/*
=============================================================================
Module Information
------------------
Name:			obj_aerith.cpp
Author:			Rich Whitehouse
Description:	server logic object: Aerith
=============================================================================
*/

#include "main.h"
#include "ai.h"

typedef enum
{
	AERANIM_CIN_TALKTIFA1 = NUM_HUMAN_ANIMS,
	AERANIM_CIN_TALKTIFA2,
	AERANIM_CIN_FLOAT1,
	AERANIM_CIN_FLOATEMBRACE,
	NUM_AERITH_ANIMS
} aerithAnims_e;

static gameAnim_t g_aerithAnims[NUM_AERITH_ANIMS] =
{
	{0, 19, 68.0f, true},			//HUMANIM_IDLE
	{92, 97, 168.0f, true},			//HUMANIM_WALK
	{92, 97, 118.0f, true},			//HUMANIM_RUNSLOW
	{92, 97, 88.0f, true},			//HUMANIM_RUN
	{0, 2, 68.0f, false},			//HUMANIM_JUMP
	{0, 2, 68.0f, false},			//HUMANIM_FALL
	{0, 2, 68.0f, false},			//HUMANIM_LAND
	{79, 81, 168.0f, false},		//HUMANIM_PAIN_HIGH1
	{79, 81, 168.0f, false},		//HUMANIM_PAIN_HIGH2
	{79, 81, 168.0f, false},		//HUMANIM_PAIN_LOW1
	{79, 81, 168.0f, false},		//HUMANIM_PAIN_LOW2
	{79, 81, 168.0f, false},		//HUMANIM_PAIN_AIR
	{79, 81, 168.0f, false},		//HUMANIM_PAIN_POPUP
	{30, 32, 168.0f, false},		//HUMANIM_GETUP_BACK
	{30, 32, 168.0f, false},		//HUMANIM_GETUP_FRONT
	{196, 196, 68.0f, false},		//HUMANIM_FLYBACK
	{196, 196, 68.0f, false},		//HUMANIM_FLYBACK2
	{19, 22, 168.0f, false},		//HUMANIM_HIT_FALLFORWARD
	{19, 22, 68.0f, false},			//HUMANIM_FALL_LAND
	{19, 22, 68.0f, false},			//HUMANIM_POPUP_LAND
	{0, 19, 1568.0f, false},		//HUMANIM_DEATH
	{486, 486, 468.0f, false},		//AERANIM_CIN_TALKTIFA1
	{487, 487, 468.0f, false},		//AERANIM_CIN_TALKTIFA2
	{488, 488, 568.0f, false},		//AERANIM_CIN_FLOAT1
	{489, 489, 3068.0f, false},		//AERANIM_CIN_FLOATEMBRACE
};

static scriptableAnim_t g_aerithScriptAnims[] =
{
	DEF_SCRIPT_ANIM(HUMANIM_IDLE),
	DEF_SCRIPT_ANIM(AERANIM_CIN_TALKTIFA1),
	DEF_SCRIPT_ANIM(AERANIM_CIN_TALKTIFA2),
	DEF_SCRIPT_ANIM(AERANIM_CIN_FLOAT1),
	DEF_SCRIPT_ANIM(AERANIM_CIN_FLOATEMBRACE),
	{0, NULL}
};

//think
void ObjAerith_Think(gameObject_t *obj, float timeMod)
{
	if (!AI_ShouldThink(obj))
	{
		return;
	}

	obj->noCullTime = g_gameTime+1000; //pay attention to aerith

	if (obj->curAnim >= AERANIM_CIN_FLOAT1)
	{
		obj->net.renderEffects |= FXFL_UNDERLIGHT;
		if (!obj->target)
		{
			objArgs_t args[5];
			args[0].key = "notmapstatic";
			args[0].val = "1";
			args[1].key = "radius";
			args[1].val = "4096";
			args[2].key = "spotlight";
			args[2].val = "1";
			args[3].key = "spotcone";
			args[3].val = "90";
			args[4].key = "shadows";
			args[4].val = "1";
			float lightAng[3] = {90.0f, 0.0f, 0.0f};
			obj->target = LServ_ObjectFromName("obj_dynamiclight", obj->net.pos, lightAng, args, 5);
			if (obj->target)
			{
				obj->target->spawnArgs = NULL;
				obj->target->numSpawnArgs = 0;
				obj->target->net.mins[0] = 0.0f;
				obj->target->net.mins[1] = 0.0f;
				obj->target->net.mins[2] = 0.0f;
			}
		}

		g_actionTime = g_gameTime+1000;
		obj->net.renderEffects |= FXFL_CONTRAST;
		obj->localTimeScale = 8.0f;

		gameObject_t *pl = &g_gameObjects[0];
		if (pl->inuse)
		{
			float angleBlendScale = 0.7f;
			float animAng[3];
			Math_VecSub(pl->net.pos, obj->net.pos, animAng);
			Math_VecToAngles(animAng, animAng);
			obj->net.ang[PITCH] = Math_BlendAngle(obj->net.ang[PITCH], 0.0f, timeMod*angleBlendScale);
			obj->net.ang[YAW] = Math_BlendAngle(obj->net.ang[YAW], animAng[YAW], timeMod*angleBlendScale);
			obj->net.ang[ROLL] = Math_BlendAngle(obj->net.ang[ROLL], 0.0f, timeMod*angleBlendScale);

			float d[3];
			Math_VecSub(pl->net.pos, obj->net.pos, d);
			float l = Math_VecLen(d);
			if (obj->target)
			{
				obj->target->net.pos[0] = obj->net.pos[0] + d[0]*0.5f;
				obj->target->net.pos[1] = obj->net.pos[1] + d[1]*0.5f;
				obj->target->net.pos[2] = obj->net.pos[2] + d[2]*0.5f;
				float f = (l+600.0f);
				if (f > 2200.0f)
				{
					f = 2200.0f;
				}
				obj->target->net.pos[2] += f;

				obj->target->net.mins[0] += timeMod*0.1f;
				if (obj->target->net.mins[0] > 1.0f)
				{
					obj->target->net.mins[0] = 1.0f;
				}
				obj->target->net.mins[1] = obj->target->net.mins[0];
				obj->target->net.mins[2] = obj->target->net.mins[0];
			}
			if (l > 1200.0f)
			{
				l = 1200.0f;
			}
			pl->plObj->camRange = l;
			if (obj->curAnim == AERANIM_CIN_FLOAT1 && l < 1200.0f)
			{
				AI_StartAnim(obj, AERANIM_CIN_FLOATEMBRACE, true);
			}
			else if (obj->curAnim == AERANIM_CIN_FLOATEMBRACE && l < 500.0f &&
				!obj->generalFlag)
			{
				obj->generalFlag = 1;
				Util_RunScript("obj_aerithscript", "aerithsave");
			}
		}
	}

	if (obj->curAnim >= AERANIM_CIN_TALKTIFA1)
	{
		Util_AnimateObject(obj, timeMod);
		return;
	}

	AI_GenericThink(obj, timeMod);
}

//pick which anim to be in
void ObjAerith_PickAnim(gameObject_t *obj, float timeMod)
{
	gameAnim_t *curAnim = obj->animTable+obj->curAnim;

	if (obj->curAnim >= AERANIM_CIN_TALKTIFA1)
	{ //script will break out
		return;
	}
	else if (obj->net.frame == curAnim->endFrame && !obj->animRestart)
	{
		AI_StartAnim(obj, HUMANIM_IDLE, true);
	}
}

//frame tick
void ObjAerith_FrameTick(gameObject_t *obj, float timeMod, int oldFrame)
{
	switch (obj->curAnim)
	{
	case HUMANIM_WALK:
	case HUMANIM_RUNSLOW:
	case HUMANIM_RUN:
		if ((obj->net.frame == 92 || obj->net.frame == 95) && obj->onGround)
		{
			int snd = g_soundFootsteps[rand()%NUM_SOUNDS_FOOTSTEPS];
			ObjSound_CreateFromIndex(obj->net.pos, snd, 1.0f, -1);
		}
		break;
	default:
		break;
	}
}

//spawn
void ObjAerith_Spawn(gameObject_t *obj, BYTE *b, const objArgs_t *args, int numArgs)
{
	AI_GenericSpawn(obj, b, args, numArgs);
	obj->death = AI_ImmortalDeath;
	obj->localFlags &= ~LFL_ENEMY;
	obj->hurtable = 0;

	obj->net.aiDescIndex = AIDESC_AERITH;
	obj->aiObj->maxHealth = obj->health;

	obj->aiObj->moveSpeed = 400.0f;
	obj->animhandler = ObjAerith_PickAnim;
	obj->animframetick = ObjAerith_FrameTick;
	obj->animTable = g_aerithAnims;
	obj->scriptAnims = g_aerithScriptAnims;
	obj->curAnim = HUMANIM_IDLE;
	obj->think = ObjAerith_Think;
}
