/*
=============================================================================
Module Information
------------------
Name:			obj_bomb.cpp
Author:			Rich Whitehouse
Description:	server logic object: bomb
=============================================================================
*/

#include "main.h"
#include "ai.h"

//just to make things a bit more readable
#define debounceFastLook		debounce5
#define debounceGrowSound		debounce4
#define debounceBlowUp			debounce3
#define debounceGonnaBlowTime	debounce2
#define debounceFlameOn			debounce
#define debounceFlameTime		debounce6
#define debounceChaseTime		debounce7

typedef enum
{
	BOMBANIM_HEADBUTT = NUM_HUMAN_ANIMS,
	NUM_BOMB_ANIMS
} bombAnims_e;

static gameAnim_t g_bombAnims[NUM_BOMB_ANIMS] =
{
	{0, 17, 68.0f, true},			//HUMANIM_IDLE
	{0, 17, 188.0f, true},			//HUMANIM_WALK
	{0, 17, 138.0f, true},			//HUMANIM_RUNSLOW
	{0, 17, 68.0f, true},			//HUMANIM_RUN
	{0, 17, 68.0f, false},			//HUMANIM_JUMP
	{0, 17, 68.0f, false},			//HUMANIM_FALL
	{0, 17, 68.0f, false},			//HUMANIM_LAND
	{18, 25, 68.0f, false},			//HUMANIM_PAIN_HIGH1
	{18, 25, 68.0f, false},			//HUMANIM_PAIN_HIGH2
	{18, 25, 68.0f, false},			//HUMANIM_PAIN_LOW1
	{18, 25, 68.0f, false},			//HUMANIM_PAIN_LOW2
	{18, 25, 68.0f, false},			//HUMANIM_PAIN_AIR
	{18, 25, 68.0f, false},			//HUMANIM_PAIN_POPUP
	{18, 22, 118.0f, false},		//HUMANIM_GETUP_BACK
	{18, 22, 118.0f, false},		//HUMANIM_GETUP_FRONT
	{18, 22, 68.0f, false},			//HUMANIM_FLYBACK
	{18, 22, 68.0f, false},			//HUMANIM_FLYBACK2
	{18, 22, 68.0f, false},			//HUMANIM_HIT_FALLFORWARD
	{18, 22, 68.0f, false},			//HUMANIM_FALL_LAND
	{18, 22, 68.0f, false},			//HUMANIM_POPUP_LAND
	{233, 233, 1568.0f, false},		//HUMANIM_DEATH
	{66, 69, 168.0f, false}			//BOMBANIM_HEADBUTT
};

static gameAnim_t g_bombAnimsHyper[NUM_BOMB_ANIMS] =
{
	{0, 17, 10.0f, true},			//HUMANIM_IDLE
	{0, 17, 50.0f, true},			//HUMANIM_WALK
	{0, 17, 40.0f, true},			//HUMANIM_RUNSLOW
	{0, 17, 10.0f, true},			//HUMANIM_RUN
	{0, 17, 20.0f, false},			//HUMANIM_JUMP
	{0, 17, 20.0f, false},			//HUMANIM_FALL
	{0, 17, 68.0f, false},			//HUMANIM_LAND
	{18, 25, 68.0f, false},			//HUMANIM_PAIN_HIGH1
	{18, 25, 68.0f, false},			//HUMANIM_PAIN_HIGH2
	{18, 25, 68.0f, false},			//HUMANIM_PAIN_LOW1
	{18, 25, 68.0f, false},			//HUMANIM_PAIN_LOW2
	{18, 25, 68.0f, false},			//HUMANIM_PAIN_AIR
	{18, 25, 68.0f, false},			//HUMANIM_PAIN_POPUP
	{18, 22, 118.0f, false},		//HUMANIM_GETUP_BACK
	{18, 22, 118.0f, false},		//HUMANIM_GETUP_FRONT
	{18, 22, 68.0f, false},			//HUMANIM_FLYBACK
	{18, 22, 68.0f, false},			//HUMANIM_FLYBACK2
	{18, 22, 68.0f, false},			//HUMANIM_HIT_FALLFORWARD
	{18, 22, 68.0f, false},			//HUMANIM_FALL_LAND
	{18, 22, 68.0f, false},			//HUMANIM_POPUP_LAND
	{233, 233, 1568.0f, false},		//HUMANIM_DEATH
	{66, 69, 168.0f, false}			//BOMBANIM_HEADBUTT
};

#define NUM_BOMB_DAMAGE_BONES	1
static damageBone_t g_bombDamageBones[NUM_BOMB_DAMAGE_BONES] =
{
	{"b19", "b18", {0.0f, 0.0f, 0.0f}, 250.0f}			//DMGBONE_FIST_R
};

//returns damage bone or -1 if not in attack
int ObjBomb_InAttack(gameObject_t *obj, int *dmgBones)
{
	if (obj->curAnim == BOMBANIM_HEADBUTT)
	{
		if (obj->net.frame >= 67 && obj->net.frame <= 69)
		{
			dmgBones[0] = DMGBONE_FIST_R;
			return 1;
		}
	}

	return -1;
}

//occupied?
bool ObjBomb_CanAct(gameObject_t *obj)
{
	if (obj->curAnim >= BOMBANIM_HEADBUTT)
	{
		return false;
	}
	if (AI_NonMovingAnim(obj))
	{
		return false;
	}
	if (obj->aiObj->noMoveTime >= g_gameTime)
	{
		return false;
	}
	return true;
}

//can i dodge?
bool ObjBomb_CanDodge(gameObject_t *obj)
{
	if (obj->generalFlag >= 0)
	{
		return false;
	}
	if (!ObjBomb_CanAct(obj))
	{
		return false;
	}

	return true;
}

//dodge (bombs currently don't dodge)
bool ObjBomb_DodgeEnemy(gameObject_t *obj, gameObject_t *enemy)
{
	return false;
}

//i'm taking you sons of bitches with me
void ObjBomb_Explode(gameObject_t *obj)
{
	float c[3];
	float ang[3] = {0.0f, 0.0f, 0.0f};
	Util_GameObjectCenter(obj, c);
	int dmg = AI_LevelDamage(obj, 500);
	obj->net.vel[0] = 0.0f;
	obj->net.vel[1] = 0.0f;
	obj->net.vel[2] = 0.0f;
	Util_RadiusDamage(obj, dmg, 2048.0f, c);
	ObjSound_Create(c, "assets/sound/cb/boom.wav", 1.0f, -1);
	ObjSound_Create(c, "assets/sound/cb/meteorsmash.wav", 1.0f, -1);
	ObjSound_Create(c, "assets/sound/cb/boom.wav", 1.0f, -1);
	ObjSound_Create(c, "assets/sound/cb/meteorsmash.wav", 1.0f, -1);
	Math_VecSub(c, obj->net.pos, c);
	ObjParticles_Create("other/bombblow", c, ang, obj->net.index);
	obj->preDeathTime = g_gameTime + (int)(200.0f*g_timeScale);
	obj->net.renderEffects2 |= FXFL2_DEATH2;
	if (obj->net.solid)
	{
		obj->net.solid = 0;
		LServ_UpdateRClip(obj);
		if (obj->generalFlag <= -1)
		{ //spawn off lesser bombs
			BYTE *b = Common_GetEntryForObject("obj_bomb1");
			if (b)
			{
				float pos[3];
				Math_VecCopy(obj->net.pos, pos);
				pos[2] += (obj->net.mins[2])+64.0f;
				if (Util_TryObjectSpawn(pos, b))
				{
					LServ_ObjectFromName("obj_bomb1", pos, obj->net.ang, NULL, 0);
				}
				pos[2] += 380.0f;
				if (Util_TryObjectSpawn(pos, b))
				{
					LServ_ObjectFromName("obj_bomb1", pos, obj->net.ang, NULL, 0);
				}
			}
		}
	}
}

//think
void ObjBomb_Think(gameObject_t *obj, float timeMod)
{
	if (!AI_ShouldThink(obj))
	{
		return;
	}

	obj->noGravTime = g_gameTime+10000;
	if (ObjBomb_CanAct(obj) && AI_GroundDistance(obj, obj->net.pos, NULL) > 64.0f)
	{
		obj->net.vel[2] -= 16.0f;
	}

	float baseMins[3] = {-120, -120, 64};
	float baseMaxs[3] = {120, 120, 360};

	float desiredScale = 1.0f;
	if (obj->debounceBlowUp)
	{
		desiredScale = 4.0f;
	}
	else if (obj->health <= (obj->aiObj->maxHealth/4))
	{
		desiredScale = 3.0f;
	}
	else if (obj->health <= (obj->aiObj->maxHealth/2))
	{
		desiredScale = 2.0f;
	}

	if (desiredScale != obj->net.modelScale[0])
	{
		if (Util_ScaledResize(obj, baseMins, baseMaxs, 200.0f, desiredScale, timeMod*0.1f))
		{
			Math_VecCopy(obj->net.pos, obj->safePos);
			if (obj->debounceGrowSound < g_gameTime)
			{
				ObjSound_Create(obj->net.pos, "assets/sound/cb/stretch.wav", 1.0f, -1);
				obj->debounceGrowSound = g_gameTime+1000;
			}
		}
	}

	if (obj->debounceBlowUp)
	{
		if (obj->debounceGonnaBlowTime < g_gameTime)
		{
			ObjSound_Create(obj->net.pos, "assets/sound/cb/gonnablow.wav", 1.0f, -1);
			ObjSound_Create(obj->net.pos, "assets/sound/cb/gonnablow.wav", 1.0f, -1);
			obj->debounceGonnaBlowTime = g_gameTime+2000;
		}
		obj->net.vel[0] = Util_RandFloat(-1024.0f, 1024.0f);
		obj->net.vel[1] = Util_RandFloat(-1024.0f, 1024.0f);
		obj->net.vel[2] = Util_RandFloat(-128.0f, 128.0f);
		if (obj->debounceBlowUp < g_gameTime)
		{
			ObjBomb_Explode(obj);
			obj->debounceBlowUp = 0;
		}
		else
		{
			AI_GenericThink(obj, timeMod);
		}
		return;
	}

	if (obj->generalFlag <= -1)
	{
		if (!obj->debounceFlameTime)
		{
			obj->debounceFlameTime = g_gameTime + 2000 + rand()%8000;
		}
		if (obj->debounceFlameTime < g_gameTime)
		{
			ObjSound_Create(obj->net.pos, "assets/sound/cb/chargeflame.wav", 1.0f, -1);
			obj->debounceFlameOn = g_gameTime + 10000 + rand()%5000;
			obj->debounceFlameTime = g_gameTime + 16000 + rand()%10000;
		}
	}

	AI_StandardGoals(obj, timeMod);
	AI_GenericThink(obj, timeMod);
	AI_GetToGoal(obj, timeMod);
	if (obj->health <= 0)
	{
		obj->net.renderEffects &= ~FXFL_FLAMES;
	}
	else if (obj->debounceFlameOn > g_gameTime)
	{
		obj->net.renderEffects |= FXFL_FLAMES;
	}
	else if (obj->isOnFire < g_gameTime)
	{
		obj->net.renderEffects &= ~FXFL_FLAMES;
	}

	damageBone_t localDamageBones[NUM_BOMB_DAMAGE_BONES];
	memcpy(localDamageBones, g_bombDamageBones, sizeof(localDamageBones));
	for (int i = 0; i < NUM_BOMB_DAMAGE_BONES; i++)
	{ //scale damage bones by bomb scale
		localDamageBones[i].radius = g_bombDamageBones[i].radius*obj->net.modelScale[0];
	}

	if (obj->debounceFastLook >= g_gameTime)
	{
		obj->net.ang[YAW] = Math_BlendAngleLinear(obj->net.ang[YAW], obj->aiObj->lookYaw, 45.0f);
	}

	dmgBoneLine_t dmgPos[NUM_BOMB_DAMAGE_BONES];
	AI_TransformDamageBones(obj, localDamageBones, dmgPos, NUM_BOMB_DAMAGE_BONES);
	int dmgBones[MAX_AI_DAMAGE_BONES];
	int numDmgBones = ObjBomb_InAttack(obj, dmgBones);
	if (obj->aiObj->hasLastDmg && numDmgBones >= 0)
	{
		int dmg = AI_LevelDamage(obj, 40);
		for (int i = 0; i < numDmgBones; i++)
		{
			AI_RunDamageBone(obj, &localDamageBones[dmgBones[i]], &obj->aiObj->lastDmg[dmgBones[i]],
				&dmgPos[dmgBones[i]], dmg);
		}
	}

	obj->atkType = ATKTYPE_NORMAL;

	memcpy(obj->aiObj->lastDmg, dmgPos, sizeof(dmgPos));
	obj->aiObj->hasLastDmg = true;

	obj->aiObj->combatRange = 700.0f;
	obj->animTable = g_bombAnims;
	if (!AI_NonAnglingAnim(obj) && obj->aiObj->enemy && AI_FacingEnemy(obj, obj->aiObj->enemy))
	{
		if (obj->aiObj->enemy->net.index < MAX_NET_CLIENTS &&
			ObjBomb_CanDodge(obj) &&
			ObjPlayer_InDodgeableAttack(obj->aiObj->enemy) &&
			ObjBomb_DodgeEnemy(obj, obj->aiObj->enemy))
		{ //dodge
			obj->aiObj->noMoveTime = g_gameTime+100;
		}
		else
		{
			bool shouldCloseIn = AI_ShouldCloseIn(obj);
			if (obj->aiObj->attackTime < g_gameTime && shouldCloseIn && ObjBomb_CanAct(obj) &&
				obj->debounceChaseTime < (g_gameTime-4000))
			{
				serverTime_t chaseDur = (obj->generalFlag <= -1) ? (serverTime_t)8000 : (serverTime_t)4000;
				obj->debounceChaseTime = g_gameTime+chaseDur;
				obj->animTable = g_bombAnimsHyper;
				AI_StartAnim(obj, HUMANIM_IDLE, true);
			}
			if (obj->debounceChaseTime > g_gameTime)
			{
				obj->animTable = g_bombAnimsHyper;

				obj->aiObj->combatRange = 400.0f;
				float closeAttackDist = 500.0f + (100.0f*obj->net.modelScale[0]);
				if (obj->aiObj->distToEnemy < closeAttackDist)
				{
					obj->debounceChaseTime = g_gameTime-1;
					AI_StartAnim(obj, BOMBANIM_HEADBUTT, true);
					if (obj->generalFlag <= -1)
					{
						obj->aiObj->attackTime = g_gameTime + Util_LocalDelay(obj, (serverTime_t)(1500+rand()%2000));
					}
					else
					{
						obj->aiObj->attackTime = g_gameTime + Util_LocalDelay(obj, (serverTime_t)(2000+rand()%4000));
					}
					obj->aiObj->noMoveTime = g_gameTime + Util_LocalDelay(obj, 1000);
				}
			}
			else
			{
				float closeDist = (obj->generalFlag <= 0) ? 600.0f : 400.0f;
				float r = (shouldCloseIn) ? closeDist : 800.0f;
				obj->aiObj->combatRange = g_ai.playerCloseDist+r;
			}
		}
	}
}

//pick which anim to be in
void ObjBomb_PickAnim(gameObject_t *obj, float timeMod)
{
	gameAnim_t *curAnim = obj->animTable+obj->curAnim;

	if (obj->net.frame == curAnim->endFrame && !obj->animRestart)
	{
		AI_StartAnim(obj, HUMANIM_IDLE, true);
	}
}

//frame tick
void ObjBomb_FrameTick(gameObject_t *obj, float timeMod, int oldFrame)
{
	switch (obj->curAnim)
	{
	case BOMBANIM_HEADBUTT:
		if (obj->net.frame == 66)
		{
			ObjSound_CreateFromIndex(obj->net.pos, g_soundPunch[rand()%NUM_SOUNDS_PUNCH], 1.0f, -1);
			float fwd[3];
			float f = 600.0f;
			Math_AngleVectors(obj->net.ang, fwd, 0, 0);
			obj->net.vel[0] += fwd[0]*f;
			obj->net.vel[1] += fwd[1]*f;
			obj->net.vel[2] = 100.0f;
		}
		break;
	default:
		break;
	}
}

//touch
void ObjBomb_Touch(gameObject_t *obj, gameObject_t *other, const collObj_t *col)
{
	AI_GenericTouch(obj, other, col);
}

//bombs currently can't block
bool ObjBomb_AttackBlock(gameObject_t *obj, gameObject_t *hurter, int dmg, collObj_t *col,
							float *impactPos, float *impactAng)
{
	return false;
}

//death
void ObjBomb_Death(gameObject_t *obj, gameObject_t *killer, int dmg)
{
	AI_DeathRoutines(obj, killer, dmg);
	obj->net.renderEffects &= ~FXFL_DEATH;
	obj->preDeathTime = 0;
	obj->debounceBlowUp = g_gameTime+4000;
	ObjSound_Create(obj->net.pos, "assets/sound/cb/gonnablow.wav", 1.0f, -1);
	obj->debounceGonnaBlowTime = g_gameTime+2000;
}

//called when inflicting damage
void ObjBomb_OnHurt(gameObject_t *obj, gameObject_t *victim, int dmg, const collObj_t *col)
{
	if (victim && victim != obj && obj->debounceFlameOn >= g_gameTime)
	{
		victim->isOnFire = g_gameTime+1000+(rand()%3000);
	}
}

//spawn
void ObjBomb_Spawn(gameObject_t *obj, BYTE *b, const objArgs_t *args, int numArgs)
{
	float p[3];
	Math_VecCopy(obj->net.pos, p);
	AI_GenericSpawn(obj, b, args, numArgs);
	Math_VecCopy(p, obj->net.pos);

	obj->touch = ObjBomb_Touch;
	obj->animframetick = ObjBomb_FrameTick;
	obj->onhurt = ObjBomb_OnHurt;
	obj->death = ObjBomb_Death;

	const char *elvalue = Common_GetValForKey(b, "elvalue");
	if (elvalue && elvalue[0])
	{
		obj->generalFlag = atoi(elvalue);
	}

	if (obj->generalFlag == -1)
	{ //can spawn other bombs on death
		LServ_CacheObj("obj_bomb1");
		obj->aiObj->makoValue = 50 + rand()%20;
		obj->net.aiDescIndex = AIDESC_BOMB2;
	}
	else
	{
		obj->aiObj->makoValue = 24 + rand()%6;
		obj->net.aiDescIndex = AIDESC_BOMB1;
	}

	obj->aiObj->dropChances[INVITEM_SHRAPNEL] = 60;
	if (obj->generalFlag == -1)
	{
		obj->aiObj->dropChances[INVITEM_SHRAPNEL] = 100;
		obj->aiObj->dropChances[INVITEM_RIGHTARM] = 30;
	}
	obj->aiObj->maxHealth = obj->health;

	obj->net.modelScale[0] = 1.0f;
	obj->net.modelScale[1] = 1.0f;
	obj->net.modelScale[2] = 1.0f;

	obj->aiObj->dmgEffect = "melee/impacten";
	obj->aiObj->dmgSounds = g_soundHitHeavy;
	obj->aiObj->numDmgSounds = NUM_SOUNDS_HITHEAVY;
	if (obj->generalFlag == -1)
	{
		obj->aiObj->moveSpeed = 180.0f;
	}
	else
	{
		obj->aiObj->moveSpeed = 120.0f;
	}
	obj->aiObj->fly = true;
	obj->animhandler = ObjBomb_PickAnim;
	obj->animTable = g_bombAnims;
	obj->curAnim = HUMANIM_IDLE;
	obj->think = ObjBomb_Think;
	obj->attackblock = ObjBomb_AttackBlock;
}
