/*
=============================================================================
Module Information
------------------
Name:			obj_light.cpp
Author:			Rich Whitehouse
Description:	map light objects (static and dynamic)

Object Arguments
radius:			Light radius
color:			Color in form of (r g b), each component ranging 0.0-1.0.
				Can be > 1.0 for oversaturation.
shadows:		0 or 1 to toggle shadows.
notmapstatic:	Set on lights that will move in code or script, so that
				static shadow volumes are not cached.
fullambient:	Becomes a full ambient light, which renders a soft lighting
				pass, which includes lightmaps on geometry that has them.
novolumes:		Allow shadowmaps to be cast, but not shadow volumes.
spotlight:		Set to 1 to turn into a spotlight instead of the default of
				omnidirectional.
spotcone:		Specify the number of degrees for a spotlight's cone to span.
spottex:		Name of the texture to project for a spotlight. If none is
				provided, the default circular attenutation is used.
startoff:		Start off
tracker:		Special case, tracks all living things
=============================================================================
*/

#include "main.h"

//think
void ObjLight_Think(gameObject_t *obj, float timeMod)
{
	if (obj->target && obj->target->inuse)
	{
		obj->net.pos[0] = obj->target->net.pos[0]+obj->spareVec[0];
		obj->net.pos[1] = obj->target->net.pos[1]+obj->spareVec[1];
		obj->net.pos[2] = obj->target->net.pos[2]+obj->spareVec[2];
	}

	if (obj->generalFlag2 && !g_cinema)
	{ //tracker
		float trackMins[3], trackMaxs[3];
		bool anyLiving = false;
		for (int i = 0; i < g_gameObjectSlots; i++)
		{
			gameObject_t *other = &g_gameObjects[i];
			if (!other->inuse || !other->hurtable)
			{
				continue;
			}

			if (!anyLiving)
			{
				Math_VecAdd(other->net.pos, other->spawnMins, trackMins);
				Math_VecAdd(other->net.pos, other->spawnMaxs, trackMaxs);
				anyLiving = true;
			}
			else
			{
				float nmins[3], nmaxs[3];
				Math_VecAdd(other->net.pos, other->spawnMins, nmins);
				Math_VecAdd(other->net.pos, other->spawnMaxs, nmaxs);
				Math_ExpandBounds(trackMins, trackMaxs, nmins, nmaxs);
			}
		}

		if (anyLiving)
		{
			float desiredPos[3];
			desiredPos[0] = trackMins[0]+(trackMins[0]-trackMaxs[0])*0.5f;
			desiredPos[1] = trackMins[1]+(trackMins[1]-trackMaxs[1])*0.5f;
			desiredPos[2] = trackMins[2]+(trackMins[2]-trackMaxs[2])*0.5f;
			desiredPos[2] += obj->net.modelScale[0]*0.25f;
			Phys_LinearMove(obj, desiredPos, 128.0f*timeMod);
		}
	}

	obj->thinkTime = g_gameTime;
}

//post spawn routine
void ObjLight_PostSpawn(gameObject_t *obj)
{
	if (obj->target && obj->target->inuse)
	{
		obj->parent = obj->target;
		obj->target->child = obj;
		obj->think = ObjLight_Think;
		obj->thinkTime = g_gameTime+50;
		obj->net.staticIndex = -1; //bound lights cannot be static
		Math_VecSub(obj->net.pos, obj->target->net.pos, obj->spareVec);
	}
}

//light spawn
void ObjLight_Spawn(gameObject_t *obj, BYTE *b, const objArgs_t *args, int numArgs)
{
	obj->net.solid = -1; //default to no light volume

	obj->postSpawn = ObjLight_PostSpawn;

	bool gotRadius = false;
	bool gotColor = false;
	float spotCone = 0.0f;
	int spotTex = 0;
	int lightVolumeIndex = -1;
	obj->net.lerpDuration = 1; //default to shadows
	obj->net.staticIndex = obj->net.index;
	obj->net.frame = 0;
	if (obj->spawnArgs && obj->numSpawnArgs > 0)
	{
		int i = 0;
		while (i < obj->numSpawnArgs)
		{
			const objArgs_t *arg = obj->spawnArgs+i;
			if (!_stricmp(arg->key, "radius"))
			{ //todo - stop being lazy, add non-uniform light support
				obj->net.modelScale[0] = (float)atof(arg->val);
				gotRadius = true;
			}
			else if (!_stricmp(arg->key, "color"))
			{
				Util_ParseVector(arg->val, obj->net.mins);
				gotColor = true;
			}
			else if (!_stricmp(arg->key, "lightVolumeIndex"))
			{
				const char *s = Common_GetValForKey(b, "static");
				if (s && s[0] && atoi(s) == 1)
				{ //made sure the light was static. if it is, use this lightVolumeIndex.
					obj->net.solid = atoi(arg->val);
				}
			}
			else if (!_stricmp(arg->key, "shadows"))
			{
				obj->net.lerpDuration = atoi(arg->val);
			}
			else if (!_stricmp(arg->key, "modshadows") && atoi(arg->val) == 1)
			{
				obj->net.frame |= LIGHTFL_MODSHADOWS;
			}
			else if (!_stricmp(arg->key, "notmapstatic"))
			{ //force non-static, for moving lights etc
				obj->net.staticIndex = -1;
			}
			else if (!_stricmp(arg->key, "double") && atoi(arg->val) == 1)
			{
				obj->net.frame |= LIGHTFL_DOUBLE;
			}
			else if (!_stricmp(arg->key, "novolumes") && atoi(arg->val) == 1)
			{
				obj->net.frame |= LIGHTFL_NOVOL;
			}
			else if (!_stricmp(arg->key, "ambprogram") && atoi(arg->val) == 1)
			{
				obj->net.frame |= LIGHTFL_AMBPROG;
			}
			else if (!_stricmp(arg->key, "fullambient") && atoi(arg->val) == 1)
			{
				obj->net.frame |= LIGHTFL_FULLAMB;
				obj->net.lerpDuration = 0; //no dynamic shadows on ambient lights
			}
			else if (!_stricmp(arg->key, "dullspec") && atoi(arg->val) == 1)
			{
				obj->net.frame |= LIGHTFL_DULLSPEC;
			}
			else if (!_stricmp(arg->key, "dullspec2") && atoi(arg->val) == 1)
			{
				obj->net.frame |= LIGHTFL_DULLSPEC2;
			}
			else if (!_stricmp(arg->key, "spotlight") && atoi(arg->val) == 1)
			{
				obj->net.frame |= LIGHTFL_SPOTLIGHT;
				if (!spotCone)
				{
					spotCone = 90.0f;
				}
			}
			else if (!_stricmp(arg->key, "spotcone"))
			{
				spotCone = (float)atof(arg->val);
			}
			else if (!_stricmp(arg->key, "spottex"))
			{
				char str[256];
				sprintf(str, "^%s", arg->val, str);
				spotTex = g_sharedFn->Common_ServerString(str);
			}
			else if (!_stricmp(arg->key, "startoff"))
			{
				obj->net.renderEffects |= FXFL_HIDDEN;
			}
			else if (!_stricmp(arg->key, "tracker"))
			{
				obj->net.staticIndex = -1;
				obj->generalFlag2 = 1;
				obj->think = ObjLight_Think;
				obj->thinkTime = g_gameTime;
			}
			i++;
		}
	}

	if (!gotRadius)
	{
		const char *s = Common_GetValForKey(b, "defaultRadius");
		if (s && s[0])
		{
			obj->net.modelScale[0] = (float)atof(s);
		}
		else
		{
			obj->net.modelScale[0] = 256.0f;
		}
	}

	if (obj->net.frame & LIGHTFL_SPOTLIGHT)
	{
		viewFrustum_t spotFrust;
		memset(&spotFrust, 0, sizeof(spotFrust));
		float axis[3][3];
		Math_AngleVectors(obj->net.ang, axis[0], axis[1], axis[2]);
		float frustp[3] = {0.0f, 0.0f, 0.0f};
		Math_CreateFrustumForViewAxis(frustp, axis[0], &spotFrust, spotCone, spotCone, 0.01f, obj->net.modelScale[0]);
		float p[8][3];
		Math_GetFrustumPoints(&spotFrust, p, NULL);
		Math_BoundsFromPoints(obj->spawnMins, obj->spawnMaxs, p[0], 8);
	}
	else
	{
		obj->spawnMins[0] = -obj->net.modelScale[0];
		obj->spawnMins[1] = -obj->net.modelScale[0];
		obj->spawnMins[2] = -obj->net.modelScale[0];
		obj->spawnMaxs[0] = obj->net.modelScale[0];
		obj->spawnMaxs[1] = obj->net.modelScale[0];
		obj->spawnMaxs[2] = obj->net.modelScale[0];
	}

	if ((obj->net.frame & LIGHTFL_SPOTLIGHT) && !spotTex)
	{
		spotTex = g_sharedFn->Common_ServerString("^assets/textures/2dattn");
	}
	obj->net.modelScale[1] = spotCone;
	obj->net.modelScale[2] = spotCone;
	obj->net.strIndexC = spotTex;

	if (!gotColor)
	{
		const char *s = Common_GetValForKey(b, "defaultColor");
		if (s && s[0])
		{
			Util_ParseVector(s, obj->net.mins);
		}
		else
		{
			obj->net.mins[0] = 1.0f;
			obj->net.mins[1] = 1.0f;
			obj->net.mins[2] = 1.0f;
		}
	}
}
