/*
=============================================================================
Module Information
------------------
Name:			obj_madouge.cpp
Author:			Rich Whitehouse
Description:	server logic object: madouge
=============================================================================
*/

#include "main.h"
#include "ai.h"

//just to make things a bit more readable
#define debounceFastLook		debounce5

typedef enum
{
	MADANIM_SPINDEATH = NUM_HUMAN_ANIMS,
	NUM_MAD_ANIMS
} madAnims_e;

static gameAnim_t g_madAnims[NUM_MAD_ANIMS] =
{
	{0, 15, 68.0f, true},			//HUMANIM_IDLE
	{26, 29, 268.0f, true},			//HUMANIM_WALK
	{26, 29, 218.0f, true},			//HUMANIM_RUNSLOW
	{26, 29, 168.0f, true},			//HUMANIM_RUN
	{27, 28, 68.0f, false},			//HUMANIM_JUMP
	{28, 28, 68.0f, false},			//HUMANIM_FALL
	{29, 29, 68.0f, false},			//HUMANIM_LAND
	{16, 23, 68.0f, false},			//HUMANIM_PAIN_HIGH1
	{16, 23, 68.0f, false},			//HUMANIM_PAIN_HIGH2
	{16, 23, 68.0f, false},			//HUMANIM_PAIN_LOW1
	{16, 23, 68.0f, false},			//HUMANIM_PAIN_LOW2
	{32, 36, 68.0f, false},			//HUMANIM_PAIN_AIR
	{128, 130, 68.0f, false},		//HUMANIM_PAIN_POPUP
	{133, 137, 118.0f, false},		//HUMANIM_GETUP_BACK
	{133, 137, 118.0f, false},		//HUMANIM_GETUP_FRONT
	{128, 130, 68.0f, false},		//HUMANIM_FLYBACK
	{128, 130, 68.0f, false},		//HUMANIM_FLYBACK2
	{128, 130, 68.0f, false},		//HUMANIM_HIT_FALLFORWARD
	{131, 133, 68.0f, false},		//HUMANIM_FALL_LAND
	{131, 133, 68.0f, false},		//HUMANIM_POPUP_LAND
	{20, 20, 1568.0f, false},		//HUMANIM_DEATH
	{96, 104, 68.0f, false}			//MADANIM_SPINDEATH
};

#define NUM_MAD_DAMAGE_BONES	1
static damageBone_t g_madDamageBones[NUM_MAD_DAMAGE_BONES] =
{
	{"b5", "b4", {0.0f, 0.0f, 0.0f}, 140.0f}			//DMGBONE_FIST_R
};

//returns damage bone or -1 if not in attack
int ObjMad_InAttack(gameObject_t *obj, int *dmgBones)
{
	return -1;
}

//occupied?
bool ObjMad_CanAct(gameObject_t *obj)
{
	if (!obj->onGround)
	{
		return false;
	}
	if (obj->curAnim >= MADANIM_SPINDEATH)
	{
		return false;
	}
	if (AI_NonMovingAnim(obj))
	{
		return false;
	}
	if (obj->aiObj->noMoveTime >= g_gameTime)
	{
		return false;
	}
	return true;
}

//can i dodge?
bool ObjMad_CanDodge(gameObject_t *obj)
{
	if (obj->generalFlag >= 0)
	{
		return false;
	}
	if (!ObjMad_CanAct(obj))
	{
		return false;
	}

	return true;
}

//dodge (mads currently don't dodge)
bool ObjMad_DodgeEnemy(gameObject_t *obj, gameObject_t *enemy)
{
	return false;
}

//think
void ObjMad_Think(gameObject_t *obj, float timeMod)
{
	if (!AI_ShouldThink(obj))
	{
		return;
	}

	AI_StandardGoals(obj, timeMod);
	AI_GenericThink(obj, timeMod);
	AI_GetToGoal(obj, timeMod);

	if (obj->debounceFastLook >= g_gameTime)
	{
		obj->net.ang[YAW] = Math_BlendAngleLinear(obj->net.ang[YAW], obj->aiObj->lookYaw, 45.0f);
	}

	dmgBoneLine_t dmgPos[NUM_MAD_DAMAGE_BONES];
	AI_TransformDamageBones(obj, g_madDamageBones, dmgPos, NUM_MAD_DAMAGE_BONES);
	int dmgBones[MAX_AI_DAMAGE_BONES];
	int numDmgBones = ObjMad_InAttack(obj, dmgBones);
	if (obj->aiObj->hasLastDmg && numDmgBones >= 0)
	{
		int dmg = AI_LevelDamage(obj, 80);
		for (int i = 0; i < numDmgBones; i++)
		{
			AI_RunDamageBone(obj, &g_madDamageBones[dmgBones[i]], &obj->aiObj->lastDmg[dmgBones[i]],
				&dmgPos[dmgBones[i]], dmg);
		}
	}

	obj->atkType = ATKTYPE_NORMAL;

	memcpy(obj->aiObj->lastDmg, dmgPos, sizeof(dmgPos));
	obj->aiObj->hasLastDmg = true;

	bool canAct = ObjMad_CanAct(obj);
	if (obj->curAnim < MADANIM_SPINDEATH && !canAct)
	{
		obj->debounce3 = g_gameTime+200;
	}

	obj->aiObj->combatRange = 700.0f;
	if (!AI_NonAnglingAnim(obj) && obj->aiObj->enemy && AI_FacingEnemy(obj, obj->aiObj->enemy))
	{
		if (obj->aiObj->enemy->net.index < MAX_NET_CLIENTS &&
			ObjMad_CanDodge(obj) &&
			ObjPlayer_InDodgeableAttack(obj->aiObj->enemy) &&
			ObjMad_DodgeEnemy(obj, obj->aiObj->enemy))
		{ //dodge
			obj->aiObj->noMoveTime = g_gameTime+100;
		}
		else
		{
			bool shouldCloseIn = AI_ShouldCloseIn(obj);
			if (obj->aiObj->attackTime < g_gameTime && shouldCloseIn && ObjMad_CanAct(obj))
			{
				obj->aiObj->combatRange = 400.0f;
				float closeAttackDist = 1000.0f;
				if (obj->aiObj->distToEnemy < closeAttackDist)
				{
					AI_StartAnim(obj, MADANIM_SPINDEATH, true);
					obj->debounce4 = g_gameTime+8000;
					if (obj->generalFlag <= -1)
					{
						obj->aiObj->attackTime = g_gameTime + Util_LocalDelay(obj, (serverTime_t)(1500+rand()%2000));
					}
					else
					{
						obj->aiObj->attackTime = g_gameTime + Util_LocalDelay(obj, (serverTime_t)(2000+rand()%4000));
					}
					obj->aiObj->noMoveTime = g_gameTime + Util_LocalDelay(obj, 1000);
				}
			}
			else
			{
				float closeDist = (obj->generalFlag <= 0) ? 600.0f : 400.0f;
				float r = (shouldCloseIn) ? closeDist : 800.0f;
				obj->aiObj->combatRange = g_ai.playerCloseDist+r;
			}
		}
	}
}

//pick which anim to be in
void ObjMad_PickAnim(gameObject_t *obj, float timeMod)
{
	gameAnim_t *curAnim = obj->animTable+obj->curAnim;

	if (obj->curAnim == MADANIM_SPINDEATH)
	{
		if (obj->net.frame == curAnim->endFrame && !obj->animRestart)
		{
			if (obj->debounce4 < g_gameTime)
			{
				AI_StartAnim(obj, HUMANIM_IDLE, true);
			}
			else
			{
				if (obj->chain)
				{
					obj->chain->debounce3 = g_gameTime+100;
				}
				obj->aiObj->noMoveTime = g_gameTime+100;
				if (obj->generalFlag <= -1)
				{
					obj->aiObj->attackTime = g_gameTime + Util_LocalDelay(obj, (serverTime_t)(1500+rand()%2000));
				}
				else
				{
					obj->aiObj->attackTime = g_gameTime + Util_LocalDelay(obj, (serverTime_t)(2000+rand()%4000));
				}
				obj->net.ang[YAW] -= timeMod*100.0f;
				if (obj->aiObj->enemy)
				{
					float speed = (obj->generalFlag <= -1) ? 300.0f : 180.0f;
					float d[3];
					Math_VecSub(obj->aiObj->enemy->net.pos, obj->net.pos, d);
					Math_VecNorm(d);
					obj->net.vel[0] = d[0]*speed;
					obj->net.vel[1] = d[1]*speed;
				}
			}
		}
	}
	else if (obj->net.frame == curAnim->endFrame && !obj->animRestart)
	{
		AI_StartAnim(obj, HUMANIM_IDLE, true);
	}
}

//frame tick
void ObjMad_FrameTick(gameObject_t *obj, float timeMod, int oldFrame)
{
	switch (obj->curAnim)
	{
	case HUMANIM_WALK:
	case HUMANIM_RUNSLOW:
	case HUMANIM_RUN:
		if (obj->onGround && (obj->net.frame == 26 || obj->net.frame == 28))
		{
			ObjSound_Create(obj->net.pos, "assets/sound/cb/objland.wav", 1.0f, -1);
		}
		break;
	case MADANIM_SPINDEATH:
		break;
	default:
		break;
	}
}

//touch
void ObjMad_Touch(gameObject_t *obj, gameObject_t *other, const collObj_t *col)
{
	AI_GenericTouch(obj, other, col);
}

//mads currently can't block
bool ObjMad_AttackBlock(gameObject_s *obj, gameObject_s *hurter, int dmg, collObj_t *col,
							float *impactPos, float *impactAng)
{
	return false;
}

//removal
void ObjMad_OnRemove(gameObject_t *obj)
{
	if (obj->chain)
	{
		obj->chain->chain = NULL;
		obj->chain->think = ObjGeneral_RemoveThink;
		obj->chain->thinkTime = g_gameTime;
	}
}

//spawn
void ObjMad_Spawn(gameObject_t *obj, BYTE *b, const objArgs_t *args, int numArgs)
{
	AI_GenericSpawn(obj, b, args, numArgs);

	obj->touch = ObjMad_Touch;
	obj->animframetick = ObjMad_FrameTick;
	obj->onremove = ObjMad_OnRemove;

	const char *elvalue = Common_GetValForKey(b, "elvalue");
	if (elvalue && elvalue[0])
	{
		obj->generalFlag = atoi(elvalue);
	}

	if (obj->generalFlag == -1)
	{
		obj->aiObj->makoValue = 120 + rand()%60;
		obj->net.aiDescIndex = AIDESC_MAD2;
	}
	else
	{
		obj->aiObj->makoValue = 40 + rand()%10;
		obj->net.aiDescIndex = AIDESC_MAD1;
	}

	obj->aiObj->dropChances[INVITEM_GRAVIBALL] = 10;
	obj->aiObj->dropChances[INVITEM_LOCOWEED] = 20;
	if (obj->generalFlag == -1)
	{
		obj->aiObj->dropChances[INVITEM_SPEEDDRINK] = 20;
		obj->aiObj->dropChances[INVITEM_LOCOWEED] = 80;
		obj->aiObj->dropChances[INVITEM_GRAVIBALL] = 50;
	}
	obj->aiObj->maxHealth = obj->health;

	obj->aiObj->dmgEffect = "melee/impacten";
	obj->aiObj->dmgSounds = g_soundHitHeavy;
	obj->aiObj->numDmgSounds = NUM_SOUNDS_HITHEAVY;
	if (obj->generalFlag == -1)
	{
		obj->aiObj->moveSpeed = 400.0f;
	}
	else
	{
		obj->aiObj->moveSpeed = 300.0f;
	}
	obj->animhandler = ObjMad_PickAnim;
	obj->animTable = g_madAnims;
	obj->curAnim = HUMANIM_IDLE;
	obj->think = ObjMad_Think;
	obj->attackblock = ObjMad_AttackBlock;
	
	float p[3];
	p[0] = obj->net.pos[0];
	p[1] = obj->net.pos[1];
	p[2] = obj->net.pos[2]+obj->net.maxs[2] + 128.0f;
	obj->chain = LServ_ObjectFromName("obj_madball", p, obj->net.ang, NULL, 0);
	if (obj->chain)
	{
		obj->chain->chain = obj;
		if (obj->generalFlag <= -1)
		{
			obj->chain->net.mins[0] = -150.0f;
			obj->chain->net.mins[1] = -150.0f;
			obj->chain->net.mins[2] = -150.0f;
			obj->chain->net.maxs[0] = 150.0f;
			obj->chain->net.maxs[1] = 150.0f;
			obj->chain->net.maxs[2] = 150.0f;
			obj->chain->radius = 256.0f;
			Math_VecCopy(obj->chain->net.mins, obj->chain->spawnMins);
			Math_VecCopy(obj->chain->net.maxs, obj->chain->spawnMaxs);
		}
		obj->chain->think(obj, 0.5f);
	}
}

//collision
void ObjMadBall_Touch(gameObject_t *obj, gameObject_t *other, const collObj_t *col)
{
	float p[3];
	float ang[3];
	if (col)
	{
		Math_VecCopy((float *)col->endPos, p);
		Math_VecToAngles((float *)col->endNormal, ang);
	}
	else
	{
		Math_VecCopy(obj->net.pos, p);
		ang[0] = 0.0f;
		ang[1] = 0.0f;
		ang[2] = 0.0f;
	}
	if (other != obj->chain && !obj->onGround &&
		(obj->chain->debounce3 < g_gameTime || !other || !other->hurtable))
	{
		ObjSound_Create(obj->net.pos, "assets/sound/cb/ironhit.wav", 1.0f, -1);
		ObjParticles_Create("melee/impact", p, ang, -1);
		if (obj->chain && other && other->hurtable)
		{
			ObjParticles_Create("melee/impacten", p, ang, -1);
			int snd = g_soundHitHeavy[rand()%NUM_SOUNDS_HITHEAVY];
			ObjSound_CreateFromIndex(obj->net.pos, snd, 1.0f, -1);
			ObjSound_CreateFromIndex(obj->net.pos, snd, 1.0f, -1);
			int atkType = obj->chain->atkType;
			obj->chain->atkType = (obj->chain->net.frame == 104) ? ATKTYPE_KNOCKBACK : ATKTYPE_KNOCKBACK2;
			int fullDmg = (obj->chain->net.frame == 104) ? 100 : 30;
			fullDmg = AI_LevelDamage(obj->chain, fullDmg);
			int dmg = (!Util_ValidEnemies(obj->chain, other)) ? 1 : fullDmg;
			Util_DamageObject(obj->chain, other, dmg, col);
			obj->chain->atkType = atkType;
		}
	}
	obj->debounce = g_gameTime + 500;
}

//madball think
void ObjMadBall_Think(gameObject_t *obj, float timeMod)
{
	gameObject_t *owner = obj->chain;

	if (owner && owner->health > 0)
	{
#if 0
		modelMatrix_t boneMat;
		if (owner->rcColModel && g_sharedFn->Coll_GetModelBoneMatrix(owner->rcColModel, "b28", &boneMat))
		{
			float fwd[3] = {0.0f, 0.0f, -1.0f};
			float posFwd[3];
			Math_TransformPointByMatrix(&boneMat, fwd, posFwd);
			Math_VecSub(posFwd, boneMat.o, posFwd);
			Math_VecNorm(posFwd);
			boneMat.o[0] += posFwd[0]*128.0f;
			boneMat.o[1] += posFwd[1]*128.0f;
			boneMat.o[2] += posFwd[2]*128.0f;
			//Util_DebugLine(obj->net.pos, boneMat.o);
			Math_VecSub(boneMat.o, obj->net.pos, obj->net.vel);
			obj->net.vel[0] *= 2.0f;
			obj->net.vel[1] *= 2.0f;
			obj->net.vel[2] *= 2.0f;
			Phys_ApplyObjectPhysics(obj, timeMod, 96.0f, 0.0f, 0.0f);
		}
#else
		modelMatrix_t boneMat;
		if (owner->rcColModel && g_sharedFn->Coll_GetModelBoneMatrix(owner->rcColModel, "b24", &boneMat))
		{
			float right[3] = {1.0f, 0.0f, 0.0f};
			float up[3] = {0.0f, 1.0f, 0.0f};
			float fwd[3] = {0.0f, 0.0f, 1.0f};
			//float posFwd[3];
			//Math_TransformPointByMatrix(&boneMat, fwd, posFwd);
			//Math_VecSub(posFwd, boneMat.o, fwd);
			//Math_VecNorm(fwd);
			float ofsR, ofsU, ofsF;
			serverTime_t soundDur;
			if (obj->debounce3 > g_gameTime)
			{
				obj->physModelOwner = obj->chain;
				ofsR = sinf(g_gameTime*0.02f)*1000.0f;
				ofsU = cosf(g_gameTime*0.02f)*1000.0f;
				ofsF = sinf(g_gameTime*0.01f)*100.0f;
				soundDur = 200;
			}
			else
			{
				obj->physModelOwner = NULL;
				ofsR = sinf(g_gameTime*0.008f)*400.0f;
				ofsU = cosf(g_gameTime*0.008f)*400.0f;
				ofsF = 50.0f + sinf(g_gameTime*0.004f)*50.0f;
				soundDur = 500;
			}
			if (obj->debounce < g_gameTime && obj->debounce2 < g_gameTime && (fabsf(ofsU) < 80.0f || soundDur < 400))
			{
				int snd = (rand()%10 < 5) ? g_soundDeepBlow[0] : g_soundDeepBlow[2];
				ObjSound_CreateFromIndex(obj->net.pos, snd, 1.0f, -1);
				obj->debounce2 = g_gameTime+soundDur;
			}
			float desiredPos[3];
			Math_VecCopy(boneMat.o, desiredPos);
			desiredPos[0] += right[0]*ofsR;
			desiredPos[1] += right[1]*ofsR;
			desiredPos[2] += right[2]*ofsR;
			desiredPos[0] += up[0]*ofsU;
			desiredPos[1] += up[1]*ofsU;
			desiredPos[2] += up[2]*ofsU;
			desiredPos[0] += fwd[0]*ofsF;
			desiredPos[1] += fwd[1]*ofsF;
			desiredPos[2] += fwd[2]*ofsF;
			//Util_DebugLine(boneMat.o, desiredPos);
			float maxChainDist = 400.0f;
			float d[3];
			Math_VecSub(obj->net.pos, boneMat.o, d);
			if (Math_VecLen(d) > maxChainDist)
			{ //end of the chain
				Math_VecNorm(d);
				Math_VecMA(boneMat.o, maxChainDist, d, obj->net.pos);
			}

			if (obj->debounce < g_gameTime && owner->debounce3 < g_gameTime)
			{
				/*
				Math_VecSub(desiredPos, obj->net.pos, obj->net.vel);
				obj->net.vel[0] *= 2.0f;
				obj->net.vel[1] *= 2.0f;
				obj->net.vel[2] *= 2.0f;
				*/
				Math_VecSub(desiredPos, obj->net.pos, d);
				Math_VecAdd(obj->net.vel, d, obj->net.vel);
			}

			float grav = (owner->debounce3 >= g_gameTime) ? 80.0f : 0.0f;
			Phys_ApplyObjectPhysics(obj, timeMod, obj->net.maxs[0], grav, 0.5f);

			Math_VecSub(obj->net.pos, boneMat.o, d);
			if (Math_VecLen(d) > maxChainDist)
			{ //end of the chain
				Math_VecNorm(d);
				Math_VecMA(boneMat.o, maxChainDist, d, obj->net.pos);
			}
		}
#endif
		Math_VecCopy(obj->net.pos, owner->net.boltOffsets);
		owner->net.boltOffsets[2] += 32.0f;
		owner->net.effectLen = (obj->net.maxs[0] > 100.0f) ? 2.0f : 1.0f;
		owner->net.renderEffects2 |= FXFL2_IKBALL;
		//owner->net.renderEffects2 &= ~FXFL2_IKBALL;
	}
	else
	{
		if (owner && owner->chain == obj)
		{
			owner->chain = NULL;
			obj->chain = NULL;
		}
		obj->think = ObjGeneral_RemoveThink;
	}

	/*
	float bmins[3], bmaxs[3], clr[3];
	Math_VecAdd(obj->net.pos, obj->net.mins, bmins);
	Math_VecAdd(obj->net.pos, obj->net.maxs, bmaxs);
	clr[0] = 0.0f;
	clr[1] = 1.0f;
	clr[2] = 0.0f;
	Util_DebugBox(bmins, bmaxs, clr);
	*/
}

//spawn ball object
void ObjMadBall_Spawn(gameObject_t *obj, BYTE *b, const objArgs_t *args, int numArgs)
{
	obj->think = ObjMadBall_Think;
	obj->touch = ObjMadBall_Touch;
}
