/*
=============================================================================
Module Information
------------------
Name:			obj_mover.cpp
Author:			Rich Whitehouse
Description:	things that rotate/translate and push other objects
=============================================================================
*/

#include "main.h"

#define MAX_DST_MATS	16

typedef struct moverInfo_s
{
	float			ivlAtDst;
	float			ivlAtSrc;
	float			moveSpeed;
	bool			hitNonSolids;
	bool			noMurder;
	serverTime_t	murderDelay;
	serverTime_t	stuckTime;

	bool			lastTransSuccess;
	serverTime_t	lastTransTime;

	float			curAmt;
	modelMatrix_t	tmat;
	modelMatrix_t	baseMat;
	modelMatrix_t	invBaseMat;
	modelMatrix_t	dstMats[MAX_DST_MATS];
	int				numDstMats;
	int				curDst;

	serverTime_t	debounce;
} moverInfo_t;

typedef struct movedObject_s
{
	gameObject_t	*obj;
	float			newPos[3];
} movedObject_t;

typedef struct dstOriParm_s
{
	float			pos[3];
	float			ang[3];
} dstOriParm_t;

//orientation to matrix
static void ObjMover_OriToMat(float *pos, float *ang, modelMatrix_t *mat)
{
	Math_AngleVectors(ang, mat->x1, mat->x2, mat->x3);
	Math_VecCopy(pos, mat->o);
	//*mat = g_gameIdent;
	//Math_TranslateMatrix(mat, pos);
	//Math_RotateMatrix(mat, -ang[1], 0.0f, 0.0f, 1.0f);
	//Math_RotateMatrix(mat, ang[0], 0.0f, 1.0f, 0.0f);
	//Math_RotateMatrix(mat, ang[2], 1.0f, 0.0f, 0.0f);
}

//crush something
static void ObjMover_CrushObject(gameObject_t *obj, moverInfo_t *mi, gameObject_t *other)
{
	if (mi->noMurder)
	{
		return;
	}
	if (mi->murderDelay)
	{
		if (!mi->stuckTime || (g_gameTime-mi->stuckTime) < mi->murderDelay)
		{
			return;
		}
	}
	Util_DamageObject(obj, other, 99999);
}

//attempt sweeping transform
bool ObjMover_AttemptTransform(gameObject_t *obj, float timeMod, modelMatrix_t *curMatL, modelMatrix_t *dstMatL,
							   moverInfo_t *mi)
{
	const int moverIterateNum = g_gameObjectSlots; //MAX_NET_CLIENTS
	movedObject_t movedObjects[MAX_GAME_OBJECTS]; //MAX_NET_CLIENTS
	int numMovedObjects = 0;
	float d[3];
	bool canMove = true;

	float oldAng[3], oldPos[3];
	Math_VecCopy(obj->net.ang, oldAng);
	Math_VecCopy(obj->net.pos, oldPos);

	Math_MatToAngles2(obj->net.ang, dstMatL);
	//ObjMover_CompAngles(obj->net.ang, dstMatL);
	Math_VecCopy(dstMatL->o, obj->net.pos);
	LServ_UpdateRClip(obj);

	modelMatrix_t curMat, dstMat;

	//set up world space transform matrices for current and next moves
	float curAng[3];
	Math_MatToAngles2(curAng, curMatL);
	//ObjMover_CompAngles(curAng, curMatL);
	curMat = g_gameIdent;
	Math_TranslateMatrix(&curMat, curMatL->o);
	Math_RotateMatrix(&curMat, -curAng[1], 0.0f, 0.0f, 1.0f);
	Math_RotateMatrix(&curMat, curAng[0], 0.0f, 1.0f, 0.0f);
	Math_RotateMatrix(&curMat, curAng[2], 1.0f, 0.0f, 0.0f);

	dstMat = g_gameIdent;
	Math_TranslateMatrix(&dstMat, obj->net.pos);
	Math_RotateMatrix(&dstMat, -obj->net.ang[1], 0.0f, 0.0f, 1.0f);
	Math_RotateMatrix(&dstMat, obj->net.ang[0], 0.0f, 1.0f, 0.0f);
	Math_RotateMatrix(&dstMat, obj->net.ang[2], 1.0f, 0.0f, 0.0f);

	modelMatrix_t invCurMat, relMoveMat;
	Math_MatrixInverse(&curMat, &invCurMat);
	Math_MatrixMultiply(&invCurMat, &dstMat, &relMoveMat);

	//Steps for moving
	//1.
	//Get offset of other from mover (other's pos transformed by inverse current matrix),
	//transform offset by relative move matrix, then transform the result by the current
	//matrix, to get the offset in world space (newTransPos).
	//2.
	//Trace from newTransPos to otherPos, clipping only against the mover in its new
	//clip position. If there is no collision, continue on, the thing was not moved.
	//3.
	//If there was a collision, store the endpos. Now, trace from otherPos (original
	//object position) to the endpos, hitting everything BUT the mover. If something was
	//hit in that collision, the other thing is going to be crushed by the mover. If not,
	//the trace endpos can be used as the new otherPos.

	if (obj->rcColModel)
	{
		bool hitNonPC = false;
		bool hitNonSolids = false;
		if (mi)
		{
			hitNonSolids = mi->hitNonSolids;
		}
		for (int i = 0; i < moverIterateNum; i++)
		{
			gameObject_t *other = &g_gameObjects[i];
			if (!other->inuse || (!other->net.solid && !hitNonSolids) || !other->rcColModel)
			{
				continue;
			}
			if (!hitNonPC && !other->plObj && !other->aiObj)
			{
				continue;
			}

			float otherPos[3];
			Math_VecCopy(other->net.pos, otherPos);

			float relPosLocal[3], newTransPosLocal[3], newTransPos[3];
			float exactTrans[3];
			Math_TransformPointByMatrix(&invCurMat, otherPos, relPosLocal);
			Math_TransformPointByMatrix(&relMoveMat, relPosLocal, newTransPosLocal);
			Math_TransformPointByMatrix(&curMat, newTransPosLocal, newTransPos);

			Math_VecCopy(newTransPos, exactTrans);
			//fudge a bit
			Math_VecSub(newTransPos, otherPos, d);
			d[0] *= 1.5f;
			d[1] *= 1.5f;
			d[2] *= 1.5f;
			Math_VecAdd(otherPos, d, newTransPos);

			//Util_DebugLine(otherPos, newTransPos);

			collObj_t col;
			g_sharedFn->Coll_MovementTranslation(other->rcColModel, &col, newTransPos, otherPos, obj->rcColModel, NULL);
			if (!col.hit)
			{ //clear to move without hitting the thing
				g_sharedFn->Coll_MovementTranslation(other->rcColModel, &col, otherPos, otherPos, obj->rcColModel, NULL);
				if (!col.hit && !col.containsSolid)
				{
					if (other->onGround && other->groundObj == obj->net.index)
					{ //move with the mover
						g_sharedFn->Coll_MovementTranslation(other->rcColModel, &col, otherPos, exactTrans, NULL, NULL);
						if (!col.containsSolid)
						{
							if (col.hit)
							{
								collObj_t solCol;
								g_sharedFn->Coll_MovementTranslation(other->rcColModel, &solCol, col.endPos, col.endPos, NULL, NULL);
								if (solCol.containsSolid)
								{
									int fixAttempts = 0;
									while (solCol.hit && solCol.containsSolid && fixAttempts < 4)
									{
										col.endPos[0] += col.endNormal[0]*16.0f;
										col.endPos[1] += col.endNormal[1]*16.0f;
										col.endPos[2] += col.endNormal[2]*16.0f;
										g_sharedFn->Coll_MovementTranslation(other->rcColModel, &solCol, col.endPos, col.endPos, NULL, NULL);
										fixAttempts++;
									}
									if (solCol.containsSolid)
									{
										col.containsSolid = true;
									}
								}
							}
						}

						if (!col.containsSolid)
						{ //change original point of movement
							movedObject_t *mo = &movedObjects[numMovedObjects];
							numMovedObjects++;
							Math_VecCopy(col.endPos, mo->newPos);
							mo->obj = other;
						}
					}
					continue;
				}
				else
				{ //error in offset translation, try to correct
					float newOfs[3];
					Math_VecSub(newTransPos, otherPos, newOfs);
					Math_VecScale(newOfs, 4.0f);
					Math_VecAdd(otherPos, newOfs, newTransPos);
					g_sharedFn->Coll_MovementTranslation(other->rcColModel, &col, newTransPos, otherPos, obj->rcColModel, NULL);
					if (!col.hit || col.containsSolid)
					{
						ObjMover_CrushObject(obj, mi, other);
						canMove = false;
						break;
					}
				}
			}

			//push in the direction the mover wants it to go
			if (fabsf(d[0])+fabsf(d[1]) > d[2]*2.0f)
			{
				float dfac = Math_VecNorm(d);
				if (dfac > 128.0f)
				{
					dfac = 128.0f;
				}
				Math_VecScale(d, dfac);
				other->net.vel[0] += d[0]*0.5f;
				other->net.vel[1] += d[1]*0.5f;
				other->net.vel[2] += d[2]*0.5f;
			}

			if (col.containsSolid)
			{
				ObjMover_CrushObject(obj, mi, other);
				canMove = false;
				break;
			}

			//hack, make the mover non-solid for the next translation
			int wasSolid = obj->rcColModel->solid;
			obj->rcColModel->solid = 0;
			collObj_t moveCol;
			g_sharedFn->Coll_MovementTranslation(other->rcColModel, &moveCol, otherPos, col.endPos, NULL, NULL);
			obj->rcColModel->solid = wasSolid;

			if (!moveCol.hit)
			{ //can be pushed
				g_sharedFn->Coll_MovementTranslation(other->rcColModel, &moveCol, col.endPos, col.endPos, NULL, NULL);
				int fixAttempts = 0;
				while (moveCol.hit && moveCol.containsSolid && fixAttempts < 4)
				{
					col.endPos[0] += col.endNormal[0]*16.0f;
					col.endPos[1] += col.endNormal[1]*16.0f;
					col.endPos[2] += col.endNormal[2]*16.0f;
					g_sharedFn->Coll_MovementTranslation(other->rcColModel, &moveCol, col.endPos, col.endPos, NULL, NULL);
					fixAttempts++;
				}

				if (!moveCol.hit && !moveCol.containsSolid)
				{
					movedObject_t *mo = &movedObjects[numMovedObjects];
					numMovedObjects++;
					Math_VecCopy(col.endPos, mo->newPos);
					mo->obj = other;
				}
				else
				{
					ObjMover_CrushObject(obj, mi, other);
					canMove = false;
					break;
				}
			}
			else
			{ //the thing is being crushed
				ObjMover_CrushObject(obj, mi, other);
				canMove = false;
				break;
			}
		}
	}

	if (canMove)
	{
		for (int i = 0; i < numMovedObjects; i++)
		{
			Math_VecCopy(movedObjects[i].newPos, movedObjects[i].obj->net.pos);
			LServ_UpdateRClip(movedObjects[i].obj);
		}
		return true;
	}

	Math_VecCopy(oldAng, obj->net.ang);
	Math_VecCopy(oldPos, obj->net.pos);
	LServ_UpdateRClip(obj);
	return false;
}

//transform
bool ObjMover_Transform(gameObject_t *obj, float timeMod, modelMatrix_t *baseMat, moverInfo_t *recState, moverInfo_t *oldMI)
{
	if (!(obj->localFlags & LFL_MOVER))
	{
		return true;
	}

	moverInfo_t *mi = (moverInfo_t *)obj->customDynamic;
	RCUBE_ASSERT(mi);
	if (mi->lastTransTime == g_gameTime)
	{ //already transformed
		return mi->lastTransSuccess;
	}

	if (recState)
	{
		*recState = *mi;
	}

	bool success = true;
	int preDst = mi->curDst;
	float preAmt = mi->curAmt;
	if (mi->debounce < g_gameTime)
	{
		mi->curAmt += mi->moveSpeed*timeMod;
		if (mi->curAmt >= 1.0f)
		{
			mi->curDst++;
			if (mi->curDst >= mi->numDstMats)
			{
				mi->curDst = 0;
			}
			mi->curAmt = 0.0f;
			float debTime = (mi->curDst == 1) ? mi->ivlAtSrc : mi->ivlAtDst;
			mi->debounce = g_gameTime+debTime;
		}
	}

	modelMatrix_t moveMat;
	if (mi->numDstMats <= 1)
	{
		moveMat = g_gameIdent;
	}
	else
	{
		int prevDst = (mi->curDst-1);
		if (prevDst < 0)
		{
			prevDst = mi->numDstMats-1;
		}
		Math_LerpMatrices(mi->dstMats[mi->curDst], mi->dstMats[prevDst], mi->curAmt, moveMat, false);
	}

	//put back in non-local space
	modelMatrix_t t = moveMat;
	Math_MatrixMultiply(baseMat, &t, &moveMat);

	if (ObjMover_AttemptTransform(obj, timeMod, &mi->tmat, &moveMat, mi))
	{
		int i;
		for (i = 0; i < obj->numPhysChildren; i++)
		{
			gameObject_t *other = obj->physChildren[i];
			if (!(other->localFlags & LFL_MOVER))
			{
				continue;
			}
			moverInfo_t *otherMI = (moverInfo_t *)other->customDynamic;
			modelMatrix_t childMat, relBaseMat;
			//could precalc relBaseMat
			Math_MatrixMultiply(&mi->invBaseMat, &otherMI->baseMat, &relBaseMat);
			Math_MatrixMultiply(&mi->tmat, &relBaseMat, &childMat);
			if (!ObjMover_Transform(other, timeMod, &childMat, &oldMI[i], oldMI))
			{
				break;
			}
		}
		if (i != obj->numPhysChildren)
		{ //child failed to move
			int failedIdx = i+1;
			for (i = 0; i < failedIdx; i++)
			{
				gameObject_t *other = obj->physChildren[i];
				if (!(other->localFlags & LFL_MOVER))
				{
					continue;
				}
				moverInfo_t *otherMI = (moverInfo_t *)other->customDynamic;
				//*otherMI = oldMI[i];
				otherMI->curDst = oldMI[i].curDst;
				otherMI->curAmt = oldMI[i].curAmt;
				otherMI->tmat = oldMI[i].tmat;

				Math_MatToAngles2(other->net.ang, &otherMI->tmat);
				Math_VecCopy(otherMI->tmat.o, other->net.pos);
				LServ_UpdateRClip(other);
			}
			success = false;
			mi->curDst = preDst;
			mi->curAmt = preAmt;
		}
		else
		{
			mi->tmat = moveMat;
		}
	}
	else
	{
		success = false;
		mi->curDst = preDst;
		mi->curAmt = preAmt;
	}

	Math_MatToAngles2(obj->net.ang, &mi->tmat);
	//ObjMover_CompAngles(obj->net.ang, &mi->tmat);
	Math_VecCopy(mi->tmat.o, obj->net.pos);
	LServ_UpdateRClip(obj);

	if (success)
	{
		mi->stuckTime = 0;
	}
	else if (!mi->stuckTime)
	{
		mi->stuckTime = g_gameTime;
	}

	mi->lastTransTime = g_gameTime;
	mi->lastTransSuccess = success;

	return mi->lastTransSuccess;
}


//think
void ObjMover_Think(gameObject_t *obj, float timeMod)
{
	if (!obj->physParent)
	{
		moverInfo_t *mi = (moverInfo_t *)obj->customDynamic;
		RCUBE_ASSERT(mi);
		moverInfo_t oldMI[MAX_PHYSICS_CHILDREN];
		ObjMover_Transform(obj, timeMod, &mi->baseMat, NULL, oldMI);
	}
	obj->thinkTime = g_gameTime;
}

//mover spawn
void ObjMover_Spawn(gameObject_t *obj, BYTE *b, const objArgs_t *args, int numArgs)
{
	obj->customDynamic = g_sharedFn->Common_RCMalloc(sizeof(moverInfo_t));
	moverInfo_t *mi = (moverInfo_t *)obj->customDynamic;
	memset(mi, 0, sizeof(moverInfo_t));

	mi->murderDelay = 500;

	dstOriParm_t dstOri[MAX_DST_MATS];
	int numDst = 1;
	for (int i = 0; i < MAX_DST_MATS; i++)
	{
		Math_VecCopy(obj->net.ang, dstOri[i].ang);
		Math_VecCopy(obj->net.pos, dstOri[i].pos);
	}
	for (int i = 0; i < numArgs; i++)
	{
		const objArgs_t *arg = args+i;
		if (!strnicmp(arg->key, "dstAng", strlen("dstAng")) ||
			!strnicmp(arg->key, "dstPosOfs", strlen("dstPosOfs")))
		{
			for (int j = 0; j < MAX_DST_MATS; j++)
			{
				char angStr[256];
				char posStr[256];
				if (j == 0)
				{
					strcpy(angStr, "dstAng");
					strcpy(posStr, "dstPosOfs");
				}
				else
				{
					sprintf(angStr, "dstAng%i", j+1);
					sprintf(posStr, "dstPosOfs%i", j+1);
				}
				if (!stricmp(arg->key, angStr))
				{
					sscanf(arg->val, "(%f %f %f)", &dstOri[j].ang[0], &dstOri[j].ang[1], &dstOri[j].ang[2]);
					if (j >= numDst)
					{
						numDst = j+1;
					}
				}
				else if (!stricmp(arg->key, posStr))
				{
					sscanf(arg->val, "(%f %f %f)", &dstOri[j].pos[0], &dstOri[j].pos[1], &dstOri[j].pos[2]);
					Math_VecAdd(dstOri[j].pos, obj->net.pos, dstOri[j].pos);
					if (j >= numDst)
					{
						numDst = j+1;
					}
				}
			}
		}
		else if (!stricmp(arg->key, "moveSpeed"))
		{
			mi->moveSpeed = (float)atof(arg->val);
		}
		else if (!stricmp(arg->key, "debounce"))
		{
			mi->ivlAtDst = (float)atof(arg->val);
			mi->ivlAtSrc = mi->ivlAtDst;
		}
		else if (!stricmp(arg->key, "debouncedst"))
		{
			mi->ivlAtDst = (float)atof(arg->val);
		}
		else if (!stricmp(arg->key, "debouncesrc"))
		{
			mi->ivlAtSrc = (float)atof(arg->val);
		}
		else if (!stricmp(arg->key, "noMurder"))
		{
			mi->noMurder = !!atoi(arg->val);
		}
		else if (!stricmp(arg->key, "murderDelay"))
		{
			mi->murderDelay = (float)atof(arg->val);
		}
	}

	mi->debounce = g_gameTime+mi->ivlAtSrc;

	ObjMover_OriToMat(obj->net.pos, obj->net.ang, &mi->baseMat);
	Math_MatrixInverse(&mi->baseMat, &mi->invBaseMat);

	if (numDst == 1)
	{ //back-and-forth default
		mi->dstMats[0] = mi->baseMat;
		ObjMover_OriToMat(dstOri[0].pos, dstOri[0].ang, &mi->dstMats[1]);

		mi->numDstMats = 2;
	}
	else
	{ //discreet angle specs
		for (int i = 0; i < numDst; i++)
		{
			ObjMover_OriToMat(dstOri[i].pos, dstOri[i].ang, &mi->dstMats[i]);
		}
		mi->numDstMats = numDst;
	}

	for (int i = 0; i < mi->numDstMats; i++)
	{
		modelMatrix_t t = mi->dstMats[i];
		Math_MatrixMultiply(&mi->invBaseMat, &t, &mi->dstMats[i]);
	}

	mi->tmat = mi->baseMat;

	obj->localFlags |= LFL_MOVER;
	obj->think = ObjMover_Think;
	obj->thinkTime = g_gameTime+50;
}
