/*
=============================================================================
Module Information
------------------
Name:			obj_sephfin.cpp
Author:			Rich Whitehouse
Description:	server logic object: sephiroth (final battle form)
=============================================================================
*/

#include "main.h"
#include "ai.h"

//just to make things a bit more readable
#define debounceNoBlock			debounce
#define debounceLungeTime		debounce2
#define debounceLungeDuration	debounce3
#define debounceRunTime			debounce4
#define debounceFastLook		debounce5
#define debounceRandSlash		debounce6
#define debounceSkyAttack		debounce7

typedef enum
{
	SEPHANIM_STRAFE_LEFT = NUM_HUMAN_ANIMS,
	SEPHANIM_STRAFE_RIGHT,
	SEPHANIM_TARG_FWD,
	SEPHANIM_TARG_BACK,
	SEPHANIM_SLASH1,
	SEPHANIM_QUICKSLASH,
	SEPHANIM_JUMPUP,
	SEPHANIM_COMEDOWN,
	SEPHANIM_DOWNMISS,
	SEPHANIM_BEATDOWN,
	SEPHANIM_BEATDOWNLOOP,
	SEPHANIM_PREPAREFORDOOM,
	SEPHANIM_DOWNHIT,
	SEPHANIM_BLOCK,
	SEPHANIM_DODGE_R,
	SEPHANIM_DODGE_L,
	NUM_SEPH_ANIMS
} sephAnims_e;

static gameAnim_t g_sephAnims[NUM_SEPH_ANIMS] =
{
	{1, 30, 68.0f, true},			//HUMANIM_IDLE
	{235, 240, 188.0f, true},		//HUMANIM_WALK
	{235, 240, 138.0f, true},		//HUMANIM_RUNSLOW
	{235, 240, 88.0f, true},		//HUMANIM_RUN
	{43, 44, 68.0f, false},			//HUMANIM_JUMP
	{281, 281, 68.0f, false},		//HUMANIM_FALL
	{274, 275, 68.0f, false},		//HUMANIM_LAND
	{62, 67, 68.0f, false},			//HUMANIM_PAIN_HIGH1
	{259, 264, 68.0f, false},		//HUMANIM_PAIN_HIGH2
	{71, 76, 68.0f, false},			//HUMANIM_PAIN_LOW1
	{71, 76, 68.0f, false},			//HUMANIM_PAIN_LOW2
	{267, 268, 68.0f, false},		//HUMANIM_PAIN_AIR
	{265, 268, 68.0f, false},		//HUMANIM_PAIN_POPUP
	{280, 281, 118.0f, false},		//HUMANIM_GETUP_BACK
	{274, 276, 118.0f, false},		//HUMANIM_GETUP_FRONT
	{269, 269, 68.0f, false},		//HUMANIM_FLYBACK
	{269, 269, 68.0f, false},		//HUMANIM_FLYBACK2
	{270, 271, 68.0f, false},		//HUMANIM_HIT_FALLFORWARD
	{272, 272, 118.0f, false},		//HUMANIM_FALL_LAND
	{277, 279, 118.0f, false},		//HUMANIM_POPUP_LAND
	{32, 32, 1568.0f, false},		//HUMANIM_DEATH
	{243, 246, 168.0f, true},		//SEPHANIM_STRAFE_LEFT
	{247, 250, 168.0f, true},		//SEPHANIM_STRAFE_RIGHT
	{251, 254, 168.0f, true},		//SEPHANIM_TARG_FWD
	{255, 258, 168.0f, true},		//SEPHANIM_TARG_BACK
	{46, 49, 118.0f, false},		//SEPHANIM_SLASH1
	{47, 49, 168.0f, false},		//SEPHANIM_QUICKSLASH
	{290, 291, 268.0f, false},		//SEPHANIM_JUMPUP
	{292, 292, 68.0f, false},		//SEPHANIM_COMEDOWN
	{293, 293, 68.0f, false},		//SEPHANIM_DOWNMISS
	{304, 349, 118.0f, false},		//SEPHANIM_BEATDOWN
	{341, 348, 68.0f, false},		//SEPHANIM_BEATDOWNLOOP
	{350, 350, 68.0f, false},		//SEPHANIM_PREPAREFORDOOM
	{294, 303, 218.0f, false},		//SEPHANIM_DOWNHIT
	{282, 285, 118.0f, false},		//SEPHANIM_BLOCK
	{288, 289, 68.0f, false},		//SEPHANIM_DODGE_R
	{286, 287, 68.0f, false}		//SEPHANIM_DODGE_L
};

#define NUM_SEPH_DAMAGE_BONES	1
static damageBone_t g_sephDamageBones[NUM_SEPH_DAMAGE_BONES] =
{
	{"", "", {0.0f, -1.0f, 0.0f}, 0.0f}			//DMGBONE_FIST_R
};

//custom movement animations
bool ObjSephFin_IdleAnim(gameObject_t *obj, float timeMod)
{
	if (!obj->onGround || obj->aiObj->moveSpeed > 150.0f)
	{
		return false;
	}

	if (obj->curAnim >= SEPHANIM_SLASH1)
	{
		return false;
	}

	float velLen = Math_VecLen(obj->net.vel);
	if (velLen <= 5.0f)
	{ //not moving enough
		return false;
	}

	float fwd[3], right[3];
	float nVel[3];
	Math_AngleVectors(obj->net.ang, fwd, right, 0);
	Math_VecCopy(obj->net.vel, nVel);
	Math_VecNorm(nVel);

	float dpfwd = Math_DotProduct(fwd, nVel);
	float dpright = Math_DotProduct(right, nVel);

	if (fabsf(dpright) > fabsf(dpfwd))
	{ //left/right
		if (dpright >= 0.0f)
		{
			AI_StartAnim(obj, SEPHANIM_STRAFE_RIGHT, false);
		}
		else
		{
			AI_StartAnim(obj, SEPHANIM_STRAFE_LEFT, false);
		}
	}
	else
	{ //forward/back
		if (dpfwd >= 0.0f)
		{
			AI_StartAnim(obj, SEPHANIM_TARG_FWD, false);
		}
		else
		{
			AI_StartAnim(obj, SEPHANIM_TARG_BACK, false);
		}
	}

	return true;
}

//returns damage bone or -1 if not in attack
int ObjSephFin_InAttack(gameObject_t *obj, int *dmgBones)
{
	if (obj->curAnim == SEPHANIM_SLASH1 || obj->curAnim == SEPHANIM_QUICKSLASH)
	{
		if (obj->net.frame >= 46 && obj->net.frame <= 49)
		{
			dmgBones[0] = DMGBONE_FIST_R;
			return 1;
		}
	}
	else if (obj->curAnim == SEPHANIM_BLOCK)
	{
		if (obj->net.frame >= 284 && obj->net.frame <= 285)
		{
			dmgBones[0] = DMGBONE_FIST_R;
			return 1;
		}
	}

	return -1;
}

//occupied?
bool ObjSephFin_CanAct(gameObject_t *obj)
{
	if (!obj->onGround)
	{
		return false;
	}
	if (obj->curAnim >= SEPHANIM_SLASH1)
	{
		return false;
	}
	if (AI_NonMovingAnim(obj))
	{
		return false;
	}
	if (obj->aiObj->noMoveTime >= g_gameTime)
	{
		return false;
	}

	if (obj->curAnim >= SEPHANIM_JUMPUP &&
		obj->curAnim <= SEPHANIM_DOWNHIT)
	{
		return false;
	}

	return true;
}

//can i dodge?
bool ObjSephFin_CanDodge(gameObject_t *obj)
{
	if (!ObjSephFin_CanAct(obj))
	{
		if (obj->curAnim != HUMANIM_FALL && obj->curAnim != HUMANIM_GETUP_BACK)
		{
			return false;
		}
	}

	return true;
}

//dodge
bool ObjSephFin_DodgeEnemy(gameObject_t *obj, gameObject_t *enemy)
{
	float d[3];
	Math_VecSub(obj->net.pos, enemy->net.pos, d);
	if (Math_VecLen(d) > 1200.0f)
	{
		return false;
	}

	float fwd[3];
	Math_AngleVectors(obj->net.ang, 0, fwd, 0);
	float nv[3];
	Math_VecCopy(enemy->net.vel, nv);
	Math_VecNorm(nv);
	float dp = Math_DotProduct(fwd, nv) + 0.5f;
	if (dp <= 0.0f || dp >= 1.0f)
	{
		return false;
	}

	if (dp < 0.5f)
	{
		AI_StartAnim(obj, SEPHANIM_DODGE_R, true);
	}
	else
	{
		AI_StartAnim(obj, SEPHANIM_DODGE_L, true);
	}
	ObjSound_Create(obj->net.pos, "assets/sound/cb/lunge.wav", 1.0f, -1);
	obj->debounceRunTime = g_gameTime + Util_LocalDelay(obj, (serverTime_t)(6000+rand()%3000));
	obj->aiObj->obstructTime = 0;
	obj->aiObj->attackTime = 0;
	return true;
}

//update sword
static void ObjSephFin_UpdateSword(gameObject_t *obj, float timeMod)
{
	if (!obj->chain)
	{
		return;
	}

	obj->chain->net.renderEffects2 &= ~(FXFL2_SPAWN);
	obj->chain->net.renderEffects2 &= ~(FXFL2_SPAWN2);

	obj->chain->net.renderEffects |= (obj->net.renderEffects & FXFL_DEATH);
	obj->chain->net.renderEffects2 |= (obj->net.renderEffects & FXFL2_DEATH2);
	obj->chain->net.renderEffects2 |= (obj->net.renderEffects & FXFL2_DEATH3);
	obj->chain->net.renderEffects2 |= (obj->net.renderEffects & FXFL2_SPAWN);
	obj->chain->net.renderEffects2 |= (obj->net.renderEffects & FXFL2_SPAWN2);

	gameObject_t *sword = obj->chain;

	sword->net.frame = 4;
	float ofs[3];
	ofs[0] = 0.0f;
	ofs[1] = 0.0f;
	ofs[2] = -90.0f;

	char *boneName = "b41";
	sword->net.boltOffsets[0] = ofs[0];
	sword->net.boltOffsets[1] = ofs[1];
	sword->net.boltOffsets[2] = ofs[2];
	sword->net.strIndexC = g_sharedFn->Common_ServerString(boneName);
	sword->net.renderEffects |= FXFL_BOLTED1;
	modelMatrix_t mat;
	if (Util_GetObjBolt(obj, boneName, &mat, ofs))
	{
		Math_VecCopy(mat.o, sword->net.pos);
		Math_MatToAngles2(sword->net.ang, &mat);
	}
}

//set sky attack delay
static void ObjSephFin_DelaySkyAttack(gameObject_t *obj)
{
	obj->debounceSkyAttack = g_gameTime+Util_LocalDelay(obj, (serverTime_t)(30000+(rand()%30000)));
}

//think
void ObjSephFin_Think(gameObject_t *obj, float timeMod)
{
	if (!AI_ShouldThink(obj))
	{
		return;
	}

	if (g_timeType == TIMETYPE_TIMEATTACK)
	{
		obj->net.renderEffects |= FXFL_CONTRAST;
		obj->localTimeScale = 4.0f;
	}
	else
	{
		obj->net.renderEffects &= ~FXFL_CONTRAST;
		obj->localTimeScale = 1.0f;
	}

	AI_StandardGoals(obj, timeMod);
	if (obj->aiObj->moveSpeed > 150.0f &&
		(!obj->aiObj->enemy || obj->aiObj->distToEnemy > 1400.0f) && Math_VecLen(obj->net.vel) > 5.0f && obj->onGround)
	{ //look where i'm going
		float ang[3];
		Math_VecToAngles(obj->net.vel, ang);
		obj->aiObj->lookYaw = ang[YAW];
	}
	AI_GenericThink(obj, timeMod);
	AI_GetToGoal(obj, timeMod);

	if (obj->debounceFastLook >= g_gameTime)
	{
		obj->net.ang[YAW] = Math_BlendAngleLinear(obj->net.ang[YAW], obj->aiObj->lookYaw, 45.0f);
	}

	if (obj->curAnim == SEPHANIM_DOWNMISS && obj->hurtable && obj->debounceNoBlock > g_gameTime)
	{
		gameObject_t *other = &g_gameObjects[0];
		if (other->inuse && other->plObj)
		{
			bool hitEnemy = false;
			float d[3];
			d[0] = other->net.pos[0]-obj->net.pos[0];
			d[1] = other->net.pos[1]-obj->net.pos[1];
			d[2] = 0.0f;
			float dist = Math_VecNorm(d);
			if (dist < 500.0f)
			{
				float fwd[3];
				Math_AngleVectors(obj->net.ang, fwd, 0, 0);
				float dp = 1.0f-Math_DotProduct(d, fwd);
				if (dp*180.0f < 30.0f)
				{
					hitEnemy = true;
				}
			}

			if (hitEnemy)
			{
				if (other->health > 0 && !GVar_GetInt("intriggersequence"))
				{ //don't trigger from jump while script says it's busy
					other->plObj->jumpUseObj = obj->net.index;
					other->plObj->canJumpUse = g_gameTime + Util_LocalDelay(other, 500);
					if (other->plObj->jumpUseTime >= g_gameTime)
					{
						int plAnim = ObjPlayer_GetSephAnim(1);
						if (plAnim >= 0)
						{
							AI_StartAnim(obj, SEPHANIM_BEATDOWN, true);
							AI_StartAnim(other, plAnim, true);
							other->animStartTime = obj->animStartTime;
							other->animRestart = obj->animRestart;
							other->animTime = obj->animTime;
						}
						other->plObj->jumpUseTime = 0;
						other->plObj->canJumpUse = 0;
					}
				}
			}
		}
	}

	float f;
	float fwd[3];
	if (obj->curAnim == SEPHANIM_DODGE_R)
	{
		obj->aiObj->noMoveTime = g_gameTime+100;
		f = 900.0f;
		Math_AngleVectors(obj->net.ang, 0, fwd, 0);
		obj->net.vel[0] += fwd[0]*f;
		obj->net.vel[1] += fwd[1]*f;
		obj->net.vel[2] += fwd[2]*f;
	}
	else if (obj->curAnim == SEPHANIM_DODGE_L)
	{
		obj->aiObj->noMoveTime = g_gameTime+100;
		f = 900.0f;
		Math_AngleVectors(obj->net.ang, 0, fwd, 0);
		obj->net.vel[0] -= fwd[0]*f;
		obj->net.vel[1] -= fwd[1]*f;
		obj->net.vel[2] -= fwd[2]*f;
	}

	ObjSephFin_UpdateSword(obj, timeMod);

	dmgBoneLine_t dmgPos[NUM_SEPH_DAMAGE_BONES];
	AI_TransformDamageBones(obj, g_sephDamageBones, dmgPos, NUM_SEPH_DAMAGE_BONES);
	//Util_DebugLine(dmgPos[0].parPos, dmgPos[0].pos);
	int dmgBones[MAX_AI_DAMAGE_BONES];
	int numDmgBones = ObjSephFin_InAttack(obj, dmgBones);
	gameObject_t *sword = NULL;
	if (obj->chain && obj->chain->swordObj)
	{
		sword = obj->chain;
	}

	obj->atkType = ATKTYPE_NORMAL;
	if (obj->curAnim == SEPHANIM_BLOCK || obj->curAnim == SEPHANIM_SLASH1 || obj->curAnim == SEPHANIM_QUICKSLASH)
	{
		obj->atkType = ATKTYPE_KNOCKBACK;
	}

	if (obj->aiObj->hasLastDmg && numDmgBones >= 0)
	{
		if (sword)
		{
			sword->net.renderEffects |= FXFL_BLADETRAIL;
			sword->net.effectLen = -(sword->spareVec[0]+sword->spareVec[1]);
			sword->net.strIndexD = 0;
		}
		int dmg = 300;
		if (obj->curAnim == SEPHANIM_SLASH1)
		{
			dmg = 350;
		}
		for (int i = 0; i < numDmgBones; i++)
		{
			AI_RunDamageBone(obj, &g_sephDamageBones[dmgBones[i]], &obj->aiObj->lastDmg[dmgBones[i]],
				&dmgPos[dmgBones[i]], dmg);
		}
	}
	else if (sword)
	{
		sword->net.renderEffects &= ~FXFL_BLADETRAIL;
	}

	if (obj->curAnim == SEPHANIM_DODGE_R || obj->curAnim == SEPHANIM_DODGE_L ||
		obj->curAnim == SEPHANIM_JUMPUP || obj->curAnim == SEPHANIM_COMEDOWN)
	{
		obj->net.renderEffects |= FXFL_MODELTRAIL;
	}

	if (obj->health > 0)
	{
		obj->hurtable = 1;
	}
	if (obj->curAnim == SEPHANIM_JUMPUP ||
		obj->curAnim == SEPHANIM_COMEDOWN)
	{
		obj->hurtable = 0;
	}

	obj->atkType = ATKTYPE_NORMAL;

	memcpy(obj->aiObj->lastDmg, dmgPos, sizeof(dmgPos));
	obj->aiObj->hasLastDmg = true;

	obj->aiObj->combatRange = 1000.0f;
	obj->aiObj->moveSpeed = 150.0f;
	if (obj->aiObj->enemy && (obj->aiObj->distToEnemy > 2400.0f || obj->debounceRunTime > g_gameTime || ObjPlayer_InVulnerableState(obj->aiObj->enemy)))
	{
		if (obj->aiObj->obstructTime)
		{
			obj->aiObj->obstructTime = 0;
			obj->aiObj->pathCheckTime = 0;
			Math_VecCopy(obj->aiObj->enemy->net.pos, obj->aiObj->goalPos);
			if (obj->aiObj->path)
			{
				AI_FreePath(obj->aiObj->path);
				obj->aiObj->path = NULL;
				obj->aiObj->pathNode = NULL;
			}
		}
		obj->aiObj->moveSpeed = 450.0f;
		if (obj->debounceRunTime < g_gameTime)
		{
			obj->debounceRunTime = g_gameTime+2000;
		}
		if (obj->aiObj->enemy && obj->aiObj->enemy->plObj &&
			g_ai.playerLastHurtTime < g_gameTime &&
			(g_gameTime-g_ai.playerLastHurtTime) > 10000)
		{
			obj->aiObj->moveSpeed = 550.0f;
		}
	}
	if (!AI_NonAnglingAnim(obj) && obj->aiObj->enemy && AI_FacingEnemy(obj, obj->aiObj->enemy))
	{
		if (obj->aiObj->enemy->net.index < MAX_NET_CLIENTS &&
			ObjSephFin_CanDodge(obj) &&
			ObjPlayer_InDodgeableAttack(obj->aiObj->enemy) &&
			ObjSephFin_DodgeEnemy(obj, obj->aiObj->enemy))
		{ //dodge
			obj->aiObj->noMoveTime = g_gameTime+100;
		}
		else
		{
			bool shouldCloseIn = true;
			if (obj->aiObj->moveSpeed <= 150.0f)
			{
				obj->aiObj->obstructTime = g_gameTime+500;
			}
			if (obj->aiObj->enemy && obj->debounceSkyAttack < g_gameTime && ObjSephFin_CanAct(obj))
			{
				AI_StartAnim(obj, SEPHANIM_JUMPUP, true);
				obj->hurtable = 0;
				ObjSephFin_DelaySkyAttack(obj);
			}
			else if (obj->aiObj->enemy && obj->aiObj->distToEnemy < 400.0f && ObjSephFin_CanAct(obj) &&
				obj->debounceNoBlock < g_gameTime)
			{
				if (!obj->debounceRandSlash)
				{
					obj->debounceRandSlash = g_gameTime + Util_LocalDelay(obj, (serverTime_t)(rand()%400));
				}
				else if (obj->debounceRandSlash < g_gameTime)
				{
					obj->debounceRandSlash = 0;
					obj->net.ang[YAW] = Math_BlendAngleLinear(obj->net.ang[YAW], obj->aiObj->lookYaw, 90.0f);
					AI_StartAnim(obj, SEPHANIM_QUICKSLASH, true);
					obj->aiObj->attackTime = g_gameTime + Util_LocalDelay(obj, (serverTime_t)(1500+rand()%2000));
					obj->aiObj->noMoveTime = g_gameTime + Util_LocalDelay(obj, 400);
				}
			}
			else if (obj->aiObj->attackTime < g_gameTime && shouldCloseIn && ObjSephFin_CanAct(obj))
			{
				if ((g_gameTime-obj->aiObj->attackTime) > 10000)
				{ //get sick of dancing and run in
					obj->debounceRunTime = g_gameTime + Util_LocalDelay(obj, (serverTime_t)(6000+rand()%3000));
				}
				obj->aiObj->combatRange = 128.0f;
				float closeAttackDist = (obj->aiObj->moveSpeed < 500.0f) ? 600.0f : 400.0f;
				if (obj->aiObj->distToEnemy < closeAttackDist)
				{
					if (!obj->debounceRandSlash && obj->debounceRunTime < g_gameTime)
					{
						obj->debounceRandSlash = g_gameTime + Util_LocalDelay(obj, (serverTime_t)(rand()%400));
					}
					else if (obj->debounceRandSlash < g_gameTime || obj->debounceRunTime < g_gameTime)
					{
						obj->debounceRunTime = 0;
						obj->debounceRandSlash = 0;
						obj->debounceNoBlock = g_gameTime + 2000;
						if (obj->aiObj->moveSpeed >= 500.0f)
						{
							AI_StartAnim(obj, SEPHANIM_QUICKSLASH, true);
							obj->aiObj->attackTime = g_gameTime + Util_LocalDelay(obj, (serverTime_t)(1500+rand()%2000));
						}
						else
						{
							AI_StartAnim(obj, SEPHANIM_SLASH1, true);
							obj->aiObj->attackTime = g_gameTime + Util_LocalDelay(obj, (serverTime_t)(1500+rand()%2000));
							obj->aiObj->noMoveTime = g_gameTime + Util_LocalDelay(obj, 400);
						}
					}
				}
				else if (obj->debounceLungeTime < g_gameTime &&
					obj->aiObj->distToEnemy < 2000.0f &&
					AI_CanReachPoint(obj, obj->aiObj->goalPos, obj->aiObj->enemy))
				{
					/*
					obj->net.ang[YAW] = Math_BlendAngleLinear(obj->net.ang[YAW], obj->aiObj->lookYaw, 90.0f);
					AI_StartAnim(obj, ObjE30_PickLungeAnim(obj), true);
					obj->aiObj->attackTime = g_gameTime + Util_LocalDelay(obj, (serverTime_t)(2000+rand()%4000));
					obj->aiObj->noMoveTime = g_gameTime + Util_LocalDelay(obj, 1000);
					obj->debounceLungeDuration = g_gameTime + Util_LocalDelay(obj, 800);
					obj->debounceLungeTime = g_gameTime + Util_LocalDelay(obj, (serverTime_t)(2000+rand()%8000));
					*/
				}
				else if (obj->aiObj->distToEnemy > 2400.0f && obj->debounceRunTime < g_gameTime)
				{
					obj->debounceRunTime = g_gameTime + Util_LocalDelay(obj, (serverTime_t)(6000+rand()%3000));
				}
			}
			else
			{
				obj->aiObj->obstructTime = g_gameTime+500;
				float closeDist = (obj->generalFlag <= 0) ? 200.0f : 100.0f;
				float r = (shouldCloseIn) ? closeDist : 500.0f;
				obj->aiObj->combatRange = g_ai.playerCloseDist+r;
			}
		}
	}
}

//position for air attack
void ObjSephFin_AirPosition(gameObject_t *obj, float timeMod)
{
	gameObject_t *enemy = obj->aiObj->enemy;
	if (!enemy)
	{
		return;
	}

	float fwd[3];
	Math_AngleVectors(enemy->net.ang, fwd, 0, 0);
	float desPos[3];
	Math_VecMA(enemy->net.pos, -300.0f, fwd, desPos);

	float d[3];
	Math_VecSub(desPos, obj->net.pos, d);
	obj->net.vel[0] = d[0];
	obj->net.vel[1] = d[1];

	obj->aiObj->lookYaw = enemy->net.ang[YAW];
	obj->net.ang[YAW] = enemy->net.ang[YAW];
}

//distance from pos to ground for object
static float ObjSephFin_GroundDistance(gameObject_t *obj, float *pos, float *groundPos)
{
	collObj_t col;
	const float maxGroundDist = 32768.0f;
	float d[3] = {pos[0], pos[1], pos[2]-maxGroundDist};
	g_sharedFn->Coll_MovementTranslation(obj->rcColModel, &col, pos, d, NULL, NULL);
	if (col.containsSolid)
	{
		return 0.0f;
	}
	if (!col.hit)
	{
		if (groundPos)
		{
			Math_VecCopy(d, groundPos);
		}
		return maxGroundDist;
	}

	if (col.distance < 64.0f && col.hitObjectIndex >= 0 && col.hitObjectIndex < g_gameObjectSlots)
	{
		gameObject_t *other = &g_gameObjects[col.hitObjectIndex];
		if (other->inuse && other->hurtable)
		{
			int atkType = obj->atkType;
			obj->atkType = ATKTYPE_KNOCKBACK;
			char *s[3] =
			{
				"assets/sound/cb/buscut01.wav",
				"assets/sound/cb/buscut02.wav",
				"assets/sound/cb/buscut03.wav"
			};
			if (other->plObj)
			{
				float ang[3];
				Math_VecToAngles(col.endNormal, ang);
				ObjParticles_Create("melee/bleed", col.endPos, ang, -1);
				ObjParticles_Create("melee/bleed", col.endPos, ang, -1);
			}
			ObjSound_Create(obj->net.pos, s[rand()%3], 1.0f, -1);
			Util_DamageObject(obj, other, 500, &col);
			obj->atkType = atkType;
			return 1000.0f;
		}
	}

	if (groundPos)
	{
		Math_VecCopy(col.endPos, groundPos);
	}
	return col.distance;
}

//getting beaten in the face
void ObjSephFin_BeatdownBlow(gameObject_t *obj)
{
	if (obj->aiObj->enemy && obj->aiObj->enemy->inuse && obj->aiObj->enemy->plObj)
	{
		ObjPlayer_ChargeUp(obj->aiObj->enemy, obj, 200);
	}
	modelMatrix_t boneMat;
	float boneOfs[3] = {0.0f, 0.0f, 0.0f};
	Util_GetObjBolt(obj, "b2", &boneMat, boneOfs);
	float ang[3];
	float fwd[3];
	Math_AngleVectors(obj->net.ang, fwd, 0, 0);
	boneMat.o[0] += fwd[0]*32.0f;
	boneMat.o[1] += fwd[1]*32.0f;
	boneMat.o[2] += fwd[2]*32.0f;
	Math_VecToAngles(boneMat.x2, ang);
	ObjParticles_Create("melee/impact", boneMat.o, ang, -1);
	ang[0] = Util_RandFloat(0.0, 360.0f);
	ang[1] = Util_RandFloat(0.0, 360.0f);
	ang[2] = 0.0f;
	ObjParticles_Create("melee/bleedless", boneMat.o, ang, -1);

	if (rand()%10 < 6)
	{
		char *s[3] =
		{
			"assets/sound/cb/crack01.wav",
			"assets/sound/cb/crack02.wav",
			"assets/sound/cb/crack03.wav"
		};
		ObjSound_CreateFromIndex(obj->net.pos, g_soundHitHeavy[rand()%NUM_SOUNDS_HITHEAVY], 1.0f, -1);
		ObjSound_Create(obj->net.pos, s[rand()%3], 1.0f, -1);
		Util_ProximityShake(obj->net.pos, 1200.0f, 2.0f, 0.25f);
	}
	else
	{
		ObjSound_CreateFromIndex(obj->net.pos, g_soundHitLight[rand()%NUM_SOUNDS_HITLIGHT], 1.0f, -1);
	}
}

//pick which anim to be in
void ObjSephFin_PickAnim(gameObject_t *obj, float timeMod)
{
	gameAnim_t *curAnim = obj->animTable+obj->curAnim;

	if (obj->curAnim >= SEPHANIM_STRAFE_LEFT && obj->curAnim <= SEPHANIM_TARG_BACK)
	{
		if (!ObjSephFin_IdleAnim(obj, timeMod))
		{
			AI_StartAnim(obj, HUMANIM_IDLE, true);
		}
	}
	else if (obj->curAnim == SEPHANIM_JUMPUP)
	{
		ObjSephFin_DelaySkyAttack(obj);

		float groundDist = ObjSephFin_GroundDistance(obj, obj->net.pos, NULL);
		if (groundDist > 8000.0f)
		{
			ObjSephFin_AirPosition(obj, timeMod);
		}
		if (obj->net.frame == curAnim->endFrame && groundDist < 20000.0f)
		{
			obj->net.vel[2] = 3000.0f;
		}
		else if (obj->net.frame == curAnim->endFrame && !obj->animRestart)
		{
			AI_StartAnim(obj, SEPHANIM_COMEDOWN, true);
		}
	}
	else if (obj->curAnim == SEPHANIM_COMEDOWN)
	{
		ObjSephFin_DelaySkyAttack(obj);

		float groundDist = ObjSephFin_GroundDistance(obj, obj->net.pos, NULL);
		if (groundDist > 1000.0f)
		{
			ObjSephFin_AirPosition(obj, timeMod);
		}
		else if (groundDist <= 32.0f)
		{
			bool hitEnemy = false;
			gameObject_t *enemy = obj->aiObj->enemy;
			if (enemy && enemy->hurtable)
			{
				float d[3];
				d[0] = enemy->net.pos[0]-obj->net.pos[0];
				d[1] = enemy->net.pos[1]-obj->net.pos[1];
				d[2] = 0.0f;
				float dist = Math_VecNorm(d);
				if (dist < 500.0f)
				{
					float fwd[3];
					Math_AngleVectors(obj->net.ang, fwd, 0, 0);
					float dp = 1.0f-Math_DotProduct(d, fwd);
					if (dp*180.0f < 30.0f)
					{
						hitEnemy = true;
					}
				}
			}

			if (hitEnemy)
			{
				if (enemy->plObj)
				{
					int plAnim = ObjPlayer_GetSephAnim(0);
					if (plAnim >= 0)
					{
						int *t = (int *)_alloca(sizeof(int)+sizeof(float)*3);
						*t = 600;
						float *c = (float *)(t+1);
						c[0] = 1.0f;
						c[1] = 0.0f;
						c[2] = 0.0f;
						g_sharedFn->Net_SendEventType(CLEV_TRIGGERFLASH, t, sizeof(int)+sizeof(float)*3, -1);
						modelMatrix_t boneMat;
						float boneOfs[3] = {0.0f, 0.0f, 80.0f};
						if (Util_GetObjBolt(enemy, "b2", &boneMat, boneOfs))
						{
							float ang[3];
							Math_VecToAngles(boneMat.x2, ang);
							ObjParticles_Create("melee/bleed", boneMat.o, ang, -1);
							ObjParticles_Create("melee/bleed", boneMat.o, ang, -1);
							ObjParticles_Create("melee/bleed", boneMat.o, ang, -1);
						}
						char *s[3] =
						{
							"assets/sound/cb/buscut01.wav",
							"assets/sound/cb/buscut02.wav",
							"assets/sound/cb/buscut03.wav"
						};
						ObjSound_Create(obj->net.pos, s[rand()%3], 1.0f, -1);

						AI_StartAnim(obj, SEPHANIM_DOWNHIT, true);
						AI_StartAnim(enemy, plAnim, true);
						enemy->animStartTime = obj->animStartTime;
						enemy->animRestart = obj->animRestart;
						enemy->animTime = obj->animTime;
					}
					else
					{
						AI_StartAnim(obj, SEPHANIM_DOWNMISS, true);
					}
				}
				else
				{
					char *s[3] =
					{
						"assets/sound/cb/buscut01.wav",
						"assets/sound/cb/buscut02.wav",
						"assets/sound/cb/buscut03.wav"
					};
					ObjSound_Create(obj->net.pos, s[rand()%3], 1.0f, -1);
					Util_DamageObject(obj, enemy, 9999);
					AI_StartAnim(obj, SEPHANIM_DOWNMISS, true);
					obj->debounceNoBlock = g_gameTime+1000;
					obj->aiObj->noMoveTime = g_gameTime+1000;
				}
			}
			else
			{
				ObjSound_Create(obj->net.pos, "assets/sound/cb/swdhit.wav", 1.0f, -1);
				AI_StartAnim(obj, SEPHANIM_DOWNMISS, true);
				obj->debounceNoBlock = g_gameTime+2000;
				obj->aiObj->noMoveTime = g_gameTime+4000;
			}
		}
		else
		{
			obj->net.vel[2] = -2000.0f;
		}
	}
	else if (obj->curAnim == SEPHANIM_DOWNMISS)
	{
		ObjSephFin_DelaySkyAttack(obj);

		if (obj->debounceNoBlock < g_gameTime)
		{
			AI_StartAnim(obj, HUMANIM_IDLE, true);
		}
	}
	else if (obj->curAnim == SEPHANIM_DOWNHIT)
	{
		ObjSephFin_DelaySkyAttack(obj);

		gameObject_t *enemy = obj->aiObj->enemy;
		if (enemy && enemy->inuse && enemy->plObj && obj->net.frame < 300)
		{
			float fwd[3];
			Math_AngleVectors(obj->net.ang, fwd, 0, 0);
			float p[3];
			Math_VecMA(obj->net.pos, 300.0f, fwd, p);
			float d[3];
			Math_VecSub(p, enemy->net.pos, d);
			enemy->net.vel[0] = d[0];
			enemy->net.vel[1] = d[1];
			enemy->net.vel[2] = d[2];
			enemy->net.ang[YAW] = obj->net.ang[YAW];
		}
		obj->aiObj->noMoveTime = g_gameTime+100;
		if (obj->net.frame == curAnim->endFrame && !obj->animRestart)
		{
			AI_StartAnim(obj, HUMANIM_IDLE, true);
		}
	}
	else if (obj->curAnim == SEPHANIM_BEATDOWN)
	{
		ObjSephFin_DelaySkyAttack(obj);

		gameObject_t *enemy = obj->aiObj->enemy;
		if (enemy && enemy->inuse && enemy->plObj && obj->net.frame < 348)
		{
			float fwd[3];
			Math_AngleVectors(obj->net.ang, fwd, 0, 0);
			float p[3];
			Math_VecMA(obj->net.pos, 300.0f, fwd, p);
			float d[3];
			Math_VecSub(p, enemy->net.pos, d);
			enemy->net.vel[0] = d[0];
			enemy->net.vel[1] = d[1];
			enemy->net.vel[2] = d[2];
			enemy->net.ang[YAW] = obj->net.ang[YAW]-180.0f;
		}
		if (!ObjPlayer_InSephAnim(enemy))
		{
			AI_StartAnim(obj, HUMANIM_IDLE, true);
		}
		else
		{
			obj->dualAnimCnt = 10;
			enemy->dualAnimCnt = 10;
			obj->aiObj->noMoveTime = g_gameTime+100;
			if (obj->net.frame == curAnim->endFrame && !obj->animRestart)
			{
				AI_StartAnim(obj, SEPHANIM_BEATDOWNLOOP, true);
			}
		}
	}
	else if (obj->curAnim == SEPHANIM_BEATDOWNLOOP)
	{
		ObjSephFin_DelaySkyAttack(obj);

		gameObject_t *enemy = obj->aiObj->enemy;
		if (!ObjPlayer_InSephAnim(enemy))
		{
			AI_StartAnim(obj, HUMANIM_IDLE, true);
		}
		else
		{
			if (obj->net.frame == curAnim->endFrame && !obj->animRestart)
			{
				if (obj->dualAnimCnt <= 0)
				{
					AI_StartAnim(obj, SEPHANIM_PREPAREFORDOOM, true);
				}
				else
				{
					obj->dualAnimCnt--;
					AI_StartAnim(obj, SEPHANIM_BEATDOWNLOOP, true);
				}
			}
			obj->aiObj->noMoveTime = g_gameTime+100;
		}
	}
	else if (obj->curAnim == SEPHANIM_PREPAREFORDOOM)
	{
		ObjSephFin_DelaySkyAttack(obj);

		gameObject_t *enemy = obj->aiObj->enemy;
		if (!ObjPlayer_InSephAnim(enemy))
		{
			AI_StartAnim(obj, HUMANIM_IDLE, true);
		}
		else
		{
			if (enemy->net.frame == 776)
			{
				modelMatrix_t boneMat;
				float boneOfs[3] = {0.0f, 0.0f, 0.0f};
				Util_GetObjBolt(obj, "b2", &boneMat, boneOfs);
				float ang[3];
				float fwd[3];
				Math_AngleVectors(obj->net.ang, fwd, 0, 0);
				boneMat.o[0] += fwd[0]*32.0f;
				boneMat.o[1] += fwd[1]*32.0f;
				boneMat.o[2] += fwd[2]*32.0f;
				ang[0] = -90.0f;
				ang[1] = 0.0f;
				ang[2] = 0.0f;
				int *t = (int *)_alloca(sizeof(int)+sizeof(float)*3);
				*t = 600;
				float *c = (float *)(t+1);
				c[0] = 1.0f;
				c[1] = 1.0f;
				c[2] = 1.0f;
				g_sharedFn->Net_SendEventType(CLEV_TRIGGERFLASH, t, sizeof(int)+sizeof(float)*3, -1);
				ObjParticles_Create("impacts/meteorimpact", boneMat.o, ang, -1);
				ObjSound_Create(boneMat.o, "assets/sound/cb/meteorsmash.wav", 1.0f, -1);
				ObjSound_Create(boneMat.o, "assets/sound/cb/boom.wav", 1.0f, -1);
				char *s[3] =
				{
					"assets/sound/cb/crack01.wav",
					"assets/sound/cb/crack02.wav",
					"assets/sound/cb/crack03.wav"
				};
				ObjSound_CreateFromIndex(boneMat.o, g_soundHitHeavy[rand()%NUM_SOUNDS_HITHEAVY], 1.0f, -1);
				ObjSound_Create(boneMat.o, s[rand()%3], 1.0f, -1);
				Util_RadiusDamage(enemy, 250, 2600.0f, boneMat.o);
				Util_DamageObject(enemy, obj, 4000);
				if (obj->health > 0)
				{
					gameObject_t *knockee = obj;
					gameObject_t *knocker = enemy;
					AI_StartAnim(knockee, HUMANIM_FLYBACK, true);
					float fwd[3];
					float knockback = 4000.0f;
					float c1[3], c2[3];
					Util_GameObjectCenter(knockee, c1);
					Util_GameObjectCenter(knocker, c2);
					Math_VecSub(c1, c2, fwd);
					Math_VecNorm(fwd);
					knockee->net.vel[0] += fwd[0]*knockback;
					knockee->net.vel[1] += fwd[1]*knockback;
					knockee->net.vel[2] += 800.0f;
					knockee->onGround = false;
					knockee->knockTime = g_gameTime + Util_LocalDelay(knockee, 100);
				}
			}
			obj->aiObj->noMoveTime = g_gameTime+100;
		}
	}
	else if (obj->net.frame == curAnim->endFrame && !obj->animRestart)
	{
		AI_StartAnim(obj, HUMANIM_IDLE, true);
	}
}

//frame tick
void ObjSephFin_FrameTick(gameObject_t *obj, float timeMod, int oldFrame)
{
	switch (obj->curAnim)
	{
	case HUMANIM_WALK:
	case HUMANIM_RUNSLOW:
	case HUMANIM_RUN:
		if ((obj->net.frame == 237 || obj->net.frame == 240) && obj->onGround)
		{
			int snd = g_soundFootsteps[rand()%NUM_SOUNDS_FOOTSTEPS];
			ObjSound_CreateFromIndex(obj->net.pos, snd, 1.0f, -1);
		}
		break;
	case SEPHANIM_STRAFE_LEFT:
		if ((obj->net.frame == 243 || obj->net.frame == 245) && obj->onGround)
		{
			int snd = g_soundFootsteps[rand()%NUM_SOUNDS_FOOTSTEPS];
			ObjSound_CreateFromIndex(obj->net.pos, snd, 1.0f, -1);
		}
		break;
	case SEPHANIM_STRAFE_RIGHT:
		if ((obj->net.frame == 247 || obj->net.frame == 2459) && obj->onGround)
		{
			int snd = g_soundFootsteps[rand()%NUM_SOUNDS_FOOTSTEPS];
			ObjSound_CreateFromIndex(obj->net.pos, snd, 1.0f, -1);
		}
		break;
	case SEPHANIM_TARG_FWD:
		if ((obj->net.frame == 252 || obj->net.frame == 254) && obj->onGround)
		{
			int snd = g_soundFootsteps[rand()%NUM_SOUNDS_FOOTSTEPS];
			ObjSound_CreateFromIndex(obj->net.pos, snd, 1.0f, -1);
		}
		break;
	case SEPHANIM_TARG_BACK:
		if ((obj->net.frame == 255 || obj->net.frame == 257) && obj->onGround)
		{
			int snd = g_soundFootsteps[rand()%NUM_SOUNDS_FOOTSTEPS];
			ObjSound_CreateFromIndex(obj->net.pos, snd, 1.0f, -1);
		}
		break;
	case HUMANIM_GETUP_BACK:
		if (obj->net.frame == 280)
		{
			ObjSound_Create(obj->net.pos, "assets/sound/cb/jump01.wav", 1.0f, -1);
			obj->net.vel[2] = 600.0f;
		}
		break;
	case SEPHANIM_BLOCK:
		if (obj->net.frame == 284)
		{
			ObjSound_CreateFromIndex(obj->net.pos, g_soundDeepBlow[rand()%NUM_SOUNDS_DEEPBLOW], 1.0f, -1);
		}
		break;
	case SEPHANIM_SLASH1:
		if (obj->net.frame == 46)
		{
			ObjSound_CreateFromIndex(obj->net.pos, g_soundDeepBlow[rand()%NUM_SOUNDS_DEEPBLOW], 1.0f, -1);
		}
		break;
	case SEPHANIM_QUICKSLASH:
		if (obj->net.frame == 47)
		{
			ObjSound_CreateFromIndex(obj->net.pos, g_soundDeepBlow[rand()%NUM_SOUNDS_DEEPBLOW], 1.0f, -1);
		}
		break;
	case SEPHANIM_JUMPUP:
		if (obj->net.frame == 291 && oldFrame != obj->net.frame)
		{
			ObjSound_Create(obj->net.pos, "assets/sound/cb/jump01.wav", 1.0f, -1);
		}
		break;
	case SEPHANIM_DOWNHIT:
		if (obj->net.frame == 300)
		{
			gameObject_t *enemy = obj->aiObj->enemy;
			if (enemy && enemy->inuse && enemy->hurtable && enemy->plObj)
			{
				int *t = (int *)_alloca(sizeof(int)+sizeof(float)*3);
				*t = 600;
				float *c = (float *)(t+1);
				c[0] = 1.0f;
				c[1] = 0.0f;
				c[2] = 0.0f;
				g_sharedFn->Net_SendEventType(CLEV_TRIGGERFLASH, t, sizeof(int)+sizeof(float)*3, -1);
				modelMatrix_t boneMat;
				float boneOfs[3] = {0.0f, 0.0f, -64.0f};
				if (Util_GetObjBolt(enemy, "b2", &boneMat, boneOfs))
				{
					float ang[3];
					Math_VecToAngles(boneMat.x2, ang);
					ObjParticles_Create("melee/bleed", boneMat.o, ang, -1);
					ObjParticles_Create("melee/bleed", boneMat.o, ang, -1);
					ObjParticles_Create("melee/bleed", boneMat.o, ang, -1);
				}
				ObjSound_Create(obj->net.pos, "assets/sound/cb/buscut01.wav", 1.0f, -1);
				ObjSound_Create(obj->net.pos, "assets/sound/cb/buscut01.wav", 1.0f, -1);
				enemy->animNeverInterrupt = true;
				Util_DamageObject(obj, enemy, 1500);
				if (enemy->health > 0)
				{
					enemy->animNeverInterrupt = false;
				}
			}
		}
		break;
	case SEPHANIM_BEATDOWN:
	case SEPHANIM_BEATDOWNLOOP:
		if (obj->net.frame == 307 ||
			obj->net.frame == 311 ||
			obj->net.frame == 315 ||
			obj->net.frame == 319 ||
			obj->net.frame == 323 ||
			obj->net.frame == 327 ||
			obj->net.frame == 330 ||
			obj->net.frame == 333 ||
			obj->net.frame == 336 ||
			obj->net.frame == 339 ||
			obj->net.frame == 342 ||
			obj->net.frame == 344 ||
			obj->net.frame == 346 ||
			obj->net.frame == 348)
		{
			ObjSephFin_BeatdownBlow(obj);
		}
		break;
	default:
		break;
	}
}

//touch
void ObjSephFin_Touch(gameObject_t *obj, gameObject_t *other, const collObj_t *col)
{
	AI_GenericTouch(obj, other, col);
}

//block
bool ObjSephFin_AttackBlock(gameObject_s *obj, gameObject_s *hurter, int dmg, collObj_t *col,
							float *impactPos, float *impactAng)
{
	if (obj->debounceNoBlock > g_gameTime)
	{
		return false;
	}

	if (hurter->atkType != ATKTYPE_NORMAL && hurter->atkType != ATKTYPE_BULLET &&
		hurter->atkType != ATKTYPE_KNOCKBACK)
	{
		return false;
	}

	if (!ObjSephFin_CanAct(obj))
	{
		if (obj->curAnim != HUMANIM_GETUP_BACK &&
			obj->curAnim != HUMANIM_GETUP_FRONT &&
			obj->curAnim != SEPHANIM_BLOCK &&
			obj->curAnim != HUMANIM_FALL &&
			obj->curAnim != SEPHANIM_DODGE_R &&
			obj->curAnim != SEPHANIM_DODGE_L)
		{ //make an exception to allow blocking while getting up and in a block
			if (!(obj->generalFlag <= -1 &&
				obj->curAnim >= HUMANIM_PAIN_HIGH1 && obj->curAnim <= HUMANIM_PAIN_LOW2 &&
				rand()%10 < 2))
			{ //small chance of blocking in pain if highly skilled
				return false;
			}
		}
	}

	if (obj->generalFlag <= -1 &&
		obj->curAnim != SEPHANIM_BLOCK &&
		!ObjSephFin_CanAct(obj))
	{ //always block in getups/pain
		obj->net.ang[YAW] = Math_BlendAngleLinear(obj->net.ang[YAW], obj->aiObj->lookYaw, 360.0f);
	}
	else
	{
		float fwd[3];
		Math_AngleVectors(obj->net.ang, fwd, 0, 0);
		float d[3];
		Math_VecSub(impactPos, obj->net.pos, d);
		d[2] = 0.0f;
		Math_VecNorm(d);
		float dp = Math_DotProduct(fwd, d);
		if (dp <= 0.0f)
		{
			return false;
		}
	}

	ObjParticles_Create("melee/impact", impactPos, impactAng, -1);
	ObjSound_Create(impactPos, "assets/sound/cb/hitmetal.wav", 1.0f, -1);
	AI_StartAnim(obj, SEPHANIM_BLOCK, true);
	obj->aiObj->noMoveTime = g_gameTime + Util_LocalDelay(obj, 300);
	return true;
}

//called when inflicting damage
void ObjSephFin_OnHurt(gameObject_s *obj, gameObject_s *victim, int dmg, const collObj_t *col)
{
}

//removal
void ObjSephFin_OnRemove(gameObject_t *obj)
{
	if (obj->chain)
	{
		obj->chain->think = ObjGeneral_RemoveThink;
		obj->chain->thinkTime = g_gameTime;
		obj->chain = NULL;
	}
	if (obj->chain2)
	{
		obj->chain2->chain = NULL;
		obj->chain2->think = ObjGeneral_RemoveThink;
		obj->chain2->thinkTime = g_gameTime;
		obj->chain2 = NULL;
	}
}

//spawn
void ObjSephFin_Spawn(gameObject_t *obj, BYTE *b, const objArgs_t *args, int numArgs)
{
	AI_GenericSpawn(obj, b, args, numArgs);

	obj->aiObj->swordBone = "b41";
	obj->aiObj->swordOfs[0] = 0.0f;
	obj->aiObj->swordOfs[1] = -25.0f;
	obj->aiObj->swordOfs[2] = -40.0f;

	obj->touch = ObjSephFin_Touch;
	obj->animframetick = ObjSephFin_FrameTick;
	obj->animidleoverride = ObjSephFin_IdleAnim;
	obj->onhurt = ObjSephFin_OnHurt;
	obj->onremove = ObjSephFin_OnRemove;

	obj->chain = LServ_ObjectFromName("obj_masamune", obj->net.pos, obj->net.ang, 0, 0);
	if (obj->chain)
	{
		gameObject_t *sword = obj->chain;
		sword->net.owner = obj->net.index;
		sword->net.spareVec1[0] = 0.0f;
		sword->net.spareVec1[1] = 25.0f;
		sword->net.spareVec1[2] = 50.0f;
	}
	ObjSephFin_UpdateSword(obj, 1.0f);

	float p[3];
	p[0] = obj->net.pos[0];
	p[1] = obj->net.pos[1];
	p[2] = obj->net.pos[2]+obj->net.maxs[2] + 128.0f;
	obj->chain2 = LServ_ObjectFromName("obj_sephhair", p, obj->net.ang, NULL, 0);
	if (obj->chain2)
	{
		obj->chain2->chain = obj;
		obj->chain2->think(obj, 0.5f);
	}

	const char *elvalue = Common_GetValForKey(b, "elvalue");
	if (elvalue && elvalue[0])
	{
		obj->generalFlag = atoi(elvalue);
	}

	obj->aiObj->makoValue = 500;
	obj->net.aiDescIndex = AIDESC_SEPHFIN;

	obj->aiObj->maxHealth = obj->health;

	static int dmgSounds[3];
	dmgSounds[0] = g_sharedFn->Common_ServerString("$assets/sound/cb/buscut01.wav");
	dmgSounds[1] = g_sharedFn->Common_ServerString("$assets/sound/cb/buscut02.wav");
	dmgSounds[2] = g_sharedFn->Common_ServerString("$assets/sound/cb/buscut03.wav");

	obj->aiObj->dmgEffect = "melee/bigslash";
	obj->aiObj->dmgSounds = dmgSounds;
	obj->aiObj->numDmgSounds = 3;
	obj->aiObj->moveSpeed = 450.0f;
	obj->animhandler = ObjSephFin_PickAnim;
	obj->animTable = g_sephAnims;
	obj->curAnim = HUMANIM_IDLE;
	obj->think = ObjSephFin_Think;
	obj->attackblock = ObjSephFin_AttackBlock;

	ObjSephFin_DelaySkyAttack(obj);
}

//seph hair think
void ObjSephHair_Think(gameObject_t *obj, float timeMod)
{
	gameObject_t *owner = obj->chain;

	if (owner && owner->health > 0)
	{
		obj->localTimeScale = owner->localTimeScale;
		modelMatrix_t boneMat;
		if (owner->rcColModel && g_sharedFn->Coll_GetModelBoneMatrix(owner->rcColModel, "b3", &boneMat))
		{
			float desiredPos[3];
			Math_VecCopy(owner->net.pos, desiredPos);
			float fwd[3], right[3], up[3];
			Math_AngleVectors(owner->net.ang, fwd, right, up);
			desiredPos[2] = boneMat.o[2];
			Math_VecMA(desiredPos, -140.0f+(sinf(g_gameTime*0.001f)*20.0f), fwd, desiredPos);
			Math_VecMA(desiredPos, 140.0f+(cosf(g_gameTime*0.001f)*20.0f), right, desiredPos);
			Math_VecMA(desiredPos, /*500.0f+*/-60.0f+(sinf(g_gameTime*0.001f)*60.0f), up, desiredPos);

			//Util_DebugLine(boneMat.o, desiredPos);
			float maxChainDist = 300.0f;//400.0f;
			float d[3];
			Math_VecSub(obj->net.pos, boneMat.o, d);
			if (Math_VecLen(d) > maxChainDist)
			{ //end of the chain
				Math_VecNorm(d);
				Math_VecMA(boneMat.o, maxChainDist, d, obj->net.pos);
			}

			Math_VecSub(desiredPos, obj->net.pos, d);
			d[0] *= 0.25f;
			d[1] *= 0.25f;
			d[2] *= 0.25f;
			Math_VecAdd(obj->net.vel, d, obj->net.vel);

			float grav = 0.0f;
			Phys_ApplyObjectPhysics(obj, timeMod, obj->net.maxs[0], grav, 0.0f);

			Math_VecSub(obj->net.pos, boneMat.o, d);
			if (Math_VecLen(d) > maxChainDist)
			{ //end of the chain
				Math_VecNorm(d);
				Math_VecMA(boneMat.o, maxChainDist, d, obj->net.pos);
			}

			Math_VecCopy(obj->net.pos, owner->net.boltOffsets);
			owner->net.boltOffsets[2] += 32.0f;
			owner->net.effectLen = 1.0f;
			owner->net.renderEffects2 |= FXFL2_IKHAIR;
		}
	}
	else
	{
		if (owner && owner->chain == obj)
		{
			owner->chain = NULL;
			obj->chain = NULL;
		}
		obj->think = ObjGeneral_RemoveThink;
	}

	/*
	float bmins[3], bmaxs[3], clr[3];
	Math_VecAdd(obj->net.pos, obj->net.mins, bmins);
	Math_VecAdd(obj->net.pos, obj->net.maxs, bmaxs);
	clr[0] = 0.0f;
	clr[1] = 1.0f;
	clr[2] = 0.0f;
	Util_DebugBox(bmins, bmaxs, clr);
	*/
}

//hair object
void ObjSephHair_Spawn(gameObject_t *obj, BYTE *b, const objArgs_t *args, int numArgs)
{
	obj->think = ObjSephHair_Think;
}
