/*
=============================================================================
Module Information
------------------
Name:			obj_tonberry.cpp
Author:			Rich Whitehouse
Description:	server logic object: tonberry
=============================================================================
*/

#include "main.h"
#include "ai.h"

//just to make things a bit more readable
#define debounceFastLook		debounce5
#define debounceTonTime			debounce4

typedef enum
{
	TONANIM_KNIFEUP = NUM_HUMAN_ANIMS,
	TONANIM_STAB,
	TONANIM_MAGICTIME,
	NUM_TON_ANIMS
} tonAnims_e;

static gameAnim_t g_tonAnims[NUM_TON_ANIMS] =
{
	{0, 20, 68.0f, true},			//HUMANIM_IDLE
	{32, 44, 68.0f, true},			//HUMANIM_WALK
	{32, 44, 68.0f, true},			//HUMANIM_RUNSLOW
	{32, 44, 68.0f, true},			//HUMANIM_RUN
	{0, 4, 68.0f, false},			//HUMANIM_JUMP
	{0, 4, 68.0f, false},			//HUMANIM_FALL
	{0, 4, 68.0f, false},			//HUMANIM_LAND
	{21, 29, 68.0f, false},			//HUMANIM_PAIN_HIGH1
	{21, 29, 68.0f, false},			//HUMANIM_PAIN_HIGH2
	{21, 29, 68.0f, false},			//HUMANIM_PAIN_LOW1
	{21, 29, 68.0f, false},			//HUMANIM_PAIN_LOW2
	{21, 29, 68.0f, false},			//HUMANIM_PAIN_AIR
	{306, 309, 68.0f, false},		//HUMANIM_PAIN_POPUP
	{122, 126, 118.0f, false},		//HUMANIM_GETUP_BACK
	{122, 126, 118.0f, false},		//HUMANIM_GETUP_FRONT
	{306, 309, 68.0f, false},		//HUMANIM_FLYBACK
	{306, 309, 68.0f, false},		//HUMANIM_FLYBACK2
	{306, 309, 68.0f, false},		//HUMANIM_HIT_FALLFORWARD
	{109, 112, 68.0f, false},		//HUMANIM_FALL_LAND
	{109, 112, 68.0f, false},		//HUMANIM_POPUP_LAND
	{310, 310, 1568.0f, false},		//HUMANIM_DEATH
	{311, 316, 68.0f, false},		//TONANIM_KNIFEUP
	{316, 317, 68.0f, false},		//TONANIM_STAB
	{50, 84, 68.0f, false}			//TONANIM_MAGICTIME
};

#define NUM_TON_DAMAGE_BONES	1
//lantern bone is b11
static damageBone_t g_tonDamageBones[NUM_TON_DAMAGE_BONES] =
{
	{"b16", "b15", {0.0f, 0.0f, 0.0f}, 140.0f}			//DMGBONE_FIST_R
};

//returns damage bone or -1 if not in attack
int ObjTon_InAttack(gameObject_t *obj, int *dmgBones)
{
	if (obj->curAnim == TONANIM_STAB)
	{
		dmgBones[0] = DMGBONE_FIST_R;
		return 1;
	}

	return -1;
}

//occupied?
bool ObjTon_CanAct(gameObject_t *obj)
{
	if (!obj->onGround)
	{
		return false;
	}
	if (obj->curAnim >= TONANIM_KNIFEUP)
	{
		return false;
	}
	if (AI_NonMovingAnim(obj))
	{
		return false;
	}
	if (obj->aiObj->noMoveTime >= g_gameTime)
	{
		return false;
	}
	return true;
}

//can i dodge?
bool ObjTon_CanDodge(gameObject_t *obj)
{
	if (obj->generalFlag >= 0)
	{
		return false;
	}
	if (!ObjTon_CanAct(obj))
	{
		return false;
	}

	return true;
}

//dodge (tons currently don't dodge)
bool ObjTon_DodgeEnemy(gameObject_t *obj, gameObject_t *enemy)
{
	return false;
}

//think
void ObjTon_Think(gameObject_t *obj, float timeMod)
{
	if (!AI_ShouldThink(obj))
	{
		return;
	}

	if (g_timeType == TIMETYPE_TONBERRY)
	{
		obj->net.renderEffects |= FXFL_CONTRAST;
		obj->localTimeScale = 8.0f;
	}
	else
	{
		obj->net.renderEffects &= ~FXFL_CONTRAST;
		obj->localTimeScale = 1.0f;
	}

	if (obj->health > 0)
	{
		obj->net.renderEffects2 |= FXFL2_LANTERN;
		if (obj->generalFlag <= -1)
		{
			obj->net.boltOffsets[0] = 2.0f;
			obj->net.boltOffsets[1] = 0.0f;
			obj->net.boltOffsets[2] = 0.0f;
		}
		else
		{
			obj->net.boltOffsets[0] = 1.0f;
			obj->net.boltOffsets[1] = 0.8f;
			obj->net.boltOffsets[2] = 0.0f;
		}
	}
	else
	{
		obj->net.renderEffects2 &= ~FXFL2_LANTERN;
	}

	AI_StandardGoals(obj, timeMod);
	AI_GenericThink(obj, timeMod);
	AI_GetToGoal(obj, timeMod);

	if (obj->debounceFastLook >= g_gameTime)
	{
		obj->net.ang[YAW] = Math_BlendAngleLinear(obj->net.ang[YAW], obj->aiObj->lookYaw, 45.0f);
	}

	dmgBoneLine_t dmgPos[NUM_TON_DAMAGE_BONES];
	AI_TransformDamageBones(obj, g_tonDamageBones, dmgPos, NUM_TON_DAMAGE_BONES);
	int dmgBones[MAX_AI_DAMAGE_BONES];
	int numDmgBones = ObjTon_InAttack(obj, dmgBones);
	if (obj->aiObj->hasLastDmg && numDmgBones >= 0)
	{
		int dmg = AI_LevelDamage(obj, 20);
		for (int i = 0; i < numDmgBones; i++)
		{
			AI_RunDamageBone(obj, &g_tonDamageBones[dmgBones[i]], &obj->aiObj->lastDmg[dmgBones[i]],
				&dmgPos[dmgBones[i]], dmg);
		}
	}

	obj->atkType = ATKTYPE_NORMAL;

	memcpy(obj->aiObj->lastDmg, dmgPos, sizeof(dmgPos));
	obj->aiObj->hasLastDmg = true;

	if (obj->aiObj->noMoveTime >= g_gameTime)
	{ //tonberry always looks
		float angleBlendScale = 0.3f;
		obj->net.ang[YAW] = Math_BlendAngle(obj->net.ang[YAW], obj->aiObj->lookYaw, timeMod*angleBlendScale);
	}

	if (g_timeType == TIMETYPE_TONBERRY)
	{
		obj->debounceTonTime = g_gameTime + Util_LocalDelay(obj, (serverTime_t)(10000+rand()%10000));
	}

	obj->aiObj->combatRange = 700.0f;
	if (!AI_NonAnglingAnim(obj) && obj->aiObj->enemy && AI_FacingEnemy(obj, obj->aiObj->enemy))
	{
		if (obj->aiObj->enemy->net.index < MAX_NET_CLIENTS &&
			ObjTon_CanDodge(obj) &&
			ObjPlayer_InDodgeableAttack(obj->aiObj->enemy) &&
			ObjTon_DodgeEnemy(obj, obj->aiObj->enemy))
		{ //dodge
			obj->aiObj->noMoveTime = g_gameTime+100;
		}
		else if (obj->aiObj->enemy && obj->aiObj->distToEnemy < /*2048.0f*/4096.0f &&
			obj->generalFlag <= -1 && obj->debounceTonTime < g_gameTime &&
			obj->aiObj->attackTime < g_gameTime && ObjTon_CanAct(obj))
		{
			if (rand()%10 < 6)
			{
				AI_StartAnim(obj, TONANIM_MAGICTIME, true);
				obj->aiObj->noMoveTime = g_gameTime+100;
			}
			obj->debounceTonTime = g_gameTime + Util_LocalDelay(obj, (serverTime_t)(5000+rand()%2000));
		}
		else
		{
			bool shouldCloseIn = AI_ShouldCloseIn(obj);
			if (obj->aiObj->attackTime < g_gameTime && shouldCloseIn && ObjTon_CanAct(obj))
			{
				obj->aiObj->combatRange = 200.0f;
				float closeAttackDist = 400.0f;
				if (obj->aiObj->distToEnemy < closeAttackDist)
				{
					AI_StartAnim(obj, TONANIM_KNIFEUP, true);
					if (obj->generalFlag <= -1)
					{
						obj->aiObj->attackTime = g_gameTime + Util_LocalDelay(obj, (serverTime_t)(1500+rand()%2000));
					}
					else
					{
						obj->aiObj->attackTime = g_gameTime + Util_LocalDelay(obj, (serverTime_t)(2000+rand()%4000));
					}
					obj->aiObj->noMoveTime = g_gameTime + Util_LocalDelay(obj, 1000);
				}
			}
			else
			{
				float closeDist = (obj->generalFlag <= 0) ? 600.0f : 400.0f;
				float r = (shouldCloseIn) ? closeDist : 800.0f;
				obj->aiObj->combatRange = g_ai.playerCloseDist+r;
			}
		}
	}
}

//pick which anim to be in
void ObjTon_PickAnim(gameObject_t *obj, float timeMod)
{
	gameAnim_t *curAnim = obj->animTable+obj->curAnim;

	if (obj->curAnim == TONANIM_KNIFEUP)
	{
		if (obj->net.frame == curAnim->endFrame && !obj->animRestart)
		{
			AI_StartAnim(obj, TONANIM_STAB, true);
			obj->generalFlag2 = 15;
		}
	}
	else if (obj->curAnim == TONANIM_STAB)
	{
		if (obj->net.frame == curAnim->endFrame && !obj->animRestart)
		{
			if (obj->generalFlag2 > 0)
			{
				AI_StartAnim(obj, TONANIM_STAB, true);
				if (obj->generalFlag <= -1)
				{
					obj->aiObj->attackTime = g_gameTime + Util_LocalDelay(obj, (serverTime_t)(1500+rand()%2000));
				}
				else
				{
					obj->aiObj->attackTime = g_gameTime + Util_LocalDelay(obj, (serverTime_t)(2000+rand()%4000));
				}
				obj->aiObj->noMoveTime = g_gameTime + Util_LocalDelay(obj, 200);
				obj->generalFlag2--;
			}
			else
			{
				AI_StartAnim(obj, HUMANIM_IDLE, true);
			}
		}
	}
	else if (obj->net.frame == curAnim->endFrame && !obj->animRestart)
	{
		AI_StartAnim(obj, HUMANIM_IDLE, true);
	}
}

//frame tick
void ObjTon_FrameTick(gameObject_t *obj, float timeMod, int oldFrame)
{
	switch (obj->curAnim)
	{
	case TONANIM_STAB:
		if (obj->net.frame == 317)
		{
			ObjSound_CreateFromIndex(obj->net.pos, g_soundPunch[rand()%NUM_SOUNDS_PUNCH], 1.0f, -1);
			float fwd[3];
			float f = 200.0f;
			Math_AngleVectors(obj->net.ang, fwd, 0, 0);
			obj->net.vel[0] += fwd[0]*f;
			obj->net.vel[1] += fwd[1]*f;
		}
		break;
	case TONANIM_MAGICTIME:
		if (obj->net.frame <= 64)
		{
			obj->aiObj->noMoveTime = g_gameTime+100;
		}
		else if (obj->net.frame == 65)
		{
			if (g_timeType == TIMETYPE_NORMAL || g_timeType == TIMETYPE_TIMEATTACK)
			{
				g_tonberryTime = g_gameTime+2000;
				ObjSound_Create(obj->net.pos, "assets/sound/cb/chargetime.wav", 1.0f, -1);
			}
		}
		break;
	default:
		break;
	}
}

//touch
void ObjTon_Touch(gameObject_t *obj, gameObject_t *other, const collObj_t *col)
{
	AI_GenericTouch(obj, other, col);
}

//tons currently can't block
bool ObjTon_AttackBlock(gameObject_s *obj, gameObject_s *hurter, int dmg, collObj_t *col,
							float *impactPos, float *impactAng)
{
	return false;
}

//spawn
void ObjTon_Spawn(gameObject_t *obj, BYTE *b, const objArgs_t *args, int numArgs)
{
	static int dmgSounds[3];
	dmgSounds[0] = g_sharedFn->Common_ServerString("$assets/sound/cb/slash01.wav");
	dmgSounds[1] = g_sharedFn->Common_ServerString("$assets/sound/cb/slash02.wav");
	dmgSounds[2] = g_sharedFn->Common_ServerString("$assets/sound/cb/slash03.wav");

	AI_GenericSpawn(obj, b, args, numArgs);

	obj->touch = ObjTon_Touch;
	obj->animframetick = ObjTon_FrameTick;

	const char *elvalue = Common_GetValForKey(b, "elvalue");
	if (elvalue && elvalue[0])
	{
		obj->generalFlag = atoi(elvalue);
	}

	if (obj->generalFlag == -1)
	{
		obj->aiObj->makoValue = 200 + rand()%100;
		obj->net.aiDescIndex = AIDESC_TON2;
	}
	else
	{
		obj->aiObj->makoValue = 70 + rand()%20;
		obj->net.aiDescIndex = AIDESC_TON1;
	}

	obj->aiObj->dropChances[INVITEM_HOURGLASS] = 40;
	if (obj->generalFlag == -1)
	{
		obj->aiObj->dropChances[INVITEM_ELIXER] = 10;
		obj->aiObj->dropChances[INVITEM_HOURGLASS] = 100;
	}
	obj->aiObj->maxHealth = obj->health;

	obj->aiObj->dmgEffect = "melee/impactslash";
	obj->aiObj->dmgSounds = dmgSounds;
	obj->aiObj->numDmgSounds = 3;
	if (obj->generalFlag == -1)
	{
		obj->aiObj->moveSpeed = 50.0f;
	}
	else
	{
		obj->aiObj->moveSpeed = 50.0f;
	}
	obj->animhandler = ObjTon_PickAnim;
	obj->animTable = g_tonAnims;
	obj->curAnim = HUMANIM_IDLE;
	obj->think = ObjTon_Think;
	obj->attackblock = ObjTon_AttackBlock;
}
