/*
=============================================================================
Module Information
------------------
Name:			obj_triggerbox.cpp
Author:			Rich Whitehouse
Description:	trigger box

Object Arguments
script:			R-Script to run when something enters the box. The targname of
				the object entering the box is pushed to the R-Script stack.
onlyuseon:		Value is the targname of an object. Box can only be used by
				this object.
removeonuse:	Remove the box after the first time it is triggered.
removeon:		Value is the targname of an object. Box is removed if it is
				used by this particular object.
startoff:		Start disabled. It is toggled by setting the rfx hidden bit.
				This can be done in R-Script.
debounce:		Milliseconds to wait between triggering.
buttonuse:		Player must push jump to use.
beambox:		Special case, make it solid and draw beams across the longest
				x/y axis.
fx:				Emit a particle system while active.
fxdelay:		Delay between fx emissions.
fxofs:			(x y z) offset from center to display particles.
fxang:			(pitch yaw roll) angles to play particle system.
enemyMax:		Don't activate if more than this many active enemies on the map.
deactonuse:		Deactivate on use.
gvar:			Only spawns if this gvar is set
holospawn:		Special hologram spawner option, value is the name off the object
				to spawn on use.
mponly:			Only present in multiplayer
=============================================================================
*/

#include "main.h"
#include "ai.h"
#include "rscript.h"

typedef struct triggerBoxInfo_s
{
	char			*script;
	char			*onlyuseon;
	int				removeonuse;
	int				buttonuse;
	char			*removeon;
	serverTime_t	debounce;
	int				fx;
	serverTime_t	fxdelay;
	float			fxofs[3];
	float			fxang[3];
	int				enemyMax;
	int				deactonuse;
	bool			playerOnly;
	bool			mpOnly;
} triggerBoxInfo_t;

//returns true if it's go-away time
bool ObjTriggerBox_Use(gameObject_t *obj, gameObject_t *other, triggerBoxInfo_t *tbi, bool &used)
{
	used = false;

	if (tbi->enemyMax && g_ai.numActiveEnemyAI > tbi->enemyMax)
	{
		return false;
	}

	if (tbi->onlyuseon && tbi->onlyuseon[0])
	{
		if (!other->targetName || stricmp(other->targetName, tbi->onlyuseon))
		{ //can't be used by this
			return false;
		}
	}

	if (tbi->buttonuse && other->net.index < MAX_NET_CLIENTS && other->plObj)
	{
		if (other->health <= 0 || GVar_GetInt("intriggersequence") || other->plObj->ride)
		{ //don't trigger from jump while script says it's busy
			return false;
		}
		if (obj->str1)
		{ //holospawn
			obj->debounce3 = g_gameTime+500;
		}
		other->plObj->jumpUseObj = obj->net.index;
		other->plObj->canJumpUse = g_gameTime + Util_LocalDelay(other, 500);
		if (other->plObj->jumpUseTime < g_gameTime)
		{
			return false;
		}
		other->plObj->jumpUseTime = 0;
		other->plObj->canJumpUse = 0;
		other->plObj->noItemTime = g_gameTime+Util_LocalDelay(other, 500);
	}

	used = true;
	if (tbi->script && tbi->script[0])
	{
		float z[3] = {0.0f, 0.0f, 0.0f};
		objArgs_t arg;
		arg.key = "script";
		arg.val = tbi->script;
		gameObject_t *scriptObj = LServ_ObjectFromName("obj_rscript", z, z, &arg, 1);
		if (!scriptObj || !scriptObj->rscript)
		{
			Util_ErrorMessage("triggerbox with bad script: %s", tbi->script);
			return true;
		}
		scriptObj->spawnArgs = NULL;
		scriptObj->numSpawnArgs = 0;
		char *otherName = (other->targetName) ? other->targetName : "";
		scriptObj->rscript->ropStack[scriptObj->rscript->ropStackPtr] = otherName;
		scriptObj->rscript->ropStackPtr--;
		if (obj->str1)
		{
			scriptObj->rscript->ropStack[scriptObj->rscript->ropStackPtr] = obj->str1;
			scriptObj->rscript->ropStackPtr--;
		}
		if (obj->targetName)
		{
			char str[256];
			sprintf(str, "%s_script", obj->targetName);
			scriptObj->targetName = Util_PooledString(str);
		}
	}

	if (tbi->removeonuse)
	{
		return true;
	}
	if (tbi->removeon && tbi->removeon[0])
	{
		if (other->targetName && !stricmp(other->targetName, tbi->removeon))
		{
			return true;
		}
	}

	if (tbi->deactonuse)
	{
		obj->net.renderEffects |= FXFL_HIDDEN;
	}

	return false;
}

//can be activated for immediate use
void ObjTriggerBox_Activate(gameObject_t *obj, gameObject_t *activator)
{
	bool used;
	triggerBoxInfo_t *tbi = (triggerBoxInfo_t *)obj->customDynamic;
	if (ObjTriggerBox_Use(obj, activator, tbi, used))
	{
		obj->think = ObjGeneral_RemoveThink;
		obj->thinkTime = g_gameTime;
	}
}

//if nothing is inside, become solid
void ObjTriggerBox_BecomeSolid(gameObject_t *obj)
{
	if (obj->net.solid)
	{
		return;
	}

	collObj_t col;
	g_sharedFn->Coll_MovementTranslation(obj->rcColModel, &col, obj->net.pos, obj->net.pos, NULL, NULL);
	if (!col.hit && !col.containsSolid)
	{
		obj->net.solid = 1;
		LServ_UpdateRClip(obj);
	}
}

//think
void ObjTriggerBox_Think(gameObject_t *obj, float timeMod)
{
	obj->thinkTime = g_gameTime;

	if (obj->chain)
	{
		if (obj->net.renderEffects & FXFL_HIDDEN)
		{
			obj->net.solid = 0;
			LServ_UpdateRClip(obj);
		}
		else
		{
			obj->net.solid = 1;
			LServ_UpdateRClip(obj);

			obj->chain->net.ang[YAW] = Math_AngleMod(obj->chain->net.ang[YAW]+(1.0f*timeMod));

			if (obj->debounce3 > g_gameTime)
			{
				obj->chain->net.renderEffects2 |= FXFL2_SKIPDEPTH;
				if (obj->chain->net.renderEffects & FXFL_HIDDEN)
				{
					obj->chain->net.renderEffects &= ~FXFL_HIDDEN;
					ObjSound_Create(obj->net.pos, "assets/sound/cb/holoon.wav", 1.0f, -1);
				}
				obj->net.renderEffects2 |= FXFL2_SKINMAP;
				obj->net.strIndexC = g_sharedFn->Common_ServerString("assets/textures/holoproj");
				obj->net.strIndexD = g_sharedFn->Common_ServerString("assets/textures/holoproj_on");
			}
			else
			{
				obj->chain->net.renderEffects2 &= ~FXFL2_SKIPDEPTH;
				if (!(obj->chain->net.renderEffects & FXFL_HIDDEN))
				{
					obj->chain->net.renderEffects |= FXFL_HIDDEN;
					ObjSound_Create(obj->net.pos, "assets/sound/cb/holooff.wav", 1.0f, -1);
				}
				obj->net.renderEffects2 &= ~FXFL2_SKINMAP;
			}
		}
	}

	if (obj->net.renderEffects & FXFL_BEAMBOX)
	{
		RCUBE_ASSERT(obj->rcColModel);
		if (obj->net.renderEffects & FXFL_HIDDEN)
		{ //become unsolid
			if (obj->net.solid)
			{
				ObjSound_Create(obj->net.pos, "assets/sound/cb/beamoff.wav", 1.0f, -1);
				obj->net.solid = 0;
				LServ_UpdateRClip(obj);
			}
		}
		else
		{
			ObjTriggerBox_BecomeSolid(obj);
		}
	}

	if (obj->net.renderEffects & FXFL_HIDDEN)
	{
		return;
	}

	triggerBoxInfo_t *tbi = (triggerBoxInfo_t *)obj->customDynamic;

	if (tbi->fx && obj->debounce2 < g_gameTime)
	{ //activated fx
		gameObject_t *cam = Util_GetCam(0);
		if (g_sharedFn->Coll_GeneralVisibilityWithBounds(obj->net.pos, cam->net.pos, obj->radius, obj->spawnMins, obj->spawnMaxs))
		{
			float fxPos[3];
			Math_VecAdd(obj->net.pos, tbi->fxofs, fxPos);
			ObjParticles_CreateFromIndex(tbi->fx, fxPos, tbi->fxang, -1);
		}
		obj->debounce2 = g_gameTime+tbi->fxdelay;
	}

	if (obj->debounce >= g_gameTime)
	{
		return;
	}

	float trigMins[3], trigMaxs[3];
	Math_VecAdd(obj->net.pos, obj->spawnMins, trigMins);
	Math_VecAdd(obj->net.pos, obj->spawnMaxs, trigMaxs);

	int totalCheck = (tbi->playerOnly) ? 1 : g_gameObjectSlots;
	if (tbi->playerOnly && tbi->mpOnly)
	{
		totalCheck = MAX_NET_CLIENTS;
	}

	for (int i = 0; i < totalCheck; i++)
	{
		gameObject_t *other = &g_gameObjects[i];
		if (i == MAP_OBJ_NUM || !other->inuse || !other->rcColModel || !other->rcColModel->solid ||
			other->net.staticIndex != -1)
		{
			continue;
		}

		float absMins[3], absMaxs[3];
		Math_VecAdd(other->rcColModel->pos, other->rcColModel->mins, absMins);
		Math_VecAdd(other->rcColModel->pos, other->rcColModel->maxs, absMaxs);
		if (Math_BoxesOverlap(trigMins, trigMaxs, absMins, absMaxs))
		{
			bool used;
			if (ObjTriggerBox_Use(obj, other, tbi, used))
			{
				obj->think = ObjGeneral_RemoveThink;
				return;
			}
			if (used)
			{
				if (tbi->debounce)
				{
					obj->debounce = g_gameTime + tbi->debounce;
					return;
				}
			}
		}
		else if (tbi->buttonuse && other->plObj && other->plObj->jumpUseObj == obj->net.index)
		{
			other->plObj->jumpUseTime = 0;
			other->plObj->canJumpUse = 0;
			other->plObj->jumpUseObj = 0;
		}
	}
}

//spawn
void ObjTriggerBox_Spawn(gameObject_t *obj, BYTE *b, const objArgs_t *args, int numArgs)
{
	obj->customDynamic = g_sharedFn->Common_RCMalloc(sizeof(triggerBoxInfo_t));
	triggerBoxInfo_t *tbi = (triggerBoxInfo_t *)obj->customDynamic;
	memset(tbi, 0, sizeof(triggerBoxInfo_t));
	for (int i = 0; i < numArgs; i++)
	{
		const objArgs_t *arg = args+i;
		if (!stricmp(arg->key, "script"))
		{
			tbi->script = arg->val;
		}
		else if (!stricmp(arg->key, "onlyuseon"))
		{
			tbi->onlyuseon = arg->val;
		}
		else if (!stricmp(arg->key, "removeonuse"))
		{
			tbi->removeonuse = atoi(arg->val);
		}
		else if (!stricmp(arg->key, "removeon"))
		{
			tbi->removeon = arg->val;
		}
		else if (!stricmp(arg->key, "startoff"))
		{
			obj->net.renderEffects |= FXFL_HIDDEN;
		}
		else if (!stricmp(arg->key, "debounce"))
		{
			tbi->debounce = (serverTime_t)atoi(arg->val);
		}
		else if (!stricmp(arg->key, "buttonuse"))
		{
			tbi->buttonuse = atoi(arg->val);
		}
		else if (!stricmp(arg->key, "beambox"))
		{
			g_sharedFn->Common_ServerString("$assets/sound/cb/beamoff.wav");
			obj->net.renderEffects |= FXFL_BEAMBOX;
			obj->rcColModel = g_sharedFn->Coll_RegisterModelInstance("*box");
			obj->net.solid = 1;
			LServ_UpdateRClip(obj);
		}
		else if (!stricmp(arg->key, "fx"))
		{
			char str[256];
			sprintf(str, "&&%s", arg->val);
			tbi->fx = g_sharedFn->Common_ServerString(str);
		}
		else if (!stricmp(arg->key, "fxdelay"))
		{
			tbi->fxdelay = (serverTime_t)atoi(arg->val);
		}
		else if (!stricmp(arg->key, "fxofs"))
		{
			sscanf(arg->val, "(%f %f %f)", &tbi->fxofs[0], &tbi->fxofs[1], &tbi->fxofs[2]);
		}
		else if (!stricmp(arg->key, "fxang"))
		{
			sscanf(arg->val, "(%f %f %f)", &tbi->fxang[0], &tbi->fxang[1], &tbi->fxang[2]);
		}
		else if (!stricmp(arg->key, "enemyMax"))
		{
			tbi->enemyMax = atoi(arg->val);
		}
		else if (!stricmp(arg->key, "deactonuse"))
		{
			tbi->deactonuse = atoi(arg->val);
		}
		else if (!stricmp(arg->key, "gvar"))
		{
			if (!GVar_GetInt(arg->val))
			{
				obj->think = ObjGeneral_RemoveThink;
				obj->thinkTime = g_gameTime;
				return;
			}
		}
		else if (!stricmp(arg->key, "holospawn"))
		{
			if (!tbi->debounce)
			{
				tbi->debounce = 50;
			}
			g_sharedFn->Common_ServerString("$assets/sound/cb/holoon.wav");
			g_sharedFn->Common_ServerString("$assets/sound/cb/holooff.wav");
			g_sharedFn->Common_ServerString("^assets/textures/holoproj_on");
			obj->str1 = arg->val;
			BYTE *b = Common_GetEntryForObject(arg->val);
			if (b)
			{
				const char *s = Common_GetValForKey(b, "assetName");
				if (s && s[0])
				{
					char assetName[256];
					obj->chain = LServ_ObjectFromName("obj_nothing", obj->net.pos, obj->net.ang, 0, 0);
					RCUBE_ASSERT(obj->chain);
					obj->chain->localFlags &= ~LFL_NONET;
					obj->chain->net.type = OBJ_TYPE_MODEL;
					Util_ParseVector(Common_GetValForKey(b, "spawnMins"), obj->chain->spawnMins);
					Util_ParseVector(Common_GetValForKey(b, "spawnMaxs"), obj->chain->spawnMaxs);
					Util_ParseVector(Common_GetValForKey(b, "mins"), obj->chain->net.mins);
					Util_ParseVector(Common_GetValForKey(b, "maxs"), obj->chain->net.maxs);
					sprintf(assetName, "&%s", s);
					obj->chain->net.strIndex = g_sharedFn->Common_ServerString(assetName);
					s = Common_GetValForKey(b, "modelAnim");
					if (s && s[0])
					{
						sprintf(assetName, "@%s", s);
						obj->chain->net.strIndexB = g_sharedFn->Common_ServerString(assetName);
					}
					obj->chain->net.pos[2] += 160.0f;
					obj->chain->net.renderEffects |= FXFL_HIDDEN;
				}
			}
		}
		else if (!stricmp(arg->key, "mponly"))
		{
			tbi->mpOnly = true;
			if (!g_runningMultiplayer)
			{
				obj->think = ObjGeneral_RemoveThink;
				obj->thinkTime = g_gameTime;
				return;
			}
		}
	}

	if (tbi->onlyuseon && !stricmp(tbi->onlyuseon, "__the_player"))
	{ //pseudo-hack optimization
		tbi->playerOnly = true;
	}

	obj->activate = ObjTriggerBox_Activate;
	if (!(obj->net.renderEffects & FXFL_BEAMBOX) && obj->net.type != OBJ_TYPE_MODEL)
	{
		obj->localFlags |= LFL_NONET;
	}
	obj->think = ObjTriggerBox_Think;
	obj->thinkTime = g_gameTime;
}
