/*
=============================================================================
Module Information
------------------
Name:			obj_visblock.cpp
Author:			Rich Whitehouse
Description:	visibility-blocking objects
=============================================================================
*/

#include "main.h"
#include <float.h>

gameObject_t *g_visBlockObjects[MAX_GAME_OBJECTS];
int g_numVisBlocks = 0;
float g_activeViewPos[3];
bool g_updatedViewPos = false;

//update the "cam" location
void ObjVisBlock_Update(float *pos)
{
	if (g_numVisBlocks <= 0)
	{
		return;
	}

	g_activeViewPos[0] = pos[0];
	g_activeViewPos[1] = pos[1];
	g_activeViewPos[2] = pos[2];
	g_updatedViewPos = true;
}

//is the bounds in this blocker?
static inline bool ObjVisBlock_BoundsInBlock(gameObject_t *block, float *mins, float *maxs)
{
	gameVisBlocker_t *vis = block->visBlockData;
	if (Math_BoxesOverlap(vis->minsS1, vis->maxsS1, mins, maxs))
	{
		return true;
	}
	if (Math_BoxesOverlap(vis->minsS2, vis->maxsS2, mins, maxs))
	{
		return true;
	}

	return false;
}

//visibility check from outside
bool ObjVisBlock_CheckVis(gameObject_t *obj)
{
	if (g_numVisBlocks <= 0 || !g_updatedViewPos)
	{ //just say visible if no vis blocks are here
		return true;
	}
	gameObject_t *cam = Util_GetCam(0);
	if (!cam->inuse)
	{ //no cam?
		return true;
	}

	float objMins[3], objMaxs[3];
	Math_VecAdd(obj->net.pos, obj->spawnMins, objMins);
	Math_VecAdd(obj->net.pos, obj->spawnMaxs, objMaxs);

	float camMins[3], camMaxs[3];
	Math_VecAdd(g_activeViewPos, cam->spawnMins, camMins);
	Math_VecAdd(g_activeViewPos, cam->spawnMaxs, camMaxs);
	camMins[0] -= 1.0f;
	camMins[1] -= 1.0f;
	camMins[2] -= 1.0f;
	camMaxs[0] += 1.0f;
	camMaxs[1] += 1.0f;
	camMaxs[2] += 1.0f;
	for (int i = 0; i < g_numVisBlocks; i++)
	{
		gameObject_t *v1 = g_visBlockObjects[i];
		if (!v1->inuse)
		{
			continue;
		}
		RCUBE_ASSERT(v1->visBlockData);
		if (!ObjVisBlock_BoundsInBlock(v1, camMins, camMaxs))
		{
			continue;
		}

		if (ObjVisBlock_BoundsInBlock(v1, objMins, objMaxs))
		{ //the object is in the same block
			int s = Math_PointRelativeToPlane(g_activeViewPos, NULL, v1->visBlockData->plane);
			float *min, *max;
			if (s >= 0)
			{
				min = v1->visBlockData->minsS1;
				max = v1->visBlockData->maxsS1;
			}
			else
			{
				min = v1->visBlockData->minsS2;
				max = v1->visBlockData->maxsS2;
			}
			if (Math_BoxesOverlap(min, max, objMins, objMaxs))
			{ //on the same side, we're good
				return true;
			}
			if (Math_BoxInFrustum(&g_wideCamFrustum, v1->net.mins, v1->net.maxs))
			{ //it's on the other side, but the box is in the frustum
				return true;
			}
		}
	}

	return false;
}

//debug draw all the blocks
void ObjVisBlock_DebugDraw(void)
{
	for (int i = 0; i < g_numVisBlocks; i++)
	{
		gameObject_t *v1 = g_visBlockObjects[i];
		if (!v1->inuse)
		{
			continue;
		}

		RCUBE_ASSERT(v1->visBlockData);

		float clr[3];
		clr[0] = 1.0f;
		clr[1] = 0.0f;
		clr[2] = 0.0f;
		Util_DebugBox(v1->net.mins, v1->net.maxs, clr);
		clr[0] = 0.0f;
		clr[1] = 1.0f;
		clr[2] = 0.0f;
		Util_DebugBox(v1->visBlockData->minsS1, v1->visBlockData->maxsS1, clr);
		clr[0] = 0.0f;
		clr[1] = 0.0f;
		clr[2] = 1.0f;
		Util_DebugBox(v1->visBlockData->minsS2, v1->visBlockData->maxsS2, clr);
	}
}

//see if an object is touching another blocker
static inline bool ObjVisBlock_TouchesOtherBlock(gameObject_t *blocker, gameObject_t *gameObj,
												 float *goMins, float *goMaxs)
{
	for (int k = 0; k < g_numVisBlocks; k++)
	{
		gameObject_t *v2 = g_visBlockObjects[k];
		if (!v2->inuse || v2 == blocker)
		{
			continue;
		}
		float blockMins[3], blockMaxs[3];
		Math_VecAdd(v2->net.pos, v2->spawnMins, blockMins);
		Math_VecAdd(v2->net.pos, v2->spawnMaxs, blockMaxs);
        if (Math_BoxesOverlap(goMins, goMaxs, blockMins,
			blockMaxs))
		{
			return true;
		}
	}

	return false;
}

//expand bounds by going through game objects
static inline void ObjVisBlock_RecurseExpandBounds(gameObject_t *blocker, float *exMinsS1, float *exMaxsS1,
												   float *exMinsS2, float *exMaxsS2, int recursionTick,
												   gameObject_t *gameObj, float *goMins, float *goMaxs)
{
	RCUBE_ASSERT(blocker->visBlockData);
	if (Math_PointRelativeToPlane(gameObj->net.pos, NULL, blocker->visBlockData->plane) >= 0)
	{
		Math_ExpandBounds(exMinsS1, exMaxsS1, goMins, goMaxs);
	}
	else
	{
		Math_ExpandBounds(exMinsS2, exMaxsS2, goMins, goMaxs);
	}
	gameObj->localRecursionTick = recursionTick;

	if (ObjVisBlock_TouchesOtherBlock(blocker, gameObj, goMins, goMaxs))
	{ //do not continue recursion if this thing touches another blocker
		return;
	}

	for (int j = 0; j < g_gameObjectSlots; j++)
	{ //go through all game objects for each vis block
		gameObject_t *subGameObj = &g_gameObjects[j];
		if (!subGameObj->inuse || subGameObj->net.type == OBJ_TYPE_VISBLOCK ||
			subGameObj->net.type == OBJ_TYPE_LIGHT || subGameObj->net.staticIndex == -1 ||
			subGameObj->localRecursionTick == recursionTick)
		{
			continue;
		}
		if (subGameObj->spawnMins[0] == subGameObj->spawnMaxs[0] ||
			subGameObj->spawnMins[1] == subGameObj->spawnMaxs[1] ||
			subGameObj->spawnMins[2] == subGameObj->spawnMaxs[2])
		{ //invalid bounds
			continue;
		}

		float subGoMins[3], subGoMaxs[3];
		Math_VecAdd(subGameObj->net.pos, subGameObj->spawnMins, subGoMins);
		Math_VecAdd(subGameObj->net.pos, subGameObj->spawnMaxs, subGoMaxs);
		if (Math_BoxesOverlap(goMins, goMaxs, subGoMins, subGoMaxs))
		{ //it's touching, expand
			ObjVisBlock_RecurseExpandBounds(blocker, exMinsS1, exMaxsS1,
				exMinsS2, exMaxsS2, recursionTick, subGameObj,
				subGoMins, subGoMaxs);
		}
	}
}

//called immediately post-map-spawn to establish visbility with visibility blockers
void ObjVisBlock_EstablishVis(void)
{
	if (g_numVisBlocks <= 0)
	{ //nothing to do
		return;
	}

	static int recursionTick = 0;
	for (int i = 0; i < g_numVisBlocks; i++)
	{
		recursionTick++;
		if (!recursionTick)
		{ //wrapped?!
			recursionTick++;
		}

		gameObject_t *v1 = g_visBlockObjects[i];
		if (!v1->inuse)
		{
			continue;
		}
		RCUBE_ASSERT(v1->visBlockData);
        float visBoundsMinsS1[3], visBoundsMaxsS1[3];
		float visBoundsMinsS2[3], visBoundsMaxsS2[3];
		Math_VecCopy(v1->net.mins, visBoundsMinsS1);
		Math_VecCopy(v1->net.maxs, visBoundsMaxsS1);
		Math_VecCopy(v1->net.mins, visBoundsMinsS2);
		Math_VecCopy(v1->net.maxs, visBoundsMaxsS2);

		for (int j = 0; j < g_gameObjectSlots; j++)
		{ //go through all game objects for each vis block
			gameObject_t *gameObj = &g_gameObjects[j];
			if (!gameObj->inuse || gameObj->net.type == OBJ_TYPE_VISBLOCK ||
				gameObj->net.type == OBJ_TYPE_LIGHT || gameObj->net.staticIndex == -1)
			{
				continue;
			}
			if (gameObj->spawnMins[0] == gameObj->spawnMaxs[0] ||
				gameObj->spawnMins[1] == gameObj->spawnMaxs[1] ||
				gameObj->spawnMins[2] == gameObj->spawnMaxs[2])
			{ //invalid bounds
				continue;
			}

			float goMins[3], goMaxs[3];
			Math_VecAdd(gameObj->net.pos, gameObj->spawnMins, goMins);
			Math_VecAdd(gameObj->net.pos, gameObj->spawnMaxs, goMaxs);
			if (Math_BoxesOverlap(v1->net.mins, v1->net.maxs, goMins, goMaxs))
			{ //it's touching, expand
				ObjVisBlock_RecurseExpandBounds(v1, visBoundsMinsS1, visBoundsMaxsS1,
					visBoundsMinsS2, visBoundsMaxsS2, recursionTick, gameObj,
					goMins, goMaxs);
			}
		}

		//put vismins/maxs somewhere
		Math_VecCopy(visBoundsMinsS1, v1->visBlockData->minsS1);
		Math_VecCopy(visBoundsMaxsS1, v1->visBlockData->maxsS1);
		Math_VecCopy(visBoundsMinsS2, v1->visBlockData->minsS2);
		Math_VecCopy(visBoundsMaxsS2, v1->visBlockData->maxsS2);
	}
}

//spawn
void ObjVisBlock_Spawn(gameObject_t *obj, BYTE *b, const objArgs_t *args, int numArgs)
{
	//never send to the client
	obj->localFlags |= LFL_NONET;
	obj->inuse |= INUSE_NONET;

	if (obj->spawnArgs && obj->numSpawnArgs > 0)
	{
		int i = 0;
		while (i < obj->numSpawnArgs)
		{
			const objArgs_t *arg = obj->spawnArgs+i;
			if (!_stricmp(arg->key, "blockMins"))
			{
				Util_ParseVector(arg->val, obj->spawnMins);
			}
			else if (!_stricmp(arg->key, "blockMaxs"))
			{
				Util_ParseVector(arg->val, obj->spawnMaxs);
			}
			i++;
		}
	}

	obj->radius = Util_RadiusFromBounds(obj->spawnMins, obj->spawnMaxs, 1.0f);

	//use abs bounds
	Math_VecAdd(obj->net.pos, obj->spawnMins, obj->net.mins);
	Math_VecAdd(obj->net.pos, obj->spawnMaxs, obj->net.maxs);

	obj->visBlockData = (gameVisBlocker_t *)g_sharedFn->Common_RCMalloc(sizeof(gameVisBlocker_t));
	memset(obj->visBlockData, 0, sizeof(gameVisBlocker_t));
	int shortestAxis = -1;
	float shortestAxisLen = 0.0f;
	for (int i = 0; i < 3; i++)
	{
		float d = obj->net.maxs[i]-obj->net.mins[i];
		if (shortestAxis == -1 || d < shortestAxisLen)
		{
			shortestAxisLen = d;
			shortestAxis = i;
		}
	}
	RCUBE_ASSERT(shortestAxis != -1);

	//create the plane
	obj->visBlockData->plane[shortestAxis] = 1.0f;
	obj->visBlockData->plane[3] = -Math_DotProduct(obj->net.pos, obj->visBlockData->plane);

	//add it to the main list of vis blockers
	g_visBlockObjects[g_numVisBlocks++] = obj;
}
