/*
=============================================================================
Module Information
------------------
Name:			rscript.cpp
Author:			Rich Whitehouse
Description:	R-Script routines
=============================================================================
*/

#include "main.h"
#include "rscript.h"

static rscriptBase_t *g_rscripts[MAX_RSCRIPTS];
static int g_numRScripts = 0;

static rscriptLabel_t *g_labels[MAX_RSCRIPT_LABELS];
static int g_numLabels = 0;

static rscriptHandler_t g_scriptHandlers[] =
{
	{"delay", RSHan_Delay, 1},
	{"delayrand", RSHan_DelayRand, 2},
	{"cacheres", RSHan_CacheRes, 1},
	{"cacheobj", RSHan_CacheObj, 1},
	{"runscript", RSHan_RunScript, 2},
	{"changemap", RSHan_ChangeMap, 1},
	{"waitonscript", RSHan_WaitOnScript, 1},
	{"waitonframe", RSHan_WaitOnFrame, 2},
	{"waitoninput", RSHan_WaitOnInput, 1},
	{"waitonenemiesdead", RSHan_WaitOnEnemiesDead, 0},
	{"waitondyingemies", RSHan_WaitOnDyingEnemies, 0},
	{"waitonenemiesnum", RSHan_WaitOnEnemiesNum, 1},
	{"waitondead", RSHan_WaitOnDead, 0},
	{"killenemies", RSHan_KillEnemies, 0},
	{"killenemiespassive", RSHan_KillEnemiesPassive, 0},
	{"killenemiesquiet", RSHan_KillEnemiesQuiet, 0},
	{"togglenotarget", RSHan_ToggleNoTarget, 0},
	{"toggleconfuse", RSHan_ToggleConfuse, 0},
	{"magicmodecheck", RSHan_MagicModeCheck, 0},
	{"statmsg", RSHan_StatMsg, 1},
	{"arenamsg", RSHan_ArenaMsg, 0},
	{"setfade", RSHan_SetFade, 1},
	{"setfadecolor", RSHan_SetFadeColor, 1},
	{"lerpfade", RSHan_LerpFade, 2},
	{"setpicfade", RSHan_SetPicFade, 1},
	{"lerppicfade", RSHan_LerpPicFade, 2},
	{"pushpic", RSHan_PushPic, 1},
	{"showhud", RSHan_ShowHUD, 1},
	{"runmenuscript", RSHan_RunMenuScript, 2},
	{"rollcredits", RSHan_RollCredits, 0},
	{"setcinema", RSHan_SetCinema, 1},
	{"setmusic", RSHan_SetMusic, 1},
	{"cyclemusic", RSHan_CycleMusic, 0},
	{"setbattlemusic", RSHan_SetBattleMusic, 1},
	{"setmagicmode", RSHan_SetMagicMode, 1},
	{"setendlessbattle", RSHan_SetEndlessBattle, 1},
	{"endlessbattlespawns", RSHan_EndlessBattleSpawns, 0},
	{"setmpversusmode", RSHan_SetMPVersusMode, 1},
	{"setskymodel", RSHan_SetSkyModel, 1},
	{"setskyterrain", RSHan_SetSkyTerrain, 1},
	{"setskycolor", RSHan_SetSkyColor, 1},
	{"setskycolorrand", RSHan_SetSkyColorRand, 0},
	{"setpropframes", RSHan_SetPropFrames, 5},
	{"forcenextanim", RSHan_ForceNextAnim, 1},
	{"setplmovespeed", RSHan_SetPlMoveSpeed, 1},
	{"setplcamtarg", RSHan_SetPlCamTarg, 1},
	{"setplcamrange", RSHan_SetPlCamRange, 1},
	{"giveplitem", RSHan_GivePlItem, 1},
	{"giveplmako", RSHan_GivePlMako, 1},
	{"giveplcharge", RSHan_GivePlCharge, 1},
	{"setcambolt", RSHan_SetCamBolt, 3},
	{"setcamabsofs", RSHan_SetCamAbsOfs, 2},
	{"setcamslack", RSHan_SetCamSlack, 1},
	{"setcamfov", RSHan_SetCamFov, 1},
	{"lerpcamfov", RSHan_LerpCamFov, 2},
	{"setactiontime", RSHan_SetActionTime, 1},
	{"setactiontimescale", RSHan_SetActionTimeScale, 1},
	{"setambientlight", RSHan_SetAmbientLight, 1},
	{"lerpambientlight", RSHan_LerpAmbientLight, 2},
	{"lerpambientlightevil", RSHan_LerpAmbientLightEvil, 2},
	{"setviewtrails", RSHan_SetViewTrails, 2},
	{"setfragfog", RSHan_SetFragFog, 4},
	{"lerpfragfog", RSHan_LerpFragFog, 5},
	{"setbloommod", RSHan_SetBloomMod, 1},
	{"setpcltimescale", RSHan_SetPclTimeScale, 1},
	{"setviewflash", RSHan_SetViewFlash, 2},
	{"setviewshake", RSHan_SetViewShake, 1},
	{"setenspawnblock", RSHan_SetEnSpawnBlock, 1},
	{"setexvar", RSHan_SetExVar, 2},
	{"mpshowstats", RSHan_MPShowStats, 0},
	{"mpmatchcheck", RSHan_MPMatchCheck, 0},
	{"mpbeginmatch", RSHan_MPBeginMatch, 0},
	{"mpendmatch", RSHan_MPEndMatch, 0},
	{"objcreate", RSHan_ObjCreate, 3},
	{"objcreatenamed", RSHan_ObjCreateNamed, 4},
	{"objcreatenamedatobj", RSHan_ObjCreateNamedAtObj, 3},
	{"objcreatenamedinobj", RSHan_ObjCreateNamedInObj, 3},
	{"objcreatenamedinobjrnd", RSHan_ObjCreateNamedInObjRnd, 5},
	{"objcreateenemy", RSHan_ObjCreateEnemy, 2},
	{"objcreateenemyforce", RSHan_ObjCreateEnemyForce, 2},
	{"objcreateenemymax", RSHan_ObjCreateEnemyMax, 3},
	{"objcreatesneaky", RSHan_ObjCreateSneaky, 3},
	{"objcreateitem", RSHan_ObjCreateItem, 4},
	{"objcreateitemalt", RSHan_ObjCreateItemAlt, 5},
	{"objremovetempitems", RSHan_ObjRemoveTempItems, 0},
	{"objremove", RSHan_ObjRemove, 1},
	{"objactivate", RSHan_ObjActivate, 1},
	{"objmovepos", RSHan_ObjMovePos, 3},
	{"objsetpos", RSHan_ObjSetPos, 2},
	{"objputground", RSHan_ObjPutGround, 1},
	{"objgivehealth", RSHan_ObjGiveHealth, 2},
	{"objsettarget", RSHan_ObjSetTarget, 2},
	{"objsetdebounce", RSHan_ObjSetDebounce, 2},
	{"objmoveang", RSHan_ObjMoveAng, 3},
	{"objlookat", RSHan_ObjLookAt, 3},
	{"objsetang", RSHan_ObjSetAng, 2},
	{"objsound", RSHan_ObjSound, 2},
	{"objsoundrand", RSHan_ObjSoundRand, 3},
	{"objfx", RSHan_ObjFX, 5},
	{"objfxbolted", RSHan_ObjFXBolted, 4},
	{"objrfxsetbit", RSHan_ObjRFXSetBit, 2},
	{"objrfxunsetbit", RSHan_ObjRFXUnsetBit, 2},
	{"objrfx2setbit", RSHan_ObjRFX2SetBit, 2},
	{"objrfx2unsetbit", RSHan_ObjRFX2UnsetBit, 2},
	{"objsetanim", RSHan_ObjSetAnim, 2},
	{"objsetanimnum", RSHan_ObjSetAnimNum, 2},
	{"objsetaigoal", RSHan_ObjSetAIGoal, 3},
	{"objsetaienemy", RSHan_ObjSetAIEnemy, 2},
	{"objsetailookgoal", RSHan_ObjSetAILookGoal, 2},
	{"objsetaimovespeed", RSHan_ObjSetAIMoveSpeed, 2},
	{"objtalkbubble", RSHan_ObjTalkBubble, 4},
	{"objdamage", RSHan_ObjDamage, 2},
	{"objsetlight", RSHan_ObjSetLight, 2},
	{"rop_jmp", RSHan_ROP_JMP, 1},
	{"rop_jr", RSHan_ROP_JR, 0},
	{"rop_je", RSHan_ROP_JE, 1},
	{"rop_jne", RSHan_ROP_JNE, 1},
	{"rop_jge", RSHan_ROP_JGE, 1},
	{"rop_jle", RSHan_ROP_JLE, 1},
	{"rop_jg", RSHan_ROP_JG, 1},
	{"rop_jl", RSHan_ROP_JL, 1},
	{"rop_push", RSHan_ROP_PUSH, 1},
	{"rop_pop", RSHan_ROP_POP, 0},
	{"rop_popgvar", RSHan_ROP_POPGVAR, 1},
	{"rop_ldri", RSHan_ROP_LDRI, 2},
	{"rop_ldru", RSHan_ROP_LDRU, 2},
	{"rop_ldrf", RSHan_ROP_LDRF, 2},
	{"rop_ldro", RSHan_ROP_LDRO, 3},
	{"rop_ldrinv", RSHan_ROP_LDRINV, 2},
	{"rop_stro", RSHan_ROP_STRO, 3},
	{"rop_strgi", RSHan_ROP_STRGI, 2},
	{"rop_strgf", RSHan_ROP_STRGF, 2},
	{"rop_addi", RSHan_ROP_ADDI, 2},
	{"rop_addu", RSHan_ROP_ADDU, 2},
	{"rop_addf", RSHan_ROP_ADDF, 2},
	{"rop_subi", RSHan_ROP_SUBI, 2},
	{"rop_subu", RSHan_ROP_SUBU, 2},
	{"rop_subf", RSHan_ROP_SUBF, 2},
	{"rop_muli", RSHan_ROP_MULI, 2},
	{"rop_mulu", RSHan_ROP_MULU, 2},
	{"rop_mulf", RSHan_ROP_MULF, 2},
	{"rop_divi", RSHan_ROP_DIVI, 2},
	{"rop_divu", RSHan_ROP_DIVU, 2},
	{"rop_divf", RSHan_ROP_DIVF, 2},
	{"rop_shl", RSHan_ROP_SHL, 2},
	{"rop_shr", RSHan_ROP_SHR, 2},
	{"rop_and", RSHan_ROP_AND, 2},
	{"rop_or", RSHan_ROP_OR, 2},
	{"rop_xor", RSHan_ROP_XOR, 2},
	{"rop_cmpi", RSHan_ROP_CMPI, 2},
	{"rop_cmpu", RSHan_ROP_CMPU, 2},
	{"rop_cmpf", RSHan_ROP_CMPF, 2},
	{"rop_cmps", RSHan_ROP_CMPS, 2},
	{NULL, NULL, -1}
};

//allocate a script base
static rscriptBase_t *RScr_AllocScriptBase(char *name)
{
	if (g_numRScripts >= MAX_RSCRIPTS)
	{
		Util_ErrorMessage("R-Script Error: g_numRScripts >= MAX_RSCRIPTS");
		return NULL;
	}

	rscriptBase_t *scrb = (rscriptBase_t *)g_sharedFn->Common_RCMalloc(sizeof(rscriptBase_t));
	RCUBE_ASSERT(strlen(name) < 63);
	memset(scrb, 0, sizeof(rscriptBase_t));
	strcpy(scrb->name, name);
	g_rscripts[g_numRScripts] = scrb;
	g_numRScripts++;
	return scrb;
}

//free base
static void RScr_FreeScriptBase(rscriptBase_t *scrb)
{
	rscriptNode_t *n = scrb->nodes;
	while (n)
	{
		rscriptNode_t *nn = n->next;
		g_sharedFn->Common_RCFree(n);
		n = nn;
	}
	g_sharedFn->Common_RCFree(scrb);
}

//init
void RScr_Init(void)
{
	memset(g_rscripts, 0, sizeof(g_rscripts));
	g_numRScripts = 0;
	memset(g_labels, 0, sizeof(g_labels));
	g_numLabels = 0;
}

//cleanup
void RScr_Shutdown(void)
{
	for (int i = 0; i < g_numRScripts; i++)
	{
		if (g_rscripts[i])
		{
			RScr_FreeScriptBase(g_rscripts[i]);
			g_rscripts[i] = NULL;
		}
	}
	g_numRScripts = 0;
	for (int i = 0; i < g_numLabels; i++)
	{
		if (g_labels[i])
		{
			g_sharedFn->Common_RCFree(g_labels[i]);
			g_labels[i] = NULL;
		}
	}
	g_numLabels = 0;
}

//parse a define list
static rscriptDefine_t *RScr_ParseDefines(textParser_t *parser, parseToken_t *token)
{
	rscriptDefine_t *defines = NULL;
	while (token->groupName && !stricmp(token->groupName, "$__module_defines"))
	{
		rscriptDefine_t *newDef = (rscriptDefine_t *)g_sharedFn->Common_RCMalloc(sizeof(rscriptDefine_t));
		newDef->name = (char *)g_sharedFn->Common_RCMalloc((int)strlen(token->text)+1);
		strcpy(newDef->name, token->text);
		if (!g_sharedFn->Parse_GetNextToken(parser, token))
		{ //bad formatting
			RCUBE_ASSERT(0);
			return NULL;
		}
		newDef->val = (char *)g_sharedFn->Common_RCMalloc((int)strlen(token->text)+1);
		strcpy(newDef->val, token->text);

		newDef->next = defines;
		defines = newDef;
		if (!g_sharedFn->Parse_GetNextToken(parser, token))
		{
			break;
		}
	}
	return defines;
}

//see if something resolves to a define
static const char *RScr_ResolveDefine(rscriptDefine_t *moduleDefines, char *defName)
{
	if (defName[0] != '$')
	{
		return NULL;
	}

	for (rscriptDefine_t *def = moduleDefines; def; def = def->next)
	{
		if (!stricmp(def->name, defName+1))
		{
			return def->val;
		}
	}
	return NULL;
}

//free defines
static void RScr_FreeDefines(rscriptDefine_t *moduleDefines)
{
	rscriptDefine_t *def = moduleDefines;
	while (def)
	{
		rscriptDefine_t *n = def->next;
		if (def->name)
		{
			g_sharedFn->Common_RCFree(def->name);
		}
		if (def->val)
		{
			g_sharedFn->Common_RCFree(def->val);
		}
		g_sharedFn->Common_RCFree(def);
		def = n;
	}
}

//add label
static rscriptLabel_t *RScr_AddLabel(char *labelName)
{
	if (g_numLabels >= MAX_RSCRIPT_LABELS)
	{
		return NULL;
	}

	rscriptLabel_t *lab = RScr_GetLabel(labelName);
	if (lab)
	{ //already exists
		return NULL;
	}

	lab = (rscriptLabel_t *)g_sharedFn->Common_RCMalloc(sizeof(rscriptLabel_t));
	memset(lab, 0, sizeof(rscriptLabel_t));
	strcpy(lab->name, labelName);
	g_labels[g_numLabels++] = lab;

	return lab;
}

//get label
rscriptLabel_t *RScr_GetLabel(char *labelName)
{
	for (int i = 0; i < g_numLabels; i++)
	{
		rscriptLabel_t *lab = g_labels[i];
		if (!strcmp(lab->name, labelName))
		{
			return lab;
		}
	}

	return NULL;
}

//parse r-script
void RScr_ParseScriptFile(const char *filename)
{
	textParser_t *parser = g_sharedFn->Parse_InitParserFromFile(filename);
	if (!parser)
	{
		return;
	}
	g_sharedFn->Parse_EnableInclude(parser, "$__include");

	parseToken_t token;
	char groupName[MAX_TOKEN_SIZE];
	groupName[0] = 0;
	rscriptBase_t *curBase = NULL;
	rscriptNode_t **nodePtr = NULL;
	rscriptDefine_t *moduleDefines = NULL;
	rscriptLabel_t *lastLabel = NULL;
	while (g_sharedFn->Parse_GetNextToken(parser, &token))
	{
		if (token.groupName && !stricmp(token.groupName, "$__module_defines"))
		{ //parse module defines out
			RScr_FreeDefines(moduleDefines);
			moduleDefines = RScr_ParseDefines(parser, &token);
			if (!token.text[0])
			{ //nothing after the define list
				break;
			}
		}

		if (token.groupName && stricmp(token.groupName, groupName))
		{
			if (strlen(token.groupName) >= MAX_RSCRIPT_NAME_STR-1)
			{
				Util_ErrorMessage("R-Script Error: Script name too long: '%s'.", token.groupName);
				break;
			}
			strcpy(groupName, token.groupName);
			curBase = RScr_AllocScriptBase(groupName);
			nodePtr = &curBase->nodes;
			lastLabel = NULL;
		}

		if (curBase)
		{
			int l = token.lineNum;
			if (!strcmp(token.text, "$l:"))
			{ //create a label
				if (!g_sharedFn->Parse_GetNextToken(parser, &token) || token.lineNum != l)
				{
					Util_ErrorMessage("R-Script Error: %s, line %i, bad label syntax.", filename, l+1);
					break;
				}
				if (strlen(token.text) >= MAX_RSCRIPT_NAME_STR-1)
				{
					Util_ErrorMessage("R-Script Error: Label name too long: '%s'.", token.text);
					break;
				}
				lastLabel = RScr_AddLabel(token.text);
				if (!lastLabel)
				{
					Util_ErrorMessage("R-Script Error: %s, line %i, duplicate or too many labels.", filename, l+1);
					break;
				}
			}
			else
			{ //look for a script command handler
				rscriptHandler_t *handler = NULL;
				for (int i = 0; g_scriptHandlers[i].name; i++)
				{
					if (!stricmp(token.text, g_scriptHandlers[i].name))
					{ //found the handler for this node command
						handler = &g_scriptHandlers[i];
						break;
					}
				}
				if (handler)
				{
					rscriptNode_t *newNode = (rscriptNode_t *)g_sharedFn->Common_RCMalloc(sizeof(rscriptNode_t));
					memset(newNode, 0, sizeof(rscriptNode_t));
					newNode->nodeHandler = handler;
					if (lastLabel)
					{ //set the label's node
						lastLabel->node = newNode;
						lastLabel = NULL;
					}
					for (int i = 0; i < handler->argNum && g_sharedFn->Parse_GetNextToken(parser, &token); i++)
					{ //parse out the handler arguments
						if (token.lineNum != l)
						{
							Util_ErrorMessage("R-Script Error: %s, line %i, wrong number of arguments.", filename, l+1);
							break;
						}
						if (strlen(token.text) >= MAX_RSCRIPT_NAME_STR-1)
						{
							Util_ErrorMessage("R-Script Error: Argument too long: '%s'.", token.text);
							break;
						}
						const char *defStr = RScr_ResolveDefine(moduleDefines, token.text);
						const char *argText = defStr ? defStr : token.text;
						strcpy(newNode->menuArgs[i].p, argText);
					}
					*nodePtr = newNode;
					nodePtr = &newNode->next;
				}
				else
				{
					Util_ErrorMessage("R-Script Error: %s, line %i, unknown command '%s'.", filename, l+1, token.text);
					break;
				}
			}
		}
	}

	RScr_FreeDefines(moduleDefines);
	g_sharedFn->Parse_FreeParser(parser);
}

//create new script object
rscript_t *RScr_CreateScript(const char *scriptName)
{
	rscriptBase_t *scrb = NULL;
	for (int i = 0; i < g_numRScripts; i++)
	{
		if (!stricmp(scriptName, g_rscripts[i]->name))
		{
			scrb = g_rscripts[i];
			break;
		}
	}
	if (!scrb)
	{
		return NULL;
	}

	rscript_t *rscr = (rscript_t *)g_sharedFn->Common_RCMalloc(sizeof(rscript_t));
	memset(rscr, 0, sizeof(rscript_t));
	rscr->curScript = scrb;
	rscr->curNode = scrb->nodes;
	rscr->ropStackPtr = MAX_RSCRIPT_STACK-1;
	return rscr;
}

//free it back up
void RScr_DestroyScript(rscript_t *rscr)
{
	g_sharedFn->Common_RCFree(rscr);
}

//run script, returns false when the script is complete
bool RScr_RunScript(rscript_t *rscr, float timeMod)
{
	if (!rscr)
	{
		return false;
	}

	while (rscr->curNode && rscr->scriptTime < g_gameTime)
	{
		rscriptNode_t *node = rscr->curNode;
		if (!node->nodeHandler->handlerFunc(node, rscr, timeMod))
		{ //the node does not want to continue to the next yet
			break;
		}
		if (rscr->curNode == node)
		{ //don't step if the handler made a jump
			rscr->curNode = rscr->curNode->next;
		}
	}

	return (rscr->curNode != 0);
}
