/*
=============================================================================
Module Information
------------------
Name:			cldraw.cpp
Author:			Rich Whitehouse
Description:	client module draw routines.
=============================================================================
*/

#include "clmain.h"
#include <gl/gl.h>

//message sort function
static int LCl_StatusMessageSort(const void *arg1, const void *arg2)
{
	hudStatusMsg_t *a1 = (hudStatusMsg_t *)arg1;
	hudStatusMsg_t *a2 = (hudStatusMsg_t *)arg2;
	if (a1->time < a2->time)
	{
		return -1;
	}
	else if (a2->time < a1->time)
	{
		return 1;
	}

	return 0;
}

//display enemy health
void LCl_UpdateEnemyHealth(int health, int maxHealth, const aiDesc_t *aiDesc)
{
	g_clg.hudState.enemyHealthName = aiDesc->name;
	g_clg.hudState.enemyHealthFrac = (float)health/(float)maxHealth;
	g_clg.hudState.enemyHealthTime = g_clg.gameTime + 2000;
}

//add a talk bubble
void LCl_AddTalkBubble(int objIndex, char *talkName, char *talkText, unsigned long duration)
{
	if (!talkName[0] && !talkText[0])
	{ //kill bubbles owned by this object
		for (int i = 0; i < MAX_TALK_BUBBLES; i++)
		{
			talkBubble_t *bubble = &g_clg.hudState.talkBubbles[i];
			if (bubble->objIndex == objIndex && (bubble->time > g_clg.gameTime || bubble->indef))
			{
				bubble->time = g_clg.gameTime;
				bubble->indef = false;
			}
		}
		return;
	}

	talkBubble_t *bestBubble = &g_clg.hudState.talkBubbles[0];
	for (int i = 1; i < MAX_TALK_BUBBLES; i++)
	{
		talkBubble_t *bubble = &g_clg.hudState.talkBubbles[i];
		if (bubble->time < bestBubble->time && !bubble->indef)
		{
			bestBubble = bubble;
		}
	}

	bestBubble->objIndex = objIndex;
	strcpy(bestBubble->talkName, talkName);
	strcpy(bestBubble->talkText, talkText);
	bestBubble->time = g_clg.gameTime+duration;
	bestBubble->indef = (duration == 0);
}

//message sort function
static int LCl_TalkBubbleSort(const void *arg1, const void *arg2)
{
	talkBubble_t *a1 = (talkBubble_t *)arg1;
	talkBubble_t *a2 = (talkBubble_t *)arg2;
	if (a1->distToPlane < a2->distToPlane)
	{
		return 1;
	}
	else if (a2->distToPlane < a1->distToPlane)
	{
		return -1;
	}

	return 0;
}

//draw talk bubbles
void LCl_DrawTalkBubbles(viewFrustum_t *viewFrust, bool waitOnInput)
{
	for (int i = 0; i < MAX_TALK_BUBBLES; i++)
	{
		talkBubble_t *bubble = &g_clg.hudState.talkBubbles[i];
		if (bubble->indef)
		{
			bubble->time = g_clg.gameTime+5000;
		}

		if (bubble->time < g_clg.gameTime && bubble->scale == 0.0f)
		{
			continue;
		}
		clientObject_t *obj = &g_clientObjects[bubble->objIndex];
		if (!obj->active && bubble->time > g_clg.gameTime)
		{
			bubble->scale = 0.0f;//bubble->time = g_clg.gameTime;
			continue;
		}

		float timeFrac = (((float)g_clg.gameTime-(float)g_clg.prvTime)/50.0f)*0.15f;
		if (bubble->time < g_clg.gameTime)
		{
			bubble->scale -= timeFrac;
			if (bubble->scale < 0.0f)
			{
				bubble->scale = 0.0f;
				continue;
			}
		}
		else
		{
			bubble->scale += timeFrac;
			if (bubble->scale > 1.0f)
			{
				bubble->scale = 1.0f;
			}
		}

		float pos[3];
		Math_VecCopy(obj->projPos, pos);
		pos[2] += obj->net.maxs[2] + 128.0f;
		bubble->distToPlane = Math_PlaneDist(viewFrust->planes[4], pos);
	}

	//sort so that bubbles get drawn in order of farthest from near plane to nearest
	qsort(g_clg.hudState.talkBubbles, MAX_TALK_BUBBLES, sizeof(talkBubble_t), LCl_TalkBubbleSort);

	float talkBubbleWGoal = 300.0f;
	float talkBubbleHGoal = 100.0f;
	for (int i = 0; i < MAX_TALK_BUBBLES; i++)
	{
		talkBubble_t *bubble = &g_clg.hudState.talkBubbles[i];
		if (bubble->scale == 0.0f)
		{
			continue;
		}

		float talkBubbleW = talkBubbleWGoal*bubble->scale;
		float talkBubbleH = talkBubbleHGoal*bubble->scale;

		float pos[3];
		clientObject_t *obj = &g_clientObjects[bubble->objIndex];
		Math_VecCopy(obj->projPos, pos);
		pos[2] += obj->net.maxs[2] + 128.0f;

		float screenPos[3];
		g_sharedCl->GL_WorldToScreen(pos, screenPos);

		float dif = (talkBubbleW-g_sharedCl->GL_RatioCorrectWidth(talkBubbleW))*0.5f;

		screenPos[0] -= talkBubbleW*0.5f;
		screenPos[1] -= talkBubbleH;
		screenPos[0] += dif;

		if (screenPos[0] < 0.0f)
		{
			screenPos[0] = 0.0f;
		}
		if (screenPos[1] < 0.0f)
		{
			screenPos[1] = 0.0f;
		}

		float d = (screenPos[0]+g_sharedCl->GL_RatioCorrectWidth(talkBubbleW))-640.0f;
		if (d > 0.0f)
		{
			screenPos[0] -= d;
		}

		//RCUBE_ASSERT(bubble->talkName[0] && bubble->talkText[0]);

		g_sharedCl->RGL_Color4f(1.0f, 1.0f, 1.0f, 0.5f);
		g_sharedCl->GL_DrawRectRot(screenPos[0], screenPos[1], talkBubbleW, talkBubbleH, &g_clRes.talkBubbleBack, 90.0f);
		char str[512];
		g_sharedCl->RGL_Color4f(1.0f, 1.0f, 1.0f, 1.0f);
		sprintf(str, "*#%s\n*(1 1 1 1)%s", bubble->talkName, bubble->talkText);
		float marginW = 16.0f*bubble->scale;
		float marginH = 8.0f*bubble->scale;
		g_sharedCl->GL_DrawRawText(screenPos[0]+g_sharedCl->GL_RatioCorrectWidth(marginW), screenPos[1]+marginH,
			1.0f*bubble->scale, 1.0f*bubble->scale, str, false, screenPos[0]+g_sharedCl->GL_RatioCorrectWidth(talkBubbleW-marginW));
		RGL_Disable(GL_ALPHA_TEST);
		RGL_Enable(GL_BLEND);

		if (waitOnInput && bubble->indef)
		{
			float scofs = sinf((float)g_clg.gameTime*0.006f)*2.0f;
			g_sharedCl->GL_DrawRect(screenPos[0]+g_sharedCl->GL_RatioCorrectWidth(talkBubbleW-21.0f-scofs*0.5f),
				screenPos[1]+talkBubbleH-18.0f-scofs*0.5f, 18.0f+scofs, 18.0f+scofs, &g_clRes.buttonJ, false, false);
		}
	}
}

//draw offscreen players as icons
void LCl_DrawClientIcons(viewFrustum_t *viewFrust)
{
	clientObject_t *frustClients[MAX_NET_CLIENTS];
	int numFrusClients = 0;
	for (int i = 0; i < MAX_NET_CLIENTS; i++)
	{
		clientObject_t *cl = &g_clientObjects[i];
		if (cl->active)
		{
			frustClients[numFrusClients] = cl;
			numFrusClients++;
		}
	}
	if (numFrusClients <= 1)
	{
		return;
	}

	for (int i = 0; i < numFrusClients; i++)
	{
		clientObject_t *cl = frustClients[i];
		float clPos[3];
		Math_VecCopy(cl->projPos, clPos);
		clPos[2] += 300.0f;
		if ((cl->net.renderEffects2 & FXFL2_PLICON) || !Math_PointInFrustum(viewFrust, clPos, 0.0f))
		{
			float screenPos[3];
			g_sharedCl->GL_WorldToScreen(clPos, screenPos);

			float icoWidth = 64.0f;
			float icoHeight = 64.0f;

			if (screenPos[0]+(icoWidth*0.5f) > 640.0f)
			{
				screenPos[0] = 640.0f-g_sharedCl->GL_RatioCorrectWidth(icoWidth*0.5f);
			}
			else if (screenPos[0]-(icoWidth*0.5f) < 0.0f)
			{
				screenPos[0] = g_sharedCl->GL_RatioCorrectWidth(icoWidth*0.5f);
			}
			if (screenPos[1]+(icoHeight*0.5f) > 480.0f)
			{
				screenPos[1] = 480.0f-(icoHeight*0.5f);
			}
			else if (screenPos[1]-(icoHeight*0.5f) < 0.0f)
			{
				screenPos[1] = icoHeight*0.5f;
			}

			g_sharedCl->RGL_Color4f(1.0f, 1.0f, 1.0f, 0.9f);
			g_sharedCl->GL_DrawRectRot(screenPos[0]-g_sharedCl->GL_RatioCorrectWidth(icoWidth*0.5f), screenPos[1]-icoHeight*0.5f,
				icoWidth, icoHeight, &g_clRes.hudP2Icon, 0.0f);
			g_sharedCl->RGL_Color4f(1.0f, 1.0f, 1.0f, 1.0f);
		}
	}
}

//jump icon
void LCl_DrawPlayerUse(clientObject_t *obj, viewFrustum_t *viewFrust)
{
	if (!obj->active)
	{
		return;
	}

	float pos[3];
	Math_VecCopy(obj->projPos, pos);
	pos[2] += obj->net.maxs[2] + 196.0f;

	float screenPos[3];
	g_sharedCl->GL_WorldToScreen(pos, screenPos);

	float scofs = sinf((float)g_clg.gameTime*0.006f)*3.0f;
	float icoW = 32.0f;
	float icoH = 32.0f;
	float icoX = screenPos[0]-g_sharedCl->GL_RatioCorrectWidth(icoW*0.5f+scofs*0.5f);
	float icoY = screenPos[1]-icoH*0.5f-scofs*0.5f;
	if (icoX < 0.0f)
	{
		icoX = 0.0f;
	}
	if (icoY < 0.0f)
	{
		icoY = 0.0f;
	}
	g_sharedCl->GL_DrawRect(icoX, icoY, icoW+scofs, icoH+scofs, &g_clRes.buttonJ, false, false);
}

//add status message
void LCl_AddStatusMessage(const char *msg, unsigned long duration)
{
	int best = -1;
	unsigned long bestTime;
	hudStatusMsg_t *m;
	for (int i = 0; i < MAX_HUD_STATUS_MSG; i++)
	{
		m = &g_clg.hudState.statusMessages[i];
		if (best == -1 || m->time < bestTime)
		{
			best = i;
			bestTime = m->time;
		}
	}
	RCUBE_ASSERT(best != -1);

	m = &g_clg.hudState.statusMessages[best];
	strcpy(m->str, msg);
	m->curY = 481.0f;
	m->time = g_clg.gameTime+duration;

	qsort(g_clg.hudState.statusMessages, MAX_HUD_STATUS_MSG, sizeof(hudStatusMsg_t), LCl_StatusMessageSort);
}

//draw credits
void LCl_DrawCredits(float ofs)
{
	typedef struct creditSeg_s
	{
		char		*text;
		float		yOfs;
		float		sizeX;
		float		sizeY;
	} creditSeg_t;
	creditSeg_t creditTexts[] =
	{
		{"*(1 1 1 1)AVALANCHE", 0.0f, 2.5f, 2.5f},
		{"Presented by Rich Whitehouse", 40.0f, 1.5f, 1.5f},
		{"*(1 1 1 1)Programming and Technology\n*(-1 -1 -1 1)Rich Whitehouse", 200.0f, 2.0f, 2.0f},
		{"*(1 1 1 1)Featuring music from *(0.5 0.5 1.0 1.0)Final Fantasy VII: Voices of the Lifestream*(1 1 1 1) and *(1 0.25 0.25 1)Nine Inch Nails\n*(-1 -1 -1 1)http://ff7.ocremix.org/\nhttp://www.nin.com/", 300.0f, 2.0f, 2.0f},
		{"*(1 1 1 1)In order of appearance.\n\n\n\n\
*(1 0.5 0 1)Nomura Limit (Fight On!)\n\
*(1 1 1 1)Artists: *(-1 -1 -1 1)zircon\n\
\n\n\n\n\
*(1 0.5 0 1)No Such Thing As the Promised Land (Mako Reactor)\n\
*(1 1 1 1)Artists: *(-1 -1 -1 1)sephfire, sgx\n\
\n\n\n\n\
*(1 0.5 0 1)Full Frontal Assault (Let the Battles Begin!)\n\
*(1 1 1 1)Artists: *(-1 -1 -1 1)norg, SnappleMan\n\
\n\n\n\n\
*(1 0.5 0 1)Serenity (Main Theme of FINAL FANTASY VII)\n\
*(1 1 1 1)Artists: *(-1 -1 -1 1)Mustin\n\
\n\n\n\n\
*(1 0.5 0 1)Frozen Landscape (Buried in Snow)\n\
*(1 1 1 1)Artists: *(-1 -1 -1 1)Tweek\n\
\n\n\n\n\
*(1 0.5 0 1)Sephiroth's Wake (Trail of Blood)\n\
*(1 1 1 1)Artists: *(-1 -1 -1 1)Tweek\n\
\n\n\n\n\
*(1 0.5 0 1)Kweh! (Electric de Chocobo)\n\
*(1 1 1 1)Artists: *(-1 -1 -1 1)Darangen\n\
\n\n\n\n\
*(1 0.5 0 1)Lights in the Sky\n\
*(1 1 1 1)Artists: *(-1 -1 -1 1)Nine Inch Nails\n\
\n\n\n\n\
*(1 0.5 0 1)Omnislash (Hurry Up!)\n\
*(1 1 1 1)Artists: *(-1 -1 -1 1)Fishy\n\
\n\n\n\n\
*(1 0.5 0 1)Beginning of the End (Birth of a God)\n\
*(1 1 1 1)Artists: *(-1 -1 -1 1)bLiNd\n\
\n\n\n\n\
*(1 0.5 0 1)Adrenalyne Kyck (Hurry!)\n\
*(1 1 1 1)Artists: *(-1 -1 -1 1)Big Giant Circles, Liontamer, zircon\n\
\n\n\n\n\
*(1 0.5 0 1)Discipline\n\
*(1 1 1 1)Artists: *(-1 -1 -1 1)Nine Inch Nails\n\
\n\n\n\n\
*(1 0.5 0 1)Head Down\n\
*(1 1 1 1)Artists: *(-1 -1 -1 1)Nine Inch Nails\n\
\n\n\n\n\
*(1 1 1 1)Design, Concept, Animation, 2D Art, Textures, Original Models, Original Sounds:\n\
\n\
*(-1 -1 -1 1)Rich Whitehouse\n\
\n\n\n\n\
*(1 1 1 1)Some sound effect samples make use of source material from the following Freesound users:\n\n\
*(-1 -1 -1 1)RHumphries\n\n\
carrigsound\n\n\
Syna-Max\n\n\
inferno\n\n\
FreqMan\n\n\
scarbelly\n\n\
Stephan Matson\n\n\
Abyssmal\n\n\
zerolagtime\n\n\
ERH\n\n\
alienbomb\n\n\
HerbertBoland\n\n\
aust_paul\n\n\
harri\n\n\
gezortenplotz\n\n\
Erdie\n\n\
Dalibor\n\n\
hello_flowers\n\n\
Halleck\n\n\
LAGtheNoggin\n\n\
\n\n\
*(1 1 1 1)http://freesound.iua.upf.edu/\n\
\n\n\n\n\
*(1 1 1 1)Additional source models were provided by the following turbosquid users (included models may be modified, animated, weighted, or re-textured outside of original specifications):\n\
\n\n\
*(1 1 1 1)Razor Sword (Omega, Ein-Sof)\n\
*(-1 -1 -1 1)Trixigt 3D\n\
\n\n\
*(1 1 1 1)Buster Sword (Buster-Yesod, Buster-Kor)\n\
*(-1 -1 -1 1)Dannyboy374\n\
\n\n\
*(1 1 1 1)Axe (Midget Force)\n\
*(-1 -1 -1 1)dexsoft\n\
\n\n\
*(1 1 1 1)Katana (Masamune)\n\
*(-1 -1 -1 1)TogaMario\n\
\n\n\
*(1 1 1 1)Chocobo\n\
*(-1 -1 -1 1)enryke01\n\
\n\n\
*(1 1 1 1)Glock\n\
*(-1 -1 -1 1)lkraan\n\
\n\n\n\n\
*(1 1 1 1)Special Thanks:\n\n\
*(-1 -1 -1 1)Shoko Nakagawa\n\n\
The original Final Fantasy VII team\n\n\
Hondo\n\n\
Josh Heidorn (testing/feedback)\n\n\
Bryan Shaw (testing/feedback)\n\n\
Mike Majernik (testing/feedback)\n\n\
\n\n\n\n\n\n\n\
*(1 1 1 1)Some character animations and models were reconstructed based on original Final Fantasy VII materials and designs.\n\
All original media and concepts from Final Fantasy VII are copyright their original owners.\n\
\n\n\n\n\n\n\n\
*(1 1 1 1)AVALANCHE is not supported by Square Enix, and is not officially related to the Final Fantasy franchise in any way.\n\
\n\n\n\n\n\n\n\
*(1 0 0.75 1)No chocobos were harmed during the making of this game.\n\
\n\n\n\n\n\n\n\n\
*(1 1 1 1)Thanks for playing.\n\
www.richwhitehouse.com\n\
", 500.0f, 2.0f, 2.0f},
		{NULL, 0, 0, 0}
	};

	for (int i = 0; creditTexts[i].text; i++)
	{
		creditSeg_t *cred = &creditTexts[i];
		g_sharedCl->GL_DrawRawText(20.0f, (520.0f+cred->yOfs)-ofs, cred->sizeX, cred->sizeY,
			cred->text, false, 640.0f);
	}

	RGL_Disable(GL_ALPHA_TEST);
	RGL_Enable(GL_BLEND);
}

//draw status messages
void LCl_DrawHudStatusMessages(void)
{
	float statBoxXBase = 32.0f;
	float statBoxY = 450.0f;
	float statLineH = 24.0f;

	if (g_clg.hudState.errorMessage[0])
	{ //the server is angry about something
		statBoxY -= statLineH*2.0f;
		g_sharedCl->GL_DrawRawText(g_sharedCl->GL_RatioCorrectWidth(statBoxXBase), statBoxY, 1.5f, 1.5f,
			g_clg.hudState.errorMessage, false, g_sharedCl->GL_RatioCorrectWidth(640.0f));
		RGL_Disable(GL_ALPHA_TEST);
		RGL_Enable(GL_BLEND);
	}

	float timeFrac = ((float)g_clg.gameTime-(float)g_clg.prvTime)/50.0f;
	for (int i = 0; i < MAX_HUD_STATUS_MSG; i++)
	{
		hudStatusMsg_t *m = &g_clg.hudState.statusMessages[i];
		//just here in case i want to do some x movement
		m->idealX = m->curX = statBoxXBase;
		if (!m->str[0])
		{
			m->curY = 481.0f;
		}
		if (m->time <= g_clg.gameTime)
		{
			m->idealY = 481.0f;
		}
		else
		{
			statBoxY -= statLineH;
			m->idealY = statBoxY;
		}

		m->curY += (m->idealY-m->curY)*0.4f*timeFrac;
		if (m->curY < 481.0f && m->str[0])
		{
			g_sharedCl->GL_DrawRawText(g_sharedCl->GL_RatioCorrectWidth(m->curX), m->curY, 1.5f, 1.5f,
				m->str, false, 0.0f);
			RGL_Disable(GL_ALPHA_TEST);
			RGL_Enable(GL_BLEND);
		}
	}
}

//hud draw function
void LCl_DrawHudMakoCharge(float x, float y, float angle, float makoPulse, float &chargeStrength)
{
	if (chargeStrength > 0.0f)
	{
		float timeDif = ((float)g_clg.gameTime-(float)g_clg.prvTime)/50.0f;
		float makoChargeGlowSize = 10.0f + 45.0f*chargeStrength;
		RGL_BlendFunc(GL_ONE, GL_ONE);
		g_sharedCl->RGL_Color4f(0.25f, 1.0f, 0.45f, 1.0f);
		g_sharedCl->GL_DrawRectRot(g_sharedCl->GL_RatioCorrectWidth(x-makoChargeGlowSize*0.5f), y-makoChargeGlowSize*0.5f, 32+makoChargeGlowSize, 32+makoChargeGlowSize, &g_clRes.hudMakoGlow, 0.0f);
		RGL_BlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

		chargeStrength -= timeDif*0.08f;
		if (chargeStrength < 0.0f)
		{
			chargeStrength = 0.0f;
		}
	}

	g_sharedCl->RGL_Color4f(0.5f, 1.0f, 0.5f, 1.0f);
	g_sharedCl->GL_DrawRectRot(g_sharedCl->GL_RatioCorrectWidth(x), y, 32, 32, &g_clRes.hudChargeSphere, angle);

	RGL_BlendFunc(GL_ONE, GL_ONE);
	g_sharedCl->RGL_Color4f(0.5f*makoPulse, 1.0f*makoPulse, 0.5f*makoPulse, 1.0f);
	g_sharedCl->GL_DrawRectRot(g_sharedCl->GL_RatioCorrectWidth(x), y, 32, 32, &g_clRes.hudChargeSphere, angle);
	RGL_BlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

	g_sharedCl->RGL_Color4f(1.0f, 1.0f, 1.0f, 1.0f);
}

//draw charges
void LCl_DrawHudMakoCharges(clientObject_t *obj, int makoCharges)
{
	//draw charge spheres
	float makoPulse;
	if (obj->net.renderEffects & FXFL_AURA)
	{
		makoPulse = (0.75f + sinf((float)g_clg.gameTime*0.006f)*0.25f)*1.0f;
	}
	else
	{
		makoPulse = (0.75f + sinf((float)g_clg.gameTime*0.003f)*0.25f)*0.7f;
	}

	for (int i = g_clg.hudState.lastMakoChargeCount+1; i <= makoCharges; i++)
	{
		g_clg.hudState.makoChargeStrength[i-1] = 1.0f;
	}
	g_clg.hudState.lastMakoChargeCount = makoCharges;

	if (makoCharges >= 1)
	{
		LCl_DrawHudMakoCharge(32.0f, 12.0f, 80.0f, makoPulse, g_clg.hudState.makoChargeStrength[0]);
	}
	if (makoCharges >= 2)
	{
		LCl_DrawHudMakoCharge(43.0f, 36.0f, 50.0f, makoPulse, g_clg.hudState.makoChargeStrength[1]);
	}
	if (makoCharges >= 3)
	{
		LCl_DrawHudMakoCharge(44.0f, 63.0f, 30.0f, makoPulse, g_clg.hudState.makoChargeStrength[2]);
	}
	if (makoCharges >= 4)
	{
		LCl_DrawHudMakoCharge(34.0f, 89.0f, 0.0f, makoPulse, g_clg.hudState.makoChargeStrength[3]);
	}
}

//draw equipped item
void LCl_DrawHudEquippedItem(int equippedItem)
{
	playerInvItem_t *plItem = &g_clg.clientState.localPl->inventory[equippedItem];
	if (plItem->itemQuantity <= 0)
	{
		return;
	}

	const invItemDef_t *item = &g_invItems[plItem->itemIndex];
	if (!item->icon || !item->icon[0])
	{
		return;
	}

	g_sharedCl->GL_DrawRect(g_sharedCl->GL_RatioCorrectWidth(85.0f), 90, 26, 26, &g_clRes.invItemIcons[plItem->itemIndex], false, false);
	g_sharedCl->GL_DrawRawText(g_sharedCl->GL_RatioCorrectWidth(85.0f+26), 95, 1.0f, 1.0f,
		"*(1 1 1 1)x", false, 0.0f);
	char str[256];
	sprintf(str, "*(1 1 1 1)%02i", plItem->itemQuantity);
	g_sharedCl->GL_DrawRawText(g_sharedCl->GL_RatioCorrectWidth(85.0f+36), 93, 1.25f, 1.25f,
		str, false, 0.0f);
	RGL_Disable(GL_ALPHA_TEST);
	RGL_Enable(GL_BLEND);

	if (g_clg.hudState.itemNameTime > g_clg.gameTime)
	{
		float nameFade = 1.0f;
		if ((g_clg.hudState.itemNameTime-1000) < g_clg.gameTime)
		{
			nameFade = (float)(g_clg.hudState.itemNameTime-g_clg.gameTime) / 1000.0f;
		}
		sprintf(str, "*(1 1 1 %f)%s", nameFade, item->name);
		g_sharedCl->GL_DrawRawText(g_sharedCl->GL_RatioCorrectWidth(85.0f+36), 110, 1.25f, 1.25f,
			str, false, 0.0f);
		RGL_Disable(GL_ALPHA_TEST);
		RGL_Enable(GL_BLEND);
	}
}

//life meter
void LCl_DrawHudKnuckles(int health, int maxHealth)
{
	float lifeW, lifeH;
	float lifeMaxW;
	int numLifeChunks = (int)ceilf((float)maxHealth/50.0f);
	const int maxLifeChunksRow = 10;
	int i;

	float glowColor[4] = {0.25f, 0.5f, 0.25f, 0.0f};
	if (health < (maxHealth/4))
	{
		glowColor[0] = 1.0f;
		glowColor[1] = 0.0f;
		glowColor[2] = 0.0f;
		float pulse = sinf((float)g_clg.gameTime*0.007f);
		for (int i = 0; i < 3; i++)
		{
			glowColor[i] *= (0.65f + (pulse*0.35f));
		}
	}
	else
	{
		float pulse = sinf((float)g_clg.gameTime*0.003f);
		for (int i = 0; i < 3; i++)
		{
			glowColor[i] *= (0.85f + (pulse*0.15f));
		}
	}

	//draw the glowing background items
	g_sharedCl->RGL_Color4fv(glowColor);
	RGL_BlendFunc(GL_ONE, GL_ONE);
	g_sharedCl->GL_DrawRect(g_sharedCl->GL_RatioCorrectWidth(45.0f), 24, 128, 64, &g_clRes.hudLifePieceGlow, false, false);
	if (numLifeChunks > maxLifeChunksRow)
	{
		g_sharedCl->GL_DrawRect(g_sharedCl->GL_RatioCorrectWidth(45.0f), 42, 128, 64, &g_clRes.hudLifePieceGlow, false, true);
	}
	g_sharedCl->GL_DrawRect(g_sharedCl->GL_RatioCorrectWidth(-20), -10, 150, 150, &g_clRes.hudKnuckleGlow, false, false);
	lifeW = g_sharedCl->GL_RatioCorrectWidth(15.0f+96.0f);
	lifeH = 42.0f;
	lifeMaxW = lifeW;
	for (i = 0; i < numLifeChunks; i++)
	{
		if (i > 0 && (i%maxLifeChunksRow) == 0)
		{
			lifeW = g_sharedCl->GL_RatioCorrectWidth(15.0f+96.0f);
			lifeH += 17.0f;
		}
		g_sharedCl->GL_DrawRect(lifeW, lifeH, 32, 30, &g_clRes.hudLifeChunkGlow, false, false);
		lifeW += g_sharedCl->GL_RatioCorrectWidth(15.0f);
		lifeMaxW = Math_Max2(lifeMaxW, lifeW);
	}
	if (numLifeChunks > maxLifeChunksRow)
	{
		g_sharedCl->GL_DrawRect(lifeMaxW-g_sharedCl->GL_RatioCorrectWidth(20.0f), 6, 60, 120, &g_clRes.hudLifeEndGlow, false, false);
	}
	else
	{
		g_sharedCl->GL_DrawRect(lifeMaxW-g_sharedCl->GL_RatioCorrectWidth(5.0f), 27, 31, 60, &g_clRes.hudLifeEndGlow, false, false);
	}

	//draw the foreground alpha tested items
	g_sharedCl->RGL_Color4f(1.0f, 1.0f, 1.0f, 1.0f);
	RGL_BlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	g_sharedCl->GL_DrawRect(g_sharedCl->GL_RatioCorrectWidth(45.0f), 24, 128, 64, &g_clRes.hudLifePiece, false, false);
	if (numLifeChunks > maxLifeChunksRow)
	{
		g_sharedCl->GL_DrawRect(g_sharedCl->GL_RatioCorrectWidth(45.0f), 42, 128, 64, &g_clRes.hudLifePiece, false, true);
	}
	g_sharedCl->GL_DrawRect(g_sharedCl->GL_RatioCorrectWidth(-20), -10, 150, 150, &g_clRes.hudKnuckle, false, false);
	lifeW = g_sharedCl->GL_RatioCorrectWidth(15.0f+96.0f);
	lifeH = 42.0f;
	lifeMaxW = lifeW;
	int buildH = health;
	int buildHRed = g_clg.hudState.healthRed;
	for (i = 0; i < numLifeChunks; i++)
	{
		if (i > 0 && (i%maxLifeChunksRow) == 0)
		{
			lifeW = g_sharedCl->GL_RatioCorrectWidth(15.0f+96.0f);
			lifeH += 17.0f;
		}
		g_sharedCl->GL_DrawRect(lifeW, lifeH, 32, 30, &g_clRes.hudLifeChunk, false, false);

		//draw the life sliver
		//RGL_BlendFunc(GL_SRC_COLOR, GL_ONE_MINUS_SRC_COLOR);
		if (buildHRed > 0)
		{
			g_sharedCl->RGL_Color4f(1.0f, 0.0f, 0.0f, 1.0f);
			float healthFrac = (buildHRed < 50) ? (float)buildHRed/50.0f : 1.0f;
			float rhFrac = 1.0f-healthFrac;
			g_sharedCl->GL_DrawRect(lifeW+g_sharedCl->GL_RatioCorrectWidth(8.0f+(4.0f*rhFrac)), lifeH+7+(1.5f*rhFrac), 15*healthFrac, 15-(3.0f*rhFrac), &g_clRes.hudLifeSliver2, false, false);
		}
		if (buildH > 0)
		{
			g_sharedCl->RGL_Color4f(0.25f, 1.0f, 0.25f, 1.0f);
			float healthFrac = (buildH < 50) ? (float)buildH/50.0f : 1.0f;
			float rhFrac = 1.0f-healthFrac;
			g_sharedCl->GL_DrawRect(lifeW+g_sharedCl->GL_RatioCorrectWidth(8.0f+(4.0f*rhFrac)), lifeH+7+(1.5f*rhFrac), 15*healthFrac, 15-(3.0f*rhFrac), &g_clRes.hudLifeSliver, false, false);
		}
		g_sharedCl->RGL_Color4f(1.0f, 1.0f, 1.0f, 1.0f);

		lifeW += g_sharedCl->GL_RatioCorrectWidth(15.0f);
		lifeMaxW = Math_Max2(lifeMaxW, lifeW);

		buildH -= 50;
		buildHRed -= 50;
	}
	if (numLifeChunks > maxLifeChunksRow)
	{
		g_sharedCl->GL_DrawRect(lifeMaxW-g_sharedCl->GL_RatioCorrectWidth(20.0f), 6, 60, 120, &g_clRes.hudLifeEnd, false, false);
	}
	else
	{
		g_sharedCl->GL_DrawRect(lifeMaxW-g_sharedCl->GL_RatioCorrectWidth(5.0f), 27, 31, 60, &g_clRes.hudLifeEnd, false, false);
	}
}

//draw charge meter
void LCl_DrawHudChargeMeter(int chargeMeter)
{
	float timeDif = ((float)g_clg.gameTime-(float)g_clg.prvTime)/50.0f;
	if (chargeMeter > 0)
	{
		g_clg.hudState.chargeAlpha += timeDif*0.4f;
		if (g_clg.hudState.chargeAlpha > 1.0f)
		{
			g_clg.hudState.chargeAlpha = 1.0f;
		}
	}
	else if (g_clg.hudState.chargeAlpha > 0.0f)
	{
		g_clg.hudState.chargeAlpha -= timeDif*0.2f;
		if (g_clg.hudState.chargeAlpha < 0.0f)
		{
			g_clg.hudState.chargeAlpha = 0.0f;
		}
	}

	if (g_clg.hudState.chargeAlpha > 0.0f)
	{
		float chargeX = 465.0f;//g_sharedCl->GL_RatioCorrectWidth(87.0f);
		float chargeY = 88.0f;//88.0f;
		float chargeFrac = (float)chargeMeter/4000.0f;
		float chargeScaleX = 1.0f;
		float chargeScaleY = 1.0f;
		float rfl1 = (float)(rand()%8192)/8192.0f;
		float rfl2 = (float)(rand()%8192)/8192.0f;

		if (chargeFrac > 0.5f)
		{
			float moveScale = (chargeFrac-0.5f)*8.0f;
			float moveOfs;
			
			moveOfs = -moveScale + rfl1*moveScale*2.0f;
			chargeX += g_sharedCl->GL_RatioCorrectWidth(moveOfs);
			moveOfs = -moveScale + rfl2*moveScale*2.0f;
			chargeY += moveOfs;
		}

		RGL_BlendFunc(GL_ONE, GL_ONE);
		float pulse = sinf((float)g_clg.gameTime*0.007f)*0.25f;
		float cr = 0.75f + pulse;
		g_sharedCl->RGL_Color4f(chargeFrac*cr, 0.0f, 0.0f, 1.0f);
		g_sharedCl->GL_DrawRect(chargeX, chargeY, 128*chargeScaleX, 32*chargeScaleY, &g_clRes.hudChargeMeterGlow, false, false);
		if (chargeFrac > 0.5f)
		{
			g_sharedCl->GL_DrawRect(chargeX, chargeY, 128*chargeScaleX, 32*chargeScaleY, &g_clRes.hudChargeMeterGlow, false, false);
		}
		RGL_BlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

		g_sharedCl->RGL_Color4f(1.0f, 1.0f, 1.0f, g_clg.hudState.chargeAlpha);
		g_sharedCl->GL_DrawRect(chargeX, chargeY, 128*chargeScaleX, 32*chargeScaleY, &g_clRes.hudChargeMeter, false, false);
		g_sharedCl->RGL_Color4f(0.5f+(chargeFrac*0.5f), 1.0f-chargeFrac, 0.0f, g_clg.hudState.chargeAlpha);
		g_sharedCl->GL_DrawRect(chargeX+g_sharedCl->GL_RatioCorrectWidth(30.0f*chargeScaleX), chargeY+10.0f*chargeScaleY, 68.0f*chargeScaleX*chargeFrac, 12.0f*chargeScaleY, &g_clRes.hudBar, false, false);
		g_sharedCl->RGL_Color4f(1.0f, 1.0f, 1.0f, 1.0f);
	}
}

//draw enemy health
void LCl_DrawEnemyHealth(void)
{
	if (g_clg.clientState.globalNet.vsNames[0][0] &&
		g_clg.clientState.globalNet.vsNames[1][0])
	{
		float healthBarX = 325.0f*(640.0f/g_sharedCl->GL_RatioCorrectWidth(640.0f));
		float healthBarY = 32.0f;//425.0f;
		float healthBarScaleX = 2.0f;
		float healthBarScaleY = 0.5f;
		int idx = (g_clg.selfClientNum == 0) ? 1 : 0;
		float healthBarFrac = (g_clg.clientState.globalNet.vsMaxHealth[idx] == 0) ? 1.0f : (float)g_clg.clientState.globalNet.vsHealth[idx]/(float)g_clg.clientState.globalNet.vsMaxHealth[idx];

		g_sharedCl->RGL_Color4f(1.0f, 1.0f, 1.0f, 1.0f);
		char str[256];
		sprintf(str, "*(1 1 1 %f)%s", 1.0f, g_clg.clientState.globalNet.vsNames[idx]);
		g_sharedCl->GL_DrawRawText(healthBarX, healthBarY, 1.25f, 1.25f,
			str, false, 0.0f);
		RGL_Enable(GL_BLEND);
		RGL_Disable(GL_ALPHA_TEST);
		healthBarY += 19.0f;
		g_sharedCl->RGL_Color4f(1.0f, 1.0f, 1.0f, 0.5f);
		g_sharedCl->GL_DrawRect(healthBarX, healthBarY, 128*healthBarScaleX, 32*healthBarScaleY, &g_clRes.hudBar, false, false);
		g_sharedCl->RGL_Color4f(0.0f, 0.0f, 0.0f, 0.5f);
		g_sharedCl->GL_DrawRect(healthBarX+g_sharedCl->GL_RatioCorrectWidth(1.0f*healthBarScaleX), healthBarY+2.0f*healthBarScaleY, 126.0f*healthBarScaleX, 28.0f*healthBarScaleY, &g_clRes.hudBar, false, false);
		g_sharedCl->RGL_Color4f(1.0f-healthBarFrac, healthBarFrac, 0.0f, 1.0f);
		g_sharedCl->GL_DrawRect(healthBarX+g_sharedCl->GL_RatioCorrectWidth(1.0f*healthBarScaleX), healthBarY+2.0f*healthBarScaleY, 126.0f*healthBarScaleX*healthBarFrac, 28.0f*healthBarScaleY, &g_clRes.hudBar, false, false);
		g_sharedCl->RGL_Color4f(1.0f, 1.0f, 1.0f, 1.0f);
		return;
	}

	if (g_clg.hudState.enemyHealthTime >= g_clg.gameTime)
	{
		g_clg.hudState.enemyHealthFade = 1.0f;
	}

	if (g_clg.hudState.enemyHealthFade <= 0.0f)
	{
		return;
	}

	float healthBarX = 365.0f;
	float healthBarY = 425.0f;
	float healthBarFrac = g_clg.hudState.enemyHealthFrac;
	float healthBarScaleX = 2.0f;
	float healthBarScaleY = 0.5f;

	g_sharedCl->RGL_Color4f(1.0f, 1.0f, 1.0f, g_clg.hudState.enemyHealthFade);
	char str[256];
	sprintf(str, "*(1 1 1 %f)%s", g_clg.hudState.enemyHealthFade, g_clg.hudState.enemyHealthName);
	g_sharedCl->GL_DrawRawText(healthBarX, healthBarY, 1.25f, 1.25f,
		str, false, 0.0f);
	RGL_Enable(GL_BLEND);
	RGL_Disable(GL_ALPHA_TEST);
	healthBarY += 19.0f;
	g_sharedCl->RGL_Color4f(1.0f, 1.0f, 1.0f, g_clg.hudState.enemyHealthFade*0.5f);
	g_sharedCl->GL_DrawRect(healthBarX, healthBarY, 128*healthBarScaleX, 32*healthBarScaleY, &g_clRes.hudBar, false, false);
	g_sharedCl->RGL_Color4f(0.0f, 0.0f, 0.0f, g_clg.hudState.enemyHealthFade*0.5f);
	g_sharedCl->GL_DrawRect(healthBarX+g_sharedCl->GL_RatioCorrectWidth(1.0f*healthBarScaleX), healthBarY+2.0f*healthBarScaleY, 126.0f*healthBarScaleX, 28.0f*healthBarScaleY, &g_clRes.hudBar, false, false);
	g_sharedCl->RGL_Color4f(1.0f-healthBarFrac, healthBarFrac, 0.0f, g_clg.hudState.enemyHealthFade);
	g_sharedCl->GL_DrawRect(healthBarX+g_sharedCl->GL_RatioCorrectWidth(1.0f*healthBarScaleX), healthBarY+2.0f*healthBarScaleY, 126.0f*healthBarScaleX*healthBarFrac, 28.0f*healthBarScaleY, &g_clRes.hudBar, false, false);
	g_sharedCl->RGL_Color4f(1.0f, 1.0f, 1.0f, 1.0f);

	float timeDif = ((float)g_clg.gameTime-(float)g_clg.prvTime)/50.0f;
	g_clg.hudState.enemyHealthFade -= timeDif*0.2f;
	if (g_clg.hudState.enemyHealthFade < 0.0f)
	{
		g_clg.hudState.enemyHealthFade = 0.0f;
	}
}

//draw stash
void LCl_DrawHudMakoStash(int makoStash)
{
	if (g_clg.hudState.makoShowTime > g_clg.gameTime)
	{ //draw mako stash count
		float makoAlpha = 1.0f;
		if (g_clg.hudState.makoShowTime < g_clg.gameTime+1000)
		{
			int d = (int)(g_clg.gameTime+1000)-(int)g_clg.hudState.makoShowTime;
			makoAlpha = 1.0f-Math_Max2((float)d/1000.0f, 0.0f);
		}
		float makoX = 500.0f;//520.0f;
		float makoY = 30.0f;
		float makoStashGlowSize = 20.0f + 35.0f*g_clg.hudState.makoStrength;
		RGL_BlendFunc(GL_ONE, GL_ONE);
		g_sharedCl->RGL_Color4f(0.25f*makoAlpha, 1.0f*makoAlpha, 0.45f*makoAlpha, 1.0f);
		g_sharedCl->GL_DrawRectRot(makoX-g_sharedCl->GL_RatioCorrectWidth(makoStashGlowSize*0.5f), makoY-makoStashGlowSize*0.5f, 28+makoStashGlowSize, 28+makoStashGlowSize, &g_clRes.hudMakoGlow, 0.0f);
		RGL_BlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
		float rot1 = sinf((float)g_clg.gameTime*0.001f)*180.0f;
		float rot2 = cosf((float)g_clg.gameTime*0.001f)*180.0f;
		g_sharedCl->RGL_Color4f(0.5f, 1.0f, 0.7f, 0.5f*makoAlpha);
		g_sharedCl->GL_DrawRectRot(makoX, makoY, 28, 28, &g_clRes.hudMako1, rot1);
		g_sharedCl->RGL_Color4f(0.5f, 1.0f, 0.7f, 0.5f*makoAlpha);
		g_sharedCl->GL_DrawRectRot(makoX, makoY, 28, 28, &g_clRes.hudMako2, rot2);
		g_sharedCl->RGL_Color4f(1.0f, 1.0f, 1.0f, 1.0f);
		char str[256];
		sprintf(str, "*(1 1 1 %f)%06i", makoAlpha, makoStash);
		g_sharedCl->GL_DrawRawText((makoX + g_sharedCl->GL_RatioCorrectWidth(34.0f)), (makoY+2.0f), 1.5f, 1.5f,
			str, false, 0.0f);
		RGL_Enable(GL_BLEND);
		RGL_Disable(GL_ALPHA_TEST);
	}
	if (g_clg.hudState.makoStrength > 0.0f)
	{
		float timeDif = ((float)g_clg.gameTime-(float)g_clg.prvTime)/50.0f;
		g_clg.hudState.makoStrength -= timeDif*0.2f;
		if (g_clg.hudState.makoStrength < 0.0f)
		{
			g_clg.hudState.makoStrength = 0.0f;
		}
	}
}

//draw endless battle score
void LCl_DrawHudEndlessScore(void)
{
	if (!g_clg.clientState.globalNet.endlScore && !g_clg.clientState.globalNet.endlHighscore)
	{
		return;
	}
	int escr = g_clg.clientState.globalNet.endlScore;
	int ehscr = g_clg.clientState.globalNet.endlHighscore;
	if (ehscr < 0)
	{
		ehscr = 0;
	}

	float x = 510.0f + 60.0f*(640.0f/g_sharedCl->GL_RatioCorrectWidth(640.0f));//520.0f;
	float y = 400.0f;

	char str[256];
	sprintf(str, "*(1 1 1 1)Score:\n*(-1 -1 -1 1)%05i\n\n*(1 1 1 1)High:\n*(-1 -1 -1 1)%05i", escr, ehscr);
	g_sharedCl->GL_DrawRawText(x, y, 1.0f, 1.0f,
		str, false, 0.0f);
	RGL_Enable(GL_BLEND);
	RGL_Disable(GL_ALPHA_TEST);
}

//fps counter
void LCl_DrawFPS(void)
{
	static int frameRate = 0;
	static int frameRateCount = 0;
	static unsigned long frameRateTime = 0;
	frameRateCount++;
	if (frameRateTime < g_clg.gameTime)
	{
		frameRate = frameRateCount;
		frameRateCount = 0;
		frameRateTime = g_clg.gameTime+1000;
	}
	char str[256];
	sprintf(str, "%i FPS", frameRate);
	g_sharedCl->GL_DrawRawText(0.0f, 0.0f, 1.25f, 1.25f,
		str, false, 0.0f);
	RGL_Disable(GL_ALPHA_TEST);
	RGL_Enable(GL_BLEND);
}

typedef struct hudTObj_s
{
	float			pos[2];
	float			size[2];
	bool			touched;
	bool			reallyTouched;
	unsigned long	touchFx;
	int				touchIdx;
	float			touchPos[2];
} hudTObj_t;

typedef enum
{
	HUDTOBJ_STICK,
	HUDTOBJ_PUNCH,
	HUDTOBJ_KICK,
	HUDTOBJ_JUMP,
	HUDTOBJ_MENU,
	NUM_HUD_TOUCHOBJ
} hudTObj_e;
static hudTObj_t g_hudTObj[NUM_HUD_TOUCHOBJ] =
{
	//{0.0f, 480.0f-140.0f-22.0f-22.0f, 140.0f + 26.0f*2, 140.0f + 44.0f, false}
	{0.0f, 480.0f-280.0f, 260.0f, 280.0f, false, false, 0, -1, 0.0f, 0.0f},
//	{640.0f-(140.0f-24.0f)*2.0f, 480.0f-238.0f, 140.0f, 140.0f, false, false, -1, 0.0f, 0.0f},
//	{640.0f-120.0f, 480.0f-238.0f, 140.0f, 140.0f, false, false, 0, -1, 0.0f, 0.0f},
//	{640.0f-177.0f, 480.0f-130.0f, 140.0f, 140.0f, false, false, 0, -1, 0.0f, 0.0f}
	{444.0f, 276.0f, 120.0f, 120.0f, false, false, 0, -1, 0.0f, 0.0f},
	{540.0f, 276.0f, 120.0f, 120.0f, false, false, 0, -1, 0.0f, 0.0f},
	{491.0f, 366.0f, 120.0f, 120.0f, false, false, 0, -1, 0.0f, 0.0f},
	{640.0f-((64.0f+8.0f)*0.85715f), 5.0f, 64.0f, 64.0f, false, false, 0, -1, 0.0f, 0.0f}
};

//touch event callback
void LCl_ProcessTouch(int touchEv, appTouch_t *touch)
{
	float x = touch->pos[0]*640.0f;
	float y = touch->pos[1]*480.0f;
	switch (touchEv)
	{
	case TOUCHEV_TOUCHDOWN:
	case TOUCHEV_TOUCHMOVE:
		{
			for (int i = 0; i < NUM_HUD_TOUCHOBJ; i++)
			{
				hudTObj_t *ho = &g_hudTObj[i];
				if (x >= ho->pos[0] && x <= ho->pos[0]+ho->size[0] &&
					y >= ho->pos[1] && y <= ho->pos[1]+ho->size[1])
				{
					if (!ho->touched)
					{
						if (i == HUDTOBJ_MENU)
						{
							waveFile_t *w = g_sharedCl->S_CacheSound("assets/sound/menu/g_move.wav");
							if (w)
							{
								g_sharedCl->S_PlayWave(w, NULL, 0.9f, false, -1, -1);
							}
						}
					}
					ho->touched = true;
					ho->touchIdx = touch->index;
					ho->touchPos[0] = x;
					ho->touchPos[1] = y;
				}
				else if (ho->touched && ho->touchIdx == touch->index)
				{
					if (ho->touched)
					{
						if (i == HUDTOBJ_MENU)
						{
							waveFile_t *w = g_sharedCl->S_CacheSound("assets/sound/menu/g_deselect.wav");
							if (w)
							{
								g_sharedCl->S_PlayWave(w, NULL, 0.9f, false, -1, -1);
							}
						}
					}
					ho->touched = false;
				}
			}
		}
		break;
	case TOUCHEV_TOUCHUP:
		for (int i = 0; i < NUM_HUD_TOUCHOBJ; i++)
		{
			hudTObj_t *ho = &g_hudTObj[i];
			if (ho->touched && ho->touchIdx == touch->index)
			{
				if (ho->touched)
				{
					if (x >= ho->pos[0] && x <= ho->pos[0]+ho->size[0] &&
						y >= ho->pos[1] && y <= ho->pos[1]+ho->size[1])
					{ //was touching
						if (i == HUDTOBJ_MENU)
						{ //menu button use
							if (g_sharedCl->Common_CanPauseToMenu())
							{
								waveFile_t *w = g_sharedCl->S_CacheSound("assets/sound/menu/g_select.wav");
								if (w)
								{
									g_sharedCl->S_PlayWave(w, NULL, 0.9f, false, -1, -1);
								}
								g_sharedCl->Menu_StartScript("activateingame");
							}
						}
					}
				}
				ho->touched = false;
			}
		}
		break;
	default:
		break;
	}
}

//draw touch button
static void LCl_DrawTouchButton(int touchBtn, float pad, texContainer_t *tex)
{
	hudTObj_t *t = &g_hudTObj[touchBtn];

	float x = t->pos[0]+pad;
	float y = t->pos[1]+pad;
	float w = t->size[0]-pad*2.0f;
	float h = t->size[1]-pad*2.0f;
	if (t->touched)
	{
		float cnt[2];
		cnt[0] = t->pos[0]+t->size[0]*0.5f;
		cnt[1] = t->pos[1]+t->size[1]*0.5f;
		float d[3];
		d[0] = t->touchPos[0]-cnt[0];
		d[1] = t->touchPos[1]-cnt[1];
		d[2] = 0.0f;
		if (Math_VecLen(d) <= t->size[0])
		{
			if (!t->reallyTouched)
			{
				t->touchFx = g_clg.gameTime + 250;
			}
			t->reallyTouched = true;
		}
		else
		{
			t->reallyTouched = false;
		}
	}
	else
	{
		t->reallyTouched = false;
	}

	if (t->reallyTouched)
	{
		g_sharedCl->RGL_Color4f(1.0f, 1.0f, 1.0f, 0.75f);
	}
	else
	{
		g_sharedCl->RGL_Color4f(1.0f, 1.0f, 1.0f, 0.35f);
	}
	g_sharedCl->GL_DrawRect(x, y, w, h, tex, false, false);

	if (t->touchFx > g_clg.gameTime)
	{
		float d = (float)(t->touchFx-g_clg.gameTime);
		float f = d/250.0f;
		g_sharedCl->RGL_Color4f(f, f, f, 1.0f);
		RGL_BlendFunc(GL_ONE, GL_ONE);
		float sizeEnd = 64.0f*(1.0f-f);
		x -= sizeEnd;
		y -= sizeEnd;
		w += sizeEnd*2.0f;
		h += sizeEnd*2.0f;
		g_sharedCl->GL_DrawRect(x, y, w, h, tex, false, false);
		RGL_BlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	}

	g_sharedCl->RGL_Color4f(1.0f, 1.0f, 1.0f, 1.0f);
}

//draw touch-screen items
static void LCl_DrawTouchInterface(void)
{
	if (!g_clTouchBased)
	{
		return;
	}

	//todo - redirect to menu input here if needed
	if (!g_sharedCl->Menu_WantTouchInput())
	{
		g_sharedCl->Common_ProcessTouchEvents(LCl_ProcessTouch);
	}

	float fade = 1.0f;
	float stickW = 140.0f;//128.0f;
	float stickH = 140.0f;//128.0f;
	float stickX = 34.0f;
	float stickY = 480.0f-stickH-32.0f;
	float stickCnt[2] = {stickX+stickW*0.5f, stickY+stickH*0.5f};
	g_sharedCl->RGL_Color4f(1.0f, 1.0f, 1.0f, fade);
	g_sharedCl->GL_DrawRect(stickX, stickY, stickW, stickH, &g_clRes.touchStickBase, false, false);

	float ofsX = 0.0f;
	float ofsY = 0.0f;
	float rangeFac = 0.0f;
	hudTObj_t *stickT = &g_hudTObj[HUDTOBJ_STICK];
	const float stickMax = 50.0f;
	WORD analogs[5] = {32767, 32767, 32767, 32767, 32767};
	if (stickT->touched)
	{
		float d[3];
		d[0] = stickT->touchPos[0]-stickCnt[0];
		d[1] = stickT->touchPos[1]-stickCnt[1];
		d[2] = 0.0f;
		float l = Math_VecNorm(d);
		if (l > stickMax)
		{
			l = stickMax;
		}
		ofsX = d[0]*l;
		ofsY = d[1]*l;
		rangeFac = l/stickMax;
		analogs[0] = (WORD)(((stickMax+ofsX)/(stickMax*2.0f))*65535.0f);
		analogs[1] = (WORD)(((stickMax+ofsY)/(stickMax*2.0f))*65535.0f);
	}

	g_sharedCl->Common_SetAnalogs(analogs);

	g_sharedCl->GL_DrawRect(stickX+ofsX, stickY+ofsY, stickW, stickH, &g_clRes.touchStickTop, false, false);

	float nf = 0.85f + sinf((float)g_clg.gameTime * 0.001f)*0.15f;
	float rg[2] = {(1.0f-rangeFac)*nf, rangeFac*nf};
	float m = Math_Max2(rg[0], rg[1]);
	if (m < 1.0f && m > 0.0f)
	{ //normalize to maximum intensity for spectrum
		m = 1.0f/m;
		rg[0] *= m;
		rg[1] *= m;
	}

	g_sharedCl->RGL_Color4f(rg[0], rg[1], 0.0f, fade);
	RGL_BlendFunc(GL_ONE, GL_ONE);
	//g_sharedCl->GL_DrawRect(stickX, stickY, stickW, stickH, &g_clRes.touchStickGlow, false, false);
	g_sharedCl->GL_DrawRectRot(stickX+ofsX, stickY+ofsY, stickW, stickH, &g_clRes.touchStickGlow, fmodf((float)g_clg.gameTime * 0.1f, 360.0f));

	g_sharedCl->RGL_Color4f(1.0f, 1.0f, 1.0f, 1.0f);
	RGL_BlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

	LCl_DrawTouchButton(HUDTOBJ_PUNCH, 12.0f, &g_clRes.buttonP);
	LCl_DrawTouchButton(HUDTOBJ_KICK, 12.0f, &g_clRes.buttonK);
	LCl_DrawTouchButton(HUDTOBJ_JUMP, 12.0f, &g_clRes.buttonJ);

	if (g_sharedCl->Common_CanPauseToMenu())
	{
		if (g_hudTObj[HUDTOBJ_PUNCH].reallyTouched)
		{
			g_sharedCl->Common_SetButton(BUTTON_ACTION, 1);
		}
		else
		{
			g_sharedCl->Common_SetButton(BUTTON_ACTION, 0);
		}

		if (g_hudTObj[HUDTOBJ_KICK].reallyTouched)
		{
			g_sharedCl->Common_SetButton(BUTTON_EVENT1, 1);
		}
		else
		{
			g_sharedCl->Common_SetButton(BUTTON_EVENT1, 0);
		}

		if (g_hudTObj[HUDTOBJ_JUMP].reallyTouched)
		{
			g_sharedCl->Common_SetButton(BUTTON_JUMP, 1);
		}
		else
		{
			g_sharedCl->Common_SetButton(BUTTON_JUMP, 0);
		}

		hudTObj_t *t = &g_hudTObj[HUDTOBJ_MENU];
		float x = t->pos[0];
		float y = t->pos[1];
		float w = t->size[0];
		float h = t->size[1];
		texContainer_t *btex = NULL;
		if (t->touched)
		{
			g_sharedCl->RGL_Color4f(1.0f, 1.0f, 1.0f, 1.0f);
			btex = &g_clRes.buttonMenuDown;
		}
		else
		{
			g_sharedCl->RGL_Color4f(1.0f, 1.0f, 1.0f, 1.0f);
			btex = &g_clRes.buttonMenu;
		}
		g_sharedCl->GL_DrawRect(x, y, w, h, btex, false, false);
		g_sharedCl->RGL_Color4f(1.0f, 1.0f, 1.0f, 1.0f);
	}
	else
	{
		for (int i = 0; i < NUM_HUD_TOUCHOBJ; i++)
		{
			g_hudTObj[i].touched = false;
			g_hudTObj[i].reallyTouched = false;
		}
		g_sharedCl->Common_SetButton(BUTTON_ACTION, 0);
		g_sharedCl->Common_SetButton(BUTTON_EVENT1, 0);
		g_sharedCl->Common_SetButton(BUTTON_JUMP, 0);
	}
}

//draw hud
void LCl_DrawHudElements(viewFrustum_t *viewFrust)
{
	clientObject_t *obj = &g_clientObjects[g_clg.selfClientNum];
	int health = (obj->net.lives&((1<<12)-1));
	int maxHealth = g_clg.clientState.localPl->maxHealth;
	int makoCharges = ((obj->net.lives>>24)&((1<<4)-1));
	int chargeMeter = ((obj->net.lives>>12)&((1<<12)-1));//((obj->net.plAmmo>>17)&((1<<12)-1));
	int equippedItem = (g_clg.clientState.desiredItem >= 0) ? g_clg.clientState.desiredItem : (obj->net.spare1&((1<<8)-1));
	int maxMakoCharges = g_clg.clientState.localPl->maxMakoCharges;
	bool waitOnInput = !!((obj->net.spare1>>12) & 1);
	bool jumpUse = !!((obj->net.spare1>>13) & 1);
	bool anyMenus = g_sharedCl->Menu_AnyActiveMenus();

	if (anyMenus)
	{
		g_clg.clientState.makoStashUpdateTime = g_clg.gameTime+1000;
	}
	if (g_clg.clientState.makoStashUpdateTime < g_clg.gameTime)
	{
		g_clg.clientState.makoStash = obj->net.plAmmo;
	}

	if (g_clg.clientState.makoStash != g_clg.hudState.lastMakoStash)
	{
		g_clg.hudState.lastMakoStash = g_clg.clientState.makoStash;
		g_clg.hudState.makoShowTime = g_clg.gameTime + 5000;
		g_clg.hudState.makoStrength = 1.0f;
	}

	if (health > g_clg.hudState.healthRed)
	{
		g_clg.hudState.healthRed = health;
	}
	else if (health < g_clg.hudState.healthRed)
	{
		float timeDif = ((float)g_clg.gameTime-(float)g_clg.prvTime)/50.0f;
		g_clg.hudState.healthRed -= (int)ceilf(timeDif*5.0f);
		if (g_clg.hudState.healthRed < health)
		{
			g_clg.hudState.healthRed = health;
		}
	}

	RGL_Disable(GL_ALPHA_TEST);
	RGL_Disable(GL_DEPTH_TEST);
	RGL_Enable(GL_BLEND);
	RGL_BlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	RGL_Disable(GL_TEXTURE_2D);
	g_sharedCl->RGL_Color4f(1.0f, 1.0f, 1.0f, 1.0f);

	RGL_Enable(GL_TEXTURE_2D);

	if (!g_clg.hudState.hideHud)
	{
		LCl_DrawHudKnuckles(health, maxHealth);
		RCUBE_ASSERT(glIsEnabled(GL_BLEND) && !glIsEnabled(GL_ALPHA_TEST));
		LCl_DrawHudMakoCharges(obj, makoCharges);
		RCUBE_ASSERT(glIsEnabled(GL_BLEND) && !glIsEnabled(GL_ALPHA_TEST));
		LCl_DrawHudEquippedItem(equippedItem);
		RCUBE_ASSERT(glIsEnabled(GL_BLEND) && !glIsEnabled(GL_ALPHA_TEST));
		LCl_DrawHudChargeMeter(chargeMeter);
		RCUBE_ASSERT(glIsEnabled(GL_BLEND) && !glIsEnabled(GL_ALPHA_TEST));
		LCl_DrawEnemyHealth();
		RCUBE_ASSERT(glIsEnabled(GL_BLEND) && !glIsEnabled(GL_ALPHA_TEST));
		if (!anyMenus)
		{
			LCl_DrawHudMakoStash(g_clg.clientState.makoStash);
		}
		RCUBE_ASSERT(glIsEnabled(GL_BLEND) && !glIsEnabled(GL_ALPHA_TEST));
		LCl_DrawHudEndlessScore();
		RCUBE_ASSERT(glIsEnabled(GL_BLEND) && !glIsEnabled(GL_ALPHA_TEST));
		LCl_DrawTouchInterface();
		RCUBE_ASSERT(glIsEnabled(GL_BLEND) && !glIsEnabled(GL_ALPHA_TEST));
	}
	RCUBE_ASSERT(glIsEnabled(GL_BLEND) && !glIsEnabled(GL_ALPHA_TEST));
	LCl_DrawTalkBubbles(viewFrust, waitOnInput);
	LCl_DrawClientIcons(viewFrust);
	if (jumpUse)
	{
		LCl_DrawPlayerUse(obj, viewFrust);
	}
	RCUBE_ASSERT(glIsEnabled(GL_BLEND) && !glIsEnabled(GL_ALPHA_TEST));

	//LCl_DrawFPS();

	g_clg.hudState.lastHealth = health;
	g_clg.hudState.lastMaxHealth = maxHealth;
	g_clg.hudState.lastMakoChargeCountMax = maxMakoCharges;
	g_clg.hudState.lastEquippedItem = equippedItem;

	LCl_UpdateStatMenu();

	if (g_clg.hudState.screenFade > 0.0f)
	{
		RGL_Disable(GL_TEXTURE_2D);
		g_sharedCl->RGL_Color4f(g_clg.hudState.screenFadeColor[0], g_clg.hudState.screenFadeColor[1],
			g_clg.hudState.screenFadeColor[2], g_clg.hudState.screenFade);
		g_sharedCl->GL_DrawRect(0.0f, 0.0f, 1000.0f, 1000.0f, NULL, false, false);
	}
	if (g_clg.hudState.screenPicFade > 0.0f && g_clg.hudState.screenPic.inuse)
	{
		float xofs = (640.0f-g_sharedCl->GL_RatioCorrectWidth(640.0f))*0.5f;
		RGL_Enable(GL_TEXTURE_2D);
		g_sharedCl->RGL_Color4f(1.0f, 1.0f, 1.0f, g_clg.hudState.screenPicFade);
		g_sharedCl->GL_DrawRect(xofs, 0.0f, 640.0f, 480.0f, &g_clg.hudState.screenPic, false, false);
	}

	if (g_clg.clientState.globalNet.creditRoll > 0.0f)
	{
		LCl_DrawCredits(g_clg.clientState.globalNet.creditRoll);
	}

	//still draw status messages over everything
	LCl_DrawHudStatusMessages();
	RCUBE_ASSERT(glIsEnabled(GL_BLEND) && !glIsEnabled(GL_ALPHA_TEST));

    RGL_Disable(GL_BLEND);
	RGL_Enable(GL_DEPTH_TEST);

	g_sharedCl->RGL_Color4f(1.0f, 1.0f, 1.0f, 1.0f);

	RGL_Disable(GL_TEXTURE_2D);
}
