/*
=============================================================================
Module Information
------------------
Name:			ai_anim.cpp
Author:			Rich Whitehouse
Description:	ai animation routines
=============================================================================
*/

#include "main.h"
#include "ai.h"

//start anim
void AI_StartAnim(gameObject_t *obj, int anim, bool restart, bool transitional)
{
	if (obj->animNeverInterrupt)
	{
		return;
	}

	gameAnim_t *animObj = obj->animTable+anim;
	if (obj->curAnim == anim)
	{
		obj->animRestart = restart;
	}
	else
	{
		obj->curAnim = anim;
		obj->animRestart = !transitional;
	}
	obj->atkLastSeq++;
}

//pick the active animation
void AI_PickAnim(gameObject_t *obj, float timeMod)
{
	gameAnim_t *curAnim = obj->animTable+obj->curAnim;

	if (obj->curAnim == HUMANIM_JUMP)
	{
		if (obj->net.frame == curAnim->endFrame && !obj->animRestart)
		{
			AI_StartAnim(obj, HUMANIM_FALL, true);
		}
	}
	else if (obj->curAnim == HUMANIM_FALL)
	{
		if (obj->onGround || (obj->aiObj && obj->aiObj->fly))
		{
			AI_StartAnim(obj, HUMANIM_LAND, true);
		}
	}
	else if (obj->curAnim == HUMANIM_FLYBACK || obj->curAnim == HUMANIM_FLYBACK2)
	{
		if ((obj->onGround || (obj->aiObj && obj->aiObj->fly)) && obj->net.frame == curAnim->endFrame && !obj->animRestart && obj->knockTime < g_glb.gameTime)
		{
			AI_StartAnim(obj, HUMANIM_POPUP_LAND, true);
		}
	}
	else if (obj->curAnim == HUMANIM_PAIN_AIR)
	{
		if (obj->onGround)
		{
			AI_StartAnim(obj, HUMANIM_POPUP_LAND, true);
		}
		else
		{
			if (obj->net.frame == curAnim->endFrame && !obj->animRestart)
			{
				AI_StartAnim(obj, HUMANIM_PAIN_POPUP, false, true);
				gameAnim_t *curAnim = obj->animTable+obj->curAnim;
				obj->net.frame = curAnim->endFrame;
			}
		}
	}
	else if (obj->curAnim == HUMANIM_POPUP_LAND)
	{
		if (obj->net.frame == curAnim->endFrame && !obj->animRestart)
		{
			AI_StartAnim(obj, HUMANIM_GETUP_BACK, true);
		}
	}
	else if (obj->curAnim == HUMANIM_FALL_LAND)
	{
		if (obj->net.frame == curAnim->endFrame && !obj->animRestart)
		{
			AI_StartAnim(obj, HUMANIM_GETUP_FRONT, true);
		}
	}
	else if (obj->curAnim == HUMANIM_LAND ||
		obj->curAnim == HUMANIM_GETUP_BACK ||
		obj->curAnim == HUMANIM_GETUP_FRONT ||
		(obj->curAnim >= HUMANIM_PAIN_HIGH1 && obj->curAnim <= HUMANIM_PAIN_LOW2))
	{
		if (obj->net.frame == curAnim->endFrame && !obj->animRestart)
		{
			AI_StartAnim(obj, HUMANIM_IDLE, true);
		}
	}
	else if (obj->curAnim == HUMANIM_PAIN_POPUP)
	{
		if ((obj->onGround || (obj->aiObj && obj->aiObj->fly)) && obj->net.frame >= curAnim->startFrame+2 && !obj->animRestart)
		{
			AI_StartAnim(obj, HUMANIM_POPUP_LAND, true);
		}
	}
	else if (obj->curAnim == HUMANIM_HIT_FALLFORWARD)
	{
		if ((obj->onGround || (obj->aiObj && obj->aiObj->fly)) && obj->net.frame >= curAnim->startFrame+2 && !obj->animRestart)
		{
			AI_StartAnim(obj, HUMANIM_FALL_LAND, true);
		}
	}
	else if (obj->curAnim == HUMANIM_DEATH)
	{ //wait in death
		if (obj->net.frame == curAnim->endFrame && !obj->animRestart)
		{
			obj->think = AI_DropMako;
			obj->thinkTime = g_glb.gameTime + (int)(1000.0f*g_glb.timeScale);
		}
	}
	else if (obj->curAnim >= NUM_HUMAN_ANIMS)
	{ //per-creature animation
		if (obj->animhandler)
		{
			obj->animhandler(obj, timeMod);
		}
		else if (obj->net.frame == curAnim->endFrame && !obj->animRestart)
		{
			AI_StartAnim(obj, HUMANIM_IDLE, true);
		}
	}
	else
	{
		float velLen = Math_VecLen(obj->net.vel);
		if (!obj->animidleoverride || !obj->animidleoverride(obj, timeMod))
		{
			if (obj->aiObj && obj->aiObj->fly && !obj->groundHugger)
			{
				AI_StartAnim(obj, HUMANIM_IDLE, false, true);
			}
			else if (!obj->onGround && !obj->groundHugger)
			{
				AI_StartAnim(obj, HUMANIM_FALL, false);
			}
			else if (velLen > 460.0f)
			{
				AI_StartAnim(obj, HUMANIM_RUN, false, true);
			}
			else if (velLen > 240.0f)
			{
				AI_StartAnim(obj, HUMANIM_RUNSLOW, false, true);
			}
			else if (velLen > 5.0f)
			{
				AI_StartAnim(obj, HUMANIM_WALK, false, true);
			}
			else
			{
				AI_StartAnim(obj, HUMANIM_IDLE, false);
			}
		}
	}
}

//non-angling?
bool AI_NonAnglingAnim(gameObject_t *obj)
{
	if (obj->curAnim >= HUMANIM_PAIN_HIGH1 && obj->curAnim <= HUMANIM_POPUP_LAND)
	{
		return true;
	}
	if (obj->curAnim == HUMANIM_DEATH || obj->preDeathTime)
	{
		return true;
	}

	return false;
}

//don't allow voluntary movement
bool AI_NonMovingAnim(gameObject_t *obj)
{
	return AI_NonAnglingAnim(obj);
}

//check anim angles
bool AI_AnglesForAnim(gameObject_t *obj, float *ang)
{
	if (obj->curAnim != HUMANIM_FLYBACK)
	{
		return false;
	}

	float invV[3];
	invV[0] = -obj->net.vel[0];
	invV[1] = -obj->net.vel[1];
	invV[2] = 0.0f;
	Math_VecToAngles(invV, ang);
	ang[PITCH] = 0.0f;
	ang[ROLL] = 0.0f;
	return true;
}

//pick a pain anim
void AI_PainAnim(gameObject_t *obj, const collObj_t *col)
{
	int anim;
	if (!obj->onGround && (!obj->aiObj || !obj->aiObj->fly))
	{
		if (obj->curAnim == HUMANIM_PAIN_POPUP ||
			obj->curAnim == HUMANIM_PAIN_AIR)
		{
			anim = HUMANIM_PAIN_AIR;
		}
		else
		{
			anim = HUMANIM_PAIN_POPUP;
		}
	}
	else
	{
		float c[3];
		Util_GameObjectCenter(obj, c);
		if (!col || col->endPos[2] >= c[2])
		{
			anim = HUMANIM_PAIN_HIGH1+(rand()%2);
		}
		else
		{
			anim = HUMANIM_PAIN_LOW1+(rand()%2);
		}
	}
	AI_StartAnim(obj, anim, true);
}

//some animations/frames will adjust velocity
bool AI_ApplyAnimationVelocity(gameObject_t *obj, float timeMod)
{
	float velFwd = 0.0f;
	float velRight = 0.0f;

	switch (obj->curAnim)
	{
	case HUMANIM_FLYBACK:
		/*
		{
			float velDir[3];
			float f = 64.0f*timeMod;
			Math_VecCopy(obj->net.vel, velDir);
			velDir[2] = 0.0f;
			Math_VecNorm(velDir);
			obj->net.vel[0] += velDir[0]*f;
			obj->net.vel[1] += velDir[1]*f;
			obj->net.vel[2] += velDir[2]*f;
		}
		*/
		break;
	default:
		break;
	}

	if (velFwd != 0.0f || velRight != 0.0f)
	{
		const float animVelFactor = 200.0f*timeMod;
		velFwd *= animVelFactor;
		velRight *= animVelFactor;

		float fwd[3], right[3];
		Math_AngleVectors(obj->net.ang, fwd, right, 0);
		obj->net.vel[0] += (fwd[0]*velFwd + right[0]*velRight);
		obj->net.vel[1] += (fwd[1]*velFwd + right[1]*velRight);
		obj->net.vel[2] += (fwd[2]*velFwd + right[2]*velRight);
		return true;
	}

	return false;
}

//set render effects for animations
void AI_ApplyAnimationEffects(gameObject_t *obj, float timeMod)
{
	obj->net.renderEffects &= ~FXFL_MODELTRAIL;
	obj->net.renderEffects &= ~FXFL_MODELTRAIL2;
	switch (obj->curAnim)
	{
	case HUMANIM_FLYBACK:
		if (Math_VecLen(obj->net.vel) > 300.0f)
		{
			obj->net.renderEffects |= FXFL_MODELTRAIL2;
		}
		break;
	default:
		break;
	}
}
