/*
=============================================================================
Module Information
------------------
Name:			ai_nav.cpp
Author:			Rich Whitehouse
Description:	ai navigation routines
=============================================================================
*/

#include "main.h"
#include "ai.h"

#define AI_CHOICE_MEM_SIZE		65536
#define AI_MAX_NODE_BRANCHES	32

static BYTE g_aiChoiceMem[AI_CHOICE_MEM_SIZE];
static unsigned int g_aiChoiceMemPtr = 0;

const float g_aiPathDensity = 256.0f;
static float g_nodeOfs[4][3] =
{
	{-g_aiPathDensity,	0.0f,				0.0f},
	{g_aiPathDensity,	0.0f,				0.0f},
	{0.0f,				-g_aiPathDensity,	0.0f},
	{0.0f,				g_aiPathDensity,	0.0f}
};

const int g_neighborMap[4] =
{
	AI_CHOICENODE_X_POS,
	AI_CHOICENODE_X_NEG,
	AI_CHOICENODE_Y_POS,
	AI_CHOICENODE_Y_NEG
};

//allocate choice node
static aiChoiceNode_t *AI_AllocChoiceNode(float *pos, float *dst)
{
	aiChoiceNode_t *p = (aiChoiceNode_t *)&g_aiChoiceMem[g_aiChoiceMemPtr];
	memset(p, 0, sizeof(aiChoiceNode_t));

	Math_VecCopy(pos, p->pos);
	float d[3];
	Math_VecSub(dst, p->pos, d);
	p->distToDest = Math_VecLen(d);//sqrtf(d[0]*d[0] + d[1]*d[1]);

	g_aiChoiceMemPtr += sizeof(aiChoiceNode_t);
	RCUBE_ASSERT(g_aiChoiceMemPtr < (AI_CHOICE_MEM_SIZE>>1));
	return p;
}

//allocate path node
static aiPathNode_t *AI_AllocPathNode(void)
{
	aiPathNode_t *p = (aiPathNode_t *)g_sharedFn->Common_RCMalloc(sizeof(aiPathNode_t));
	memset(p, 0, sizeof(aiPathNode_t));
	return p;
}

//distance from pos to ground for object
float AI_GroundDistance(gameObject_t *obj, float *pos, float *groundPos)
{
	collObj_t col;
	const float maxGroundDist = 8192.0f;//1024.0f;
	float d[3] = {pos[0], pos[1], pos[2]-maxGroundDist};
	g_sharedFn->Coll_MovementTranslation(obj->rcColModel, &col, pos, d, NULL, NULL);
	if (col.containsSolid)
	{
		return 0.0f;
	}
	if (!col.hit)
	{
		if (groundPos)
		{
			Math_VecCopy(d, groundPos);
		}
		return maxGroundDist;
	}

	if (groundPos)
	{
		Math_VecCopy(col.endPos, groundPos);
	}
	return col.distance;
}

//sort check nodes
static int AI_SortCheckNodes(const void *arg1, const void *arg2)
{
	aiChoiceNode_t *check1 = *(aiChoiceNode_t **)arg1;
	aiChoiceNode_t *check2 = *(aiChoiceNode_t **)arg2;
	if (check1->distToDest > check2->distToDest)
	{
		return 1;
	}
	if (check2->distToDest > check1->distToDest)
	{
		return -1;
	}
	return 0;
}

//recursively branch down choice nodes
bool AI_RecursiveChoiceNode(gameObject_t *obj, float *dst, aiChoiceNode_t *node, int &numBranches, aiChoiceNode_t **bestNode)
{
	if (numBranches >= AI_MAX_NODE_BRANCHES)
	{
		return false;
	}
	numBranches++;

	if (node->distToDest <= g_aiPathDensity*1.5f)
	{
		*bestNode = node;
		return true;
	}

	if (((*bestNode)->weight < g_aiPathDensity*4.0f && node->weight > (*bestNode)->weight) ||
		node->distToDest < (*bestNode)->distToDest)
	{ //keep track of the best node so far
		*bestNode = node;
	}

	aiChoiceNode_t *checkNeighbors[4];
	int numCheck = 0;
	for (int i = 0; i < 4; i++)
	{
		if (node->neighbors[i])
		{ //came from this direction
			continue;
		}

		float pos[3];
		Math_VecAdd(node->pos, g_nodeOfs[i], pos);
		collObj_t col;
		g_sharedFn->Coll_MovementTranslation(obj->rcColModel, &col, node->pos, pos, NULL, NULL);
		if (col.hit)
		{ //not a clear path, try climbing
			collObj_t stair;
			float up[3];
			float stairHop = 64.0f;//196.0f;
			bool climbed = false;
			Math_VecCopy(node->pos, up);
			up[2] += stairHop;
			g_sharedFn->Coll_MovementTranslation(obj->rcColModel, &stair, node->pos, up, NULL, NULL);
			if (!stair.hit)
			{
				float upb[3];
				Math_VecCopy(pos, upb);
				upb[2] += stairHop;
				g_sharedFn->Coll_MovementTranslation(obj->rcColModel, &stair, up, upb, NULL, NULL);
				if (!stair.hit)
				{
					pos[2] += stairHop;
					upb[2] = pos[2]-stairHop*8.0f;
					g_sharedFn->Coll_MovementTranslation(obj->rcColModel, &stair, pos, upb, NULL, NULL);
					if (stair.hit && !stair.containsSolid)
					{
						Math_VecCopy(stair.endPos, pos);
						climbed = true;
					}
				}
			}

			if (!climbed)
			{
				continue;
			}
		}
		AI_GroundDistance(obj, pos, pos);

		aiChoiceNode_t *neighbor = AI_AllocChoiceNode(pos, dst);
		neighbor->neighbors[g_neighborMap[i]] = node;
		int checkIdx = (i < 2) ? 2 : 0;
		for (int j = checkIdx; j < checkIdx+2; j++)
		{ //associate neighbors with the neighbor being placed
			if (!node->neighbors[j])
			{
				continue;
			}
			aiChoiceNode_t *exNeighbor = node->neighbors[j];
			if (exNeighbor->neighbors[i])
			{
				neighbor->neighbors[j] = exNeighbor->neighbors[i];
			}
		}

		neighbor->weight = node->weight+g_aiPathDensity;
		node->neighbors[i] = neighbor;
		checkNeighbors[numCheck] = neighbor;
		numCheck++;
	}

	qsort(checkNeighbors, numCheck, sizeof(checkNeighbors[0]), AI_SortCheckNodes);
	for (int i = 0; i < numCheck; i++)
	{
		//Util_DebugLine(node->pos, checkNeighbors[i]->pos);
		if (AI_RecursiveChoiceNode(obj, dst, checkNeighbors[i], numBranches, bestNode))
		{
			return true;
		}
	}

	return false;
}

//chart a path between the object's current location and the destination point
aiPath_t *AI_ChartPath(gameObject_t *obj, float *dst)
{
	g_aiChoiceMemPtr = 0;

	aiChoiceNode_t *node = AI_AllocChoiceNode(obj->net.pos, dst);
	int numBranches = 0;
	aiChoiceNode_t *bestNode = node;
	AI_RecursiveChoiceNode(obj, dst, node, numBranches, &bestNode);
	
	aiPathNode_t *pathNode = AI_AllocPathNode();
	Math_VecCopy(bestNode->pos, pathNode->pos);
	while (bestNode->weight > 0.0f)
	{
		aiChoiceNode_t *bestNeighbor = bestNode;
		for (int i = 0; i < 4; i++)
		{
			if (!bestNode->neighbors[i])
			{
				continue;
			}

			if (bestNode->neighbors[i]->weight < bestNeighbor->weight && fabsf(bestNode->pos[2]-bestNode->neighbors[i]->pos[2]) < 196.0f)
			{
				bestNeighbor = bestNode->neighbors[i];
			}
		}
		if (bestNeighbor == bestNode)
		{
			break;
		}

		aiChoiceNode_t *choiceToPath = NULL;
		if (bestNeighbor->weight == 0.0f)
		{
			choiceToPath = bestNeighbor;
		}
		else
		{
			collObj_t col;
			g_sharedFn->Coll_MovementTranslation(obj->rcColModel, &col, pathNode->pos, bestNeighbor->pos, NULL, NULL);
			if (col.hit)
			{
				choiceToPath = bestNode;
			}
		}

		if (choiceToPath)
		{ //insert a path node
			aiPathNode_t *nextPathNode = AI_AllocPathNode();
			Math_VecCopy(choiceToPath->pos, nextPathNode->pos);
			nextPathNode->next = pathNode;
			pathNode = nextPathNode;

			float d[3];
			Math_VecSub(pathNode->pos, pathNode->next->pos, d);
			pathNode->len = Math_VecLen(d);
		}

		bestNode = bestNeighbor;
	}

	aiPath_t *path = (aiPath_t *)g_sharedFn->Common_RCMalloc(sizeof(aiPath_t));
	path->nodes = pathNode;
	path->userCount = 1;

	return path;
}

//free path chain
void AI_FreePath(aiPath_t *path)
{
	if (!path)
	{
		return;
	}
	path->userCount--;
	if (path->userCount > 0)
	{ //someone is still using it
		return;
	}

	aiPathNode_t *pathNode = path->nodes;
	while (pathNode)
	{
		aiPathNode_t *n = pathNode->next;
		g_sharedFn->Common_RCFree(pathNode);
		pathNode = n;
	}
	g_sharedFn->Common_RCFree(path);
}

//move
void AI_Move(gameObject_t *obj, float timeMod, float *moveVec, bool cancelObstructed)
{
	if (AI_NonMovingAnim(obj) || obj->aiObj->noMoveTime > g_glb.gameTime)
	{
		return;
	}

	collObj_t col;
	float moveTest[3];
	moveTest[0] = obj->net.pos[0]+moveVec[0]*64.0f;
	moveTest[1] = obj->net.pos[1]+moveVec[1]*64.0f;
	moveTest[2] = obj->net.pos[2];
	g_sharedFn->Coll_MovementTranslation(obj->rcColModel, &col, obj->net.pos, moveTest, NULL, NULL);
	if (col.hit && col.hitObjectIndex >= 0)
	{
		if (cancelObstructed)
		{
			return;
		}
		gameObject_t *block = &g_gameObjects[col.hitObjectIndex];
		if (block->inuse && block->rcColModel && (block->rcColModel->clipFlags & CLIPFL_BOXMOVE))
		{ //try to go around it
			float d[3];
			Math_VecSub(obj->net.pos, block->net.pos, d);
			d[2] = 0.0f;
			Math_VecNorm(d);
			Math_VecAdd(moveVec, d, moveVec);
			Math_VecNorm(moveVec);
		}
	}

	float fwd[3];
	Math_AngleVectors(obj->net.ang, fwd, 0, 0);
	float moveSpeed = (Math_DotProduct(fwd, moveVec) < -0.25f) ? obj->aiObj->moveSpeed*0.5f : obj->aiObj->moveSpeed;
	if (!obj->onGround && !obj->aiObj->fly)
	{
		moveSpeed *= 0.15f;
	}

	float f = moveSpeed*timeMod;
	obj->net.vel[0] += moveVec[0]*f;
	obj->net.vel[1] += moveVec[1]*f;
}

//try borrowing another ai's path
aiPath_t *AI_BorrowPath(gameObject_t *obj, float *dst)
{
	float objCenter[3];
	Util_GameObjectCenter(obj, objCenter);

	for (int i = 0; i < g_glb.gameObjectSlots; i++)
	{
		gameObject_t *other = &g_gameObjects[i];
		if (!other->inuse || !other->aiObj || !other->aiObj->path)
		{
			continue;
		}

		float otherCenter[3];
		Util_GameObjectCenter(other, otherCenter);

		float d[3];
		Math_VecSub(otherCenter, objCenter, d);
		if (Math_VecLen(d) > 1024.0f)
		{ //too far
			continue;
		}

		//something else with a path, see if it's accessable
		if (!g_sharedFn->Coll_GeneralVisibilityWithBounds(objCenter, otherCenter, obj->radius,
			obj->spawnMins, obj->spawnMaxs))
		{
			continue;
		}

		//finally resort to a line trace
		collObj_t col;
		collOpt_t opt;
		memset(&opt, 0, sizeof(opt));
		opt.ignoreBoxModels = true;
		g_sharedFn->Coll_RadiusTranslation(obj->rcColModel, &col, objCenter, otherCenter, 0.0f, NULL, &opt);
		if (col.hit && col.hitObjectIndex != other->net.index)
		{
			continue;
		}

		//could possibly continue going through ai and see which has the closest path to dst, but for now,
		//tale the cheap approach
		other->aiObj->path->userCount++;
		return other->aiObj->path;
	}

	return NULL;
}

//see if it can get to a point
bool AI_CanReachPoint(gameObject_t *obj, float *pos, gameObject_t *ignore)
{
	float objCenter[3];
	Util_GameObjectCenter(obj, objCenter);
	float p[3] = {pos[0], pos[1], pos[2]+16.0f};
	if (!g_sharedFn->Coll_GeneralVisibilityWithBounds(objCenter, p, obj->radius, obj->spawnMins, obj->spawnMaxs))
	{
		return false;
	}

	collObj_t col;
	g_sharedFn->Coll_MovementTranslation(obj->rcColModel, &col, obj->net.pos, pos, NULL, NULL);
	if (col.hit)
	{
		if (!ignore || col.hitObjectIndex != ignore->net.index)
		{
			return false;
		}
	}

	return true;
}

//get the closest node on a path
aiPathNode_t *AI_ClosestPathNode(gameObject_t *obj, float *pos, aiPath_t *path)
{
	float bestLen;
	aiPathNode_t *best = NULL;
	float bestVisibleLen;
	aiPathNode_t *bestVisible = NULL;
	aiPathNode_t *pathNode = path->nodes;
	float objCenter[3];
	Util_GameObjectCenter(obj, objCenter);
	while (pathNode)
	{
		float d[3];
		Math_VecSub(obj->net.pos, pathNode->pos, d);
		float len = Math_VecLen(d);
		if (!best || len < bestLen)
		{
			best = pathNode;
			bestLen = len;
		}

		//check visibility
		float p[3] = {pathNode->pos[0], pathNode->pos[1], pathNode->pos[2]+16.0f};
		if (g_sharedFn->Coll_GeneralVisibilityWithBounds(objCenter, p, obj->radius, obj->spawnMins, obj->spawnMaxs))
		{
			collObj_t col;
			g_sharedFn->Coll_MovementTranslation(obj->rcColModel, &col, obj->net.pos, pathNode->pos, NULL, NULL);
			if (!col.hit || (obj->aiObj->goalObj && col.hitObjectIndex == obj->aiObj->goalObj->net.index))
			{
				if (!bestVisible || len < bestVisibleLen)
				{
					bestVisible = pathNode;
					bestVisibleLen = len;
				}
			}
		}

		pathNode = pathNode->next;
	}

	return bestVisible ? bestVisible : best;
}
