/*
=============================================================================
Module Information
------------------
Name:			ai_obj.cpp
Author:			Rich Whitehouse
Description:	ai object routines
=============================================================================
*/

#include "main.h"
#include "ai.h"

//free ai objects
void AI_FreeAIObject(aiObject_t *aiObj)
{
	if (aiObj->path)
	{
		AI_FreePath(aiObj->path);
	}
	g_sharedFn->Common_RCFree(aiObj);
}

//get gravity factor
float AI_GetGravity(gameObject_t *obj, float timeMod)
{
	if (obj->aiObj && obj->aiObj->forceGravity)
	{
		return obj->aiObj->forceGravity;
	}
	if (obj->net.renderEffects & FXFL_DEATH)
	{ //dead, turning into lifestream
		return 0.0f;
	}
	float normalGravity = (obj->noGravTime < g_glb.gameTime) ? /*80.0f*/100.0f : 0.0f;
	gameAnim_t *curAnim = obj->animTable+obj->curAnim;
	if (obj->net.frame < curAnim->startFrame || obj->net.frame > curAnim->endFrame)
	{
		return normalGravity;
	}

	switch (obj->curAnim)
	{
	case HUMANIM_PAIN_POPUP:
		if (obj->net.frame < curAnim->endFrame)
		{
			return 20.0f;
		}
		break;
	case HUMANIM_FLYBACK:
		//if (Math_VecLen(obj->net.vel) > 300.0f)
		if (obj->knockTime > g_glb.gameTime)
		{
			return 20.0f;
		}
		break;
	case HUMANIM_FLYBACK2:
		if (obj->knockTime > g_glb.gameTime)
		{
			return 0.0f;
		}
		break;
	case HUMANIM_HIT_FALLFORWARD:
		if (obj->net.frame < curAnim->startFrame+2 && obj->net.frame < curAnim->endFrame)
		{
			return 0.0f;
		}
		break;
	default:
		break;
	}
	return normalGravity;
}

//checks random chance factor for pain
bool AI_PainAnimChance(gameObject_t *obj)
{
	if (!obj->aiObj || !obj->aiObj->painChance)
	{
		return true;
	}
	if (obj->aiObj->painChance < 0)
	{ //never do pain anims
		return false;
	}

	if (rand()%100 < obj->aiObj->painChance)
	{
		return true;
	}
	return false;
}

//generic pain
void AI_GenericPain(gameObject_t *obj, gameObject_t *hurter, int dmg, const collObj_t *col)
{
	if (obj->aiObj && obj->aiObj->confuseTime)
	{ //snap out of it
		obj->aiObj->confuseTime = 0;
	}

	bool doPainAnim = AI_PainAnimChance(obj);
	if (obj->plObj)
	{
		const invItemDef_t *item = ObjPlayer_GetEquippedItem(obj);
		if (item)
		{
			if (item->itemBehavior == INVITEM_BEHAVIOR_NOPAIN ||
				item->itemBehavior == INVITEM_BEHAVIOR_FINALHEAVEN)
			{
				doPainAnim = false;
			}
			else if (item->itemBehavior == INVITEM_BEHAVIOR_VIEWHEALTH)
			{
				ObjPlayer_HealthDisplay(obj, hurter);
			}
		}
		ObjPlayer_ChargeDown(obj, hurter, dmg);
	}
	else if (hurter->plObj)
	{
		g_glb.ai.playerLastHitIndex = obj->net.index;
		g_glb.ai.playerLastHitTime = g_glb.gameTime;

		const invItemDef_t *item = ObjPlayer_GetEquippedItem(hurter);
		if (item)
		{
			if (item->itemBehavior == INVITEM_BEHAVIOR_VIEWHEALTH)
			{
				ObjPlayer_HealthDisplay(hurter, obj);
			}
		}
	}

	if (hurter && hurter->plObj)
	{
		ObjPlayer_ChargeUp(hurter, obj, dmg);
		if (hurter->plObj->fistOfFlames)
		{
			obj->isOnFire = g_glb.gameTime+4000+(rand()%2000);
		}
	}

	if (hurter && hurter->atkType == ATKTYPE_EXPLOSION)
	{
		float p[3];
		if (!col)
		{
			Math_VecCopy(hurter->atkRadiusPos, p);
		}
		else
		{
			Math_VecCopy((float *)col->endPos, p);
		}

		float explDir[3];
		float c[3];
		Util_GameObjectCenter(obj, c);
		Math_VecSub(c, p, explDir);
		float dist = Math_VecNorm(explDir);
		float dmgFactor = 256.0f + (float)dmg*12.0f;
		float force = (1.0f-(dist/hurter->atkRadiusCore))*dmgFactor;
		if (force < 1.0f)
		{
			force = 1.0f;
		}
		obj->net.vel[0] += explDir[0]*force;
		obj->net.vel[1] += explDir[1]*force;
		obj->net.vel[2] += explDir[2]*force;
		if (dist < hurter->atkRadiusCore*0.75f)
		{
			float fwd[3];
			Math_AngleVectors(obj->net.ang, fwd, 0, 0);
			float dp = Math_DotProduct(fwd, explDir);
			if (dp < 0.0f)
			{
				obj->knockTime = g_glb.gameTime + Util_LocalDelay(obj, 100);
				explDir[0] = -explDir[0];
				explDir[1] = -explDir[1];
				explDir[2] = -explDir[2];
				Math_VecToAngles(explDir, obj->net.ang);
				if (doPainAnim)
				{
					AI_StartAnim(obj, HUMANIM_FLYBACK2, true);
				}
			}
			else
			{
				Math_VecToAngles(explDir, obj->net.ang);
				if (doPainAnim)
				{
					AI_StartAnim(obj, HUMANIM_HIT_FALLFORWARD, true);
				}
			}
		}
		else if (doPainAnim)
		{
			AI_PainAnim(obj, col);
		}
		return;
	}

	if (!col)
	{
		if ((hurter->atkType != ATKTYPE_BULLET && hurter->atkType != ATKTYPE_BURNING) || rand()%10 < 1)
		{
			if (doPainAnim)
			{
				AI_PainAnim(obj, col);
			}
		}
		return;
	}
	float ang[3];
	Math_VecToAngles(col->endNormal, ang);
	if (!hurter || hurter->atkType != ATKTYPE_KNOCKBACK3)
	{
		if (dmg <= 10)
		{
			ObjParticles_Create("melee/bleedless", col->endPos, ang, -1);
		}
		else
		{
			ObjParticles_Create("melee/bleed", col->endPos, ang, -1);
		}
	}
	if (hurter && hurter->atkType == ATKTYPE_POPUP3)
	{
		if (doPainAnim)
		{
			AI_StartAnim(obj, HUMANIM_PAIN_POPUP, true);
		}
		float fwd[3];
		Math_AngleVectors(hurter->net.ang, fwd, 0, 0);
		obj->net.vel[0] += fwd[0]*200.0f;
		obj->net.vel[1] += fwd[1]*200.0f;
		obj->net.vel[2] += 1200.0f;
	}
	else if (hurter && hurter->atkType == ATKTYPE_POPUP)
	{
		if (doPainAnim)
		{
			AI_StartAnim(obj, HUMANIM_PAIN_POPUP, true);
		}
		if (obj->onGround)
		{
			obj->net.vel[0] = 0.0f;
			obj->net.vel[1] = 0.0f;
		}
		obj->net.vel[2] += 1000.0f;
	}
	else if (hurter && hurter->atkType == ATKTYPE_POPUP2)
	{
		if (doPainAnim)
		{
			AI_StartAnim(obj, HUMANIM_PAIN_POPUP, true);
		}
		float fwd[3];
		Math_AngleVectors(hurter->net.ang, fwd, 0, 0);
		obj->net.vel[0] += fwd[0]*300.0f;
		obj->net.vel[1] += fwd[1]*300.0f;
		obj->net.vel[2] = 800.0f;
	}
	else if (hurter && (hurter->atkType == ATKTYPE_KNOCKBACK || hurter->atkType == ATKTYPE_KNOCKBACK2 ||
		hurter->atkType == ATKTYPE_KNOCKBACK3))
	{
		if (doPainAnim)
		{
			AI_StartAnim(obj, HUMANIM_FLYBACK, true);
		}
		float fwd[3];
		float knockback = (hurter->atkType == ATKTYPE_KNOCKBACK) ? 2400.0f : 600.0f;
		if (hurter->atkType == ATKTYPE_KNOCKBACK3)
		{
			float c1[3], c2[3];
			Util_GameObjectCenter(obj, c1);
			Util_GameObjectCenter(hurter, c2);
			Math_VecSub(c1, c2, fwd);
			Math_VecNorm(fwd);
		}
		else
		{
			Math_AngleVectors(hurter->net.ang, fwd, 0, 0);
		}
		obj->net.vel[0] += fwd[0]*knockback;
		obj->net.vel[1] += fwd[1]*knockback;
		obj->net.vel[2] += (hurter->atkType == ATKTYPE_KNOCKBACK) ? 400.0f : 300.0f;
		obj->onGround = false;
		obj->knockTime = g_glb.gameTime + Util_LocalDelay(obj, 100);
	}
	else if (hurter && hurter->atkType == ATKTYPE_SMACKDOWN)
	{
		if (!obj->onGround && !hurter->onGround)
		{
			obj->net.vel[0] = 0.0f;
			obj->net.vel[1] = 0.0f;
			if (doPainAnim)
			{
				AI_PainAnim(obj, col);
			}
			if (hurter && hurter->plObj && hurter->plObj->explosiveBlows && obj->health > 0)
			{
				hurter->net.vel[2] += 600.0f;
				obj->net.vel[2] = -600.0f;
				g_glb.actionTime = g_glb.gameTime+500;
				obj->meteorAttack = 2;
			}
			else
			{
				obj->net.vel[2] = -2000.0f;
				obj->meteorAttack = 1;
			}

			//push them off a bit, so they don't land on top of me
			float c1[3], c2[3], d[3];
			Util_GameObjectCenter(obj, c1);
			Util_GameObjectCenter(hurter, c2);
			Math_VecSub(c2, c1, d);
			Math_VecNorm(d);
			hurter->net.vel[0] += d[0]*128.0f;
			hurter->net.vel[1] += d[1]*128.0f;
		}
		else
		{
			if (doPainAnim)
			{
				AI_StartAnim(obj, HUMANIM_FALL_LAND, true);
			}
			float fwd[3];
			float knockback = 800.0f;
			Math_AngleVectors(hurter->net.ang, fwd, 0, 0);
			obj->net.vel[0] += fwd[0]*knockback;
			obj->net.vel[1] += fwd[1]*knockback;
		}
	}
	else
	{
		if (!obj->onGround && hurter && hurter->atkType == ATKTYPE_INAIR &&
			(!obj->aiObj || !obj->aiObj->fly))
		{
			float d[3];
			float c1[3], c2[3];
			//Util_GameObjectCenter(obj, c1);
			Math_VecCopy(obj->net.pos, c1);
			c1[2] += (obj->net.maxs[2]-obj->net.mins[2])*0.5f;
			Util_GameObjectCenter(hurter, c2);
			Math_VecSub(c2, c1, d);
			obj->net.vel[0] = d[0];
			obj->net.vel[1] = d[1];
			obj->net.vel[2] = d[2];
			obj->noGravTime = g_glb.gameTime + Util_LocalDelay(obj, 500);
		}
		else if (!obj->onGround && hurter && hurter->atkType == ATKTYPE_BULLET &&
			(!obj->aiObj || !obj->aiObj->fly))
		{
			/*
			float ang[3];
			float d[3];
			Math_VecSub(hurter->net.pos, obj->net.pos, d);
			Math_VecToAngles(d, ang);
			obj->net.ang[YAW] = ang[YAW];
			*/
			if (!hurter->onGround)
			{
				float d[3];
				float c1[3], c2[3];
				//Util_GameObjectCenter(obj, c1);
				Math_VecCopy(obj->net.pos, c1);
				c1[2] += (obj->net.maxs[2]-obj->net.mins[2])*0.5f;
				Util_GameObjectCenter(hurter, c2);
				Math_VecSub(c2, c1, d);
				obj->net.vel[0] = d[0]*0.1f;
				obj->net.vel[1] = d[1]*0.1f;
				obj->net.vel[2] = d[2];
			}
			else
			{
				obj->net.vel[2] = 10.0f;
			}
			if (doPainAnim)
			{
				AI_StartAnim(obj, HUMANIM_HIT_FALLFORWARD, true);
			}
			return;
		}

		if ((hurter->atkType != ATKTYPE_BULLET && hurter->atkType != ATKTYPE_BURNING) || rand()%10 < 1)
		{
			if (doPainAnim)
			{
				AI_PainAnim(obj, col);
			}
		}
	}
}

//effects
void AI_CheckEffects(gameObject_t *obj, float timeMod)
{
	if (obj->isOnFire)
	{
		obj->net.renderEffects |= FXFL_FLAMES;
		if (obj->isOnFire < g_glb.gameTime)
		{
			obj->net.renderEffects &= ~FXFL_FLAMES;
			obj->isOnFire = 0;
		}
	}

	if (obj->isOnFire && obj->fireBurnTime < g_glb.gameTime)
	{
		int atkType = obj->atkType;
		obj->atkType = ATKTYPE_BURNING;
		Util_DamageObject(obj, obj, 10, NULL);
		obj->atkType = atkType;
		ObjSound_Create(obj->net.pos, "assets/sound/cb/fire01.wav", 1.0f, -1);
		obj->fireBurnTime = g_glb.gameTime + 100 + (rand()%100);
	}

	if (obj->meteorAttack == 2)
	{
		obj->net.renderEffects |= FXFL_METEORSTREAM;
	}
	else
	{
		obj->net.renderEffects &= ~FXFL_METEORSTREAM;
	}
}

//for anything client-side that may need this
void AI_SetGroundDistance(gameObject_t *obj)
{
	collObj_t col;
	float ground[3];
	Math_VecCopy(obj->net.pos, ground);
	ground[2] -= 4096.0f;
	RCUBE_ASSERT(obj->rcColModel);
	g_sharedFn->Coll_MovementTranslation(obj->rcColModel, &col, obj->net.pos, ground, NULL, NULL);
	if (col.hit && !col.containsSolid)
	{
		obj->net.groundZ = col.endPos[2];
		obj->net.renderEffects2 |= FXFL2_GROUNDZ;
	}
	else
	{
		obj->net.renderEffects2 &= ~FXFL2_GROUNDZ;
	}
}

//should i do anything?
bool AI_ShouldThink(gameObject_t *obj)
{
	if (g_glb.cinema == 1)
	{
		return false;
	}
	return true;
}

//generic think
void AI_GenericThink(gameObject_t *obj, float timeMod)
{
	if (obj->preDeathTime && obj->preDeathTime < g_glb.gameTime)
	{
		obj->nonSolidTime = 0;
		obj->net.solid = 0;
		obj->rcColModel->solid = 0;
		AI_StartAnim(obj, HUMANIM_DEATH, true);
		obj->animNeverInterrupt = true;
		obj->preDeathTime = 0;
	}
	if (obj->postSpawnTime && obj->postSpawnTime < g_glb.gameTime)
	{
		obj->net.renderEffects2 &= ~FXFL2_SPAWN;
		obj->net.renderEffects2 &= ~FXFL2_SPAWN2;
		obj->postSpawnTime = 0;
	}
	if (obj->aiObj->confuseTime >= g_glb.gameTime)
	{
		obj->net.renderEffects2 |= FXFL2_CONFUSE;
	}
	else
	{
		obj->net.renderEffects2 &= ~FXFL2_CONFUSE;
	}

	AI_PickAnim(obj, timeMod);
	Util_AnimateObject(obj, timeMod);

	AI_ApplyAnimationVelocity(obj, timeMod);
	AI_ApplyAnimationEffects(obj, timeMod);
	AI_CheckEffects(obj, timeMod);

	//renormalize angles
	obj->net.ang[0] = (360.0f / 65536) * ((int)(obj->net.ang[0] * (65536 / 360.0f)) & 65535);
	obj->net.ang[1] = (360.0f / 65536) * ((int)(obj->net.ang[1] * (65536 / 360.0f)) & 65535);
	obj->net.ang[2] = (360.0f / 65536) * ((int)(obj->net.ang[2] * (65536 / 360.0f)) & 65535);
	obj->aiObj->lookYaw = (360.0f / 65536) * ((int)(obj->aiObj->lookYaw * (65536 / 360.0f)) & 65535);
	obj->aiObj->lookPitch = (360.0f / 65536) * ((int)(obj->aiObj->lookPitch * (65536 / 360.0f)) & 65535);

	float animAng[3];
	if (AI_AnglesForAnim(obj, animAng))
	{
		float angleBlendScale = 0.7f;
		obj->net.ang[PITCH] = Math_BlendAngle(obj->net.ang[PITCH], animAng[PITCH], timeMod*angleBlendScale);
		obj->net.ang[YAW] = Math_BlendAngle(obj->net.ang[YAW], animAng[YAW], timeMod*angleBlendScale);
		obj->net.ang[ROLL] = Math_BlendAngle(obj->net.ang[ROLL], animAng[ROLL], timeMod*angleBlendScale);
	}
	else
	{
		float angleBlendScale = 0.3f;
		if (!AI_NonAnglingAnim(obj) && obj->aiObj->noMoveTime < g_glb.gameTime)
		{
			obj->net.ang[PITCH] = Math_BlendAngle(obj->net.ang[PITCH], obj->aiObj->lookPitch, timeMod*angleBlendScale);
			obj->net.ang[YAW] = Math_BlendAngle(obj->net.ang[YAW], obj->aiObj->lookYaw, timeMod*angleBlendScale);
		}
		else
		{
			obj->net.ang[PITCH] = Math_BlendAngle(obj->net.ang[PITCH], 0.0f, timeMod*angleBlendScale);
		}
		obj->net.ang[ROLL] = Math_BlendAngle(obj->net.ang[ROLL], 0.0f, timeMod*angleBlendScale);
	}

	if (obj->net.solid || obj->nonSolidTime)
	{
		Util_SetSafeSpot(obj);
		Phys_ApplyObjectPhysics(obj, timeMod, obj->radius, AI_GetGravity(obj, timeMod), 0.0f);
		LServ_UpdateRClip(obj);
	}

	AI_SetGroundDistance(obj);

	if (obj->thinkTime < g_glb.gameTime)
	{
		obj->thinkTime = g_glb.gameTime;
	}
}

//touch
void AI_GenericTouch(gameObject_t *obj, gameObject_t *other, const collObj_t *col)
{
	if (!col || col->containsSolid)
	{
		return;
	}

	if (obj->toucher && obj->meteorAttack)
	{
		if (other->hurtable && other->net.index != obj->seqDmg.dmgLastIndex)
		{
			int a = obj->atkType;
			obj->atkType = ATKTYPE_KNOCKBACK3;
			Util_DamageObject(obj, other, 5, col);
			obj->atkType = a;
		}
		float ang[3], p[3];
		Math_VecToAngles(col->endNormal, ang);
		Math_VecMA((float *)col->endPos, 8.0f, (float *)col->endNormal, p);
		ObjParticles_Create("impacts/flyback", p, ang, -1);
		ObjSound_CreateFromIndex(p, g_soundHitHeavy[rand()%NUM_SOUNDS_HITHEAVY], 1.0f, -1);

		if (obj->meteorAttack == 2)
		{
			if (g_glb.actionTime)
			{
				g_glb.actionTime = 1;
			}
			gameObject_t *pl = &g_gameObjects[0];
			if (pl->inuse)
			{
				ObjParticles_Create("impacts/meteorimpact", p, ang, -1);
				ObjSound_Create(p, "assets/sound/cb/meteorsmash.wav", 1.0f, -1);
				Util_RadiusDamage(pl, 250, 2600.0f, p);
			}
		}

		obj->meteorAttack = 0;
	}

	if (obj->toucher && obj->curAnim == HUMANIM_FLYBACK)
	{
		if (other->hurtable && other->net.index != obj->seqDmg.dmgLastIndex)
		{
			int a = obj->atkType;
			obj->atkType = ATKTYPE_KNOCKBACK3;
			Util_DamageObject(obj, other, 5, col);
			obj->atkType = a;
		}
		float nVel[3];
		Math_VecCopy(obj->net.vel, nVel);
		Math_VecNorm(nVel);
		float d = Math_DotProduct(nVel, col->endNormal);
		float groundPl[3] = {0.0f, 0.0f, 1.0f};
		float groundDot = 0.4f;
		float gd = Math_DotProduct(groundPl, col->endNormal);
		if (gd < groundDot && gd > -groundDot)
		{
			if (d < -0.5f || obj->knockTime < g_glb.gameTime)
			{
				float ang[3], p[3];
				Math_VecToAngles(col->endNormal, ang);
				Math_VecMA((float *)col->endPos, -(obj->net.maxs[0]*0.75f), (float *)col->endNormal, p);
				p[2] += (obj->net.maxs[2]-obj->net.mins[2])*0.5f;
				ObjParticles_Create("impacts/flyback", p, ang, -1);
				ObjSound_CreateFromIndex(p, g_soundHitHeavy[rand()%NUM_SOUNDS_HITHEAVY], 1.0f, -1);

				Math_VecCopy((float *)col->endPos, obj->net.pos);
				LServ_UpdateRClip(obj);

				float f = 256.0f;
				obj->net.vel[0] = col->endNormal[0]*f;
				obj->net.vel[1] = col->endNormal[1]*f;
				obj->net.vel[2] = col->endNormal[2]*f;
				Math_VecToAngles(col->endNormal, obj->net.ang);
				AI_StartAnim(obj, HUMANIM_HIT_FALLFORWARD, true);
			}
		}
		else if (d < -0.5f || d == 0.0f || obj->knockTime < g_glb.gameTime)
		{
			AI_StartAnim(obj, HUMANIM_POPUP_LAND, true);
		}
	}
	else if (obj->toucher && (obj->curAnim == HUMANIM_PAIN_AIR || obj->curAnim == HUMANIM_PAIN_POPUP || obj->curAnim == HUMANIM_HIT_FALLFORWARD || obj->curAnim == HUMANIM_FLYBACK2))
	{
		gameAnim_t *curAnim = obj->animTable+obj->curAnim;
		if (obj->net.frame == curAnim->endFrame)
		{
			float groundPl[3] = {0.0f, 0.0f, 1.0f};
			float groundDot = 0.0f;
			float gd = Math_DotProduct(groundPl, col->endNormal);
			if (gd > groundDot)
			{
				int anim = (obj->curAnim == HUMANIM_HIT_FALLFORWARD) ? HUMANIM_FALL_LAND : HUMANIM_POPUP_LAND;
				AI_StartAnim(obj, anim, true);
			}
		}
	}
}

//drop item
void AI_DropItem(gameObject_t *obj, float timeMod)
{
	gameObject_t *pl = &g_gameObjects[0];
	if (obj->aiObj && pl->inuse && pl->plObj)
	{
		for (int i = 0; i < NUM_INVENTORY_ITEM_DEFS; i++)
		{
			if (obj->aiObj->dropChances[i] > 0)
			{ //consider dropping one of these
				int luckStat = (int)pl->plObj->plData.statLuck;
				const invItemDef_t *item = ObjPlayer_GetEquippedItem(pl);
				if (item && item->itemBehavior == INVITEM_BEHAVIOR_MODLUCK)
				{
					luckStat += item->itemValue;
				}
				int dropFactor = (int)obj->aiObj->dropChances[i] * luckStat;
				if (dropFactor > (rand()%65536))
				{ //success, spit one out
					bool itemFull = false;
					if (i == INVITEM_POTION)
					{ //hack because potions tend to build up and it gets annoying seeing full messages
						for (int j = 0; j < MAX_PLAYER_INVENTORY_ITEMS; j++)
						{
							playerInvItem_t *inv = &pl->plObj->plData.inventory[j];
							if (inv->itemIndex == i && inv->itemQuantity >= item->maxAmount)
							{
								itemFull = true;
							}
						}
					}
					if (!itemFull)
					{
						float c[3];
						Util_GameObjectCenter(obj, c);
						c[0] += obj->net.mins[0]+(rand()%(int)(obj->net.maxs[0]-obj->net.mins[0]));
						c[1] += obj->net.mins[1]+(rand()%(int)(obj->net.maxs[1]-obj->net.mins[1]));
						c[2] += (rand()%(int)(obj->net.maxs[2]));
						gameObject_t *item = LServ_ObjectFromName("obj_item_drop", c, obj->net.ang, NULL, 0);
						item->makoCount = i;
					}
				}
			}
		}
	}
	obj->think = ObjGeneral_RemoveThink;
	obj->thinkTime = g_glb.gameTime + 4000;
}

//drop mako
void AI_DropMako(gameObject_t *obj, float timeMod)
{
	if (g_glb.magicMode)
	{
		obj->think = ObjGeneral_RemoveThink;
		obj->thinkTime = g_glb.gameTime + 4000;
		return;
	}

	if (obj->makoCount > 0)
	{
		float c[3];
		Util_GameObjectCenter(obj, c);
		c[0] += obj->net.mins[0]+(rand()%(int)(obj->net.maxs[0]-obj->net.mins[0]));
		c[1] += obj->net.mins[1]+(rand()%(int)(obj->net.maxs[1]-obj->net.mins[1]));
		c[2] = obj->net.pos[2]+obj->net.mins[2]+64.0f;
		float f = (float)(rand()%(int)(obj->net.maxs[2]));
		if (f > 64.0f)
		{
			f -= 64.0f;
		}
		c[2] += f;
		gameObject_t *mako = LServ_ObjectFromName("obj_mako_drop", c, obj->net.ang, NULL, 0);
		if (mako && obj->aiObj && obj->aiObj->makoValue)
		{
			mako->generalFlag = obj->aiObj->makoValue*2;
		}
		obj->makoCount--;
	}

	if (obj->makoCount > 0)
	{
		obj->thinkTime = g_glb.gameTime + (int)(100.0f*g_glb.timeScale);
	}
	else
	{
		obj->think = AI_DropItem;
		obj->thinkTime = g_glb.gameTime + (int)(100.0f*g_glb.timeScale);
	}
}

//undeath
void AI_ImmortalDeath(gameObject_t *obj, gameObject_t *killer, int dmg)
{
	obj->health = 1;
}

//routines to perform on death
void AI_DeathRoutines(gameObject_t *obj, gameObject_t *killer, int dmg)
{
	if (g_glb.endlessBattle.active)
	{ //in endless battle mode
		if (g_glb.endlessBattle.numKills < 536870912)
		{
			g_glb.endlessBattle.numKills++;
			if ((g_glb.endlessBattle.numKills%100) == 0)
			{
				int *t = (int *)_alloca(sizeof(int)+sizeof(float)*3);
				*t = 600;
				float *c = (float *)(t+1);
				c[0] = 1.0f;
				c[1] = 1.0f;
				c[2] = 1.0f;
				g_sharedFn->Net_SendEventType(CLEV_TRIGGERFLASH, t, sizeof(int)+sizeof(float)*3, -1);
				Util_StatusMessage("*(1 1 1 1)Defeated: *(1 0 0 1)%i*(1 1 1 1)!", g_glb.endlessBattle.numKills);
			}
			else if ((g_glb.endlessBattle.numKills%10) == 0)
			{
				Util_StatusMessage("*(1 1 1 1)Defeated: %i", g_glb.endlessBattle.numKills);
			}
			if (g_glb.endlessBattle.numKills > GVar_GetInt("endl_killrecord"))
			{
				GVar_SetInt("endl_killrecord", g_glb.endlessBattle.numKills);
			}

			if (g_glb.endlessBattle.numKills == 500)
			{
				Util_RunScript("obj_awardscr", "endl_award1");
			}
			else if (g_glb.endlessBattle.numKills == 1000)
			{
				Util_RunScript("obj_awardscr", "endl_award2");
			}

			g_globalNet.endlScore = g_glb.endlessBattle.numKills;
			g_globalNet.endlHighscore = GVar_GetInt("endl_killrecord");
		}
	}

	if (killer && killer->plObj)
	{
		if (g_glb.ai.weakAI && obj->aiObj && obj->aiObj->aiLevel < 15 && !ObjPlayer_IsWeakSauce(killer))
		{ //don't follow up a successful kill with an attack from ai if we are still n00bish
			g_glb.ai.playerLastHurtTime = g_glb.gameTime;
		}

		ObjPlayer_ChargeUp(killer, obj, 75);
		if (obj->net.entNameIndex)
		{
			char *objName = g_sharedFn->Common_ServerStringFromIndex(obj->net.entNameIndex);
			if (objName && objName[0])
			{
				char gvName[128];
				sprintf(gvName, "endef_%s", objName);
				GVar_SetInt(gvName, 1);
			}
		}
	}
	gameObject_t *pl = &g_gameObjects[0];
	obj->isOnFire = 0;
	obj->net.renderEffects &= ~FXFL_FLAMES;

	obj->meteorAttack = 0;
	obj->net.renderEffects &= ~FXFL_METEORSTREAM;

	obj->net.renderEffects &= ~FXFL_AURA;
	obj->net.renderEffects &= ~FXFL_AURA2;
	obj->net.renderEffects2 &= ~FXFL2_INVINCIBLE;

	obj->makoCount = (pl && pl->plObj) ? 1+(int)(((float)pl->plObj->chargeMeter/4000.0f)*7.0f): 1;
	if (obj->aiObj && obj->aiObj->aiLevel <= 15 && obj->makoCount > 3)
	{
		obj->makoCount = 3;
	}
	obj->hurtable = 0;
	obj->net.renderEffects |= FXFL_DEATH;
	obj->preDeathTime = g_glb.gameTime + (int)(200.0f*g_glb.timeScale);
}

//death
void AI_GenericDeath(gameObject_t *obj, gameObject_t *killer, int dmg)
{
	if (killer && killer->inuse && obj->seqDmg.dmgLastIndex == killer->net.index)
	{
		if (obj->aiObj->boneDmgTime == g_glb.gameTime)
		{
			float force = (float)dmg * 8.0f;
			if (force < 800.0f)
			{
				force = 800.0f;
			}
			else if (force > 1200.0f)
			{
				force = 1200.0f;
			}
			float d[3];
			Math_VecCopy(obj->aiObj->boneDmgDir, d);
			d[2] *= 0.2f;
			Math_VecNorm(d);
			obj->net.vel[0] += d[0]*force;
			obj->net.vel[1] += d[1]*force;
			obj->net.vel[2] += d[2]*force;
		}
	}
	ObjSound_Create(obj->net.pos, "assets/sound/cb/lstreamsh.wav", 1.0f, -1);
	AI_DeathRoutines(obj, killer, dmg);
}

//generic spawn
void AI_GenericSpawn(gameObject_t *obj, BYTE *b, const objArgs_t *args, int numArgs)
{
	obj->aiObj = (aiObject_t *)g_sharedFn->Common_RCMalloc(sizeof(aiObject_t));
	memset(obj->aiObj, 0, sizeof(aiObject_t));

	const char *aiLevel = Common_GetValForKey(b, "ailevel");
	if (aiLevel && aiLevel[0])
	{
		obj->aiObj->aiLevel = atoi(aiLevel);
	}

	const char *aiFov = Common_GetValForKey(b, "aiFov");
	if (aiFov && aiFov[0])
	{
		obj->aiObj->aiFov = (float)atof(aiFov);
	}
	const char *aiRange = Common_GetValForKey(b, "aiRange");
	if (aiRange && aiRange[0])
	{
		obj->aiObj->aiRange = (float)atof(aiRange);
	}

	obj->aiObj->combatRange = 512.0f;
	obj->aiObj->moveSpeed = 128.0f;
	obj->aiObj->lookYaw = obj->net.ang[YAW];
	obj->aiObj->lookPitch = 0.0f;
	Math_VecCopy(obj->net.pos, obj->aiObj->goalPos);

	obj->postSpawnTime = g_glb.gameTime+5000;

	obj->pain = AI_GenericPain;
	obj->death = AI_GenericDeath;
	obj->touch = AI_GenericTouch;
	obj->think = AI_GenericThink;
	obj->thinkTime = g_glb.gameTime+50;

	obj->canBePushed = true;
	obj->localFlags |= LFL_ENEMY;

	Phys_PutOnGround(obj);
	Math_VecCopy(obj->net.pos, obj->safePos);
}
