/*
=============================================================================
Module Information
------------------
Name:			main.cpp
Author:			Rich Whitehouse
Description:	core server logic module implementation.
=============================================================================
*/

#include "main.h"
#include "ai.h"
#include "rscript.h"

gameGlobals_t g_glb;

int g_gameTouchBased = 0;
sharedSVFunctions_t *g_sharedFn;
globalNetData_t g_globalNet;

static sessHeader_t *g_sessLoad = NULL;

int g_soundFootsteps[NUM_SOUNDS_FOOTSTEPS];
int g_soundHitLight[NUM_SOUNDS_HITLIGHT];
int g_soundHitHeavy[NUM_SOUNDS_HITHEAVY];
int g_soundKick[NUM_SOUNDS_KICK];
int g_soundPunch[NUM_SOUNDS_PUNCH];
int g_soundDeepBlow[NUM_SOUNDS_DEEPBLOW];

gameObject_t g_gameObjects[MAX_GAME_OBJECTS];

//return api version
int LServ_GetAPIVersion(void)
{
	return LOGIC_API_VERSION;
}

//find one that isn't in use
gameObject_t *LServ_CreateObject(void)
{
	int i = MAP_OBJ_NUM+1;
	while (i < MAX_GAME_OBJECTS)
	{
		if (!g_gameObjects[i].inuse && g_gameObjects[i].noReuseTime <= g_glb.gameTime)
		{
			g_gameObjects[i].inuse = INUSE_ACTIVE;
			g_gameObjects[i].net.index = i;
			g_gameObjects[i].net.owner = MAP_OBJ_NUM;
			g_gameObjects[i].net.staticIndex = -1;
			g_gameObjects[i].creationCount = g_glb.creationCount;
			g_glb.creationCount++;
			g_gameObjects[i].localTimeScale = 1.0f;

			if (g_glb.gameObjectSlots <= i)
			{
				g_glb.gameObjectSlots = i+1;
			}

			return &g_gameObjects[i];
		}
		i++;
	}

	return NULL;
}

//unlink from physics parent/children, if applicable
void LServ_UnlinkPhysics(gameObject_t *obj)
{
	for (int i = 0; i < obj->numPhysChildren; i++)
	{
		gameObject_t *child = obj->physChildren[i];
		if (child->physParent != obj)
		{
			continue;
		}
		child->physParent = NULL;
	}
	obj->numPhysChildren = 0;

	if (obj->physParent)
	{
		gameObject_t *parent = obj->physParent;
		for (int i = 0; i < parent->numPhysChildren; i++)
		{
			if (parent->physChildren[i] == obj)
			{
				if (i < parent->numPhysChildren-1)
				{
					memcpy(&parent->physChildren[i], &parent->physChildren[i+1], parent->numPhysChildren-i-1);
				}
				parent->numPhysChildren--;
				break;
			}
		}
	}
}

//kill it
void LServ_FreeObject(gameObject_t *obj)
{
	if (obj->onremove)
	{
		obj->onremove(obj);
	}
	LServ_UnlinkPhysics(obj);
	if (obj->physChildren)
	{
		g_sharedFn->Common_RCFree(obj->physChildren);
	}
	if (obj->rcColModel)
	{
		g_sharedFn->Coll_DestroyModelInstance(obj->rcColModel);
	}
	if (obj->plObj)
	{
		if (obj->plObj->targetObj)
		{
			obj->plObj->targetObj->net.renderEffects &= ~ObjPlayer_TargetBit(obj);
		}
		g_sharedFn->Common_RCFree(obj->plObj);
	}
	if (obj->aiObj)
	{
		AI_FreeAIObject(obj->aiObj);
	}
	if (obj->cinObj)
	{
		g_sharedFn->Common_RCFree(obj->cinObj);
	}
	if (obj->rscript)
	{
		RScr_DestroyScript(obj->rscript);
	}
	if (obj->swordObj)
	{
		g_sharedFn->Common_RCFree(obj->swordObj);
	}
	if (obj->rigid)
	{
		g_sharedFn->Common_RCFree(obj->rigid);
	}
	if (obj->customDynamic)
	{
		g_sharedFn->Common_RCFree(obj->customDynamic);
	}
	if (obj->lodInfo)
	{
		g_sharedFn->Common_RCFree(obj->lodInfo);
	}
	memset(obj, 0, sizeof(gameObject_t));
	obj->noReuseTime = g_glb.gameTime + 1000; //don't reuse it so that the client can get an update on its removal first
}

//flag INUSE_NONET based on visibility within camera frustum
void LServ_NetCull(gameObject_t *obj)
{
	if (!g_glb.camFrustumFresh)
	{
		return;
	}

	if (obj->net.type == OBJ_TYPE_VISBLOCK)
	{ //don't attempt to cull these
		return;
	}

	if (obj->noCullTime >= g_glb.gameTime || obj->alwaysSend)
	{
		obj->inuse &= ~INUSE_NONET;
		return;
	}

	if (obj->localFlags & LFL_NONET)
	{ //never send
		obj->inuse |= INUSE_NONET;
		return;
	}

	if (obj->net.type == OBJ_TYPE_LIGHT &&
		(obj->net.renderEffects & FXFL_HIDDEN))
	{ //cull out invisible lights
		obj->inuse |= INUSE_NONET;
		return;
	}

	if (obj->cinObj)
	{
		if (obj->cinObj->hideable && (obj->net.renderEffects & FXFL_HIDDEN))
		{
			obj->inuse |= INUSE_NONET;
			return;
		}
		else if (!obj->net.solid)
		{ //hack for cinematic objects so they can animate way outside of their bounds
			obj->inuse &= ~INUSE_NONET;
			return;
		}
	}

	if (obj->spawnMins[0] == 0.0f &&
		obj->spawnMins[1] == 0.0f &&
		obj->spawnMins[2] == 0.0f &&
		obj->spawnMaxs[0] == 0.0f &&
		obj->spawnMaxs[1] == 0.0f &&
		obj->spawnMaxs[2] == 0.0f)
	{ //spawn bounds not set
		return;
	}

	if (!Math_PointInFrustum(&g_glb.wideCamFrustum, obj->net.pos, Util_RadiusFromBounds(obj->spawnMins, obj->spawnMaxs, 2.0f)))
	{ //not in frustum
		if (!(obj->net.renderEffects & FXFL_DEATH))
		{ //don't frustum-cull things with death fx
			obj->inuse |= INUSE_NONET;
			return;
		}
	}

	if (!ObjVisBlock_CheckVis(obj))
	{ //visibility system culled it
		obj->inuse |= INUSE_NONET;
		return;
	}

	if (!obj->rcColModel || !g_sharedFn->Coll_IsTreeModel(obj->rcColModel))
	{
		gameObject_t *cam = Util_GetCam(0);
		float rad;
		float midPos[3];
		if (obj->net.type == OBJ_TYPE_LIGHT)
		{
			rad = obj->net.modelScale[0];
			Math_VecCopy(obj->net.pos, midPos);
		}
		else
		{
			rad = Math_Max3(obj->spawnMaxs[0]-obj->spawnMins[0],
								obj->spawnMaxs[1]-obj->spawnMins[1],
								obj->spawnMaxs[2]-obj->spawnMins[2])*0.5f;
			midPos[0] = obj->net.pos[0] + obj->spawnMins[0] + (obj->spawnMaxs[0]-obj->spawnMins[0])*0.5f;
			midPos[1] = obj->net.pos[1] + obj->spawnMins[1] + (obj->spawnMaxs[1]-obj->spawnMins[1])*0.5f;
			midPos[2] = obj->net.pos[2] + obj->spawnMins[2] + (obj->spawnMaxs[2]-obj->spawnMins[2])*0.5f;
		}
		if (!g_sharedFn->Coll_GeneralVisibilityWithBounds(midPos, cam->net.pos, rad, obj->spawnMins, obj->spawnMaxs))
		{ //sphere visibility failed
			obj->inuse |= INUSE_NONET;
			return;
		}
	}

	//visible
	obj->inuse &= ~INUSE_NONET;
}

//update an object's rclip
void LServ_UpdateRClip(gameObject_t *obj)
{
	if (!obj || !obj->rcColModel)
	{
		return;
	}
	Math_VecCopy(obj->net.mins, obj->rcColModel->mins);
	Math_VecCopy(obj->net.maxs, obj->rcColModel->maxs);
	Math_VecCopy(obj->net.pos, obj->rcColModel->pos);
	Math_VecCopy(obj->net.ang, obj->rcColModel->ang);
	Math_VecCopy(obj->net.modelScale, obj->rcColModel->modelScale);
	obj->rcColModel->gameOwner = obj->net.index;
	obj->rcColModel->solid = obj->net.solid;
	if (obj->rcColModel->frame != obj->net.frame)
	{
		obj->rcColModel->frame = obj->net.frame;
		obj->rcColModel->blendFrame = obj->net.frame;
	}
	g_sharedFn->Coll_UpdateModel(obj->rcColModel, NULL);
}

//check the new score against records and such
void LServ_MusicalScoreLog(gameObject_t *client)
{
	/*
	if (client->net.lives > g_musicHighScore)
	{
		g_musicHighScore = client->net.lives;
		g_sharedFn->Net_SendEventType(CLEV_UPDATEHIGHSCORE, &g_musicHighScore, sizeof(g_musicHighScore), -1);
	}
	*/
	//wait until after the song is done to update i guess, in case they started screwing up and dying a lot
}

//in musical mode, modify the music string
void LServ_ChangeMusic(char *music)
{
	if (g_glb.musicStrIndex == -1)
	{
		g_glb.musicStrIndex = g_sharedFn->Common_ServerString(music);
	}
	else
	{
		g_sharedFn->Common_ModifyString(g_glb.musicStrIndex, ""); //force cleansing
		g_sharedFn->Common_ModifyString(g_glb.musicStrIndex, music);
	}
}

//event
void LServ_ServerEvent(int clientIndex, serverEventTypes_e ev, BYTE *data, int dataSize)
{
	switch (ev)
	{
	case SVEV_FORGEABILITY:
		if (dataSize == sizeof(int))
		{
			gameObject_t *obj = &g_gameObjects[clientIndex];
			if (obj->inuse && obj->plObj)
			{
				Shared_ForgeAbility(&obj->plObj->plData, *((int *)data), obj->plObj->makoStash);
				obj->plObj->makoCharges = obj->plObj->plData.maxMakoCharges;
			}
		}
		break;
	case SVEV_FORGEITEM:
		if (dataSize == sizeof(int))
		{
			gameObject_t *obj = &g_gameObjects[clientIndex];
			if (obj->inuse && obj->plObj)
			{
				Shared_ForgeItem(&obj->plObj->plData, *((int *)data), obj->plObj->makoStash);
			}
		}
		break;
	case SVEV_FORGESTAT:
		if (dataSize == sizeof(int))
		{
			gameObject_t *obj = &g_gameObjects[clientIndex];
			if (obj->inuse && obj->plObj)
			{
				Shared_ForgeStat(&obj->plObj->plData, *((int *)data), obj->plObj->makoStash);
				obj->health = obj->plObj->plData.maxHealth;
			}
		}
		break;
	case SVEV_INVSORT:
		if (dataSize == sizeof(int)*2)
		{
			gameObject_t *obj = &g_gameObjects[clientIndex];
			if (obj->inuse && obj->plObj)
			{
				int src = *((int *)data);
				int dst = *(((int *)data)+1);
				playerInvItem_t itemDst = obj->plObj->plData.inventory[dst];
				obj->plObj->plData.inventory[dst] = obj->plObj->plData.inventory[src];
				obj->plObj->plData.inventory[src] = itemDst;
			}
		}
		break;
	default:
		break;
	}
}

//a development-only command
void LServ_DevCommand(char *cmd)
{
	GVar_SetInt("useddevcmd", 1);

	textParser_t *parser = g_sharedFn->Parse_InitParser(cmd);
	if (!parser)
	{
		return;
	}
	parseToken_t token;
	while (g_sharedFn->Parse_GetNextToken(parser, &token))
	{
		if (!stricmp(token.text, "spawn"))
		{
			if (g_sharedFn->Parse_GetNextToken(parser, &token))
			{
				gameObject_t *cam = Util_GetCam(0);
				float pos[3], ang[3], dir[3];
				Math_AngleVectors(cam->net.ang, dir, 0, 0);
				ang[0] = 0.0f;
				ang[1] = cam->net.ang[1];
				ang[2] = 0.0f;
				Math_VecMA(cam->net.pos, 2500.0f, dir, pos);
				gameObject_t *obj = LServ_ObjectFromName(token.text, pos, ang, NULL, 0);
				if (!obj)
				{
					Util_StatusMessage("Object '%s' is not valid.", token.text);
				}
			}
			else
			{
				Util_StatusMessage("spawn command requires an object name.");
			}
		}
		else if (!stricmp(token.text, "<3"))
		{
			gameObject_t *pl = &g_gameObjects[0];
			if (pl->plObj)
			{
				pl->plObj->plData.maxHealth = 1000;
				pl->health = 1000;
				pl->plObj->plData.maxMakoCharges = 4;
				pl->plObj->makoCharges = 4;
				pl->plObj->plData.abilities = (1<<PLABIL_WALLJUMP)|(1<<PLABIL_GETUP)|(1<<PLABIL_FLIPKICK)|(1<<PLABIL_FLURRY)|(1<<PLABIL_THRUST)|(1<<PLABIL_TACKLE)|
						(1<<PLABIL_GROUNDCRUSH)|(1<<PLABIL_METEORKICK)|(1<<PLABIL_MAKOFLAME)|(1<<PLABIL_MAKOTIME)|(1<<PLABIL_MAKOHEAL)|(1<<PLABIL_MAKOAURA)|
						(1<<PLABIL_CHARGEFORCE);
			}
		}
		else if (!stricmp(token.text, "mako"))
		{
			gameObject_t *pl = &g_gameObjects[0];
			if (pl->plObj)
			{
				if (g_sharedFn->Parse_GetNextToken(parser, &token))
				{
					ObjPlayer_GiveMako(pl, atoi(token.text));
					if (pl->plObj->makoStash < 0)
					{
						pl->plObj->makoStash = 0;
					}
				}
				else
				{
					Util_StatusMessage("mako command requires a value.");
				}
			}
		}
		else if (!stricmp(token.text, "str") || !stricmp(token.text, "def") || !stricmp(token.text, "dex") ||
			!stricmp(token.text, "luck"))
		{
			char cmdName[64];
			strcpy(cmdName, token.text);
			if (g_sharedFn->Parse_GetNextToken(parser, &token))
			{
				gameObject_t *pl = &g_gameObjects[0];
				if (pl->plObj)
				{
					if (!stricmp(cmdName, "str"))
					{
						pl->plObj->plData.statStr = (BYTE)atoi(token.text);
					}
					else if (!stricmp(cmdName, "def"))
					{
						pl->plObj->plData.statDef = (BYTE)atoi(token.text);
					}
					else if (!stricmp(cmdName, "dex"))
					{
						pl->plObj->plData.statDex = (BYTE)atoi(token.text);
					}
					else if (!stricmp(cmdName, "luck"))
					{
						pl->plObj->plData.statLuck = (BYTE)atoi(token.text);
					}
				}
			}
			else
			{
				Util_StatusMessage("%s command requires value.", cmdName);
			}
		}
		else if (!stricmp(token.text, "noclip"))
		{
			gameObject_t *pl = &g_gameObjects[0];
			if (pl->plObj)
			{
				pl->plObj->noClip = !pl->plObj->noClip;
			}
		}
		else if (!stricmp(token.text, "jesus"))
		{
			gameObject_t *pl = &g_gameObjects[0];
			if (pl->plObj)
			{
				pl->plObj->invincible = !pl->plObj->invincible;
			}
		}
		else if (!stricmp(token.text, "notarget"))
		{
			gameObject_t *pl = &g_gameObjects[0];
			if (pl->plObj)
			{
				pl->plObj->noTarget = !pl->plObj->noTarget;
			}
		}
		else if (!stricmp(token.text, "npsngs"))
		{
			g_glb.noDeath = !g_glb.noDeath;
		}
		else if (!stricmp(token.text, "aipath"))
		{
			g_glb.ai.showPaths = !g_glb.ai.showPaths;
		}
		else if (!stricmp(token.text, "getpos"))
		{
			gameObject_t *pl = &g_gameObjects[0];
			Util_StatusMessage("pos: (%f %f %f)", pl->net.pos[0], pl->net.pos[1], pl->net.pos[2]);
		}
		else if (!stricmp(token.text, "setpos"))
		{
			if (g_sharedFn->Parse_GetNextToken(parser, &token))
			{
				gameObject_t *pl = &g_gameObjects[0];
				sscanf(token.text, "(%f %f %f)", &pl->net.pos[0], &pl->net.pos[1], &pl->net.pos[2]);
			}
			else
			{
				Util_StatusMessage("setpos command requires argument.");
			}
		}
		else if (!stricmp(token.text, "killemall"))
		{
			gameObject_t *pl = &g_gameObjects[0];
			for (int i = 0; i < g_glb.gameObjectSlots; i++)
			{
				gameObject_t *obj = &g_gameObjects[i];
				if (obj->inuse && obj->hurtable && (obj->localFlags & LFL_ENEMY))
				{
					Util_DamageObject(pl, obj, 99999999, 0);
				}
			}
		}
		else if (!stricmp(token.text, "giveitup"))
		{
			gameObject_t *pl = &g_gameObjects[0];
			if (pl->inuse && pl->plObj)
			{
				for (int i = 0; i < NUM_INVENTORY_ITEM_DEFS; i++)
				{
					const invItemDef_t *item = &g_invItems[i];
					pl->plObj->plData.inventory[i].itemIndex = i;
					pl->plObj->plData.inventory[i].itemQuantity = item->maxAmount;
				}
			}
		}
		else if (!stricmp(token.text, "timescale"))
		{
			if (g_sharedFn->Parse_GetNextToken(parser, &token))
			{
				g_glb.timeScale = (float)atof(token.text);
			}
			else
			{
				Util_StatusMessage("timescale command requires argument.");
			}
		}
		else if (!stricmp(token.text, "testfx"))
		{
			if (g_sharedFn->Parse_GetNextToken(parser, &token))
			{
				gameObject_t *pl = &g_gameObjects[0];
				float c[3];
				float ang[3] = {0.0f, 0.0f, 0.0f};
				Util_GameObjectCenter(pl, c);
				ObjParticles_Create(token.text, c, ang, -1);
			}
			else
			{
				Util_StatusMessage("testfx command requires argument.");
			}
		}
		else if (!stricmp(token.text, "rfx"))
		{
			if (g_sharedFn->Parse_GetNextToken(parser, &token))
			{
				gameObject_t *pl = &g_gameObjects[0];
				pl->net.renderEffects ^= (1<<atoi(token.text));
			}
			else
			{
				Util_StatusMessage("rfx command requires argument.");
			}
		}
		else if (!stricmp(token.text, "rfx2"))
		{
			if (g_sharedFn->Parse_GetNextToken(parser, &token))
			{
				gameObject_t *pl = &g_gameObjects[0];
				pl->net.renderEffects2 ^= (1<<atoi(token.text));
			}
			else
			{
				Util_StatusMessage("rfx2 command requires argument.");
			}
		}
		else if (!stricmp(token.text, "setfog"))
		{
			float *fogparms = (float *)_alloca(sizeof(float)*6 + sizeof(int));
			memset(fogparms, 0, sizeof(float)*6 + sizeof(int));
			for (int i = 0; i < 6; i++)
			{
				if (g_sharedFn->Parse_GetNextToken(parser, &token))
				{
					fogparms[i] = (float)atof(token.text);
				}
			}
			g_sharedFn->Net_SendEventType(CLEV_SETFOG, fogparms, sizeof(float)*6 + sizeof(int), -1);
		}
		else if (!stricmp(token.text, "^#$^%"))
		{
			gameObject_t *pl = &g_gameObjects[0];
			if (pl->plObj)
			{
				ObjPlBar_Init(pl);
			}
		}
		else if (!stricmp(token.text, "runscript"))
		{
			if (g_sharedFn->Parse_GetNextToken(parser, &token))
			{
				Util_RunScript("obj_testrun", token.text);
			}
			else
			{
				Util_StatusMessage("runscript command requires a script name.");
			}
		}
		else if (!stricmp(token.text, "map"))
		{
			if (g_sharedFn->Parse_GetNextToken(parser, &token))
			{
				g_sharedFn->Common_MapChange(token.text, 1000);
			}
			else
			{
				Util_StatusMessage("map command requires a map name.");
			}
		}
		else if (!stricmp(token.text, "gvar"))
		{
			if (g_sharedFn->Parse_GetNextToken(parser, &token))
			{
				char gvarName[2048];
				strcpy(gvarName, token.text);
				if (g_sharedFn->Parse_GetNextToken(parser, &token))
				{
					if (!stricmp(gvarName, "useddevcmd") || !stricmp(gvarName, "cdt_usingmod"))
					{
						Util_StatusMessage("Why would you want to do that?");
					}
					else
					{
						GVar_SetValue(gvarName, token.text);
					}
				}
				else
				{
					Util_StatusMessage("gvar '%s' is '%s'.", gvarName, GVar_GetValue(gvarName));
				}
			}
			else
			{
				Util_StatusMessage("gvar command requires a gvar name.");
			}
		}
		else if (!stricmp(token.text, "gvarlist"))
		{
			GVar_ListGVars();
		}
		else if (!stricmp(token.text, "clearerror"))
		{
			g_sharedFn->Net_SendEventType(CLEV_ERRORMESSAGE, "\0", 1, -1);
		}
		else if (!stricmp(token.text, "mscript"))
		{
			if (g_sharedFn->Parse_GetNextToken(parser, &token))
			{
				g_sharedFn->Net_SendEventType(CLEV_RUNMENUSCRIPT, token.text, (int)strlen(token.text)+1, -1);
			}
			else
			{
				Util_StatusMessage("mscript command requires a script name.");
			}
		}
		else if (!stricmp(token.text, "debugrigid"))
		{
			if (g_glb.debugRigid)
			{
				g_glb.debugRigid = false;
				Util_StatusMessage("Rigid body debug disabled.");
			}
			else
			{
				g_glb.debugRigid = true;
				Util_StatusMessage("Rigid body debug enabled.");
			}
		}
		else if (!stricmp(token.text, "debugsave"))
		{
			LServ_SaveSession("assets/saves/session.rse");
		}
		else
		{
			Util_StatusMessage("Unknown dev command: '%s'", token.text);
		}
	}

	g_sharedFn->Parse_FreeParser(parser);
}

//music logic
void LServ_MusicLogic(void)
{
	gameObject_t *pl = &g_gameObjects[0];
	if (g_glb.battleTime > g_glb.gameTime && g_glb.battleMusicEnabled)
	{
		if (!g_glb.battleMusic)
		{
			LServ_ChangeMusic("$#assets/music/FullFrontalAssault.ogg");
			g_glb.battleMusic = true;
		}
	}
	else if (!g_glb.battleMusic && pl->inuse && pl->plObj && pl->plObj->ride)
	{ //chocobo!
		if (!g_glb.chocoboMusic)
		{
			LServ_ChangeMusic("$#assets/music/Kweh.ogg");
			g_glb.chocoboMusic = true;
		}
	}
	else
	{
		if (g_glb.battleMusic || g_glb.chocoboMusic)
		{
			LServ_ChangeMusic(g_glb.levelMusic);
			g_glb.battleMusic = false;
			g_glb.chocoboMusic = false;
		}
	}
}

//show stats
void LServ_ShowMPStats(void)
{
	bool tied = false;
	int bestCl = -1;
	int bestScore = -1;
	for (int i = 0; i < MAX_NET_CLIENTS; i++)
	{
		gameObject_t *pl = &g_gameObjects[i];
		if (!pl->inuse || !pl->plObj || !g_glb.clientInfo[i].infoStr)
		{
			continue;
		}

		char str[64];
		sprintf(str, "pls_%s", g_glb.clientInfo[i].infoStr);
		int score = GVar_GetInt(str);
		bool cheating = (i == 0) ? (GVar_GetInt("useddevcmd") || GVar_GetInt("cdt_usingmod")) : pl->plObj->clCheater;
		char *winStr = (score == 1) ? "win" : "wins";
		if (cheating)
		{
			Util_StatusMessage("%s has %i %s, but may be cheating.", g_glb.clientInfo[i].infoStr, score, winStr);
		}
		else
		{
			Util_StatusMessage("%s has %i %s.", g_glb.clientInfo[i].infoStr, score, winStr);
		}
		if (score > bestScore)
		{
			bestScore = score;
			bestCl = i;
		}
		else if (score == bestScore)
		{
			tied = true;
		}
	}

	if (tied)
	{
		Util_StatusMessage("The winning position tied.");
	}
	else if (bestCl != -1)
	{
		Util_StatusMessage("%s is winning with %i.", g_glb.clientInfo[bestCl].infoStr, bestScore);
	}
}

//mp versus mode checks
static void LServ_VersusLogic(void)
{
	if (g_glb.mpCountDown > 0)
	{
		if (g_glb.mpCountTime < g_glb.gameTime)
		{
			gameObject_t *cam = Util_GetCam(0);
			g_glb.mpCountDown--;
			if (g_glb.mpCountDown == 0)
			{
				int *t = (int *)_alloca(sizeof(int)+sizeof(float)*3);
				*t = 600;
				float *c = (float *)(t+1);
				c[0] = 1.0f;
				c[1] = 1.0f;
				c[2] = 1.0f;
				g_sharedFn->Net_SendEventType(CLEV_TRIGGERFLASH, t, sizeof(int)+sizeof(float)*3, -1);
				ObjSound_Create(cam->net.pos, "assets/sound/cb/fistpower.wav", 1.0f, -1);
				Util_StatusMessage("*(1 0 0 1)FIGHT!");
				for (int i = 0; i < MAX_NET_CLIENTS; i++)
				{
					gameObject_t *obj = &g_gameObjects[i];
					if (!obj->inuse || !obj->plObj)
					{
						continue;
					}
					AI_StartAnim(obj, HUMANIM_IDLE, true);
				}
			}
			else
			{
				ObjSound_Create(cam->net.pos, "assets/sound/cb/bigfoot1.wav", 1.0f, -1);
				Util_StatusMessage("*(1 1 1 1)%i...", g_glb.mpCountDown);
				g_glb.mpCountTime = g_glb.gameTime+1000;
			}
		}
	}

	gameObject_t *living[MAX_NET_CLIENTS];
	int numLiving = 0;
	for (int i = 0; i < MAX_NET_CLIENTS; i++)
	{
		gameObject_t *obj = &g_gameObjects[i];
		if (!obj->inuse || !obj->plObj)
		{
			continue;
		}

		if (i < 2)
		{
			g_globalNet.vsHealth[i] = obj->health;
			if (g_globalNet.vsHealth[i] < 0)
			{
				g_globalNet.vsHealth[i] = 0;
			}
		}

		if (obj->health <= 0)
		{
			continue;
		}

		living[numLiving] = obj;
		numLiving++;
	}

	if (numLiving >= 2)
	{
		return;
	}

	if (numLiving <= 0)
	{
		Util_StatusMessage("Double K.O.!");
	}
	else
	{
		int clIndex = living[0]->net.index;
		Util_StatusMessage("%s is the victor!", g_glb.clientInfo[clIndex].infoStr);
		char str[64];
		sprintf(str, "pls_%s", g_glb.clientInfo[clIndex].infoStr);
		int score = GVar_GetInt(str);
		score++;
		GVar_SetInt(str, score);
	}

	LServ_ShowMPStats();

	g_glb.mpVersusMode = 0;
	Util_RunScript("obj_mpscript", "lshub_mpfinished");
}

//determine active lod level for object
void LServ_SetObjectLod(gameObject_t *obj)
{
	if (!obj || !obj->inuse || !obj->lodInfo)
	{
		return;
	}

	gameObject_t *cam = Util_GetCam(0);
	if (!cam->inuse)
	{
		return;
	}

	if (obj->lodInfo->lodDist <= 0.0f)
	{
		return;
	}

	float d[3];
	Math_VecSub(obj->net.pos, cam->net.pos, d);
	float camDist = Math_VecLen(d);
	float stp = floorf(camDist/obj->lodInfo->lodDist);
	int lodNum = (int)stp;
	if (lodNum >= obj->lodInfo->numLods)
	{
		lodNum = obj->lodInfo->numLods-1;
	}

	if (lodNum == obj->lodInfo->curLod)
	{ //no change
		return;
	}

	obj->net.strIndex = obj->lodInfo->lodList[lodNum];
	obj->lodInfo->curLod = lodNum;
}

//logic frame
void LServ_RunFrame(unsigned long prvTime, unsigned long curTime, bool paused)
{
	static float secondCounter = 0.0f;
	float timeDif = (float)(curTime-prvTime);

	secondCounter += timeDif;
	while (secondCounter >= 1000.0f)
	{ //increment persistent play time counter
		secondCounter -= 1000.0f;
		int sec = GVar_GetInt("playsec");
		sec++;
		GVar_SetInt("playsec", sec);
	}

	if (timeDif > 100.0f)
	{ //cap to something sane
		timeDif = 100.0f;
	}

	if (g_glb.actionTime)
	{
		if (g_glb.actionTime > g_glb.gameTime)
		{
			if (g_glb.timeType == TIMETYPE_NORMAL)
			{
				g_glb.timeType = TIMETYPE_ACTION;
			}
		}
		else
		{
			g_glb.actionTime = 0;
			if (g_glb.timeType == TIMETYPE_ACTION)
			{
				g_glb.timeType = TIMETYPE_NORMAL;
			}
		}
	}

	if (g_glb.tonberryTime)
	{
		if (g_glb.tonberryTime > g_glb.gameTime)
		{
			if (g_glb.timeType == TIMETYPE_NORMAL)
			{
				g_glb.timeType = TIMETYPE_TONBERRY;
			}
		}
		else
		{
			g_glb.tonberryTime = 0;
			if (g_glb.timeType == TIMETYPE_TONBERRY)
			{
				g_glb.timeType = TIMETYPE_NORMAL;
			}
		}
	}

	if (g_glb.cinema == 1 && g_gameObjects[0].inuse && g_gameObjects[0].plObj &&
		g_gameObjects[0].plObj->clButtons[BUTTON_EVENT1])
	{ //fast-forward
		g_glb.timeScale = 8.0f;
		g_glb.lastTimeType = -1;
	}
	else if (g_glb.timeType != g_glb.lastTimeType)
	{
		float plTime;
		switch (g_glb.timeType)
		{
		case TIMETYPE_NORMAL:
			g_glb.timeScale = 1.0f;
			plTime = 1.0f;
			break;
		case TIMETYPE_TIMEATTACK:
			g_glb.timeScale = 0.25f;
			plTime = 4.0f;
			break;
		case TIMETYPE_TONBERRY:
			g_glb.timeScale = 0.125f;
			plTime = 1.0f;
			break;
		case TIMETYPE_DEATH:
			g_glb.timeScale = 0.25f;
			plTime = 1.0f;
			break;
		case TIMETYPE_ACTION:
			g_glb.timeScale = g_glb.actionTimeScale;
			plTime = 1.0f;
			break;
		default:
			g_glb.timeScale = 1.0f;
			plTime = 1.0f;
			break;
		}
		for (int i = 0; i < MAX_NET_CLIENTS; i++)
		{
			gameObject_t *obj = &g_gameObjects[i];
			if (g_glb.timeType == TIMETYPE_TIMEATTACK)
			{
				obj->net.renderEffects |= FXFL_CONTRAST;
			}
			else
			{
				obj->net.renderEffects &= ~FXFL_CONTRAST;
			}
			obj->localTimeScale = plTime;
		}
		Util_GetCam(0)->localTimeScale = plTime;
		int ltt = g_glb.lastTimeType;
		g_glb.lastTimeType = g_glb.timeType;
		if (g_glb.timeType != TIMETYPE_DEATH && ltt != -1)
		{
			int *t = (int *)_alloca(sizeof(int)+sizeof(float)*3);
			*t = 300;
			float *c = (float *)(t+1);
			c[0] = 1.0f;
			c[1] = 1.0f;
			c[2] = 1.0f;
			g_sharedFn->Net_SendEventType(CLEV_TRIGGERFLASH, t, sizeof(int)+sizeof(float)*3, -1);
		}
	}
	else if (g_glb.timeType == TIMETYPE_ACTION)
	{
		g_glb.timeScale = g_glb.actionTimeScale;
	}

	g_glb.invTimeScale = 1.0f/g_glb.timeScale;
	timeDif *= g_glb.timeScale;

	g_glb.timeFactor = timeDif/50.0f; //compensate for dropped frames
	if (g_glb.timeFactor > 2.0f)
	{
		g_glb.timeFactor = 2.0f;
	}
	else if (g_glb.timeFactor < 0.1f)
	{
		g_glb.timeFactor = 0.1f;
	}

	if (paused)
	{
		return;
	}

	g_sharedFn->Coll_SetupVisibility(Util_GetCam(0)->net.pos);

	if (!g_glb.gameTime || prvTime == curTime)
	{
		g_glb.gameTime = (serverTime_t)curTime;
	}
	else
	{
		g_glb.gameTime += (serverTime_t)timeDif;
	}

	LServ_MusicLogic();

	AI_GlobalFrame();

	gameObject_t *activeLights[MAX_GAME_OBJECTS];
	int numActiveLights = 0;
	gameObject_t *obj;
	for (int i = 0; i < g_glb.gameObjectSlots; i++)
	{
		obj = &g_gameObjects[i];
		if (obj->inuse)
		{
			LServ_NetCull(obj);
			if (obj->net.type == OBJ_TYPE_LIGHT &&
				!(obj->inuse & INUSE_NONET) &&
				obj->net.lerpDuration)
			{ //add to the unculled lights list (only if it's a shadowing light)
				activeLights[numActiveLights++] = obj;
			}
			if (obj->lodInfo)
			{ //set the lod level based on cam position
				LServ_SetObjectLod(obj);
			}
			if (obj->think && obj->thinkTime < g_glb.gameTime && !obj->parent)
			{
				obj->think(obj, g_glb.timeFactor*obj->localTimeScale);
				if (obj->child)
				{ //think for the child
					obj->child->think(obj->child, g_glb.timeFactor*obj->localTimeScale);
				}
			}
		}
	}

	if (g_glb.mpVersusMode)
	{
		LServ_VersusLogic();
	}

	if (numActiveLights > 0)
	{ //check culled objects against lights
		for (int i = 0; i < g_glb.gameObjectSlots; i++)
		{
			obj = &g_gameObjects[i];

			if (!obj->inuse || !(obj->inuse & INUSE_NONET))
			{ //only interested if it is inuse and culled
				continue;
			}

			if (obj->net.type == OBJ_TYPE_LIGHT)
			{ //lights don't affect each other
				continue;
			}

			if (obj->localFlags & LFL_NONET)
			{ //never wants to be seen by the client
				continue;
			}

			if (obj->spawnMins[0] == 0.0f &&
				obj->spawnMins[1] == 0.0f &&
				obj->spawnMins[2] == 0.0f &&
				obj->spawnMaxs[0] == 0.0f &&
				obj->spawnMaxs[1] == 0.0f &&
				obj->spawnMaxs[2] == 0.0f)
			{ //spawn bounds not set
				continue;
			}

			for (int j = 0; j < numActiveLights; j++)
			{
				if (activeLights[j]->net.frame & LIGHTFL_FULLAMB)
				{ //do not consider pure ambient lights
					continue;
				}
				if (Util_GameObjectsOverlap(obj, activeLights[j]))
				{ //it interacts with a light in view, so make sure the client gets it.
					obj->inuse &= ~INUSE_NONET;
					break;
				}
			}
		}
	}

	//update client state data
	for (int i = 0; i < MAX_NET_CLIENTS; i++)
	{
		obj = &g_gameObjects[i];
		if (!obj->inuse || !obj->plObj)
		{
			continue;
		}
		ObjPlayer_UpdatePlayerStates(obj);
		RCUBE_ASSERT(sizeof(g_globalNet) < 32768); //make sure the delta can't get too big
		if (memcmp(&g_globalNet, &obj->plObj->lastGlobal, sizeof(g_globalNet)) != 0)
		{ //update global net data
			int msgSize = sizeof(g_globalNet)+1024;
			WORD *msgOut = (WORD *)_alloca(msgSize);
			BYTE *deltaOut = (BYTE *)(msgOut+1);
			int deltaSize = msgSize-4;

			int realDeltaSize = g_sharedFn->Net_CreateDelta(&obj->plObj->lastGlobal, &g_globalNet, sizeof(g_globalNet),
				deltaOut, deltaSize);
			memcpy(&obj->plObj->lastGlobal, &g_globalNet, sizeof(g_globalNet));
			RCUBE_ASSERT(realDeltaSize > 0);
		
			*msgOut = realDeltaSize; //first word for the event type is the delta size

			g_sharedFn->Net_SendEventType(CLEV_GLOBALDATA, msgOut, realDeltaSize+2, obj->net.index);
		}
	}

	//ObjVisBlock_DebugDraw();
}

//make sure global resources are cached by the client
void LServ_GlobalResources(void)
{
	g_sharedFn->Common_ServerString("^assets/textures/2dattn");
	g_sharedFn->Common_ServerString("^assets/textures/flashattn");
	g_sharedFn->Common_ServerString("^assets/textures/fetusredshell");
	g_sharedFn->Common_ServerString("$assets/sound/menu/introsnd.wav");

	//battle music
	g_sharedFn->Common_ServerString("$assets/music/FullFrontalAssault.ogg");

	g_sharedFn->Common_ServerString("$assets/sound/cb/jump01.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/meteorsmash.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/objland.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/hitbounce.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/swingaura.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/walljump01.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/groundsmash.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/hitaura.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/lstream.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/lstreamsh.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/lstreamint.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/lstreamint2.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/lunge.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/makopickup.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/makopop.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/roll01.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/swingflame.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/swingflame02.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/chargeaura.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/chargeflame.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/chargeheal.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/chargetime.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/chargeoff.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/chargeup.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/fire01.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/healstep.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/pldeath.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/jesus.wav");
	g_sharedFn->Common_ServerString("$assets/sound/items/itempickup.wav");
	g_sharedFn->Common_ServerString("$assets/sound/items/itempop.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/ironhit.wav");
	g_soundHitHeavy[0] = g_sharedFn->Common_ServerString("$assets/sound/cb/hitheavy01.wav");
	g_soundHitHeavy[1] = g_sharedFn->Common_ServerString("$assets/sound/cb/hitheavy02.wav");
	g_soundHitHeavy[2] = g_sharedFn->Common_ServerString("$assets/sound/cb/hitheavy03.wav");
	g_soundHitLight[0] = g_sharedFn->Common_ServerString("$assets/sound/cb/hitlight01.wav");
	g_soundHitLight[1] = g_sharedFn->Common_ServerString("$assets/sound/cb/hitlight02.wav");
	g_soundHitLight[2] = g_sharedFn->Common_ServerString("$assets/sound/cb/hitlight03.wav");
	g_soundHitLight[3] = g_sharedFn->Common_ServerString("$assets/sound/cb/hitlight04.wav");
	g_soundPunch[0] = g_sharedFn->Common_ServerString("$assets/sound/cb/punch01.wav");
	g_soundPunch[1] = g_sharedFn->Common_ServerString("$assets/sound/cb/punch02.wav");
	g_soundPunch[2] = g_sharedFn->Common_ServerString("$assets/sound/cb/punch03.wav");
	g_soundKick[0] = g_soundPunch[0];//g_sharedFn->Common_ServerString("$assets/sound/cb/kick01.wav");
	g_soundKick[1] = g_soundPunch[1];//g_sharedFn->Common_ServerString("$assets/sound/cb/kick02.wav");
	g_soundKick[2] = g_soundPunch[2];//g_sharedFn->Common_ServerString("$assets/sound/cb/kick03.wav");
	g_soundDeepBlow[0] = g_sharedFn->Common_ServerString("$assets/sound/cb/kick01.wav");
	g_soundDeepBlow[1] = g_sharedFn->Common_ServerString("$assets/sound/cb/kick02.wav");
	g_soundDeepBlow[2] = g_sharedFn->Common_ServerString("$assets/sound/cb/kick03.wav");
	g_soundFootsteps[0] = g_sharedFn->Common_ServerString("$assets/sound/cb/step01.wav");
	g_soundFootsteps[1] = g_sharedFn->Common_ServerString("$assets/sound/cb/step02.wav");
	g_soundFootsteps[2] = g_sharedFn->Common_ServerString("$assets/sound/cb/step03.wav");
	g_soundFootsteps[3] = g_sharedFn->Common_ServerString("$assets/sound/cb/step04.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/swdun.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/swdre.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/swdhit.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/slash01.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/slash02.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/slash03.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/bite01.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/bite02.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/buscut01.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/buscut02.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/buscut03.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/hitmetal.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/espawn.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/suck.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/quake.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/confuse.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cin/fast.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/crack01.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/crack02.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/crack03.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/fistpower.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/bigfoot1.wav");

	g_sharedFn->Common_ServerString("$assets/sound/cb/glock.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/bullet.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/bulflesh.wav");
	g_sharedFn->Common_ServerString("&&bulletspark");
	g_sharedFn->Common_ServerString("&&melee/bigslash");

	g_sharedFn->Common_ServerString("$assets/sound/items/rifle.wav");
	g_sharedFn->Common_ServerString("$assets/sound/items/usepotion.wav");
	g_sharedFn->Common_ServerString("$assets/sound/other/mmts.wav");
	g_sharedFn->Common_ServerString("$assets/sound/items/tr_silver.wav");
	g_sharedFn->Common_ServerString("$assets/sound/items/tr_gold.wav");
	for (int i = 0; i < NUM_INVENTORY_ITEM_DEFS; i++)
	{
		char s[256];
		const invItemDef_t *item = &g_invItems[i];
		if (item->useFX && item->useFX[0] && item->itemBehavior != INVITEM_BEHAVIOR_COSTUME)
		{
			if (item->useFX[0] != '*')
			{
				sprintf(s, "&&%s", item->useFX);
				g_sharedFn->Common_ServerString(s);
			}
			else
			{
				LServ_CacheObj(&item->useFX[1]);
			}
		}
	}

	g_sharedFn->Common_ServerString("&&melee/bleed");
	g_sharedFn->Common_ServerString("&&melee/bleedless");
	g_sharedFn->Common_ServerString("&&melee/impact");
	g_sharedFn->Common_ServerString("&&melee/impacten");
	g_sharedFn->Common_ServerString("&&melee/impactexplosive");
	g_sharedFn->Common_ServerString("&&melee/impactslash");
	g_sharedFn->Common_ServerString("&&impacts/flyback");
	g_sharedFn->Common_ServerString("&&impacts/meteorimpact");
	g_sharedFn->Common_ServerString("&&impacts/aircrush");
	g_sharedFn->Common_ServerString("&&impacts/aircrushsuper");
	g_sharedFn->Common_ServerString("&&impacts/airsword");
	g_sharedFn->Common_ServerString("&&general/makopickup");
	g_sharedFn->Common_ServerString("&&general/makofade");
	g_sharedFn->Common_ServerString("&&general/makodeath");
	g_sharedFn->Common_ServerString("&&general/makosit");
	g_sharedFn->Common_ServerString("&&general/itemfade");
	g_sharedFn->Common_ServerString("&&general/itempickup");
	g_sharedFn->Common_ServerString("&&general/droppeditem");
	g_sharedFn->Common_ServerString("&&general/makochargeup");
	g_sharedFn->Common_ServerString("&&impacts/wallhop");
	g_sharedFn->Common_ServerString("&&playerfire");
	g_sharedFn->Common_ServerString("&&items/vortex");

	LServ_CacheObj("obj_mako_drop");
	LServ_CacheObj("obj_item_drop");

	LServ_CacheObj("obj_aerithshell");

	ObjProjectile_Init();

	g_globalNet.skyboxModel = g_sharedFn->Common_ServerString("&assets/models/levels/skyshell4.rdm");
	//g_globalNet.skyboxModel = g_sharedFn->Common_ServerString("&assets/models/levels/skyshell5.rdm");
	//g_globalNet.skyboxTerrain = g_sharedFn->Common_ServerString("&assets/models/terrain2_01.rdm");
}

//set up the main map object
void LServ_InitGlobalMapObj(void)
{
	gameObject_t *map = &g_gameObjects[MAP_OBJ_NUM];

	map->inuse = INUSE_ACTIVE;
	map->net.index = MAP_OBJ_NUM;
	map->creationCount = g_glb.creationCount;
	g_glb.creationCount++;
	map->localTimeScale = 1.0f;
	map->net.type = OBJ_TYPE_MAP;
	map->net.pos[0] = 0.0f;
	map->net.pos[1] = 0.0f;
	map->net.pos[2] = 0.0f;

	map->net.ang[0] = 0.0f;
	map->net.ang[1] = 0.0f;
	map->net.ang[2] = 0.0f;

	map->net.entNameIndex = g_sharedFn->Common_ServerString("obj_map");

	map->inuse |= INUSE_NONET; //for fetus blaster

	map->net.solid = 1;

	//Fetus Blaster - init the camera tracker
	gameObject_t *cam = Util_GetCam(0);

	g_glb.camFrustumFresh = false;
	g_glb.camBoneBolt[0] = 0;
	g_glb.camBoltOffset[0] = 0.0f;
	g_glb.camBoltOffset[1] = 0.0f;
	g_glb.camBoltOffset[2] = 0.0f;
	g_glb.camAbsOffset[0] = 0.0f;
	g_glb.camAbsOffset[1] = 0.0f;
	g_glb.camAbsOffset[2] = 0.0f;
	g_glb.camAbsAngOffset[0] = 0.0f;
	g_glb.camAbsAngOffset[1] = 0.0f;
	g_glb.camAbsAngOffset[2] = 0.0f;
	g_glb.camSlack = 1.0f;
	cam->inuse = INUSE_ACTIVE;
	cam->targetName = "__the_camera";
	cam->net.staticIndex = -1;
	cam->net.index = CAM_TRACK_NUM;
	cam->creationCount = g_glb.creationCount;
	g_glb.creationCount++;
	cam->localTimeScale = 1.0f;
	cam->net.type = OBJ_TYPE_CAM;
	cam->net.pos[0] = 0.0f;
	cam->net.pos[1] = 0.0f;
	cam->net.pos[2] = DEFAULT_CAM_Z;

	cam->camTrackPos[0] = 0.0f;
	cam->camTrackPos[1] = 0.0f;
	cam->camTrackPos[2] = DEFAULT_CAM_Z;

	cam->net.ang[0] = 180.0f;
	cam->net.ang[1] = 0.0f;
	cam->net.ang[2] = 0.0f;

	//g_levelScroll[0] = 0.0f;
	//g_levelScroll[1] = 0.0f;
	//g_levelScroll[2] = 0.0f;

	cam->net.maxs[0] = 0.0f;//g_levelScroll[0];
	cam->net.maxs[1] = 0.0f;//g_levelScroll[1];

	cam->net.modelScale[0] = 70.0f; //default fov

	//cam->net.renderEffects |= FXFL_FPSMODE;

	cam->net.entNameIndex = g_sharedFn->Common_ServerString("obj_camtrack");

	cam->think = ObjCam_Think;
	cam->thinkTime = 0;
}

//client disconnected
void LServ_FreePlayerObj(int clientNum)
{
	gameObject_t *pl = &g_gameObjects[clientNum];
	if (pl->inuse)
	{
		if (pl->plObj)
		{
			if (pl->plObj->targetObj)
			{
				pl->plObj->targetObj->net.renderEffects &= ~ObjPlayer_TargetBit(pl);
			}
			if (pl->plObj->ride)
			{
				ObjPlayer_UnmountRide(pl, true);
				pl->plObj->ride = NULL;
			}
			if (pl->plObj->flashlight)
			{
				pl->plObj->flashlight->think = ObjGeneral_RemoveThink;
				pl->plObj->flashlight = NULL;
			}
			if (pl->plObj->sword)
			{
				pl->plObj->sword->think = ObjGeneral_RemoveThink;
				pl->plObj->sword = NULL;
			}
			if (pl->plObj->gun)
			{
				pl->plObj->gun->think = ObjGeneral_RemoveThink;
				pl->plObj->gun = NULL;
			}
			if (pl->plObj->aerith)
			{
				pl->plObj->aerith->think = ObjGeneral_RemoveThink;
				pl->plObj->aerith = NULL;
			}
		}
		LServ_FreeObject(pl);
	}
}

//cache all stuff applicable
void LServ_CacheObj(const char *objName)
{
	const char *s;
	char assetName[2048];
	int i;
	int type;
	BYTE *b = Common_GetEntryForObject(objName);
	if (!b)
	{ //no entry for this object then
		return;
	}

	Util_ParseObjType(Common_GetValForKey(b, "type"), &type);

	s = Common_GetValForKey(b, "swordcache");
	if (s && atoi(s))
	{
		ObjSword_RegMedia(b, "swdHitSnd", NULL, false);
		ObjSword_RegMedia(b, "swdSwing", NULL, false);
		ObjSword_RegMedia(b, "swdImpact", NULL, false);
		ObjSword_RegMedia(b, "swdUnholst", NULL, false);
		ObjSword_RegMedia(b, "swdHolst", NULL, false);
		ObjSword_RegMedia(b, "swdHitFx", NULL, true);
		ObjSword_RegMedia(b, "swdImpactFx", NULL, true);
	}

	char resCacheStr[256];
	i = 0;
	sprintf(resCacheStr, "soundcache%i", i);
	s = Common_GetValForKey(b, resCacheStr);
	while (s && s[0] && i < 256)
	{
		sprintf(assetName, "$%s", s);
		g_sharedFn->Common_ServerString(assetName);
		i++;
		sprintf(resCacheStr, "soundcache%i", i);
		s = Common_GetValForKey(b, resCacheStr);
	}

	i = 0;
	sprintf(resCacheStr, "rplcache%i", i);
	s = Common_GetValForKey(b, resCacheStr);
	while (s && s[0] && i < 256)
	{
		sprintf(assetName, "&&%s", s);
		g_sharedFn->Common_ServerString(assetName);
		i++;
		sprintf(resCacheStr, "rplcache%i", i);
		s = Common_GetValForKey(b, resCacheStr);
	}

	s = Common_GetValForKey(b, "customTex");
	if (s && s[0])
	{ //make sure the client caches this texture
		sprintf(assetName, "^%s", s);
		g_sharedFn->Common_ServerString(assetName);
	}

	//precache particle effects
	s = Common_GetValForKey(b, "rpl_muzzle");
	if (s && s[0])
	{
		sprintf(assetName, "&&%s", s);
		g_sharedFn->Common_ServerString(assetName);
	}
	s = Common_GetValForKey(b, "rpl_muzzlechg");
	if (s && s[0])
	{
		sprintf(assetName, "&&%s", s);
		g_sharedFn->Common_ServerString(assetName);
	}
	s = Common_GetValForKey(b, "rpl_impact");
	if (s && s[0])
	{
		sprintf(assetName, "&&%s", s);
		g_sharedFn->Common_ServerString(assetName);
	}

	//precache decals
	s = Common_GetValForKey(b, "dcl_impact");
	if (s && s[0])
	{
		sprintf(assetName, "^%s", s);
		g_sharedFn->Common_ServerString(assetName);
	}

	//base on the type we need a different prepended char so the client
	//knows what to cache based on string name alone
	i = 0;
	if (type == OBJ_TYPE_SPRITE || type == OBJ_TYPE_PROJECTILE)
	{
		assetName[i++] = '#';
	}
	else if (type == OBJ_TYPE_MODEL)
	{
		assetName[i++] = '&';
	}
	assetName[i] = 0;

	//cache any lod levels as well
	s = Common_GetValForKey(b, "numLods");
	if (s && s[0])
	{
		int numLods = atoi(s);
		for (int i = 0; i < numLods; i++)
		{
			char lodStr[256];
			sprintf(lodStr, "lod%i", i);
			s = Common_GetValForKey(b, lodStr);
			if (s && s[0])
			{
				char lodName[2048];
				sprintf(lodName, "%s%s", assetName, s);
				g_sharedFn->Common_ServerString(lodName);
			}
		}
	}

	s = Common_GetValForKey(b, "assetName");
	if (s && s[0])
	{
		strcat(assetName, s);
		g_sharedFn->Common_ServerString(assetName);
	}

	//check modelAnim
	s = Common_GetValForKey(b, "modelAnim");
	if (s && s[0])
	{
		assetName[0] = '@';
		assetName[1] = 0;
		strcat(assetName, s);
		g_sharedFn->Common_ServerString(assetName);
	}

	//rclip functionality
	s = Common_GetValForKey(b, "clipModel");
	if (!s)
	{
		s = "*sphere";
	}
	if (s[0] != '*')
	{ //if it's a custom clip model, it needs to be cached
		const char *clipAnim = Common_GetValForKey(b, "clipAnim");
		g_sharedFn->Coll_CacheModel(s, clipAnim);
	}
}

gameObject_t *LServ_ObjectFromName(const char *objName, float *pos, float *ang, const objArgs_t *args, int numArgs,
								   gameObject_t *useObj)
{
	const char *s;
	char assetName[2048];
	BYTE *b = Common_GetEntryForObject(objName);
	int i;
	if (!b)
	{ //no entry for this object then
		return NULL;
	}

	gameObject_t *obj;
	if (useObj)
	{
		obj = useObj;
	}
	else
	{
		obj = LServ_CreateObject();
		if (!obj)
		{ //oh no
			return NULL;
		}
	}
	obj->spawnArgs = args;
	obj->numSpawnArgs = numArgs;

	obj->objData = b;

	Util_ParseObjType(Common_GetValForKey(b, "type"), &obj->net.type);

	Util_ParseObjSpawn(Common_GetValForKey(b, "customSpawn"), obj);

	obj->think = ObjGeneral_Think;
	obj->thinkTime = 0;

	if (pos)
	{
		obj->net.pos[0] = pos[0];
		obj->net.pos[1] = pos[1];
		obj->net.pos[2] = pos[2];
	}

	if (ang)
	{
		obj->net.ang[0] = ang[0];
		obj->net.ang[1] = ang[1];
		obj->net.ang[2] = ang[2];
	}

	Util_ParseVector(Common_GetValForKey(b, "spawnMins"), obj->spawnMins);
	Util_ParseVector(Common_GetValForKey(b, "spawnMaxs"), obj->spawnMaxs);

	Util_ParseVector(Common_GetValForKey(b, "mins"), obj->net.mins);
	Util_ParseVector(Common_GetValForKey(b, "maxs"), obj->net.maxs);
	obj->radius = Util_RadiusFromBounds(obj->net.mins, obj->net.maxs, 1.0f);
	Util_ParseVector(Common_GetValForKey(b, "modelScale"), obj->net.modelScale);

	Util_ParseInt(Common_GetValForKey(b, "hurtable"), &obj->hurtable);
	Util_ParseInt(Common_GetValForKey(b, "health"), &obj->health);

	Util_ParseInt(Common_GetValForKey(b, "solid"), &obj->net.solid);

	Util_ParseInt(Common_GetValForKey(b, "renderfx"), &obj->net.renderEffects);

	s = Common_GetValForKey(b, "customTex");
	if (s && s[0])
	{ //make sure the client caches this texture
		sprintf(assetName, "^%s", s);
		g_sharedFn->Common_ServerString(assetName);
	}

	obj->net.entNameIndex = g_sharedFn->Common_ServerString(objName);

	//base on the type we need a different prepended char so the client
	//knows what to cache based on string name alone
	i = 0;
	if (obj->net.type == OBJ_TYPE_SPRITE || obj->net.type == OBJ_TYPE_PROJECTILE)
	{
		assetName[i++] = '#';
	}
	else if (obj->net.type == OBJ_TYPE_MODEL)
	{
		assetName[i++] = '&';
	}
	else if (obj->net.type == OBJ_TYPE_BEAM)
	{
		assetName[i++] = '^';
	}
	assetName[i] = 0;

	//create/cache lod levels
	s = Common_GetValForKey(b, "numLods");
	if (s && s[0])
	{
		int numLods = atoi(s);
		if (numLods > 0)
		{
			obj->lodInfo = (lodInfo_t *)g_sharedFn->Common_RCMalloc(sizeof(lodInfo_t));
			memset(obj->lodInfo, 0, sizeof(lodInfo_t));

			if (numLods >= MAX_GAME_LODS)
			{
				numLods = MAX_GAME_LODS-1;
			}
			obj->lodInfo->numLods = numLods+1;
			s = Common_GetValForKey(b, "lodDist");
			obj->lodInfo->lodDist = (s && s[0]) ? (float)atof(s) : 256.0f;
			for (int i = 0; i < numLods; i++)
			{
				char lodStr[256];
				sprintf(lodStr, "lod%i", i);
				s = Common_GetValForKey(b, lodStr);
				if (s && s[0])
				{
					char lodName[2048];
					sprintf(lodName, "%s%s", assetName, s);
					obj->lodInfo->lodList[i+1] = g_sharedFn->Common_ServerString(lodName);
				}
			}
		}
	}

	s = Common_GetValForKey(b, "assetName");
	if (s && s[0])
	{
		strcat(assetName, s);
		obj->net.strIndex = g_sharedFn->Common_ServerString(assetName);
		if (obj->lodInfo)
		{ //first lod level is the normal asset
			obj->lodInfo->lodList[0] = obj->net.strIndex;
		}
	}

	//check modelAnim
	s = Common_GetValForKey(b, "modelAnim");
	if (s && s[0])
	{
		assetName[0] = '@';
		assetName[1] = 0;
		strcat(assetName, s);
		obj->net.strIndexB = g_sharedFn->Common_ServerString(assetName);
	}
	else
	{
		obj->net.strIndexB = 0;
	}

	s = Common_GetValForKey(b, "lightBlocking");
	if (s && s[0] && atoi(s) == 1)
	{ //blocks static light, take this to mean it is static
		obj->net.staticIndex = obj->net.index;
	}

	if (obj->spawnArgs && obj->numSpawnArgs > 0)
	{
		for (int i = 0; i < obj->numSpawnArgs; i++)
		{
			const objArgs_t *arg = obj->spawnArgs+i;
			if (!_stricmp(arg->key, "objMins"))
			{ //mins override
				Util_ParseVector(arg->val, obj->spawnMins);
				Math_VecCopy(obj->spawnMins, obj->net.mins);
			}
			else if (!_stricmp(arg->key, "objMaxs"))
			{ //maxs override
				Util_ParseVector(arg->val, obj->spawnMaxs);
				Math_VecCopy(obj->spawnMaxs, obj->net.maxs);
			}
		}
	}

	//rclip functionality
	if (obj->net.solid)
	{
		//update nxactor
		s = Common_GetValForKey(b, "clipModel");
		if (s && s[0])
		{
            obj->rcColModel = g_sharedFn->Coll_RegisterModelInstance(s);
			if (obj->rcColModel)
			{
				const char *clipRadStr = Common_GetValForKey(b, "clipRadius");
				obj->rcColModel->radius = (clipRadStr && clipRadStr[0]) ? (float)atof(clipRadStr) : Math_Max2(obj->net.maxs[0], obj->net.maxs[1]);
				s = Common_GetValForKey(b, "clipBox");
				if (s && atoi(s))
				{
					obj->rcColModel->clipFlags = CLIPFL_BOXMOVE;
				}
				Math_VecCopy(obj->net.mins, obj->rcColModel->mins);
				Math_VecCopy(obj->net.maxs, obj->rcColModel->maxs);
				Math_VecCopy(obj->net.pos, obj->rcColModel->pos);
				Math_VecCopy(obj->net.ang, obj->rcColModel->ang);
				Math_VecCopy(obj->net.modelScale, obj->rcColModel->modelScale);
				obj->rcColModel->gameOwner = obj->net.index;
				obj->rcColModel->solid = obj->net.solid;
				obj->rcColModel->frame = obj->net.frame;
				obj->rcColModel->blendFrame = obj->net.frame;
				const char *clipAnim = Common_GetValForKey(b, "clipAnim");
				g_sharedFn->Coll_UpdateModel(obj->rcColModel, clipAnim);
			}
		}
	}

	if (obj->customSpawn)
	{
		obj->customSpawn(obj, b, args, numArgs);
	}

	return obj;
}

//time update outside of frame when necessary
void LServ_UpdateTime(unsigned long time)
{
	g_glb.gameTime = (serverTime_t)time;
}

//the engine will call this function for each map object.
//it's our responsibility to create an object in the game structure.
void LServ_InitMapObj(const char *objName, float *pos, float *ang, const objArgs_t *args, int numArgs)
{
	if (!g_sessLoad)
	{
		LServ_ObjectFromName(objName, pos, ang, args, numArgs);
	}
}

//perform any tasks that should be performed immediately after map spawn
void LServ_PostMapSpawn(void)
{
	//set up target name pointers
	for (int i = 0; i < g_glb.gameObjectSlots; i++)
	{
		gameObject_t *obj = &g_gameObjects[i];
		if (obj->inuse && obj->spawnArgs && obj->numSpawnArgs > 0)
		{
			for (int j = 0; j < obj->numSpawnArgs; j++)
			{
				const objArgs_t *arg = obj->spawnArgs+j;
				if (!_stricmp(arg->key, "targname"))
				{
					obj->targetName = arg->val;
					break;
				}
			}
		}
	}

	for (int i = 0; i < g_glb.gameObjectSlots; i++)
	{
		//target objects to each other
		gameObject_t *obj = &g_gameObjects[i];
		if (obj->inuse && obj->spawnArgs && obj->numSpawnArgs > 0)
		{
			for (int j = 0; j < obj->numSpawnArgs; j++)
			{
				const objArgs_t *arg = obj->spawnArgs+j;
				if (!_stricmp(arg->key, "camstart"))
				{ //camera's target
					gameObject_t *cam = Util_GetCam(0);
					if (cam->inuse)
					{
                        //cam->target = obj;
						cam->net.pos[0] = obj->net.pos[0];
						cam->net.pos[1] = obj->net.pos[1];
						cam->net.pos[2] = obj->net.pos[2];
						cam->camTrackPos[0] = obj->net.pos[0];
						cam->camTrackPos[1] = obj->net.pos[1];
						cam->camTrackPos[2] = obj->net.pos[2];
					}
				}
				else if (!_stricmp(arg->key, "targobj"))
				{ //targeting something else
					obj->target = Util_FindTargetObject(arg->val);
				}
				else if (!_stricmp(arg->key, "physpar"))
				{ //parented to something for physics
					obj->physParent = Util_FindTargetObject(arg->val);
					if (obj->physParent)
					{
						if (!obj->physParent->physChildren)
						{
							obj->physParent->physChildren = (gameObject_t **)g_sharedFn->Common_RCMalloc(sizeof(gameObject_t **)*MAX_PHYSICS_CHILDREN);
						}
						if (obj->physParent->numPhysChildren < MAX_PHYSICS_CHILDREN)
						{
							obj->physParent->physChildren[obj->numPhysChildren] = obj;
							obj->physParent->numPhysChildren++;
						}
						else
						{
							obj->physParent = NULL;
						}
					}
				}
			}
		}
	}

	ObjVisBlock_EstablishVis();

	//call post-spawns
	for (int i = 0; i < g_glb.gameObjectSlots; i++)
	{
		gameObject_t *obj = &g_gameObjects[i];
		if (obj->postSpawn)
		{
			obj->postSpawn(obj);
		}
	}
}

//physics impact callback (for novodex)
void LServ_PhysicsImpact(int objNum, int otherNum, const float *planeDir, const float *point)
{
}

//get information about the game object
void LServ_ObjectInfo(void **baseAddr, int *size, int *num, int *netSize, int **activeObjPtr)
{
	*baseAddr = &g_gameObjects[0];
	*size = sizeof(gameObject_t);
	*num = MAX_GAME_OBJECTS;
	*netSize = sizeof(gameObjNet_t);
	*activeObjPtr = &g_glb.gameObjectSlots;
}

//new input from a client
void LServ_ClientInput(int clientNum, BYTE *buttons, BYTE *userData, int userDataSize)
{ //it would be better to cache these and run them all in a single server frame for smoother input.
	gameObject_t *pl = &g_gameObjects[clientNum];
	if (!pl->plObj)
	{
		return;
	}
	if (userDataSize == sizeof(int))
	{
		int eq = *((int *)userData);
		if (eq >= 0 && eq < MAX_PLAYER_INVENTORY_ITEMS)
		{
			pl->plObj->desiredItem = eq;
		}
	}
	memcpy(pl->plObj->clButtonsBackBuffer[pl->plObj->numButtonsBuffered], pl->plObj->clButtons, sizeof(pl->plObj->clButtons));
	if (pl->plObj->numButtonsBuffered < MAX_BUTTON_BUFFER-1)
	{
		pl->plObj->numButtonsBuffered++;
	}
	memcpy(pl->plObj->clButtons, buttons, sizeof(pl->plObj->clButtons));
}

//the client's view angles changed (in respect to the player object)
void LServ_ClientInputAngle(int clientNum, float *angles)
{
	gameObject_t *pl = &g_gameObjects[clientNum];
	if (!pl->plObj)
	{
		return;
	}
	memcpy(pl->plObj->clAngles, angles, sizeof(pl->plObj->clAngles));
	if (!pl->plObj->hasLastClAngles)
	{
		Math_VecCopy(pl->plObj->clAngles, pl->plObj->lastClAngles);
		pl->plObj->hasLastClAngles = true;
	}
}

//the client's analog controller movement
void LServ_ClientInputAnalog(int clientNum, WORD *analogs)
{
	gameObject_t *pl = &g_gameObjects[clientNum];
	if (!pl->plObj)
	{
		return;
	}
	memcpy(pl->plObj->clAnalog, analogs, sizeof(pl->plObj->clAnalog));
	pl->plObj->hasClAnalog = true;
}

//music was restarted/changed
void LServ_MusicalReset(const char *musicName)
{
}

//set game global defaults
void LServ_SetGameGlobals(void)
{
	g_glb.timeScale = 1.0f;
	g_glb.invTimeScale = 1.0f;
	g_glb.actionTimeScale = 0.125f;
	g_glb.musicStrIndex = -1;

	g_glb.levelMusic[0] = '$';
	g_glb.levelMusic[1] = '$';
	g_glb.levelMusic[2] = 0;
	g_glb.battleMusicEnabled = true;

	g_glb.initialFPSAng[1] = 210.0f;

	g_glb.scriptFogValues[3] = 1.0f;
}

//load session
void LServ_LoadSessionData(void)
{
	if (g_sessLoad)
	{ //load session data now
		LServ_LoadSession(g_sessLoad);
		g_sessLoad = NULL;
	}
}

//module init
int LServ_Initialize(sharedSVFunctions_t *sharedFunc, const char *mapName, void *val, int valSize, bool queryOnly, bool multiplayer,
					 sessHeader_t *sessLoad)
{
	memset(&g_glb, 0, sizeof(g_glb));
	LServ_SetGameGlobals();
	strcpy(g_glb.mapName, mapName);
	g_sessLoad = sessLoad;

	g_sharedFn = sharedFunc;
	g_sharedFn->LServ_RunFrame = LServ_RunFrame;
	g_sharedFn->LServ_ObjectInfo = LServ_ObjectInfo;
	g_sharedFn->LServ_ClientSaveGame = LServ_ClientSaveGame;
	g_sharedFn->LServ_ClientInput = LServ_ClientInput;
	g_sharedFn->LServ_ClientInputAngle = LServ_ClientInputAngle;
	g_sharedFn->LServ_ClientInputAnalog = LServ_ClientInputAnalog;
	g_sharedFn->LServ_InitPlayerObj = LServ_InitPlayerObj;
	g_sharedFn->LServ_FreePlayerObj = LServ_FreePlayerObj;
	g_sharedFn->LServ_UpdateTime = LServ_UpdateTime;
	g_sharedFn->LServ_InitMapObj = LServ_InitMapObj;
	g_sharedFn->LServ_PostMapSpawn = LServ_PostMapSpawn;
	g_sharedFn->LServ_ChangeMusic = LServ_ChangeMusic;
	g_sharedFn->LServ_DevCommand = LServ_DevCommand;
	g_sharedFn->LServ_ServerEvent = LServ_ServerEvent;
	g_sharedFn->LServ_MusicalReset = LServ_MusicalReset;
	g_sharedFn->LServ_PhysicsImpact = LServ_PhysicsImpact;
	g_sharedFn->LServ_SaveGame = LServ_SaveGame;
	g_sharedFn->LServ_LoadGame = LServ_LoadGame;
	g_sharedFn->LServ_LoadSessionData = LServ_LoadSessionData;

	g_glb.runningMultiplayer = multiplayer;

	g_sharedFn->Common_GetEnOp(ENOP_TOUCHBASED, &g_gameTouchBased);
	if (!queryOnly)
	{ //don't bother loading stuff if this is only a query
		sharedInterface_t si;
		si.Parse_InitParser = g_sharedFn->Parse_InitParser;
		si.Parse_InitParserFromFile = g_sharedFn->Parse_InitParserFromFile;
		si.Parse_FreeParser = g_sharedFn->Parse_FreeParser;
		si.Parse_EnableInclude = g_sharedFn->Parse_EnableInclude;
		si.Parse_GetNextToken = g_sharedFn->Parse_GetNextToken;
		Shared_LoadUserItems(&si);

		g_glb.creationCount = 0;
		LServ_InitGlobalMapObj();
		LServ_GlobalResources();
		RScr_Init();
		GVar_Init();
		Phys_InitRigidBodies();

		if (!g_sessLoad)
		{
			g_glb.ai.weakAI = true;
			if (val)
			{
				cntStream_t *st = Stream_Alloc(val, valSize);
				LServ_ReadGameStream(st);
				Stream_Free(st);
			}
		}

		//don't preserve this gvar across map changes
		GVar_SetInt("intriggersequence", 0);

		//parse the general script file now
		RScr_ParseScriptFile("assets/rscript/general.rsc");
	}

	return 1;
}

//module shutdown
void LServ_Shutdown(void)
{
	gameObject_t *pl = &g_gameObjects[0];
	if (!pl->inuse || pl->health > 0)
	{ //don't write out values if the player is dead (let the values from last time they were alive be restored instead)
		cntStream_t *st = Stream_Alloc(NULL, 8192);
		LServ_WriteGameStream(st);
		g_sharedFn->Common_SaveGameValues(Stream_Buffer(st), Stream_Size(st));
		Stream_Free(st);
	}

	RScr_Shutdown();
	GVar_Shutdown();
	Phys_ShutdownRigidBodies();

	int i = 0;
	while (i < MAX_GAME_OBJECTS)
	{
		if (g_gameObjects[i].inuse)
		{
			LServ_FreeObject(&g_gameObjects[i]);
		}
		i++;
	}
	memset(&g_glb, 0, sizeof(g_glb));
}
