/*
=============================================================================
Module Information
------------------
Name:			obj_camtrack.cpp
Author:			Rich Whitehouse
Description:	server logic object: fetus camera tracker
=============================================================================
*/

#include "main.h"

//do stuff if appropriate
void ObjCamMarker_Trigger(gameObject_t *obj, gameObject_t *other, const collObj_t *col)
{
	if (obj->spawnArgs && obj->numSpawnArgs > 0)
	{
		int i = 0;
		while (i < obj->numSpawnArgs)
		{
			const objArgs_t *arg = obj->spawnArgs+i;
			if (!_stricmp(arg->key, "camflip"))
			{
				//g_fetusSideMode = !!atoi(arg->val);
				//g_noConfineTime = g_glb.gameTime+2000;
			}
			else if (!_stricmp(arg->key, "camspeed"))
			{
				//sscanf(arg->val, "(%f %f %f)", &g_levelScroll[0], &g_levelScroll[1], &g_levelScroll[2]);
			}
			i++;
		}
	}
}

//camera marker spawn function
void ObjCamMarker_Spawn(gameObject_t *obj, BYTE *b, const objArgs_t *args, int numArgs)
{
	obj->localFlags |= LFL_NONET;
	obj->inuse |= INUSE_NONET; //don't need markers sent to the client ever.
	obj->touch = ObjCamMarker_Trigger;

	if (obj->spawnArgs && obj->numSpawnArgs > 0)
	{
		int i = 0;
		while (i < obj->numSpawnArgs)
		{
			const objArgs_t *arg = obj->spawnArgs+i;
			if (!_stricmp(arg->key, "fps"))
			{
				if (atoi(arg->val) == 1)
				{
					g_glb.initialFPSPos[0] = obj->net.pos[0];
					g_glb.initialFPSPos[1] = obj->net.pos[1];
					g_glb.initialFPSPos[2] = obj->net.pos[2];
					g_glb.initialFPSAng[0] = obj->net.ang[0];
					g_glb.initialFPSAng[1] = obj->net.ang[1];
					g_glb.initialFPSAng[2] = obj->net.ang[2];
				}
				break;
			}
			else if (!_stricmp(arg->key, "musical"))
			{
				gameObject_t *cam = Util_GetCam(0);
				if (cam->inuse)
				{
					cam->net.renderEffects2 |= FXFL2_MUSICAL;
				}
				//g_musical = true;
			}
			i++;
		}
	}
}

//remove something if it is not within the camera visibility range
void ObjCam_Removal(gameObject_t *obj)
{
	if (!g_glb.camFrustumFresh)
	{
		return;
	}

	float radius = Util_RadiusFromBounds(obj->spawnMins, obj->spawnMaxs, 3.0f);
	if (!radius)
	{
		radius = 128.0f;
	}
	if (!Math_PointInFrustum(&g_glb.camFrustum, obj->net.pos, radius))
	{
		LServ_FreeObject(obj);
	}
}

//lerp cam angle
static float ObjCam_LerpAngle(float in, float goal, float factor)
{
	if (factor > 1.0f)
	{
		factor = 1.0f;
	}

	float current = Math_AngleMod(in);
	if (current == goal)
	{
		return goal;
	}

	float thresh = 25.0f;
	float dif = goal-current;

	if (goal > current)
	{
		if (dif >= 180.0f)
		{
			dif = dif-360.0f;
		}
	}
	else
	{
		if (dif <= -180.0f)
		{
			dif = dif+360.0f;
		}
	}

	if (dif > 0.0f)
	{
		dif -= thresh;
		if (dif < 0.0f)
		{
			dif = 0.0f;
		}
	}
	else if (dif < 0.0f)
	{
		dif += thresh;
		if (dif > 0.0f)
		{
			dif = 0.0f;
		}
	}

	if (fabsf(dif) < factor*0.2f)
	{
		return current;
	}

	dif *= factor;

	return Math_AngleMod(current+dif);
}

//mod and wrap to 180
float ObjCam_NormalizedAngle(float angle)
{
	angle = Math_AngleMod(angle);
	if (angle > 180.0f)
	{
		angle -= 360.0f;
	}
	return angle;
}

//frame function
void ObjCam_Think(gameObject_t *obj, float timeMod)
{
	//keep cam angles normalized
	obj->net.ang[0] = ObjCam_NormalizedAngle(obj->net.ang[0]);
	obj->net.ang[1] = ObjCam_NormalizedAngle(obj->net.ang[1]);
	obj->net.ang[2] = ObjCam_NormalizedAngle(obj->net.ang[2]);

	bool autoAngling = false;
	bool trackingTarg = false;
	gameObject_t *camPlayer = &g_gameObjects[0];
	RCUBE_ASSERT(camPlayer->plObj);
	if (g_glb.cinema != 1)
	{
		if (camPlayer->plObj->camTarget && !g_glb.runningMultiplayer)
		{
			float angleBlendScale = 3.0f;
			float ang[3];
			Math_VecSub(camPlayer->plObj->camTarget->net.pos, camPlayer->net.pos, ang);
			Math_VecToAngles(ang, ang);
			float f = camPlayer->plObj->camRange*0.03f;
			if (f > 90.0f)
			{
				f = 90.0f;
			}
			ang[PITCH] += f;
			ang[YAW] += 35.0f;
			ang[0] = Math_AngleMod(ang[0]);
			ang[1] = Math_AngleMod(ang[1]);
			ang[2] = Math_AngleMod(ang[2]);
			obj->net.ang[PITCH] = Math_BlendAngleLinear(obj->net.ang[PITCH], ang[PITCH], timeMod*angleBlendScale);
			obj->net.ang[YAW] = Math_BlendAngleLinear(obj->net.ang[YAW], ang[YAW], timeMod*angleBlendScale);
			obj->net.ang[ROLL] = Math_BlendAngleLinear(obj->net.ang[ROLL], ang[ROLL], timeMod*angleBlendScale);
		}
		else
		{
			float preAng[3];
			Math_VecCopy(obj->net.ang, preAng);
			if (g_glb.mpVersusMode)
			{
				obj->net.ang[0] = 30.0f;
				obj->net.ang[1] = 90.0f;
				obj->net.ang[2] = 0.0f;
			}
			else
			{
				ObjPlayer_AdjustTurnAngles(camPlayer, timeMod, obj->net.ang);
				if (g_glb.runningMultiplayer)
				{ //let everyone fuck with the camera
					for (int i = 1; i < MAX_NET_CLIENTS; i++)
					{
						gameObject_t *otherPl = &g_gameObjects[i];
						if (otherPl->inuse && otherPl->plObj)
						{
							ObjPlayer_AdjustTurnAngles(otherPl, timeMod, obj->net.ang);
						}
					}
				}
			}

			if (preAng[0] != obj->net.ang[0] ||
				preAng[1] != obj->net.ang[1] ||
				preAng[2] != obj->net.ang[2])
			{
				obj->debounce2 = g_glb.gameTime + 1000;
			}
			if (!g_glb.runningMultiplayer && obj->debounce2 < g_glb.gameTime && camPlayer->plObj->targetObj &&
				camPlayer->plObj->targetObj->inuse)
			{ //adjust angles for them if they haven't tried and are targetting something
				float angleBlendScale = 0.2f;
				float ang[3];
				float targCenter[3], plCenter[3];
				Util_GameObjectCenter(camPlayer, plCenter);
				plCenter[2] += 285.0f;
				Util_GameObjectCenter(camPlayer->plObj->targetObj, targCenter);
				Math_VecSub(targCenter, plCenter, ang);
				Math_VecToAngles(ang, ang);
				ang[0] = Math_AngleMod(ang[0]);
				ang[1] = Math_AngleMod(ang[1]);
				ang[2] = Math_AngleMod(ang[2]);
				if (ang[PITCH] > 15.0f)
				{
					ang[PITCH] = 15.0f;
				}
				obj->net.ang[PITCH] = Math_BlendAngle(obj->net.ang[PITCH], ang[PITCH], timeMod*angleBlendScale);
				obj->net.ang[YAW] = ObjCam_LerpAngle(obj->net.ang[YAW], ang[YAW], timeMod*0.25f);
				obj->net.ang[ROLL] = Math_BlendAngle(obj->net.ang[ROLL], ang[ROLL], timeMod*angleBlendScale);
				autoAngling = true;
			}
		}
	}

	float xyFriction = 0.09f;
	float zFriction = 0.09f;
	float velMul = 0.2f;

	//calculate the desired pos
	float desiredPos[3];
	if (obj->target)
	{
		velMul = 0.8f;
		velMul *= g_glb.camSlack;
		modelMatrix_t oriMat;
		Math_VecCopy(obj->target->net.pos, desiredPos);
		modelMatrix_t boneMat;
		if (g_glb.camBoneBolt[0] && obj->target->rcColModel &&
			g_sharedFn->Coll_GetModelBoneMatrix(obj->target->rcColModel, g_glb.camBoneBolt, &boneMat))
		{
			Math_VecCopy(boneMat.o, desiredPos);
			Math_TransformPointByMatrix(&boneMat, g_gameIdent.x1, oriMat.x1);
			Math_VecSub(oriMat.x1, boneMat.o, oriMat.x1);
			Math_TransformPointByMatrix(&boneMat, g_gameIdent.x2, oriMat.x2);
			Math_VecSub(oriMat.x2, boneMat.o, oriMat.x2);
			Math_TransformPointByMatrix(&boneMat, g_gameIdent.x3, oriMat.x3);
			Math_VecSub(oriMat.x3, boneMat.o, oriMat.x3);
			Math_VecNorm(oriMat.x1);
			Math_VecNorm(oriMat.x2);
			Math_VecNorm(oriMat.x3);
		}
		else
		{
			oriMat = g_gameIdent;
		}

		float ang[3];
		if (g_glb.camBoltOffset[0] != 0.0f ||
			g_glb.camBoltOffset[1] != 0.0f ||
			g_glb.camBoltOffset[2] != 0.0f)
		{
			float basePos[3];
			Math_VecCopy(desiredPos, basePos);

			desiredPos[0] += oriMat.x1[0]*g_glb.camBoltOffset[0];
			desiredPos[1] += oriMat.x1[1]*g_glb.camBoltOffset[0];
			desiredPos[2] += oriMat.x1[2]*g_glb.camBoltOffset[0];
			desiredPos[0] += oriMat.x2[0]*g_glb.camBoltOffset[1];
			desiredPos[1] += oriMat.x2[1]*g_glb.camBoltOffset[1];
			desiredPos[2] += oriMat.x2[2]*g_glb.camBoltOffset[1];
			desiredPos[0] += oriMat.x3[0]*g_glb.camBoltOffset[2];
			desiredPos[1] += oriMat.x3[1]*g_glb.camBoltOffset[2];
			desiredPos[2] += oriMat.x3[2]*g_glb.camBoltOffset[2];
			float d[3];
			Math_VecSub(basePos, desiredPos, d);
			Math_VecToAngles(d, ang);
			Math_VecAdd(ang, g_glb.camAbsAngOffset, ang);
		}
		else
		{
			//Math_VecAdd(obj->net.ang, g_glb.camAbsAngOffset, ang);
			Math_VecCopy(g_glb.camAbsAngOffset, ang);
		}
		float angleBlendScale = 0.4f*g_glb.camSlack;
		if (angleBlendScale > 1.0f)
		{
			angleBlendScale = 1.0f;
		}

		ang[0] = Math_AngleMod(ang[0]);
		ang[1] = Math_AngleMod(ang[1]);
		ang[2] = Math_AngleMod(ang[2]);

		obj->net.ang[PITCH] = Math_BlendAngle(obj->net.ang[PITCH], ang[PITCH], timeMod*angleBlendScale);
		obj->net.ang[YAW] = Math_BlendAngle(obj->net.ang[YAW], ang[YAW], timeMod*angleBlendScale);
		obj->net.ang[ROLL] = Math_BlendAngle(obj->net.ang[ROLL], ang[ROLL], timeMod*angleBlendScale);
	}
	else
	{
		//Math_VecCopy(camPlayer->net.pos, desiredPos);
		Util_GameObjectCenter(camPlayer, desiredPos);
		float distFactor = 0.0f;
		if (g_glb.runningMultiplayer)
		{
			bool anyPl = false;
			float totalMins[3], totalMaxs[3];
			for (int i = 0; i < MAX_NET_CLIENTS; i++)
			{
				gameObject_t *otherPl = &g_gameObjects[i];
				if (!otherPl->inuse || !otherPl->plObj)
				{
					continue;
				}
				float gmins[3], gmaxs[3];
				Math_VecAdd(otherPl->net.pos, otherPl->spawnMins, gmins);
				Math_VecAdd(otherPl->net.pos, otherPl->spawnMaxs, gmaxs);
				if (!anyPl)
				{
					Math_VecCopy(gmins, totalMins);
					Math_VecCopy(gmaxs, totalMaxs);
					anyPl = true;
				}
				else
				{
					Math_ExpandBounds(totalMins, totalMaxs, gmins, gmaxs);
				}
			}
			if (anyPl)
			{
				desiredPos[0] = totalMins[0]+(totalMaxs[0]-totalMins[0])*0.5f;
				desiredPos[1] = totalMins[1]+(totalMaxs[1]-totalMins[1])*0.5f;
				desiredPos[2] = totalMins[2]+(totalMaxs[2]-totalMins[2])*0.5f;
				distFactor += Math_Max3(totalMaxs[0]-totalMins[0], totalMaxs[1]-totalMins[1], totalMaxs[2]-totalMins[2]);
				if (g_glb.mpVersusMode)
				{
					distFactor *= 0.75f;
				}
			}
		}

		if (camPlayer->plObj->targetObj && !g_glb.runningMultiplayer)
		{
			float d[3];
			float tpos[3];
			Util_GameObjectCenter(camPlayer->plObj->targetObj, tpos);
			Math_VecSub(tpos, desiredPos, d);
			desiredPos[0] += d[0]*0.5f;
			desiredPos[1] += d[1]*0.5f;
			desiredPos[2] += d[2]*0.5f;
			float dist = Math_VecLen(d);
			if (dist > 900.0f)
			{
				float dfact = (dist-900.0f)*0.25f;
				Math_VecNorm(d);
				desiredPos[0] -= d[0]*dfact;
				desiredPos[1] -= d[1]*dfact;
				desiredPos[2] -= d[2]*dfact;
				distFactor += dfact;
			}
			trackingTarg = true;
		}
		else
		{
			if (g_glb.runningMultiplayer)
			{ //take the highest moving velocity from all players
				float velPush[3];
				velPush[0] = 0.0f;
				velPush[1] = 0.0f;
				velPush[2] = 0.0f;
				float maxPush = 0.0f;
				for (int i = 0; i < MAX_NET_CLIENTS; i++)
				{
					gameObject_t *otherPl = &g_gameObjects[i];
					if (otherPl->inuse && otherPl->plObj)
					{
						float lv[3];
						gameObject_t *velObj = (otherPl->plObj->ride) ? otherPl->plObj->ride : camPlayer;
						lv[0] = velObj->net.vel[0]*1.5f;
						lv[1] = velObj->net.vel[1]*1.5f;
						if (velObj->net.vel[2] > 0.0f)
						{
							lv[2] = velObj->net.vel[2]*0.5f;
						}
						else
						{
							lv[2] = velObj->net.vel[2]*1.5f;
						}

						float f = Math_VecNorm(lv);
						if (f > maxPush)
						{
							maxPush = f;
						}
						Math_VecAdd(velPush, lv, velPush);
					}
				}
				if (maxPush > 0.0f)
				{
					Math_VecNorm(velPush);
					desiredPos[0] += velPush[0]*maxPush;
					desiredPos[1] += velPush[1]*maxPush;
					desiredPos[2] += velPush[2]*maxPush;
				}
			}
			else
			{
				gameObject_t *velObj = (camPlayer->plObj->ride) ? camPlayer->plObj->ride : camPlayer;
				desiredPos[0] += velObj->net.vel[0]*1.5f;
				desiredPos[1] += velObj->net.vel[1]*1.5f;
				if (velObj->net.vel[2] > 0.0f)
				{
					desiredPos[2] += velObj->net.vel[2]*0.5f;
				}
				else
				{
					desiredPos[2] += velObj->net.vel[2]*1.5f;
				}
			}
		}
		float fwd[3], up[3];
		Math_AngleVectors(obj->net.ang, fwd, 0, up);
		//Math_VecMA(desiredPos, 400.0f, up, desiredPos);
		if (camPlayer->plObj->camRange != 0.0f)
		{
			desiredPos[2] += camPlayer->plObj->camRange*0.5f;
			Math_VecMA(desiredPos, -camPlayer->plObj->camRange, fwd, desiredPos);
		}
		else
		{
			desiredPos[2] += 385.0f;
			Math_VecMA(desiredPos, -(1600.0f+distFactor), fwd, desiredPos);
		}
	}

	desiredPos[0] += g_glb.camAbsOffset[0];
	desiredPos[1] += g_glb.camAbsOffset[1];
	desiredPos[2] += g_glb.camAbsOffset[2];

	//slowly move toward the desired pos with velocity
	float v[3];
	Math_VecSub(desiredPos, obj->net.pos, v);
	if (Math_VecLen(v) > 8192.0f)
	{
		Math_VecCopy(desiredPos, obj->net.pos);
	}
	else
	{
		v[0] *= timeMod;
		v[1] *= timeMod;
		v[2] *= timeMod;
		Math_VecAdd(obj->net.vel, v, obj->net.vel);
	}

	if (trackingTarg && autoAngling)
	{
		velMul = 0.4f;
	}

	//Phys_LinearMove(obj, desiredPos, 800.0f*velMul*timeMod);
	Phys_DirectVelocity(obj, timeMod, xyFriction, zFriction, velMul);

	if (g_glb.debugRigid)
	{ //fire bullets out of the camera to push stuff around
		if (camPlayer->plObj->clButtons[BUTTON_ACTION] && obj->debounce6 < g_glb.gameTime)
		{
			float dir[3];
			Math_AngleVectors(obj->net.ang, dir, 0, 0);
			Util_FireBullet(camPlayer, obj->net.pos, dir, NULL);
			obj->debounce6 = g_glb.gameTime + 100;
		}
	}

	if (!obj->target)
	{
		collObj_t col;
		float p[3];
		Math_VecCopy(camPlayer->net.pos, p);
		p[2] += 360.0f;
		collOpt_t opt;
		memset(&opt, 0, sizeof(opt));
		opt.ignoreBoxModels = true;
		opt.ignoreTexNum = 1;
		char *t = "textures/ava/noclip";
		opt.ignoreTexList = &t;
		g_sharedFn->Coll_RadiusTranslation(camPlayer->rcColModel, &col, p, obj->net.pos, 32.0f, NULL, &opt);
		if (col.hit)
		{
			Math_VecCopy(col.endPos, obj->net.pos);
			obj->net.vel[0] = obj->net.vel[1] = obj->net.vel[2] = 0.0f;
		}
	}

	obj->net.ang[0] = ObjCam_NormalizedAngle(obj->net.ang[0]);
	obj->net.ang[1] = ObjCam_NormalizedAngle(obj->net.ang[1]);
	obj->net.ang[2] = ObjCam_NormalizedAngle(obj->net.ang[2]);

	//update cam frustum
	float ang[3];
	ang[0] = obj->net.ang[0];
	ang[1] = obj->net.ang[1];
	ang[2] = obj->net.ang[2];
	Math_CreateFrustum(70.0f, obj->net.pos, ang, &g_glb.camFrustum);
	Math_CreateFrustum(93.33334f, obj->net.pos, ang, &g_glb.wideCamFrustum);
	g_glb.camFrustumFresh = true;
	ObjVisBlock_Update(obj->net.pos);
}
