/*
=============================================================================
Module Information
------------------
Name:			obj_cinemaprop.cpp
Author:			Rich Whitehouse
Description:	generic animating object used for cinematics

Object Arguments
model:			Name of the RDM model to use.
anim:			Name of the RDA animation to use.

In the following, # is the number of any given animation frame. On that
frame, the given event will trigger.
fs_#:			Frame sound. Value is the name of a wave/ogg file.
fx_#:			Frame effect. Value is the name of a particle system.
fr_#:			R-Script. Value is the name of an R-Script routine.
=============================================================================
*/

#include "main.h"

//think
void ObjCinProp_Think(gameObject_t *obj, float timeMod)
{
	if (obj->cinObj->hideable && (obj->net.renderEffects & FXFL_HIDDEN) &&
		!(obj->net.renderEffects2 & FXFL2_DEATH3))
	{
		return;
	}

	Util_AnimateObject(obj, timeMod);
}

//frame tick
void ObjCinProp_FrameTick(gameObject_t *obj, float timeMod, int oldFrame)
{
	for (int i = 0; i < obj->cinObj->numFrameEvents; i++)
	{
		animFrameEvent_t *fe = &obj->cinObj->frameEvents[i];
		if (fe->frame == obj->net.frame && obj->net.frame != oldFrame)
		{
			if (fe->sound)
			{
				ObjSound_CreateFromIndex(obj->net.pos, fe->sound, 1.0f, -1);
			}
			if (fe->fx)
			{
				ObjParticles_CreateFromIndex(fe->fx, obj->net.pos, obj->net.ang, -1);
			}
			if (fe->script)
			{
				float z[3] = {0.0f, 0.0f, 0.0f};
				objArgs_t arg;
				arg.key = "script";
				arg.val = fe->script;
				gameObject_t *scriptObj = LServ_ObjectFromName("obj_rscript", z, z, &arg, 1);
				if (scriptObj && obj->targetName)
				{
					scriptObj->spawnArgs = NULL;
					scriptObj->numSpawnArgs = 0;
					char str[256];
					sprintf(str, "%s_script", obj->targetName);
					scriptObj->targetName = Util_PooledString(str);
				}
			}
		}
	}
}

//get frame event
static animFrameEvent_t *ObjCinProp_GetFrameEvent(gameObject_t *obj, int frame, bool freeSound, bool freeFx, bool freeScript)
{
	if (!obj->cinObj)
	{
		return NULL;
	}

	for (int i = 0; i < obj->cinObj->numFrameEvents; i++)
	{
		animFrameEvent_t *fe = &obj->cinObj->frameEvents[i];
		if (fe->frame == frame)
		{
			if (!freeSound || !fe->sound)
			{
				if (!freeFx || !fe->fx)
				{
					if (!freeScript || !fe->script)
					{
						return fe;
					}
				}
			}
		}
	}

	if (obj->cinObj->numFrameEvents >= MAX_CIN_FRAME_EVENTS)
	{
		return NULL;
	}

	animFrameEvent_t *fe = &obj->cinObj->frameEvents[obj->cinObj->numFrameEvents++];
	fe->frame = frame;
	fe->sound = 0;
	fe->fx = 0;
	fe->script = NULL;
	return fe;
}

//spawn
void ObjCinProp_Spawn(gameObject_t *obj, BYTE *b, const objArgs_t *args, int numArgs)
{
	obj->cinObj = (cinObject_t *)g_sharedFn->Common_RCMalloc(sizeof(cinObject_t));
	memset(obj->cinObj, 0, sizeof(cinObject_t));

	for (int i = 0; i < numArgs; i++)
	{
		const objArgs_t *arg = args+i;
		if (!stricmp(arg->key, "model"))
		{
			char str[256];
			sprintf(str, "&%s", arg->val);
			obj->net.strIndex = g_sharedFn->Common_ServerString(str);
			obj->rcColModel = g_sharedFn->Coll_RegisterModelInstance(arg->val);
			g_sharedFn->Coll_UpdateModel(obj->rcColModel, NULL);
		}
		else if (!stricmp(arg->key, "anim"))
		{
			char str[256];
			sprintf(str, "@%s", arg->val);
			obj->net.strIndexB = g_sharedFn->Common_ServerString(str);
			if (obj->rcColModel)
			{
				g_sharedFn->Coll_UpdateModel(obj->rcColModel, arg->val);
			}
		}
		else if (!stricmp(arg->key, "hideable"))
		{
			obj->cinObj->hideable = !!atoi(arg->val);
		}
		else if (!strncmp(arg->key, "fs_", strlen("fs_")))
		{
			int frame = atoi(arg->key+strlen("fs_"));

			animFrameEvent_t *fe = ObjCinProp_GetFrameEvent(obj, frame, true, false, false);
			if (fe)
			{
				char str[256];
				sprintf(str, "$%s", arg->val);
				fe->sound = g_sharedFn->Common_ServerString(str);
			}
		}
		else if (!strncmp(arg->key, "fx_", strlen("fx_")))
		{
			int frame = atoi(arg->key+strlen("fx_"));

			animFrameEvent_t *fe = ObjCinProp_GetFrameEvent(obj, frame, false, true, false);
			if (fe)
			{
				char str[256];
				sprintf(str, "&&%s", arg->val);
				fe->fx = g_sharedFn->Common_ServerString(str);
			}
		}
		else if (!strncmp(arg->key, "fr_", strlen("fr_")))
		{
			int frame = atoi(arg->key+strlen("fr_"));

			animFrameEvent_t *fe = ObjCinProp_GetFrameEvent(obj, frame, false, false, true);
			if (fe)
			{
				fe->script = arg->val;
			}
		}
	}

	if (!obj->net.solid && !obj->cinObj->hideable)
	{
		float propSize = 99999.0f;
		obj->radius = propSize;
		obj->spawnMins[0] = -propSize;
		obj->spawnMins[1] = -propSize;
		obj->spawnMins[2] = -propSize;
		obj->spawnMaxs[0] = propSize;
		obj->spawnMaxs[1] = propSize;
		obj->spawnMaxs[2] = propSize;
	}
	else if (obj->rcColModel)
	{
		Math_VecCopy(obj->spawnMins, obj->net.mins);
		Math_VecCopy(obj->spawnMaxs, obj->net.maxs);
		obj->rcColModel->solid = 1;
		obj->rcColModel->clipFlags = CLIPFL_BOXMOVE;
	}

	if (obj->cinObj->hideable)
	{ //default to hidden
		obj->net.renderEffects |= FXFL_HIDDEN;
	}

	LServ_UpdateRClip(obj);

	obj->animTable = &obj->cinObj->animStub;
	obj->curAnim = 0;

	obj->animTable->looping = true;
	obj->animTable->duration = 68.0f;

	obj->animframetick = ObjCinProp_FrameTick;

	obj->think = ObjCinProp_Think;
	obj->thinkTime = g_glb.gameTime;
}
