/*
=============================================================================
Module Information
------------------
Name:			obj_eater.cpp
Author:			Rich Whitehouse
Description:	server logic object: whole eater
=============================================================================
*/

#include "main.h"
#include "ai.h"

//just to make things a bit more readable
#define debounceFastLook		debounce5
#define debounceSuckTime		debounce4
#define debounceDoSuck			debounce3
#define debounceRoamTime		debounce2

typedef enum
{
	EATANIM_BITE = NUM_HUMAN_ANIMS,
	EATANIM_SUCK_WARMUP,
	EATANIM_SUCK,
	NUM_EAT_ANIMS
} eatAnims_e;

static gameAnim_t g_eatAnims[NUM_EAT_ANIMS] =
{
	{0, 17, 68.0f, true},			//HUMANIM_IDLE
	{57, 61, 188.0f, true},			//HUMANIM_WALK
	{57, 61, 138.0f, true},			//HUMANIM_RUNSLOW
	{57, 61, 68.0f, true},			//HUMANIM_RUN
	{43, 44, 68.0f, false},			//HUMANIM_JUMP
	{41, 41, 68.0f, false},			//HUMANIM_FALL
	{39, 40, 68.0f, false},			//HUMANIM_LAND
	{36, 41, 68.0f, false},			//HUMANIM_PAIN_HIGH1
	{36, 41, 68.0f, false},			//HUMANIM_PAIN_HIGH2
	{36, 41, 68.0f, false},			//HUMANIM_PAIN_LOW1
	{36, 41, 68.0f, false},			//HUMANIM_PAIN_LOW2
	{32, 36, 68.0f, false},			//HUMANIM_PAIN_AIR
	{31, 34, 68.0f, false},			//HUMANIM_PAIN_POPUP
	{37, 41, 118.0f, false},		//HUMANIM_GETUP_BACK
	{37, 41, 118.0f, false},		//HUMANIM_GETUP_FRONT
	{31, 34, 68.0f, false},			//HUMANIM_FLYBACK
	{31, 34, 68.0f, false},			//HUMANIM_FLYBACK2
	{31, 34, 68.0f, false},			//HUMANIM_HIT_FALLFORWARD
	{39, 43, 68.0f, false},			//HUMANIM_FALL_LAND
	{39, 43, 68.0f, false},			//HUMANIM_POPUP_LAND
	{32, 32, 1568.0f, false},		//HUMANIM_DEATH
	{61, 66, 68.0f, false},			//EATANIM_BITE
	{111, 113, 168.0f, false},		//EATANIM_SUCK_WARMUP
	{109, 110, 168.0f, false}		//EATANIM_SUCK
};

#define NUM_EAT_DAMAGE_BONES	1
static damageBone_t g_eatDamageBones[NUM_EAT_DAMAGE_BONES] =
{
	{"b5", "b4", {0.0f, 0.0f, 0.0f}, 140.0f}			//DMGBONE_FIST_R
};

//returns damage bone or -1 if not in attack
int ObjEat_InAttack(gameObject_t *obj, int *dmgBones)
{
	if (obj->curAnim == EATANIM_BITE)
	{
		if (obj->net.frame >= 63 && obj->net.frame <= 66)
		{
			dmgBones[0] = DMGBONE_FIST_R;
			return 1;
		}
	}
	else if (obj->curAnim == EATANIM_SUCK)
	{
		dmgBones[0] = DMGBONE_FIST_R;
		return 1;
	}

	return -1;
}

//occupied?
bool ObjEat_CanAct(gameObject_t *obj)
{
	if (!obj->onGround)
	{
		return false;
	}
	if (obj->curAnim >= EATANIM_BITE)
	{
		return false;
	}
	if (AI_NonMovingAnim(obj))
	{
		return false;
	}
	if (obj->aiObj->noMoveTime >= g_glb.gameTime)
	{
		return false;
	}
	return true;
}

//can i dodge?
bool ObjEat_CanDodge(gameObject_t *obj)
{
	if (obj->generalFlag >= 0)
	{
		return false;
	}
	if (!ObjEat_CanAct(obj))
	{
		return false;
	}

	return true;
}

//dodge (eats currently don't dodge)
bool ObjEat_DodgeEnemy(gameObject_t *obj, gameObject_t *enemy)
{
	return false;
}

//think
void ObjEat_Think(gameObject_t *obj, float timeMod)
{
	if (!AI_ShouldThink(obj))
	{
		return;
	}

	gameObject_t *en = obj->aiObj->enemy;

	AI_StandardGoals(obj, timeMod);
	if (!obj->aiObj->enemy && obj->debounceRoamTime < g_glb.gameTime && obj->aiObj->aiRange)
	{
		int *objIdx = (int *)_alloca(sizeof(int)*g_glb.gameObjectSlots);
		int numObj = 0;
		for (int i = MAX_NET_CLIENTS; i < g_glb.gameObjectSlots; i++)
		{
			if (g_gameObjects[i].inuse && !(g_gameObjects[i].net.renderEffects & FXFL_BOLTED1))
			{
				if (g_sharedFn->Coll_GeneralVisibility(obj->net.pos, g_gameObjects[i].net.pos, 200.0f))
				{
					objIdx[numObj] = i;
					numObj++;
				}
			}
		}
		if (numObj > 1)
		{
			char *roamSnd[3] = {
				"assets/sound/cb/eatidle1.wav",
				"assets/sound/cb/eatidle2.wav",
				"assets/sound/cb/eatidle3.wav"
			};
			gameObject_t *cam = Util_GetCam(0);
			float d[3];
			Math_VecSub(obj->net.pos, cam->net.pos, d);
			if (Math_VecLen(d) < 8192.0f)
			{
				ObjSound_Create(obj->net.pos, roamSnd[rand()%3], 1.0f, -1);
			}

			obj->aiObj->goalObj = &g_gameObjects[objIdx[rand()%numObj]];
			obj->debounceRoamTime = g_glb.gameTime + 1000 + rand()%15000;
		}
	}
	AI_GenericThink(obj, timeMod);
	AI_GetToGoal(obj, timeMod);

	if (en != obj->aiObj->enemy && obj->aiObj->aiRange)
	{
		char *spotSnd[3] = {
			"assets/sound/cb/eatspot1.wav",
			"assets/sound/cb/eatspot2.wav",
			"assets/sound/cb/eatspot3.wav"
		};
		char *snd = spotSnd[rand()%3];
		ObjSound_Create(obj->net.pos, snd, 1.0f, -1);
		ObjSound_Create(obj->net.pos, snd, 1.0f, -1);
	}

	if (obj->debounceFastLook >= g_glb.gameTime)
	{
		obj->net.ang[YAW] = Math_BlendAngleLinear(obj->net.ang[YAW], obj->aiObj->lookYaw, 45.0f);
	}

	dmgBoneLine_t dmgPos[NUM_EAT_DAMAGE_BONES];
	AI_TransformDamageBones(obj, g_eatDamageBones, dmgPos, NUM_EAT_DAMAGE_BONES);
	int dmgBones[MAX_AI_DAMAGE_BONES];
	int numDmgBones = ObjEat_InAttack(obj, dmgBones);
	if (obj->aiObj->hasLastDmg && numDmgBones >= 0)
	{
		int dmg = (obj->curAnim == EATANIM_SUCK) ? 20 : 80;
		dmg = AI_LevelDamage(obj, dmg);
		for (int i = 0; i < numDmgBones; i++)
		{
			AI_RunDamageBone(obj, &g_eatDamageBones[dmgBones[i]], &obj->aiObj->lastDmg[dmgBones[i]],
				&dmgPos[dmgBones[i]], dmg);
		}
	}

	obj->atkType = ATKTYPE_NORMAL;

	memcpy(obj->aiObj->lastDmg, dmgPos, sizeof(dmgPos));
	obj->aiObj->hasLastDmg = true;

	if (g_glb.ai.weakAI && obj->aiObj->aiLevel < 15)
	{
		obj->aiObj->moveSpeed = 420.0f;
	}

	obj->aiObj->combatRange = 700.0f;
	obj->canBePushed = true;
	if (!AI_NonAnglingAnim(obj) && obj->aiObj->enemy && AI_FacingEnemy(obj, obj->aiObj->enemy))
	{
		if (obj->curAnim == EATANIM_SUCK)
		{ //suck them in
			obj->canBePushed = false;
			float angleBlendScale = 0.3f;
			obj->net.ang[YAW] = Math_BlendAngle(obj->net.ang[YAW], obj->aiObj->lookYaw, timeMod*angleBlendScale);
			float d[3];
			gameObject_t *suckee = obj->aiObj->enemy;
			Math_VecSub(obj->net.pos, suckee->net.pos, d);
			float dist = Math_VecNorm(d);
			if (dist < 512.0f)
			{
				Math_VecSub(obj->net.pos, suckee->net.pos, d);
				Math_VecCopy(d, suckee->net.vel);
			}
			else if (dist < 2048.0f)
			{
				float f;
				if (obj->generalFlag <= -1)
				{
					f = (2048.0f-dist)*0.25f;
				}
				else
				{
					f = (2048.0f-dist)*0.1f;
				}
				if (f > 256.0f)
				{
					f = 256.0f;
				}
				d[0] *= f;
				d[1] *= f;
				d[2] *= f;
				Math_VecAdd(suckee->net.vel, d, suckee->net.vel);
			}
		}
		else if (obj->aiObj->enemy->net.index < MAX_NET_CLIENTS &&
			ObjEat_CanDodge(obj) &&
			ObjPlayer_InDodgeableAttack(obj->aiObj->enemy) &&
			ObjEat_DodgeEnemy(obj, obj->aiObj->enemy))
		{ //dodge
			obj->aiObj->noMoveTime = g_glb.gameTime+100;
		}
		else
		{
			bool shouldCloseIn = AI_ShouldCloseIn(obj);
			if (obj->aiObj->attackTime < g_glb.gameTime && ObjEat_CanAct(obj))
			{
				if (shouldCloseIn)
				{
					bool okToAttack = ObjPlayer_OkayToAttack(obj, obj->aiObj->enemy);
					obj->aiObj->combatRange = (okToAttack) ? 400.0f : Util_RandFloat(400.0f, 1300.0f);
					float closeAttackDist = 500.0f;
					if (obj->aiObj->distToEnemy < closeAttackDist &&
						okToAttack)
					{
						AI_StartAnim(obj, EATANIM_BITE, true);
						if (obj->generalFlag <= -1)
						{
							obj->aiObj->attackTime = g_glb.gameTime + Util_LocalDelay(obj, (serverTime_t)(1500+rand()%2000));
						}
						else
						{
							if (g_glb.ai.weakAI && obj->aiObj->aiLevel < 14)
							{
								obj->aiObj->attackTime = g_glb.gameTime + Util_LocalDelay(obj, (serverTime_t)(4000+rand()%4000));
							}
							else
							{
								obj->aiObj->attackTime = g_glb.gameTime + Util_LocalDelay(obj, (serverTime_t)(2000+rand()%4000));
							}
						}
						obj->aiObj->noMoveTime = g_glb.gameTime + Util_LocalDelay(obj, 1000);
					}
				}
				if (obj->aiObj->attackTime < g_glb.gameTime && obj->aiObj->distToEnemy < 2048.0f && obj->debounceSuckTime < g_glb.gameTime &&
					ObjPlayer_OkayToAttack(obj, obj->aiObj->enemy))
				{
					if (obj->generalFlag <= -1 || g_glb.ai.numActiveAI < 8)
					{
						if (g_glb.ai.weakAI && obj->aiObj->aiLevel < 15 && ObjPlayer_IsWeakSauce(obj->aiObj->enemy))
						{ //don't suck right now
							obj->debounceSuckTime = g_glb.gameTime + Util_LocalDelay(obj, (serverTime_t)(4000+rand()%6000));
						}
						else
						{
							AI_StartAnim(obj, EATANIM_SUCK_WARMUP, true);
							obj->aiObj->attackTime = g_glb.gameTime + Util_LocalDelay(obj, (serverTime_t)(2000+rand()%4000));
							obj->aiObj->noMoveTime = g_glb.gameTime + Util_LocalDelay(obj, 1000);
							obj->debounceDoSuck = g_glb.gameTime + Util_LocalDelay(obj, 2000);
						}
					}
				}
			}
			else
			{
				if (g_glb.ai.weakAI && obj->aiObj->aiLevel < 14)
				{
					obj->aiObj->moveSpeed = 200.0f;
				}
				float closeDist = (obj->generalFlag <= 0) ? 600.0f : 400.0f;
				float r = (shouldCloseIn) ? closeDist : 800.0f;
				obj->aiObj->combatRange = g_glb.ai.playerCloseDist+r;
			}
		}
	}
}

//pick which anim to be in
void ObjEat_PickAnim(gameObject_t *obj, float timeMod)
{
	gameAnim_t *curAnim = obj->animTable+obj->curAnim;

	if (obj->curAnim == EATANIM_SUCK_WARMUP)
	{
		if (obj->net.frame == curAnim->endFrame && !obj->animRestart)
		{
			obj->generalFlag2 = 0;
			AI_StartAnim(obj, EATANIM_SUCK, true);
		}
	}
	else if (obj->curAnim == EATANIM_SUCK)
	{
		if (obj->net.frame == curAnim->endFrame && !obj->animRestart)
		{
			int rfac = (obj->generalFlag <= -1) ? 9 : 8;
			int suckDel = (obj->generalFlag <= -1) ? 8000 : 16000;
			obj->debounceSuckTime = g_glb.gameTime + Util_LocalDelay(obj, (serverTime_t)suckDel);
			if ((rand()%10 < rfac || obj->debounceDoSuck >= g_glb.gameTime) && obj->generalFlag2 < 4)
			{
				AI_StartAnim(obj, EATANIM_SUCK, true);
				ObjSound_Create(obj->net.pos, "assets/sound/cb/suck.wav", 1.0f, -1);
				if (obj->aiObj->hasLastDmg)
				{
					char *fx = (obj->generalFlag <= -1) ? "other/hellsuck" : "other/suckcone";
					ObjParticles_Create(fx, obj->aiObj->lastDmg[0].pos, obj->net.ang, -1);
				}
				obj->aiObj->attackTime = g_glb.gameTime + Util_LocalDelay(obj, (serverTime_t)(2000+rand()%4000));
				obj->aiObj->noMoveTime = g_glb.gameTime + Util_LocalDelay(obj, 500);
			}
			else
			{
				AI_StartAnim(obj, HUMANIM_IDLE, true);
			}
		}
	}
	else if (obj->net.frame == curAnim->endFrame && !obj->animRestart)
	{
		AI_StartAnim(obj, HUMANIM_IDLE, true);
	}
}

//frame tick
void ObjEat_FrameTick(gameObject_t *obj, float timeMod, int oldFrame)
{
	switch (obj->curAnim)
	{
	case HUMANIM_WALK:
	case HUMANIM_RUNSLOW:
	case HUMANIM_RUN:
		{
			char *stepSnd[4] = {
				"assets/sound/cb/clawstep01.wav",
				"assets/sound/cb/clawstep02.wav",
				"assets/sound/cb/clawstep03.wav",
				"assets/sound/cb/clawstep04.wav"
			};
			gameObject_t *cam = Util_GetCam(0);
			float d[3];
			Math_VecSub(obj->net.pos, cam->net.pos, d);
			if (Math_VecLen(d) < 4096.0f)
			{
				ObjSound_Create(obj->net.pos, stepSnd[rand()%4], 1.0f, -1);
			}
		}
		break;
	case EATANIM_BITE:
		if (obj->net.frame == 62)
		{
			ObjSound_CreateFromIndex(obj->net.pos, g_soundPunch[rand()%NUM_SOUNDS_PUNCH], 1.0f, -1);
			float fwd[3];
			float f = 400.0f;
			Math_AngleVectors(obj->net.ang, fwd, 0, 0);
			obj->net.vel[0] += fwd[0]*f;
			obj->net.vel[1] += fwd[1]*f;
			obj->net.vel[2] = 300.0f;
		}
		break;
	default:
		break;
	}
}

//touch
void ObjEat_Touch(gameObject_t *obj, gameObject_t *other, const collObj_t *col)
{
	AI_GenericTouch(obj, other, col);
}

//eats currently can't block
bool ObjEat_AttackBlock(gameObject_t *obj, gameObject_t *hurter, int dmg, collObj_t *col,
							float *impactPos, float *impactAng)
{
	return false;
}

//called when inflicting damage
void ObjEat_OnHurt(gameObject_t *obj, gameObject_t *victim, int dmg, const collObj_t *col)
{
	if (victim && victim->hurtable && Util_ValidEnemies(obj, victim))
	{
		obj->generalFlag2++;
		if (obj->generalFlag <= -1)
		{
			obj->health += dmg;
		}
		else
		{
			obj->health += dmg/2;
		}
		if (obj->health > obj->aiObj->maxHealth)
		{
			obj->health = obj->aiObj->maxHealth;
		}
	}
}

//spawn
void ObjEat_Spawn(gameObject_t *obj, BYTE *b, const objArgs_t *args, int numArgs)
{
	g_sharedFn->Common_ServerString("$assets/sound/cb/clawstep01.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/clawstep02.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/clawstep03.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/clawstep04.wav");

	AI_GenericSpawn(obj, b, args, numArgs);

	obj->touch = ObjEat_Touch;
	obj->animframetick = ObjEat_FrameTick;
	obj->onhurt = ObjEat_OnHurt;

	obj->debounceRoamTime = g_glb.gameTime + 1000 + rand()%10000;

	const char *elvalue = Common_GetValForKey(b, "elvalue");
	if (elvalue && elvalue[0])
	{
		obj->generalFlag = atoi(elvalue);
	}

	if (obj->generalFlag == -1)
	{
		obj->aiObj->makoValue = 60 + rand()%30;
		obj->net.aiDescIndex = AIDESC_EAT2;
	}
	else
	{
		obj->aiObj->makoValue = 25 + rand()%10;
		obj->net.aiDescIndex = AIDESC_EAT1;
	}

	obj->aiObj->dropChances[INVITEM_VAMPIREFANG] = 50;
	if (obj->generalFlag == -1)
	{
		obj->aiObj->dropChances[INVITEM_VAMPIREFANG] = 90;
		obj->aiObj->dropChances[INVITEM_HELLFANG] = 20;
	}
	obj->aiObj->maxHealth = obj->health;

	static int biteSounds[2];
	biteSounds[0] = g_sharedFn->Common_ServerString("$assets/sound/cb/bite01.wav");
	biteSounds[1] = g_sharedFn->Common_ServerString("$assets/sound/cb/bite02.wav");
	obj->aiObj->dmgEffect = "melee/impactbite";
	obj->aiObj->dmgSounds = biteSounds;
	obj->aiObj->numDmgSounds = 2;
	if (obj->generalFlag == -1)
	{
		obj->aiObj->moveSpeed = 480.0f;
	}
	else
	{
		obj->aiObj->moveSpeed = 420.0f;
	}
	obj->animhandler = ObjEat_PickAnim;
	obj->animTable = g_eatAnims;
	obj->curAnim = HUMANIM_IDLE;
	obj->think = ObjEat_Think;
	obj->attackblock = ObjEat_AttackBlock;
}
