/*
=============================================================================
Module Information
------------------
Name:			obj_ironman.cpp
Author:			Rich Whitehouse
Description:	server logic object: iron man
=============================================================================
*/

#include "main.h"
#include "ai.h"

//just to make things a bit more readable
#define debounceFastLook		debounce5
#define debouncePowerup			debounce4
#define debounceUsePowerup		debounce3

typedef enum
{
	IRONANIM_POWERSLASH = NUM_HUMAN_ANIMS,
	IRONANIM_POWERSLASH2,
	IRONANIM_FISTBLOW,
	IRONANIM_POWERUP,
	NUM_IRON_ANIMS
} ironAnims_e;

static gameAnim_t g_ironAnims[NUM_IRON_ANIMS] =
{
	{0, 29, 68.0f, true},			//HUMANIM_IDLE
	{364, 369, 168.0f, true},		//HUMANIM_WALK
	{364, 369, 168.0f, true},		//HUMANIM_RUNSLOW
	{364, 369, 168.0f, true},		//HUMANIM_RUN
	{0, 4, 68.0f, false},			//HUMANIM_JUMP
	{0, 4, 68.0f, false},			//HUMANIM_FALL
	{0, 4, 68.0f, false},			//HUMANIM_LAND
	{36, 41, 68.0f, false},			//HUMANIM_PAIN_HIGH1
	{36, 41, 68.0f, false},			//HUMANIM_PAIN_HIGH2
	{36, 41, 68.0f, false},			//HUMANIM_PAIN_LOW1
	{36, 41, 68.0f, false},			//HUMANIM_PAIN_LOW2
	{36, 41, 68.0f, false},			//HUMANIM_PAIN_AIR
	{36, 41, 68.0f, false},			//HUMANIM_PAIN_POPUP
	{36, 41, 118.0f, false},		//HUMANIM_GETUP_BACK
	{36, 41, 118.0f, false},		//HUMANIM_GETUP_FRONT
	{36, 41, 68.0f, false},			//HUMANIM_FLYBACK
	{36, 41, 68.0f, false},			//HUMANIM_FLYBACK2
	{36, 41, 68.0f, false},			//HUMANIM_HIT_FALLFORWARD
	{0, 4, 68.0f, false},			//HUMANIM_FALL_LAND
	{0, 4, 68.0f, false},			//HUMANIM_POPUP_LAND
	{96, 96, 1568.0f, false},		//HUMANIM_DEATH
	{127, 148, 68.0f, false},		//IRONANIM_POWERSLASH
	{222, 250, 68.0f, false},		//IRONANIM_POWERSLASH2
	{72, 86, 68.0f, false},			//IRONANIM_FISTBLOW
	{154, 200, 68.0f, false}		//IRONANIM_POWERUP
};

#define NUM_IRON_DAMAGE_BONES	2
static damageBone_t g_ironDamageBones[NUM_IRON_DAMAGE_BONES] =
{
	{"b16", "b16", {0.0f, 0.0f, -3000.0f}, 300.0f},		//DMGBONE_FIST_R
	{"b25", "b24", {0.0f, 0.0f, 0.0f}, 350.0f}			//DMGBONE_FIST_L
};

//returns damage bone or -1 if not in attack
int ObjIron_InAttack(gameObject_t *obj, int *dmgBones)
{
	if (obj->curAnim == IRONANIM_POWERSLASH)
	{
		if (obj->net.frame >= 135 && obj->net.frame <= 139)
		{
			dmgBones[0] = DMGBONE_FIST_R;
			return 1;
		}
	}
	else if (obj->curAnim == IRONANIM_POWERSLASH2)
	{
		if (obj->net.frame >= 232 && obj->net.frame <= 238)
		{
			dmgBones[0] = DMGBONE_FIST_R;
			return 1;
		}
	}
	else if (obj->curAnim == IRONANIM_FISTBLOW)
	{
		if (obj->net.frame >= 78 && obj->net.frame <= 82)
		{
			dmgBones[0] = DMGBONE_FIST_L;
			return 1;
		}
	}

	return -1;
}

//occupied?
bool ObjIron_CanAct(gameObject_t *obj)
{
	if (!obj->onGround)
	{
		return false;
	}
	if (obj->curAnim >= IRONANIM_POWERSLASH)
	{
		return false;
	}
	if (AI_NonMovingAnim(obj))
	{
		return false;
	}
	if (obj->aiObj->noMoveTime >= g_glb.gameTime)
	{
		return false;
	}
	return true;
}

//can i dodge?
bool ObjIron_CanDodge(gameObject_t *obj)
{
	if (obj->generalFlag >= 0)
	{
		return false;
	}
	if (!ObjIron_CanAct(obj))
	{
		return false;
	}

	return true;
}

//dodge (irons currently don't dodge)
bool ObjIron_DodgeEnemy(gameObject_t *obj, gameObject_t *enemy)
{
	return false;
}

//think
void ObjIron_Think(gameObject_t *obj, float timeMod)
{
	if (!AI_ShouldThink(obj))
	{
		return;
	}

	if (obj->debouncePowerup > g_glb.gameTime && obj->health > 0)
	{
		obj->net.renderEffects2 |= FXFL2_INVINCIBLE;
	}
	else
	{
		obj->net.renderEffects2 &= ~FXFL2_INVINCIBLE;
	}

	AI_StandardGoals(obj, timeMod);
	if (!obj->onGround && obj->aiObj->noMoveTime < g_glb.gameTime+500)
	{
		obj->aiObj->noMoveTime = g_glb.gameTime+500;
	}
	if ((!obj->aiObj->enemy || obj->aiObj->distToEnemy > 2000.0f) && Math_VecLen(obj->net.vel) > 5.0f && obj->onGround)
	{ //look where i'm going
		float ang[3];
		Math_VecToAngles(obj->net.vel, ang);
		obj->aiObj->lookYaw = ang[YAW];
	}
	AI_GenericThink(obj, timeMod);
	AI_GetToGoal(obj, timeMod);

	if (obj->debounceFastLook >= g_glb.gameTime)
	{
		obj->net.ang[YAW] = Math_BlendAngleLinear(obj->net.ang[YAW], obj->aiObj->lookYaw, 45.0f);
	}

	dmgBoneLine_t dmgPos[NUM_IRON_DAMAGE_BONES];
	AI_TransformDamageBones(obj, g_ironDamageBones, dmgPos, NUM_IRON_DAMAGE_BONES);
	int dmgBones[MAX_AI_DAMAGE_BONES];
	int numDmgBones = ObjIron_InAttack(obj, dmgBones);
	if (obj->aiObj->hasLastDmg && numDmgBones >= 0)
	{
		int dmg = (obj->curAnim == IRONANIM_FISTBLOW || obj->curAnim == IRONANIM_POWERSLASH) ? 100 : 80;
		dmg = AI_LevelDamage(obj, dmg);
		for (int i = 0; i < numDmgBones; i++)
		{
			AI_RunDamageBone(obj, &g_ironDamageBones[dmgBones[i]], &obj->aiObj->lastDmg[dmgBones[i]],
				&dmgPos[dmgBones[i]], dmg);
		}
	}

	if (obj->curAnim == IRONANIM_FISTBLOW || obj->curAnim == IRONANIM_POWERSLASH2)
	{
		obj->atkType = ATKTYPE_KNOCKBACK;
	}
	else if (obj->curAnim == IRONANIM_POWERSLASH)
	{
		obj->atkType = ATKTYPE_SMACKDOWN;
	}
	else
	{
		obj->atkType = ATKTYPE_NORMAL;
	}

	memcpy(obj->aiObj->lastDmg, dmgPos, sizeof(dmgPos));
	obj->aiObj->hasLastDmg = true;

	if (obj->curAnim == IRONANIM_POWERSLASH || obj->curAnim == IRONANIM_POWERSLASH2)
	{
		obj->aiObj->dmgEffect = "melee/bigslash";
		static int dmgSounds[3] = {0, 0, 0};
		if (!dmgSounds[0])
		{
			dmgSounds[0] = g_sharedFn->Common_ServerString("$assets/sound/cb/buscut01.wav");
			dmgSounds[1] = g_sharedFn->Common_ServerString("$assets/sound/cb/buscut02.wav");
			dmgSounds[2] = g_sharedFn->Common_ServerString("$assets/sound/cb/buscut03.wav");
		}
		obj->aiObj->dmgSounds = dmgSounds;
		obj->aiObj->numDmgSounds = 3;
		obj->net.renderEffects |= FXFL_BLADETRAIL;
		obj->net.effectLen = -800.0f;
		obj->net.strIndexC = g_sharedFn->Common_ServerString(g_ironDamageBones[0].boneName);
		obj->net.strIndexD = obj->net.strIndexC;
	}
	else
	{
		obj->aiObj->dmgEffect = "melee/impacten";
		obj->aiObj->dmgSounds = g_soundHitHeavy;
		obj->aiObj->numDmgSounds = NUM_SOUNDS_HITHEAVY;
		obj->net.renderEffects &= ~FXFL_BLADETRAIL;
	}

	obj->aiObj->combatRange = 1200.0f;
	if (!AI_NonAnglingAnim(obj) && obj->aiObj->enemy && AI_FacingEnemy(obj, obj->aiObj->enemy))
	{
		if (obj->aiObj->enemy->net.index < MAX_NET_CLIENTS &&
			ObjIron_CanDodge(obj) &&
			ObjPlayer_InDodgeableAttack(obj->aiObj->enemy) &&
			ObjIron_DodgeEnemy(obj, obj->aiObj->enemy))
		{ //dodge
			obj->aiObj->noMoveTime = g_glb.gameTime+100;
		}
		else
		{
			bool shouldCloseIn = AI_ShouldCloseIn(obj);
			if (obj->generalFlag <= -1 && obj->aiObj->distToEnemy > 1600.0f && obj->health < obj->aiObj->maxHealth/2 &&
				obj->debouncePowerup < g_glb.gameTime && obj->debounceUsePowerup < g_glb.gameTime && ObjIron_CanAct(obj))
			{
				AI_StartAnim(obj, IRONANIM_POWERUP, true);
				obj->aiObj->noMoveTime = g_glb.gameTime + Util_LocalDelay(obj, 3000);
				obj->aiObj->attackTime = g_glb.gameTime + Util_LocalDelay(obj, (serverTime_t)(3000+rand()%2000));
				obj->debounceUsePowerup = g_glb.gameTime + 20000;
			}
			else if (obj->aiObj->attackTime < g_glb.gameTime && shouldCloseIn && ObjIron_CanAct(obj))
			{
				obj->aiObj->combatRange = 900.0f;
				float closeAttackDist = 1000.0f;
				if (obj->aiObj->distToEnemy < closeAttackDist)
				{
					serverTime_t atkDel;
					int r = rand()%30;
					if (obj->aiObj->distToEnemy < 800.0f)
					{
						AI_StartAnim(obj, IRONANIM_FISTBLOW, true);
						atkDel = 2000;
					}
					else if (r >= 20)
					{
						AI_StartAnim(obj, IRONANIM_POWERSLASH2, true);
						atkDel = 2000;
					}
					else if (r >= 10)
					{
						AI_StartAnim(obj, IRONANIM_POWERSLASH, true);
						atkDel = 2000;
					}
					else
					{
						AI_StartAnim(obj, IRONANIM_FISTBLOW, true);
						atkDel = 2000;
					}
					if (obj->generalFlag <= -1)
					{
						obj->aiObj->attackTime = g_glb.gameTime + Util_LocalDelay(obj, (serverTime_t)(atkDel+rand()%2000));
					}
					else
					{
						obj->aiObj->attackTime = g_glb.gameTime + Util_LocalDelay(obj, (serverTime_t)(atkDel+1000+rand()%4000));
					}
					obj->aiObj->noMoveTime = g_glb.gameTime + Util_LocalDelay(obj, atkDel);
				}
			}
			else
			{
				float closeDist = 1200.0f;
				float r = (shouldCloseIn) ? closeDist : 1600.0f;
				obj->aiObj->combatRange = g_glb.ai.playerCloseDist+r;
			}
		}
	}
}

//pick which anim to be in
void ObjIron_PickAnim(gameObject_t *obj, float timeMod)
{
	gameAnim_t *curAnim = obj->animTable+obj->curAnim;

	if (obj->net.frame == curAnim->endFrame && !obj->animRestart)
	{
		AI_StartAnim(obj, HUMANIM_IDLE, true);
	}
}

//frame tick
void ObjIron_FrameTick(gameObject_t *obj, float timeMod, int oldFrame)
{
	switch (obj->curAnim)
	{
	case HUMANIM_WALK:
	case HUMANIM_RUNSLOW:
	case HUMANIM_RUN:
		if (obj->onGround && (obj->net.frame == 367 || obj->net.frame == 364))
		{
			char *stepSounds[3] =
			{
				"assets/sound/cb/bigfoot1.wav",
				"assets/sound/cb/bigfoot2.wav",
				"assets/sound/cb/bigfoot3.wav"
			};
			ObjSound_Create(obj->net.pos, stepSounds[rand()%3], 1.0f, -1);
			Util_ProximityShake(obj->net.pos, 4096.0f, 5.0f, 0.5f);
		}
		break;
	case IRONANIM_POWERSLASH:
		if (obj->net.frame == 135)
		{
			ObjSound_CreateFromIndex(obj->net.pos, g_soundDeepBlow[rand()%NUM_SOUNDS_DEEPBLOW], 1.0f, -1);
		}
		break;
	case IRONANIM_POWERSLASH2:
		if (obj->net.frame == 232)
		{
			ObjSound_CreateFromIndex(obj->net.pos, g_soundDeepBlow[rand()%NUM_SOUNDS_DEEPBLOW], 1.0f, -1);
		}
		break;
	case IRONANIM_FISTBLOW:
		if (obj->net.frame == 79)
		{
			ObjSound_CreateFromIndex(obj->net.pos, g_soundDeepBlow[rand()%NUM_SOUNDS_DEEPBLOW], 1.0f, -1);
		}
		break;
	case IRONANIM_POWERUP:
		if (obj->net.frame == 175)
		{
			float p[3];
			float fwd[3];
			Math_AngleVectors(obj->net.ang, fwd, 0, 0);
			Math_VecCopy(obj->net.pos, p);
			Math_VecMA(p, 800.0f, fwd, p);
			p[2] += 8.0f;
			float ang[3] = {-90.0f, 0.0f, 0.0f};
			ObjSound_Create(p, "assets/sound/cb/fistpower.wav", 1.0f, -1);
			ObjParticles_Create("impacts/ironpow", p, ang, -1);
			Util_ProximityShake(p, 4096.0f, 12.0f, 1.0f);
			obj->debouncePowerup = g_glb.gameTime + 10000;
			obj->net.renderEffects2 |= FXFL2_INVINCIBLE;
		}
		break;
	default:
		break;
	}
}

//touch
void ObjIron_Touch(gameObject_t *obj, gameObject_t *other, const collObj_t *col)
{
	AI_GenericTouch(obj, other, col);
}

//become invincible (mostly) with powerup
bool ObjIron_AttackBlock(gameObject_t *obj, gameObject_t *hurter, int dmg, collObj_t *col,
							float *impactPos, float *impactAng)
{
	if (dmg > 999 || obj->debouncePowerup < g_glb.gameTime)
	{
		return false;
	}

	ObjParticles_Create("melee/impact", impactPos, impactAng, -1);
	ObjSound_Create(impactPos, "assets/sound/cb/hitmetal.wav", 1.0f, -1);
	//ObjParticles_Create("melee/impactexplosive", impactPos, impactAng, -1);
	//ObjSound_Create(impactPos, "assets/sound/cb/hitaura.wav", 1.0f, -1);
	//Util_RadiusDamage(obj, dmg, 900.0f, impactPos);
	return true;
}

//spawn
void ObjIron_Spawn(gameObject_t *obj, BYTE *b, const objArgs_t *args, int numArgs)
{
	AI_GenericSpawn(obj, b, args, numArgs);

	obj->touch = ObjIron_Touch;
	obj->animframetick = ObjIron_FrameTick;
	obj->aiObj->painChance = 5;
	obj->aiObj->forceGravity = 300.0f;

	const char *elvalue = Common_GetValForKey(b, "elvalue");
	if (elvalue && elvalue[0])
	{
		obj->generalFlag = atoi(elvalue);
	}

	if (obj->generalFlag == -1)
	{
		obj->aiObj->makoValue = 200;
		obj->net.aiDescIndex = AIDESC_IRON2;
	}
	else
	{
		obj->aiObj->makoValue = 100;
		obj->net.aiDescIndex = AIDESC_IRON1;
	}

	obj->aiObj->dropChances[INVITEM_GRAVIBALL] = 20;
	if (obj->generalFlag == -1)
	{
		obj->aiObj->dropChances[INVITEM_GRAVIBALL] = 80;
	}
	obj->aiObj->maxHealth = obj->health;

	obj->aiObj->dmgEffect = "melee/impacten";
	obj->aiObj->dmgSounds = g_soundHitHeavy;
	obj->aiObj->numDmgSounds = NUM_SOUNDS_HITHEAVY;
	if (obj->generalFlag == -1)
	{
		obj->aiObj->moveSpeed = 150.0f;
	}
	else
	{
		obj->aiObj->moveSpeed = 150.0f;
	}
	obj->animhandler = ObjIron_PickAnim;
	obj->animTable = g_ironAnims;
	obj->curAnim = HUMANIM_IDLE;
	obj->think = ObjIron_Think;
	obj->attackblock = ObjIron_AttackBlock;
}
