/*
=============================================================================
Module Information
------------------
Name:			obj_partemit.cpp
Author:			Rich Whitehouse
Description:	server logic object: particle emitter

Object Arguments
pcl:			Name of the particle system to emit.
pclofs:			Vector in the format of "(f f f)" specifying an absolute positional
				offset for the particle system being emitted. Useful to emit a particle
				system from within a solid object or somewhere where visibility checks
				would fail on the emitter object.
delay:			Interval in milliseconds between emission.
onetime:		Remove after emitting the particle system.
startoff:		Starts off, must be activated to begin emission.
=============================================================================
*/

#include "main.h"

#define PEFL_ONETIME		(1<<1)
#define PEFL_ENABLED		(1<<2)

//frame function
void ObjPartEmit_Think(gameObject_t *obj, float timeMod)
{
	if (!(obj->generalFlag & PEFL_ENABLED))
	{
		obj->thinkTime = g_glb.gameTime;
		return;
	}

	gameObject_t *cam = Util_GetCam(0);
	if (!g_sharedFn->Coll_GeneralVisibilityWithBounds(obj->net.pos, cam->net.pos, obj->radius, obj->spawnMins, obj->spawnMaxs))
	{ //don't emit when not visible
		obj->thinkTime = g_glb.gameTime;
		return;
	}

	float p[3];
	Math_VecAdd(obj->net.pos, obj->net.modelScale, p);
	ObjParticles_CreateFromIndex(obj->net.strIndex, p, obj->net.ang, -1);
	obj->thinkTime = g_glb.gameTime+obj->debounce;
	if (obj->generalFlag & PEFL_ONETIME)
	{ //one-time
		obj->think = ObjGeneral_RemoveThink;
	}
}

//activate
void ObjPartEmit_Activate(gameObject_t *obj, gameObject_t *activator)
{
	obj->generalFlag ^= PEFL_ENABLED;
}

//spawn
void ObjPartEmit_Spawn(gameObject_t *obj, BYTE *b, const objArgs_t *args, int numArgs)
{
	obj->localFlags |= LFL_NONET;
	obj->inuse |= INUSE_NONET; //don't need emitters sent to the client ever.

	int oneTime = 0;
	int delay = 50;
	obj->net.modelScale[0] = 0.0f;
	obj->net.modelScale[1] = 0.0f;
	obj->net.modelScale[2] = 0.0f;
	const char *pcl = NULL;
	obj->generalFlag = PEFL_ENABLED;
	if (obj->spawnArgs && obj->numSpawnArgs > 0)
	{
		int i = 0;
		while (i < obj->numSpawnArgs)
		{
			const objArgs_t *arg = obj->spawnArgs+i;
			if (!_stricmp(arg->key, "onetime"))
			{
				oneTime = atoi(arg->val);
			}
			else if (!_stricmp(arg->key, "pcl"))
			{
				pcl = arg->val;
			}
			else if (!_stricmp(arg->key, "pclofs"))
			{
				sscanf(arg->val, "(%f %f %f)", &obj->net.modelScale[0], &obj->net.modelScale[1],
					&obj->net.modelScale[2]);
			}
			else if (!_stricmp(arg->key, "delay"))
			{
				delay = atoi(arg->val);
			}
			else if (!_stricmp(arg->key, "startoff"))
			{
				obj->generalFlag &= ~PEFL_ENABLED;
			}
			i++;
		}
	}

	if (!pcl)
	{
		obj->think = ObjGeneral_RemoveThink;
		obj->thinkTime = g_glb.gameTime+50;
		return;
	}

	obj->debounce = (serverTime_t)delay;

	obj->think = ObjPartEmit_Think;
	if (oneTime)
	{
		obj->generalFlag |= PEFL_ONETIME;
		obj->thinkTime = g_glb.gameTime+obj->debounce;
	}
	else
	{
		obj->thinkTime = g_glb.gameTime+50;
	}

	obj->radius = Util_RadiusFromBounds(obj->spawnMins, obj->spawnMaxs, 1.0f);

	obj->activate = ObjPartEmit_Activate;

	char str[64];
    sprintf(str, "&&%s", pcl);
	obj->net.strIndex = g_sharedFn->Common_ServerString(str);
}
