/*
=============================================================================
Module Information
------------------
Name:			obj_plbar.cpp
Author:			Rich Whitehouse
Description:	server logic object: player barrett
=============================================================================
*/

#include "main.h"
#include "ai.h"

extern bool ObjPlayer_ButtonTapped(gameObject_t *obj, int button);
extern bool ObjPlayer_TargetValid(gameObject_t *obj, gameObject_t *target);
extern gameObject_t *ObjPlayer_FindTarget(gameObject_t *obj);
extern void ObjPlayer_AnalogMove(gameObject_t *obj, float &move, float &moveRight,
							  const float moveSpeed, const float timeMod);
extern bool ObjPlayer_CanTakeAction(gameObject_t *obj);
extern bool ObjPlayer_AnglingAnim(gameObject_t *obj);
extern bool ObjPlayer_NonAnglingAnim(gameObject_t *obj);
extern void ObjPlayer_PickAnim(gameObject_t *obj, float timeMod);
extern void ObjPlayer_EncodeNetData(gameObject_t *obj);
extern void ObjPlayer_CheckCharge(gameObject_t *obj, float timeMod);

typedef enum
{
	PLBAR_WALLJUMP = NUM_HUMAN_ANIMS,
	PLBAR_DIVE_BOUNCE,
	PLBAR_DODGE_LEFT,
	PLBAR_DODGE_RIGHT,
	PLBAR_BACKFLIP,
	PLBAR_LUNGE,
	PLBAR_TACKLE,
	PLBAR_STRAFE_LEFT,
	PLBAR_STRAFE_RIGHT,
	PLBAR_TARG_FWD,
	PLBAR_TARG_BACK,
	PLBAR_PUNCH_R1,
	PLBAR_PUNCH_R2,
	PLBAR_C1_COMBO1,
	PLBAR_C1_COMBO2,
	PLBAR_C1_COMBO3,
	PLBAR_PUNCH_L1,
	PLBAR_PUNCH_UPPERCUT,
	PLBAR_PUNCH_AIRCRUSH,
	PLBAR_PUNCH_AIRCRUSH_LAND,
	PLBAR_DIVE_LAND,
	PLBAR_ITEM_USE,
	PLBAR_KICK_R1,
	PLBAR_KICK_R2,
	PLBAR_KICK_R3,
	PLBAR_KICK_R4,
	PLBAR_KICK_BACKFLIP,
	PLBAR_KICK_AIR_FLIP,
	PLBAR_KICK_DIVE,
	PLBAR_KICK_AIR1,
	PLBAR_KICK_AIR2,
	PLBAR_KICK_AIR3,
	PLBAR_KICK_AIR4,
	PLBAR_LUNGE_FINISH,
	PLBAR_TACKLE_FINISH,
	PLBAR_RISE,
	PLBAR_SHOOT,
	PLBAR_DEATH,
	PLBAR_DEATH_LAND,
	PLBAR_CIN_IDLE_1,
	PLBAR_CIN_COMBATREADY,
	PLBAR_CIN_SITTING,
	PLBAR_CIN_GETUP,
	PLBAR_CIN_CALLOUT,
	PLBAR_CIN_CALLDONE,
	PLBAR_CIN_TALKAERITH,
	NUM_PLBAR_ANIMS
} plBarAnims_e;

static scriptableAnim_t g_plScriptAnims[] =
{
	DEF_SCRIPT_ANIM(HUMANIM_IDLE),
	DEF_SCRIPT_ANIM(PLBAR_PUNCH_AIRCRUSH_LAND),
	DEF_SCRIPT_ANIM(PLBAR_CIN_IDLE_1),
	DEF_SCRIPT_ANIM(PLBAR_CIN_COMBATREADY),
	DEF_SCRIPT_ANIM(PLBAR_CIN_SITTING),
	DEF_SCRIPT_ANIM(PLBAR_CIN_GETUP),
	DEF_SCRIPT_ANIM(PLBAR_CIN_CALLOUT),
	DEF_SCRIPT_ANIM(PLBAR_CIN_CALLDONE),
	DEF_SCRIPT_ANIM(PLBAR_CIN_TALKAERITH),
	{0, NULL}
};

static gameAnim_t g_playerAnims[NUM_PLBAR_ANIMS] =
{
	{0, 19, 68.0f, true},			//HUMANIM_IDLE
	{92, 97, 168.0f, true},			//HUMANIM_WALK
	{92, 97, 118.0f, true},			//HUMANIM_RUNSLOW
	{92, 97, 88.0f, true},			//HUMANIM_RUN
	{0, 2, 68.0f, false},			//HUMANIM_JUMP
	{93, 93, 68.0f, false},			//HUMANIM_FALL
	{0, 2, 68.0f, false},			//HUMANIM_LAND
	{79, 81, 168.0f, false},		//HUMANIM_PAIN_HIGH1
	{79, 81, 168.0f, false},		//HUMANIM_PAIN_HIGH2
	{79, 81, 168.0f, false},		//HUMANIM_PAIN_LOW1
	{79, 81, 168.0f, false},		//HUMANIM_PAIN_LOW2
	{79, 81, 168.0f, false},		//HUMANIM_PAIN_AIR
	{79, 81, 168.0f, false},		//HUMANIM_PAIN_POPUP
	{30, 32, 168.0f, false},		//HUMANIM_GETUP_BACK
	{30, 32, 168.0f, false},		//HUMANIM_GETUP_FRONT
	{196, 196, 68.0f, false},		//HUMANIM_FLYBACK
	{196, 196, 68.0f, false},		//HUMANIM_FLYBACK2
	{19, 22, 168.0f, false},		//HUMANIM_HIT_FALLFORWARD
	{19, 22, 68.0f, false},			//HUMANIM_FALL_LAND
	{19, 22, 68.0f, false},			//HUMANIM_POPUP_LAND
	{0, 19, 1568.0f, false},		//HUMANIM_DEATH
	{457, 462, 68.0f, false},		//PLBAR_WALLJUMP
	{511, 515, 68.0f, false},		//PLBAR_DIVE_BOUNCE
	{495, 500, 68.0f, false},		//PLBAR_DODGE_LEFT
	{501, 506, 68.0f, false},		//PLBAR_DODGE_RIGHT
	{507, 515, 68.0f, false},		//PLBAR_BACKFLIP
	{516, 517, 68.0f, false},		//PLBAR_LUNGE
	{619, 619, 68.0f, false},		//PLBAR_TACKLE
	{477, 481, 118.0f, true},		//PLBAR_STRAFE_LEFT
	{482, 486, 118.0f, true},		//PLBAR_STRAFE_RIGHT
	{487, 490, 118.0f, true},		//PLBAR_TARG_FWD
	{491, 494, 118.0f, true},		//PLBAR_TARG_BACK
	{336, 341, 68.0f, false},		//PLBAR_PUNCH_R1
	{134, 143, 68.0f, false},		//PLBAR_PUNCH_R2
	{138, 143, 68.0f, false},		//PLBAR_C1_COMBO1
	{394, 401, 68.0f, false},		//PLBAR_C1_COMBO2
	{371, 375, 68.0f, false},		//PLBAR_C1_COMBO3
	{370, 375, 68.0f, false},		//PLBAR_PUNCH_L1
	{467, 476, 68.0f, false},		//PLBAR_PUNCH_UPPERCUT
	{560, 563, 118.0f, false},		//PLBAR_PUNCH_AIRCRUSH
	{564, 571, 168.0f, false},		//PLBAR_PUNCH_AIRCRUSH_LAND
	{568, 571, 118.0f, false},		//PLBAR_DIVE_LAND
	{120, 131, 118.0f, false},		//PLBAR_ITEM_USE
	{343, 350, 68.0f, false},		//PLBAR_KICK_R1
	{355, 367, 68.0f, false},		//PLBAR_KICK_R2
	{375, 383, 68.0f, false},		//PLBAR_KICK_R3
	{389, 401, 68.0f, false},		//PLBAR_KICK_R4
	{600, 608, 68.0f, false},		//PLBAR_KICK_BACKFLIP
	{609, 614, 68.0f, false},		//PLBAR_KICK_AIR_FLIP
	{616, 618, 68.0f, false},		//PLBAR_KICK_DIVE
	{539, 546, 68.0f, false},		//PLBAR_KICK_AIR1
	{547, 552, 68.0f, false},		//PLBAR_KICK_AIR2
	{526, 532, 68.0f, false},		//PLBAR_KICK_AIR3
	{533, 538, 68.0f, false},		//PLBAR_KICK_AIR4
	{518, 525, 68.0f, false},		//PLBAR_LUNGE_FINISH
	{384, 389, 68.0f, false},		//PLBAR_TACKLE_FINISH
	{620, 632, 68.0f, false},		//PLBAR_RISE
	{481, 482, 68.0f, false},		//PLBAR_SHOOT
	{218, 222, 68.0f, false},		//PLBAR_DEATH
	{223, 225, 68.0f, false},		//PLBAR_DEATH_LAND
	{655, 656, 1068.0f, true},		//PLBAR_CIN_IDLE_1
	{0, 13, 68.0f, true},			//PLBAR_CIN_COMBATREADY
	{657, 657, 68.0f, false},		//PLBAR_CIN_SITTING
	{658, 670, 168.0f, false},		//PLBAR_CIN_GETUP
	{671, 672, 368.0f, false},		//PLBAR_CIN_CALLOUT
	{672, 674, 268.0f, false},		//PLBAR_CIN_CALLDONE
	{675, 675, 468.0f, false}		//PLBAR_CIN_TALKAERITH
};

//frame function
void ObjPlBar_Think(gameObject_t *obj, float timeMod)
{
	if (g_glb.cinema == 1)
	{
		return;
	}

	if (obj->plObj->waitOnInput > 0)
	{ //script input
		if (ObjPlayer_ButtonTapped(obj, obj->plObj->waitOnInput-1))
		{
			obj->plObj->waitOnInput = -1;
		}
	}

	gameObject_t *cam = Util_GetCam(obj->net.index);

	if (cam->inuse)
	{
		if (obj->plObj->playerSpawnSeq)
		{
			obj->net.pos[0] = g_glb.initialFPSPos[0];
			obj->net.pos[1] = g_glb.initialFPSPos[1];
			obj->net.pos[2] = g_glb.initialFPSPos[2];
			obj->net.ang[0] = g_glb.initialFPSAng[0];
			obj->net.ang[1] = g_glb.initialFPSAng[1];
			obj->net.ang[2] = g_glb.initialFPSAng[2];
			Math_VecCopy(obj->net.pos, obj->safePos);
			g_sharedFn->Coll_GetWorldBounds(g_glb.worldMins, g_glb.worldMaxs);
			obj->plObj->playerSpawnSeq = false;
			Phys_PutOnGround(obj);
		}

		//terrible hackery to prevent player from falling out of world
		float plAbsMins[3], plAbsMaxs[3];
		for (int k = 0; k < 3; k++)
		{
			plAbsMins[k] = obj->net.pos[k]+obj->net.mins[k];
			plAbsMaxs[k] = obj->net.pos[k]+obj->net.maxs[k];
		}
		if (!Math_BoxesOverlap(plAbsMins, plAbsMaxs, g_glb.worldMins, g_glb.worldMaxs))
		{
			obj->net.pos[0] = g_glb.initialFPSPos[0];
			obj->net.pos[1] = g_glb.initialFPSPos[1];
			obj->net.pos[2] = g_glb.initialFPSPos[2];
			obj->net.ang[0] = g_glb.initialFPSAng[0];
			obj->net.ang[1] = g_glb.initialFPSAng[1];
			obj->net.ang[2] = g_glb.initialFPSAng[2];
			Math_VecCopy(obj->net.pos, obj->safePos);
		}
	}

	AI_CheckEffects(obj, timeMod);

	if (obj->plObj->clButtons[BUTTON_EVENT7] && obj->health > 0)
	{ //find a target
		if (!ObjPlayer_TargetValid(obj, obj->plObj->targetObj))
		{
			if (obj->plObj->targetObj)
			{
				obj->plObj->targetObj->net.renderEffects &= ~ObjPlayer_TargetBit(obj);
				obj->plObj->targetObj = NULL;
			}
		}

		if (!obj->plObj->targetObj/* && (!ObjPlayer_InChain(obj) || ObjPlayer_InChainTransition(obj)) && !ObjPlayer_NonMovingAnim(obj)*/)
		{ //find a new target
			obj->plObj->targetObj = ObjPlayer_FindTarget(obj);
			if (obj->plObj->targetObj)
			{
				obj->plObj->targetObj->net.renderEffects |= ObjPlayer_TargetBit(obj);
			}
		}
	}
	else
	{
		if (obj->plObj->targetObj)
		{
			obj->plObj->targetObj->net.renderEffects &= ~ObjPlayer_TargetBit(obj);
			obj->plObj->targetObj = NULL;
		}
	}

	float moveSpeed = 400.0f*obj->plObj->moveSpeedScale;//48.0f;
	float move = 0.0f, rightMove = 0.0f;
	float fwd[3], right[3];

	if (!obj->onGround)
	{
		moveSpeed *= 0.085f;
	}

	if (obj->plObj->clButtons[BUTTON_UP])
	{ //up
		move = moveSpeed;
	}
	else if (obj->plObj->clButtons[BUTTON_DOWN])
	{ //down
		move = -moveSpeed;
	}
	if (obj->plObj->clButtons[BUTTON_RIGHT])
	{ //right
		rightMove = moveSpeed;
	}
	else if (obj->plObj->clButtons[BUTTON_LEFT])
	{ //left
		rightMove = -moveSpeed;
	}

	if (move && rightMove)
	{
		move *= 0.5f;
		rightMove *= 0.5f;
	}

	ObjPlayer_AnalogMove(obj, move, rightMove, moveSpeed, timeMod);

	//ObjPlayer_AdjustTurnAngles(obj, timeMod, obj->net.ang);

	//compensate for time
	move *= timeMod;
	rightMove *= timeMod;
	float moveAng[3];
	moveAng[0] = 0.0f;
	moveAng[1] = cam->net.ang[1];
	moveAng[2] = 0.0f;

	Math_AngleVectors(moveAng, fwd, right, 0);
	float moveVec[3];
	moveVec[0] = (fwd[0]*move)+(right[0]*rightMove);
	moveVec[1] = (fwd[1]*move)+(right[1]*rightMove);
	moveVec[2] = (fwd[2]*move)+(right[2]*rightMove);

	AI_ApplyAnimationEffects(obj, timeMod);

	if (obj->curAnim != PLBAR_SHOOT)
	{
		obj->net.vel[0] += moveVec[0];
		obj->net.vel[1] += moveVec[1];
		obj->net.vel[2] += moveVec[2];
	}

	Math_VecNorm(moveVec);

	if (obj->onGround)
	{
		if (obj->plObj->clButtons[BUTTON_EVENT8])
		{
			AI_StartAnim(obj, PLBAR_SHOOT, false);
		}
		else if (ObjPlayer_CanTakeAction(obj) && ObjPlayer_ButtonTapped(obj, BUTTON_JUMP))
		{
			ObjSound_Create(obj->net.pos, "assets/sound/cb/jump01.wav", 1.0f, -1);
			obj->net.vel[2] = 1200.0f;
			obj->net.vel[0] *= 0.25f;
			obj->net.vel[1] *= 0.25f;
		}
	}

	//update model angles
	float animAng[3];
	if (obj->curAnim == PLBAR_SHOOT)
	{
		float angleBlendScale = 60.0f;
		obj->net.ang[PITCH] = Math_BlendAngleLinear(obj->net.ang[PITCH], 0.0f, timeMod*angleBlendScale);
		obj->net.ang[YAW] = Math_BlendAngleLinear(obj->net.ang[YAW], Math_AngleMod(cam->net.ang[YAW]), timeMod*angleBlendScale);
		obj->net.ang[ROLL] = Math_BlendAngleLinear(obj->net.ang[ROLL], 0.0f, timeMod*angleBlendScale);
	}
	else if (AI_AnglesForAnim(obj, animAng))
	{
		float angleBlendScale = 0.7f;
		obj->net.ang[PITCH] = Math_BlendAngle(obj->net.ang[PITCH], animAng[PITCH], timeMod*angleBlendScale);
		obj->net.ang[YAW] = Math_BlendAngle(obj->net.ang[YAW], animAng[YAW], timeMod*angleBlendScale);
		obj->net.ang[ROLL] = Math_BlendAngle(obj->net.ang[ROLL], animAng[ROLL], timeMod*angleBlendScale);
	}
	else if (!ObjPlayer_NonAnglingAnim(obj))
	{
		if (obj->plObj->targetObj && !ObjPlayer_AnglingAnim(obj))
		{ //turn toward the target
			float targCenter[3], plCenter[3], targVec[3], targAng[3];
			float angleBlendScale = 0.7f;
			Util_GameObjectCenter(obj->plObj->targetObj, targCenter);
			Util_GameObjectCenter(obj, plCenter);
			Math_VecSub(targCenter, plCenter, targVec);
			Math_VecToAngles(targVec, targAng);
			obj->net.ang[YAW] = Math_BlendAngle(obj->net.ang[YAW], targAng[YAW], timeMod*angleBlendScale);
		}
		else
		{
			float moveAngTol = obj->onGround ? 1.0f : 100.0f;
			float velCheck[3] = {obj->net.vel[0], obj->net.vel[1], 0.0f};
			float velCheckLen = Math_VecLen(velCheck);
			if ((move || rightMove) && velCheckLen > moveAngTol)
			{
				float velBasedTurn = (velCheckLen > 64.0f) ? 0.5f : 0.05f;
				float angleBlendScale = velBasedTurn;
				float velAng[3];
				Math_VecToAngles(obj->net.vel, velAng);
				obj->net.ang[YAW] = Math_BlendAngle(obj->net.ang[YAW], velAng[YAW], timeMod*angleBlendScale);
			}
		}
	}

	float idealPitch = 0.0f;
	float idealRoll = 0.0f;
	/*
	switch (obj->curAnim)
	{
	case PLANIM_DODGE_LEFT:
		idealRoll = -30.0f;
		break;
	case PLANIM_DODGE_RIGHT:
		idealRoll = 30.0f;
		break;
	default:
		break;
	}
	*/
	obj->net.ang[PITCH] = Math_BlendAngle(obj->net.ang[PITCH], idealPitch, timeMod*0.3f);
	obj->net.ang[ROLL] = Math_BlendAngle(obj->net.ang[ROLL], idealRoll, timeMod*0.3f);

	AI_PickAnim(obj, timeMod);
	Util_AnimateObject(obj, timeMod);

	if (!obj->plObj->playerSpawnSeq)
	{ //translation safety checking
		Util_SetSafeSpot(obj);
		Phys_ApplyObjectPhysics(obj, timeMod, obj->radius, AI_GetGravity(obj, timeMod), 0.0f);
	}
	LServ_UpdateRClip(obj);

	ObjPlayer_CheckCharge(obj, timeMod);

	if (obj->plObj->timeAttackTime && obj->plObj->timeAttackTime < g_glb.gameTime)
	{
		ObjSound_Create(obj->net.pos, "assets/sound/cb/chargeoff.wav", 1.0f, -1);
		if (g_glb.timeType == TIMETYPE_TIMEATTACK)
		{
			g_glb.timeType = TIMETYPE_NORMAL;
		}
		obj->plObj->timeAttackTime = 0;
	}

	ObjPlayer_EncodeNetData(obj);

	obj->plObj->numButtonsBuffered = 0; //processed all input
}

//frame tick
void ObjPlBar_FrameTick(gameObject_t *obj, float timeMod, int oldFrame)
{
	switch (obj->curAnim)
	{
	case HUMANIM_WALK:
	case HUMANIM_RUNSLOW:
	case HUMANIM_RUN:
		if ((obj->net.frame == 92 || obj->net.frame == 95) && obj->onGround)
		{
			int snd = g_soundFootsteps[rand()%NUM_SOUNDS_FOOTSTEPS];
			ObjSound_CreateFromIndex(obj->net.pos, snd, 1.0f, -1);
		}
		break;
	case PLBAR_SHOOT:
		if (oldFrame == 481 || oldFrame == 482)
		{
			float pos[3] = {0.0f, -116.0f, -450.0f};
			float ang[3] = {0.0f, 0.0f, -1.0f};
			modelMatrix_t boneMat;
			if (g_sharedFn->Coll_GetModelBoneMatrix(obj->rcColModel, "b12", &boneMat))
			{
				float t[3];
				Math_VecCopy(pos, t);
				Math_TransformPointByMatrix(&boneMat, t, pos);

				gameObject_t *cam = Util_GetCam(obj->net.index);
				if (1)
				{ //fire on cam angle
					Math_AngleVectors(cam->net.ang, ang, 0, 0);

					//add some spread
					float maxSpread = 0.05f;
					float r;
					r = (float)(rand()%16384)/16384.0f;
					float spreadX = -maxSpread + r*maxSpread*2.0f;
					r = (float)(rand()%16384)/16384.0f;
					float spreadY =  -maxSpread + r*maxSpread*2.0f;
					Math_VecNorm(ang);
					float right[3], up[3];
					Math_AxisForNormal(ang, 0, right, up);
					Math_VecScale(right, spreadX);
					Math_VecScale(up, spreadY);
					Math_VecAdd(ang, right, ang);
					Math_VecAdd(ang, up, ang);
				}
				else
				{ //fire where the barrel is pointed
					Math_VecCopy(ang, t);
					Math_TransformPointByMatrix(&boneMat, t, ang);
					Math_VecSub(ang, boneMat.o, ang);
				}
			}
			Math_VecNorm(ang);
			Util_FireBullet(obj, pos, ang, NULL);

			Math_VecToAngles(ang, ang);
			ObjParticles_Create("barrettfire", pos, ang, -1);

			ObjSound_Create(obj->net.pos, "assets/sound/cin/shot02.wav", 1.0f, -1);
		}
		break;
	default:
		break;
	}
}

//pick which anim to be in
void ObjPlBar_PickAnim(gameObject_t *obj, float timeMod)
{
	gameAnim_t *curAnim = obj->animTable+obj->curAnim;

	if (obj->curAnim >= PLBAR_CIN_IDLE_1)
	{ //script will break out
		return;
	}
	else if (obj->curAnim == PLBAR_SHOOT)
	{
		if (obj->net.frame == curAnim->endFrame && !obj->animRestart)
		{
			if (obj->plObj->clButtons[BUTTON_EVENT8])
			{
				obj->animRestart = true;
			}
			else
			{
				AI_StartAnim(obj, HUMANIM_IDLE, true);
			}
		}
	}
	else
	{
		ObjPlayer_PickAnim(obj, timeMod);
	}
}

//client joined, create its map object.
void ObjPlBar_Init(gameObject_t *pl)
{
	pl->plObj->moveSpeedScale = 1.0f;

	pl->animTable = g_playerAnims;
	pl->numAnims = NUM_PLBAR_ANIMS;
	pl->scriptAnims = g_plScriptAnims;
	pl->curAnim = HUMANIM_IDLE;
	pl->animhandler = ObjPlBar_PickAnim;
	pl->animframetick = ObjPlBar_FrameTick;
	pl->animidleoverride = 0;//ObjPlayer_IdleAnim;

	pl->health = pl->plObj->plData.maxHealth;
	pl->hurtable = 1;

	pl->plObj->playerSpawnSeq = true;

	pl->net.ang[0] = 0.0f;
	pl->net.ang[1] = 0.0f;
	pl->net.ang[2] = 0.0f;

	pl->net.mins[0] = -100.0f;
	pl->net.mins[1] = -100.0f;
	pl->net.mins[2] = 0.0f;
	pl->net.maxs[0] = 100.0f;
	pl->net.maxs[1] = 100.0f;
	pl->net.maxs[2] = 430.0f;
	pl->radius = 200.0f;

	pl->net.plAmmo = 0;

	pl->net.solid = 1;

	pl->net.strIndex = g_sharedFn->Common_ServerString("&assets/models/barrett.rdm");
	pl->net.strIndexB = g_sharedFn->Common_ServerString("@assets/models/barrett.rda");

	pl->think = ObjPlBar_Think;

	pl->rcColModel = g_sharedFn->Coll_RegisterModelInstance("assets/models/barrett.rdm");
	if (pl->rcColModel)
	{
		pl->rcColModel->radius = pl->radius;
		pl->rcColModel->clipFlags = CLIPFL_BOXMOVE;
		Math_VecCopy(pl->net.mins, pl->rcColModel->mins);
		Math_VecCopy(pl->net.maxs, pl->rcColModel->maxs);
		Math_VecCopy(pl->net.pos, pl->rcColModel->pos);
		Math_VecCopy(pl->net.ang, pl->rcColModel->ang);
		Math_VecCopy(pl->net.modelScale, pl->rcColModel->modelScale);
		pl->rcColModel->gameOwner = pl->net.index;
		pl->rcColModel->solid = pl->net.solid;
		pl->rcColModel->frame = pl->net.frame;
		g_sharedFn->Coll_UpdateModel(pl->rcColModel, "assets/models/barrett.rda");
	}
}
