/*
=============================================================================
Module Information
------------------
Name:			physics.cpp
Author:			Rich Whitehouse
Description:	server logic module physics routines.

This is using my own collision system and disregarding novodex support.
This code is all horrible and from several years ago.
=============================================================================
*/

#include "main.h"

static float g_gravFactor = DEFAULT_GRAVITY;
static const bool g_checkForSolid = true;

#ifdef _DEBUG
//#define _DEBUG_INSOLID
#endif

//interpolate difference
float Phys_InterpolateAngle(float old, float cur, float val, float tresh)
{
	if ((cur-old) > 180.0f)
	{
		cur -= 360.0f;
	}
	if ((cur-old) < -180.0f)
	{
		cur += 360.0f;
	}

	if (fabsf(cur-old) > tresh)
	{ //too big of a difference.
		return cur;
	}

	return Math_AngleMod(old + (val*(cur-old)));
}

//clip velocity
static void Phys_ClipVel(float *vel, float *normal)
{
	float dp = Math_DotProduct(vel, normal);
	if (dp < 0.0f)
	{
		/*
		for (int i = 0; i < 3; i++)
		{
			float f = dp*normal[i];
			if (fabsf(f) > fabsf(vel[i]))
			{
				vel[i] = 0.0f;
			}
			else
			{
				vel[i] -= dp*normal[i];
			}
		}
		*/
		vel[0] -= dp*normal[0];
		vel[1] -= dp*normal[1];
		vel[2] -= dp*normal[2];
	}
}

//decrement velocity
static float Phys_DecVel(float vel, float amt)
{
	if (vel > 0.0f)
	{
		vel -= amt;
		if (vel < 0.0f)
		{
			vel = 0.0f;
		}
	}
	else if (vel < 0.0f)
	{
		vel += amt;
		if (vel > 0.0f)
		{
			vel = 0.0f;
		}
	}
	return vel;
}

//collide
void Phys_Collide(gameObject_t *obj, float objRad, collObj_t *collision, float *start, float *end, rcColModel_t *other, collOpt_t *opt)
{
	if (obj && obj->rcColModel)
	{
		g_sharedFn->Coll_MovementTranslation(obj->rcColModel, collision, start, end, other, opt);
	}
	else
	{
		rcColModel_t *rc = (obj->physModelOwner) ? obj->physModelOwner->rcColModel : NULL;
		g_sharedFn->Coll_RadiusTranslation(rc, collision, start, end, objRad, other, opt);
	}
}

//translate an object which pushes other collidable objects
void Phys_TranslatePusher(gameObject_t *pusher, float *dest)
{
	if (pusher->net.pos[0] == dest[0])

	if (!pusher || !pusher->rcColModel)
	{
		RCUBE_ASSERT(!"bad object for Phys_TranslatePusher");
		return;
	}

	bool pushBlocked = false;
	float travelVecTowardPusher[3];
	float travelVecAwayFromPusherN[3];
	Math_VecSub(pusher->net.pos, dest, travelVecTowardPusher);
	float moveLen = Math_VecLen(travelVecTowardPusher);
	travelVecAwayFromPusherN[0] = -travelVecTowardPusher[0];
	travelVecAwayFromPusherN[1] = -travelVecTowardPusher[1];
	travelVecAwayFromPusherN[2] = -travelVecTowardPusher[2];
	Math_VecNorm(travelVecAwayFromPusherN);

	for (int i = 0; i < g_glb.gameObjectSlots; i++)
	{
		gameObject_t *other = &g_gameObjects[i];
		if (other->inuse && other->rcColModel && other->radius != 0.0f)
		{
			if (other->net.staticIndex != -1)
			{ //don't care about static things
				continue;
			}
			float goalPos[3];
			Math_VecAdd(other->net.pos, travelVecTowardPusher, goalPos);
			collObj_t col;
			Phys_Collide(other, other->radius, &col, other->net.pos,
				goalPos, pusher->rcColModel, NULL);
			if (col.hit)
			{ //the pusher will hit this thing during the translation
				Util_TouchObjects(pusher, other, &col);
				Math_VecMA(other->net.pos, col.distance+moveLen+1.0f, travelVecAwayFromPusherN, goalPos);
				g_sharedFn->Coll_MovementTranslation(other->rcColModel, &col, other->net.pos,
					goalPos, NULL, NULL);
				if (!col.hit)
				{ //successful push
					Math_VecCopy(col.endPos, other->net.pos);
					LServ_UpdateRClip(other);
				}
				else
				{ //crush!
					Util_DamageObject(pusher, other, 10000);
					pushBlocked = true;
				}
			}
		}
	}

	if (!pushBlocked)
	{
		Math_VecCopy(dest, pusher->net.pos);
		LServ_UpdateRClip(pusher);
	}
}

//get it off
void Phys_ThrowOffHead(gameObject_t *obj, gameObject_t *other, float timeMod)
{
	float c1[3], c2[3];
	Util_GameObjectCenter(obj, c1);
	Util_GameObjectCenter(other, c2);

	float d[3];
	Math_VecSub(c1, c2, d);
	if (!d[0] && !d[1])
	{
		d[0] = 1.0f;
	}
	d[2] = 0.0f;
	Math_VecNorm(d);
	Math_VecMA(obj->net.vel, 400.0f, d, obj->net.vel);
}

//apply friction
void Phys_ApplyFriction(gameObject_t *obj, float timeMod, float xyFriction, float zFriction)
{
	if (obj->net.vel[2] > 0.0f)
	{
		zFriction *= 2.0f;
	}
	float fxy = (xyFriction*timeMod);
	float fz = (zFriction*timeMod);
	if (fxy > 1.0f)
	{
		fxy = 1.0f;
	}
	if (fz > 1.0f)
	{
		fz = 1.0f;
	}
	obj->net.vel[0] *= 1.0f-fxy;
	obj->net.vel[1] *= 1.0f-fxy;
	obj->net.vel[2] *= 1.0f-fz;
}

//apply physics
void Phys_ApplyObjectPhysics(gameObject_t *obj, float timeMod, float objRad, float gravFactor, float bounceFactor)
{
#ifdef _DEBUG_INSOLID
	collObj_t testCol;
#endif
	collObj_t collision;
	float ground[3];
	float projected[3];
	float velDir[3];
	float velFactor;
	float xyFriction = 0.5f;//0.4f;
	float zFriction = 0.05f;
	float groundPl[3] = {0.0f, 0.0f, 1.0f};
	float groundDot = 0.4f;//0.6f;
	float groundDist = 16.0f;//8.0f
	if (gravFactor == 0.0f)
	{
		groundDot = 1.0f;
	}
	float oldPos[3];
	Math_VecCopy(obj->net.pos, oldPos);
	if (gravFactor != 0.0f)
	{
		//gravity/ground check
		Math_VecCopy(obj->net.pos, ground);
		ground[2] -= groundDist;
		Phys_Collide(obj, objRad, &collision, obj->net.pos, ground, NULL, NULL);
		if (collision.hit && collision.containsSolid && g_checkForSolid)
		{
			collObj_t col;
			Phys_Collide(obj, objRad, &col, obj->safePos, obj->safePos, NULL, NULL);
			if (col.hit && col.hitObjectIndex >= 0)
			{
				gameObject_t *other = &g_gameObjects[col.hitObjectIndex];
				//Util_DamageObject(other, other, 999999);
				if (/*other->health > 0 &&*/!obj->nonSolidTime && other->net.solid)
				{ //it can't be killed, no choice, make it non-solid
					if (other->net.index < MAX_NET_CLIENTS)
					{
						Util_HarmlessKnockback(obj, other);
						obj->nonSolidTime = g_glb.gameTime + 500;
					}
					else
					{
						Util_HarmlessKnockback(other, obj);
						other->nonSolidTime = g_glb.gameTime + 500;
					}
				}
			}
			Math_VecCopy(obj->safePos, obj->net.pos);
			Math_VecCopy(obj->net.pos, oldPos);
		}
		
		if (!collision.hit)
		{ //add it
			obj->onGround = false;
			obj->net.vel[2] -= gravFactor*timeMod;
		}
		else if (!collision.containsSolid)
		{ //be even with ground
			if (collision.hitObjectIndex != -1)
			{
				Util_TouchObjects(obj, &g_gameObjects[collision.hitObjectIndex], &collision);
			}
			float dp = Math_DotProduct(groundPl, collision.endNormal);
			if (dp >= groundDot)
			{
				if (!obj->onGround && obj->health > 0 && obj->rcColModel)
				{
					if (obj->preImpactVel[2] < -400.0f)
					{
						ObjSound_Create(obj->net.pos, "assets/sound/cb/objland.wav", 1.0f, -1);
					}
					else
					{
						int snd = g_soundFootsteps[rand()%NUM_SOUNDS_FOOTSTEPS];
						ObjSound_CreateFromIndex(obj->net.pos, snd, 1.0f, -1);
					}
				}
				obj->onGround = true;
				obj->groundObj = collision.hitObjectIndex;
				Math_VecCopy(collision.endPos, obj->net.pos);
				if (obj->net.vel[2] < 0.001f)
				{
					obj->net.vel[2] = 0.0f;
				}
#ifdef _DEBUG_INSOLID
				Phys_Collide(obj, objRad, &testCol, obj->net.pos, obj->net.pos, NULL, NULL);
				RCUBE_ASSERT(!testCol.containsSolid);
#endif
				gameObject_t *other = collision.hitObjectIndex >= 0 ? &g_gameObjects[collision.hitObjectIndex] : NULL;
				if (other && other->hurtable && other->rcColModel && (other->rcColModel->clipFlags & CLIPFL_BOXMOVE) &&
					obj->rcColModel && (obj->rcColModel->clipFlags & CLIPFL_BOXMOVE))
				{ //bounce off of head
					Phys_ThrowOffHead(obj, other, timeMod);
				}
			}
			else
			{
				obj->onGround = false;
				float d[3];
				d[0] = (-collision.endNormal[0]) + (-groundPl[0]);
				d[1] = (-collision.endNormal[1]) + (-groundPl[1]);
				d[2] = (-collision.endNormal[2]) + (-groundPl[2]);
				Math_VecNorm(d);
				obj->net.vel[0] += d[0]*(gravFactor*timeMod);
				obj->net.vel[1] += d[1]*(gravFactor*timeMod);
				obj->net.vel[2] += d[2]*(gravFactor*timeMod);
				//clip the velocity by the slope we may be on
				d[0] = -d[0];
				d[1] = -d[1];
				d[2] = -d[2];
				Phys_ClipVel(obj->net.vel, d);
				obj->net.vel[0] += collision.endNormal[0]*16.0f;
				obj->net.vel[1] += collision.endNormal[1]*16.0f;
			}
		}
		if (!obj->onGround)
		{
//			xyFriction = (obj->curAnim == HUMANIM_FLYBACK && Math_VecLen(obj->net.vel) > 2000.0f) ? 0.006f : 0.06f; //low friction in air
			xyFriction *= 0.25f;
		}
	}
	else
	{
		obj->onGround = false;
//		xyFriction = 0.06f; //low friction in air
		xyFriction *= 0.25f;
	}
	//check projected spot for velocity
	Math_VecCopy(obj->net.vel, velDir);
	velFactor = Math_VecNorm(velDir);
	if (!velDir[0] && !velDir[1] && !velDir[2])
	{
		velFactor = 1.0f;
	}
	velFactor *= 0.2f;
	velFactor *= timeMod;
	Math_VecMA(obj->net.pos, velFactor, velDir, projected);

	//todo: check for bounce
	//g_sharedFn->Common_Collide(obj->net.pos, projected, obj->net.mins, obj->net.maxs, obj->net.index, &collision);
	Phys_Collide(obj, objRad, &collision, obj->net.pos, projected, NULL, NULL);
#ifdef _DEBUG_INSOLID
	Phys_Collide(obj, objRad, &testCol, collision.endPos, collision.endPos, NULL, NULL);
	RCUBE_ASSERT(!testCol.containsSolid);
#endif
	//Util_DebugSphere(obj->net.pos, projected, objRad);

	if (collision.hit && !collision.containsSolid)
	{ //slide along the wall
		gameObject_t *other = collision.hitObjectIndex >= 0 ? &g_gameObjects[collision.hitObjectIndex] : NULL;
		if (other)
		{
			Util_TouchObjects(obj, other, &collision);
		}
		if (other && other->rigid)
		{ //push rigids
			float shoveVel[3];
			shoveVel[0] = obj->net.vel[0]*0.4f;
			shoveVel[1] = obj->net.vel[1]*0.4f;
			shoveVel[2] = obj->net.vel[2]*0.4f;
			Phys_ShoveRigid(other, collision.endPos, shoveVel);
		}
		else if (other && obj->canPush && other->canBePushed)
		{ //try pushing it
			Phys_ApplyFriction(obj, timeMod, xyFriction, zFriction);

			Math_VecCopy(obj->net.vel, velDir);
			velFactor = Math_VecNorm(velDir);
			if (!velDir[0] && !velDir[1] && !velDir[2])
			{
				velFactor = 1.0f;
			}
			velFactor *= 0.2f;
			velFactor *= timeMod;
			Math_VecMA(obj->net.pos, velFactor, velDir, projected);

			collObj_t colPush;
			float pushVec[3];
			Math_VecSub(projected, collision.endPos, pushVec);
			pushVec[2] = 0.0f;
			Math_VecAdd(other->net.vel, pushVec, other->net.vel);
			float pushLen = Math_VecNorm(pushVec)+4.0f;

			float testPos[3];
			Math_VecMA(other->net.pos, pushLen, pushVec, testPos);
			float prePos[3];
			Math_VecCopy(other->net.pos, prePos);
			g_sharedFn->Coll_MovementTranslation(other->rcColModel, &colPush, other->net.pos, testPos, NULL, NULL);
			if (!colPush.containsSolid)
			{
				float dt[3];
				Math_VecSub(other->net.pos, obj->net.pos, dt);
				float l1 = Math_VecLen(dt);
				Math_VecSub(colPush.endPos, obj->net.pos, dt);
				float l2 = Math_VecLen(dt);
				if (l2 >= l1)
				{
					Math_VecCopy(colPush.endPos, other->net.pos);
					LServ_UpdateRClip(other);
#ifdef _DEBUG_INSOLID
					g_sharedFn->Coll_MovementTranslation(other->rcColModel, &testCol, other->net.pos, other->net.pos, NULL, NULL);
					RCUBE_ASSERT(!testCol.containsSolid);
#endif
				}
			}
			Phys_Collide(obj, objRad, &collision, obj->net.pos, projected, NULL, NULL);
			if (collision.containsSolid)
			{ //bad push
				Math_VecCopy(prePos, other->net.pos);
				LServ_UpdateRClip(other);
				Phys_Collide(obj, objRad, &collision, obj->net.pos, projected, NULL, NULL);
			}
		}
	}

	if (collision.hit && !collision.containsSolid)
	{ //slide along the wall
		Math_VecCopy(collision.endPos, obj->net.pos);
#ifdef _DEBUG_INSOLID
		Phys_Collide(obj, objRad, &testCol, obj->net.pos, obj->net.pos, NULL, NULL);
		RCUBE_ASSERT(!testCol.containsSolid);
#endif
		gameObject_t *other = collision.hitObjectIndex >= 0 ? &g_gameObjects[collision.hitObjectIndex] : NULL;
		if (bounceFactor != 0.0f)
		{ //bounce
			float velL = Math_VecNorm(obj->net.vel);
			obj->net.vel[0] += collision.endNormal[0]*2.0f;
			obj->net.vel[1] += collision.endNormal[1]*2.0f;
			obj->net.vel[2] += collision.endNormal[2]*2.0f;
			Math_VecNorm(obj->net.vel);
            obj->net.vel[0] *= (velL*bounceFactor);
			obj->net.vel[1] *= (velL*bounceFactor);
			obj->net.vel[2] *= (velL*bounceFactor);
		}
		else if (other && other->hurtable && other->rcColModel && (other->rcColModel->clipFlags & CLIPFL_BOXMOVE) &&
			obj->rcColModel && (obj->rcColModel->clipFlags & CLIPFL_BOXMOVE) && fabsf(Math_DotProduct(groundPl, collision.endNormal)) > groundDot)
		{ //bounce off of head
			Phys_ThrowOffHead(obj, other, timeMod);
		}
		else
		{ //slide
			Math_VecCopy(obj->net.vel, obj->preImpactVel);
			Phys_ClipVel(obj->net.vel, collision.endNormal);
			Math_VecCopy(obj->net.vel, velDir);
			Math_VecNorm(velDir);
			float oldVelDir[3];
			Math_VecCopy(velDir, oldVelDir);
			//oldVelDir[2] = 0.0f;
			float dp = Math_DotProduct(groundPl, collision.endNormal);
			if (dp < groundDot && dp > -groundDot &&
				(velDir[0] || velDir[1] || velDir[2]))
			{
				Math_VecCopy(obj->net.vel, velDir);
				velFactor = Math_VecNorm(velDir);
				if (!velDir[0] && !velDir[1] && !velDir[2])
				{
					velFactor = 1.0f;
				}
				velFactor *= 0.2f;
				velFactor *= timeMod;
				Math_VecMA(obj->net.pos, velFactor, velDir, projected);
				//g_sharedFn->Common_Collide(obj->net.pos, projected, obj->net.mins, obj->net.maxs, obj->net.index, &collision);
				Phys_Collide(obj, objRad, &collision, obj->net.pos, projected, NULL, NULL);
				if (collision.hit)
				{ //smooth over intersecting planes
					float e[3];
					Math_VecCopy(collision.endPos, e);
					dp = Math_DotProduct(velDir, collision.endNormal);
					if (1)//dp < 0.0f && dp != -1.0f)// && dp > -0.99f)
					{
						Phys_ClipVel(obj->net.vel, collision.endNormal);
						Math_VecCopy(obj->net.vel, velDir);
						velFactor = Math_VecNorm(velDir);
						if (!velDir[0] && !velDir[1] && !velDir[2])
						{
							velFactor = 1.0f;
						}
						velFactor *= 0.2f;
						velFactor *= timeMod;

						float mFact = 0.25f;
						Math_VecAdd(velDir, collision.endNormal, velDir);
						velDir[2] = 0.0f;
						Math_VecNorm(velDir);
						Math_VecMA(e, velFactor*mFact, velDir, projected);
						Phys_Collide(obj, objRad, &collision, e, projected, NULL, NULL);
						if (!collision.hit)
						{
							Math_VecCopy(oldVelDir, velDir);
							Math_VecMA(collision.endPos, velFactor*mFact, velDir, projected);
							Math_VecCopy(collision.endPos, e);
							Phys_Collide(obj, objRad, &collision, e, projected, NULL, NULL);
						}
#if 1
						else if (obj->net.vel[2] != 0.0f)
						{ //hack, assume stuck in corner
							obj->net.vel[0] = 0.0f;
							obj->net.vel[1] = 0.0f;
							Math_VecCopy(obj->net.vel, velDir);
							velFactor = Math_VecNorm(velDir);
							if (!velDir[0] && !velDir[1] && !velDir[2])
							{
								velFactor = 1.0f;
							}
							velFactor *= 0.2f;
							velFactor *= timeMod;

							Math_VecMA(e, velFactor*mFact, velDir, projected);
							Phys_Collide(obj, objRad, &collision, e, projected, NULL, NULL);
						}
#endif
					}
				}
				//Util_DebugFX(collision.endPos, velDir);
			}
		}
	}

	if (obj->onGround && collision.hit && !collision.containsSolid && gravFactor != 0.0f)
	{ //try stair-climbing
		//float dp = Math_DotProduct(groundPl, collision.endNormal);
		//if (dp < groundDot || dp > -groundDot)
		{
			collObj_t stair;
			float up[3];
			float stairHop = 48.0f;//32.0f;
			Math_VecCopy(collision.endPos, up);
			up[2] += stairHop;
			Phys_Collide(obj, objRad, &stair, obj->net.pos, up, NULL, NULL);
			if (!stair.hit)
			{
				float stepFwdLen = velFactor-collision.distance;
				if (stepFwdLen < 0.1f)
				{
					stepFwdLen = 0.1f;
				}
				Math_VecCopy(obj->net.vel, velDir);
				velDir[2] = 0.0f;
				Math_VecNorm(velDir);
				float n[3];
				n[0] = up[0] + (velDir[0]*stepFwdLen);
				n[1] = up[1] + (velDir[1]*stepFwdLen);
				n[2] = up[2];
				Phys_Collide(obj, objRad, &stair, up, n, NULL, NULL);
				while (stair.hit && stepFwdLen > 16.0f)
				{
					stepFwdLen *= 0.5f;
					n[0] = up[0] + (velDir[0]*stepFwdLen);
					n[1] = up[1] + (velDir[1]*stepFwdLen);
					n[2] = up[2];
					Phys_Collide(obj, objRad, &stair, up, n, NULL, NULL);
				}
				if (!stair.hit)
				{
					up[0] = n[0];
					up[1] = n[1];
					n[2] -= (stairHop+1.0f);
					Phys_Collide(obj, objRad, &stair, up, n, NULL, NULL);
					if (stair.hit && !stair.containsSolid)
					{
						Math_VecCopy(stair.endPos, obj->net.pos);
#ifdef _DEBUG_INSOLID
						Phys_Collide(obj, objRad, &testCol, obj->net.pos, obj->net.pos, NULL, NULL);
						RCUBE_ASSERT(!testCol.containsSolid);
#endif
						Math_VecMA(obj->net.pos, velFactor, velDir, projected);
						Phys_Collide(obj, objRad, &collision, obj->net.pos, projected, NULL, NULL);
					}
				}
			}
		}
	}

	const float maxVel = 8192.0f;//4096.0f;
	for (int i = 0; i < 3; i++)
	{
		if (obj->net.vel[i] > maxVel)
		{
			obj->net.vel[i] = maxVel;
		}
		else if (obj->net.vel[i] < -maxVel)
		{
			obj->net.vel[i] = -maxVel;
		}
	}

	if (!collision.containsSolid)
	{ //i pulled these numbers out of my ass.
		Math_VecCopy(collision.endPos, obj->net.pos);
#ifdef _DEBUG_INSOLID
		Phys_Collide(obj, objRad, &testCol, obj->net.pos, obj->net.pos, NULL, NULL);
		RCUBE_ASSERT(!testCol.containsSolid);
#endif
		Phys_ApplyFriction(obj, timeMod, xyFriction, zFriction);

		for (int k = 0; k < 3; k++)
		{
			if (fabsf(obj->net.vel[k]) < 0.0001f)
			{
				obj->net.vel[k] = 0.0f;
			}
		}
	}
	else if (g_checkForSolid)
	{ //oh no!
		//obj->net.pos[2] += 200.0f;
		Math_VecCopy(oldPos, obj->net.pos);
	}

	if (obj->net.pos[0] != oldPos[0] ||
		obj->net.pos[1] != oldPos[1] ||
		obj->net.pos[2] != oldPos[2])
	{ //update the clip model
		Phys_Collide(obj, objRad, &collision, obj->net.pos, obj->net.pos, NULL, NULL);
		if (collision.containsSolid)
		{
			Math_VecCopy(oldPos, obj->net.pos);
		}
		LServ_UpdateRClip(obj);
	}
}

//linear movement, returns true if at destPos
bool Phys_LinearMove(gameObject_t *obj, float *destPos, float moveSpeed)
{
	bool atGoal = false;
	float d[3];
	Math_VecSub(destPos, obj->net.pos, d);
	float dist = Math_VecNorm(d);
	if (moveSpeed >= dist)
	{
		moveSpeed = dist;
		atGoal = true;
	}
	obj->net.pos[0] += d[0]*moveSpeed;
	obj->net.pos[1] += d[1]*moveSpeed;
	obj->net.pos[2] += d[2]*moveSpeed;

	return atGoal;
}

//just apply velocity to position
void Phys_DirectVelocity(gameObject_t *obj, float timeMod, float xyFriction, float zFriction, float velMul)
{
	float velDir[3];
	Math_VecCopy(obj->net.vel, velDir);
	float velFactor = Math_VecNorm(velDir);
	float velLen = velFactor;
	if (!velDir[0] && !velDir[1] && !velDir[2])
	{
		velFactor = 1.0f;
	}
	velFactor *= velMul;
	//velFactor *= timeMod;
	if (velFactor > velLen)
	{
		velFactor = velLen;
	}
	Math_VecMA(obj->net.pos, velFactor, velDir, obj->net.pos);

	obj->net.vel[0] = 0.0f;
	obj->net.vel[1] = 0.0f;
	obj->net.vel[2] = 0.0f;
	/*
	//apply friction
	float fxy = Math_Min2((xyFriction*timeMod), 1.0f);
	float fz = Math_Min2((zFriction*timeMod), 1.0f);
	obj->net.vel[0] *= 1.0f-fxy;
	obj->net.vel[1] *= 1.0f-fxy;
	obj->net.vel[2] *= 1.0f-fz;
	for (int k = 0; k < 3; k++)
	{
		if (fabsf(obj->net.vel[k]) < 0.0001f)
		{
			obj->net.vel[k] = 0.0f;
		}
	}
	*/
}

//put it on the ground
void Phys_PutOnGround(gameObject_t *obj)
{
	collObj_t col;
	float ground[3];
	Math_VecCopy(obj->net.pos, ground);
	ground[2] -= 16384.0f;
	Phys_Collide(obj, obj->radius, &col, obj->net.pos, ground, NULL, NULL);
	if (col.hit && !col.containsSolid)
	{
		Math_VecCopy(col.endPos, obj->net.pos);
		LServ_UpdateRClip(obj);

		float groundPl[3] = {0.0f, 0.0f, 1.0f};
		float groundDot = 0.4f;
		float dp = Math_DotProduct(groundPl, col.endNormal);
		if (dp >= groundDot)
		{
			obj->onGround = true;
			obj->groundObj = col.hitObjectIndex;
		}
	}
}
