/*
=============================================================================
Module Information
------------------
Name:			physics_rigid.cpp
Author:			Rich Whitehouse
Description:	rigid body physics routines
=============================================================================
*/

#include "main.h"

#define MAX_RIGID_PROPS		512
static rigidPropObj_t *g_rigidProps[MAX_RIGID_PROPS];
static int g_numRigidProps = 0;

//load rigid body primitive data
void Phys_InitRigidBodies(void)
{
	textParser_t *parser = g_sharedFn->Parse_InitParserFromFile("assets/objects/physdefs.rbp");
	if (!parser)
	{
		return;
	}

	char lastGroup[256];
	lastGroup[0] = 0;

	g_numRigidProps = 0;

	parseToken_t tok;
	rigidPropObj_t *rp = NULL;
	float parseScale = 1.0f;
	int parsePrim = 0;
	int lastPrimGroup = -1;
	while (g_sharedFn->Parse_GetNextToken(parser, &tok))
	{
		if (tok.groupName && tok.groupName[0])
		{
			if (!stricmp(tok.groupName, "prim"))
			{
				if (lastPrimGroup != tok.groupNum)
				{
					parsePrim++;
					lastPrimGroup = tok.groupNum;
				}
				RCUBE_ASSERT(rp && parsePrim > 0 && parsePrim <= rp->numPrims);
				rigidPrim_t *prim = rp->prims+(parsePrim-1);
				if (!stricmp(tok.text, "type"))
				{
					g_sharedFn->Parse_GetNextToken(parser, &tok);
					if (!stricmp(tok.text, "sphere"))
					{
						prim->type = RIGPRIM_SPHERE;
					}
					else if (!stricmp(tok.text, "stitchpt"))
					{
						prim->type = RIGPRIM_STITCHPT;
					}
				}
				else if (!stricmp(tok.text, "pos"))
				{
					g_sharedFn->Parse_GetNextToken(parser, &tok);
					sscanf(tok.text, "(%f %f %f)", &prim->pos[0], &prim->pos[1], &prim->pos[2]);
					Math_VecScale(prim->pos, parseScale);
				}
				else if (!stricmp(tok.text, "radius"))
				{
					g_sharedFn->Parse_GetNextToken(parser, &tok);
					prim->radius = (float)atof(tok.text) * parseScale;
				}
			}
			else
			{
				if (strcmp(tok.groupName, lastGroup))
				{
					strcpy(lastGroup, tok.groupName);
					g_rigidProps[g_numRigidProps] = (rigidPropObj_t *)g_sharedFn->Common_RCMalloc(sizeof(rigidPropObj_t));
					memset(g_rigidProps[g_numRigidProps], 0, sizeof(rigidPropObj_t));
					rp = g_rigidProps[g_numRigidProps];
					strcpy(rp->name, tok.groupName);
					g_numRigidProps++;
					parseScale = 1.0f;
					parsePrim = 0;
				}

				if (!stricmp(tok.text, "scale"))
				{
					RCUBE_ASSERT(rp);
					g_sharedFn->Parse_GetNextToken(parser, &tok);
					parseScale = (float)atof(tok.text);
				}
				else if (!stricmp(tok.text, "centerOfMass"))
				{
					RCUBE_ASSERT(rp);
					g_sharedFn->Parse_GetNextToken(parser, &tok);
					sscanf(tok.text, "(%f %f %f)", &rp->centerOfMass[0], &rp->centerOfMass[1], &rp->centerOfMass[2]);
					Math_VecScale(rp->centerOfMass, parseScale);
				}
				else if (!stricmp(tok.text, "numPrims"))
				{
					RCUBE_ASSERT(rp);
					g_sharedFn->Parse_GetNextToken(parser, &tok);
					rp->numPrims = atoi(tok.text);
					rp->prims = (rigidPrim_t *)g_sharedFn->Common_RCMalloc(sizeof(rigidPrim_t)*rp->numPrims);
					memset(rp->prims, 0, sizeof(rigidPrim_t)*rp->numPrims);
				}
			}
		}
	}

	g_sharedFn->Parse_FreeParser(parser);
}

//free rigid body primitive data
void Phys_ShutdownRigidBodies(void)
{
	for (int i = 0; i < g_numRigidProps; i++)
	{
		if (g_rigidProps[i])
		{
			if (g_rigidProps[i]->prims)
			{
				g_sharedFn->Common_RCFree(g_rigidProps[i]->prims);
			}
			g_sharedFn->Common_RCFree(g_rigidProps[i]);
		}
	}
	g_numRigidProps = 0;
}

//load rigid body property data
rigidPropObj_t *Phys_LoadRigidBody(const char *rbName)
{
	if (!rbName || !rbName[0])
	{
		return NULL;
	}

	for (int i = 0; i < g_numRigidProps; i++)
	{
		if (!strcmp(g_rigidProps[i]->name, rbName))
		{
			return g_rigidProps[i];
		}
	}
	return NULL;
}

//apply torque
void Phys_ApplyTorque(gameObject_t *obj, float *pt, float *vel, float torqueDamp)
{
	RCUBE_ASSERT(obj && obj->rigid && obj->rigid->rp);
	rigidPropObj_t *rp = obj->rigid->rp;

	float cnt[3];
	Math_TransformPointByMatrix(&obj->rigid->mat, rp->centerOfMass, cnt);
	Math_VecAdd(cnt, obj->net.pos, cnt);

	float lpt[3];
	if (!pt)
	{
		Math_VecCopy(cnt, lpt);
	}
	else
	{
		Math_VecCopy(pt, lpt);
	}

	float lvel[3];
	Math_TransformPointByMatrix(&obj->rigid->invMat, vel, lvel);

	float d[3];
	float ld[3];
	Math_VecSub(lpt, cnt, d);
	Math_TransformPointByMatrix(&obj->rigid->invMat, d, ld);

	float ltor[3];
	Math_CrossProduct(ld, lvel, ltor);

	Math_VecScale(ltor, torqueDamp);
	Math_VecAdd(obj->rigid->torque, ltor, obj->rigid->torque);
}

//apply torque and velocity based on a point and direction
void Phys_ShoveRigidLocal(gameObject_t *obj, float *pt, float *vel)
{
	if (!obj || !obj->rigid || !obj->rigid->rp)
	{
		return;
	}
	Math_VecAdd(obj->net.vel, vel, obj->net.vel);
	Phys_ApplyTorque(obj, pt, vel, 0.00015f);
}

//called externally
void Phys_ShoveRigid(gameObject_t *obj, float *pt, float *vel)
{
	if (!obj || !obj->rigid || !obj->rigid->rp)
	{
		return;
	}
	Math_VecAdd(obj->net.vel, vel, obj->net.vel);
	Phys_ApplyTorque(obj, pt, vel, 0.00005f);
}

//register touch
void Phys_RigidTouch(gameObject_t *obj, collObj_t *col)
{
	if (!obj || !col || !col->hit || col->hitObjectIndex < 0)
	{
		return;
	}
	gameObject_t *other = &g_gameObjects[col->hitObjectIndex];
	if (!other->inuse)
	{
		return;
	}

	if (obj->touch)
	{
		obj->touch(obj, other, col);
	}
	if (other->touch)
	{
		other->touch(other, obj, col);
	}
	if (other->rigid)
	{
		float v[3];
		v[0] = obj->net.vel[0]*0.1f;
		v[1] = obj->net.vel[1]*0.1f;
		v[2] = obj->net.vel[2]*0.1f;
		Phys_ShoveRigid(other, col->endPos, v);
	}
}

//attempt to move to a new transform with collision checks
void Phys_RigidSweep(gameObject_t *obj, modelMatrix_t *newTestMat, float *newTestPos, bool rotating)
{
	RCUBE_ASSERT(obj && obj->rigid && obj->rigid->rp);

	if (!memcmp(&obj->rigid->mat, newTestMat, sizeof(modelMatrix_t)) &&
		!memcmp(obj->net.pos, newTestPos, sizeof(obj->net.pos)))
	{ //no movement
		return;
	}

	rigidPropObj_t *rp = obj->rigid->rp;
	float *oldPosSet;
	float *newPosSet;
	oldPosSet = (float *)_alloca(sizeof(float)*3*rp->numPrims);
	newPosSet = (float *)_alloca(sizeof(float)*3*rp->numPrims);

	//run through and transform all primitives for quick reference later
	for (int i = 0; i < rp->numPrims; i++)
	{
		rigidPrim_t *prim = rp->prims+i;
		float *oldPos = oldPosSet+i*3;
		float *newPos = newPosSet+i*3;
		if (rotating)
		{
			Math_TransformPointByMatrix(&obj->rigid->mat, prim->pos, oldPos);
			Math_VecAdd(oldPos, obj->net.pos, oldPos);
			Math_TransformPointByMatrix(newTestMat, prim->pos, newPos);
			Math_VecAdd(newPos, newTestPos, newPos);
		}
		else
		{
			Math_TransformPointByMatrix(&obj->rigid->mat, prim->pos, oldPos);
			Math_VecCopy(oldPos, newPos);
			Math_VecAdd(oldPos, obj->net.pos, oldPos);
			Math_VecAdd(newPos, newTestPos, newPos);
		}
	}

	float resPos[3];
	modelMatrix_t resMat;
	Math_VecCopy(newTestPos, resPos);
	resMat = *newTestMat;
	float mainD[3];
	float mainL;
	Math_VecSub(newTestPos, obj->net.pos, mainD);
	mainL = Math_VecLen(mainD);
	float closestDist = mainL;
	Math_VecNorm(mainD);
	bool impact = false;
	float impactNormal[3];
	float impactPos[3];
	float impactDir[3];

	//do actual collision tests
	for (int i = 0; i < rp->numPrims; i++)
	{
		rigidPrim_t *prim = rp->prims+i;
		float *oldPos = oldPosSet+i*3;
		float *newPos = newPosSet+i*3;
		collObj_t col;
		g_sharedFn->Coll_RadiusTranslation(obj->rcColModel, &col, oldPos, newPos, prim->radius, NULL, NULL);
		if (col.hit || col.containsSolid)
		{
			float d[3];
			Math_VecSub(oldPos, col.endPos, d);
			float l = Math_VecLen(d);
			Phys_RigidTouch(obj, &col);
			if (rotating)
			{
				resMat = obj->rigid->mat;
				if (!col.containsSolid)
				{
					Math_VecCopy(col.endNormal, impactNormal);
					Math_VecCopy(col.endPos, impactPos);
					Math_VecSub(oldPos, newPos, impactDir);
					float dp = Math_DotProduct(obj->net.vel, impactNormal);
					float l = 80.0f;//Math_VecLen(obj->net.vel);
					if (dp < 0.0f)
					{
						float push[3];
						push[0] = -dp*impactNormal[0];
						push[1] = -dp*impactNormal[1];
						push[2] = -dp*impactNormal[2];
						Phys_ShoveRigidLocal(obj, impactPos, push);
					}
					Math_VecNorm(impactNormal);
					Math_VecScale(obj->net.vel, 0.5f);
					Math_VecScale(impactNormal, l*0.5f);
					Phys_ShoveRigidLocal(obj, impactPos, impactNormal);
				}
			}
			else
			{
				if (l < closestDist)
				{
					if (!col.containsSolid)
					{
						resPos[0] = obj->net.pos[0] + mainD[0]*l;
						resPos[1] = obj->net.pos[1] + mainD[1]*l;
						resPos[2] = obj->net.pos[2] + mainD[2]*l;
						closestDist = l;
					}
					else
					{
						resPos[0] = obj->net.pos[0];
						resPos[1] = obj->net.pos[1];
						resPos[2] = obj->net.pos[2];
						closestDist = 0.0f;
					}
					if (!col.containsSolid)
					{
						Math_VecCopy(col.endNormal, impactNormal);
						//Math_VecNorm(d);
						//Math_VecAdd(impactNormal, d, impactNormal);
						//Math_VecNorm(impactNormal);
						Math_VecCopy(col.endPos, impactPos);
						Math_VecSub(oldPos, newPos, impactDir);
						impact = true;

						float dp = Math_DotProduct(obj->net.vel, impactNormal);
						float l = Math_VecLen(obj->net.vel);
						if (dp < 0.0f)
						{
							//obj->net.vel[0] -= dp*impactNormal[0];
							//obj->net.vel[1] -= dp*impactNormal[1];
							//obj->net.vel[2] -= dp*impactNormal[2];
							float push[3];
							push[0] = -dp*impactNormal[0];
							push[1] = -dp*impactNormal[1];
							push[2] = -dp*impactNormal[2];
							Phys_ShoveRigidLocal(obj, impactPos, push);
						}
						Math_VecNorm(impactNormal);
						Math_VecScale(obj->net.vel, 0.5f);
						Math_VecScale(impactNormal, l*0.5f);
						//obj->rigid->torque[0] = 0.0f;
						//obj->rigid->torque[1] = 0.0f;
						//obj->rigid->torque[2] = 0.0f;
						//Math_VecScale(impactNormal, 5.0f);
						Phys_ShoveRigidLocal(obj, impactPos, impactNormal);
						/*
						float gvec[3];
						Phys_GetRigidGravAx(obj, gvec);
						Math_VecScale(gvec, 80.0f);
						Phys_ApplyTorque(obj, impactPos, gvec);
						*/
						//Math_VecAdd(obj->net.vel, impactNormal, obj->net.vel);
						//Math_VecSub(oldPos, newPos, d);
						//Math_VecScale(d, 20.0f);
						//Phys_ApplyTorque(obj, impactPos, d);
					}
				}
			}
		}
	}

	if (!rotating)
	{
		Math_VecCopy(resPos, obj->net.pos);
	}
	obj->rigid->mat = resMat;
}

//do ground check
void Phys_RigidGroundCheck(gameObject_t *obj, float *lowestPt)
{
	RCUBE_ASSERT(obj && obj->rigid && obj->rigid->rp);
	obj->onGround = false;

	rigidPropObj_t *rp = obj->rigid->rp;

	float grDist = 16.0f;
	//run through and transform all primitives for quick reference later
	for (int i = 0; i < rp->numPrims; i++)
	{
		rigidPrim_t *prim = rp->prims+i;
		float tpos[3];
		Math_TransformPointByMatrix(&obj->rigid->mat, prim->pos, tpos);
		Math_VecAdd(tpos, obj->net.pos, tpos);
		float dpos[3];
		dpos[0] = tpos[0];
		dpos[1] = tpos[1];
		dpos[2] = tpos[2]-grDist;
		collObj_t col;
		g_sharedFn->Coll_RadiusTranslation(obj->rcColModel, &col, tpos, dpos, prim->radius, NULL, NULL);
		if (col.hit || col.containsSolid)
		{
			obj->onGround = true;
		}
		if (lowestPt)
		{
			if (i == 0 || tpos[2] < lowestPt[2])
			{
				lowestPt[0] = tpos[0];
				lowestPt[1] = tpos[1];
				lowestPt[2] = tpos[2];
			}
		}
	}
}

//get axis for rigid body closest to gravity vector
void Phys_GetRigidGravAx(gameObject_t *obj, float *axOut)
{
	float gravVec[3] = {0.0f, 0.0f, 1.0f};
	float ax[3][3];
	Math_TransformPointByMatrix(&obj->rigid->mat, g_gameIdent.x1, ax[0]);
	Math_TransformPointByMatrix(&obj->rigid->mat, g_gameIdent.x2, ax[1]);
	Math_TransformPointByMatrix(&obj->rigid->mat, g_gameIdent.x3, ax[2]);
	float closeDot = Math_DotProduct(gravVec, ax[0]);
	int idx = 0;
	for (int i = 1; i < 3; i++)
	{
		float dp = Math_DotProduct(gravVec, ax[i]);
		if (fabsf(dp) > fabsf(closeDot))
		{
			closeDot = dp;
			idx = i;
		}
	}

	if (closeDot < 0.0f)
	{
		axOut[0] = -ax[idx][0];
		axOut[1] = -ax[idx][1];
		axOut[2] = -ax[idx][2];
	}
	else
	{
		axOut[0] = ax[idx][0];
		axOut[1] = ax[idx][1];
		axOut[2] = ax[idx][2];
	}
}

//stitch check
bool Phys_CheckRigidStitches(gameObject_t *obj, modelMatrix_t *newTestMat)
{
	RCUBE_ASSERT(obj && obj->rigid && obj->rigid->rp);

	rigidPropObj_t *rp = obj->rigid->rp;
	float *oldPosSet;
	float *newPosSet;
	oldPosSet = (float *)_alloca(sizeof(float)*3*rp->numPrims);
	newPosSet = (float *)_alloca(sizeof(float)*3*rp->numPrims);

	//run through and transform all primitives for quick reference later
	for (int i = 0; i < rp->numPrims; i++)
	{
		rigidPrim_t *prim = rp->prims+i;
		float *oldPos = oldPosSet+i*3;
		float *newPos = newPosSet+i*3;
		Math_TransformPointByMatrix(&obj->rigid->mat, prim->pos, oldPos);
		Math_VecAdd(oldPos, obj->net.pos, oldPos);
		Math_TransformPointByMatrix(newTestMat, prim->pos, newPos);
		Math_VecAdd(newPos, obj->net.pos, newPos);
	}

	bool anyImpact = false;
	bool wasSolid = false;
	float peneDist = 0.0f;
	float penePt[3];
	float peneNormal[3];
	float peneDif[3];
	int peneIdx = -1;
	for (int i = 0; i < rp->numPrims; i++)
	{
		rigidPrim_t *prim = rp->prims+i;
		if (prim->type != RIGPRIM_STITCHPT)
		{
			continue;
		}
		for (int j = 0; j < rp->numPrims; j++)
		{
			rigidPrim_t *prim2 = rp->prims+j;
			if (i == j || prim2->type != RIGPRIM_STITCHPT)
			{
				continue;
			}

			float *p1 = newPosSet+i*3;
			float *p2 = newPosSet+j*3;
			collObj_t col;
			g_sharedFn->Coll_RadiusTranslation(obj->rcColModel, &col, p1, p2, prim->radius, NULL, NULL);
			if (col.hit || col.containsSolid)
			{
				float d[3];
				Math_VecSub(col.endPos, p2, d);
				float l = Math_VecLen(d);
				if (!anyImpact || l > peneDist)
				{
					if (col.containsSolid)
					{
						wasSolid = true;
					}
					else
					{
						Math_VecCopy(col.endNormal, peneNormal);
						Phys_RigidTouch(obj, &col);
					}
					float *p2Old = oldPosSet+j*3;
					Math_VecCopy(p2, penePt);
					peneIdx = j;
					peneDist = l;
					Math_VecSub(p2, p2Old, peneDif);
					anyImpact = true;
				}
			}
		}
	}

	if (anyImpact && !wasSolid && peneDist > 20.0f)
	{
		float dp = Math_DotProduct(obj->net.vel, peneNormal);
		float push[3];
		if (dp < 0.0f)
		{
			push[0] = -dp*peneNormal[0];
			push[1] = -dp*peneNormal[1];
			push[2] = -dp*peneNormal[2];
			Phys_ShoveRigidLocal(obj, penePt, push);
		}
		Phys_GetRigidGravAx(obj, peneNormal);
		push[0] = peneNormal[0]*peneDist;
		push[1] = peneNormal[1]*peneDist;
		push[2] = peneNormal[2]*peneDist;
		Phys_ShoveRigidLocal(obj, penePt, push);
	}

	/*
	if (anyImpact && !wasSolid)
	{
		//obj->net.vel[0] += peneNormal[0]*peneDist;
		//obj->net.vel[1] += peneNormal[1]*peneDist;
		//obj->net.vel[2] += peneNormal[2]*peneDist;
		//Math_VecScale(peneNormal, Math_VecLen(obj->net.vel)*1.0f);
#if 0
		float push[3];
		push[0] = -peneDif[0];
		push[1] = -peneDif[1];
		push[2] = -peneDif[2];
#else
		float dp = Math_DotProduct(obj->net.vel, peneNormal);
		if (dp < 0.0f)
		{
			obj->net.vel[0] -= dp*peneNormal[0];
			obj->net.vel[1] -= dp*peneNormal[1];
			obj->net.vel[2] -= dp*peneNormal[2];
		}
		float push[3];
		push[0] = peneNormal[0];
		push[1] = peneNormal[1];
		push[2] = peneNormal[2];
//		Phys_GetRigidGravAx(obj, push);
//		push[0] += peneNormal[0];
//		push[1] += peneNormal[1];
//		push[2] += peneNormal[2];
		Math_VecNorm(push);
		float pl = Math_VecLen(peneDif);
		if (pl < peneDist)
		{
			pl = peneDist;
		}
		if (pl < 2.0f)
		{
			pl = 2.0f;
		}
		Math_VecScale(push, pl);
#endif

//		obj->rigid->torque[0] = 0.0f;
//		obj->rigid->torque[1] = 0.0f;
//		obj->rigid->torque[2] = 0.0f;
		//Math_VecScale(push, 5.0f);
//		Phys_ShoveRigidLocal(obj, penePt, push);
		//Phys_ApplyTorque(obj, penePt, push);
		obj->net.vel[0] += peneNormal[0]*pl;
		obj->net.vel[1] += peneNormal[1]*pl;
		obj->net.vel[2] += peneNormal[2]*pl;
		//Math_VecScale(obj->rigid->torque, 0.75f);
	}
	*/

	return !anyImpact;
}

//apply rigid-body physics
void Phys_RunRigid(gameObject_t *obj, float timeMod, float gravFactor)
{
	if (!obj || !obj->rigid || !obj->rigid->rp)
	{
		RCUBE_ASSERT(0);
		return;
	}

	modelMatrix_t rigMat = obj->rigid->mat;
	float newPos[3];
	Math_VecCopy(obj->net.pos, newPos);

	float lowestPt[3];
	Phys_RigidGroundCheck(obj, lowestPt);
	if (gravFactor)// && !obj->onGround)
	{ //apply gravity to velocity
		obj->net.vel[2] -= gravFactor*timeMod;
	}

	if (obj->rigid->torque[0] ||
		obj->rigid->torque[1] ||
		obj->rigid->torque[2])
	{ //apply torque
		float rotAng[3];
		float dampRot = 2.0f*timeMod;
		float torqueLocal[3];
		//Math_TransformPointByMatrix3x3(&obj->rigid->invMat, obj->rigid->torque, torqueLocal);
		Math_VecCopy(obj->rigid->torque, torqueLocal);
		rotAng[0] = torqueLocal[0];
		rotAng[1] = torqueLocal[1];
		rotAng[2] = torqueLocal[2];
		Math_RotateMatrix(&rigMat, 1.0f, 0.0f, 0.0f, -rotAng[2]*dampRot);
		Math_RotateMatrix(&rigMat, 1.0f, 0.0f, rotAng[1]*dampRot, 0.0f);
		Math_RotateMatrix(&rigMat, 1.0f, -rotAng[0]*dampRot, 0.0f, 0.0f);
		float torqueDec = 0.5f*timeMod;
		float l = Math_VecNorm(obj->rigid->torque);
		if (l > 30.0f)
		{
			l = 30.0f;
		}
		if (l > 0.0f)
		{
			l -= torqueDec;
			if (l < 0.0f)
			{
				l = 0.0f;
			}
		}
		Math_VecScale(obj->rigid->torque, l);
	}

	modelMatrix_t oldMat = obj->rigid->mat;

	Phys_RigidSweep(obj, &rigMat, obj->net.pos, true);
	if (!Phys_CheckRigidStitches(obj, &obj->rigid->mat))
	{ //failed rotation
		obj->rigid->mat = oldMat;
	}

	if (obj->net.vel[0] ||
		obj->net.vel[1] ||
		obj->net.vel[2])
	{ //apply velocity
		float l = Math_VecNorm(obj->net.vel);
		float velFactor = l*0.2f;
		velFactor *= timeMod;
		Math_VecMA(obj->net.pos, velFactor, obj->net.vel, newPos);
		float velDec = 10.0f*timeMod;
		if (l > 0.0f)
		{
			l -= velDec;
			if (l < 0.0f)
			{
				l = 0.0f;
			}
		}
		Math_VecScale(obj->net.vel, l);
	}

	Phys_RigidSweep(obj, &obj->rigid->mat, newPos, false);

	if (g_glb.debugRigid)
	{ //draw primitives
		rigidPropObj_t *rp = obj->rigid->rp;
		for (int i = 0; i < rp->numPrims; i++)
		{
			rigidPrim_t *prim = rp->prims+i;
			if (prim->type == RIGPRIM_SPHERE)
			{
				float pos[3];
				Math_TransformPointByMatrix(&obj->rigid->mat, prim->pos, pos);
				Math_VecAdd(pos, obj->net.pos, pos);
				Util_DebugSphere(pos, pos, prim->radius);
			}
			else if (prim->type == RIGPRIM_STITCHPT)
			{
				float pos[3];
				Math_TransformPointByMatrix(&obj->rigid->mat, prim->pos, pos);
				Math_VecAdd(pos, obj->net.pos, pos);

				for (int j = 0; j < rp->numPrims; j++)
				{
					if (j == i)
					{
						continue;
					}
					rigidPrim_t *p2 = rp->prims+j;
					if (p2->type != RIGPRIM_STITCHPT)
					{
						continue;
					}

					float otherPos[3];
					Math_TransformPointByMatrix(&obj->rigid->mat, p2->pos, otherPos);
					Math_VecAdd(otherPos, obj->net.pos, otherPos);
					Util_DebugLine(pos, otherPos);
				}
			}
		}
	}

	Math_MatrixInverse(&obj->rigid->mat, &obj->rigid->invMat);
}
