/*
=============================================================================
Module Information
------------------
Name:			rscript_han.cpp
Author:			Rich Whitehouse
Description:	R-Script node handlers
=============================================================================
*/

#include "main.h"
#include "ai.h"
#include "rscript.h"

//get argument string
char *RScr_GetArgStr(rscriptNode_t *node, rscript_t *rscript, int argIdx)
{
	char *str = node->menuArgs[argIdx].p;
	if (!strncmp(str, "#stack", 6))
	{
		int stackStrNum = rscript->ropStackPtr+1+atoi(str+6);
		if (stackStrNum < rscript->ropStackPtr || stackStrNum >= MAX_RSCRIPT_STACK)
		{
			Util_ErrorMessage("%s: attempt to access stack out of range: %i", node->nodeHandler->name, stackStrNum);
			return "";
		}
		return rscript->ropStack[stackStrNum];
	}
	else if (!strncmp(str, "$$", 2))
	{ //gvar
		return GVar_GetValue(str+2);
	}
	return str;
}

//"delay" command
bool RSHan_Delay(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	rscript->scriptTime = g_glb.gameTime + atoi(RScr_GetArgStr(node, rscript, 0));
	return true;
}

//"delayrand" command
bool RSHan_DelayRand(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	int minDelay = atoi(RScr_GetArgStr(node, rscript, 0));
	int maxDelay = atoi(RScr_GetArgStr(node, rscript, 1));
	rscript->scriptTime = g_glb.gameTime + minDelay + rand()%(maxDelay-minDelay);
	return true;
}

//"cacheres" command
bool RSHan_CacheRes(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	g_sharedFn->Common_ServerString(RScr_GetArgStr(node, rscript, 0));
	return true;
}

//"cacheobj" command
bool RSHan_CacheObj(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	LServ_CacheObj(RScr_GetArgStr(node, rscript, 0));
	return true;
}

//"runscript" command
bool RSHan_RunScript(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	Util_RunScript(RScr_GetArgStr(node, rscript, 0), RScr_GetArgStr(node, rscript, 1));
	return true;
}

//"changemap" command
bool RSHan_ChangeMap(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	char *mapName = RScr_GetArgStr(node, rscript, 0);
	strcpy(g_glb.changeToMap, mapName);
	g_glb.scheduledMapChange = true;
	g_sharedFn->Common_MapChange(mapName, 1000);
	return true;
}

//"waitonscript" command
bool RSHan_WaitOnScript(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *obj = Util_FindTargetObject(RScr_GetArgStr(node, rscript, 0));
	if (!obj || !obj->inuse || !obj->rscript)
	{
		return true;
	}

	return false;
}

//"waitonframe" command
bool RSHan_WaitOnFrame(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *obj = Util_FindTargetObject(RScr_GetArgStr(node, rscript, 0));
	if (!obj || !obj->inuse)
	{
		return true;
	}

	if (obj->net.frame == atoi(RScr_GetArgStr(node, rscript, 1)))
	{
		return true;
	}

	return false;
}

//"waitoninput" command
bool RSHan_WaitOnInput(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *obj = Util_FindTargetObject("__the_player");
	if (!obj || !obj->inuse || !obj->plObj)
	{
		return true;
	}

	if (obj->plObj->waitOnInput == -1)
	{
		obj->plObj->waitOnInput = 0;
		return true;
	}

	obj->plObj->waitOnInput = atoi(RScr_GetArgStr(node, rscript, 0));
	if (obj->plObj->waitOnInput < 1)
	{
		obj->plObj->waitOnInput = 1;
	}
	else if (obj->plObj->waitOnInput > MAX_BUTTONS)
	{
		obj->plObj->waitOnInput = MAX_BUTTONS;
	}
	return false;
}

//"waitonenemiesdead" command
bool RSHan_WaitOnEnemiesDead(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	for (int i = 0; i < g_glb.gameObjectSlots; i++)
	{
		gameObject_t *obj = &g_gameObjects[i];
		if (obj->inuse && obj->aiObj && obj->net.aiDescIndex == AIDESC_SEPHFIN &&
			obj->health > 0)
		{
			return false;
		}
		else if (obj->inuse && obj->hurtable && (obj->localFlags & LFL_ENEMY))
		{
			if (!obj->aiObj || !obj->aiObj->nonActive)
			{
				return false;
			}
		}
	}
	return true;
}

//"waitondyingemies" command
bool RSHan_WaitOnDyingEnemies(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	for (int i = 0; i < g_glb.gameObjectSlots; i++)
	{
		gameObject_t *obj = &g_gameObjects[i];
		if (obj->inuse && (obj->localFlags & LFL_ENEMY))
		{
			if (!obj->aiObj || !obj->aiObj->nonActive)
			{
				if (obj->hurtable ||
					(!(obj->net.renderEffects & FXFL_DEATH) &&
					 !(obj->net.renderEffects2 & FXFL2_DEATH2) &&
					 !(obj->net.renderEffects2 & FXFL2_DEATH3)))
				{
					if (obj->net.aiDescIndex != AIDESC_JENOVA)
					{
						return false;
					}
				}
			}
		}
	}
	return true;
}

//"waitonenemiesnum" command
bool RSHan_WaitOnEnemiesNum(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	int maxNum = atoi(RScr_GetArgStr(node, rscript, 0));
	int num = 0;
	for (int i = 0; i < g_glb.gameObjectSlots; i++)
	{
		gameObject_t *obj = &g_gameObjects[i];
		if (obj->inuse && obj->aiObj && obj->net.aiDescIndex == AIDESC_SEPHFIN &&
			obj->health > 0)
		{
			num++;
		}
		else if (obj->inuse && obj->hurtable && (obj->localFlags & LFL_ENEMY))
		{
			if (!obj->aiObj || !obj->aiObj->nonActive)
			{
				num++;
			}
		}
	}

	if (num > maxNum)
	{
		return false;
	}
	return true;
}

//"waitondead" command
bool RSHan_WaitOnDead(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *pl = &g_gameObjects[0];
	if (pl->inuse && pl->health <= 0)
	{
		return false;
	}
	return true;
}

//"killenemies" command
bool RSHan_KillEnemies(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *pl = &g_gameObjects[0];
	for (int i = 0; i < g_glb.gameObjectSlots; i++)
	{
		gameObject_t *obj = &g_gameObjects[i];
		if (obj->inuse && obj->hurtable && (obj->localFlags & LFL_ENEMY))
		{
			if (obj->aiObj && obj->aiObj->nonActive)
			{
				continue;
			}
			Util_DamageObject(pl, obj, 99999999, 0);
		}
	}
	return true;
}

//"killenemiespassive" command
bool RSHan_KillEnemiesPassive(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *killer = &g_gameObjects[MAP_OBJ_NUM];
	for (int i = 0; i < g_glb.gameObjectSlots; i++)
	{
		gameObject_t *obj = &g_gameObjects[i];
		if (obj->inuse && obj->hurtable && (obj->localFlags & LFL_ENEMY))
		{
			if (obj->aiObj && obj->aiObj->nonActive)
			{
				continue;
			}
			Util_DamageObject(killer, obj, 99999999, 0);
			obj->makoCount = 0;
			obj->net.renderEffects2 |= FXFL2_DEATH3;
			if (obj->aiObj)
			{
				memset(obj->aiObj->dropChances, 0, sizeof(obj->aiObj->dropChances));
			}
		}
	}
	return true;
}

//"killenemiesquiet" command
bool RSHan_KillEnemiesQuiet(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *killer = &g_gameObjects[MAP_OBJ_NUM];
	for (int i = 0; i < g_glb.gameObjectSlots; i++)
	{
		gameObject_t *obj = &g_gameObjects[i];
		if (obj->inuse && obj->hurtable && (obj->localFlags & LFL_ENEMY))
		{
			if (obj->aiObj && obj->aiObj->nonActive)
			{
				continue;
			}
			obj->think = ObjGeneral_RemoveThink;
			obj->thinkTime = g_glb.gameTime;
		}
	}
	return true;
}

//"togglenotarget" command
bool RSHan_ToggleNoTarget(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *pl = &g_gameObjects[0];
	if (!pl->inuse || !pl->plObj)
	{
		return true;
	}
	bool nt = !pl->plObj->noTarget;
	for (int i = 0; i < MAX_NET_CLIENTS; i++)
	{
		gameObject_t *pl = &g_gameObjects[i];
		if (!pl->inuse || !pl->plObj)
		{
			continue;
		}
		pl->plObj->noTarget = nt;
	}

	if (nt)
	{
		Util_StatusMessage("They have become blind to you.");
	}
	else
	{
		Util_StatusMessage("They seek you once more.");
	}
	return true;
}

//"toggleconfuse" command
bool RSHan_ToggleConfuse(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	if (g_glb.ai.inChaos)
	{
		Util_StatusMessage("Order has returned.");
		g_glb.ai.inChaos = false;
	}
	else
	{
		Util_StatusMessage("Chaos has been released.");
		g_glb.ai.inChaos = true;
	}
	return true;
}

//"magicmodecheck" command
bool RSHan_MagicModeCheck(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	if (g_glb.magicMode == 2)
	{
		bool dollarLives = false;
		for (int i = 0; i < g_glb.gameObjectSlots; i++)
		{
			gameObject_t *obj = &g_gameObjects[i];
			if (obj->inuse && obj->aiObj && obj->net.aiDescIndex == AIDESC_THEDOLLAR &&
				obj->health > 0)
			{
				dollarLives = true;
				break;
			}
		}

		if (dollarLives)
		{
			Util_StatusMessage("The Dollar forbids it!");
			gameObject_t *cam = Util_GetCam(0);
			if (cam->inuse)
			{
				ObjSound_Create(cam->net.pos, "assets/sound/menu/menuerr.wav", 1.0f, -1);
			}
			rscript->curNode = NULL;
			return true;
		}
	}
	return true;
}

//"statmsg" command
bool RSHan_StatMsg(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	Util_StatusMessage(RScr_GetArgStr(node, rscript, 0));
	return true;
}

//"arenamsg" command
bool RSHan_ArenaMsg(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	int l = GVar_GetInt("arenatier")+1;
	if (l == 16)
	{
		Util_StatusMessage("*(1 0 0 1)CRAZY 88");
	}
	else if (l == 29)
	{
		Util_StatusMessage("*(0.75 0.75 1 1)RELAXATION TIME");
	}
	else if (l == 35)
	{
		Util_StatusMessage("*(1 1 1 1)Final Plane");
	}
	else
	{
		Util_StatusMessage("*(1 1 1 1)Plane %02i", l);
	}
	return true;
}

//"setfade" command
bool RSHan_SetFade(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	g_glb.scriptScreenFade = (float)atof(RScr_GetArgStr(node, rscript, 0));
	g_sharedFn->Net_SendEventType(CLEV_SETFADE, &g_glb.scriptScreenFade, sizeof(g_glb.scriptScreenFade), -1);
	return true;
}

//"setfadecolor" command
bool RSHan_SetFadeColor(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	float clr[3];
	sscanf(RScr_GetArgStr(node, rscript, 0), "(%f %f %f)", &clr[0], &clr[1], &clr[2]);
	g_sharedFn->Net_SendEventType(CLEV_SETFADECOLOR, clr, sizeof(clr), -1);
	return true;
}

//"lerpfade" command
bool RSHan_LerpFade(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	float fadeTarget = (float)atof(RScr_GetArgStr(node, rscript, 0));
	float fadeSpeed = (float)atof(RScr_GetArgStr(node, rscript, 1))*timeMod;
	if (g_glb.scriptScreenFade > fadeTarget)
	{
		g_glb.scriptScreenFade -= fadeSpeed;
		if (g_glb.scriptScreenFade < fadeTarget)
		{
			g_glb.scriptScreenFade = fadeTarget;
		}
	}
	else if (g_glb.scriptScreenFade < fadeTarget)
	{
		g_glb.scriptScreenFade += fadeSpeed;
		if (g_glb.scriptScreenFade > fadeTarget)
		{
			g_glb.scriptScreenFade = fadeTarget;
		}
	}
	g_sharedFn->Net_SendEventType(CLEV_SETFADE, &g_glb.scriptScreenFade, sizeof(g_glb.scriptScreenFade), -1);

	return (g_glb.scriptScreenFade == fadeTarget);
}

//"setpicfade" command
bool RSHan_SetPicFade(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	g_glb.scriptPicFade = (float)atof(RScr_GetArgStr(node, rscript, 0));
	g_sharedFn->Net_SendEventType(CLEV_SETPICFADE, &g_glb.scriptPicFade, sizeof(g_glb.scriptPicFade), -1);
	return true;
}

//"lerppicfade" command
bool RSHan_LerpPicFade(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	float fadeTarget = (float)atof(RScr_GetArgStr(node, rscript, 0));
	float fadeSpeed = (float)atof(RScr_GetArgStr(node, rscript, 1))*timeMod;
	if (g_glb.scriptPicFade > fadeTarget)
	{
		g_glb.scriptPicFade -= fadeSpeed;
		if (g_glb.scriptPicFade < fadeTarget)
		{
			g_glb.scriptPicFade = fadeTarget;
		}
	}
	else if (g_glb.scriptPicFade < fadeTarget)
	{
		g_glb.scriptPicFade += fadeSpeed;
		if (g_glb.scriptPicFade > fadeTarget)
		{
			g_glb.scriptPicFade = fadeTarget;
		}
	}
	g_sharedFn->Net_SendEventType(CLEV_SETPICFADE, &g_glb.scriptPicFade, sizeof(g_glb.scriptPicFade), -1);

	return (g_glb.scriptPicFade == fadeTarget);
}

//"pushpic" command
bool RSHan_PushPic(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	char *pic = RScr_GetArgStr(node, rscript, 0);
	g_sharedFn->Net_SendEventType(CLEV_SETPIC, pic, (int)strlen(pic), -1);
	return true;
}

//"showhud" command
bool RSHan_ShowHUD(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	int showHud = atoi(RScr_GetArgStr(node, rscript, 0));
	g_sharedFn->Net_SendEventType(CLEV_SHOWHUD, &showHud, sizeof(int), -1);
	return true;
}

//"runmenuscript" command
bool RSHan_RunMenuScript(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	int clIndex = atoi(RScr_GetArgStr(node, rscript, 0));
	char *scrName = RScr_GetArgStr(node, rscript, 1);
	g_sharedFn->Net_SendEventType(CLEV_RUNMENUSCRIPT, scrName, (int)strlen(scrName)+1, clIndex);
	return true;
}

//"rollcredits" command
bool RSHan_RollCredits(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	float credSpeed = 2.0f;
	if (g_gameObjects[0].inuse && g_gameObjects[0].plObj &&
		g_gameObjects[0].plObj->clButtons[BUTTON_EVENT1])
	{
		credSpeed = 16.0f;
	}
	g_globalNet.creditRoll += credSpeed*timeMod;
	if (g_globalNet.creditRoll > 7450.0f)
	{
		g_globalNet.creditRoll = 7450.0f;
		return true;
	}
	return false;
}

//"setcinema" command
bool RSHan_SetCinema(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	g_glb.cinema = atoi(RScr_GetArgStr(node, rscript, 0));
	return true;
}

//"setmusic" command
bool RSHan_SetMusic(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	sprintf(g_glb.levelMusic, "$$%s", RScr_GetArgStr(node, rscript, 0));
	LServ_ChangeMusic(g_glb.levelMusic);
	return true;
}

//"cyclemusic" command
bool RSHan_CycleMusic(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	if (!stricmp(g_glb.levelMusic, "$$assets/music/AdrenalyneKyck.ogg"))
	{
		Util_StatusMessage("Nine Inch Nails - Head Down");
		strcpy(g_glb.levelMusic, "$$assets/music/HeadDown.ogg");
		LServ_ChangeMusic(g_glb.levelMusic);
	}
	else if (!stricmp(g_glb.levelMusic, "$$assets/music/HeadDown.ogg"))
	{
		Util_StatusMessage("Tweek - Sephiroth's Wake");
		strcpy(g_glb.levelMusic, "$$assets/music/SephirothsWake.ogg");
		LServ_ChangeMusic(g_glb.levelMusic);
	}
	else if (!stricmp(g_glb.levelMusic, "$$assets/music/SephirothsWake.ogg"))
	{
		Util_StatusMessage("bLiNd - Beginning of the End");
		strcpy(g_glb.levelMusic, "$$assets/music/BeginningoftheEnd.ogg");
		LServ_ChangeMusic(g_glb.levelMusic);
	}
	else if (!stricmp(g_glb.levelMusic, "$$assets/music/BeginningoftheEnd.ogg"))
	{
		Util_StatusMessage("Tweek - Frozen Landscape");
		strcpy(g_glb.levelMusic, "$$assets/music/FrozenLandscape.ogg");
		LServ_ChangeMusic(g_glb.levelMusic);
	}
	else if (!stricmp(g_glb.levelMusic, "$$assets/music/FrozenLandscape.ogg"))
	{
		Util_StatusMessage("Nine Inch Nails - Discipline");
		strcpy(g_glb.levelMusic, "$$assets/music/Discipline.ogg");
		LServ_ChangeMusic(g_glb.levelMusic);
	}
	else if (!stricmp(g_glb.levelMusic, "$$assets/music/Discipline.ogg"))
	{
		Util_StatusMessage("norg, SnappleMan - Full Frontal Assault");
		strcpy(g_glb.levelMusic, "$$assets/music/FullFrontalAssault.ogg");
		LServ_ChangeMusic(g_glb.levelMusic);
	}
	else if (!stricmp(g_glb.levelMusic, "$$assets/music/FullFrontalAssault.ogg"))
	{
		Util_StatusMessage("Darangen - Kweh!");
		strcpy(g_glb.levelMusic, "$$assets/music/Kweh.ogg");
		LServ_ChangeMusic(g_glb.levelMusic);
	}
	else if (!stricmp(g_glb.levelMusic, "$$assets/music/Kweh.ogg"))
	{
		Util_StatusMessage("zircon - Nomura Limit");
		strcpy(g_glb.levelMusic, "$$assets/music/NomuraLimit.ogg");
		LServ_ChangeMusic(g_glb.levelMusic);
	}
	else if (!stricmp(g_glb.levelMusic, "$$assets/music/NomuraLimit.ogg"))
	{
		Util_StatusMessage("Nine Inch Nails - Lights in the Sky");
		strcpy(g_glb.levelMusic, "$$assets/music/LightsintheSky.ogg");
		LServ_ChangeMusic(g_glb.levelMusic);
	}
	else if (!stricmp(g_glb.levelMusic, "$$assets/music/LightsintheSky.ogg"))
	{
		Util_StatusMessage("sephfire, sgx - No Such Thing As the Promised Land");
		strcpy(g_glb.levelMusic, "$$assets/music/NoSuchThingAsthePromisedLand.ogg");
		LServ_ChangeMusic(g_glb.levelMusic);
	}
	else if (!stricmp(g_glb.levelMusic, "$$assets/music/NoSuchThingAsthePromisedLand.ogg"))
	{
		Util_StatusMessage("Fishy - Omnislash");
		strcpy(g_glb.levelMusic, "$$assets/music/Omnislash.ogg");
		LServ_ChangeMusic(g_glb.levelMusic);
	}
	else if (!stricmp(g_glb.levelMusic, "$$assets/music/Omnislash.ogg"))
	{
		Util_StatusMessage("Mustin - Serenity");
		strcpy(g_glb.levelMusic, "$$assets/music/Serenity.ogg");
		LServ_ChangeMusic(g_glb.levelMusic);
	}
	else if (!stricmp(g_glb.levelMusic, "$$assets/music/Serenity.ogg"))
	{
		Util_StatusMessage("Music off.");
		strcpy(g_glb.levelMusic, "$$");
		LServ_ChangeMusic(g_glb.levelMusic);
	}
	else
	{
		Util_StatusMessage("BigGiantCircles, Liontamer, zircon - Adrenalyne Kyck");
		strcpy(g_glb.levelMusic, "$$assets/music/AdrenalyneKyck.ogg");
		LServ_ChangeMusic(g_glb.levelMusic);
	}
	return true;
}

//"setbattlemusic" command
bool RSHan_SetBattleMusic(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	g_glb.battleTime = 0;
	g_glb.battleMusicEnabled = !!atoi(RScr_GetArgStr(node, rscript, 0));
	return true;
}

//"setmagicmode" command
bool RSHan_SetMagicMode(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	g_glb.magicMode = atoi(RScr_GetArgStr(node, rscript, 0));
	return true;
}

//"setendlessbattle" command
bool RSHan_SetEndlessBattle(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	for (int i = 0; i < MAX_NET_CLIENTS; i++)
	{
		gameObject_t *pl = &g_gameObjects[i];
		if (pl->inuse && pl->plObj && pl->health > 0)
		{
			pl->health = pl->plObj->plData.maxHealth;
			pl->plObj->makoCharges = pl->plObj->plData.maxMakoCharges;
			pl->hurtable = 1;
		}
	}
	int numKills = g_glb.endlessBattle.numKills;
	bool wasActive = (g_glb.endlessBattle.active != 0);

	g_globalNet.endlScore = 0;
	g_globalNet.endlHighscore = 0;

	g_glb.timeType = TIMETYPE_NORMAL;

	memset(&g_glb.endlessBattle, 0, sizeof(g_glb.endlessBattle));
	g_glb.endlessBattle.active = atoi(RScr_GetArgStr(node, rscript, 0));
	if (g_glb.endlessBattle.active)
	{
		g_globalNet.endlHighscore = GVar_GetInt("endl_killrecord");
		if (!g_globalNet.endlHighscore)
		{
			g_globalNet.endlHighscore = -1;
		}
	}
	if (wasActive && !g_glb.endlessBattle.active)
	{
		int recKills = GVar_GetInt("endl_killrecord");
		Util_StatusMessage("Record score: *(1 0.1 0.1 1)%i", recKills);
		Util_StatusMessage("Final score: *(1 1 1 1)%i", numKills);
		if (recKills == numKills)
		{
			Util_StatusMessage("*(1 0.1 0.1 1)NEW RECORD!");
		}
	}
	else if (!wasActive && g_glb.endlessBattle.active)
	{
		Util_StatusMessage("Record to beat: *(1 0.1 0.1 1)%i", GVar_GetInt("endl_killrecord"));
	}
	return true;
}

//"endlessbattlespawns" command
bool RSHan_EndlessBattleSpawns(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	if (!g_glb.endlessBattle.active)
	{
		return true;
	}

	int numLiveAI = 0;

	for (int i = 0; i < g_glb.gameObjectSlots; i++)
	{
		gameObject_t *obj = &g_gameObjects[i];
		if (!obj->inuse || !obj->aiObj || obj->aiObj->nonActive ||
			!(obj->localFlags & LFL_ENEMY) || obj->health <= 0)
		{
			continue;
		}
		numLiveAI++;
	}

	int minAI = 0;
	int &nk = g_glb.endlessBattle.numKills;
	if (nk > 200)
	{
		minAI = 20;
	}
	else if (nk > 100)
	{
		minAI = 15;
	}
	else if (nk > 30)
	{
		minAI = 10;
	}
	else
	{
		minAI = 5;
	}

	if (numLiveAI < minAI)
	{
		int r = rand()%3;
		if (r == 2 && nk > 500)
		{
			Util_RunScript("obj_battlespawn", "endl_spawn3");
		}
		else if (r == 1 && nk > 300)
		{
			Util_RunScript("obj_battlespawn", "endl_spawn2");
		}
		else
		{
			Util_RunScript("obj_battlespawn", "endl_spawn1");
		}
		rscript->scriptTime = g_glb.gameTime + 100;
	}

	return false;
}

//"setmpversusmode" command
bool RSHan_SetMPVersusMode(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	g_glb.mpVersusMode = atoi(RScr_GetArgStr(node, rscript, 0));
	return true;
}

//"setskymodel" command
bool RSHan_SetSkyModel(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	char str[256];
	sprintf(str, "&%s", RScr_GetArgStr(node, rscript, 0));
	g_globalNet.skyboxModel = g_sharedFn->Common_ServerString(str);
	return true;
}

//"setskyterrain" command
bool RSHan_SetSkyTerrain(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	char str[256];
	sprintf(str, "&%s", RScr_GetArgStr(node, rscript, 0));
	g_globalNet.skyboxTerrain = g_sharedFn->Common_ServerString(str);
	return true;
}

//"setskycolor" command
bool RSHan_SetSkyColor(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	sscanf(RScr_GetArgStr(node, rscript, 0), "(%f %f %f)", &g_globalNet.skyboxColor[0],
		&g_globalNet.skyboxColor[1], &g_globalNet.skyboxColor[2]);
	return true;
}

//"setskycolorrand" command
bool RSHan_SetSkyColorRand(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	g_globalNet.skyboxColor[0] = Util_RandFloat(0.5f, 2.0f);
	g_globalNet.skyboxColor[1] = Util_RandFloat(0.5f, 2.0f);
	g_globalNet.skyboxColor[2] = Util_RandFloat(0.5f, 2.0f);

	for (int i = 0; i < g_glb.gameObjectSlots; i++)
	{
		gameObject_t *obj = &g_gameObjects[i];
		if (obj->inuse && obj->net.type == OBJ_TYPE_LIGHT &&
			(obj->net.frame & LIGHTFL_FULLAMB))
		{
			obj->net.mins[0] = g_globalNet.skyboxColor[0];
			obj->net.mins[1] = g_globalNet.skyboxColor[1];
			obj->net.mins[2] = g_globalNet.skyboxColor[2];
		}
	}
	return true;
}

//"setpropframes" command
bool RSHan_SetPropFrames(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *obj = Util_FindTargetObject(RScr_GetArgStr(node, rscript, 0));
	if (!obj || !obj->cinObj)
	{
		return true;
	}

	obj->cinObj->animStub.startFrame = atoi(RScr_GetArgStr(node, rscript, 1));
	obj->cinObj->animStub.endFrame = atoi(RScr_GetArgStr(node, rscript, 2));
	obj->cinObj->animStub.duration = (float)atof(RScr_GetArgStr(node, rscript, 3));
	obj->cinObj->animStub.looping = !!atoi(RScr_GetArgStr(node, rscript, 4));
	return true;
}

//"forcenextanim" command
bool RSHan_ForceNextAnim(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *obj = Util_FindTargetObject(RScr_GetArgStr(node, rscript, 0));
	if (!obj)
	{
		return true;
	}
	obj->animTime = 0;
	return true;
}

//"setplmovespeed" command
bool RSHan_SetPlMoveSpeed(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *obj = Util_FindTargetObject("__the_player");
	if (!obj || !obj->plObj)
	{
		return true;
	}

	obj->plObj->moveSpeedScale = (float)atof(RScr_GetArgStr(node, rscript, 0));
	return true;
}

//"setplcamtarg" command
bool RSHan_SetPlCamTarg(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *obj = Util_FindTargetObject("__the_player");
	if (!obj || !obj->plObj)
	{
		return true;
	}

	char *targStr = RScr_GetArgStr(node, rscript, 0);
	gameObject_t *targ = (targStr && targStr[0]) ? Util_FindTargetObject(targStr) : NULL;
	obj->plObj->camTarget = targ;
	return true;
}

//"setplcamrange" command
bool RSHan_SetPlCamRange(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *obj = Util_FindTargetObject("__the_player");
	if (!obj || !obj->plObj)
	{
		return true;
	}

	obj->plObj->camRange = (float)atof(RScr_GetArgStr(node, rscript, 0));
	return true;
}

//"giveplitem" command
bool RSHan_GivePlItem(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *obj = Util_FindTargetObject("__the_player");
	if (!obj || !obj->plObj)
	{
		return true;
	}

	char *item = RScr_GetArgStr(node, rscript, 0);
	for (int i = 0; i < NUM_INVENTORY_ITEM_DEFS; i++)
	{
		const invItemDef_t *itemDef = &g_invItems[i];
		if (!strcmp(item, itemDef->name))
		{
			ObjPlayer_GiveItem(obj, i);
			ObjSound_Create(obj->net.pos, "assets/sound/items/itempickup.wav", 1.0f, -1);
			return true;
		}
	}

	return true;
}

//"giveplmako" command
bool RSHan_GivePlMako(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *obj = Util_FindTargetObject("__the_player");
	if (!obj || !obj->plObj)
	{
		return true;
	}
	ObjSound_Create(obj->net.pos, "assets/sound/cb/makopickup.wav", 1.0f, -1);
	ObjPlayer_GiveMako(obj, atoi(RScr_GetArgStr(node, rscript, 0)));
	return true;
}

//"giveplcharge" command
bool RSHan_GivePlCharge(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *obj = Util_FindTargetObject("__the_player");
	if (!obj || !obj->plObj)
	{
		return true;
	}

	ObjPlayer_ChargeUp(obj, NULL, atoi(RScr_GetArgStr(node, rscript, 0)));

	return true;
}

//"setcambolt" command
bool RSHan_SetCamBolt(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *cam = Util_GetCam(0);
	if (!cam->inuse)
	{
		return true;
	}

	char *targ = RScr_GetArgStr(node, rscript, 0);
	if (!targ[0])
	{
		cam->target = NULL;
		g_glb.camBoneBolt[0] = 0;
		g_glb.camBoltOffset[0] = 0.0f;
		g_glb.camBoltOffset[1] = 0.0f;
		g_glb.camBoltOffset[2] = 0.0f;
		return true;
	}

	gameObject_t *obj = Util_FindTargetObject(targ);
	if (!obj)
	{
		return true;
	}

	cam->target = obj;
	strcpy(g_glb.camBoneBolt, RScr_GetArgStr(node, rscript, 1));
	char *ofsStr = RScr_GetArgStr(node, rscript, 2);
	if (ofsStr[0])
	{
		sscanf(ofsStr, "(%f %f %f)", &g_glb.camBoltOffset[0], &g_glb.camBoltOffset[1],
			&g_glb.camBoltOffset[2]);
	}
	else
	{
		g_glb.camBoltOffset[0] = 0.0f;
		g_glb.camBoltOffset[1] = 0.0f;
		g_glb.camBoltOffset[2] = 0.0f;
	}

	return true;
}

//"setcamabsofs" command
bool RSHan_SetCamAbsOfs(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *cam = Util_GetCam(0);
	if (!cam->inuse)
	{
		return true;
	}

	char *ofsStr = RScr_GetArgStr(node, rscript, 0);
	if (ofsStr[0])
	{
		sscanf(ofsStr, "(%f %f %f)", &g_glb.camAbsOffset[0], &g_glb.camAbsOffset[1],
			&g_glb.camAbsOffset[2]);
	}
	else
	{
		g_glb.camAbsOffset[0] = 0.0f;
		g_glb.camAbsOffset[1] = 0.0f;
		g_glb.camAbsOffset[2] = 0.0f;
	}

	char *angStr = RScr_GetArgStr(node, rscript, 1);
	if (angStr[0])
	{
		sscanf(angStr, "(%f %f %f)", &g_glb.camAbsAngOffset[0], &g_glb.camAbsAngOffset[1],
			&g_glb.camAbsAngOffset[2]);
	}
	else
	{
		g_glb.camAbsAngOffset[0] = 0.0f;
		g_glb.camAbsAngOffset[1] = 0.0f;
		g_glb.camAbsAngOffset[2] = 0.0f;
	}
	return true;
}

//"setactiontime" command
bool RSHan_SetActionTime(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	g_glb.actionTime = g_glb.gameTime+atoi(RScr_GetArgStr(node, rscript, 0));
	return true;
}

//"setactiontimescale" command
bool RSHan_SetActionTimeScale(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	g_glb.actionTimeScale = (float)atof(RScr_GetArgStr(node, rscript, 0));
	return true;
}

//"setmbientlight" command
bool RSHan_SetAmbientLight(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	float clr[3];
	sscanf(RScr_GetArgStr(node, rscript, 0), "(%f %f %f)", &clr[0], &clr[1], &clr[2]);
	for (int i = 0; i < g_glb.gameObjectSlots; i++)
	{
		gameObject_t *obj = &g_gameObjects[i];
		if (obj->inuse && obj->net.type == OBJ_TYPE_LIGHT &&
			(obj->net.frame & LIGHTFL_FULLAMB))
		{
			obj->net.mins[0] = clr[0];
			obj->net.mins[1] = clr[1];
			obj->net.mins[2] = clr[2];
		}
	}
	return true;
}

//"lerpambientlight" command
bool RSHan_LerpAmbientLight(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	float clr[3];
	sscanf(RScr_GetArgStr(node, rscript, 0), "(%f %f %f)", &clr[0], &clr[1], &clr[2]);
	float lerpScale = (float)atof(RScr_GetArgStr(node, rscript, 1))*timeMod;
	bool anyNotThere = false;
	for (int i = 0; i < g_glb.gameObjectSlots; i++)
	{
		gameObject_t *obj = &g_gameObjects[i];
		if (obj->inuse && obj->net.type == OBJ_TYPE_LIGHT &&
			(obj->net.frame & LIGHTFL_FULLAMB))
		{
			for (int j = 0; j < 3; j++)
			{
				obj->net.mins[j] += (clr[j]-obj->net.mins[j])*lerpScale;
				if (fabsf(clr[j]-obj->net.mins[j]) <= lerpScale)
				{
					obj->net.mins[j] = clr[j];
				}
				else
				{
					anyNotThere = true;
				}
			}
		}
	}
	return !anyNotThere;
}

//"lerpambientlightevil" command
bool RSHan_LerpAmbientLightEvil(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	float clr[3];
	sscanf(RScr_GetArgStr(node, rscript, 0), "(%f %f %f)", &clr[0], &clr[1], &clr[2]);
	float lerpScale = (float)atof(RScr_GetArgStr(node, rscript, 1))*timeMod;
	bool anyNotThere = false;
	for (int i = 0; i < g_glb.gameObjectSlots; i++)
	{
		gameObject_t *obj = &g_gameObjects[i];
		if (obj->inuse && obj->net.type == OBJ_TYPE_LIGHT)
		{
			float evilClr[3] = {1.0f, 0.0f, 0.0f};
			float *dclr = clr;
			if (obj->net.frame & LIGHTFL_FULLAMB)
			{
				dclr = clr;
			}
			else
			{
				dclr = evilClr;
			}
			for (int j = 0; j < 3; j++)
			{
				obj->net.mins[j] += (dclr[j]-obj->net.mins[j])*lerpScale;
				if (fabsf(dclr[j]-obj->net.mins[j]) <= lerpScale)
				{
					obj->net.mins[j] = dclr[j];
				}
				else
				{
					anyNotThere = true;
				}
			}
		}
	}
	return !anyNotThere;
}

//"setviewtrails" command
bool RSHan_SetViewTrails(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	float *t = (float *)_alloca(sizeof(float)*2);
	*t = (float)atof(RScr_GetArgStr(node, rscript, 0));
	*(t+1) = (float)atof(RScr_GetArgStr(node, rscript, 1));
	g_sharedFn->Net_SendEventType(CLEV_VIEWTRAILS, t, sizeof(float)*2, -1);
	return true;
}

//"setfragfog"
bool RSHan_SetFragFog(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	float *t = (float *)_alloca(sizeof(float)*6 + sizeof(int));
	char *str = RScr_GetArgStr(node, rscript, 0);
	sscanf(str, "(%f %f %f %f)", &t[0], &t[1], &t[2], &t[3]);
	t[4] = (float)atof(RScr_GetArgStr(node, rscript, 1));
	t[5] = (float)atof(RScr_GetArgStr(node, rscript, 2));
	int *fbm = (int *)(t+6);
	*fbm = atoi(RScr_GetArgStr(node, rscript, 3));
	g_sharedFn->Net_SendEventType(CLEV_SETFOG, t, sizeof(float)*6 + sizeof(int), -1);
	memcpy(g_glb.scriptFogValues, t, sizeof(g_glb.scriptFogValues));
	g_glb.scriptFogBlendMode = *fbm;
	return true;
}

//"lerpfragfog"
bool RSHan_LerpFragFog(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	float *t = (float *)_alloca(sizeof(float)*6 + sizeof(int));
	char *str = RScr_GetArgStr(node, rscript, 0);
	sscanf(str, "(%f %f %f %f)", &t[0], &t[1], &t[2], &t[3]);
	t[4] = (float)atof(RScr_GetArgStr(node, rscript, 1));
	t[5] = (float)atof(RScr_GetArgStr(node, rscript, 2));
	int *fbm = (int *)(t+6);
	*fbm = atoi(RScr_GetArgStr(node, rscript, 3));
	float defLerpSpeed = (float)atof(RScr_GetArgStr(node, rscript, 4));
	float clrLerpSpeed = defLerpSpeed * timeMod;
	float distLerpSpeed = defLerpSpeed * 16384.0f * timeMod;
	bool doneLerping = true;
	for (int i = 0; i < 6; i++)
	{
		float lerpSpeed = (i >= 4) ? distLerpSpeed : clrLerpSpeed;
		if (g_glb.scriptFogValues[i] > t[i])
		{
			doneLerping = false;
			g_glb.scriptFogValues[i] -= lerpSpeed;
			if (g_glb.scriptFogValues[i] < t[i])
			{
				g_glb.scriptFogValues[i] = t[i];
			}
		}
		else if (g_glb.scriptFogValues[i] < t[i])
		{
			doneLerping = false;
			g_glb.scriptFogValues[i] += lerpSpeed;
			if (g_glb.scriptFogValues[i] > t[i])
			{
				g_glb.scriptFogValues[i] = t[i];
			}
		}
	}
	memcpy(t, g_glb.scriptFogValues, sizeof(g_glb.scriptFogValues));	
	g_glb.scriptFogBlendMode = *fbm;
	g_sharedFn->Net_SendEventType(CLEV_SETFOG, t, sizeof(float)*6 + sizeof(int), -1);

	return doneLerping;
}

//"setbloommod" command
bool RSHan_SetBloomMod(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	float amount = (float)atof(RScr_GetArgStr(node, rscript, 0));
	g_sharedFn->Net_SendEventType(CLEV_BLOOMMOD, &amount, sizeof(amount), -1);
	return true;
}

//"setpcltimescale" command
bool RSHan_SetPclTimeScale(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	float ts = (float)atof(RScr_GetArgStr(node, rscript, 0));
	g_sharedFn->Net_SendEventType(CLEV_PCLTIME, &ts, sizeof(ts), -1);
	return true;
}

//"setviewflash" command
bool RSHan_SetViewFlash(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	int *t = (int *)_alloca(sizeof(int)+sizeof(float)*3);
	*t = atoi(RScr_GetArgStr(node, rscript, 0));
	float *c = (float *)(t+1);
	sscanf(RScr_GetArgStr(node, rscript, 1), "(%f %f %f)", &c[0], &c[1], &c[2]);
	g_sharedFn->Net_SendEventType(CLEV_TRIGGERFLASH, t, sizeof(int)+sizeof(float)*3, -1);
	return true;
}

//"setviewshake" command
bool RSHan_SetViewShake(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	float shake = (float)atof(RScr_GetArgStr(node, rscript, 0));
	g_sharedFn->Net_SendEventType(CLEV_THESHAKES, &shake, sizeof(shake), -1);
	return true;
}

//"setenspawnblock" command
bool RSHan_SetEnSpawnBlock(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	g_glb.enSpawnBlock = atoi(RScr_GetArgStr(node, rscript, 0));
	return true;
}

//"setexvar" command
bool RSHan_SetExVar(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	char *varName = RScr_GetArgStr(node, rscript, 0);
	char *varVal = RScr_GetArgStr(node, rscript, 1);
	RCUBE_ASSERT(strlen(varName) <= 30);
	RCUBE_ASSERT(strlen(varVal) <= 30);
	char str[64];
	sprintf(str, "^^%s;", varName);
	int sidx = g_sharedFn->Common_ServerStringPfx(str);
	if (sidx >= 0)
	{
		strcat(str, varVal);
		g_sharedFn->Common_ModifyString(sidx, str);
	}

	return true;
}

//"mpshowstats" command
bool RSHan_MPShowStats(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	if (!g_glb.runningMultiplayer)
	{
		return true;
	}

	LServ_ShowMPStats();

	return true;
}

//"mpmatchcheck" command
bool RSHan_MPMatchCheck(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	if (!g_glb.runningMultiplayer)
	{
		rscript->curNode = NULL;
		return true;
	}
	int numLive = 0;
	for (int i = 0; i < MAX_NET_CLIENTS; i++)
	{
		gameObject_t *pl = &g_gameObjects[i];
		if (!pl->inuse || !pl->plObj || pl->health <= 0)
		{
			continue;
		}
		numLive++;
	}
	if (numLive < 2)
	{
		Util_StatusMessage("You need another player to fight.");
		rscript->curNode = NULL;
	}
	return true;
}

//"mpbeginmatch" command
bool RSHan_MPBeginMatch(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	if (!g_glb.runningMultiplayer)
	{
		return true;
	}

	g_globalNet.vsNames[0][0] = 0;
	g_globalNet.vsNames[1][0] = 0;
	g_glb.timeType = TIMETYPE_NORMAL;
	gameObject_t *pl;
	for (int i = 0; i < MAX_NET_CLIENTS; i++)
	{
		pl = &g_gameObjects[i];
		if (!pl->inuse || !pl->plObj)
		{
			continue;
		}
		if (i < 2)
		{
			strcpy_s(g_globalNet.vsNames[i], 15, g_glb.clientInfo[i].infoStr);
			g_globalNet.vsNames[i][15] = 0;
			g_globalNet.vsHealth[i] = pl->health;
			g_globalNet.vsMaxHealth[i] = pl->plObj->plData.maxHealth;
		}
		pl->isOnFire = 0;
		pl->plObj->fistOfFlames = 0;
		pl->net.renderEffects &= ~FXFL_FLAMES;
		pl->plObj->explosiveBlows = 0;
		pl->net.renderEffects &= ~FXFL_AURA2;
		pl->net.renderEffects &= ~FXFL_BONEAURA;
		pl->plObj->healingAura = 0;
		pl->net.renderEffects &= ~FXFL_HEALINGAURA;
		pl->meteorAttack = 0;
		pl->net.renderEffects &= ~FXFL_METEORSTREAM;
		pl->health = pl->plObj->plData.maxHealth;
		pl->plObj->makoCharges = pl->plObj->plData.maxMakoCharges;

		if (!pl->scriptAnims)
		{
			continue;
		}
		for (int j = 0; pl->scriptAnims[j].animName; j++)
		{
			scriptableAnim_t *a = &pl->scriptAnims[j];
			if (!stricmp(a->animName, "PLANIM_CIN_COMBATREADY"))
			{
				AI_StartAnim(pl, a->animNum, true);
				break;
			}
		}
	}

	g_glb.mpCountDown = 4;
	g_glb.mpCountTime = g_glb.gameTime+1000;

	pl = &g_gameObjects[0];
	pl->net.pos[0] = -75681.0f;
	pl->net.pos[1] = 20000.0f;
	pl->net.pos[2] = 150.0f;
	Phys_PutOnGround(pl);
	pl->net.ang[0] = 0.0f;
	pl->net.ang[1] = 0.0f;
	pl->net.ang[2] = 0.0f;

	pl = &g_gameObjects[1];
	pl->net.pos[0] = -71721.0f;
	pl->net.pos[1] = 20000.0f;
	pl->net.pos[2] = 150.0f;
	Phys_PutOnGround(pl);
	pl->net.ang[0] = 0.0f;
	pl->net.ang[1] = 180.0f;
	pl->net.ang[2] = 0.0f;

	g_glb.mpVersusMode = 1;

	return true;
}

//"mpendmatch" command
bool RSHan_MPEndMatch(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	if (!g_glb.runningMultiplayer)
	{
		return true;
	}

	g_globalNet.vsNames[0][0] = 0;
	g_globalNet.vsNames[1][0] = 0;

	for (int i = 0; i < MAX_NET_CLIENTS; i++)
	{
		gameObject_t *pl = &g_gameObjects[i];
		if (!pl->inuse || !pl->plObj)
		{
			continue;
		}
		pl->meteorAttack = 0;
		pl->health = pl->plObj->plData.maxHealth;
		pl->plObj->makoCharges = pl->plObj->plData.maxMakoCharges;
		pl->hurtable = 1;
		pl->net.solid = 1;
		pl->animNeverInterrupt = false;
		AI_StartAnim(pl, HUMANIM_IDLE, true);
		LServ_UpdateRClip(pl);
		if (i > 0)
		{
			ObjPlayer_TeleportToPlZero(pl);
		}
	}

	g_glb.timeType = TIMETYPE_NORMAL;

	return true;
}

//"setcamslack" command
bool RSHan_SetCamSlack(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *cam = Util_GetCam(0);
	if (!cam->inuse)
	{
		return true;
	}

	g_glb.camSlack = (float)atof(RScr_GetArgStr(node, rscript, 0));

	return true;
}

//"setcamfov" command
bool RSHan_SetCamFov(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *cam = Util_GetCam(0);
	if (!cam->inuse)
	{
		return true;
	}

	cam->net.modelScale[0] = (float)atof(RScr_GetArgStr(node, rscript, 0));

	return true;
}

//"lerpcamfov" command
bool RSHan_LerpCamFov(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *cam = Util_GetCam(0);
	if (!cam->inuse)
	{
		return true;
	}

	float desiredFov = (float)atof(RScr_GetArgStr(node, rscript, 0));
	float lerpScale = (float)atof(RScr_GetArgStr(node, rscript, 1))*timeMod;
	cam->net.modelScale[0] += (desiredFov-cam->net.modelScale[0])*lerpScale;

	if (fabsf(desiredFov-cam->net.modelScale[0]) <= lerpScale)
	{
		cam->net.modelScale[0] = desiredFov;
		return true;
	}

	return false;
}

//"objcreate" command
bool RSHan_ObjCreate(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	float pos[3], ang[3];
	sscanf(RScr_GetArgStr(node, rscript, 1), "(%f %f %f)", &pos[0], &pos[1], &pos[2]);
	sscanf(RScr_GetArgStr(node, rscript, 2), "(%f %f %f)", &ang[0], &ang[1], &ang[2]);
	LServ_ObjectFromName(RScr_GetArgStr(node, rscript, 0), pos, ang, NULL, 0);
	return true;
}

//"objcreatenamed" command
bool RSHan_ObjCreateNamed(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	float pos[3], ang[3];
	sscanf(RScr_GetArgStr(node, rscript, 2), "(%f %f %f)", &pos[0], &pos[1], &pos[2]);
	sscanf(RScr_GetArgStr(node, rscript, 3), "(%f %f %f)", &ang[0], &ang[1], &ang[2]);
	gameObject_t *obj = LServ_ObjectFromName(RScr_GetArgStr(node, rscript, 1), pos, ang, NULL, 0);
	if (obj)
	{
		obj->targetName = Util_PooledString(RScr_GetArgStr(node, rscript, 0));
	}
	return true;
}

//"objcreatenamedatobj" command
bool RSHan_ObjCreateNamedAtObj(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *spot = Util_FindTargetObject(RScr_GetArgStr(node, rscript, 2));
	if (!spot)
	{
		return true;
	}
	gameObject_t *obj = LServ_ObjectFromName(RScr_GetArgStr(node, rscript, 1), spot->net.pos, spot->net.ang, NULL, 0);
	if (obj)
	{
		obj->targetName = Util_PooledString(RScr_GetArgStr(node, rscript, 0));
	}
	return true;
}

//"objcreatenamedinobj" command
bool RSHan_ObjCreateNamedInObj(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *spot = Util_FindTargetObject(RScr_GetArgStr(node, rscript, 2));
	if (!spot)
	{
		return true;
	}

	char *objName = RScr_GetArgStr(node, rscript, 1);
	BYTE *b = Common_GetEntryForObject(objName);
	if (!b)
	{ //no entry for this object then
		return true;
	}

	int spawnTries;
	float pos[3];
	for (spawnTries = 0; spawnTries < 10; spawnTries++)
	{
		Math_VecCopy(spot->net.pos, pos);
		pos[0] += Util_RandFloat(spot->spawnMins[0], spot->spawnMaxs[0]);
		pos[1] += Util_RandFloat(spot->spawnMins[1], spot->spawnMaxs[1]);
		pos[2] += Util_RandFloat(spot->spawnMins[2], spot->spawnMaxs[2]);
		if (Util_TryObjectSpawn(pos, b))
		{
			break;
		}
	}

	if (spawnTries == 10)
	{ //wait 200ms and try again
		rscript->scriptTime = g_glb.gameTime + 200;
		return false;
	}

	float ang[3] = {0.0f, 0.0f, 0.0f};
	gameObject_t *pl = &g_gameObjects[0];
	if (pl->inuse && pl->plObj)
	{
		float d[3];
		Math_VecSub(pl->net.pos, pos, d);
		Math_VecToAngles(d, d);
		ang[YAW] = d[YAW];
	}
	gameObject_t *obj = LServ_ObjectFromName(objName, pos, ang, NULL, 0);
	if (obj)
	{
		obj->targetName = Util_PooledString(RScr_GetArgStr(node, rscript, 0));
		obj->net.renderEffects2 |= FXFL2_SPAWN2;
		ObjSound_Create(obj->net.pos, "assets/sound/cb/espawn.wav", 1.0f, -1);
		ObjSound_Create(obj->net.pos, "assets/sound/cb/espawn.wav", 1.0f, -1);
	}
	return true;
}

//"objcreatenamedinobjrnd" command
bool RSHan_ObjCreateNamedInObjRnd(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	if (!rscript->ropIntRegs[ROP_INTREG_STORE1])
	{
		int numLow = atoi(RScr_GetArgStr(node, rscript, 3));
		int numHigh = atoi(RScr_GetArgStr(node, rscript, 4));
		rscript->ropIntRegs[ROP_INTREG_STORE1] = Util_RandInt(numLow, numHigh);
		if (!rscript->ropIntRegs[ROP_INTREG_STORE1])
		{
			rscript->ropIntRegs[ROP_INTREG_COUNT] = 0;
			return true;
		}
	}

	if (rscript->ropIntRegs[ROP_INTREG_COUNT] >= rscript->ropIntRegs[ROP_INTREG_STORE1])
	{
		rscript->ropIntRegs[ROP_INTREG_STORE1] = 0;
		rscript->ropIntRegs[ROP_INTREG_COUNT] = 0;
		return true;
	}

	if (!RSHan_ObjCreateNamedInObj(node, rscript, timeMod))
	{
		return false;
	}

	rscript->ropIntRegs[ROP_INTREG_COUNT]++;
	rscript->scriptTime = g_glb.gameTime + Util_RandInt(100, 500);
	return false;
}

//"objcreateenemy" command
bool RSHan_ObjCreateEnemy(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	if (g_glb.enSpawnBlock)
	{
		return true;
	}
	return RSHan_ObjCreateEnemyForce(node, rscript, timeMod);
}

//"objcreateenemyforce" command
bool RSHan_ObjCreateEnemyForce(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *spot = Util_FindTargetObject(RScr_GetArgStr(node, rscript, 1));
	if (!spot)
	{
		return true;
	}

	char *objName = RScr_GetArgStr(node, rscript, 0);
	BYTE *b = Common_GetEntryForObject(objName);
	if (!b)
	{ //no entry for this object then
		return true;
	}

	float spawnMins[3], spawnMaxs[3];
	Math_VecCopy(spot->spawnMins, spawnMins);
	Math_VecCopy(spot->spawnMaxs, spawnMaxs);

	if (g_gameObjects[0].inuse)
	{ //spawn in the two quadrants furthest from the player
		float *awayPos = g_gameObjects[0].net.pos;
		spawnMins[0] *= 0.5f;
		spawnMins[1] *= 0.5f;
		spawnMaxs[0] *= 0.5f;
		spawnMaxs[1] *= 0.5f;
		float spawnQuads[4][3];
		int quadNeighbors[4][2] = 
		{
			{1, 3},
			{0, 2},
			{1, 3},
			{2, 0}
		};
		spawnQuads[0][0] = spot->net.pos[0]+spawnMins[0];
		spawnQuads[0][1] = spot->net.pos[1]+spawnMins[1];
		spawnQuads[0][2] = spot->net.pos[2];
		spawnQuads[1][0] = spot->net.pos[0]+spawnMaxs[0];
		spawnQuads[1][1] = spot->net.pos[1]+spawnMins[1];
		spawnQuads[1][2] = spot->net.pos[2];
		spawnQuads[2][0] = spot->net.pos[0]+spawnMaxs[0];
		spawnQuads[2][1] = spot->net.pos[1]+spawnMaxs[1];
		spawnQuads[2][2] = spot->net.pos[2];
		spawnQuads[3][0] = spot->net.pos[0]+spawnMins[0];
		spawnQuads[3][1] = spot->net.pos[1]+spawnMaxs[1];
		spawnQuads[3][2] = spot->net.pos[2];
		int farthestQuad = -1;
		float farthestQuadDist = 0.0f;
		float quadDistances[4];
		for (int x = 0; x < 4; x++)
		{
			float d[3];
			Math_VecSub(awayPos, spawnQuads[x], d);
			quadDistances[x] = Math_VecLen(d);
			if (farthestQuad == -1 || quadDistances[x] > farthestQuadDist)
			{
				farthestQuad = x;
				farthestQuadDist = quadDistances[x];
			}
		}
		RCUBE_ASSERT(farthestQuad != -1);
		int *qn = quadNeighbors[farthestQuad];
		int neighborIndex = (quadDistances[qn[0]] > quadDistances[qn[1]]) ? qn[0] : qn[1];
		float ex[4][3];
		Math_VecAdd(spawnQuads[farthestQuad], spawnMins, ex[0]);
		Math_VecAdd(spawnQuads[farthestQuad], spawnMaxs, ex[1]);
		Math_VecAdd(spawnQuads[neighborIndex], spawnMins, ex[2]);
		Math_VecAdd(spawnQuads[neighborIndex], spawnMaxs, ex[3]);
		float absMins[3], absMaxs[3];
		Math_BoundsFromPoints(absMins, absMaxs, ex[0], 4);
		Math_VecSub(absMins, spot->net.pos, spawnMins);
		Math_VecSub(absMaxs, spot->net.pos, spawnMaxs);
	}

	/*
	float clr[3] = {1.0f, 0.0f, 0.0f};
	float absMins[3], absMaxs[3];
	Math_VecAdd(spot->net.pos, spawnMins, absMins);
	Math_VecAdd(spot->net.pos, spawnMaxs, absMaxs);
	Util_DebugBox(absMins, absMaxs, clr);
	*/

	int spawnTries;
	float pos[3];
	for (spawnTries = 0; spawnTries < 10; spawnTries++)
	{
		Math_VecCopy(spot->net.pos, pos);
		pos[0] += Util_RandFloat(spawnMins[0], spawnMaxs[0]);
		pos[1] += Util_RandFloat(spawnMins[1], spawnMaxs[1]);
		pos[2] += Util_RandFloat(spawnMins[2], spawnMaxs[2]);
		if (Util_TryObjectSpawn(pos, b, spot->net.pos))
		{
			break;
		}
	}

	if (spawnTries == 10)
	{ //wait 200ms and try again
		rscript->scriptTime = g_glb.gameTime + 200;
		return false;
	}

	float ang[3] = {0.0f, 0.0f, 0.0f};
	gameObject_t *pl = &g_gameObjects[0];
	if (pl->inuse && pl->plObj)
	{
		float d[3];
		Math_VecSub(pl->net.pos, pos, d);
		Math_VecToAngles(d, d);
		ang[YAW] = d[YAW];
	}
	gameObject_t *obj = LServ_ObjectFromName(objName, pos, ang, NULL, 0);
	if (obj)
	{
		obj->net.renderEffects2 |= FXFL2_SPAWN2;
		ObjSound_Create(obj->net.pos, "assets/sound/cb/espawn.wav", 1.0f, -1);
		ObjSound_Create(obj->net.pos, "assets/sound/cb/espawn.wav", 1.0f, -1);
	}
	return true;
}

//"objcreateenemymax" command
bool RSHan_ObjCreateEnemyMax(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	int maxEn = atoi(RScr_GetArgStr(node, rscript, 2));
	gameObject_t *cam = Util_GetCam(0);
	if (g_glb.ai.numActiveAI >= maxEn)
	{
		if (cam->inuse)
		{
			ObjSound_Create(cam->net.pos, "assets/sound/menu/menuerr.wav", 1.0f, -1);
		}
		return true;
	}

	if (cam->inuse)
	{
		ObjSound_Create(cam->net.pos, "assets/sound/cb/espawn.wav", 1.0f, -1);
	}

	return RSHan_ObjCreateEnemy(node, rscript, timeMod);
}

//"objcreatesneaky" command
bool RSHan_ObjCreateSneaky(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	float idealDist = (float)atof(RScr_GetArgStr(node, rscript, 2));
	gameObject_t *targ = Util_FindTargetObject("__the_camera");
	if (!targ)
	{
		return true;
	}
	char *objName = RScr_GetArgStr(node, rscript, 1);
	BYTE *b = Common_GetEntryForObject(objName);
	if (!b)
	{ //no entry for this object then
		return true;
	}

	float pos[3], ang[3];
	if (!Util_FindHiddenSpawn(targ, idealDist, b, pos, ang))
	{ //wait 200ms and try again
		rscript->scriptTime = g_glb.gameTime + 200;
		return false;
	}

	gameObject_t *obj = LServ_ObjectFromName(objName, pos, ang, NULL, 0);
	if (obj)
	{
		obj->targetName = Util_PooledString(RScr_GetArgStr(node, rscript, 0));
		Phys_PutOnGround(obj);
	}
	return true;
}

//"objcreateitem" command
bool RSHan_ObjCreateItem(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	float pos[3], ang[3];
	int itemDur = atoi(RScr_GetArgStr(node, rscript, 3));
	sscanf(RScr_GetArgStr(node, rscript, 1), "(%f %f %f)", &pos[0], &pos[1], &pos[2]);
	sscanf(RScr_GetArgStr(node, rscript, 2), "(%f %f %f)", &ang[0], &ang[1], &ang[2]);
	char *itemName = RScr_GetArgStr(node, rscript, 0);
	for (int i = 0; i < NUM_INVENTORY_ITEM_DEFS; i++)
	{
		const invItemDef_t *item = &g_invItems[i];
		if (item->name && !stricmp(item->name, itemName))
		{
			gameObject_t *item = LServ_ObjectFromName("obj_item_drop", pos, ang, NULL, 0);
			item->makoCount = i;
			item->debounce = (!itemDur) ? 0 : g_glb.gameTime + itemDur;
			return true;
		}
	}
	return true;
}

//"objcreateitemalt" command
bool RSHan_ObjCreateItemAlt(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	char *itemName = RScr_GetArgStr(node, rscript, 0);
	gameObject_t *pl = &g_gameObjects[0];
	if (pl->inuse && pl->plObj)
	{
		for (int i = 0; i < MAX_PLAYER_INVENTORY_ITEMS; i++)
		{
			playerInvItem_t *inv = &pl->plObj->plData.inventory[i];
			if (inv->itemQuantity > 0 && inv->itemIndex > 0)
			{
				if (!stricmp(g_invItems[inv->itemIndex].name, itemName))
				{
					itemName = RScr_GetArgStr(node, rscript, 4);
					break;
				}
			}
		}
	}

	float pos[3], ang[3];
	int itemDur = atoi(RScr_GetArgStr(node, rscript, 3));
	sscanf(RScr_GetArgStr(node, rscript, 1), "(%f %f %f)", &pos[0], &pos[1], &pos[2]);
	sscanf(RScr_GetArgStr(node, rscript, 2), "(%f %f %f)", &ang[0], &ang[1], &ang[2]);
	for (int i = 0; i < NUM_INVENTORY_ITEM_DEFS; i++)
	{
		const invItemDef_t *item = &g_invItems[i];
		if (item->name && !stricmp(item->name, itemName))
		{
			gameObject_t *item = LServ_ObjectFromName("obj_item_drop", pos, ang, NULL, 0);
			item->makoCount = i;
			item->debounce = (!itemDur) ? 0 : g_glb.gameTime + itemDur;
			return true;
		}
	}
	return true;
}

//"objremovetempitems" command
bool RSHan_ObjRemoveTempItems(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	int tempItemTypes[2] =
	{
		g_sharedFn->Common_ServerString("obj_item_drop"),
		g_sharedFn->Common_ServerString("obj_mako_drop")
	};
	for (int i = 0; i < g_glb.gameObjectSlots; i++)
	{
		gameObject_t *obj = &g_gameObjects[i];
		if (!obj->inuse)
		{
			continue;
		}
		if (obj->net.entNameIndex != tempItemTypes[0] &&
			obj->net.entNameIndex != tempItemTypes[1])
		{
			continue;
		}
		obj->think = ObjGeneral_RemoveThink;
		obj->thinkTime = g_glb.gameTime;
	}
	return true;
}

//"objremove" command
bool RSHan_ObjRemove(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *obj = Util_FindTargetObject(RScr_GetArgStr(node, rscript, 0));
	if (!obj)
	{
		return true;
	}

	obj->think = ObjGeneral_RemoveThink;
	obj->thinkTime = g_glb.gameTime;
	return true;
}

//"objactivate" command
bool RSHan_ObjActivate(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *obj = Util_FindTargetObject(RScr_GetArgStr(node, rscript, 0));
	if (!obj || !obj->activate)
	{
		return true;
	}

	obj->activate(obj, obj);
	return true;
}

//"objmovepos" command
bool RSHan_ObjMovePos(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *obj = Util_FindTargetObject(RScr_GetArgStr(node, rscript, 0));
	if (!obj)
	{
		return true;
	}
	float destPos[3];
	sscanf(RScr_GetArgStr(node, rscript, 1), "(%f %f %f)", &destPos[0], &destPos[1],
		&destPos[2]);

	float moveSpeed = (float)atof(RScr_GetArgStr(node, rscript, 2))*timeMod;
	float d[3];
	Math_VecSub(destPos, obj->net.pos, d);
	float dist = Math_VecNorm(d);
	if (moveSpeed > dist)
	{
		moveSpeed = dist;
	}
	obj->net.pos[0] += d[0]*moveSpeed;
	obj->net.pos[1] += d[1]*moveSpeed;
	obj->net.pos[2] += d[2]*moveSpeed;

	LServ_UpdateRClip(obj);

	return (dist <= moveSpeed);
}

//"objsetpos" command
bool RSHan_ObjSetPos(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *obj = Util_FindTargetObject(RScr_GetArgStr(node, rscript, 0));
	if (!obj)
	{
		return true;
	}
	sscanf(RScr_GetArgStr(node, rscript, 1), "(%f %f %f)", &obj->net.pos[0], &obj->net.pos[1],
		&obj->net.pos[2]);
	LServ_UpdateRClip(obj);
	return true;
}

//"objputground" command
bool RSHan_ObjPutGround(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *obj = Util_FindTargetObject(RScr_GetArgStr(node, rscript, 0));
	if (!obj)
	{
		return true;
	}
	Phys_PutOnGround(obj);
	Math_VecCopy(obj->net.pos, obj->safePos);
	return true;
}

//"objgivehealth" command
bool RSHan_ObjGiveHealth(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *obj = Util_FindTargetObject(RScr_GetArgStr(node, rscript, 0));
	if (!obj)
	{
		return true;
	}
	int amt = atoi(RScr_GetArgStr(node, rscript, 1));
	obj->health += amt;
	if (obj->plObj && obj->health > obj->plObj->plData.maxHealth)
	{
		obj->health = obj->plObj->plData.maxHealth;
	}
	if (obj->aiObj && obj->health > obj->aiObj->maxHealth)
	{
		obj->health = obj->aiObj->maxHealth;
	}
	return true;
}

//"objsettarget" command
bool RSHan_ObjSetTarget(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *obj = Util_FindTargetObject(RScr_GetArgStr(node, rscript, 0));
	if (!obj)
	{
		return true;
	}
	char *targStr = RScr_GetArgStr(node, rscript, 1);
	gameObject_t *targ = (targStr && targStr[0]) ? Util_FindTargetObject(targStr) : NULL;
	obj->target = targ;
	return true;
}

//"objsetdebounce" command
bool RSHan_ObjSetDebounce(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *obj = Util_FindTargetObject(RScr_GetArgStr(node, rscript, 0));
	if (!obj)
	{
		return true;
	}
	int dur = atoi(RScr_GetArgStr(node, rscript, 1));
	obj->debounce = g_glb.gameTime+dur;
	return true;
}

//"objmoveang" command
bool RSHan_ObjMoveAng(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *obj = Util_FindTargetObject(RScr_GetArgStr(node, rscript, 0));
	if (!obj)
	{
		return true;
	}
	float ang[3];
	sscanf(RScr_GetArgStr(node, rscript, 1), "(%f %f %f)", &ang[0], &ang[1],
		&ang[2]);
	float angleBlendScale = (float)atof(RScr_GetArgStr(node, rscript, 2));

	ang[0] = Math_AngleMod(ang[0]);
	ang[1] = Math_AngleMod(ang[1]);
	ang[2] = Math_AngleMod(ang[2]);
	obj->net.ang[PITCH] = Math_BlendAngle(obj->net.ang[PITCH], ang[PITCH], timeMod*angleBlendScale);
	obj->net.ang[YAW] = Math_BlendAngle(obj->net.ang[YAW], ang[YAW], timeMod*angleBlendScale);
	obj->net.ang[ROLL] = Math_BlendAngle(obj->net.ang[ROLL], ang[ROLL], timeMod*angleBlendScale);
	LServ_UpdateRClip(obj);
	if (fabsf(ang[0]-obj->net.ang[0]) > 5.0f ||
		fabsf(ang[1]-obj->net.ang[1]) > 5.0f ||
		fabsf(ang[2]-obj->net.ang[2]) > 5.0f)
	{
		return false;
	}
	return true;
}

//"objlookat" command
bool RSHan_ObjLookAt(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *obj = Util_FindTargetObject(RScr_GetArgStr(node, rscript, 0));
	if (!obj)
	{
		return true;
	}

	gameObject_t *targ = Util_FindTargetObject(RScr_GetArgStr(node, rscript, 1));
	if (!targ)
	{
		return true;
	}

	float ang[3];
	Math_VecSub(targ->net.pos, obj->net.pos, ang);
	ang[2] = 0.0f;
	Math_VecToAngles(ang, ang);

	float angleBlendScale = (float)atof(RScr_GetArgStr(node, rscript, 2));
	ang[0] = Math_AngleMod(ang[0]);
	ang[1] = Math_AngleMod(ang[1]);
	ang[2] = Math_AngleMod(ang[2]);
	obj->net.ang[PITCH] = Math_BlendAngleLinear(obj->net.ang[PITCH], ang[PITCH], timeMod*angleBlendScale);
	obj->net.ang[YAW] = Math_BlendAngleLinear(obj->net.ang[YAW], ang[YAW], timeMod*angleBlendScale);
	obj->net.ang[ROLL] = Math_BlendAngleLinear(obj->net.ang[ROLL], ang[ROLL], timeMod*angleBlendScale);
	LServ_UpdateRClip(obj);
	if (fabsf(ang[0]-obj->net.ang[0]) > 1.0f ||
		fabsf(ang[1]-obj->net.ang[1]) > 1.0f ||
		fabsf(ang[2]-obj->net.ang[2]) > 1.0f)
	{
		return false;
	}
	return true;
}

//"objsetang" command
bool RSHan_ObjSetAng(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *obj = Util_FindTargetObject(RScr_GetArgStr(node, rscript, 0));
	if (!obj)
	{
		return true;
	}
	sscanf(RScr_GetArgStr(node, rscript, 1), "(%f %f %f)", &obj->net.ang[0], &obj->net.ang[1],
		&obj->net.ang[2]);
	LServ_UpdateRClip(obj);
	return true;
}

//"objsound" command
bool RSHan_ObjSound(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *obj = Util_FindTargetObject(RScr_GetArgStr(node, rscript, 0));
	if (!obj)
	{
		return true;
	}
	ObjSound_Create(obj->net.pos, RScr_GetArgStr(node, rscript, 1), 1.0f, -1);
	return true;
}

//"objsoundrand" command
bool RSHan_ObjSoundRand(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	int r = atoi(RScr_GetArgStr(node, rscript, 2));
	if (rand()%100 < r)
	{
		return RSHan_ObjSound(node, rscript, timeMod);
	}
	return true;
}

//"objfx" command
bool RSHan_ObjFX(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *obj = Util_FindTargetObject(RScr_GetArgStr(node, rscript, 0));
	if (!obj)
	{
		return true;
	}

	float pos[3] = {0.0f, 0.0f, 0.0f};
	float ang[3] = {0.0f, 0.0f, 0.0f};
	char *posStr = RScr_GetArgStr(node, rscript, 2);
	if (posStr[0])
	{
		sscanf(posStr, "(%f %f %f)", &pos[0], &pos[1], &pos[2]);
	}
	char *angStr = RScr_GetArgStr(node, rscript, 3);
	if (angStr[0])
	{
		sscanf(angStr, "(%f %f %f)", &ang[0], &ang[1], &ang[2]);
	}

	char *boneStr = RScr_GetArgStr(node, rscript, 4);
	if (boneStr[0] && obj->rcColModel)
	{ //play on a bone
		modelMatrix_t boneMat;
		if (g_sharedFn->Coll_GetModelBoneMatrix(obj->rcColModel, boneStr, &boneMat))
		{
			float t[3];
			Math_VecCopy(pos, t);
			Math_TransformPointByMatrix(&boneMat, t, pos);

			Math_VecCopy(ang, t);
			Math_TransformPointByMatrix(&boneMat, t, ang);
			Math_VecSub(ang, boneMat.o, ang);
			Math_VecToAngles(ang, ang);
		}
	}

	ObjParticles_Create(RScr_GetArgStr(node, rscript, 1), pos, ang, -1);

	return true;
}

//"objfxbolted" command
bool RSHan_ObjFXBolted(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *obj = Util_FindTargetObject(RScr_GetArgStr(node, rscript, 0));
	if (!obj)
	{
		return true;
	}

	float pos[3] = {0.0f, 0.0f, 0.0f};
	float ang[3] = {0.0f, 0.0f, 0.0f};
	char *posStr = RScr_GetArgStr(node, rscript, 2);
	if (posStr[0])
	{
		sscanf(posStr, "(%f %f %f)", &pos[0], &pos[1], &pos[2]);
	}
	char *angStr = RScr_GetArgStr(node, rscript, 3);
	if (angStr[0])
	{
		sscanf(angStr, "(%f %f %f)", &ang[0], &ang[1], &ang[2]);
	}

	ObjParticles_Create(RScr_GetArgStr(node, rscript, 1), pos, ang, obj->net.index);

	return true;
}

//"objrfxsetbit" command
bool RSHan_ObjRFXSetBit(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *obj = Util_FindTargetObject(RScr_GetArgStr(node, rscript, 0));
	if (!obj)
	{
		return true;
	}

	obj->net.renderEffects |= (1<<atoi(RScr_GetArgStr(node, rscript, 1)));

	return true;
}

//"objrfxunsetbit" command
bool RSHan_ObjRFXUnsetBit(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *obj = Util_FindTargetObject(RScr_GetArgStr(node, rscript, 0));
	if (!obj)
	{
		return true;
	}

	obj->net.renderEffects &= ~(1<<atoi(RScr_GetArgStr(node, rscript, 1)));

	return true;
}

//"objrfx2setbit" command
bool RSHan_ObjRFX2SetBit(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *obj = Util_FindTargetObject(RScr_GetArgStr(node, rscript, 0));
	if (!obj)
	{
		return true;
	}

	obj->net.renderEffects2 |= (1<<atoi(RScr_GetArgStr(node, rscript, 1)));

	return true;
}

//"objrfx2unsetbit" command
bool RSHan_ObjRFX2UnsetBit(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *obj = Util_FindTargetObject(RScr_GetArgStr(node, rscript, 0));
	if (!obj)
	{
		return true;
	}

	obj->net.renderEffects2 &= ~(1<<atoi(RScr_GetArgStr(node, rscript, 1)));

	return true;
}

//"objsetanim" command
bool RSHan_ObjSetAnim(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *obj = Util_FindTargetObject(RScr_GetArgStr(node, rscript, 0));
	if (!obj || !obj->scriptAnims)
	{
		return true;
	}

	for (int i = 0; obj->scriptAnims[i].animName; i++)
	{
		scriptableAnim_t *a = &obj->scriptAnims[i];
		if (!stricmp(a->animName, RScr_GetArgStr(node, rscript, 1)))
		{
			AI_StartAnim(obj, a->animNum, true);
			break;
		}
	}

	return true;
}

//"objsetanimnum" command
bool RSHan_ObjSetAnimNum(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *obj = Util_FindTargetObject(RScr_GetArgStr(node, rscript, 0));
	if (!obj)
	{
		return true;
	}

	int animNum = atoi(RScr_GetArgStr(node, rscript, 1));
	if (animNum < 0 || animNum >= obj->numAnims)
	{
		return true;
	}

	AI_StartAnim(obj, animNum, true);

	return true;
}

//"objsetaigoal" command
bool RSHan_ObjSetAIGoal(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *obj = Util_FindTargetObject(RScr_GetArgStr(node, rscript, 0));
	if (!obj || !obj->aiObj)
	{
		return true;
	}

	char *goalStr = RScr_GetArgStr(node, rscript, 1);
	if (!goalStr[0])
	{
		obj->aiObj->scriptGoal = NULL;
		obj->aiObj->scriptGoalRange = 0.0f;
		return true;
	}

	gameObject_t *goal = Util_FindTargetObject(goalStr);
	if (!goal)
	{
		return true;
	}

	obj->aiObj->scriptGoal = goal;
	obj->aiObj->scriptGoalRange = (float)atof(RScr_GetArgStr(node, rscript, 2));
	return true;
}

//"objsetaienemy" command
bool RSHan_ObjSetAIEnemy(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *obj = Util_FindTargetObject(RScr_GetArgStr(node, rscript, 0));
	if (!obj || !obj->aiObj)
	{
		return true;
	}

	char *goalStr = RScr_GetArgStr(node, rscript, 1);
	if (!goalStr[0])
	{
		return true;
	}

	gameObject_t *goal = Util_FindTargetObject(goalStr);
	if (!goal)
	{
		return true;
	}

	obj->aiObj->enemy = goal;
	obj->aiObj->goalObj = goal;
	return true;
}

//"objsetailookgoal" command
bool RSHan_ObjSetAILookGoal(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *obj = Util_FindTargetObject(RScr_GetArgStr(node, rscript, 0));
	if (!obj || !obj->aiObj)
	{
		return true;
	}

	char *goalStr = RScr_GetArgStr(node, rscript, 1);
	if (!goalStr[0])
	{
		obj->aiObj->scriptLookGoal = NULL;
		return true;
	}

	gameObject_t *goal = Util_FindTargetObject(goalStr);
	if (!goal)
	{
		return true;
	}

	obj->aiObj->scriptLookGoal = goal;
	return true;
}

//"objsetaimovespeed" command
bool RSHan_ObjSetAIMoveSpeed(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *obj = Util_FindTargetObject(RScr_GetArgStr(node, rscript, 0));
	if (!obj || !obj->aiObj)
	{
		return true;
	}

	obj->aiObj->moveSpeed = (float)atof(RScr_GetArgStr(node, rscript, 1));
	return true;
}

//"objtalkbubble" command
bool RSHan_ObjTalkBubble(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *obj = Util_FindTargetObject(RScr_GetArgStr(node, rscript, 0));
	if (!obj)
	{
		return true;
	}

	char *talkName = RScr_GetArgStr(node, rscript, 1);
	char *talkText = RScr_GetArgStr(node, rscript, 2);
	int duration = atoi(RScr_GetArgStr(node, rscript, 3));
	if (duration > 0)
	{
		obj->noCullTime = g_glb.gameTime+duration+2000;
	}
	int l1 = (int)strlen(talkName)+1;
	int l2 = (int)strlen(talkText)+1;
	BYTE *msg = (BYTE *)_alloca(l1+l2+sizeof(int)*2);
	memcpy(msg, &obj->net.index, sizeof(int));
	memcpy(msg+sizeof(int), &duration, sizeof(int));
	memcpy(msg+sizeof(int)*2, talkName, l1);
	memcpy(msg+sizeof(int)*2+l1, talkText, l2);
	g_sharedFn->Net_SendEventType(CLEV_TALKBUBBLE, msg, l1+l2+sizeof(int)*2, -1);
	return true;
}

//"objdamage" command
bool RSHan_ObjDamage(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *obj = Util_FindTargetObject(RScr_GetArgStr(node, rscript, 0));
	if (!obj)
	{
		return true;
	}

	Util_DamageObject(obj, obj, atoi(RScr_GetArgStr(node, rscript, 1)));
	return true;
}

//"objsetlight" command
bool RSHan_ObjSetLight(rscriptNode_t *node, rscript_t *rscript, float timeMod)
{
	gameObject_t *obj = Util_FindTargetObject(RScr_GetArgStr(node, rscript, 0));
	if (!obj || obj->net.type != OBJ_TYPE_LIGHT)
	{
		return true;
	}

	char *clr = RScr_GetArgStr(node, rscript, 1);
	sscanf(clr, "(%f %f %f)", &obj->net.mins[0], &obj->net.mins[1], &obj->net.mins[2]);
	return true;
}
