/*
=============================================================================
Module Information
------------------
Name:			saveload.cpp
Author:			Rich Whitehouse
Description:	handle saving/loading of game state and session data
=============================================================================
*/

#include "main.h"
#include "ai.h"
#include "rscript.h"

extern void LServ_PostMapSpawn(void);

//savegame from a client
void LServ_ClientSaveGame(int clientIndex, BYTE *saveData, int saveSize)
{
	if (saveSize < sizeof(saveHeader_t))
	{
		Util_StatusMessage("Client sent a malformed save header.");
		return;
	}
	saveHeader_t *hdr = (saveHeader_t *)saveData;
	if (hdr->version != SAVE_GAME_VERSION)
	{
		Util_StatusMessage("Client sent a bad savegame version (%i).", hdr->version);
		return;
	}
	saveSize -= sizeof(saveHeader_t);
	if (saveSize != hdr->dataSize)
	{
		Util_StatusMessage("Client sent corrupted save data (%i != %i).", saveSize, hdr->dataSize);
		return;
	}

	cntStream_t *st = Stream_Alloc(hdr+1, hdr->dataSize);

	ObjPlayer_ReadGameData(&g_gameObjects[clientIndex], st);
	for (int i = 1; i < MAX_NET_CLIENTS; i++)
	{
		ObjPlayer_ReadGameData(NULL, st);
	}
	gameObject_t *clObj = &g_gameObjects[clientIndex];
	int c = Stream_ReadInt(st);
	for (int i = 0; i < c; i++)
	{
		char gvName[MAX_GVAR_LEN];
		char gvVal[MAX_GVAR_LEN];
		Stream_ReadString(st, gvName, MAX_GVAR_LEN);
		Stream_ReadString(st, gvVal, MAX_GVAR_LEN);
		if (!stricmp(gvName, "useddevcmd") && atoi(gvVal))
		{
			Util_StatusMessage("Connecting client is using a developer save.");
			if (clObj->plObj)
			{
				clObj->plObj->clCheater = true;
			}
		}
		else if (!stricmp(gvName, "cdt_usingmod") && atoi(gvVal))
		{
			Util_StatusMessage("Connecting client is using a mod save.");
			if (clObj->plObj)
			{
				clObj->plObj->clCheater = true;
			}
		}
	}

	Stream_Free(st);
}

//fill up a stream with persistent game data
void LServ_WriteGameStream(cntStream_t *st)
{
	for (int i = 0; i < MAX_NET_CLIENTS; i++)
	{
		ObjPlayer_WriteGameData(&g_gameObjects[i], st);
	}
	GVar_WriteGVars(st);
}

//restore game data from stream
void LServ_ReadGameStream(cntStream_t *st)
{
	for (int i = 0; i < MAX_NET_CLIENTS; i++)
	{
		ObjPlayer_ReadGameData(&g_gameObjects[i], st);
	}
	GVar_ReadGVars(st);

	//make ai weak (only in terms of logic) until after chocobo wonderland becomes accessable
	g_glb.ai.weakAI = !GVar_GetInt("lshub_gotchoco");
}

//save game
void LServ_SaveGame(const char *saveName, const char *modName)
{
	int fh = g_sharedFn->FileSys_OpenFile(saveName, _O_WRONLY|_O_BINARY|_O_CREAT, _S_IWRITE);
	if (fh == -1)
	{
		return;
	}

	if (modName && modName[0])
	{
		GVar_SetInt("cdt_usingmod", 1);
	}

	cntStream_t *st = Stream_Alloc(NULL, 8192);
	LServ_WriteGameStream(st);

	//write out values now in case we somehow die before changing maps
	g_sharedFn->Common_SaveGameValues(Stream_Buffer(st), Stream_Size(st));

	saveHeader_t hdr;
	memset(&hdr, 0, sizeof(hdr));
	hdr.version = SAVE_GAME_VERSION;
	hdr.dataSize = Stream_Size(st);
	hdr.timeSec = GVar_GetInt("playsec");
	strcpy(hdr.map, g_glb.mapName);
	gameObject_t *pl = &g_gameObjects[0];
	if (!pl->inuse || !pl->plObj)
	{
		strcpy(hdr.desc, "Unknown");
	}
	else
	{
		sprintf(hdr.desc, "\nHP*(1 1 1 1):*(-1 -1 -1 1) %i*(1 1 1 1)/*(-1 -1 -1 1)%i\n"
			"STR*(1 1 1 1)/*(-1 -1 -1 1)DEF*(1 1 1 1)/*(-1 -1 -1 1)DEX*(1 1 1 1)/*(-1 -1 -1 1)LUCK*(1 1 1 1): "
			"*(-1 -1 -1 1)%i*(1 1 1 1)/*(-1 -1 -1 1)%i*(1 1 1 1)/*(-1 -1 -1 1)%i*(1 1 1 1)/*(-1 -1 -1 1)%i\n"
			"Mako*(1 1 1 1):*(-1 -1 -1 1) %i\n",
			pl->health, pl->plObj->plData.maxHealth,
			pl->plObj->plData.statStr, pl->plObj->plData.statDef, pl->plObj->plData.statDex,
			pl->plObj->plData.statLuck, pl->plObj->makoStash);
	}
	g_sharedFn->FileSys_Write(fh, &hdr, sizeof(hdr));
	g_sharedFn->FileSys_Write(fh, Stream_Buffer(st), hdr.dataSize);
	Stream_Free(st);
	g_sharedFn->FileSys_CloseFile(fh);
}

//load game
void LServ_LoadGame(const char *saveName)
{
	int fh = g_sharedFn->FileSys_OpenFile(saveName, _O_RDONLY|_O_BINARY, _S_IREAD);
	if (fh == -1)
	{
		return;
	}

	saveHeader_t hdr;
	g_sharedFn->FileSys_Read(fh, &hdr, sizeof(hdr));
	if (hdr.version != SAVE_GAME_VERSION)
	{
		Util_StatusMessage("Bad savegame version %i.", hdr.version);
		return;
	}

	unsigned char *data = (unsigned char *)g_sharedFn->Common_RCMalloc(hdr.dataSize);
	g_sharedFn->FileSys_Read(fh, data, hdr.dataSize);
	cntStream_t *st = Stream_Alloc(data, hdr.dataSize);
	LServ_ReadGameStream(st);

	g_sharedFn->Common_SaveGameValues(Stream_Buffer(st), Stream_Size(st));

	Stream_Free(st);
	g_sharedFn->Common_RCFree(data);
}


typedef enum
{
	SESSPTR_CLIPMODEL,
	SESSPTR_PLAYER,
	SESSPTR_AI,
	SESSPTR_CIN,
	SESSPTR_RSCRIPT,
	SESSPTR_RIGID,
	SESSPTR_CUSTOM,
	NUM_SESS_PTRS
} sessPtrType_e;

typedef struct sessFnPtr_s
{
	void			*ptr;
} sessFnPtr_t;
static sessFnPtr_t g_sessFnPtrs[] =
{
	{ObjGeneral_RemoveThink},
	{ObjGeneral_Think},
	{AI_DropMako},
	{AI_DropItem},
	{AI_GenericThink},
	{AI_GenericTouch},
	{AI_GenericPain},
	{AI_GenericDeath},
	{AI_ImmortalDeath},

	{ObjAerith_Think},
	{ObjBarForge_Think},
	{ObjBarrett_Think},
	{ObjBeam_Think},
	{ObjBomb_Think},
	{ObjCCon_Think},
	{ObjChoco_Think},
	{ObjCinProp_Think},
	{ObjCloud_Think},
	{ObjE30_Think},
	{ObjEat_Think},
	{ObjForceField_Think},
	{ObjGig_Think},
	{ObjHorn_Think},
	{ObjDog_Think},
	{ObjIron_Think},
	{ObjItemDrop_Think},
	{ObjJenova_Think},
	{ObjLift_Think},
	{ObjLight_Think},
	{ObjMad_Think},
	{ObjMadBall_Think},
	{ObjMako_Think},
	{ObjMover_Think},
	{ObjPartEmit_Think},
	{ObjParticles_Think},
	{ObjPlBar_Think},
	{ObjVortex_Think},
	{ObjProjectile_Think},
	{ObjRigidBody_Think},
	{ObjRSRunner_Think},
	{ObjSephFin_Think},
	{ObjSephHair_Think},
	{ObjSound_Think},
	{ObjSoundEmit_Think},
	{ObjSupernova_Think},
	{ObjTon_Think},
	{ObjTriggerBox_Think},
	{ObjCam_Think},
	{ObjPlayer_Think},

	{ObjBomb_Touch},
	{ObjCamMarker_Trigger},
	{ObjE30_Touch},
	{ObjEat_Touch},
	{ObjGig_Touch},
	{ObjHorn_Touch},
	{ObjDog_Touch},
	{ObjIron_Touch},
	{ObjMad_Touch},
	{ObjMadBall_Touch},
	{ObjPlayer_Touch},
	{ObjProjectile_Touch},
	{ObjSephFin_Touch},
	{ObjTon_Touch},

	{ObjCCon_Pain},

	{ObjBomb_Death},
	{ObjCCon_Death},
	{ObjJenova_Death},
	{ObjPlayer_Death},

	{ObjBeam_Activate},
	{ObjForceField_Activate},
	{ObjPartEmit_Activate},
	{ObjSoundEmit_Activate},
	{ObjTriggerBox_Activate},

	{NULL}
};

#define SESS_SAVE(a) Stream_WriteBytes(st, &a, sizeof(a))

//write game object pointer
static void LServ_WriteGameObjPtr(cntStream_t *st, gameObject_t *obj)
{
	if (!obj || !obj->inuse)
	{
		Stream_WriteInt(st, -1);
		return;
	}
	Stream_WriteInt(st, obj->net.index);
}

//write inventory item
static void LServ_WriteInvItem(cntStream_t *st, const invItemDef_t *item)
{
	if (!item)
	{
		Stream_WriteInt(st, -1);
		return;
	}
	int n = (int)(item-g_invItems);
	Stream_WriteInt(st, n);
}

//write session globals
static void LServ_WriteSessGlobals(cntStream_t *st)
{
	SESS_SAVE(g_glb.ai.numActiveAI);
	SESS_SAVE(g_glb.ai.numActiveEnemyAI);
	SESS_SAVE(g_glb.ai.playerCloseDist);
	SESS_SAVE(g_glb.ai.numCloseToPlayer);
	SESS_SAVE(g_glb.ai.numWantClose);
	LServ_WriteGameObjPtr(st, g_glb.ai.closestToPlayer);
	SESS_SAVE(g_glb.ai.playerLastHurtTime);
	SESS_SAVE(g_glb.ai.playerLastHitIndex);
	SESS_SAVE(g_glb.ai.playerLastHitTime);
	SESS_SAVE(g_glb.ai.inChaos);
	SESS_SAVE(g_glb.ai.showPaths);
	SESS_SAVE(g_glb.ai.weakAI);

	SESS_SAVE(g_glb.aiPathTime);
	SESS_SAVE(g_glb.gameTime);
	SESS_SAVE(g_glb.timeFactor);
	SESS_SAVE(g_glb.timeScale);
	SESS_SAVE(g_glb.invTimeScale);
	SESS_SAVE(g_glb.timeType);
	SESS_SAVE(g_glb.lastTimeType);
	SESS_SAVE(g_glb.actionTime);
	SESS_SAVE(g_glb.actionTimeScale);
	SESS_SAVE(g_glb.tonberryTime);
	SESS_SAVE(g_glb.enSpawnBlock);
	SESS_SAVE(g_glb.noDeath);
	SESS_SAVE(g_glb.cinema);
	SESS_SAVE(g_glb.debugRigid);
	SESS_SAVE(g_glb.mapName);
	SESS_SAVE(g_glb.musicStrIndex);
	SESS_SAVE(g_glb.levelMusic);
	SESS_SAVE(g_glb.battleTime);
	SESS_SAVE(g_glb.battleMusicEnabled);
	SESS_SAVE(g_glb.battleMusic);
	SESS_SAVE(g_glb.chocoboMusic);
	SESS_SAVE(g_glb.magicMode);

	SESS_SAVE(g_glb.endlessBattle);

	SESS_SAVE(g_glb.mpVersusMode);
	SESS_SAVE(g_glb.mpCountDown);
	SESS_SAVE(g_glb.mpCountTime);
	SESS_SAVE(g_glb.runningMultiplayer);
	SESS_SAVE(g_glb.initialFPSPos);
	SESS_SAVE(g_glb.initialFPSAng);
	SESS_SAVE(g_glb.camFrustum);
	SESS_SAVE(g_glb.wideCamFrustum);
	SESS_SAVE(g_glb.camFrustumFresh);
	SESS_SAVE(g_glb.camBoneBolt);
	SESS_SAVE(g_glb.camBoltOffset);
	SESS_SAVE(g_glb.camAbsOffset);
	SESS_SAVE(g_glb.camAbsAngOffset);
	SESS_SAVE(g_glb.camSlack);
	SESS_SAVE(g_glb.worldMins);
	SESS_SAVE(g_glb.worldMaxs);
	SESS_SAVE(g_glb.gameObjectSlots);
	SESS_SAVE(g_glb.creationCount);
	SESS_SAVE(g_glb.clientInfo);
	SESS_SAVE(g_glb.numChocoSpots);
	SESS_SAVE(g_glb.scriptScreenFade);
	SESS_SAVE(g_glb.scriptPicFade);
	SESS_SAVE(g_glb.scriptFogValues);
	SESS_SAVE(g_glb.scriptFogBlendMode);
	SESS_SAVE(g_glb.changeToMap);
	SESS_SAVE(g_glb.scheduledMapChange);
}

//write session object function pointer
void LServ_WriteFuncPtr(cntStream_t *st, gameObject_t *obj, void *ptr)
{
	if (!ptr)
	{
		Stream_WriteInt(st, -1);
		return;
	}
	for (int i = 0; g_sessFnPtrs[i].ptr; i++)
	{
		if (g_sessFnPtrs[i].ptr == ptr)
		{
			Stream_WriteInt(st, i);
			return;
		}
	}

	RCUBE_ASSERT(!"Object pointer not in save-load session table.");
	Stream_WriteInt(st, -1);
}

//write session object pointer data
void LServ_WriteSessObjPtr(cntStream_t *st, gameObject_t *obj, void *ptr, int type)
{
	if (!ptr)
	{
		Stream_WriteInt(st, 0);
		return;
	}

	Stream_WriteInt(st, 1);
	if (type == SESSPTR_CLIPMODEL)
	{
		rcColModel_t *o = (rcColModel_t *)ptr;
		SESS_SAVE(o->frame);
		SESS_SAVE(o->blendFrame);
		SESS_SAVE(o->lerpAmount);
		SESS_SAVE(o->pos);
		SESS_SAVE(o->ang);
		SESS_SAVE(o->modelScale);
		SESS_SAVE(o->mins);
		SESS_SAVE(o->maxs);
		SESS_SAVE(o->radius);
		SESS_SAVE(o->gameOwner);
		SESS_SAVE(o->gameIgnore);
		SESS_SAVE(o->solid);
		SESS_SAVE(o->clipFlags);
	}
	else if (type == SESSPTR_PLAYER)
	{
		playerObject_t *o = (playerObject_t *)ptr;
		SESS_SAVE(o->clButtons);
		SESS_SAVE(o->clButtonsBackBuffer);
		SESS_SAVE(o->numButtonsBuffered);
		SESS_SAVE(o->clAngles);
		SESS_SAVE(o->clAnalog);
		SESS_SAVE(o->hasClAnalog);
		SESS_SAVE(o->lastClAngles);
		SESS_SAVE(o->hasLastClAngles);
		SESS_SAVE(o->playerSpawnSeq);
		SESS_SAVE(o->chainNum);
		SESS_SAVE(o->nextChain);
		LServ_WriteGameObjPtr(st, o->targetObj);
		SESS_SAVE(o->healthUpdateTime);
		SESS_SAVE(o->lungeTime);
		SESS_SAVE(o->lastDmg);
		SESS_SAVE(o->hasLastDmg);
		SESS_SAVE(o->noClip);
		SESS_SAVE(o->invincible);
		SESS_SAVE(o->noTarget);
		SESS_SAVE(o->makoCharges);
		SESS_SAVE(o->makoStash);
		SESS_SAVE(o->chargeMeter);
		SESS_SAVE(o->chargeDrain);
		SESS_SAVE(o->chargeDrainTime);
		SESS_SAVE(o->chargeToMakoPower);
		SESS_SAVE(o->timeAttackTime);
		SESS_SAVE(o->fistOfFlames);
		SESS_SAVE(o->explosiveBlows);
		SESS_SAVE(o->healingAura);
		SESS_SAVE(o->healingTime);
		SESS_SAVE(o->magicHealTime);
		SESS_SAVE(o->desiredItem);
		SESS_SAVE(o->equippedItem);
		SESS_SAVE(o->waitOnInput);
		SESS_SAVE(o->moveSpeedScale);
		LServ_WriteGameObjPtr(st, o->camTarget);
		SESS_SAVE(o->camRange);
		SESS_SAVE(o->jumpUseTime);
		SESS_SAVE(o->canJumpUse);
		SESS_SAVE(o->jumpUseObj);
		SESS_SAVE(o->hasteTime);

		LServ_WriteGameObjPtr(st, o->gun);
		LServ_WriteInvItem(st, o->gunItem);
		SESS_SAVE(o->gunFireTime);
		SESS_SAVE(o->gunFirePose);
		LServ_WriteGameObjPtr(st, o->sword);
		LServ_WriteInvItem(st, o->swordItem);
		LServ_WriteGameObjPtr(st, o->flashlight);
		LServ_WriteInvItem(st, o->flashlightItem);
		LServ_WriteGameObjPtr(st, o->ride);
		LServ_WriteGameObjPtr(st, o->aerith);
		SESS_SAVE(o->noItemTime);
		SESS_SAVE(o->plData);
		SESS_SAVE(o->camSeeTime);
		SESS_SAVE(o->camTeleWarn);
		SESS_SAVE(o->clCheater);
		SESS_SAVE(o->plSeqDmg);
		SESS_SAVE(o->numPlSeqDmg);
	}
	else if (type == SESSPTR_AI)
	{
		aiObject_t *o = (aiObject_t *)ptr;
		SESS_SAVE(o->maxHealth);
		SESS_SAVE(o->moveSpeed);
		SESS_SAVE(o->lookYaw);
		SESS_SAVE(o->lookPitch);
		SESS_SAVE(o->fly);
		SESS_SAVE(o->painChance);
		SESS_SAVE(o->forceGravity);
		SESS_SAVE(o->combatRange);
		SESS_SAVE(o->goalRange);
		SESS_SAVE(o->goalPos);
		LServ_WriteGameObjPtr(st, o->goalObj);
		SESS_SAVE(o->distToGoal);
		SESS_SAVE(o->obstructTime);
		SESS_SAVE(o->makoValue);
		SESS_SAVE(o->aiLevel);
		SESS_SAVE(o->aiFov);
		SESS_SAVE(o->aiRange);
		LServ_WriteGameObjPtr(st, o->scriptGoal);
		LServ_WriteGameObjPtr(st, o->scriptLookGoal);
		SESS_SAVE(o->scriptGoalRange);
		LServ_WriteGameObjPtr(st, o->enemy);
		SESS_SAVE(o->distToEnemy);
		SESS_SAVE(o->attackTime);
		SESS_SAVE(o->noMoveTime);
		SESS_SAVE(o->confuseTime);
		SESS_SAVE(o->lastDmg);
		SESS_SAVE(o->hasLastDmg);
		SESS_SAVE(o->nonActive);
		SESS_SAVE(o->boneDmgTime);
		SESS_SAVE(o->boneDmgDir);
	}
	else if (type == SESSPTR_CIN)
	{
		cinObject_t *o = (cinObject_t *)ptr;
		SESS_SAVE(o->animStub);
		SESS_SAVE(o->hideable);
	}
	else if (type == SESSPTR_RSCRIPT)
	{
		rscript_t *o = (rscript_t *)ptr;
		Stream_WriteString(st, o->curScript->name);

		rscriptNode_t *nodeBase;
		if (o->curLabel)
		{
			Stream_WriteInt(st, 1);
			Stream_WriteString(st, o->curLabel->name);
			nodeBase = o->curLabel->node;
		}
		else
		{
			Stream_WriteInt(st, 0);
			nodeBase = o->curScript->nodes;
		}

		int curNum = -1;
		if (o->curNode)
		{
			int num = 0;
			for (rscriptNode_t *n = nodeBase; n; n = n->next)
			{
				if (n == o->curNode)
				{
					curNum = num;
				}
				num++;
			}
		}
		Stream_WriteInt(st, curNum);

		if (o->retLabel)
		{
			Stream_WriteInt(st, 1);
			Stream_WriteString(st, o->retLabel->name);
			nodeBase = o->retLabel->node;
		}
		else
		{
			Stream_WriteInt(st, 0);
			nodeBase = o->curScript->nodes;
		}

		int retNum = -1;
		if (o->retNode)
		{
			int num = 0;
			for (rscriptNode_t *n = nodeBase; n; n = n->next)
			{
				if (n == o->retNode)
				{
					retNum = num;
				}
				num++;
			}
		}
		Stream_WriteInt(st, retNum);

		SESS_SAVE(o->scriptTime);
		SESS_SAVE(o->ropRegs);
		SESS_SAVE(o->ropIntRegs);
		SESS_SAVE(o->ropStackPtr);
		for (int i = MAX_RSCRIPT_STACK-1; i > o->ropStackPtr; i--)
		{
			Stream_WriteString(st, o->ropStack[i]);
		}
	}
	else if (type == SESSPTR_RIGID)
	{
		rigidBody_t *o = (rigidBody_t *)ptr;
		SESS_SAVE(o->mat);
		SESS_SAVE(o->invMat);
		SESS_SAVE(o->torque);
		SESS_SAVE(o->gravity);
	}
	else if (type == SESSPTR_CUSTOM)
	{
		if (obj->sessSave)
		{
			obj->sessSave(obj, st);
		}
	}
}

//write out a game object
void LServ_WriteGameObj(cntStream_t *st, gameObject_t *obj)
{
	if (!obj || !obj->inuse)
	{
		Stream_WriteInt(st, -1);
		return;
	}
	
	Stream_WriteInt(st, obj->inuse);
	char *objName = g_sharedFn->Common_ServerStringFromIndex(obj->net.entNameIndex);
	RCUBE_ASSERT(objName && objName[0]);
	Stream_WriteString(st, objName);

	Stream_WriteInt(st, obj->numSpawnArgs);
	for (int i = 0; i < obj->numSpawnArgs; i++)
	{
		const objArgs_t *arg = obj->spawnArgs+i;
		Stream_WriteString(st, arg->key);
		Stream_WriteString(st, arg->val);
	}

	SESS_SAVE(obj->net);
	SESS_SAVE(obj->thinkTime);
	SESS_SAVE(obj->spawnMins);
	SESS_SAVE(obj->spawnMaxs);
	SESS_SAVE(obj->creationCount);
	SESS_SAVE(obj->hurtable);
	SESS_SAVE(obj->health);
	SESS_SAVE(obj->curAnim);
	SESS_SAVE(obj->animTime);
	SESS_SAVE(obj->animStartTime);
	SESS_SAVE(obj->animRestart);
	SESS_SAVE(obj->animNeverInterrupt);
	SESS_SAVE(obj->seqDmg);
	SESS_SAVE(obj->atkLastSeq);
	SESS_SAVE(obj->atkLastTime);
	SESS_SAVE(obj->atkType);
	SESS_SAVE(obj->atkRadiusCore);
	SESS_SAVE(obj->atkRadiusPos);
	SESS_SAVE(obj->meteorAttack);
	SESS_SAVE(obj->isOnFire);
	SESS_SAVE(obj->fireBurnTime);
	SESS_SAVE(obj->makoCount);
	SESS_SAVE(obj->localTimeScale);
	SESS_SAVE(obj->alwaysSend);
	SESS_SAVE(obj->noCullTime);
	SESS_SAVE(obj->nonSolidTime);
	SESS_SAVE(obj->preDeathTime);
	SESS_SAVE(obj->postSpawnTime);
	SESS_SAVE(obj->noGravTime);
	SESS_SAVE(obj->knockTime);
	SESS_SAVE(obj->toucher);
	SESS_SAVE(obj->radius);
	SESS_SAVE(obj->canPush);
	SESS_SAVE(obj->canBePushed);
	SESS_SAVE(obj->debounce);
	SESS_SAVE(obj->debounce2);
	SESS_SAVE(obj->debounce3);
	SESS_SAVE(obj->debounce4);
	SESS_SAVE(obj->debounce5);
	SESS_SAVE(obj->debounce6);
	SESS_SAVE(obj->debounce7);
	SESS_SAVE(obj->swordTime);
	SESS_SAVE(obj->generalFlag);
	SESS_SAVE(obj->generalFlag2);
	SESS_SAVE(obj->generalFlag3);
	SESS_SAVE(obj->dualAnimCnt);
	SESS_SAVE(obj->spareVec);
	SESS_SAVE(obj->spawnLife);
	SESS_SAVE(obj->localFlags);
	SESS_SAVE(obj->bounceFactor);
	SESS_SAVE(obj->onGround);
	SESS_SAVE(obj->groundObj);
	SESS_SAVE(obj->camTrackPos);
	SESS_SAVE(obj->groundHugger);
	SESS_SAVE(obj->preImpactVel);
	SESS_SAVE(obj->projVelAng);
	SESS_SAVE(obj->projVelocity);
	SESS_SAVE(obj->projFriction);
	SESS_SAVE(obj->projDamage);
	SESS_SAVE(obj->safePos);
	SESS_SAVE(obj->localRecursionTick);
	SESS_SAVE(obj->localAngles);
	SESS_SAVE(obj->customDynSize);
	LServ_WriteGameObjPtr(st, obj->enemy);
	LServ_WriteGameObjPtr(st, obj->target);
	LServ_WriteGameObjPtr(st, obj->child);
	LServ_WriteGameObjPtr(st, obj->parent);
	LServ_WriteGameObjPtr(st, obj->chain);
	LServ_WriteGameObjPtr(st, obj->chain2);
	LServ_WriteGameObjPtr(st, obj->physModelOwner);

	LServ_WriteFuncPtr(st, obj, obj->think);
	LServ_WriteFuncPtr(st, obj, obj->touch);
	LServ_WriteFuncPtr(st, obj, obj->pain);
	LServ_WriteFuncPtr(st, obj, obj->death);
	LServ_WriteFuncPtr(st, obj, obj->activate);

	LServ_WriteSessObjPtr(st, obj, obj->rcColModel, SESSPTR_CLIPMODEL);
	LServ_WriteSessObjPtr(st, obj, obj->plObj, SESSPTR_PLAYER);
	LServ_WriteSessObjPtr(st, obj, obj->aiObj, SESSPTR_AI);
	LServ_WriteSessObjPtr(st, obj, obj->cinObj, SESSPTR_CIN);
	LServ_WriteSessObjPtr(st, obj, obj->rscript, SESSPTR_RSCRIPT);
	LServ_WriteSessObjPtr(st, obj, obj->rigid, SESSPTR_RIGID);
	LServ_WriteSessObjPtr(st, obj, obj->customDynamic, SESSPTR_CUSTOM);
}

//write out the complete session data
void LServ_SaveSession(const char *saveName)
{
	int fh = g_sharedFn->FileSys_OpenFile(saveName, _O_WRONLY|_O_BINARY|_O_CREAT, _S_IWRITE);
	if (fh == -1)
	{
		return;
	}

	sessHeader_t hdr;
	hdr.version = SAVE_SESSION_VERSION;
	strcpy(hdr.map, g_glb.mapName);

	cntStream_t *st = Stream_Alloc(NULL, 8192);
	int schk = (int)RScr_GetScriptChk();
	Stream_WriteInt(st, schk);

	GVar_WriteGVars(st);

	//write server strings
	for (int i = 0; i < MAX_GAME_STRINGS; i++)
	{
		char *s = g_sharedFn->Common_ServerStringFromIndex(i);
		if (!s)
		{
			s = "";
		}
		Stream_WriteString(st, s);
	}

	//write game globals
	LServ_WriteSessGlobals(st);

	//write all game objects
	for (int i = 0; i < g_glb.gameObjectSlots; i++)
	{
		LServ_WriteGameObj(st, &g_gameObjects[i]);
	}

	hdr.dataSize = Stream_Size(st);
	g_sharedFn->FileSys_Write(fh, &hdr, sizeof(hdr));
	g_sharedFn->FileSys_Write(fh, Stream_Buffer(st), hdr.dataSize);
	Stream_Free(st);
	g_sharedFn->FileSys_CloseFile(fh);
}

#define SESS_LOAD(a) Stream_ReadBytes(st, &a, sizeof(a))

//read game object pointer
static gameObject_t *LServ_ReadGameObjPtr(cntStream_t *st)
{
	int i = Stream_ReadInt(st);
	if (i == -1)
	{
		return NULL;
	}
	return &g_gameObjects[i];
}

//read inventory item
static const invItemDef_t *LServ_ReadInvItem(cntStream_t *st)
{
	int i = Stream_ReadInt(st);
	if (i == -1)
	{
		return NULL;
	}
	return &g_invItems[i];
}

//read session globals
static void LServ_ReadSessGlobals(cntStream_t *st, int &prvMusicStrIndex)
{
	SESS_LOAD(g_glb.ai.numActiveAI);
	SESS_LOAD(g_glb.ai.numActiveEnemyAI);
	SESS_LOAD(g_glb.ai.playerCloseDist);
	SESS_LOAD(g_glb.ai.numCloseToPlayer);
	SESS_LOAD(g_glb.ai.numWantClose);
	g_glb.ai.closestToPlayer = LServ_ReadGameObjPtr(st);
	SESS_LOAD(g_glb.ai.playerLastHurtTime);
	SESS_LOAD(g_glb.ai.playerLastHitIndex);
	SESS_LOAD(g_glb.ai.playerLastHitTime);
	SESS_LOAD(g_glb.ai.inChaos);
	SESS_LOAD(g_glb.ai.showPaths);
	SESS_LOAD(g_glb.ai.weakAI);

	SESS_LOAD(g_glb.aiPathTime);
	SESS_LOAD(g_glb.gameTime);
	SESS_LOAD(g_glb.timeFactor);
	SESS_LOAD(g_glb.timeScale);
	SESS_LOAD(g_glb.invTimeScale);
	SESS_LOAD(g_glb.timeType);
	SESS_LOAD(g_glb.lastTimeType);
	SESS_LOAD(g_glb.actionTime);
	SESS_LOAD(g_glb.actionTimeScale);
	SESS_LOAD(g_glb.tonberryTime);
	SESS_LOAD(g_glb.enSpawnBlock);
	SESS_LOAD(g_glb.noDeath);
	SESS_LOAD(g_glb.cinema);
	SESS_LOAD(g_glb.debugRigid);
	SESS_LOAD(g_glb.mapName);
	SESS_LOAD(prvMusicStrIndex);
	SESS_LOAD(g_glb.levelMusic);
	SESS_LOAD(g_glb.battleTime);
	SESS_LOAD(g_glb.battleMusicEnabled);
	SESS_LOAD(g_glb.battleMusic);
	SESS_LOAD(g_glb.chocoboMusic);
	SESS_LOAD(g_glb.magicMode);

	SESS_LOAD(g_glb.endlessBattle);

	SESS_LOAD(g_glb.mpVersusMode);
	SESS_LOAD(g_glb.mpCountDown);
	SESS_LOAD(g_glb.mpCountTime);
	SESS_LOAD(g_glb.runningMultiplayer);
	SESS_LOAD(g_glb.initialFPSPos);
	SESS_LOAD(g_glb.initialFPSAng);
	SESS_LOAD(g_glb.camFrustum);
	SESS_LOAD(g_glb.wideCamFrustum);
	SESS_LOAD(g_glb.camFrustumFresh);
	SESS_LOAD(g_glb.camBoneBolt);
	SESS_LOAD(g_glb.camBoltOffset);
	SESS_LOAD(g_glb.camAbsOffset);
	SESS_LOAD(g_glb.camAbsAngOffset);
	SESS_LOAD(g_glb.camSlack);
	SESS_LOAD(g_glb.worldMins);
	SESS_LOAD(g_glb.worldMaxs);
	SESS_LOAD(g_glb.gameObjectSlots);
	SESS_LOAD(g_glb.creationCount);
	SESS_LOAD(g_glb.clientInfo);
	SESS_LOAD(g_glb.numChocoSpots);
	SESS_LOAD(g_glb.scriptScreenFade);
	SESS_LOAD(g_glb.scriptPicFade);
	SESS_LOAD(g_glb.scriptFogValues);
	SESS_LOAD(g_glb.scriptFogBlendMode);
	SESS_LOAD(g_glb.changeToMap);
	SESS_LOAD(g_glb.scheduledMapChange);
}

//read session object function pointer
void *LServ_ReadFuncPtr(cntStream_t *st, gameObject_t *obj)
{
	int i = Stream_ReadInt(st);
	if (i == -1)
	{
		return NULL;
	}
	return g_sessFnPtrs[i].ptr;
}

//read session object pointer data
void LServ_ReadSessObjPtr(cntStream_t *st, gameObject_t *obj, int type)
{
	int r = Stream_ReadInt(st);
	if (!r)
	{
		return;
	}

	if (type == SESSPTR_CLIPMODEL)
	{
		rcColModel_t *o = obj->rcColModel;
		RCUBE_ASSERT(o);
		SESS_LOAD(o->frame);
		SESS_LOAD(o->blendFrame);
		SESS_LOAD(o->lerpAmount);
		SESS_LOAD(o->pos);
		SESS_LOAD(o->ang);
		SESS_LOAD(o->modelScale);
		SESS_LOAD(o->mins);
		SESS_LOAD(o->maxs);
		SESS_LOAD(o->radius);
		SESS_LOAD(o->gameOwner);
		SESS_LOAD(o->gameIgnore);
		SESS_LOAD(o->solid);
		SESS_LOAD(o->clipFlags);
	}
	else if (type == SESSPTR_PLAYER)
	{
		playerObject_t *o = obj->plObj;
		RCUBE_ASSERT(o);
		SESS_LOAD(o->clButtons);
		SESS_LOAD(o->clButtonsBackBuffer);
		SESS_LOAD(o->numButtonsBuffered);
		SESS_LOAD(o->clAngles);
		SESS_LOAD(o->clAnalog);
		SESS_LOAD(o->hasClAnalog);
		SESS_LOAD(o->lastClAngles);
		SESS_LOAD(o->hasLastClAngles);
		SESS_LOAD(o->playerSpawnSeq);
		SESS_LOAD(o->chainNum);
		SESS_LOAD(o->nextChain);
		o->targetObj = LServ_ReadGameObjPtr(st);
		SESS_LOAD(o->healthUpdateTime);
		SESS_LOAD(o->lungeTime);
		SESS_LOAD(o->lastDmg);
		SESS_LOAD(o->hasLastDmg);
		SESS_LOAD(o->noClip);
		SESS_LOAD(o->invincible);
		SESS_LOAD(o->noTarget);
		SESS_LOAD(o->makoCharges);
		SESS_LOAD(o->makoStash);
		SESS_LOAD(o->chargeMeter);
		SESS_LOAD(o->chargeDrain);
		SESS_LOAD(o->chargeDrainTime);
		SESS_LOAD(o->chargeToMakoPower);
		SESS_LOAD(o->timeAttackTime);
		SESS_LOAD(o->fistOfFlames);
		SESS_LOAD(o->explosiveBlows);
		SESS_LOAD(o->healingAura);
		SESS_LOAD(o->healingTime);
		SESS_LOAD(o->magicHealTime);
		SESS_LOAD(o->desiredItem);
		SESS_LOAD(o->equippedItem);
		SESS_LOAD(o->waitOnInput);
		SESS_LOAD(o->moveSpeedScale);
		o->camTarget = LServ_ReadGameObjPtr(st);
		SESS_LOAD(o->camRange);
		SESS_LOAD(o->jumpUseTime);
		SESS_LOAD(o->canJumpUse);
		SESS_LOAD(o->jumpUseObj);
		SESS_LOAD(o->hasteTime);

		o->gun = LServ_ReadGameObjPtr(st);
		o->gunItem = LServ_ReadInvItem(st);
		SESS_LOAD(o->gunFireTime);
		SESS_LOAD(o->gunFirePose);
		o->sword = LServ_ReadGameObjPtr(st);
		o->swordItem = LServ_ReadInvItem(st);
		o->flashlight = LServ_ReadGameObjPtr(st);
		o->flashlightItem = LServ_ReadInvItem(st);
		o->ride = LServ_ReadGameObjPtr(st);
		o->aerith = LServ_ReadGameObjPtr(st);
		SESS_LOAD(o->noItemTime);
		SESS_LOAD(o->plData);
		SESS_LOAD(o->camSeeTime);
		SESS_LOAD(o->camTeleWarn);
		SESS_LOAD(o->clCheater);
		SESS_LOAD(o->plSeqDmg);
		SESS_LOAD(o->numPlSeqDmg);
	}
	else if (type == SESSPTR_AI)
	{
		aiObject_t *o = obj->aiObj;
		RCUBE_ASSERT(o);
		SESS_LOAD(o->maxHealth);
		SESS_LOAD(o->moveSpeed);
		SESS_LOAD(o->lookYaw);
		SESS_LOAD(o->lookPitch);
		SESS_LOAD(o->fly);
		SESS_LOAD(o->painChance);
		SESS_LOAD(o->forceGravity);
		SESS_LOAD(o->combatRange);
		SESS_LOAD(o->goalRange);
		SESS_LOAD(o->goalPos);
		o->goalObj = LServ_ReadGameObjPtr(st);
		SESS_LOAD(o->distToGoal);
		SESS_LOAD(o->obstructTime);
		SESS_LOAD(o->makoValue);
		SESS_LOAD(o->aiLevel);
		SESS_LOAD(o->aiFov);
		SESS_LOAD(o->aiRange);
		o->scriptGoal = LServ_ReadGameObjPtr(st);
		o->scriptLookGoal = LServ_ReadGameObjPtr(st);
		SESS_LOAD(o->scriptGoalRange);
		o->enemy = LServ_ReadGameObjPtr(st);
		SESS_LOAD(o->distToEnemy);
		SESS_LOAD(o->attackTime);
		SESS_LOAD(o->noMoveTime);
		SESS_LOAD(o->confuseTime);
		SESS_LOAD(o->lastDmg);
		SESS_LOAD(o->hasLastDmg);
		SESS_LOAD(o->nonActive);
		SESS_LOAD(o->boneDmgTime);
		SESS_LOAD(o->boneDmgDir);
	}
	else if (type == SESSPTR_CIN)
	{
		cinObject_t *o = obj->cinObj;
		RCUBE_ASSERT(o);
		SESS_LOAD(o->animStub);
		SESS_LOAD(o->hideable);
	}
	else if (type == SESSPTR_RSCRIPT)
	{
		if (obj->rscript)
		{
			RScr_DestroyScript(obj->rscript);
		}
		char scrName[512];
		Stream_ReadString(st, scrName, 512);
		obj->rscript = RScr_CreateScript(scrName);
		rscript_t *o = obj->rscript;

		int r;
		rscriptNode_t *nodeBase;

		r = Stream_ReadInt(st);
		if (r)
		{
			char labName[512];
			Stream_ReadString(st, labName, 512);
			rscriptLabel_t *lab = RScr_GetLabel(labName);
			RCUBE_ASSERT(lab);
			nodeBase = lab->node;
		}
		else
		{
			nodeBase = o->curScript->nodes;
		}

		int curNum = Stream_ReadInt(st);
		if (curNum == -1)
		{
			o->curNode = NULL;
		}
		else
		{
			int num = 0;
			for (rscriptNode_t *n = nodeBase; n; n = n->next)
			{
				if (num == curNum)
				{
					o->curNode = n;
					break;
				}
				num++;
			}
		}

		r = Stream_ReadInt(st);
		if (r)
		{
			char labName[512];
			Stream_ReadString(st, labName, 512);
			rscriptLabel_t *lab = RScr_GetLabel(labName);
			RCUBE_ASSERT(lab);
			nodeBase = lab->node;
		}
		else
		{
			nodeBase = o->curScript->nodes;
		}

		int retNum = Stream_ReadInt(st);
		if (retNum == -1)
		{
			o->retNode = NULL;
		}
		else
		{
			int num = 0;
			for (rscriptNode_t *n = nodeBase; n; n = n->next)
			{
				if (num == retNum)
				{
					o->retNode = n;
					break;
				}
				num++;
			}
		}

		SESS_LOAD(o->scriptTime);
		SESS_LOAD(o->ropRegs);
		SESS_LOAD(o->ropIntRegs);
		SESS_LOAD(o->ropStackPtr);
		for (int i = MAX_RSCRIPT_STACK-1; i > o->ropStackPtr; i--)
		{
			char str[512];
			Stream_ReadString(st, str, 512);
			o->ropStack[i] = Util_PooledString(str);
		}
	}
	else if (type == SESSPTR_RIGID)
	{
		rigidBody_t *o = obj->rigid;
		RCUBE_ASSERT(o);
		SESS_LOAD(o->mat);
		SESS_LOAD(o->invMat);
		SESS_LOAD(o->torque);
		SESS_LOAD(o->gravity);
	}
	else if (type == SESSPTR_CUSTOM)
	{
		if (obj->sessLoad)
		{
			obj->sessLoad(obj, st);
		}
	}
}

//read out a game object
void LServ_ReadGameObj(cntStream_t *st, gameObject_t *obj, int objIdx)
{
	int inuse = Stream_ReadInt(st);
	if (inuse == -1)
	{
		if (obj->inuse)
		{
			obj->think = ObjGeneral_RemoveThink;
			obj->thinkTime = g_glb.gameTime;
		}
		return;
	}

	char objName[512];
	Stream_ReadString(st, objName, 512);
	int numSpawnArgs = Stream_ReadInt(st);
	objArgs_t *args = NULL;
	if (numSpawnArgs > 0)
	{
		args = (objArgs_t *)Util_PooledAlloc(sizeof(objArgs_t)*numSpawnArgs);
		for (int i = 0; i < numSpawnArgs; i++)
		{
			objArgs_t *arg = args+i;
			char str[64];
			Stream_ReadString(st, str, 64);
			arg->key = Util_PooledString(str);
			Stream_ReadString(st, str, 64);
			arg->val = Util_PooledString(str);
		}
	}
	if (objIdx > MAX_NET_CLIENTS && objIdx != CAM_TRACK_NUM && objIdx != MAP_OBJ_NUM)
	{
		LServ_ObjectFromName(objName, NULL, NULL, args, numSpawnArgs, obj);
	}
	obj->inuse = inuse;

	SESS_LOAD(obj->net);
	SESS_LOAD(obj->thinkTime);
	SESS_LOAD(obj->spawnMins);
	SESS_LOAD(obj->spawnMaxs);
	SESS_LOAD(obj->creationCount);
	SESS_LOAD(obj->hurtable);
	SESS_LOAD(obj->health);
	SESS_LOAD(obj->curAnim);
	SESS_LOAD(obj->animTime);
	SESS_LOAD(obj->animStartTime);
	SESS_LOAD(obj->animRestart);
	SESS_LOAD(obj->animNeverInterrupt);
	SESS_LOAD(obj->seqDmg);
	SESS_LOAD(obj->atkLastSeq);
	SESS_LOAD(obj->atkLastTime);
	SESS_LOAD(obj->atkType);
	SESS_LOAD(obj->atkRadiusCore);
	SESS_LOAD(obj->atkRadiusPos);
	SESS_LOAD(obj->meteorAttack);
	SESS_LOAD(obj->isOnFire);
	SESS_LOAD(obj->fireBurnTime);
	SESS_LOAD(obj->makoCount);
	SESS_LOAD(obj->localTimeScale);
	SESS_LOAD(obj->alwaysSend);
	SESS_LOAD(obj->noCullTime);
	SESS_LOAD(obj->nonSolidTime);
	SESS_LOAD(obj->preDeathTime);
	SESS_LOAD(obj->postSpawnTime);
	SESS_LOAD(obj->noGravTime);
	SESS_LOAD(obj->knockTime);
	SESS_LOAD(obj->toucher);
	SESS_LOAD(obj->radius);
	SESS_LOAD(obj->canPush);
	SESS_LOAD(obj->canBePushed);
	SESS_LOAD(obj->debounce);
	SESS_LOAD(obj->debounce2);
	SESS_LOAD(obj->debounce3);
	SESS_LOAD(obj->debounce4);
	SESS_LOAD(obj->debounce5);
	SESS_LOAD(obj->debounce6);
	SESS_LOAD(obj->debounce7);
	SESS_LOAD(obj->swordTime);
	SESS_LOAD(obj->generalFlag);
	SESS_LOAD(obj->generalFlag2);
	SESS_LOAD(obj->generalFlag3);
	SESS_LOAD(obj->dualAnimCnt);
	SESS_LOAD(obj->spareVec);
	SESS_LOAD(obj->spawnLife);
	SESS_LOAD(obj->localFlags);
	SESS_LOAD(obj->bounceFactor);
	SESS_LOAD(obj->onGround);
	SESS_LOAD(obj->groundObj);
	SESS_LOAD(obj->camTrackPos);
	SESS_LOAD(obj->groundHugger);
	SESS_LOAD(obj->preImpactVel);
	SESS_LOAD(obj->projVelAng);
	SESS_LOAD(obj->projVelocity);
	SESS_LOAD(obj->projFriction);
	SESS_LOAD(obj->projDamage);
	SESS_LOAD(obj->safePos);
	SESS_LOAD(obj->localRecursionTick);
	SESS_LOAD(obj->localAngles);
	SESS_LOAD(obj->customDynSize);
	obj->enemy = LServ_ReadGameObjPtr(st);
	obj->target = LServ_ReadGameObjPtr(st);
	obj->child = LServ_ReadGameObjPtr(st);
	obj->parent = LServ_ReadGameObjPtr(st);
	obj->chain = LServ_ReadGameObjPtr(st);
	obj->chain2 = LServ_ReadGameObjPtr(st);
	obj->physModelOwner = LServ_ReadGameObjPtr(st);

	typedef void (*pfnObjThink)(gameObject_t *obj, float timeMod);
	typedef void (*pfnObjTouch)(gameObject_t *obj, gameObject_t *other, const collObj_t *col);
	typedef void (*pfnObjPain)(gameObject_t *obj, gameObject_t *hurter, int dmg, const collObj_t *col);
	typedef void (*pfnObjDeath)(gameObject_t *obj, gameObject_t *killer, int dmg);
	typedef void (*pfnObjActivate)(gameObject_t *obj, gameObject_t *activator);
	obj->think = (pfnObjThink)LServ_ReadFuncPtr(st, obj);
	obj->touch = (pfnObjTouch)LServ_ReadFuncPtr(st, obj);
	obj->pain = (pfnObjPain)LServ_ReadFuncPtr(st, obj);
	obj->death = (pfnObjDeath)LServ_ReadFuncPtr(st, obj);
	obj->activate = (pfnObjActivate)LServ_ReadFuncPtr(st, obj);

	LServ_ReadSessObjPtr(st, obj, SESSPTR_CLIPMODEL);
	LServ_ReadSessObjPtr(st, obj, SESSPTR_PLAYER);
	LServ_ReadSessObjPtr(st, obj, SESSPTR_AI);
	LServ_ReadSessObjPtr(st, obj, SESSPTR_CIN);
	LServ_ReadSessObjPtr(st, obj, SESSPTR_RSCRIPT);
	LServ_ReadSessObjPtr(st, obj, SESSPTR_RIGID);
	LServ_ReadSessObjPtr(st, obj, SESSPTR_CUSTOM);
}

//read out the complete session data
bool LServ_LoadSession(sessHeader_t *hdr)
{
	if (hdr->version != SAVE_SESSION_VERSION)
	{
		return false;
	}

	cntStream_t *st = Stream_Alloc(hdr+1, hdr->dataSize);
	int schk = (int)RScr_GetScriptChk();
	int testChk = Stream_ReadInt(st);
	if (schk != testChk)
	{ //script change made the session data incompatible
		Stream_Free(st);
		return false;
	}

	GVar_ReadGVars(st);

	//read server strings
	typedef struct prvSessString_s
	{
		char		p[64];
	} prvSessString_t;
	prvSessString_t prvStr[MAX_GAME_STRINGS];
	for (int i = 0; i < MAX_GAME_STRINGS; i++)
	{
		Stream_ReadString(st, prvStr[i].p, 64);
	}

	int oldObjSlots = g_glb.gameObjectSlots;

	//read game globals
	int prvMusicStrIndex = -1;
	LServ_ReadSessGlobals(st, prvMusicStrIndex);

	if (prvMusicStrIndex != -1)
	{
		LServ_ChangeMusic(prvStr[prvMusicStrIndex].p);
	}

#define SESS_STR_REMAP(a) \
	if (a) \
	{ \
		a = g_sharedFn->Common_ServerString(prvStr[a].p); \
	}

	//free any objects over the top of the load object array
	for (int i = g_glb.gameObjectSlots; i < oldObjSlots; i++)
	{
		gameObject_t *obj = &g_gameObjects[i];
		obj->think = ObjGeneral_RemoveThink;
		obj->thinkTime = g_glb.gameTime;
	}

	//read all game objects
	for (int i = 0; i < g_glb.gameObjectSlots; i++)
	{
		gameObject_t *obj = &g_gameObjects[i];
		LServ_ReadGameObj(st, obj, i);

		//remap old game string indices
		SESS_STR_REMAP(obj->net.entNameIndex);
		SESS_STR_REMAP(obj->net.strIndex);
		SESS_STR_REMAP(obj->net.strIndexB);
		SESS_STR_REMAP(obj->net.strIndexC);
		SESS_STR_REMAP(obj->net.strIndexD);
		SESS_STR_REMAP(obj->net.strIndexE);

		LServ_UpdateRClip(obj);
	}

	Stream_Free(st);

	LServ_PostMapSpawn();

	if (g_glb.scheduledMapChange)
	{ //resume map transition
		if (!g_glb.changeToMap[0])
		{
			g_sharedFn->Common_MapChange(NULL, 6000);
		}
		else
		{
			g_sharedFn->Common_MapChange(g_glb.changeToMap, 1000);
		}
	}

	if (g_gameObjects[0].plObj &&
		g_gameObjects[0].plObj->waitOnInput)
	{ //hack for restoring session in the middle of a script that was waiting on input
		g_gameObjects[0].plObj->waitOnInput = -1;
	}

	return true;
}
