/*
=============================================================================
Module Information
------------------
Name:			util.cpp
Author:			Rich Whitehouse
Description:	random server logic utility/convenience functions
=============================================================================
*/

#include "main.h"
#include "ai.h"

modelMatrix_t g_gameIdent ={	{1.0f, 0.0f, 0.0f},
								{0.0f, 1.0f, 0.0f},
								{0.0f, 0.0f, 1.0f},
								{0.0f, 0.0f, 0.0f}	};

typedef struct objSpawnMatch_s
{
	void				(*a)(gameObject_t *obj, BYTE *b, const objArgs_t *args, int numArgs);
	char				*b;
} objSpawnMatch_t;

static const objSpawnMatch_t g_objSpawnTable[] =
{
	{ObjGeneral_SpawnNoNet, "ObjGeneral_SpawnNoNet"},
	{ObjGeneral_Spawn, "ObjGeneral_Spawn"},
	{ObjProjectile_Spawn, "ObjProjectile_Spawn"},
	{ObjCamMarker_Spawn, "ObjCamMarker_Spawn"},
	{ObjLift_Spawn, "ObjLift_Spawn"},
	{ObjMover_Spawn, "ObjMover_Spawn"},
	{ObjLight_Spawn, "ObjLight_Spawn"},
	{ObjVisBlock_Spawn, "ObjVisBlock_Spawn"},
	{ObjSoundEmit_Spawn, "ObjSoundEmit_Spawn"},
	{ObjPartEmit_Spawn, "ObjPartEmit_Spawn"},
	{ObjBeam_Spawn, "ObjBeam_Spawn"},
	{ObjSupernova_Spawn, "ObjSupernova_Spawn"},
	{ObjMako_Spawn, "ObjMako_Spawn"},
	{ObjItemDrop_Spawn, "ObjItemDrop_Spawn"},
	{ObjRSRunner_Spawn, "ObjRSRunner_Spawn"},
	{ObjTriggerBox_Spawn, "ObjTriggerBox_Spawn"},
	{ObjForceField_Spawn, "ObjForceField_Spawn"},
	{ObjCinProp_Spawn, "ObjCinProp_Spawn"},
	{ObjE30_Spawn, "ObjE30_Spawn"},
	{ObjDog_Spawn, "ObjDog_Spawn"},
	{ObjEat_Spawn, "ObjEat_Spawn"},
	{ObjBomb_Spawn, "ObjBomb_Spawn"},
	{ObjGig_Spawn, "ObjGig_Spawn"},
	{ObjIron_Spawn, "ObjIron_Spawn"},
	{ObjTon_Spawn, "ObjTon_Spawn"},
	{ObjHorn_Spawn, "ObjHorn_Spawn"},
	{ObjMad_Spawn, "ObjMad_Spawn"},
	{ObjMadBall_Spawn, "ObjMadBall_Spawn"},
	{ObjCloud_Spawn, "ObjCloud_Spawn"},
	{ObjBarrett_Spawn, "ObjBarrett_Spawn"},
	{ObjBarForge_Spawn, "ObjBarForge_Spawn"},
	{ObjAerith_Spawn, "ObjAerith_Spawn"},
	{ObjChocoSpot_Spawn, "ObjChocoSpot_Spawn"},
	{ObjChoco_Spawn, "ObjChoco_Spawn"},
	{ObjCCon_Spawn, "ObjCCon_Spawn"},
	{ObjJenova_Spawn, "ObjJenova_Spawn"},
	{ObjSephFin_Spawn, "ObjSephFin_Spawn"},
	{ObjSephHair_Spawn, "ObjSephHair_Spawn"},
	{ObjVortex_Spawn, "ObjVortex_Spawn"},
	{ObjSword_Spawn, "ObjSword_Spawn"},
	{ObjRigidBody_Spawn, "ObjRigidBody_Spawn"},
	{0, 0}
};

//parse a spawn function from the table
bool Util_ParseObjSpawn(const char *str, gameObject_t *obj)
{
	obj->customSpawn = NULL;

	if (!str || !str[0])
	{
		return false;
	}

	int i = 0;
	while (g_objSpawnTable[i].b)
	{
		if (!strcmp(str, g_objSpawnTable[i].b))
		{ //got it
			obj->customSpawn = g_objSpawnTable[i].a;
			return true;
		}
		i++;
	}

	return false;
}

typedef struct objValMatch_s
{
	objTypes_e	a;
	char		*b;
} objValMatch_t;

static const objValMatch_t g_objValTable[] =
{
	{OBJ_TYPE_SPRITE, "OBJ_TYPE_SPRITE"},
	{OBJ_TYPE_MAP, "OBJ_TYPE_MAP"},
	{OBJ_TYPE_MODEL, "OBJ_TYPE_MODEL"},
	{OBJ_TYPE_NETEVENT, "OBJ_TYPE_NETEVENT"},
	{OBJ_TYPE_CAM, "OBJ_TYPE_CAM"},
	{OBJ_TYPE_PROJECTILE, "OBJ_TYPE_PROJECTILE"},
	{OBJ_TYPE_GENERAL, "OBJ_TYPE_GENERAL"},
	{OBJ_TYPE_LIGHT, "OBJ_TYPE_LIGHT"},
	{OBJ_TYPE_VISBLOCK, "OBJ_TYPE_VISBLOCK"},
	{OBJ_TYPE_BEAM, "OBJ_TYPE_BEAM"},
	{OBJ_TYPE_USER, "OBJ_TYPE_USER"},
	{NUM_OBJ_TYPES, NULL}
};

//parse an object type from the table
bool Util_ParseObjType(const char *str, int *out)
{
	*out = OBJ_TYPE_SPRITE;

	if (!str || !str[0])
	{
		return false;
	}

	int i = 0;
	while (g_objValTable[i].b)
	{
		if (!strcmp(str, g_objValTable[i].b))
		{ //got it
			*out = g_objValTable[i].a;
			return true;
		}
		i++;
	}

	return false;
}

//parse string vector to float
bool Util_ParseVector(const char *str, float *out)
{
	if (!str || !str[0])
	{
		out[0] = 0.0f;
		out[1] = 0.0f;
		out[2] = 0.0f;
		return false;
	}
	sscanf(str, "(%f %f %f)", &out[0], &out[1], &out[2]);
	return true;
}

//parse string to int
bool Util_ParseInt(const char *str, int *out)
{
	if (!str || !str[0])
	{
		*out = 0;
		return false;
	}
	*out = atoi(str);
	return true;
}

//nudge value
float Util_NudgeValue(float cur, float dst, float fact)
{
	if (fact > 1.0f)
	{
		fact = 1.0f;
	}
	else if (fact < 0.000001f)
	{
		fact = 0.000001f;
	}

	float dif = (dst-cur);
	if (fabsf(dif) < 0.00001f)
	{
		return dst;
	}

	float f = cur+(dif*fact);

	return f;
}

//cheesy function to get a radius from a bounds for very quick frustum checking
float Util_RadiusFromBounds(float *mins, float *maxs, float padMul)
{
	float b = 0.0f;
	for (int i = 0; i < 3; i++)
	{
		if (fabsf(mins[i]) > b)
		{
			b = fabsf(mins[i]);
		}
		if (maxs[i] > b)
		{
			b = maxs[i];
		}
	}

	return b*padMul;
}

//random int not based on the local seed
int Util_RandInt(int low, int high)
{
	if (low == high)
	{
		return low;
	}
	return low + (rand()%(high-low));
}

//random float not based on the local seed
float Util_RandFloat(float low, float high)
{
	if (low == high)
	{
		return low;
	}
	unsigned int ir = rand()%(1<<15);
	float r = ((float)ir)/(float)((unsigned int)(1<<15));
	return low + (r*(high-low));
}

//gets center of object position
void Util_GameObjectCenter(gameObject_t *obj, float *v)
{
	float b[3];
	b[0] = (obj->net.maxs[0]+obj->net.mins[0])*0.5f;
	b[1] = (obj->net.maxs[1]+obj->net.mins[1])*0.5f;
	b[2] = (obj->net.maxs[2]+obj->net.mins[2])*0.5f;
	v[0] = obj->net.pos[0]+b[0];
	v[1] = obj->net.pos[1]+b[1];
	v[2] = obj->net.pos[2]+b[2];
}

//checks if two game objects overlap
bool Util_GameObjectsOverlap(gameObject_t *obja, gameObject_t *objb)
{
	for (int i = 0; i < 3; i++)
	{
		float mina = obja->net.pos[i]+obja->spawnMins[i];
		float maxa = obja->net.pos[i]+obja->spawnMaxs[i];
		float minb = objb->net.pos[i]+objb->spawnMins[i];
		float maxb = objb->net.pos[i]+objb->spawnMaxs[i];
		if (mina < minb &&
			maxa < minb)
		{
			return false;
		}
		if (mina > maxb &&
			maxa > maxb)
		{
			return false;
		}
	}
	return true;
}

//touch objects to each other
void Util_TouchObjects(gameObject_t *obj1, gameObject_t *obj2, const collObj_t *col)
{
	if (!obj1 || !obj2)
	{
		return;
	}
	obj1->toucher = true;
	if (obj1->touch)
	{
		obj1->touch(obj1, obj2, col);
	}
	if (obj2->touch)
	{
		obj2->touch(obj2, obj1, col);
	}
	obj1->toucher = false;
}

//project a decal
void Util_ProjectDecal(gameObject_t *obj, const float *start, const float *end, const char *decal,
					   float decalSize, int decalLife, int decalFade)
{
	if (!obj || !obj->inuse)
	{
		return;
	}

	gameObject_t *netEvent = LServ_CreateObject();
	if (!netEvent)
	{
		return;
	}

	netEvent->net.type = OBJ_TYPE_NETEVENT;
	netEvent->net.frame = NETEVENT_DECAL;
	netEvent->net.pos[0] = start[0];
	netEvent->net.pos[1] = start[1];
	netEvent->net.pos[2] = start[2];

	netEvent->net.ang[0] = end[0];
	netEvent->net.ang[1] = end[1];
	netEvent->net.ang[2] = end[2];

	netEvent->net.modelScale[0] = decalSize; //projected decal size
	netEvent->net.modelScale[1] = (float)decalLife; //lifetime (in ms)
	netEvent->net.modelScale[2] = (float)decalFade; //fadetime (in ms)

	netEvent->net.solid = obj->net.index;

	netEvent->think = ObjGeneral_RemoveThink;
	netEvent->thinkTime = g_glb.gameTime+50;

	char str[64];
    sprintf(str, "^%s", decal);
	netEvent->net.strIndex = g_sharedFn->Common_ServerString(str);

}

//animate something
void Util_AnimateObject(gameObject_t *obj, float timeMod)
{
	if (!obj->animTable)
	{
		return;
	}
	/*
	float c[3] = {1.0f, 0.0f, 0.0f};
	float mins[3], maxs[3];
	Math_VecAdd(obj->net.pos, obj->net.mins, mins);
	Math_VecAdd(obj->net.pos, obj->net.maxs, maxs);
	Util_DebugBox(mins, maxs, c);
	*/
	gameAnim_t *gameAnim = obj->animTable+obj->curAnim;
	float gd = gameAnim->duration;
	if (gd >= 68.0f)
	{ //hack because old anim times were off
		gd -= 18.0f;
	}
	float animDur = gd*(1.0f/obj->localTimeScale);
	if (obj->animTime < g_glb.gameTime)
	{
		if (obj->animTime)
		{
			animDur -= (float)(g_glb.gameTime-obj->animTime);//*0.1f;
		}
		if (animDur < 1.0f)
		{
			animDur = 1.0f;
		}

		int lastFrame = obj->net.frame;
		if (lastFrame < gameAnim->startFrame || lastFrame > gameAnim->endFrame || obj->animRestart)
		{
			obj->net.frame = gameAnim->startFrame;
			obj->animRestart = false;
		}
		else
		{
			obj->net.frame++;
		}

		if (obj->net.frame > gameAnim->endFrame)
		{
			if (gameAnim->looping)
			{
				obj->net.frame = gameAnim->startFrame+(obj->net.frame-gameAnim->endFrame-1);
			}
			else
			{
				obj->net.frame = gameAnim->endFrame;
			}
		}

		if (obj->animframetick)
		{
			obj->animframetick(obj, timeMod, lastFrame);
		}

		obj->net.lerpDuration = (int)(animDur*g_glb.invTimeScale);
		
		if (obj->rcColModel)
		{
			obj->rcColModel->blendFrame = lastFrame;
			obj->rcColModel->frame = obj->net.frame;
			obj->animStartTime = g_glb.gameTime;
			g_sharedFn->Coll_UpdateModel(obj->rcColModel, NULL);
		}
		float compVal = 0.0f;//18.0f;//animDur*0.2f;
		obj->animTime = g_glb.gameTime + (serverTime_t)(animDur-(compVal*(1.0f/obj->localTimeScale)));
	}

	if (obj->rcColModel && obj->animStartTime)
	{ //update collision model
		serverTime_t lerpDur = (serverTime_t)animDur;
		serverTime_t dif = g_glb.gameTime-obj->animStartTime;
		if (dif > lerpDur)
		{
			dif = lerpDur;
		}
		obj->rcColModel->lerpAmount = 1.0f-((float)dif/(float)lerpDur);
		g_sharedFn->Coll_UpdateModel(obj->rcColModel, NULL);
	}
}

//is sequence damage possible?
bool Util_SequenceDamage(gameObject_t *obj, gameObject_t *other)
{
	if (other->seqDmg.dmgLastIndex == obj->net.index &&
		other->seqDmg.dmgLastCount == obj->creationCount &&
		other->seqDmg.dmgLastSeq == obj->atkLastSeq)
	{ //already hit this thing in this sequence
		return false;
	}
	if (other->plObj)
	{
		for (int i = 0; i < other->plObj->numPlSeqDmg; i++)
		{
			if (other->plObj->plSeqDmg[i].dmgLastIndex == obj->net.index &&
				other->plObj->plSeqDmg[i].dmgLastCount == obj->creationCount &&
				other->plObj->plSeqDmg[i].dmgLastSeq == obj->atkLastSeq)
			{ //already hit this thing in this sequence
				return false;
			}
		}
	}
	return true;
}

//doing sequence damage
void Util_SetSequenceDamage(gameObject_t *obj, gameObject_t *other)
{
	other->seqDmg.dmgLastIndex = obj->net.index;
	other->seqDmg.dmgLastCount = obj->creationCount;
	other->seqDmg.dmgLastSeq = obj->atkLastSeq;
	obj->atkLastTime = g_glb.gameTime;
	if (other->plObj)
	{
		if (other->plObj->numPlSeqDmg > 0)
		{
			int numCopy = (other->plObj->numPlSeqDmg > 7) ? 7 : other->plObj->numPlSeqDmg;
			memcpy(&other->plObj->plSeqDmg[1], &other->plObj->plSeqDmg[0], sizeof(seqDmg_t)*numCopy);
		}
		other->plObj->plSeqDmg[0].dmgLastIndex = obj->net.index;
		other->plObj->plSeqDmg[0].dmgLastCount = obj->creationCount;
		other->plObj->plSeqDmg[0].dmgLastSeq = obj->atkLastSeq;
		other->plObj->numPlSeqDmg++;
		if (other->plObj->numPlSeqDmg > 8)
		{
			other->plObj->numPlSeqDmg = 8;
		}
	}
}

//are these two things enemies?
bool Util_ValidEnemies(gameObject_t *obj, gameObject_t *other)
{
	if (obj == other)
	{
		return false;
	}
	
	if (obj->net.index < MAX_NET_CLIENTS && other->net.index < MAX_NET_CLIENTS &&
		g_glb.mpVersusMode)
	{ //versus mode, pvp
		return true;
	}

	if (obj->aiObj && (obj->aiObj->confuseTime >= g_glb.gameTime || g_glb.ai.inChaos))
	{
		return true;
	}
	if (other->aiObj && (other->aiObj->confuseTime >= g_glb.gameTime || g_glb.ai.inChaos))
	{
		return true;
	}

	if ((obj->localFlags & LFL_ENEMY) == (other->localFlags & LFL_ENEMY))
	{
		return false;
	}
	return true;
}

//do damage
bool Util_DamageObject(gameObject_t *damager, gameObject_t *victim, int damage, const collObj_t *col)
{
	if (!victim->hurtable)
	{
		return false;
	}

	if (victim->net.index < MAX_NET_CLIENTS)
	{
		if (victim->debounce2 > g_glb.gameTime)
		{ //player invincible
			return false;
		}
		if (damager != victim &&
			(damager->net.index < MAX_NET_CLIENTS ||
			damager->net.owner < MAX_NET_CLIENTS) &&
			!Util_ValidEnemies(damager, victim))
		{ //no friendly fire
			return false;
		}

		if (damager && damager->atkType != ATKTYPE_BURNING)
		{ //can't def-absorb burning
			damage = ObjPlayer_DefMod(victim, damage, damager);
		}
	}

	if (victim->net.index >= MAX_NET_CLIENTS || !victim->plObj || !victim->plObj->invincible)
	{
		if (!g_glb.noDeath)
		{
			victim->health -= damage;
			if (victim->plObj && g_glb.magicMode)
			{
				if (victim->health < 1)
				{
					victim->health = 1;
					if (g_glb.magicMode == 2 && damager && damager->net.aiDescIndex == AIDESC_THEDOLLAR &&
						damager->hurtable)
					{ //see if the dollar should forsake us
						damager->hurtable = 0;
						damager->think = ObjGeneral_RemoveThink;
						damager->thinkTime = g_glb.gameTime;

						float c[3];
						float ang[3] = {0.0f, 0.0f, 0.0f};
						Util_GameObjectCenter(damager, c);
						int dmg = 1000;
						damager->net.vel[0] = 0.0f;
						damager->net.vel[1] = 0.0f;
						damager->net.vel[2] = 0.0f;
						Util_RadiusDamage(damager, dmg, 2048.0f, c);
						ObjSound_Create(c, "assets/sound/cb/boom.wav", 1.0f, -1);
						ObjSound_Create(c, "assets/sound/cb/meteorsmash.wav", 1.0f, -1);
						ObjSound_Create(c, "assets/sound/cb/boom.wav", 1.0f, -1);
						ObjSound_Create(c, "assets/sound/cb/meteorsmash.wav", 1.0f, -1);
						Math_VecSub(c, damager->net.pos, c);
						ObjParticles_Create("other/bombblow", c, ang, damager->net.index);

						Util_StatusMessage("The Dollar is disappointed.");
					}
				}
				victim->plObj->magicHealTime = g_glb.gameTime+4000;
			}
		}
	}

	if (victim->pain)
	{
		victim->pain(victim, damager, damage, col);
	}
	if (victim->health <= 0 && victim->death)
	{
		victim->death(victim, damager, damage);
	}
	if (damager && damager->onhurt)
	{
		damager->onhurt(damager, victim, damage, col);
	}
	return true;
}

//radius damage
void Util_RadiusDamage(gameObject_t *damager, int damage, float radius, float *damageOrigin, bool friendlyFire)
{
	float c1[3];
	if (damageOrigin)
	{
		Math_VecCopy(damageOrigin, c1);
	}
	else
	{
		Util_GameObjectCenter(damager, c1);
	}
	for (int i = 0; i < g_glb.gameObjectSlots; i++)
	{
		gameObject_t *obj = &g_gameObjects[i];
		if (obj == damager || !obj->inuse || !obj->hurtable)
		{
			continue;
		}
		if (!friendlyFire && !Util_ValidEnemies(damager, obj))
		{
			continue;
		}
		float c2[3];
		float d[3];
		Util_GameObjectCenter(obj, c2);
		Math_VecSub(c1, c2, d);
		float dist = Math_VecLen(d);
		if (dist > radius)
		{ //too far
			continue;
		}

		if (!g_sharedFn->Coll_GeneralVisibilityWithBounds(c1, c2, damager->radius, damager->spawnMins, damager->spawnMaxs))
		{ //not in the same visibility cluster
			continue;
		}

		collObj_t col;
		collOpt_t opt;
		memset(&opt, 0, sizeof(opt));
		opt.ignoreBoxModels = true;
		g_sharedFn->Coll_RadiusTranslation(damager->rcColModel, &col, c1, c2, 0.0f, NULL, &opt);
		if (col.hit && col.hitObjectIndex != obj->net.index)
		{ //no clear trace
			continue;
		}

		int damageForRad = (int)((float)damage*(1.0f-(dist/radius)));
		if (damageForRad > 0)
		{
			int a = damager->atkType;
			damager->atkType = ATKTYPE_EXPLOSION;
			damager->atkRadiusCore = radius;
			Math_VecCopy(c1, damager->atkRadiusPos);
			Util_DamageObject(damager, obj, damageForRad, NULL);
			damager->atkType = a;
		}
	}
}

//create an object to do a specific attack type on an object
void Util_DamageThroughHurter(gameObject_t *obj, int damage, int atkType)
{
	gameObject_t *hurter = LServ_CreateObject();
	hurter->think = ObjGeneral_RemoveThink;
	hurter->thinkTime = g_glb.gameTime+50;
	hurter->atkType = atkType;
	hurter->atkLastSeq = (int)g_glb.gameTime;
	float p[3], pd[3];
	Math_VecCopy(obj->net.pos, p);
	Math_VecCopy(obj->net.pos, pd);
	p[2] += ((obj->net.maxs[2]-obj->net.mins[2]) + 512.0f);
	pd[2] -= ((obj->net.maxs[2]-obj->net.mins[2]) + 512.0f);
	collObj_t col;
	g_sharedFn->Coll_RadiusTranslation(NULL, &col, p, pd, 1.0f, obj->rcColModel, false);
	if (col.hit && col.hitObjectIndex == obj->net.index)
	{
		Util_DamageObject(hurter, obj, damage, &col);
	}
}

//trace to offset from point within object
void Util_GetSafeOffset(gameObject_t *obj, float *pos, float *outPos)
{
	float c[3];
	Util_GameObjectCenter(obj, c);
	collObj_t col;
	g_sharedFn->Coll_RadiusTranslation(obj->rcColModel, &col, c, pos, 0.0f, NULL, false);
	Math_VecCopy(col.endPos, outPos);
	if (col.hit)
	{
		outPos[0] += col.endNormal[0]*0.1f;
		outPos[1] += col.endNormal[1]*0.1f;
		outPos[2] += col.endNormal[2]*0.1f;
	}
}

//fire a bullet, returns object shot if hit
gameObject_t *Util_FireBullet(gameObject_t *obj, float *pos, float *dir, bulletDesc_t *bullet, bool friendlyFire)
{
	const float maxBulletDist = 10000.0f;
	static bulletDesc_t defaultBullet =
	{
		5,
		"melee/impactslash",
		"assets/sound/cb/bulflesh.wav",
		"bulletspark",
		"assets/sound/cb/bullet.wav"
	};
	if (!bullet)
	{
		bullet = &defaultBullet;
	}

	//start the bullet at safe spot by tracing from the center of the shooter to the desired spot
	float c[3];
	Util_GameObjectCenter(obj, c);
	collObj_t col;
	g_sharedFn->Coll_RadiusTranslation(obj->rcColModel, &col, c, pos, 0.0f, NULL, false);
	float bulletPos[3], bulletEndPos[3];
	Math_VecCopy(col.endPos, bulletPos);
	if (col.hit)
	{
		bulletPos[0] += col.endNormal[0]*2.0f;
		bulletPos[1] += col.endNormal[1]*2.0f;
		bulletPos[2] += col.endNormal[2]*2.0f;
	}
	Math_VecMA(bulletPos, maxBulletDist, dir, bulletEndPos);

	//trace the bullet
	g_sharedFn->Coll_RadiusTranslation(obj->rcColModel, &col, bulletPos, bulletEndPos, 0.0f, NULL, false);
	if (!col.hit)
	{
		return NULL;
	}

	char *impactFx, *impactSound;
	gameObject_t *other = (col.hitObjectIndex >= 0) ? &g_gameObjects[col.hitObjectIndex] : NULL;
	if (other && other->inuse && other->hurtable)
	{
		if (!friendlyFire && !Util_ValidEnemies(obj, other))
		{
			return NULL;
		}
		float ang[3];
		Math_VecToAngles(col.endNormal, ang);
		if (other->attackblock && other->attackblock(other, obj, bullet->dmg, &col, col.endPos, ang))
		{
			impactFx = bullet->impactFxUnhurt;
			impactSound = bullet->impactSoundUnhurt;
		}
		else
		{
			impactFx = bullet->impactFxHurt;
			impactSound = bullet->impactSoundHurt;
			Util_DamageObject(obj, other, bullet->dmg, &col);
		}
	}
	else
	{
		impactFx = bullet->impactFxUnhurt;
		impactSound = bullet->impactSoundUnhurt;
	}

	if (other)
	{
		float vel[3];
		float bulletFac = (float)bullet->dmg * 32.0f;
		vel[0] = dir[0]*bulletFac;
		vel[1] = dir[1]*bulletFac;
		vel[2] = dir[2]*bulletFac;
		Phys_ShoveRigid(other, col.endPos, vel);
	}

	if (impactFx)
	{
		float ang[3];
		Math_VecToAngles(col.endNormal, ang);
		ObjParticles_Create(impactFx, col.endPos, ang, -1);
	}
	if (impactSound)
	{
		ObjSound_Create(col.endPos, impactSound, 1.0f, -1);
	}

	return other;
}

//get bolt matrix
bool Util_GetObjBolt(gameObject_t *obj, char *boneName, modelMatrix_t *matOut, float *ofs)
{
	if (!obj->inuse || !obj->rcColModel)
	{
		return false;
	}

	modelMatrix_t boneMat;
	if (!g_sharedFn->Coll_GetModelBoneMatrix(obj->rcColModel, boneName, &boneMat))
	{
		return false;
	}

	Math_VecCopy(boneMat.o, matOut->o);
	Math_TransformPointByMatrix(&boneMat, g_gameIdent.x1, matOut->x1);
	Math_TransformPointByMatrix(&boneMat, g_gameIdent.x2, matOut->x2);
	Math_TransformPointByMatrix(&boneMat, g_gameIdent.x3, matOut->x3);
	Math_VecSub(matOut->x1, boneMat.o, matOut->x1);
	Math_VecNorm(matOut->x1);
	Math_VecSub(matOut->x2, boneMat.o, matOut->x2);
	Math_VecNorm(matOut->x2);
	Math_VecSub(matOut->x3, boneMat.o, matOut->x3);
	Math_VecNorm(matOut->x3);
	if (ofs)
	{
		matOut->o[0] += matOut->x1[0]*ofs[0];
		matOut->o[1] += matOut->x1[1]*ofs[0];
		matOut->o[2] += matOut->x1[2]*ofs[0];
		matOut->o[0] += matOut->x2[0]*ofs[1];
		matOut->o[1] += matOut->x2[1]*ofs[1];
		matOut->o[2] += matOut->x2[2]*ofs[1];
		matOut->o[0] += matOut->x3[0]*ofs[2];
		matOut->o[1] += matOut->x3[1]*ofs[2];
		matOut->o[2] += matOut->x3[2]*ofs[2];
	}

	return true;
}

//used occasionally for cheesing out of bad collision situations
void Util_HarmlessKnockback(gameObject_t *knocker, gameObject_t *knockee)
{
	if (!knocker || !knocker->inuse ||
		!knockee || !knockee->inuse ||
		!knocker->hurtable || !knockee->hurtable)
	{
		return;
	}

	if (knocker->net.index < MAX_NET_CLIENTS && knockee->net.index < MAX_NET_CLIENTS)
	{ //clients ending up inside each other is normal for out-of-camera teleports
		return;
	}

	if (knocker->aiObj &&
		knocker->net.aiDescIndex == AIDESC_JENOVA &&
		knockee->aiObj)
	{ //jenova hack
		return;
	}

	AI_StartAnim(knockee, HUMANIM_FLYBACK, true);
	float fwd[3];
	float knockback = 2400.0f;
	float c1[3], c2[3];
	Util_GameObjectCenter(knockee, c1);
	Util_GameObjectCenter(knocker, c2);
	Math_VecSub(c1, c2, fwd);
	Math_VecNorm(fwd);
	knockee->net.vel[0] += fwd[0]*knockback;
	knockee->net.vel[1] += fwd[1]*knockback;
	knockee->net.vel[2] += 400.0f;
	knockee->onGround = false;
	knockee->knockTime = g_glb.gameTime + Util_LocalDelay(knockee, 100);
}

//maintain the object's safe spot
void Util_SetSafeSpot(gameObject_t *obj)
{
	if (obj->nonSolidTime > g_glb.gameTime)
	{
		obj->net.solid = 0;
		if (obj->rcColModel)
		{
			obj->rcColModel->solid = 0;
		}
	}
	collObj_t col;
	float testPos[3];
	float h = (obj->rcColModel->radius+obj->net.mins[2]);
	if (h > 0.001f)
	{
		h += 0.1f;
	}
	Math_VecCopy(obj->net.pos, testPos);
	testPos[2] += h;
	collOpt_t opt;
	memset(&opt, 0, sizeof(opt));
	opt.ignoreBoxModels = true;
	g_sharedFn->Coll_RadiusTranslation(obj->rcColModel, &col, testPos, testPos, obj->rcColModel->radius, NULL, &opt);
	if (!col.containsSolid)
	{
		g_sharedFn->Coll_MovementTranslation(obj->rcColModel, &col, obj->net.pos, obj->net.pos, NULL, NULL);
		if (!col.containsSolid)
		{
			Math_VecCopy(obj->net.pos, obj->safePos);
		}
		if (obj->nonSolidTime && obj->nonSolidTime <= g_glb.gameTime && !obj->net.solid)
		{
			if (!col.hit && !col.containsSolid)
			{
				obj->nonSolidTime = 0;
				obj->net.solid = 1;
				if (obj->rcColModel)
				{
					obj->rcColModel->solid = 1;
				}
			}
		}
	}
	else
	{
		//g_sharedFn->Coll_RadiusTranslation(obj->rcColModel, &col, testPos, testPos, obj->rcColModel->radius*0.95f, NULL, true);
		g_sharedFn->Coll_MovementTranslation(obj->rcColModel, &col, testPos, testPos, NULL, NULL);
		if (col.containsSolid)
		{ //ended up in solid
			g_sharedFn->Coll_MovementTranslation(obj->rcColModel, &col, obj->safePos, obj->safePos, NULL, NULL);
			if (col.hit && col.hitObjectIndex >= 0)
			{
				gameObject_t *other = &g_gameObjects[col.hitObjectIndex];
				//Util_DamageObject(other, other, 999999);
				if (!obj->nonSolidTime && other->net.solid)
				{ //make it non-solid
					if (other->net.index < MAX_NET_CLIENTS)
					{
						Util_HarmlessKnockback(obj, other);
						obj->nonSolidTime = g_glb.gameTime + 500;
					}
					else
					{
						Util_HarmlessKnockback(other, obj);
						other->nonSolidTime = g_glb.gameTime + 500;
					}
				}
			}
			Math_VecCopy(obj->safePos, obj->net.pos);
		}
	}
}

//find a targeted object based on target name
gameObject_t *Util_FindTargetObject(const char *targetName)
{
	if (!stricmp(targetName, "__the_player"))
	{
		return &g_gameObjects[0];
	}
	if (!stricmp(targetName, "__the_camera"))
	{
		return Util_GetCam(0);
	}

	for (int i = 0; i < g_glb.gameObjectSlots; i++)
	{
		gameObject_t *obj = &g_gameObjects[i];
		if (obj->inuse && obj->targetName && obj->targetName[0])
		{
			if (!_stricmp(obj->targetName, targetName))
			{
				return obj;
			}
		}
	}

	return NULL;
}

//try to find a spot relative to a target object which is out of camera view to spawn something
bool Util_FindHiddenSpawn(gameObject_t *targ, float dist, BYTE *rco, float *outPos, float *outAng)
{
	float mins[3], maxs[3];
	Util_ParseVector(Common_GetValForKey(rco, "spawnMins"), mins);
	Util_ParseVector(Common_GetValForKey(rco, "spawnMaxs"), maxs);
	const char *clipRadStr = Common_GetValForKey(rco, "clipRadius");
	float rad = (clipRadStr && clipRadStr[0]) ? (float)atof(clipRadStr) : Math_Max2(maxs[0], maxs[1]);
	int solid = 0;
	Util_ParseInt(Common_GetValForKey(rco, "solid"), &solid);

	const char *s = Common_GetValForKey(rco, "clipModel");
	if (!s || !s[0] || s[0] == '*' || !solid)
	{ //not supported for hidden spawn
		return false;
	}

	rcColModel_t *cm = g_sharedFn->Coll_RegisterModelInstance(s);
	if (!cm)
	{
		return false;
	}

	Math_VecCopy(mins, cm->mins);
	Math_VecCopy(maxs, cm->maxs);
	Math_VecCopy(targ->net.pos, cm->pos);
	float a[3] = {0.0f, targ->net.ang[1], 0.0f};
	Math_VecCopy(a, cm->ang);
	cm->gameOwner = targ->net.index;
	cm->solid = solid;
	cm->radius = rad;

	float fwd[3];
	Math_AngleVectors(a, fwd, 0, 0);
	fwd[0] = -fwd[0];
	fwd[1] = -fwd[1];
	fwd[2] = -fwd[2];
	float targPos[3];
	Math_VecMA(cm->pos, dist, fwd, targPos);
	collObj_t col;
	g_sharedFn->Coll_MovementTranslation(cm, &col, cm->pos, cm->pos, NULL, NULL);
	if (col.hit && col.containsSolid)
	{
		g_sharedFn->Coll_DestroyModelInstance(cm);
		return false;
	}
	g_sharedFn->Coll_MovementTranslation(cm, &col, cm->pos, targPos, NULL, NULL);
	g_sharedFn->Coll_DestroyModelInstance(cm);
	if (!col.containsSolid)
	{
		if (!col.hit || col.distance >= dist*0.75f)
		{
			outPos[0] = col.endPos[0];
			outPos[1] = col.endPos[1];
			outPos[2] = col.endPos[2];
			outAng[0] = a[0];
			outAng[1] = a[1];
			outAng[2] = a[2];
			return true;
		}
	}
	return false;
}

//try to spawn something at a position
bool Util_TryObjectSpawn(float *pos, BYTE *rco, float *originPos)
{
	float mins[3], maxs[3];
	Util_ParseVector(Common_GetValForKey(rco, "spawnMins"), mins);
	Util_ParseVector(Common_GetValForKey(rco, "spawnMaxs"), maxs);
	const char *clipRadStr = Common_GetValForKey(rco, "clipRadius");
	float rad = (clipRadStr && clipRadStr[0]) ? (float)atof(clipRadStr) : Math_Max2(maxs[0], maxs[1]);
	int solid = 0;
	Util_ParseInt(Common_GetValForKey(rco, "solid"), &solid);

	const char *s = Common_GetValForKey(rco, "clipModel");
	if (!s || !s[0] || s[0] == '*' || !solid)
	{ //not supported for hidden spawn
		return false;
	}

	rcColModel_t *cm = g_sharedFn->Coll_RegisterModelInstance(s);
	if (!cm)
	{
		return false;
	}

	Math_VecCopy(mins, cm->mins);
	Math_VecCopy(maxs, cm->maxs);
	Math_VecCopy(pos, cm->pos);
	float a[3] = {0.0f, 0.0f, 0.0f};
	Math_VecCopy(a, cm->ang);
	cm->gameOwner = MAX_GAME_OBJECTS+1;
	cm->solid = solid;
	cm->radius = rad;

	collObj_t col;
	g_sharedFn->Coll_MovementTranslation(cm, &col, cm->pos, cm->pos, NULL, NULL);
	g_sharedFn->Coll_DestroyModelInstance(cm);
	if (col.hit && col.containsSolid)
	{
		return false;
	}

	if (originPos)
	{
		collOpt_t opt;
		memset(&opt, 0, sizeof(opt));
		opt.ignoreBoxModels = true;
		g_sharedFn->Coll_RadiusTranslation(NULL, &col, originPos, pos, 0.0f, NULL, &opt);
		if (col.containsSolid || col.hit)
		{
			return false;
		}
	}

	return true;
}

//attempt to resize to a new scale
bool Util_ScaledResize(gameObject_t *obj, float *baseMins, float *baseMaxs, float baseRadius, float desiredScale, float stepSize)
{
	if (!obj->rcColModel)
	{
		return false;
	}
	if (obj->net.modelScale[0] == desiredScale)
	{
		return true;
	}

	float dif = (desiredScale-obj->net.modelScale[0]);
	if (dif > 0.0f && dif > stepSize)
	{
		dif = stepSize;
	}
	else if (dif < -stepSize)
	{
		dif = -stepSize;
	}
	float stepScale;
	if (fabsf(dif) < 0.001f)
	{
		stepScale = desiredScale;
	}
	else
	{
		stepScale = obj->net.modelScale[0] + dif;
	}

	float preR = obj->rcColModel->radius;
	float preMins[3], preMaxs[3];
	Math_VecCopy(obj->rcColModel->mins, preMins);
	Math_VecCopy(obj->rcColModel->maxs, preMaxs);

	obj->rcColModel->mins[0] = baseMins[0]*stepScale;
	obj->rcColModel->mins[1] = baseMins[1]*stepScale;
	obj->rcColModel->mins[2] = baseMins[2]*stepScale;
	obj->rcColModel->maxs[0] = baseMaxs[0]*stepScale;
	obj->rcColModel->maxs[1] = baseMaxs[1]*stepScale;
	obj->rcColModel->maxs[2] = baseMaxs[2]*stepScale;
	obj->rcColModel->radius = baseRadius*stepScale;
	collObj_t col;
	g_sharedFn->Coll_MovementTranslation(obj->rcColModel, &col, obj->net.pos, obj->net.pos, NULL, NULL);
	if (col.hit || col.containsSolid)
	{
		Math_VecCopy(preMins, obj->rcColModel->mins);
		Math_VecCopy(preMaxs, obj->rcColModel->maxs);
		obj->rcColModel->radius = preR;
		return false;
	}

	obj->net.modelScale[0] = stepScale;
	obj->net.modelScale[1] = obj->net.modelScale[0];
	obj->net.modelScale[2] = obj->net.modelScale[0];
	Math_VecCopy(obj->rcColModel->mins, obj->net.mins);
	Math_VecCopy(obj->rcColModel->maxs, obj->net.maxs);
	Math_VecCopy(obj->rcColModel->mins, obj->spawnMins);
	Math_VecCopy(obj->rcColModel->maxs, obj->spawnMaxs);
	return true;
}

//shake it
void Util_ClientShake(gameObject_t *cl, float shakeAmount, float shakeDec)
{
	float *shakes = (float *)_alloca(sizeof(float)*2);
	shakes[0] = shakeAmount;
	shakes[1] = shakeDec;
	int clIndex = (cl) ? cl->net.index : -1;
	g_sharedFn->Net_SendEventType(CLEV_THESHAKES2, shakes, sizeof(float)*2, clIndex);
}

//shake all clients in radius
void Util_ProximityShake(float *pos, float radius, float shakeAmount, float shakeDec)
{
	for (int i = 0; i < MAX_NET_CLIENTS; i++)
	{
		gameObject_t *cl = &g_gameObjects[i];
		if (!cl->inuse || !cl->plObj)
		{
			continue;
		}
		float d[3];
		Math_VecSub(cl->net.pos, pos, d);
		float len = Math_VecLen(d);
		if (len > radius)
		{
			continue;
		}

		float shakeFactor = 1.0f-(len/radius);
		Util_ClientShake(cl, shakeAmount*shakeFactor, shakeDec);
	}
}

//run r-script
void Util_RunScript(char *objName, char *scriptName)
{
	float z[3] = {0.0f, 0.0f, 0.0f};
	objArgs_t arg;
	arg.key = "script";
	arg.val = scriptName;
	gameObject_t *scriptObj = LServ_ObjectFromName("obj_rscript", z, z, &arg, 1);
	if (!scriptObj)
	{
		return;
	}
	scriptObj->spawnArgs = NULL;
	scriptObj->numSpawnArgs = 0;
	scriptObj->targetName = Util_PooledString(objName);
}

//cheap check for general cam proximity on xy
bool Util_CamProx2D(float *pos, const float d)
{
	gameObject_t *cam = Util_GetCam(0);
	if (cam->inuse)
	{
		float xyD[2] = {cam->net.pos[0]-pos[0], cam->net.pos[1]-pos[1]};
		float xyDist = (xyD[0]*xyD[0] + xyD[1]*xyD[1]);
		if (xyDist < d)
		{
			return true;
		}
	}

	return false;
}

//hax
void Util_DebugBox(float *mins, float *maxs, float *color)
{
	gameObject_t *netEvent = LServ_CreateObject();
	if (!netEvent)
	{
		return;
	}

	netEvent->net.type = OBJ_TYPE_NETEVENT;
	netEvent->net.frame = NETEVENT_DEBUGBOX;
	
	netEvent->net.mins[0] = mins[0];
	netEvent->net.mins[1] = mins[1];
	netEvent->net.mins[2] = mins[2];
	netEvent->net.maxs[0] = maxs[0];
	netEvent->net.maxs[1] = maxs[1];
	netEvent->net.maxs[2] = maxs[2];
	netEvent->net.ang[0] = color[0];
	netEvent->net.ang[1] = color[1];
	netEvent->net.ang[2] = color[2];

	netEvent->think = ObjGeneral_RemoveThink;
	netEvent->thinkTime = g_glb.gameTime+50;
}

//more hax
void Util_DebugLine(float *start, float *end)
{
	gameObject_t *netEvent = LServ_CreateObject();
	if (!netEvent)
	{
		return;
	}

	netEvent->net.type = OBJ_TYPE_NETEVENT;
	netEvent->net.frame = NETEVENT_DEBUGLINE;
	
	netEvent->net.mins[0] = start[0];
	netEvent->net.mins[1] = start[1];
	netEvent->net.mins[2] = start[2];
	netEvent->net.maxs[0] = end[0];
	netEvent->net.maxs[1] = end[1];
	netEvent->net.maxs[2] = end[2];

	netEvent->think = ObjGeneral_RemoveThink;
	netEvent->thinkTime = g_glb.gameTime+50;
}

//debug sphere
void Util_DebugSphere(float *start, float *end, float radius)
{
	gameObject_t *netEvent = LServ_CreateObject();
	if (!netEvent)
	{
		return;
	}

	netEvent->net.type = OBJ_TYPE_NETEVENT;
	netEvent->net.frame = NETEVENT_DEBUGSPHERE;
	
	netEvent->net.mins[0] = start[0];
	netEvent->net.mins[1] = start[1];
	netEvent->net.mins[2] = start[2];
	netEvent->net.maxs[0] = end[0];
	netEvent->net.maxs[1] = end[1];
	netEvent->net.maxs[2] = end[2];
	netEvent->net.modelScale[0] = radius;

	netEvent->think = ObjGeneral_RemoveThink;
	netEvent->thinkTime = g_glb.gameTime+50;
}

//more hax
void Util_DebugFX(float *pos, float *dir)
{
	float impactFxPos[3], impactFxAng[3];
	Math_VecToAngles(dir, impactFxAng);
	Math_VecCopy(pos, impactFxPos);
	ObjParticles_Create("weapons/laserimpact", impactFxPos, impactFxAng, -1);
}

//print status message to clients
void Util_StatusMessage(const char *msg, ...)
{
	char finalString[128];
	va_list args;
	
	va_start(args, msg);
	vsprintf_s(finalString, 128, msg, args);
	va_end(args);

	g_sharedFn->Net_SendEventType(CLEV_STATMESSAGE, finalString, (int)strlen(finalString), -1);
}

//print status message to one client
void Util_StatusMessageToClient(int cl, const char *msg, ...)
{
	char finalString[128];
	va_list args;
	
	va_start(args, msg);
	vsprintf_s(finalString, 128, msg, args);
	va_end(args);

	g_sharedFn->Net_SendEventType(CLEV_STATMESSAGE, finalString, (int)strlen(finalString), cl);
}

//print error message to clients
void Util_ErrorMessage(const char *msg, ...)
{
	char finalString[128];
	va_list args;
	
	va_start(args, msg);
	vsprintf_s(finalString, 128, msg, args);
	va_end(args);

	g_sharedFn->Net_SendEventType(CLEV_ERRORMESSAGE, finalString, (int)strlen(finalString), -1);
}

//scale by object-local time
serverTime_t Util_LocalDelay(gameObject_t *obj, serverTime_t del)
{
	float f = (float)del * (1.0f/obj->localTimeScale);
	return (serverTime_t)f;
}

//get camera object for a client
gameObject_t *Util_GetCam(int clIndex)
{
	return &g_gameObjects[CAM_TRACK_NUM];
}

//gets memory in a global string pool for a string
char *Util_PooledString(const char *str)
{
	const int stringPoolSize = 131072;//32768;
	static char stringPool[stringPoolSize];
	int lenForStr = (int)strlen(str);

	char *p = stringPool;//strstr(stringPool, str);
	int i = 0;
	while ((i+lenForStr+1) < g_glb.stringPoolPtr)
	{
		int j;
		for (j = 0; j < lenForStr; j++)
		{
			if (str[j] != p[i+j])
			{
				break;
			}
		}
		if (!str[j] && !p[i+j])
		{ //found it
			return &p[i];
		}
		i++;
	}

	if (g_glb.stringPoolPtr+lenForStr+1 >= stringPoolSize)
	{
		RCUBE_ASSERT(0);
		return NULL;
	}

	p = &stringPool[g_glb.stringPoolPtr];
	strcpy(p, str);
	g_glb.stringPoolPtr += lenForStr;
	stringPool[g_glb.stringPoolPtr] = 0;
	g_glb.stringPoolPtr++;

	return p;
}

//gets memory in a global pool
BYTE *Util_PooledAlloc(int size)
{
	const int poolSize = 65536;
	static BYTE memPool[poolSize];
	if (g_glb.regPoolPtr+size >= poolSize)
	{
		RCUBE_ASSERT(0); //ran out of room
		return NULL;
	}

	BYTE *p = &memPool[g_glb.regPoolPtr];
	g_glb.regPoolPtr += size;
	return p;
}
