/*
 * dat2pak
 *
 * Copyright (C) 2012 Florian Zwoch <fzwoch@gmail.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdint.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>

#ifdef __BIG_ENDIAN__
static uint16_t swap_16(uint16_t x)
{
	return (x >> 8) | (x << 8);
}

static uint32_t swap_32(uint32_t x)
{
	return (swap_16(x & 0xffff) << 16) | (swap_16(x >> 16));
}
#endif

typedef struct
{
	char magic[4];
	uint32_t offset;
	uint32_t size;
}
pak_header_t;

typedef struct
{
	char name[56];
	uint32_t offset;
	uint32_t size;
}
pak_entry_t;

int main(int argc, char *argv[])
{
	FILE *fp;
	uint32_t i;
	pak_header_t pak_header;
	pak_entry_t pak_entries[1024];
	size_t size;
	
	if (argc < 3)
	{
		printf("usage: %s <.pak> <.dat> [..]\n", argv[0]);
		return 1;
	}
	
	if (argc > sizeof(pak_entries) / sizeof(pak_entry_t) + 2)
	{
		printf("too many entries\n");
		return 1;
	}
	
	fp = fopen(argv[1], "wb");
	if (fp == NULL)
	{
		printf("error opening pak file: %s\n", argv[1]);
		return 1;
	}
	
	size = fwrite(&pak_header, sizeof(pak_header_t), 1, fp);
	if (size != 1)
	{
		printf("error writing pak header\n");
		return 1;
	}
	
	pak_header.magic[0] = 'P';
	pak_header.magic[1] = 'A';
	pak_header.magic[2] = 'C';
	pak_header.magic[3] = 'K';
	
	pak_header.offset = sizeof(pak_header_t);
	pak_header.size = (argc - 2) * sizeof(pak_entry_t);
	
	for (i = 0; i < argc - 2; i++)
	{
		uint32_t j;
		FILE *in;
		uint8_t buffer[0x1ffff];
		
		in = fopen(argv[i + 2], "rb");
		if (in == NULL)
		{
			printf("error opening in file: %s\n", argv[i + 2]);
			return 1;
		}
		
		size = fread(&buffer, 1, sizeof(buffer), in);
		if (feof(in) == 0)
		{
			printf("buffer too small\n");
			return 1;
		}
		
		fclose(in);
		
		snprintf(pak_entries[i].name, sizeof(pak_entries[i].name), "%s", argv[i + 2]);
		
		for (j = 0; j < sizeof(pak_entries[i].name); j++)
		{
			pak_entries[i].name[j] = tolower((int32_t)pak_entries[i].name[j]);
			
			if (pak_entries[i].name[j] == '\0')
			{
				memset(pak_entries[i].name + j, 0, sizeof(pak_entries[i].name) - j);
				break;
			}
		}
		
		pak_entries[i].offset = pak_header.offset;
		pak_entries[i].size = size;
		
		size = fwrite(buffer, size, 1, fp);
		if (size != 1)
		{
			printf("error writing data\n");
			return 1;
		}
		
		pak_header.offset += pak_entries[i].size;
		
#ifdef __BIG_ENDIAN__
		pak_entries[i].offset = swap_32(pak_entries[i].offset);
		pak_entries[i].size = swap_32(pak_entries[i].size);
#endif
	}
	
	size = fwrite(pak_entries, pak_header.size, 1, fp);
	if (size != 1)
	{
		printf("error writing pak directory\n");
		return 1;
	}
	
#ifdef __BIG_ENDIAN__
	pak_header.offset = swap_32(pak_header.offset);
	pak_header.size = swap_32(pak_header.size);
#endif
	
	fseek(fp, 0, SEEK_SET);
	fwrite(&pak_header, sizeof(pak_header_t), 1, fp);
	
	fclose(fp);
	
	return 0;
}
