/*
 * lmp2wad
 *
 * Copyright (C) 2012 Florian Zwoch <fzwoch@gmail.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdint.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>

#ifdef __BIG_ENDIAN__
static uint16_t swap_16(uint16_t x)
{
	return (x >> 8) | (x << 8);
}

static uint32_t swap_32(uint32_t x)
{
	return (swap_16(x & 0xffff) << 16) | (swap_16(x >> 16));
}
#endif

typedef struct
{
	char magic[4];
	uint32_t entries;
	uint32_t offset;
}
wad_header_t;

typedef struct
{
	uint32_t offset;
	uint32_t dsize;
	uint32_t size;
	uint8_t type;
	uint8_t cmprs;
	uint16_t unused;
	char name[16];
}
wad_entry_t;

typedef struct
{
	int32_t width;
	int32_t height;
}
lmp_header_t;

int main(int argc, char *argv[])
{
	FILE *fp;
	uint32_t i;
	wad_header_t wad_header;
	wad_entry_t wad_entries[1024];
	size_t size;
	
	if (argc < 3)
	{
		printf("usage: %s <.wad> <.lmp> [..]\n", argv[0]);
		return 1;
	}
	
	if (argc > sizeof(wad_entries) / sizeof(wad_entry_t) + 2)
	{
		printf("too many entries\n");
		return 1;
	}
	
	fp = fopen(argv[1], "wb");
	if (fp == NULL)
	{
		printf("error opening wad file: %s\n", argv[1]);
		return 1;
	}
	
	size = fwrite(&wad_header, sizeof(wad_header_t), 1, fp);
	if (size != 1)
	{
		printf("error writing wad header\n");
		return 1;
	}
	
	wad_header.magic[0] = 'W';
	wad_header.magic[1] = 'A';
	wad_header.magic[2] = 'D';
	wad_header.magic[3] = '2';
	
	wad_header.entries = argc - 2;
	wad_header.offset = sizeof(wad_header_t);
	
	for (i = 0; i < argc - 2; i++)
	{
		uint32_t j;
		FILE *lmp;
		uint8_t buffer[0xffff];
		lmp_header_t lmp_header;
		char *basename;
		
		lmp = fopen(argv[i + 2], "rb");
		if (lmp == NULL)
		{
			printf("error opening lmp file: %s\n", argv[i + 2]);
			return 1;
		}
		
		size = fread(&lmp_header, sizeof(lmp_header_t), 1, lmp);
		if (size != 1)
		{
			printf("error reading lmp header\n");
			return 1;
		}
		
		size = fread(&buffer, 1, sizeof(buffer), lmp);
		if (feof(lmp) == 0)
		{
			printf("buffer too small\n");
			return 1;
		}
		
		fclose(lmp);
		
		basename = strrchr(argv[i + 2], '/');
		if (basename == NULL)
		{
			basename = argv[i + 2];
		}
		else
		{
			basename++;
		}
		
		snprintf(wad_entries[i].name, sizeof(wad_entries[i].name), "%s", basename);
		
		for (j = 0; j < sizeof(wad_entries[i].name); j++)
		{
			wad_entries[i].name[j] = toupper((int32_t)wad_entries[i].name[j]);
			
			if (wad_entries[i].name[j] == '.')
			{
				memset(wad_entries[i].name + j, 0, sizeof(wad_entries[i].name) - j);
				break;
			}
		}
		
		wad_entries[i].offset = wad_header.offset;
		wad_entries[i].size = sizeof(lmp_header_t) + lmp_header.width * lmp_header.height;
		wad_entries[i].dsize = wad_entries[i].size;
		wad_entries[i].type = 0x42;
		wad_entries[i].cmprs = 0;
		wad_entries[i].unused = 0;
		
		if (strcmp(wad_entries[i].name, "CONCHARS") != 0)
		{
			size = fwrite(&lmp_header, sizeof(lmp_header_t), 1, fp);
			if (size != 1)
			{
				printf("error writing lmp header\n");
				return 1;
			}
		}
		else
		{
			wad_entries[i].size = lmp_header.width * lmp_header.height;
			wad_entries[i].dsize = wad_entries[i].size;
			wad_entries[i].type = 0x44;
		}
		
		size = fwrite(buffer, lmp_header.width * lmp_header.height, 1, fp);
		if (size != 1)
		{
			printf("error writing lmp data\n");
			return 1;
		}
		
		wad_header.offset += wad_entries[i].size;
		
#ifdef __BIG_ENDIAN__
		wad_entries[i].offset = swap_32(wad_entries[i].offset);
		wad_entries[i].dsize = swap_32(wad_entries[i].dsize);
		wad_entries[i].size = swap_32(wad_entries[i].size);
#endif
	}
	
	size = fwrite(wad_entries, wad_header.entries * sizeof(wad_entry_t), 1, fp);
	if (size != 1)
	{
		printf("error writing wad directory\n");
		return 1;
	}
	
#ifdef __BIG_ENDIAN__
	wad_header.entries = swap_32(wad_header.entries);
	wad_header.offset = swap_32(wad_header.offset);
#endif

	fseek(fp, 0, SEEK_SET);
	fwrite(&wad_header, sizeof(wad_header_t), 1, fp);
	
	fclose(fp);
	
	return 0;
}
