#/*
Copyright (C) 1996-1997 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/
// models.c -- model loading and caching

// models are the only shared resource between a client and server running
// on the same machine.

#include "quakedef.h"
#include "r_local.h"

#include "ds.h"

#define USE_3D
#define DS_MIPLEVEL 1
//#define USE_DEBUGGER
#define LOWMEM_LOADER

#ifdef USE_DEBUGGER

#include <user_debugger.h>
#define DEBUG_MSG(x) debug_msg(x)

#else
#define DEBUG_MSG(x)
#endif

model_t	*loadmodel;
char	loadname[32];	// for hunk tags

void Mod_LoadSpriteModel (model_t *mod, void *buffer);
void Mod_LoadBrushModel (model_t *mod, void *buffer);
void Mod_LoadAliasModel (model_t *mod, void *buffer);
model_t *Mod_LoadModel (model_t *mod, qboolean crash);

byte	mod_novis[MAX_MAP_LEAFS/8];

//for malice
//#define	MAX_MOD_KNOWN	170
//and quake
#define	MAX_MOD_KNOWN	256

model_t	mod_known[MAX_MOD_KNOWN];
int		mod_numknown = 0;

// values for model_t's needload
#define NL_PRESENT		0
#define NL_NEEDS_LOADED	1
#define NL_UNREFERENCED	2

extern unsigned int force_load_size;
extern unsigned int force_load_seek;
extern int force_no_memset;

extern int cache_uses_ex;

/*
===============
Mod_Init
===============
*/
void Mod_Init (void)
{
	ds_memset (mod_novis, 0xff, sizeof(mod_novis));
}

/*
===============
Mod_Extradata

Caches the data if needed
===============
*/
void *Mod_Extradata (model_t *mod)
{
	void	*r;
	
	r = Cache_Check (&mod->cache);
	if (r)
		return r;

	Mod_LoadModel (mod, true);
	
	if (!mod->cache.data)
		Sys_Error ("Mod_Extradata: caching failed");
	return mod->cache.data;
}

/*
===============
Mod_PointInLeaf
===============
*/
//mleaf_t *Mod_PointInLeaf (vec3_t p, model_t *model)
//{
//	mnode_t		*node;
//	float		d;
//	fixed_point f_d;
//	mplane_t	*plane;
//	
//	fixed_point f_p[3];
//	f_p[0] = p[0];
//	f_p[1] = p[1];
//	f_p[2] = p[2];
//	
//	if (!model || !model->nodes)
//		Sys_Error ("Mod_PointInLeaf: bad model");
//
//	node = model->nodes;
//	while (1)
//	{
//		if (node->contents < 0)
//			return (mleaf_t *)node;
//		plane = node->plane;
//		
////		f_d = DotProduct (f_p,plane->normal) - plane->dist;
//		f_d.value = ((long long)plane->normal[0].value * (long long)f_p[0].value
//						+ (long long)plane->normal[1].value * (long long)f_p[1].value
//						+ (long long)plane->normal[2].value * (long long)f_p[2].value) >> FP_SCALE_SHIFT;
//		f_d.value -= plane->dist.value;
//		
//		if (f_d.value > 0)
//			node = node->children[0];
//		else
//			node = node->children[1];
//	}
//	
//	return NULL;	// never reached
//}
//mleaf_t *Mod_PointInLeaf (vec3_t p, model_t *model)
//{
//	mnode_t		*node;
//	float		d;
//	mplane_t	*plane;
//	
//	if (!model || !model->nodes)
//		Sys_Error ("Mod_PointInLeaf: bad model");
//		
////	int big_p[3];
////	big_p[0] = p[0] * 1024;
////	big_p[1] = p[1] * 1024;
////	big_p[2] = p[2] * 1024;
//
//	node = model->nodes;
//	while (1)
//	{
//		if (node->contents < 0)
//			return (mleaf_t *)node;
//		plane = node->plane;
////		d = DotProduct (p,plane->normal) - plane->dist;
//		
//		d = DotProduct (p,plane->normal);
//		d -= plane->dist;
//		
////		d = DotProduct (p,plane->normal) - (plane->bigdist >> 10);
////		d = (DotProductII(big_p, plane->bignormal) - plane->bigdist) >> 10;
//		if (d > 0)
//			node = node->children[0];
//		else
//			node = node->children[1];
//	}
//	
//	return NULL;	// never reached
//}


/*
===================
Mod_DecompressVis
===================
*/
byte *Mod_DecompressVis (byte *in, model_t *model)
{
	static byte	decompressed[MAX_MAP_LEAFS/8];
	int		c;
	byte	*out;
	int		row;

	row = (model->numleafs+7)>>3;	
	out = decompressed;

	if (!in)
	{	// no vis info, so make all visible
		while (row)
		{
#ifdef USE_EXTRA_RAM
			byte_write(out, 0xff);
			out++;
#else
			*out++ = 0xff;
#endif
			row--;
		}
		return decompressed;		
	}

	do
	{
		if (*in)
		{
#ifdef USE_EXTRA_RAM
			byte_write(out, *in);
			out++;
			in++;
#else
			*out++ = *in++;
#endif
			continue;
		}
	
		c = in[1];
		in += 2;
		while (c)
		{
#ifdef USE_EXTRA_RAM
			byte_write(out, 0);
			out++;
#else
			*out++ = 0;
#endif
			c--;
		}
	} while (out - decompressed < row);
	
	return decompressed;
}

byte *Mod_LeafPVS (mleaf_t *leaf, model_t *model)
{
	if (leaf == model->leafs)
		return mod_novis;
	return Mod_DecompressVis (leaf->compressed_vis, model);
}

/*
===================
Mod_ClearAll
===================
*/
void Mod_ClearAll (void)
{
	int		i;
	model_t	*mod;

//	printf("MODEL CLEAR ALL!\n");

#ifdef USE_EXTRA_RAM
	for (i=0 , mod=mod_known ; i<mod_numknown ; i++, mod++) {
		byte_write(&mod->needload, NL_UNREFERENCED);
#else
	for (i=0 , mod=mod_known ; i<mod_numknown ; i++, mod++) {
		mod->needload = NL_UNREFERENCED;
#endif
//FIX FOR CACHE_ALLOC ERRORS:
		if (mod->type == mod_sprite) mod->cache.data = NULL;
	}
	
//	mod_numknown = 0;
}

/*
==================
Mod_FindName

==================
*/
model_t *Mod_FindName (char *name)
{
	int		i;
	model_t	*mod;
	model_t	*avail = NULL;

	if (!name[0])
		Sys_Error ("Mod_ForName: NULL name");
		
//
// search the currently loaded models
//
	for (i=0 , mod=mod_known ; i<mod_numknown ; i++, mod++)
	{
		if (!strcmp (mod->name, name) )
			break;
		if (mod->needload == NL_UNREFERENCED)
			if (!avail || mod->type != mod_alias)
				avail = mod;
	}
			
	if (i == mod_numknown)
	{
		if (mod_numknown == MAX_MOD_KNOWN)
		{
			if (avail)
			{
				mod = avail;
				if (mod->type == mod_alias)
					if (Cache_Check (&mod->cache))
						Cache_Free (&mod->cache);
			}
			else
				Sys_Error ("mod_numknown == MAX_MOD_KNOWN");
		}
		else
			mod_numknown++;
		Q_strcpy (mod->name, name);
#ifdef USE_EXTRA_RAM
		byte_write(&mod->needload, NL_NEEDS_LOADED);
#else
		mod->needload = NL_NEEDS_LOADED;
#endif
		
//		if (mod_numknown > 20)
//			while(1);
//		
//		printf("adding model %d, %s\n", mod_numknown, mod->name);
	}

	return mod;
}

/*
==================
Mod_TouchModel

==================
*/
void Mod_TouchModel (char *name)
{
	model_t	*mod;
	
	mod = Mod_FindName (name);
	
	if (mod->needload == NL_PRESENT)
	{
		if (mod->type == mod_alias)
			Cache_Check (&mod->cache);
	}
}

/*
==================
Mod_LoadModel

Loads a model into the cache
==================
*/
model_t *Mod_LoadModel (model_t *mod, qboolean crash)
{
//	printf("loading model %s\n", mod->name);
	unsigned *buf;
	byte	stackbuf[1024];		// avoid dirtying the cache heap
	
//	Hunk_Check ();
	
	if (strstr(mod->name, "sphere"))
		printf("sphere\n");

	if (mod->type == mod_alias)
	{
//		Hunk_Check ();
		if (Cache_Check (&mod->cache))
		{
//			Hunk_Check ();
#ifdef USE_EXTRA_RAM
			byte_write(&mod->needload, NL_PRESENT);
#else
			mod->needload = NL_PRESENT;
#endif
			return mod;
		}
	}
	else
	{
//		Hunk_Check ();
		if (mod->needload == NL_PRESENT)
			return mod;
	}
	
	Hunk_Check ();

//
// because the world is so huge, load it one piece at a time
//

//
// load the file
//      
	force_load_seek = 0;
	force_load_size = 4;

	unsigned int *header_id = (unsigned int *)COM_LoadTempFile(mod->name);
	
	force_load_seek = -1;
	force_load_size = -1;
	
	Hunk_Check ();
	
	if (header_id == NULL)
	{
		if (crash)
			Sys_Error ("Mod_NumForName: %s not found", mod->name);
		return NULL;
	}

	*header_id = LittleLong(*header_id);
	
	//
// allocate a new model
//
	COM_FileBase (mod->name, loadname);
	
//	Hunk_Check ();

	loadmodel = mod;

//
// fill it in
//

// call the apropriate loader
#ifdef USE_EXTRA_RAM
	byte_write(&mod->needload, NL_PRESENT);
#else
	mod->needload = NL_PRESENT;
#endif

#ifndef LOWMEM_LOADER
	if ((*header_id == IDPOLYHEADER) || (*header_id == IDSPRITEHEADER))
#else
	if (*header_id == IDSPRITEHEADER)
#endif
	{
		buf = (unsigned *)COM_LoadStackFile (mod->name, stackbuf, sizeof(stackbuf));
		if (!buf)
		{
			if (crash)
				Sys_Error ("Mod_NumForName: %s not found", mod->name);
			return NULL;
		}
		
		switch (LittleLong(*(unsigned *)buf))
		{
#ifndef LOWMEM_LOADER
		case IDPOLYHEADER:
//			Hunk_Check ();
			Mod_LoadAliasModel (mod, buf);
//			Hunk_Check ();
			break;
#endif
	
		case IDSPRITEHEADER:
//			Hunk_Check ();
			Mod_LoadSpriteModel (mod, buf);
//			Hunk_Check ();
			break;
		
		default:
			Sys_Error("it could be a brush model...\n");
			break;
		}
	}
#ifdef LOWMEM_LOADER
	else if (*header_id == IDPOLYHEADER)
	{
//		Hunk_Check ();
		Mod_LoadAliasModel (mod, NULL);
//		Hunk_Check ();
	}
#endif
	else
	{
//		printf("it's a brush model, so let\'s not load it\n");
//		Hunk_Check ();
		Mod_LoadBrushModel (mod, NULL);

#ifdef USE_EXTRA_RAM
//		printf("generating texture co-ordinates\n");
		
		for (int i = 0; i < mod->numsurfaces; i++)
		{
			mod->surfaces[i].texture_coordinates = NULL;
			
			if ( mod->surfaces[i].flags & SURF_DRAWTURB )
				continue;

			if ( mod->surfaces[i].flags & SURF_DRAWSKY )
				continue;
			
			generate_texcoords(mod->surfaces + i, mod);
		}
#endif
		
//		Hunk_Check ();
	}

//	printf("real header is %08x\n", *buf);
	
	Hunk_Check();

	return mod;
}

/*
==================
Mod_ForName

Loads in a model for the given name
==================
*/
model_t *Mod_ForName (char *name, qboolean crash)
{
	model_t	*mod;

	mod = Mod_FindName (name);

	return Mod_LoadModel (mod, crash);
}


/*
===============================================================================

					BRUSHMODEL LOADING

===============================================================================
*/

byte	*mod_base;

extern int lower_sky_handle, upper_sky_handle;
/*
=================
Mod_LoadTextures
=================
*/

char *get_last_found_filename(void);
int get_last_found_handle(void);
int get_last_found_seekpos(void);
bool bind_texture(int id);
void load_textures(void);

void Mod_LoadTextures (lump_t *l)
{
	int		i, j, pixels, num, max, altmax;
	miptex_t	*mt;
	texture_t	*tx, *tx2;
	texture_t	*anims[10];
	texture_t	*altanims[10];
	dmiptexlump_t *m;
	
	int used_texture_bytes = 0;

	if (!l->filelen)
	{
		loadmodel->textures = NULL;
		loadmodel->numtextures = 0;
		return;
	}
	m = (dmiptexlump_t *)(mod_base + l->fileofs);
	
	m->nummiptex = LittleLong (m->nummiptex);
//	printf("%d mip maps\n", m->nummiptex);
	
	loadmodel->numtextures = m->nummiptex;
	loadmodel->textures = (texture_t **)Hunk_AllocName (m->nummiptex * sizeof(*loadmodel->textures) , loadname);

	for (i=0 ; i<m->nummiptex ; i++)
	{
		m->dataofs[i] = LittleLong(m->dataofs[i]);
		if (m->dataofs[i] == -1)
			continue;
		mt = (miptex_t *)((byte *)m + m->dataofs[i]);
		mt->width = LittleLong (mt->width);
		mt->height = LittleLong (mt->height);
		for (j=0 ; j<MIPLEVELS ; j++)
			mt->offsets[j] = LittleLong (mt->offsets[j]);
		
		if ( (mt->width & 15) || (mt->height & 15) )
			Sys_Error ("Texture %s is not 16 aligned", mt->name);
#ifndef USE_3D
		pixels = mt->width*mt->height/64*85;
#else
		pixels = mt->width * mt->height;
#endif
		used_texture_bytes += pixels;
#ifndef USE_3D
		tx = Hunk_AllocName (sizeof(texture_t) +pixels, loadname );
#else
		tx = (texture_t *)Hunk_AllocName (sizeof(texture_t), loadname );
#endif
		loadmodel->textures[i] = tx;

		ds_memcpy (tx->name, mt->name, sizeof(tx->name));
		
		tx->width = mt->width;
		tx->height = mt->height;
		
		for (j=0 ; j<MIPLEVELS ; j++)
			tx->offsets[j] = mt->offsets[j] + sizeof(texture_t) - sizeof(miptex_t);
#ifndef USE_3D
		// the pixels immediately follow the structures
		ds_memcpy ( tx+1, mt+1, pixels);
		
		unsigned char *tex_base = (byte *)tx + tx->offsets[0];
		int count;
		for (count = 0; count < pixels; count++)
			tex_base[count] = vid.colormap[0xFF00 + tex_base[count]];
#endif
		
		bool trans = false;
		if (!Q_strncmp(mt->name,"sky",3))
		{
			R_InitSky (/*(byte *)mt + tx->offsets[DS_MIPLEVEL]*/(byte *)(mt + 1) + 256 * 128);
			tx->texture_handle = upper_sky_handle;
		}
		else
		{
#ifdef USE_3D

#if DS_MIPLEVEL == 1
			unsigned char *tex_data = (byte *)(mt + 1) + tx->width * tx->height;
#elif DS_MIPLEVEL == 0
			unsigned char *tex_data = (byte *)(mt + 1);
#elif DS_MIPLEVEL == 2
			unsigned char *tex_data = (byte *)(mt + 1) + tx->width * tx->height + ((tx->width * tx->height) >> 2);
#endif

			tx->texture_handle = register_texture(true, loadmodel->name, get_last_found_handle(),
				(unsigned int)tex_data - (unsigned int)mod_base + get_last_found_seekpos(),
				tx->width >> DS_MIPLEVEL, tx->height >> DS_MIPLEVEL, trans, 0, 0, 0);

			if (tx->texture_handle == -1)
			{
				tx->texture_handle = r_notexture_mip->texture_handle;
	//			printf("couldn\'t load brush model texture\n");
			}
			
#endif
		}
	}

//
// sequence the animations
//
	for (i=0 ; i<m->nummiptex ; i++)
	{
		tx = loadmodel->textures[i];
		if (!tx || tx->name[0] != '+')
			continue;
		if (tx->anim_next)
			continue;	// allready sequenced

	// find the number of frames in the animation
		ds_memset (anims, 0, sizeof(anims));
		ds_memset (altanims, 0, sizeof(altanims));

		max = tx->name[1];
		altmax = 0;
		if (max >= 'a' && max <= 'z')
			max -= 'a' - 'A';
		if (max >= '0' && max <= '9')
		{
			max -= '0';
			altmax = 0;
			anims[max] = tx;
			max++;
		}
		else if (max >= 'A' && max <= 'J')
		{
			altmax = max - 'A';
			max = 0;
			altanims[altmax] = tx;
			altmax++;
		}
		else
			Sys_Error ("Bad animating texture %s", tx->name);

		for (j=i+1 ; j<m->nummiptex ; j++)
		{
			tx2 = loadmodel->textures[j];
			if (!tx2 || tx2->name[0] != '+')
				continue;
			if (strcmp (tx2->name+2, tx->name+2))
				continue;

			num = tx2->name[1];
			if (num >= 'a' && num <= 'z')
				num -= 'a' - 'A';
			if (num >= '0' && num <= '9')
			{
				num -= '0';
				anims[num] = tx2;
				if (num+1 > max)
					max = num + 1;
			}
			else if (num >= 'A' && num <= 'J')
			{
				num = num - 'A';
				altanims[num] = tx2;
				if (num+1 > altmax)
					altmax = num+1;
			}
			else
				Sys_Error ("Bad animating texture %s", tx->name);
		}
		
#define	ANIM_CYCLE	2
	// link them all together
		for (j=0 ; j<max ; j++)
		{
			tx2 = anims[j];
			if (!tx2)
				Sys_Error ("Missing frame %i of %s",j, tx->name);
			tx2->anim_total = max * ANIM_CYCLE;
			tx2->anim_min = j * ANIM_CYCLE;
			tx2->anim_max = (j+1) * ANIM_CYCLE;
			tx2->anim_next = anims[ (j+1)%max ];
			if (altmax)
				tx2->alternate_anims = altanims[0];
		}
		for (j=0 ; j<altmax ; j++)
		{
			tx2 = altanims[j];
			if (!tx2)
				Sys_Error ("Missing frame %i of %s",j, tx->name);
			tx2->anim_total = altmax * ANIM_CYCLE;
			tx2->anim_min = j * ANIM_CYCLE;
			tx2->anim_max = (j+1) * ANIM_CYCLE;
			tx2->anim_next = altanims[ (j+1)%altmax ];
			if (max)
				tx2->alternate_anims = anims[0];
		}
	}
}

/*
=================
Mod_LoadLighting
=================
*/
void Mod_LoadLighting (lump_t *l)
{
	if (!l->filelen)
	{
		loadmodel->lightdata = NULL;
		return;
	}
	loadmodel->lightdata = (unsigned char *)Hunk_AllocName ( l->filelen, loadname);	
	ds_memcpy (loadmodel->lightdata, mod_base + l->fileofs, l->filelen);
}


/*
=================
Mod_LoadVisibility
=================
*/
void Mod_LoadVisibility (lump_t *l)
{
	if (!l->filelen)
	{
		loadmodel->visdata = NULL;
		return;
	}
	loadmodel->visdata = (unsigned char *)Hunk_AllocName ( l->filelen, loadname);	
	ds_memcpy (loadmodel->visdata, mod_base + l->fileofs, l->filelen);
}


/*
=================
Mod_LoadEntities
=================
*/
void Mod_LoadEntities (lump_t *l)
{
	if (!l->filelen)
	{
		loadmodel->entities = NULL;
		return;
	}
	loadmodel->entities = (char *)Hunk_AllocName ( l->filelen, loadname);	
	ds_memcpy (loadmodel->entities, mod_base + l->fileofs, l->filelen);
}


/*
=================
Mod_LoadVertexes
=================
*/
#define floattov16(n)         ((short int)((n) * (1 << 12)))
void Mod_LoadVertexes (lump_t *l)
{
	dvertex_t	*in;
#ifdef USE_3D
	short	*out;
	short int	*real_out;
#else
	mvertex_t	*out;
#endif
	int			i, count;

	in = (dvertex_t *)(mod_base + l->fileofs);
	if (l->filelen % sizeof(*in))
		Sys_Error ("MOD_LoadBmodel: funny lump size in %s",loadmodel->name);
	count = l->filelen / sizeof(*in);
	
#ifdef USE_3D
//	out = (short int *)Hunk_AllocName ( count*sizeof(*out)*3, loadname);
//
//	loadmodel->tex_vertexes = out;
	loadmodel->numvertexes = count;

//	for ( i=0 ; i<count ; i++, in++, out += 3)
//	{
//		out[0] = (float)(LittleFloat (in->point[0]) * 4);
//		out[1] = (float)(LittleFloat (in->point[1]) * 4);
//		out[2] = (float)(LittleFloat (in->point[2]) * 4);
//	}
//	
//	in = (dvertex_t *)(mod_base + l->fileofs);
	real_out = (short int *)Hunk_AllocName ( 3 * count*sizeof(*real_out), loadname);
	loadmodel->real_vertexes = real_out;
	
	extern float tri2_scale;

	for ( i=0 ; i<count ; i++, in++, real_out += 3)
	{
//		real_out[0] = floattov16(tri2_scale * LittleFloat (in->point[0]));
//		real_out[1] = floattov16(tri2_scale * LittleFloat (in->point[1]));
//		real_out[2] = floattov16(tri2_scale * LittleFloat (in->point[2]));

		real_out[0] = LittleFloat (in->point[0]);
		real_out[1] = LittleFloat (in->point[1]);
		real_out[2] = LittleFloat (in->point[2]);
	}
#else
	out = Hunk_AllocName ( count*sizeof(*out), loadname);	
	
	loadmodel->vertexes = out;
	loadmodel->numvertexes = count;

	for ( i=0 ; i<count ; i++, in++, out++)
	{
		out->position[0] = LittleFloat (in->point[0]);
		out->position[1] = LittleFloat (in->point[1]);
		out->position[2] = LittleFloat (in->point[2]);
	}
#endif
}

/*
=================
Mod_LoadSubmodels
=================
*/
void Mod_LoadSubmodels (lump_t *l)
{
	dmodel_t	*in;
	dmodel_t	*out;
	int			i, j, count;

	in = (dmodel_t *)(mod_base + l->fileofs);
	if (l->filelen % sizeof(*in))
		Sys_Error ("MOD_LoadBmodel: funny lump size in %s",loadmodel->name);
	count = l->filelen / sizeof(*in);
	out = (dmodel_t *)Hunk_AllocName ( count*sizeof(*out), loadname);	

	loadmodel->submodels = out;
	loadmodel->numsubmodels = count;

	for ( i=0 ; i<count ; i++, in++, out++)
	{
		for (j=0 ; j<3 ; j++)
		{	// spread the mins / maxs by a pixel
			out->mins[j] = LittleFloat (in->mins[j]) - 1;
			out->maxs[j] = LittleFloat (in->maxs[j]) + 1;
			out->origin[j] = LittleFloat (in->origin[j]);
		}
		for (j=0 ; j<MAX_MAP_HULLS ; j++)
			out->headnode[j] = LittleLong (in->headnode[j]);
		out->visleafs = LittleLong (in->visleafs);
		out->firstface = LittleLong (in->firstface);
		out->numfaces = LittleLong (in->numfaces);
	}
}

/*
=================
Mod_LoadEdges
=================
*/
void Mod_LoadEdges (lump_t *l)
{
	dedge_t *in;
	medge_t *out;
	int 	i, count;

	in = (dedge_t *)(mod_base + l->fileofs);
	if (l->filelen % sizeof(*in))
		Sys_Error ("MOD_LoadBmodel: funny lump size in %s",loadmodel->name);
	count = l->filelen / sizeof(*in);
	out = (medge_t *)Hunk_AllocName ( (count + 1) * sizeof(*out), loadname);	

	loadmodel->edges = out;
	loadmodel->numedges = count;

	for ( i=0 ; i<count ; i++, in++, out++)
	{
		out->v[0] = (unsigned short)LittleShort(in->v[0]);
		out->v[1] = (unsigned short)LittleShort(in->v[1]);
	}
}

/*
=================
Mod_LoadTexinfo
=================
*/
void Mod_LoadTexinfo (lump_t *l)
{
	texinfo_t *in;
	mtexinfo_t *out;
	int 	i, j, count;
	int		miptex;
	float	len1, len2;
	
	//return;

	in = (texinfo_t *)(mod_base + l->fileofs);
	if (l->filelen % sizeof(*in))
		Sys_Error ("MOD_LoadBmodel: funny lump size in %s",loadmodel->name);
	count = l->filelen / sizeof(*in);
	
//	if (loadmodel->texinfo == NULL)
	{
		out = (mtexinfo_t *)Hunk_AllocName ( count*sizeof(*out), loadname);	
	
		loadmodel->texinfo = out;
	}
	loadmodel->numtexinfo = count;
	
	i = 0;

	for ( i=0 ; i<count ; i++)
	{
		for (j=0 ; j<8 ; j++)
//			out->vecs[0][j] = LittleFloat (in->vecs[0][j]);
			out->vecs[0][j].set_value_by_float(in->vecs[0][j]);
//		len1 = Length (out->vecs[0]);
//		len2 = Length (out->vecs[1]);
		len1 = LengthFP (out->vecs[0]);
		len2 = LengthFP (out->vecs[1]); 
		len1 = (len1 + len2)/2;
		if (len1 < 0.32)
			out->mipadjust = 4;
		else if (len1 < 0.49)
			out->mipadjust = 3;
		else if (len1 < 0.99)
			out->mipadjust = 2;
		else
			out->mipadjust = 1;
#if 0
		if (len1 + len2 < 0.001)
			out->mipadjust = 1;		// don't crash
		else
			out->mipadjust = 1 / floor( (len1+len2)/2 + 0.1 );
#endif

		miptex = LittleLong (in->miptex);
		out->flags = LittleLong (in->flags);
	
		if (!loadmodel->textures)
		{
			out->texture = r_notexture_mip;	// checkerboard texture
			out->flags = 0;
		}
		else
		{
			if (miptex >= loadmodel->numtextures)
				Sys_Error ("miptex >= loadmodel->numtextures");
			out->texture = loadmodel->textures[miptex];
			if (!out->texture)
			{
				out->texture = r_notexture_mip; // texture not found
				out->flags = 0;
			}
		}
		
		in++; out++;
	}
}

/*
================
CalcSurfaceExtents

Fills in s->texturemins[] and s->extents[]
================
*/
#define v16toint(n)          ((n) >> 12)
//void CalcSurfaceExtents (msurface_t *s)
//{
//	float	mins[2], maxs[2], val;
//	int		i,j, e;
//	mvertex_t	*v;
//	mtexinfo_t	*tex;
//	int		bmins[2], bmaxs[2];
//
//	mins[0] = mins[1] = 999999;
//	maxs[0] = maxs[1] = -99999;
//
//	tex = s->texinfo;
//	
//	for (i=0 ; i<s->numedges ; i++)
//	{
//		e = loadmodel->surfedges[s->firstedge+i];
//		if (e >= 0)
//			v = &loadmodel->vertexes[loadmodel->edges[e].v[0]];
//		else
//			v = &loadmodel->vertexes[loadmodel->edges[-e].v[1]];
//		
//		for (j=0 ; j<2 ; j++)
//		{
//			val = v->position[0] * tex->vecs[j][0] + 
//				v->position[1] * tex->vecs[j][1] +
//				v->position[2] * tex->vecs[j][2] +
//				tex->vecs[j][3];
//			if (val < mins[j])
//				mins[j] = val;
//			if (val > maxs[j])
//				maxs[j] = val;
//		}
//	}
//
//	for (i=0 ; i<2 ; i++)
//	{	
//		bmins[i] = floor(mins[i]/16);
//		bmaxs[i] = ceil(maxs[i]/16);
//
//		s->texturemins[i] = bmins[i] * 16;
//		s->extents[i] = (bmaxs[i] - bmins[i]) * 16;
//		tex->flags = TEX_SPECIAL;
//		if ( !(tex->flags & TEX_SPECIAL) && s->extents[i] > 256)
//			Sys_Error ("Bad surface extents");
//	}
//}

//void CalcSurfaceExtents (msurface_t *s)
//{
//	float	mins[2], maxs[2], val;
//	int		i,j, e;
//#ifdef USE_3D
//	short	*v;
//#else
//	mvertex_t	*v;
//#endif
//	mtexinfo_t	*tex;
//	int		bmins[2], bmaxs[2];
//
//	mins[0] = mins[1] = 999999;
//	maxs[0] = maxs[1] = -99999;
//
//	tex = s->texinfo;
//	
//	for (i=0 ; i<s->numedges ; i++)
//	{
//		e = loadmodel->surfedges[s->firstedge+i];
//#ifdef USE_3D
//		if (e >= 0)
//			v = &loadmodel->real_vertexes[loadmodel->edges[e].v[0] * 3];
//		else
//			v = &loadmodel->real_vertexes[loadmodel->edges[-e].v[1] * 3];
//#else
//		if (e >= 0)
//			v = &loadmodel->vertexes[loadmodel->edges[e].v[0]];
//		else
//			v = &loadmodel->vertexes[loadmodel->edges[-e].v[1]];
//#endif
//		
//		for (j=0 ; j<2 ; j++)
//		{
//#ifdef USE_3D
//			val = ((v[0])) * tex->vecs[j][0] + 
//				((v[1])) * tex->vecs[j][1] +
//				((v[2])) * tex->vecs[j][2] +
//				tex->vecs[j][3];
//#else
//			val = v->position[0] * tex->vecs[j][0] + 
//				v->position[1] * tex->vecs[j][1] +
//				v->position[2] * tex->vecs[j][2] +
//				tex->vecs[j][3];
//#endif
//			if (val < mins[j])
//				mins[j] = val;
//			if (val > maxs[j])
//				maxs[j] = val;
//		}
//	}
//
//	for (i=0 ; i<2 ; i++)
//	{	
//		bmins[i] = floor(mins[i]/16);
//		bmaxs[i] = ceil(maxs[i]/16);
//
//		s->texturemins[i] = bmins[i] * 16;
//		s->extents[i] = (bmaxs[i] - bmins[i]) * 16;
//		tex->flags = TEX_SPECIAL;
//		if ( !(tex->flags & TEX_SPECIAL) && s->extents[i] > 256)
//			Sys_Error ("Bad surface extents");
//	}
//}

void CalcSurfaceExtents (msurface_t *s)
{
	float	mins[2], maxs[2], val;
	int		i,j, e;
	short	*v;
	mtexinfo_t	*tex;
	int		bmins[2], bmaxs[2];

	mins[0] = mins[1] = 999999;
	maxs[0] = maxs[1] = -99999;

	tex = s->texinfo;
	
//	printf("%d edges\n", s->numedges);
	
	for (i=0 ; i<s->numedges ; i++)
	{
		e = loadmodel->surfedges[s->firstedge+i];

		if (e >= 0)
			v = &loadmodel->real_vertexes[loadmodel->edges[e].v[0] * 3];
		else
			v = &loadmodel->real_vertexes[loadmodel->edges[-e].v[1] * 3];
		
		for (j=0 ; j<2 ; j++)
		{
			val = ((v[0])) * tex->vecs[j][0] + 
				((v[1])) * tex->vecs[j][1] +
				((v[2])) * tex->vecs[j][2] +
				tex->vecs[j][3];
			
//			printf("e %d v %d is %.2f\n", i, j, val);

			if (val < mins[j])
				mins[j] = val;
			if (val > maxs[j])
				maxs[j] = val;
		}
	}

	s->texturemins[0] = 0;
	s->texturemins[1] = 0;
	
	if (next_size_up(tex->texture->width) != tex->texture->width)
	{
		int x_shunt = 0;
		while (mins[0] + x_shunt < 0)
			x_shunt += tex->texture->width;
		
		while (mins[0] + x_shunt > tex->texture->width)
			x_shunt -= tex->texture->width;
		
		while (maxs[0] + x_shunt < 0)
			x_shunt += tex->texture->width;
		
		while (maxs[0] + x_shunt > tex->texture->width)
			x_shunt -= tex->texture->width;
		
		if ((maxs[0] + x_shunt >= 0) && (maxs[0] + x_shunt <= tex->texture->width)
			&& (mins[0] + x_shunt >= 0) && (mins[0] + x_shunt <= tex->texture->width))
		{
//			printf("texture \"%s\" fits ok with x shunt %d\n", tex->texture->name, x_shunt);
		}
		else
		{
//			printf("texture \"%s\" does not fit ok in x\n", tex->texture->name, x_shunt);
//			while(1);
		}
		s->texturemins[0] = x_shunt << FP_SCALE_SHIFT;
	}
		
	if (next_size_up(tex->texture->height) != tex->texture->height)
	{
		int y_shunt = 0;
		
		while (mins[1] + y_shunt < 0)
			y_shunt += tex->texture->height;
		
		while (mins[1] + y_shunt > tex->texture->height)
			y_shunt -= tex->texture->height;
		
		while (maxs[1] + y_shunt < 0)
			y_shunt += tex->texture->height;
		
		while (maxs[1] + y_shunt > tex->texture->height)
			y_shunt -= tex->texture->height;
		
		if ((maxs[1] + y_shunt >= 0) && (maxs[1] + y_shunt <= tex->texture->height)
			&& (mins[1] + y_shunt >= 0) && (mins[1] + y_shunt <= tex->texture->height))
		{
//			printf("texture \"%s\" fits ok with y shunt %d\n", tex->texture->name, y_shunt);
		}
		else
		{
//			printf("texture \"%s\" does not fit ok in y\n", tex->texture->name, y_shunt);
//			while(1);
		}
		
		s->texturemins[1] = y_shunt << FP_SCALE_SHIFT;
	}
}



/*
=================
Mod_LoadFaces
=================
*/
unsigned char *fake_lightmaps = NULL;
void Mod_LoadFaces (lump_t *l)
{
	dface_t		*in;
	msurface_t 	*out;
	int			i, count, surfnum;
	int			planenum, side;
	
	//return;

	in = (dface_t *)(mod_base + l->fileofs);
	if (l->filelen % sizeof(*in))
		Sys_Error ("MOD_LoadBmodel: funny lump size in %s",loadmodel->name);
	count = l->filelen / sizeof(*in);
	
	unsigned int orig_i = LittleLong(in->lightofs);
	
	out = (msurface_t *)Hunk_AllocName ( count*sizeof(*out), loadname);
	
	unsigned int new_i = LittleLong(in->lightofs);
	
	if (new_i != orig_i)
		Sys_Error("ran out of memory whilst loading in faces and lightmaps\n");	

	loadmodel->surfaces = out;
	loadmodel->numsurfaces = count;

	for ( surfnum=0 ; surfnum<count ; surfnum++, in++, out++)
	{
		out->firstedge = LittleLong(in->firstedge);
		out->numedges = LittleShort(in->numedges);		
		out->flags = 0;

		planenum = LittleShort(in->planenum);
		side = LittleShort(in->side);
		if (side)
			out->flags |= SURF_PLANEBACK;			

		out->plane = loadmodel->planes + planenum;

		out->texinfo = loadmodel->texinfo + LittleShort (in->texinfo);

		CalcSurfaceExtents (out);

	// lighting info

#ifdef USE_EXTRA_RAM
		for (i=0 ; i<MAXLIGHTMAPS ; i++)
			byte_write(&out->styles[i], in->styles[i]);
#else
		for (i=0 ; i<MAXLIGHTMAPS ; i++)
			out->styles[i] = in->styles[i];
#endif

		i = LittleLong(in->lightofs);
		if (i == -1)
			out->samples = (unsigned char *)(1 << 10);
//			out->samples = NULL;
		else
		{
			int total = 255;
//			if (fake_lightmaps != NULL)
//				total = fake_lightmaps[i];
			if (fake_lightmaps != NULL)
			{
				int smax = (out->extents[0]>>4)+1;
				int tmax = (out->extents[1]>>4)+1;
				
				total = 0;
				
//				printf("lightmaps at %08x, i is %d, s/t %d %d\n", fake_lightmaps, i, smax, tmax);
//				while(1);
				
				for (int index = 0; index < smax * tmax; index++)
					total += fake_lightmaps[i + index];
				
				total = total / (smax * tmax);
			}
			
			out->samples = (unsigned char *)total;//loadmodel->lightdata + i;
		}
		
	// set the drawing flags flag
		
		if (!Q_strncmp(out->texinfo->texture->name,"sky",3))	// sky
		{
			out->flags |= (SURF_DRAWSKY | SURF_DRAWTILED);
			continue;
		}
		
		if (!Q_strncmp(out->texinfo->texture->name,"*",1))		// turbulent
		{
			out->flags |= (SURF_DRAWTURB | SURF_DRAWTILED);
			for (i=0 ; i<2 ; i++)
			{
				out->extents[i] = 16384;
				out->texturemins[i] = -8192;
			}
			continue;
		}
	}
	fake_lightmaps = NULL;
}


/*
=================
Mod_SetParent
=================
*/
void Mod_SetParent (mnode_t *node, mnode_t *parent)
{
	node->parent = parent;
	if (node->contents < 0)
		return;
	Mod_SetParent (node->children[0], node);
	Mod_SetParent (node->children[1], node);
}

/*
=================
Mod_LoadNodes
=================
*/
void Mod_LoadNodes (lump_t *l)
{
	int			i, j, count, p;
	dnode_t		*in;
	mnode_t 	*out;

	in = (dnode_t *)(mod_base + l->fileofs);
	if (l->filelen % sizeof(*in))
		Sys_Error ("MOD_LoadBmodel: funny lump size in %s",loadmodel->name);
	count = l->filelen / sizeof(*in);
	out = (mnode_t *)Hunk_AllocName ( count*sizeof(*out), loadname);	

	loadmodel->nodes = out;
	loadmodel->numnodes = count;

	for ( i=0 ; i<count ; i++, in++, out++)
	{
		for (j=0 ; j<3 ; j++)
		{
			out->minmaxs[j] = LittleShort (in->mins[j]);
			out->minmaxs[3+j] = LittleShort (in->maxs[j]);
		}
	
		p = LittleLong(in->planenum);
		out->plane = loadmodel->planes + p;

		out->firstsurface = LittleShort (in->firstface);
		out->numsurfaces = LittleShort (in->numfaces);
		
		for (j=0 ; j<2 ; j++)
		{
			p = LittleShort (in->children[j]);
			if (p >= 0)
				out->children[j] = loadmodel->nodes + p;
			else
				out->children[j] = (mnode_t *)(loadmodel->leafs + (-1 - p));
		}
	}
	
	Mod_SetParent (loadmodel->nodes, NULL);	// sets nodes and leafs
}

/*
=================
Mod_LoadLeafs
=================
*/
void Mod_LoadLeafs (lump_t *l)
{
	dleaf_t 	*in;
	mleaf_t 	*out;
	int			i, j, count, p;

	in = (dleaf_t *)(mod_base + l->fileofs);
	if (l->filelen % sizeof(*in))
		Sys_Error ("MOD_LoadBmodel: funny lump size in %s",loadmodel->name);
	count = l->filelen / sizeof(*in);
	out = (mleaf_t *)Hunk_AllocName ( count*sizeof(*out), loadname);	

	loadmodel->leafs = out;
	loadmodel->numleafs = count;

	for ( i=0 ; i<count ; i++, in++, out++)
	{
		for (j=0 ; j<3 ; j++)
		{
			out->minmaxs[j] = LittleShort (in->mins[j]);
			out->minmaxs[3+j] = LittleShort (in->maxs[j]);
		}

		p = LittleLong(in->contents);
		out->contents = p;

		out->firstmarksurface = loadmodel->marksurfaces +
			LittleShort(in->firstmarksurface);
		out->nummarksurfaces = LittleShort(in->nummarksurfaces);
		
		p = LittleLong(in->visofs);
		if (p == -1)
			out->compressed_vis = NULL;
		else
			out->compressed_vis = loadmodel->visdata + p;
		out->efrags = NULL;
		
#ifdef USE_EXTRA_RAM
		for (j=0 ; j<4 ; j++)
			byte_write(&out->ambient_sound_level[j], in->ambient_level[j]);
#else
		for (j=0 ; j<4 ; j++)
			out->ambient_sound_level[j] = in->ambient_level[j];
#endif
	}	
}

/*
=================
Mod_LoadClipnodes
=================
*/
void Mod_LoadClipnodes (lump_t *l)
{
	dclipnode_t *in, *out;
	int			i, count;
	hull_t		*hull;

	in = (dclipnode_t *)(mod_base + l->fileofs);
	if (l->filelen % sizeof(*in))
		Sys_Error ("MOD_LoadBmodel: funny lump size in %s",loadmodel->name);
	count = l->filelen / sizeof(*in);
	out = (dclipnode_t *)Hunk_AllocName ( count*sizeof(*out), loadname);	

	loadmodel->clipnodes = out;
	loadmodel->numclipnodes = count;

	hull = &loadmodel->hulls[1];
	hull->clipnodes = out;
	hull->firstclipnode = 0;
	hull->lastclipnode = count-1;
	hull->planes = loadmodel->planes;
	hull->clip_mins[0] = -16;
	hull->clip_mins[1] = -16;
	hull->clip_mins[2] = -24;
	hull->clip_maxs[0] = 16;
	hull->clip_maxs[1] = 16;
	hull->clip_maxs[2] = 32;

	hull = &loadmodel->hulls[2];
	hull->clipnodes = out;
	hull->firstclipnode = 0;
	hull->lastclipnode = count-1;
	hull->planes = loadmodel->planes;
	hull->clip_mins[0] = -32;
	hull->clip_mins[1] = -32;
	hull->clip_mins[2] = -24;
	hull->clip_maxs[0] = 32;
	hull->clip_maxs[1] = 32;
	hull->clip_maxs[2] = 64;

	for (i=0 ; i<count ; i++, out++, in++)
	{
		out->planenum = LittleLong(in->planenum);
		out->children[0] = LittleShort(in->children[0]);
		out->children[1] = LittleShort(in->children[1]);
	}
}

/*
=================
Mod_MakeHull0

Deplicate the drawing hull structure as a clipping hull
=================
*/
void Mod_MakeHull0 (void)
{
	mnode_t		*in, *child;
	dclipnode_t *out;
	int			i, j, count;
	hull_t		*hull;
	
	hull = &loadmodel->hulls[0];	
	
	in = loadmodel->nodes;
	count = loadmodel->numnodes;
	out = (dclipnode_t *)Hunk_AllocName ( count*sizeof(*out), loadname);	

	hull->clipnodes = out;
	hull->firstclipnode = 0;
	hull->lastclipnode = count-1;
	hull->planes = loadmodel->planes;

	for (i=0 ; i<count ; i++, out++, in++)
	{
		out->planenum = in->plane - loadmodel->planes;
		for (j=0 ; j<2 ; j++)
		{
			child = in->children[j];
			if (child->contents < 0)
				out->children[j] = child->contents;
			else
				out->children[j] = child - loadmodel->nodes;
		}
	}
}

/*
=================
Mod_LoadMarksurfaces
=================
*/
void Mod_LoadMarksurfaces (lump_t *l)
{	
	int		i, j, count;
	short		*in;
	msurface_t **out;
	
	//return;
	
	in = (short *)(mod_base + l->fileofs);
	if (l->filelen % sizeof(*in))
		Sys_Error ("MOD_LoadBmodel: funny lump size in %s",loadmodel->name);
	count = l->filelen / sizeof(*in);
	out = (msurface_t **)Hunk_AllocName ( count*sizeof(*out), loadname);	

	loadmodel->marksurfaces = out;
	loadmodel->nummarksurfaces = count;

	for ( i=0 ; i<count ; i++)
	{
		j = LittleShort(in[i]);
		if (j >= loadmodel->numsurfaces)
			Sys_Error ("Mod_ParseMarksurfaces: bad surface number");
		out[i] = loadmodel->surfaces + j;
	}
}

/*
=================
Mod_LoadSurfedges
=================
*/
void Mod_LoadSurfedges (lump_t *l)
{	
	int		i, count;
	int		*in, *out;
	
	in = (int *)(mod_base + l->fileofs);
	if (l->filelen % sizeof(*in))
		Sys_Error ("MOD_LoadBmodel: funny lump size in %s",loadmodel->name);
	count = l->filelen / sizeof(*in);
	out = (int *)Hunk_AllocName ( count*sizeof(*out), loadname);	

	loadmodel->surfedges = out;
	loadmodel->numsurfedges = count;

	for ( i=0 ; i<count ; i++)
		out[i] = LittleLong (in[i]);
}

/*
=================
Mod_LoadPlanes
=================
*/
void Mod_LoadPlanes (lump_t *l)
{
	int			i, j;
	mplane_t	*out;
	dplane_t 	*in;
	int			count;
	int			bits;

	//return;

	in = (dplane_t *)(mod_base + l->fileofs);
	if (l->filelen % sizeof(*in))
		Sys_Error ("MOD_LoadBmodel: funny lump size in %s",loadmodel->name);
	count = l->filelen / sizeof(*in);
	out = (mplane_t *)Hunk_AllocName ( count*2*sizeof(*out), loadname);	
	
	loadmodel->planes = out;
	loadmodel->numplanes = count;

	for ( i=0 ; i<count ; i++, in++, out++)
	{
		bits = 0;
		for (j=0 ; j<3 ; j++)
		{
//			out->normal[j] = LittleFloat (in->normal[j]);
			out->normal[j].set_value_by_float(in->normal[j]);
			if (out->normal[j] < 0)
				bits |= 1<<j;
//			out->bignormal[j] = LittleFloat (in->normal[j]) * 1024;
//			if (out->bignormal[j] < 0)
//				bits |= 1<<j;
		}

//		out->dist = LittleFloat (in->dist);
		out->dist.set_value_by_float(in->dist);
//		out->bigdist = LittleFloat (in->dist) * 1024;
#ifdef USE_EXTRA_RAM
		byte_write(&out->type, LittleLong (in->type));
		byte_write(&out->signbits, bits);
#else
		out->type = LittleLong (in->type);
		out->signbits = bits;
#endif
	}
}

/*
=================
RadiusFromBounds
=================
*/
float RadiusFromBounds (vec3_t mins, vec3_t maxs)
{
	int		i;
	vec3_t	corner;

	for (i=0 ; i<3 ; i++)
	{
		corner[i] = fabs(mins[i]) > fabs(maxs[i]) ? fabs(mins[i]) : fabs(maxs[i]);
	}

	return Length (corner);
}

/*
=================
Mod_LoadBrushModel
=================
*/

//int need_reloading_count = 0;
//char need_reloading[50][50];

//void load_textures_and_lightmaps(char *mod_name)
//{
//	int i;
//	char print_buf[50];
//	dheader_t	*header = (dheader_t *)malloc(sizeof(dheader_t));;
//	
//	loadmodel = Mod_FindName(mod_name);
//	
//	force_load_seek = -1;
//	force_load_size = sizeof(dheader_t);
//	
//	void *temp_load = COM_LoadTempFile(mod_name);
//	memcpy(header, temp_load, sizeof(dheader_t));
//
//	i = LittleLong (header->version);
//	if (i != BSPVERSION)
//		Sys_Error ("Mod_LoadBrushModel: %s has wrong version number (%i should be %i)", mod_name, i, BSPVERSION);
//
//// swap all the lumps
//	mod_base = (byte *)header;
//
//	for (i=0 ; i<sizeof(dheader_t)/4 ; i++)
//		((int *)header)[i] = LittleLong ( ((int *)header)[i]);
//	
/////
//	force_load_seek = header->lumps[LUMP_TEXTURES].fileofs;
//	force_load_size = header->lumps[LUMP_TEXTURES].filelen;
//	sprintf(print_buf, "textures, %d bytes\n", force_load_size);
//	DEBUG_MSG(print_buf);
//	temp_load = COM_LoadTempFile(mod_name);
//	mod_base = (unsigned char *)temp_load - header->lumps[LUMP_TEXTURES].fileofs;
//
//	Mod_LoadTextures (&header->lumps[LUMP_TEXTURES]);
///*///
//	force_load_seek = header->lumps[LUMP_LIGHTING].fileofs;
//	force_load_size = header->lumps[LUMP_LIGHTING].filelen;
//	sprintf(print_buf, "lighting, %d bytes\n", force_load_size);
//	DEBUG_MSG(print_buf);
//	temp_load = COM_LoadTempFile(mod_name);
//	mod_base = temp_load - header->lumps[LUMP_LIGHTING].fileofs;
//
//	Mod_LoadLighting (&header->lumps[LUMP_LIGHTING]);*/
/////
//	force_load_seek = header->lumps[LUMP_TEXINFO].fileofs;
//	force_load_size = header->lumps[LUMP_TEXINFO].filelen;
//	sprintf(print_buf, "tex info, %d bytes\n", force_load_size);
//	DEBUG_MSG(print_buf);
//	temp_load = COM_LoadTempFile(mod_name);
//	mod_base = (unsigned char *)temp_load - header->lumps[LUMP_TEXINFO].fileofs;
//
//	Mod_LoadTexinfo (&header->lumps[LUMP_TEXINFO]);
/////
//	force_load_seek = -1;
//	force_load_size = -1;
//	
//	free(header);
//}

void Mod_LoadBrushModel (model_t *mod, void *buffer)
{
	int			i, j;
	dheader_t	*header = (dheader_t *)malloc(sizeof(dheader_t));;
	dmodel_t 	*bm;
	
	if (header == NULL)
		Sys_Error("not enough memory to allocate a file header!\n");
		
//	extern void *hunk_base;
//	extern unsigned int hunk_size;
	
//	dump_ram("after_ex1.bin", hunk_base, hunk_size);

#ifdef USE_EXTRA_RAM
	byte_write(&loadmodel->type, mod_brush);
#else
	loadmodel->type = mod_brush;
#endif

//	header = (dheader_t *)buffer;

	force_load_seek = 0;
	force_load_size = sizeof(dheader_t);
	
	void *temp_load = COM_LoadTempFile(mod->name);
	ds_memcpy(header, temp_load, sizeof(dheader_t));

	i = LittleLong (header->version);
	if (i != BSPVERSION)
		Sys_Error ("Mod_LoadBrushModel: %s has wrong version number (%i should be %i)", mod->name, i, BSPVERSION);

// swap all the lumps
	mod_base = (byte *)header;

	for (i=0 ; i<sizeof(dheader_t)/4 ; i++)
		((int *)header)[i] = LittleLong ( ((int *)header)[i]);
	
//	dump_ram("after_ex2.bin", hunk_base, hunk_size);

// load into heap

	char print_buf[50];
	sprintf(print_buf, "loading brush model %s\n", mod->name);
///
	force_load_seek = header->lumps[LUMP_VERTEXES].fileofs;
	force_load_size = header->lumps[LUMP_VERTEXES].filelen;
	sprintf(print_buf, "vertices, %d bytes\n", force_load_size);
//	DEBUG_MSG(print_buf);
	temp_load = COM_LoadTempFile(mod->name);
	mod_base = (unsigned char *)temp_load - header->lumps[LUMP_VERTEXES].fileofs;

	Mod_LoadVertexes (&header->lumps[LUMP_VERTEXES]);
	
//	dump_ram("after_ex3.bin", hunk_base, hunk_size);
///
	force_load_seek = header->lumps[LUMP_EDGES].fileofs;
	force_load_size = header->lumps[LUMP_EDGES].filelen;
	sprintf(print_buf, "edges, %d bytes\n", force_load_size);
//	DEBUG_MSG(print_buf);
	temp_load = COM_LoadTempFile(mod->name);
	mod_base = (unsigned char *)temp_load - header->lumps[LUMP_EDGES].fileofs;

	Mod_LoadEdges (&header->lumps[LUMP_EDGES]);
	
//	dump_ram("after_ex4.bin", hunk_base, hunk_size);
///
	force_load_seek = header->lumps[LUMP_SURFEDGES].fileofs;
	force_load_size = header->lumps[LUMP_SURFEDGES].filelen;
	sprintf(print_buf, "surf edges, %d bytes\n", force_load_size);
//	DEBUG_MSG(print_buf);
	temp_load = COM_LoadTempFile(mod->name);
	mod_base = (unsigned char *)temp_load - header->lumps[LUMP_SURFEDGES].fileofs;

	Mod_LoadSurfedges (&header->lumps[LUMP_SURFEDGES]);
	
//	dump_ram("after_ex5.bin", hunk_base, hunk_size);
	
//	strcpy(need_reloading[need_reloading_count], mod->name);
//	need_reloading_count++;
//	
//	if (need_reloading_count > 49)
//		Sys_Error("not enough reload space\n");
///
	force_load_seek = header->lumps[LUMP_TEXTURES].fileofs;
	force_load_size = header->lumps[LUMP_TEXTURES].filelen;
	sprintf(print_buf, "textures, %d bytes\n", force_load_size);
//	DEBUG_MSG(print_buf);
	temp_load = COM_LoadTempFile(mod->name);
	mod_base = (unsigned char *)temp_load - header->lumps[LUMP_TEXTURES].fileofs;

	Mod_LoadTextures (&header->lumps[LUMP_TEXTURES]);
	
//	dump_ram("after_ex6.bin", hunk_base, hunk_size);
///
//	force_load_seek = header->lumps[LUMP_LIGHTING].fileofs;
//	force_load_size = header->lumps[LUMP_LIGHTING].filelen;
//	sprintf(print_buf, "lighting, %d bytes\n", force_load_size);
////	DEBUG_MSG(print_buf);
//	temp_load = COM_LoadTempFile(mod->name);
//	mod_base = (unsigned char *)temp_load - header->lumps[LUMP_LIGHTING].fileofs;
//
//	Mod_LoadLighting (&header->lumps[LUMP_LIGHTING]);
///
	force_load_seek = header->lumps[LUMP_PLANES].fileofs;
	force_load_size = header->lumps[LUMP_PLANES].filelen;
	sprintf(print_buf, "planes, %d bytes\n", force_load_size);
//	DEBUG_MSG(print_buf);
	temp_load = COM_LoadTempFile(mod->name);
	mod_base = (unsigned char *)temp_load - header->lumps[LUMP_PLANES].fileofs;

	Mod_LoadPlanes (&header->lumps[LUMP_PLANES]);
	
//	dump_ram("after_ex7.bin", hunk_base, hunk_size);
///
	force_load_seek = header->lumps[LUMP_TEXINFO].fileofs;
	force_load_size = header->lumps[LUMP_TEXINFO].filelen;
	sprintf(print_buf, "tex info, %d bytes\n", force_load_size);
//	DEBUG_MSG(print_buf);
	temp_load = COM_LoadTempFile(mod->name);
	mod_base = (unsigned char *)temp_load - header->lumps[LUMP_TEXINFO].fileofs;

	Mod_LoadTexinfo (&header->lumps[LUMP_TEXINFO]);
	
//	dump_ram("after_ex8.bin", hunk_base, hunk_size);
///
	force_load_seek = header->lumps[LUMP_FACES].fileofs;
	force_load_size = header->lumps[LUMP_FACES].filelen;
	sprintf(print_buf, "faces, %d bytes\n", force_load_size);
//	DEBUG_MSG(print_buf);
//	unsigned char *face_buffer = (unsigned char *)ds_exram_base();
	unsigned char *face_buffer = (unsigned char *)Hunk_TempAlloc(
		header->lumps[LUMP_FACES].filelen
		+ header->lumps[LUMP_LIGHTING].filelen + 2);
	temp_load = COM_LoadTempFile(mod->name);
	temp_load = COM_LoadStackFile(mod->name, face_buffer, force_load_size + 1);
	
	force_load_seek = header->lumps[LUMP_LIGHTING].fileofs;
	force_load_size = header->lumps[LUMP_LIGHTING].filelen;
	fake_lightmaps = face_buffer + header->lumps[LUMP_FACES].filelen + 1;
	/*temp_load = */COM_LoadStackFile(mod->name,
		fake_lightmaps,
		force_load_size + 1);
	
	mod_base = (unsigned char *)temp_load - header->lumps[LUMP_FACES].fileofs;
//	while(1);

	Mod_LoadFaces (&header->lumps[LUMP_FACES]);
	
//	dump_ram("after_ex9.bin", hunk_base, hunk_size);
///
	force_load_seek = header->lumps[LUMP_MARKSURFACES].fileofs;
	force_load_size = header->lumps[LUMP_MARKSURFACES].filelen;
	sprintf(print_buf, "mark surfaces, %d bytes\n", force_load_size);
//	DEBUG_MSG(print_buf);
	temp_load = COM_LoadTempFile(mod->name);
	mod_base = (unsigned char *)temp_load - header->lumps[LUMP_MARKSURFACES].fileofs;

	Mod_LoadMarksurfaces (&header->lumps[LUMP_MARKSURFACES]);
	
//	dump_ram("after_ex10.bin", hunk_base, hunk_size);
///
	force_load_seek = header->lumps[LUMP_VISIBILITY].fileofs;
	force_load_size = header->lumps[LUMP_VISIBILITY].filelen;
	sprintf(print_buf, "visibility, %d bytes\n", force_load_size);
//	DEBUG_MSG(print_buf);
	temp_load = COM_LoadTempFile(mod->name);
	mod_base = (unsigned char *)temp_load - header->lumps[LUMP_VISIBILITY].fileofs;

	Mod_LoadVisibility (&header->lumps[LUMP_VISIBILITY]);
	
//	dump_ram("after_ex11.bin", hunk_base, hunk_size);
///
	force_load_seek = header->lumps[LUMP_LEAFS].fileofs;
	force_load_size = header->lumps[LUMP_LEAFS].filelen;
	sprintf(print_buf, "leaves, %d bytes\n", force_load_size);
//	DEBUG_MSG(print_buf);
	temp_load = COM_LoadTempFile(mod->name);
	mod_base = (unsigned char *)temp_load - header->lumps[LUMP_LEAFS].fileofs;

	Mod_LoadLeafs (&header->lumps[LUMP_LEAFS]);
	
//	dump_ram("after_ex12.bin", hunk_base, hunk_size);
///
	force_load_seek = header->lumps[LUMP_NODES].fileofs;
	force_load_size = header->lumps[LUMP_NODES].filelen;
	sprintf(print_buf, "nodes, %d bytes\n", force_load_size);
//	DEBUG_MSG(print_buf);
	temp_load = COM_LoadTempFile(mod->name);
	mod_base = (unsigned char *)temp_load - header->lumps[LUMP_NODES].fileofs;

	Mod_LoadNodes (&header->lumps[LUMP_NODES]);
	
//	dump_ram("after_ex13.bin", hunk_base, hunk_size);
///
	force_load_seek = header->lumps[LUMP_CLIPNODES].fileofs;
	force_load_size = header->lumps[LUMP_CLIPNODES].filelen;
	sprintf(print_buf, "clip nodes, %d bytes\n", force_load_size);
//	DEBUG_MSG(print_buf);
	temp_load = COM_LoadTempFile(mod->name);
	mod_base = (unsigned char *)temp_load - header->lumps[LUMP_CLIPNODES].fileofs;

	Mod_LoadClipnodes (&header->lumps[LUMP_CLIPNODES]);
	
//	dump_ram("after_ex14.bin", hunk_base, hunk_size);
///
	force_load_seek = header->lumps[LUMP_ENTITIES].fileofs;
	force_load_size = header->lumps[LUMP_ENTITIES].filelen;
	sprintf(print_buf, "entities, %d bytes\n", force_load_size);
//	DEBUG_MSG(print_buf);
	temp_load = COM_LoadTempFile(mod->name);
	mod_base = (unsigned char *)temp_load - header->lumps[LUMP_ENTITIES].fileofs;

	Mod_LoadEntities (&header->lumps[LUMP_ENTITIES]);
	
//	dump_ram("after_ex15.bin", hunk_base, hunk_size);
///
	force_load_seek = header->lumps[LUMP_MODELS].fileofs;
	force_load_size = header->lumps[LUMP_MODELS].filelen;
	sprintf(print_buf, "sub models, %d bytes\n", force_load_size);
//	DEBUG_MSG(print_buf);
	temp_load = COM_LoadTempFile(mod->name);
	mod_base = (unsigned char *)temp_load - header->lumps[LUMP_MODELS].fileofs;

	Mod_LoadSubmodels (&header->lumps[LUMP_MODELS]);
	
//	dump_ram("after_ex16.bin", hunk_base, hunk_size);
	
	force_load_seek = -1;
	force_load_size = -1;
///

	Mod_MakeHull0 ();
	
//	dump_ram("after_ex17.bin", hunk_base, hunk_size);
	
	mod->numframes = 2;		// regular and alternate animation
	mod->flags = 0;
	
//
// set up the submodels (FIXME: this is confusing)
//
	for (i=0 ; i<mod->numsubmodels ; i++)
	{
		bm = &mod->submodels[i];

		mod->hulls[0].firstclipnode = bm->headnode[0];
		for (j=1 ; j<MAX_MAP_HULLS ; j++)
		{
			mod->hulls[j].firstclipnode = bm->headnode[j];
			mod->hulls[j].lastclipnode = mod->numclipnodes-1;
		}

		mod->firstmodelsurface = bm->firstface;
		mod->nummodelsurfaces = bm->numfaces;
		
		VectorCopy (bm->maxs, mod->maxs);
		VectorCopy (bm->mins, mod->mins);
		mod->radius = RadiusFromBounds (mod->mins, mod->maxs);
		
		mod->numleafs = bm->visleafs;

		if (i < mod->numsubmodels-1)
		{	// duplicate the basic information
			char	name[10];

			sprintf (name, "*%i", i+1);
			loadmodel = Mod_FindName (name);
//			*loadmodel = *mod;
			ds_memcpy(loadmodel, mod, sizeof(*mod));
			Q_strcpy (loadmodel->name, name);
			mod = loadmodel;
		}
	}
	
//	dump_ram("after_ex18.bin", hunk_base, hunk_size);
//	while(1);
	
	free(header);
}

/*
==============================================================================

ALIAS MODELS

==============================================================================
*/

/*
=================
Mod_LoadAliasFrame
=================
*/
void * Mod_LoadAliasFrame (void * pin, int *pframeindex, int numv,
	trivertx_t *pbboxmin, trivertx_t *pbboxmax, aliashdr_t *pheader, char *name)
{
	trivertx_t		*pframe, *pinframe;
	int				i, j;
	daliasframe_t	*pdaliasframe;

	pdaliasframe = (daliasframe_t *)pin;

	Q_strcpy (name, pdaliasframe->name);
	
//	Sys_Printf("load name: %s\n", name);

#ifdef USE_EXTRA_RAM
	for (i=0 ; i<3 ; i++)
	{
	// these are byte values, so we don't have to worry about
	// endianness
		byte_write(&pbboxmin->v[i], pdaliasframe->bboxmin.v[i]);
		byte_write(&pbboxmax->v[i], pdaliasframe->bboxmax.v[i]);
	}
#else
	for (i=0 ; i<3 ; i++)
	{
	// these are byte values, so we don't have to worry about
	// endianness
		pbboxmin->v[i] = pdaliasframe->bboxmin.v[i];
		pbboxmax->v[i] = pdaliasframe->bboxmax.v[i];
	}
#endif

	pinframe = (trivertx_t *)(pdaliasframe + 1);
	pframe = (trivertx_t *)Hunk_AllocName (numv * sizeof(*pframe), loadname);

	*pframeindex = (byte *)pframe - (byte *)pheader;
	
//	Sys_Printf("%d verts\n", numv);

	for (j=0 ; j<numv ; j++)
	{
		int		k;

#ifdef USE_EXTRA_RAM
	// these are all byte values, so no need to deal with endianness
		byte_write(&pframe[j].lightnormalindex, pinframe[j].lightnormalindex);

		for (k=0 ; k<3 ; k++)
		{
			byte_write(&pframe[j].v[k], pinframe[j].v[k]);
		}
#else
	// these are all byte values, so no need to deal with endianness
		pframe[j].lightnormalindex = pinframe[j].lightnormalindex;

		for (k=0 ; k<3 ; k++)
		{
			pframe[j].v[k] = pinframe[j].v[k];
		}
#endif
	}

	pinframe += numv;

	return (void *)pinframe;
}


/*
=================
Mod_LoadAliasGroup
=================
*/
void * Mod_LoadAliasGroup (void * pin, int *pframeindex, int numv,
	trivertx_t *pbboxmin, trivertx_t *pbboxmax, aliashdr_t *pheader, char *name)
{
	daliasgroup_t		*pingroup;
	maliasgroup_t		*paliasgroup;
	int					i, numframes;
	daliasinterval_t	*pin_intervals;
	float				*poutintervals;
	void				*ptemp;
	
	pingroup = (daliasgroup_t *)pin;

	numframes = LittleLong (pingroup->numframes);
	paliasgroup = (maliasgroup_t *)Hunk_AllocName (sizeof (maliasgroup_t) +
			(numframes - 1) * sizeof (paliasgroup->frames[0]), loadname);

	paliasgroup->numframes = numframes;

#ifdef USE_EXTRA_RAM
	for (i=0 ; i<3 ; i++)
	{
	// these are byte values, so we don't have to worry about endianness
		byte_write(&pbboxmin->v[i], pingroup->bboxmin.v[i]);
		byte_write(&pbboxmax->v[i], pingroup->bboxmax.v[i]);
	}
#else
	for (i=0 ; i<3 ; i++)
	{
	// these are byte values, so we don't have to worry about endianness
		pbboxmin->v[i] = pingroup->bboxmin.v[i];
		pbboxmax->v[i] = pingroup->bboxmax.v[i];
	}
#endif

	*pframeindex = (byte *)paliasgroup - (byte *)pheader;

	pin_intervals = (daliasinterval_t *)(pingroup + 1);

	poutintervals = (float *)Hunk_AllocName (numframes * sizeof (float), loadname);

	paliasgroup->intervals = (byte *)poutintervals - (byte *)pheader;

	for (i=0 ; i<numframes ; i++)
	{
		*poutintervals = LittleFloat (pin_intervals->interval);
		if (*poutintervals <= 0.0)
			Sys_Error ("Mod_LoadAliasGroup: interval<=0");

		poutintervals++;
		pin_intervals++;
	}

	ptemp = (void *)pin_intervals;

	for (i=0 ; i<numframes ; i++)
	{
		ptemp = Mod_LoadAliasFrame (ptemp,
									&paliasgroup->frames[i].frame,
									numv,
									&paliasgroup->frames[i].bboxmin,
									&paliasgroup->frames[i].bboxmax,
									pheader, name);
	}

	return ptemp;
}


/*
=================
Mod_LoadAliasSkin
=================
*/
void * Mod_LoadAliasSkin (void * pin, int *pskinindex, int skinsize,
	aliashdr_t *pheader)
{
	int		i;
	byte	*pskin, *pinskin;
	unsigned short	*pusskin;

//	char text_buf[50];
//	sprintf(text_buf, "%.2fkb alloc for skin textures\n", (float)skinsize * r_pixbytes / 1024);
//	DEBUG_MSG(text_buf);
//	printf(text_buf);
//	while(1);
	
//	pskin = (byte *)Hunk_AllocName (skinsize * r_pixbytes, loadname);

	pinskin = (byte *)pin;
	
//	*pskinindex = (byte *)pskin - (byte *)pheader;
//
//	if (r_pixbytes == 1)
//	{
//		Q_memcpy (pskin, pinskin, skinsize);
//	}
//	else if (r_pixbytes == 2)
//	{
//		pusskin = (unsigned short *)pskin;
//
//		for (i=0 ; i<skinsize ; i++)
//			pusskin[i] = d_8to16table[pinskin[i]];
//	}
//	else
//	{
//		Sys_Error ("Mod_LoadAliasSkin: driver set invalid r_pixbytes: %d\n",
//				 r_pixbytes);
//	}
	*pskinindex = NULL;

	pinskin += skinsize;
	return ((void *)pinskin);
}


/*
=================
Mod_LoadAliasSkinGroup
=================
*/
#ifndef LOWMEM_LOADER
void * Mod_LoadAliasSkinGroup (void * pin, int *pskinindex, int skinsize,
	aliashdr_t *pheader)
{
	daliasskingroup_t		*pinskingroup;
	maliasskingroup_t		*paliasskingroup;
	int						i, numskins;
	daliasskininterval_t	*pinskinintervals;
	float					*poutskinintervals;
	void					*ptemp;

	pinskingroup = (daliasskingroup_t *)pin;

	numskins = LittleLong (pinskingroup->numskins);
	printf("-->%d skins\n", numskins);

	printf("allocating %d bytes\n", sizeof (maliasskingroup_t) +
			(numskins - 1) * sizeof (paliasskingroup->skindescs[0]));
	paliasskingroup = (maliasskingroup_t *)Hunk_AllocName (sizeof (maliasskingroup_t) +
			(numskins - 1) * sizeof (paliasskingroup->skindescs[0]),
			loadname);

	paliasskingroup->numskins = numskins;

	printf("difference between pointers is %d\n", (unsigned int)paliasskingroup - (unsigned int)pheader);
	*pskinindex = (byte *)paliasskingroup - (byte *)pheader;

	pinskinintervals = (daliasskininterval_t *)(pinskingroup + 1);

	poutskinintervals = (float *)Hunk_AllocName (numskins * sizeof (float),loadname);

	paliasskingroup->intervals = (byte *)poutskinintervals - (byte *)pheader;
	printf("there are %d intervals\n", paliasskingroup->intervals);

	for (i=0 ; i<numskins ; i++)
	{
//		*poutskinintervals = LittleFloat (pinskinintervals->interval);
//		printf("interval %d is %.2f\n", *poutskinintervals);
//		if (*poutskinintervals <= 0)
//			Sys_Error ("Mod_LoadAliasSkinGroup: interval<=0");

		poutskinintervals++;
		pinskinintervals++;
	}

	ptemp = (void *)pinskinintervals;

	for (i=0 ; i<numskins ; i++)
	{
		ptemp = Mod_LoadAliasSkin (ptemp,
				&paliasskingroup->skindescs[i].skin, skinsize, pheader);
	}

	return ptemp;
}

#endif

void ds_resize_in_place(int sizeX, int sizeY, unsigned char *image);

/*
=================
Mod_LoadAliasModel
=================
*/
#ifdef LOWMEM_LOADER
void Mod_LoadAliasModel (model_t *mod, void *buffer)
{
	int					i;
	mdl_t				*pmodel, *pinmodel;
	stvert_t			*pstverts, *pinstverts;
	aliashdr_t			*pheader;
	mtriangle_t			*ptri;
	dtriangle_t			*pintriangles;
	int					version, numframes, numskins;
	int					size;
	daliasframetype_t	*pframetype;
	daliasskintype_t	*pskintype;
	maliasskindesc_t	*pskindesc;
	int					skinsize;
	int					start, end, total;
	
//	int pwn = 0;
//	if (strcmp(mod->name, "progs/armor.mdl") == 0)
//		pwn = 1;
//	printf("loading alias model %s\n", mod->name);
	
	start = Hunk_LowMark ();
	
	force_load_seek = 0;
	force_load_size = sizeof(mdl_t);	//to include the skin type

//	pinmodel = (mdl_t *)buffer;
	pinmodel = (mdl_t *)malloc(sizeof(mdl_t));
	
	if (pinmodel == NULL)
		Sys_Error("not enough memory to allocate model header!\n");
	
	void *temp_load = COM_LoadTempFile(mod->name);
	ds_memcpy(pinmodel, temp_load, sizeof(mdl_t));

	version = LittleLong (pinmodel->version);
	if (version != ALIAS_VERSION)
		Sys_Error ("%s has wrong version number (%i should be %i)",
				 mod->name, version, ALIAS_VERSION);

//
// allocate space for a working header, plus all the data except the frames,
// skin and group info
//
	size = 	sizeof (aliashdr_t) + (LittleLong (pinmodel->numframes) - 1) *
			 sizeof (pheader->frames[0]) +
			sizeof (mdl_t) +
			LittleLong (pinmodel->numverts) * sizeof (stvert_t) +
			LittleLong (pinmodel->numtris) * sizeof (mtriangle_t);

	pheader = (aliashdr_t *)Hunk_AllocName (size, loadname);
	pmodel = (mdl_t *) ((byte *)&pheader[1] +
			(LittleLong (pinmodel->numframes) - 1) *
			 sizeof (pheader->frames[0]));
	
//	mod->cache.data = pheader;
	mod->flags = LittleLong (pinmodel->flags);

//
// endian-adjust and copy the data, starting with the alias model header
//
	pmodel->boundingradius = LittleFloat (pinmodel->boundingradius);
	pmodel->numskins = LittleLong (pinmodel->numskins);
	pmodel->skinwidth = LittleLong (pinmodel->skinwidth);
	pmodel->skinheight = LittleLong (pinmodel->skinheight);

	if (pmodel->skinheight > MAX_LBM_HEIGHT)
		Sys_Error ("model %s has a skin taller than %d", mod->name,
				   MAX_LBM_HEIGHT);

	pmodel->numverts = LittleLong (pinmodel->numverts);

	if (pmodel->numverts <= 0)
		Sys_Error ("model %s has no vertices", mod->name);

	if (pmodel->numverts > MAXALIASVERTS)
		Sys_Error ("model %s has too many vertices", mod->name);

	pmodel->numtris = LittleLong (pinmodel->numtris);

	if (pmodel->numtris <= 0)
		Sys_Error ("model %s has no triangles", mod->name);

	pmodel->numframes = LittleLong (pinmodel->numframes);
	pmodel->size = LittleFloat (pinmodel->size) * ALIAS_BASE_SIZE_RATIO;
#ifdef USE_EXTRA_RAM
	byte_write(&mod->synctype, (synctype_t)LittleLong (pinmodel->synctype));
#else
	mod->synctype = (synctype_t)LittleLong (pinmodel->synctype);
#endif
	mod->numframes = pmodel->numframes;

	for (i=0 ; i<3 ; i++)
	{
		pmodel->scale[i] = LittleFloat (pinmodel->scale[i]);
		pmodel->scale_origin[i] = LittleFloat (pinmodel->scale_origin[i]);
		pmodel->eyeposition[i] = LittleFloat (pinmodel->eyeposition[i]);
	}

	numskins = pmodel->numskins;
	numframes = pmodel->numframes;
	//Sys_Printf("%d frames\n", numframes);

	if (pmodel->skinwidth & 0x03)
		Sys_Error ("Mod_LoadAliasModel: skinwidth not multiple of 4");

	pheader->model = (byte *)pmodel - (byte *)pheader;

//
// load the skins
//
	skinsize = pmodel->skinheight * pmodel->skinwidth;
//	printf("skin size %dx%d\n", pmodel->skinwidth, pmodel->skinheight);

	if (numskins < 1)
		Sys_Error ("Mod_LoadAliasModel: Invalid # of skins: %d\n", numskins);

	pskintype = (daliasskintype_t *)&pinmodel[1];

	pskindesc = (maliasskindesc_t *)Hunk_AllocName (numskins * sizeof (maliasskindesc_t),
								loadname);

	pheader->skindesc = (byte *)pskindesc - (byte *)pheader;

	int seek_change = 0;
	
//	printf("parsing %d skins\n", numskins);
	
	for (i=0 ; i<numskins ; i++)
	{
//		printf("skin %d...", i);
		aliasskintype_t	skintype;
		
		force_load_seek += force_load_size + seek_change;
		force_load_size = sizeof(daliasskintype_t) * 2;
		
		temp_load = COM_LoadTempFile(mod->name);
		force_load_size -= sizeof(daliasskintype_t);
		
		skintype = (aliasskintype_t)LittleLong(((daliasskintype_t *)temp_load)->type);

//		skintype = (aliasskintype_t)LittleLong (pskintype->type);
		pskindesc[i].type = skintype;

//		Sys_Printf("skin loads, type %d\n", skintype);
		if (skintype == ALIAS_SKIN_SINGLE)
		{
//			printf("single\n");

			unsigned char *tex_base = (unsigned char *)(force_load_seek + 4);
			
			seek_change = (unsigned int)pskintype;
			pskintype = (daliasskintype_t *)
					Mod_LoadAliasSkin (pskintype + 1,
									   &pskindesc[i].skin,
									   skinsize, pheader);
			seek_change = (unsigned int)pskintype - seek_change - 4;

			int texture_handle;
			
			if (strcmp(mod->name, "progs/v_shot.mdl") == 0)
			{
//				printf("making shotgun transparent\n");
				texture_handle = register_texture(false, mod->name, get_last_found_handle(),
					(unsigned int)tex_base - (unsigned int)buffer + get_last_found_seekpos(),
					pmodel->skinwidth, pmodel->skinheight / 2, 1, 208, 0, 1);
			}
			else
				texture_handle = register_texture(false, mod->name, get_last_found_handle(),
					(unsigned int)tex_base - (unsigned int)buffer + get_last_found_seekpos(),
					pmodel->skinwidth, pmodel->skinheight / 2, 0, 0, 0, 1);

			if (texture_handle == -1)			
			{
				texture_handle = r_notexture_mip->texture_handle;
			}
			
			pskindesc[i].pcachespot = (void *)texture_handle;
		}
		else
		{
			int groupskins = LittleLong(((daliasskingroup_t *)temp_load)[1].numskins);
			seek_change = sizeof(daliasskintype_t) + sizeof(daliasskingroup_t)
					+ (skinsize + sizeof(daliasskininterval_t)) * groupskins - 4;
			
			pskintype = (daliasskintype_t*)((unsigned int)pskintype + seek_change + 4);
			
			int texture_handle = r_notexture_mip->texture_handle;
			pskindesc[i].pcachespot = (void *)texture_handle;
		}
	}
	
//	printf("done skin loads\n");

//
// set base s and t vertices
//
	pstverts = (stvert_t *)&pmodel[1];
//	pinstverts = (stvert_t *)pskintype;

//	printf("data's from %d\n", (unsigned int)pskintype - (unsigned int)buffer);
//	printf("seeking to %d\n", (unsigned int)pskintype - (unsigned int)&pinmodel[0]);

	pheader->stverts = (byte *)pstverts - (byte *)pheader;
	
//	force_load_seek = (unsigned int)pstverts - (unsigned int)pheader;
	force_load_seek = (unsigned int)pskintype - (unsigned int)&pinmodel[0];
	force_load_size = sizeof(stvert_t) * pmodel->numverts;

	temp_load = COM_LoadTempFile(mod->name);
	
	pinstverts = (stvert_t *)temp_load;
	
//	Sys_Printf("pheader->stverts %d\n", pheader->stverts);
	for (i=0 ; i<pmodel->numverts ; i++)
	{
		/*Sys_Printf("%d os %d s %d t %d\n",
			LittleLong (pinstverts[i].onseam),
			LittleLong (pinstverts[i].s) << 16,
			LittleLong (pinstverts[i].t) << 16);*/
			
		pstverts[i].onseam = LittleLong (pinstverts[i].onseam);
	// put s and t in 16.16 format
		pstverts[i].s = LittleLong (pinstverts[i].s);
		pstverts[i].t = LittleLong (pinstverts[i].t);
	}
	
//	printf("done verts\n");

//	printf("vert %d %d, %d\n", pstverts[0].s >> 16, pstverts[0].t >> 16, pstverts[0].onseam);

//
// set up the triangles
//
	ptri = (mtriangle_t *)&pstverts[pmodel->numverts];
//	pintriangles = (dtriangle_t *)&pinstverts[pmodel->numverts];

	pheader->triangles = (byte *)ptri - (byte *)pheader;
	
//	force_load_seek = (unsigned int)ptri - (unsigned int)pheader;
	force_load_seek += sizeof(stvert_t) * pmodel->numverts;
	force_load_size = sizeof(dtriangle_t) * pmodel->numtris;

	temp_load = COM_LoadTempFile(mod->name);
	
	pintriangles = (dtriangle_t *)temp_load;

	for (i=0 ; i<pmodel->numtris ; i++)
	{
		int		j;

		ptri[i].facesfront = LittleLong (pintriangles[i].facesfront);

		for (j=0 ; j<3 ; j++)
		{
			ptri[i].vertindex[j] =
					LittleLong (pintriangles[i].vertindex[j]);
		}
	}
	
//	printf("done %d triangles\n", pmodel->numtris);

//	printf("tri %d %d %d\n", ptri[0].vertindex[0], ptri[0].vertindex[1], ptri[0].vertindex[2]);
	
//
// load the frames
//
	if (numframes < 1)
		Sys_Error ("Mod_LoadAliasModel: Invalid # of frames: %d\n", numframes);

//	pframetype = (daliasframetype_t *)&pintriangles[pmodel->numtris];
	
	force_load_seek += pmodel->numtris * sizeof(dtriangle_t);
	force_load_size = -2;
	
//	printf("seek is %d\n", force_load_seek);

	temp_load = COM_LoadTempFile(mod->name);
	
	pframetype = (daliasframetype_t *)temp_load;

	for (i=0 ; i<numframes ; i++)
	{
		aliasframetype_t	frametype;

		frametype = (aliasframetype_t)LittleLong (pframetype->type);
		pheader->frames[i].type = frametype;
		
//		Sys_Printf("frame %d, type %d\n", i, frametype);

		if (frametype == ALIAS_SINGLE)
		{
			pframetype = (daliasframetype_t *)
					Mod_LoadAliasFrame (pframetype + 1,
										&pheader->frames[i].frame,
										pmodel->numverts,
										&pheader->frames[i].bboxmin,
										&pheader->frames[i].bboxmax,
										pheader, pheader->frames[i].name);
		}
		else
		{
			pframetype = (daliasframetype_t *)
					Mod_LoadAliasGroup (pframetype + 1,
										&pheader->frames[i].frame,
										pmodel->numverts,
										&pheader->frames[i].bboxmin,
										&pheader->frames[i].bboxmax,
										pheader, pheader->frames[i].name);
		}
	}
	
//	printf("done model loading\n");

#ifdef USE_EXTRA_RAM
	byte_write(&mod->type, mod_alias);
#else
	mod->type = mod_alias;
#endif

// FIXME: do this right
	mod->mins[0] = mod->mins[1] = mod->mins[2] = -16;
	mod->maxs[0] = mod->maxs[1] = mod->maxs[2] = 16;

//
// move the complete, relocatable alias model to the cache
//	
	end = Hunk_LowMark ();
	total = end - start;
	
	free(pinmodel);
	force_load_seek = -1;
	force_load_size = -1;
	
//	cache_uses_ex = 1;
	Cache_Alloc (&mod->cache, total, loadname);
//	cache_uses_ex = 0;
	if (!mod->cache.data)
		return;
	ds_memcpy (mod->cache.data, pheader, total);

	Hunk_FreeToLowMark (start);
	
//	printf("out\n");
}

#else

void Mod_LoadAliasModel (model_t *mod, void *buffer)
{
	int					i;
	mdl_t				*pmodel, *pinmodel;
	stvert_t			*pstverts, *pinstverts;
	aliashdr_t			*pheader;
	mtriangle_t			*ptri;
	dtriangle_t			*pintriangles;
	int					version, numframes, numskins;
	int					size;
	daliasframetype_t	*pframetype;
	daliasskintype_t	*pskintype;
	maliasskindesc_t	*pskindesc;
	int					skinsize;
	int					start, end, total;
	
	start = Hunk_LowMark ();

	pinmodel = (mdl_t *)buffer;

	version = LittleLong (pinmodel->version);
	if (version != ALIAS_VERSION)
		Sys_Error ("%s has wrong version number (%i should be %i)",
				 mod->name, version, ALIAS_VERSION);

//
// allocate space for a working header, plus all the data except the frames,
// skin and group info
//
	size = 	sizeof (aliashdr_t) + (LittleLong (pinmodel->numframes) - 1) *
			 sizeof (pheader->frames[0]) +
			sizeof (mdl_t) +
			LittleLong (pinmodel->numverts) * sizeof (stvert_t) +
			LittleLong (pinmodel->numtris) * sizeof (mtriangle_t);

	pheader = (aliashdr_t *)Hunk_AllocName (size, loadname);
	pmodel = (mdl_t *) ((byte *)&pheader[1] +
			(LittleLong (pinmodel->numframes) - 1) *
			 sizeof (pheader->frames[0]));
	
//	mod->cache.data = pheader;
	mod->flags = LittleLong (pinmodel->flags);

//
// endian-adjust and copy the data, starting with the alias model header
//
	pmodel->boundingradius = LittleFloat (pinmodel->boundingradius);
	pmodel->numskins = LittleLong (pinmodel->numskins);
	pmodel->skinwidth = LittleLong (pinmodel->skinwidth);
	pmodel->skinheight = LittleLong (pinmodel->skinheight);

	if (pmodel->skinheight > MAX_LBM_HEIGHT)
		Sys_Error ("model %s has a skin taller than %d", mod->name,
				   MAX_LBM_HEIGHT);

	pmodel->numverts = LittleLong (pinmodel->numverts);

	if (pmodel->numverts <= 0)
		Sys_Error ("model %s has no vertices", mod->name);

	if (pmodel->numverts > MAXALIASVERTS)
		Sys_Error ("model %s has too many vertices", mod->name);

	pmodel->numtris = LittleLong (pinmodel->numtris);

	if (pmodel->numtris <= 0)
		Sys_Error ("model %s has no triangles", mod->name);

	pmodel->numframes = LittleLong (pinmodel->numframes);
	pmodel->size = LittleFloat (pinmodel->size) * ALIAS_BASE_SIZE_RATIO;
	mod->synctype = (synctype_t)LittleLong (pinmodel->synctype);
	mod->numframes = pmodel->numframes;

	for (i=0 ; i<3 ; i++)
	{
		pmodel->scale[i] = LittleFloat (pinmodel->scale[i]);
		pmodel->scale_origin[i] = LittleFloat (pinmodel->scale_origin[i]);
		pmodel->eyeposition[i] = LittleFloat (pinmodel->eyeposition[i]);
	}

	numskins = pmodel->numskins;
	numframes = pmodel->numframes;
	//Sys_Printf("%d frames\n", numframes);

	if (pmodel->skinwidth & 0x03)
		Sys_Error ("Mod_LoadAliasModel: skinwidth not multiple of 4");

	pheader->model = (byte *)pmodel - (byte *)pheader;

//
// load the skins
//
	skinsize = pmodel->skinheight * pmodel->skinwidth;
//	printf("skin size %dx%d\n", pmodel->skinwidth, pmodel->skinheight);

	if (numskins < 1)
		Sys_Error ("Mod_LoadAliasModel: Invalid # of skins: %d\n", numskins);

	pskintype = (daliasskintype_t *)&pinmodel[1];

	pskindesc = (maliasskindesc_t *)Hunk_AllocName (numskins * sizeof (maliasskindesc_t),
								loadname);

	pheader->skindesc = (byte *)pskindesc - (byte *)pheader;

	for (i=0 ; i<numskins ; i++)
	{
		aliasskintype_t	skintype;

		skintype = (aliasskintype_t)LittleLong (pskintype->type);
		pskindesc[i].type = skintype;

		//Sys_Printf("skin loads, type %d\n", skintype);
		if (skintype == ALIAS_SKIN_SINGLE)
		{
			unsigned char *tex_base = (unsigned char *)(pskintype + 1);
			
			pskintype = (daliasskintype_t *)
					Mod_LoadAliasSkin (pskintype + 1,
									   &pskindesc[i].skin,
									   skinsize, pheader);

			int texture_handle;
			
//			printf("%d is the word\n",
//				(unsigned int)tex_base - 
//				(unsigned int)buffer);
//			while(1);

			texture_handle = register_texture(false, mod->name, get_last_found_handle(),
				(unsigned int)tex_base - (unsigned int)buffer + get_last_found_seekpos(),
				pmodel->skinwidth, pmodel->skinheight / 2, 0, 0, 0, 1);

			if (texture_handle == -1)			
			{
				texture_handle = r_notexture_mip->texture_handle;
			}

skip_texture:
			
			pskindesc[i].pcachespot = (void *)texture_handle;
		}
		else
		{
			int texture_handle;
			
			extern int fail_textures;
			
			fail_textures = 0;
			
			unsigned int seek_change = (unsigned int)pskintype;
			pskintype = (daliasskintype_t *)
					Mod_LoadAliasSkinGroup (pskintype + 1,
											&pskindesc[i].skin,
											skinsize, pheader);
			seek_change = (unsigned int)pskintype - seek_change;
//			printf("seek change is %d\n", seek_change);
			
//			if (ds_teximage2d(pmodel->skinwidth, pmodel->skinheight, (unsigned char *)(pskintype + 1), 0, 0) == -1)
			{
				texture_handle = r_notexture_mip->texture_handle;
//				printf("couldn\'t load alias model texture\n");
//				while(1);
			}
			
			fail_textures = 1;
			
//			pskindesc[i].pcachespot = (void *)texture_handle;
		}
	}

//
// set base s and t vertices
//
	pstverts = (stvert_t *)&pmodel[1];
	pinstverts = (stvert_t *)pskintype;
	
//	printf("data's from %d\n", (unsigned int)pskintype - (unsigned int)buffer);

	pheader->stverts = (byte *)pstverts - (byte *)pheader;
	
//	Sys_Printf("pheader->stverts %d\n", pheader->stverts);
	for (i=0 ; i<pmodel->numverts ; i++)
	{
		/*Sys_Printf("%d os %d s %d t %d\n",
			LittleLong (pinstverts[i].onseam),
			LittleLong (pinstverts[i].s) << 16,
			LittleLong (pinstverts[i].t) << 16);*/
			
		pstverts[i].onseam = LittleLong (pinstverts[i].onseam);
	// put s and t in 16.16 format
		pstverts[i].s = LittleLong (pinstverts[i].s);
		pstverts[i].t = LittleLong (pinstverts[i].t);
	}
	
//	printf("vert %d %d, %d\n", pstverts[0].s >> 16, pstverts[0].t >> 16, pstverts[0].onseam);

//
// set up the triangles
//
	ptri = (mtriangle_t *)&pstverts[pmodel->numverts];
	pintriangles = (dtriangle_t *)&pinstverts[pmodel->numverts];

	pheader->triangles = (byte *)ptri - (byte *)pheader;

	for (i=0 ; i<pmodel->numtris ; i++)
	{
		int		j;

		ptri[i].facesfront = LittleLong (pintriangles[i].facesfront);

		for (j=0 ; j<3 ; j++)
		{
			ptri[i].vertindex[j] =
					LittleLong (pintriangles[i].vertindex[j]);
		}
	}
	
//	printf("tri %d %d %d\n", ptri[0].vertindex[0], ptri[0].vertindex[1], ptri[0].vertindex[2]);

//
// load the frames
//
	if (numframes < 1)
		Sys_Error ("Mod_LoadAliasModel: Invalid # of frames: %d\n", numframes);

	pframetype = (daliasframetype_t *)&pintriangles[pmodel->numtris];

	for (i=0 ; i<numframes ; i++)
	{
		aliasframetype_t	frametype;

		frametype = (aliasframetype_t)LittleLong (pframetype->type);
		pheader->frames[i].type = frametype;
		
//		Sys_Printf("frame %d, type %d\n", i, frametype);

		if (frametype == ALIAS_SINGLE)
		{
			pframetype = (daliasframetype_t *)
					Mod_LoadAliasFrame (pframetype + 1,
										&pheader->frames[i].frame,
										pmodel->numverts,
										&pheader->frames[i].bboxmin,
										&pheader->frames[i].bboxmax,
										pheader, pheader->frames[i].name);
		}
		else
		{
			pframetype = (daliasframetype_t *)
					Mod_LoadAliasGroup (pframetype + 1,
										&pheader->frames[i].frame,
										pmodel->numverts,
										&pheader->frames[i].bboxmin,
										&pheader->frames[i].bboxmax,
										pheader, pheader->frames[i].name);
		}
	}

	mod->type = mod_alias;

// FIXME: do this right
	mod->mins[0] = mod->mins[1] = mod->mins[2] = -16;
	mod->maxs[0] = mod->maxs[1] = mod->maxs[2] = 16;

//
// move the complete, relocatable alias model to the cache
//	
	end = Hunk_LowMark ();
	total = end - start;
	
	Cache_Alloc (&mod->cache, total, loadname);
	if (!mod->cache.data)
		return;
	ds_memcpy (mod->cache.data, pheader, total);

	Hunk_FreeToLowMark (start);
}

#endif

//=============================================================================

/*
=================
Mod_LoadSpriteFrame
=================
*/
void * Mod_LoadSpriteFrame (void * pin, mspriteframe_t **ppframe, unsigned int base)
{
	dspriteframe_t		*pinframe;
	mspriteframe_t		*pspriteframe;
	int					i, width, height, size, origin[2];
	unsigned short		*ppixout;
	byte				*ppixin;

	pinframe = (dspriteframe_t *)pin;

	width = LittleLong (pinframe->width);
	height = LittleLong (pinframe->height);
//	printf("sprite %dx%d\n", width, height);
	size = width * height;

#ifndef USE_3D
	pspriteframe = (mspriteframe_t *)Hunk_AllocName (sizeof (mspriteframe_t) + size*r_pixbytes,
								   loadname);

	Q_memset (pspriteframe, 0, sizeof (mspriteframe_t) + size);
#else
	pspriteframe = (mspriteframe_t *)Hunk_AllocName (sizeof (mspriteframe_t),
								   loadname);

	Q_memset (pspriteframe, 0, sizeof (mspriteframe_t));
#endif
	*ppframe = pspriteframe;

	pspriteframe->width = width;
	pspriteframe->height = height;
	origin[0] = LittleLong (pinframe->origin[0]);
	origin[1] = LittleLong (pinframe->origin[1]);

	pspriteframe->up = origin[1];
	pspriteframe->down = origin[1] - height;
	pspriteframe->left = origin[0];
	pspriteframe->right = width + origin[0];

	if (r_pixbytes == 1)
	{
#ifdef USE_3D
//		printf("loading sprite %s, %dx%d\n", loadmodel->name, width, height);
//		printf("data is %d bytes in\n", (unsigned int)(pinframe + 1) - base);
//		while(1);
		
//		pspriteframe->texture_handle = ds_gentexture();
//		ds_bindtexture(pspriteframe->texture_handle);
//		
		pspriteframe->texture_handle = register_texture(false, loadmodel->name,
			get_last_found_handle(), get_last_found_seekpos() + ((unsigned int)(pinframe + 1) - base),
			pspriteframe->width, pspriteframe->height,
			true, 255, 0, 0);
		
//		printf("handle is %d\n", pspriteframe->texture_handle);
		
		if (pspriteframe->texture_handle == -1)
		{
			printf("sprite texture loading for sprite %s failed, size %dx%d\n", loadmodel->name, pspriteframe->width, pspriteframe->height);
			*(int *)0 = 0;
			while(1);
		}
//		
//		if (ds_teximage2d(width, height, (byte *)(pinframe + 1), false, 255) == -1)
//		{
//			pspriteframe->texture_handle = r_notexture_mip->texture_handle;
//			pspriteframe->texture_handle = -1;
//			printf("failed\n");
//		}
#else
		Q_memcpy (&pspriteframe->pixels[0], (byte *)(pinframe + 1), size);
#endif
	}
//	else if (r_pixbytes == 2)
//	{
//		ppixin = (byte *)(pinframe + 1);
//		ppixout = (unsigned short *)&pspriteframe->pixels[0];
//
//		for (i=0 ; i<size ; i++)
//			ppixout[i] = d_8to16table[ppixin[i]];
//	}
//	else
//	{
//		Sys_Error ("Mod_LoadSpriteFrame: driver set invalid r_pixbytes: %d\n",
//				 r_pixbytes);
//	}

	return (void *)((byte *)pinframe + sizeof (dspriteframe_t) + size);
}


/*
=================
Mod_LoadSpriteGroup
=================
*/
void * Mod_LoadSpriteGroup (void * pin, mspriteframe_t **ppframe, unsigned int base)
{
	dspritegroup_t		*pingroup;
	mspritegroup_t		*pspritegroup;
	int					i, numframes;
	dspriteinterval_t	*pin_intervals;
	float				*poutintervals;
	void				*ptemp;

	pingroup = (dspritegroup_t *)pin;

	numframes = LittleLong (pingroup->numframes);

	pspritegroup = (mspritegroup_t *)Hunk_AllocName (sizeof (mspritegroup_t) +
				(numframes - 1) * sizeof (pspritegroup->frames[0]), loadname);

	pspritegroup->numframes = numframes;

	*ppframe = (mspriteframe_t *)pspritegroup;

	pin_intervals = (dspriteinterval_t *)(pingroup + 1);

	poutintervals = (float *)Hunk_AllocName (numframes * sizeof (float), loadname);

	pspritegroup->intervals = poutintervals;

	for (i=0 ; i<numframes ; i++)
	{
//		*poutintervals = LittleFloat (pin_intervals->interval);
//		if (*poutintervals <= 0.0)
//			Sys_Error ("Mod_LoadSpriteGroup: interval<=0");

		poutintervals++;
		pin_intervals++;
	}

	ptemp = (void *)pin_intervals;

	for (i=0 ; i<numframes ; i++)
	{
		ptemp = Mod_LoadSpriteFrame (ptemp, &pspritegroup->frames[i], base);
	}

	return ptemp;
}


/*
=================
Mod_LoadSpriteModel
=================
*/
void Mod_LoadSpriteModel (model_t *mod, void *buffer)
{
	int					i;
	int					version;
	dsprite_t			*pin;
	msprite_t			*psprite;
	int					numframes;
	int					size;
	dspriteframetype_t	*pframetype;
	
//	printf("sprite load %s\n", mod->name);
	
	pin = (dsprite_t *)buffer;

	version = LittleLong (pin->version);
	if (version != SPRITE_VERSION)
		Sys_Error ("%s has wrong version number "
				 "(%i should be %i)", mod->name, version, SPRITE_VERSION);

	numframes = LittleLong (pin->numframes);

	size = sizeof (msprite_t) +	(numframes - 1) * sizeof (psprite->frames);

	psprite = (msprite_t *)Hunk_AllocName (size, loadname);

	mod->cache.data = psprite;

	psprite->type = LittleLong (pin->type);
	psprite->maxwidth = LittleLong (pin->width);
	psprite->maxheight = LittleLong (pin->height);
	psprite->beamlength = LittleFloat (pin->beamlength);
#ifdef USE_EXTRA_RAM
	byte_write(&mod->synctype, (synctype_t)LittleLong (pin->synctype));
#else
	mod->synctype = (synctype_t)LittleLong (pin->synctype);
#endif
	psprite->numframes = numframes;
	
//	printf("max size %dx%d\n", psprite->maxwidth, psprite->maxheight);

	mod->mins[0] = mod->mins[1] = -psprite->maxwidth/2;
	mod->maxs[0] = mod->maxs[1] = psprite->maxwidth/2;
	mod->mins[2] = -psprite->maxheight/2;
	mod->maxs[2] = psprite->maxheight/2;
	
//
// load the frames
//
	if (numframes < 1)
		Sys_Error ("Mod_LoadSpriteModel: Invalid # of frames: %d\n", numframes);

	mod->numframes = numframes;
	mod->flags = 0;

	pframetype = (dspriteframetype_t *)(pin + 1);
//	printf("size %d\n", sizeof(dsprite_t));
	
//	unsigned char *b = (unsigned char *)buffer;
//	b += 0x1b90;
//	for (int count = 0; count < 4; count++)
//		printf("%02x%02x%02x%02x ", b[count * 4 + 0], b[count * 4 + 1], b[count * 4 + 2], b[count * 4 + 3]);
//	
//	printf("\n%08x\n", *(unsigned int *)(&b[7]));
	
	for (i=0 ; i<numframes ; i++)
	{
		spriteframetype_t	frametype;

//		frametype = (spriteframetype_t)(LittleLong (pframetype->type));
		frametype = (spriteframetype_t)(*(unsigned char *)pframetype);
		psprite->frames[i].type = frametype;
		
//		printf("frame %d of %d, type %d\n", i, numframes, frametype);

		if (frametype == SPR_SINGLE)
		{
//			printf("single\n");
			pframetype = (dspriteframetype_t *)
					Mod_LoadSpriteFrame (pframetype + 1,
										 &psprite->frames[i].frameptr,
										 (unsigned int)buffer);
		}
		else
		{
//			printf("group\n");
			pframetype = (dspriteframetype_t *)
					Mod_LoadSpriteGroup (pframetype + 1,
										 &psprite->frames[i].frameptr,
										 (unsigned int)buffer);
		}
		
//		printf("difference %d\n", (unsigned int)pframetype - (unsigned int)buffer);
	}
#ifdef USE_EXTRA_RAM
	byte_write(&mod->type, mod_sprite);
#else
	mod->type = mod_sprite;
#endif
}

//=============================================================================

/*
================
Mod_Print
================
*/
void Mod_Print (void)
{
	int		i;
	model_t	*mod;

	Con_Printf ("Cached models:\n");
	for (i=0, mod=mod_known ; i < mod_numknown ; i++, mod++)
	{
		Con_Printf ("%8p : %s",mod->cache.data, mod->name);
		if (mod->needload & NL_UNREFERENCED)
			Con_Printf (" (!R)");
		if (mod->needload & NL_NEEDS_LOADED)
			Con_Printf (" (!P)");
		Con_Printf ("\n");
	}
}


