/*
Copyright (C) 1996-1997 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 3
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/
// gl_draw_textures.c -- drawing functions involving textures

#include "quakedef.h"

typedef struct
{
	int			texnum;
	char		identifier[MAX_QPATH];
	char		*pathname_explicit;
	int			width, height;
	int			scaled_width, scaled_height;
	int			texmode;
	unsigned 	crc;
	int			bpp;
} gltexture_t;

gltexture_t	gltextures[MAX_GLTEXTURES];
int		numgltextures;


/* FitzQuake version

  enum srcformat {SRC_INDEXED, SRC_LIGHTMAP, SRC_RGBA};
// Indexed = scrap and wad pics
typedef struct gltexture_s {
//managed by texture manager
	unsigned int		texnum;
	struct gltexture_s	*next;
	model_t				*owner;
//managed by image loading
	char				name[64];
	unsigned int		width; //size of image as it exists in opengl
	unsigned int		height; //size of image as it exists in opengl
	unsigned int		flags;
	char				source_file[MAX_QPATH]; //relative filepath to data source, or "" if source is in memory
	unsigned int		source_offset; //byte offset into file, or memory address
	enum srcformat		source_format; //format of pixel data (indexed, lightmap, or rgba)
	unsigned int		source_width; //size of image in source data
	unsigned int		source_height; //size of image in source data
	unsigned short		source_crc; //generated by source data before modifications
	char				shirt; //0-13 shirt color, or -1 if never colormapped
	char				pants; //0-13 pants color, or -1 if never colormapped
//used for rendering
	int					visframe; //matches r_framecount if texture was bound this frame
} gltexture_t;
*/


int	texture_extension_number = 1;

int		gl_max_size_default;

// internal format - The number of color components in the texture
int		gl_solid_bytes_per_pixel = 3;
int		gl_alpha_bytes_per_pixel = 4;

// format - The format of the pixel data (like GL_RGB, GL_RGBA, GL_LUMINANCE)
const int	gl_lightmap_format = GL_RGB;


int		gl_filter_min = GL_LINEAR_MIPMAP_NEAREST;
int		gl_filter_max = GL_LINEAR;




// mode (UPLOAD 32)
// Classic particles	TEX_MIPMAP | TEX_ALPHA_TEST
// Skybox				TEX_NONE (Is reason we shouldn't picmip, this is a world texture ... what about mipmap?)

// mode (UPLOAD8)
// Player skin trans	picmip_flag = tex_picmip_allmodels.value ? TEX_MIPMAP : 0;  Only flag
// Fullbright skin		picmip_flag | TEX_FULLBRIGHT
// Scrap				TEX_ALPHA_TEST

// mode (GL_LOADTEXTURE)
// Charset				TEX_ALPHA_TEST (may include preprocessing)
//

// mode (DIRECT:  eglTexImage2D ... upload rogue)
// Sky

/*
===============
GL_Upload32
===============
*/



// Skybox
void GL_Upload32 (unsigned *data, int width, int height, int mode)
{
	int				internal_format;
	int				miplevel;
	unsigned int	*scaled;

// STAGE 1: Make power of 2 size; copy data into malloc'd structure
	// Multiply times 2 until size is multiple of 2 that can contain the width/height
	int scaled_width  = Find_Power_Of_Two_Size(width);
	int scaled_height =  Find_Power_Of_Two_Size(height);

	scaled = ImageWork_malloc (scaled_width * scaled_height * 4, "temp scaled image");

	// If texture adjusted to power of 2 size isn't exactly width and size, resample it.
	if (width < scaled_width || height < scaled_height)
	{
		// Only scaled data is altered by following function (rest of args are const)
		ResampleTexture (data, width, height, scaled, scaled_width, scaled_height, !!tex_lerp.integer);
		width = scaled_width;
		height = scaled_height;
	}
	else
	{
		memcpy (scaled, data, width * height * 4);
	}

// STAGE 2: This is NOT world mip mapping.  Scale down based on gl_picmip value if mip mapped texture
	ScaleDimensions (width, height, &scaled_width, &scaled_height, mode);

	while (width > scaled_width || height > scaled_height)	//  Occurs regardless of mipmap flag, we are just adjusting the texture taking glmaxsize and gl_picmip into consideration
		MipMap ((byte *)scaled, &width, &height);	// Function does not use any cvars or texture flags

// STAGE 3: Determine internal format

	internal_format = (mode & TEX_ALPHA_TEST) ? gl_alpha_bytes_per_pixel : gl_solid_bytes_per_pixel;

// STAGE 4: Mip map if flagged

	miplevel = 0;
	eglTexImage2D (GL_TEXTURE_2D, 0, internal_format, width, height, 0, GL_RGBA, GL_UNSIGNED_BYTE, scaled); // UPLOAD
	if (mode & TEX_MIPMAP)
	{
		while ((width > 1 || height > 1) /* && miplevel < 4 *//* Baker: Is this right? */)
		{
			MipMap ((byte *)scaled, &width, &height);
			miplevel++;
			eglTexImage2D (GL_TEXTURE_2D, miplevel, internal_format, width, height, 0, GL_RGBA, GL_UNSIGNED_BYTE, scaled); // UPLOAD
		}

		MeglTexParameterf (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, gl_filter_min);
		MeglTexParameterf (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, gl_filter_max);
	}
	else
	{
		MeglTexParameterf (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, gl_filter_max);
		MeglTexParameterf (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, gl_filter_max);
	}

	ImageWork_free (scaled);
}

/*
===============
GL_Upload8
===============
*/
void GL_Upload8 (byte *data, int width, int height, int mode)
{
	int		size; // i,  p;
 // Baker: why did we increase this?  I can't remember now.  :(
	static unsigned	trans[1280*960]; // Was 640x480

	size = width * height;

	if (size * 4 > sizeof(trans))
		Sys_Error ("GL_Upload8: image too big");

	if (mode & TEX_FULLBRIGHT)
	{
	// this is a fullbright mask, so make all non-fullbright colors transparent
		mode |= TEX_ALPHA_TEST;
		Check_QPAL_Copy_To_RGBA_Make_NonFullBright_Pixels_Transparent (data, trans, size);
	}
	else if (mode & TEX_ALPHA_TEST)
	{
		// if there are no transparent pixels, make it a 3 component texture even
		// if it was specified as otherwise
		if (Check_QPAL_Copy_To_RGBA_Return_True_If_AlphaChannel(data, trans, size) == false)
			mode &= ~TEX_ALPHA_TEST;	 // No actual alpha pixels so strip alpha flag
	}
	else
	{
		if (size & 3)
			Sys_Error ("GL_Upload8: bad size (%d)", size);

		QPAL_To_RGBA (data, trans, size);
	}

	GL_Upload32 (trans, width, height, mode);
}

/*
================
GL_LoadTexture
================
*/
int GL_LoadTexture (const char *identifier, int in_width, int in_height, byte *data, int mode, const int bpp)
{
	int		width = in_width;
	int		height = in_height;

	int		i, scaled_width, scaled_height, crc = 0;
	gltexture_t	*glt;

	ScaleDimensions (width, height, &scaled_width, &scaled_height, mode);

	// see if the texture is already present
	if (identifier[0])
	{
		crc = CRC_Block (data, width * height * bpp);
		for (i=0, glt=gltextures ; i<numgltextures ; i++, glt++)
		{
			if (!strncmp(identifier, glt->identifier, sizeof(glt->identifier)-1)) // Baker: I'm not at all fond of how this line of code is written
#pragma message ("GL_LoadTexture: I'm not convinced this the above line of identifier comparison is right at all.  Shouldn't it be strlen and not sizeof?")
			{
				if (width == glt->width && height == glt->height &&
					scaled_width == glt->scaled_width && scaled_height == glt->scaled_height &&
					crc == glt->crc && bpp == glt->bpp &&
					(mode & ~TEX_COMPLAIN) == (glt->texmode & ~TEX_COMPLAIN))
				{
					GL_Bind (gltextures[i].texnum);
					return gltextures[i].texnum;	// texture cached
				}
				else
				{
					goto GL_LoadTexture_setup;	// reload the texture into the same slot
				}
			}
		}
	}

	if (numgltextures == MAX_GLTEXTURES)
		Sys_Error ("GL_LoadTexture: numgltextures == MAX_GLTEXTURES");

	glt = &gltextures[numgltextures];
	numgltextures++;

	StringLCopy (glt->identifier, identifier);
	glt->texnum = texture_extension_number;
	texture_extension_number++;

GL_LoadTexture_setup:
	glt->width = width;
	glt->height = height;
	glt->scaled_width = scaled_width;
	glt->scaled_height = scaled_height;
	glt->texmode = mode;
	glt->crc = crc;
	glt->bpp = bpp;
	if (glt->pathname_explicit)
	{
		Z_Free (glt->pathname_explicit);
		glt->pathname_explicit = NULL;
	}

	if (bpp == RGBA_BYTES_PER_PIXEL_IS_4 && qfs_lastload.datapath[0])
		glt->pathname_explicit = Z_Strdup (qfs_lastload.datapath_explicit /* explicitly must be correct file in a pak */);

	GL_Bind (glt->texnum);

	switch (bpp)
	{
	case QPAL_BYTES_PER_PIXEL_IS_1:
		GL_Upload8 (data, width, height, mode);
		break;

	case RGBA_BYTES_PER_PIXEL_IS_4:
		GL_Upload32 ((void *)data, width, height, mode);
		break;

	default:
		Sys_Error ("GL_LoadTexture: unknown bpp");
		break;
	}

	return glt->texnum;
}

/*
================
GL_FindTexture
================
*/
gltexture_t *GL_FindTexture (const char *identifier)
{
	int	i;

	for (i=0 ; i<numgltextures ; i++)
	{
		if (COM_StringMatch (identifier, gltextures[i].identifier))
			return &gltextures[i];
	}

	return NULL;
}


int GL_FindTextureNum (const char *identifier)
{
	int	i;

	for (i=0 ; i<numgltextures ; i++)
	{
		if (COM_StringMatch (identifier, gltextures[i].identifier))
			return i;
	}

	return -1;
}

static	gltexture_t	*active_gltexture = NULL;

#define CHECK_TEXTURE_ALREADY_LOADED		\
	if (CheckTextureLoaded(mode))			\
	{										\
		active_gltexture = NULL;			\
		fclose (f);							\
		return NULL;						\
	}


static qbool CheckTextureLoaded (int mode)
{
	int	scaled_width, scaled_height;

	if (active_gltexture && active_gltexture->pathname_explicit && COM_StringMatch (qfs_lastload.datapath_explicit, active_gltexture->pathname_explicit))
	{
		ScaleDimensions (active_gltexture->width, active_gltexture->height, &scaled_width, &scaled_height, mode);
		if (active_gltexture->scaled_width == scaled_width && active_gltexture->scaled_height == scaled_height)
			return true;
	}

	return false;
}

qbool		no24bit;

//stays ... no preprocessing
byte *GL_LoadExternalImage_RGBAPixels (const char *filename, int *owidth, int *oheight, int mode, const char *media_owner_path)
{
	char	basename[MAX_QPATH], name[MAX_QPATH];
	byte	*c, *data;
	FILE	*f;
	int		getwidth, getheight;

	if (strstr(filename, ".mdl"))
		StringLCopy (basename, filename);		// File is a model, keep the extension
	else
		COM_Copy_StripExtension (filename, basename);			// File is not a model, strip the extension


	for (c = basename ; *c ; c++)	// Fix up liquids names.  Can't have * in external filenames you know ...
		if (*c == '*')
			*c = '#';

	snprintf (name, sizeof(name), "%s.pcx", basename);
//	Con_Printf("... Attempting texture %s ... ", name);
	if (QFS_FOpenFile(name, &f, media_owner_path /*PATH LIMITED*/))
	{
		CHECK_TEXTURE_ALREADY_LOADED;	// Baker: <---- this is bad please eliminate it
		data = Image_LoadPCX_FromOpenFile (f, &getwidth, &getheight);
		fclose (f);

		if (data) {*owidth = getwidth; *oheight = getheight; return data; }
	}

	snprintf (name, sizeof(name), "%s.tga", basename);

//	Con_Printf("... Attempting texture %s ... ", name);
	if (QFS_FOpenFile(name, &f, media_owner_path /*PATH LIMITED*/))
	{
		CHECK_TEXTURE_ALREADY_LOADED;	// Baker: <---- this is bad please eliminate it
		data = Image_LoadTGA_FromOpenFile (f, &getwidth, &getheight);
		fclose (f);

		if (data) {*owidth = getwidth; *oheight = getheight; return data; }
	}




#if SUPPORTS_LIBPNG						// Baker: <---- this is bad please eliminate it
	snprintf (name, sizeof(name), "%s.png", basename);
	if (QFS_FOpenFile(name, &f, media_owner_path /*PATH LIMITED*/))
	{
		CHECK_TEXTURE_ALREADY_LOADED;
		data = Image_LoadPNG_FromOpenFile (f, &getwidth, &getheight);
		fclose (f);

		if (data) {*owidth = getwidth; *oheight = getheight; return data; }
	}
#endif

#if defined(SUPPORTS_LIBJPEG) && defined (SUPPORTS_JPEG_EXTERNAL_TEXTURES)
	snprintf (name, sizeof(name), "%s.jpg", basename);
	if (QFS_FOpenFile(name, &f, media_owner_path /*PATH LIMITED*/))
	{
		CHECK_TEXTURE_ALREADY_LOADED;		// Baker: <---- this is bad please eliminate it
		data = Image_LoadJPEG_FromOpenFile (f, &getwidth, &getheight);
		fclose (f);

		if (data) {*owidth = getwidth; *oheight = getheight; return data; }
	}
#endif

	if (mode & TEX_COMPLAIN)
		if (!no24bit)
			Con_Printf ("Couldn't load %s image\n", StringTemp_SkipPath(filename));

	return NULL;
}

// This handles RGBA as an intermediary to GL_LoadTexture
// Applies palette gamma (not typically used)
// Strip alpha rendering flag if no alpha channel found


int GL_LoadExternalTextureImage (const char *filename, const char *identifier, int mode, const char *media_owner_path)
{
	int			texnum;
	byte		*data;
	gltexture_t	*old_gltexture;
	int			mwidth, mheight;

	if (no24bit)
		return 0;

	if (!identifier)
		identifier = filename;

	old_gltexture = active_gltexture = GL_FindTexture (identifier);	// Check and see if texture is already loaded?

	if (!(data = GL_LoadExternalImage_RGBAPixels(filename, &mwidth, &mheight, mode, media_owner_path)))
	{
		texnum = (old_gltexture && !active_gltexture) ? old_gltexture->texnum : 0;
	}
	else
	{
		texnum = GL_LoadPreprocess_Texture_RGBA_Pixels_And_Upload (data, identifier, mwidth, mheight, mode);
//		// save 32 bit texels if hi-res player skin found
//		if (!strncmp(identifier, "player_", 7))
//		{
//			memcpy (player_texels32, data, image_width * image_height * 4);
//			player32_width = image_width;
//			player32_height = image_height;
//		}
		ImageWork_free (data);
	}

	active_gltexture = NULL;
	return texnum;
}


qbool IsLumaTex(const int texturenum)
{
	return (gltextures[texturenum].texmode & TEX_LUMA);

}

#if SUPPORTS_GL_DELETETEXTURES
/*
================
GL_FreeTextures -- BPJ
================
*/
void TexMgr_FreeTextures_With_ClearMemory (void)
{
	int i, j;

	if (!tex_free_on_newmap.integer)
	{
		Con_DevPrintf(DEV_MODEL, "GL_FreeTextures: Not clearing loaded map textures.\n");
		return;
	}

	for (i = j = 0; i < numgltextures; ++i, ++j)
	{
		if (gltextures[i].texmode & TEX_WORLD)
		{
			Con_DevPrintf(DEV_MODEL, "GL_FreeTextures: Clearing texture: %s.\n", gltextures[i].identifier);
			eglDeleteTextures(1, &gltextures[i].texnum);
			--j;
		}
		else if (j < i)
			gltextures[j] = gltextures[i];
	}

	numgltextures = j;
}
#endif