#include "common.h"
#include "net.h"
#include "protocol.h"
#include "renderer.h"
#include "glrenderer.h"

//#include <SDL_scancode.h>

//doesn't consider origins. angles are in radians.
void EntAngles(float *mat, float *ang)
{
	vec cy = (vec)cos(ang[1]);
	vec sy = (vec)sin(ang[1]);
	vec cp = (vec)cos(ang[0]);
	vec sp = (vec)sin(ang[0]);
	vec cr = (vec)cos(ang[2]);
	vec sr = (vec)sin(ang[2]);

	mat[0]  = cy * cp;
	mat[1]  = sy * cp;
	mat[2]  = -sp;
	mat[3]  = 0;

	mat[4]  = -sy * cr + cy * sp * sr;
	mat[5]  = cy * cr + sy * sp * sr;
	mat[6]  = cp * sr;
	mat[7]  = 0;

	mat[8]  = -sy * -sr + cy * sp * cr;
	mat[9]  = cy * -sr + sy * sp * cr;
	mat[10] = cp * cr;
	mat[11]  = 0;

	mat[15]  = 1;
}

//doesn't consider origins.
template<typename srcang>void LerpEntAngles(float *mat, srcang *newang, srcang *oldang, float newfrac, float scale, float pitchscale)
{
	float ang[3];
	scale = (M_PI*2 / scale);
	//interpolate the angles and attempt to cope with wrapping
	for (unsigned int j = 0; j < 3; j++)
	{
		float diff = newang[j] - oldang[j];
		diff *= scale;
		if (diff > M_PI)
			diff -= (float)M_PI*2;
		if (diff < -M_PI)
			diff += (float)M_PI*2;
		ang[j] = oldang[j]*scale + diff * newfrac;
	}

	ang[0] *= pitchscale;

	EntAngles(mat, ang);
}


typedef struct
{
	unsigned short modelindex;
	signed short origin[3];

	unsigned char angles[3];
	unsigned char effects;

	unsigned char colourmap;
	unsigned char skin;
	unsigned short frame;
} entstate_t;

class gamestate
{
public:
	unsigned int maxclients;
	unsigned int protocol;
	bool deathmatch;
	bool paused;
	char mapname[256];

	model_c *modelindexes[256];
	char *lightstylestrings[256];	//note: 255 is invalid.

	entstate_t staticents[128];
	unsigned int numstaticents;

	entstate_t *baselines;
	unsigned int baselines_max;

	float			servertime;
	float			oldservertime;
	float			localtime;
	float			oldlocaltime;
	struct curentities_s
	{
		entstate_t state;
		short oldorg[3];
		byte oldang[3];
		short oldframe;
		float oldtime;
	} *entities;
	unsigned int entities_max;	//max array size
	unsigned int entities_visible;	//maximum entity index currently visible. might be nice to compact the list or something, but whatever.

	int stats[32];
	unsigned int viewentity;

	vec3 viewang;
	vec3 vieworg;

	//when playing demos, the angles are from the client and so are handled completely separately from everything else
	vec3 demo_angles[2];
	float demo_time[2];

	gamestate()
	{
		for (unsigned int i = 0; i < countof(modelindexes); i++)
			modelindexes[i] = NULL;
		for (unsigned int i = 0; i < countof(lightstylestrings); i++)
		{
			lightstylestrings[i] = NULL;
		}

		baselines = NULL;
		baselines_max = 0;
		entities = NULL;
		entities_max = 0;
	}

	void ClearState(void)
	{
		viewang = vec3(0,0,0);
		for (unsigned int i = 0; i < countof(modelindexes); i++)
			modelindexes[i] = NULL;
		for (unsigned int i = 0; i < countof(lightstylestrings); i++)
		{
			free(lightstylestrings[i]);
			lightstylestrings[i] = NULL;
		}
		servertime = 0;
		paused = false;
		viewentity = 0;
		entities_visible = 0;
		numstaticents = 0;
		delete[] baselines;
		delete[] entities;
		baselines = NULL;
		baselines_max = 0;
		entities = NULL;
		entities_max = 0;
	}

	void ParseLightStyle(netbuf *buf)
	{
		char stylestring[2048];
		unsigned int style = buf->ReadByte();
		buf->ReadString(stylestring, sizeof(stylestring));

		if (style < countof(lightstylestrings))
		{
			free(lightstylestrings[style]);
			lightstylestrings[style] = _strdup(stylestring);
		}
	}

	bool ParseServerInfo(netbuf *buf)
	{
		unsigned int idx;
		unsigned char mapname[2048];

		ClearState();

		protocol = buf->ReadULong();
		if (protocol != 15 && protocol != 666)
		{
			Com_Printf(PRINT_INFO, "Unsupported protocol version %i - ", protocol);
			return false;
		}
		unsigned int maxclients = buf->ReadByte();
		unsigned char deathmatch = buf->ReadByte();

		buf->ReadString(mapname, sizeof(mapname));
		Com_Printf(PRINT_INFO, "\n%s\n\n", mapname);

		modelindexes[1] = NULL;
		for(idx = 1; ; idx++)
		{
			char model[2048];
			buf->ReadString(model, sizeof(model));
			if (!*model)
				break;
//			Com_Printf(PRINT_INFO, "model %i %s\n", idx, model);

			if (idx == countof(modelindexes))
				return false;
			if (*model == '*' && modelindexes[1])
				modelindexes[idx] = modelindexes[1]->GetSubmodel(atoi(model+1));
			else
				modelindexes[idx] = R_Model_Get(model);
		}
		for(idx = 1; ; idx++)
		{
			unsigned char sound[2048];
			buf->ReadString(sound, sizeof(sound));
			if (!*sound)
				break;
//			Com_Printf(PRINT_INFO, "sound %i %s\n", idx, sound);
		}

		return true;
	}

	void AddStatic(entstate_t es)
	{
		if (numstaticents < countof(staticents))
			staticents[numstaticents++] = es;
	}
	void ParseSpawnStatic(netbuf *buf)
	{
		entstate_t es;

		es.modelindex	= buf->ReadByte();
		es.frame		= buf->ReadByte();
		es.colourmap	= buf->ReadByte();
		es.skin			= buf->ReadByte();
		for (unsigned int i = 0; i < 3; i++)
		{
		  es.origin[i]	= buf->ReadSShort();
		  es.angles[i]	= buf->ReadByte();
		}

		AddStatic(es);
	}
	void ParseSpawnStatic_Fitz(netbuf *buf)
	{
		entstate_t es;
		unsigned char flags = buf->ReadByte();

		es.modelindex	= (flags&(1u<<0))?buf->ReadUShort():buf->ReadByte();
		es.frame		= (flags&(1u<<1))?buf->ReadUShort():buf->ReadByte();
		es.colourmap	= buf->ReadByte();
		es.skin			= buf->ReadByte();
		for (unsigned int i = 0; i < 3; i++)
		{
		  es.origin[i]	= buf->ReadSShort();
		  es.angles[i]	= buf->ReadByte();
		}

		if (flags&(1u<<2))
			/*alpha =*/ buf->ReadByte();
		if (flags&(1u<<3))
			/*scale =*/ buf->ReadByte();

		AddStatic(es);
	}

	void ParseSpawnBaseline(netbuf *buf)
	{
		entstate_t es;
		unsigned int num = buf->ReadUShort();
		es.modelindex	= buf->ReadByte();
		es.frame		= buf->ReadByte();
		es.colourmap	= buf->ReadByte();
		es.skin			= buf->ReadByte();
		for (unsigned int i = 0; i < 3; i++)
		{
			es.origin[i] = buf->ReadSShort();
			es.angles[i] = buf->ReadByte();
		}

		if (num >= baselines_max)
		{
			entstate_t *newbase = new entstate_t[num+1];
			memcpy(newbase, baselines, sizeof(*newbase)*baselines_max);
			memset(newbase+baselines_max, 0, sizeof(*newbase)*(num+1-baselines_max));
			delete[] baselines;
			baselines = newbase;
			baselines_max = num+1;
		}
		baselines[num] = es;
	}
	void ResetEntities(netbuf *buf, float time)
	{
		oldservertime = servertime;
		servertime = buf->ReadFloat();
		oldlocaltime = localtime;
		localtime = time;

		entities_visible = 0;
		if (!entities_max)
		{
			entities_max = 300;
			entities = new curentities_s[entities_max];
		}
		memset(&entities[0], 0, sizeof(entities[0]));
		entities[0].state.modelindex = 1;
	}
	void ParseEntity(netbuf *buf, unsigned int flags)
	{
		static entstate_t default_baseline;
		entstate_t *base, es;
		unsigned char fitzflags = 0;
#define E_MOREFLAGS	(1u<<0)
#define E_ORIGIN_X	(1u<<1)
#define E_ORIGIN_Y	(1u<<2)
#define E_ORIGIN_Z	(1u<<3)
#define E_ANGLE_Y	(1u<<4)
#define E_NOLERP	(1u<<5)
#define E_FRAME		(1u<<6)
#define E_GAP		(1u<<7)	//could renumber to avoid this gap. although without, x86 can just load dh or whatever.
#define E_ANGLE_X	(1u<<8)
#define E_ANGLE_Z	(1u<<9)
#define E_MODELIDX	(1u<<10)
#define E_COLOURMAP	(1u<<11)
#define E_SKIN		(1u<<12)
#define E_EFFECTS	(1u<<13)
#define E_LARGE		(1u<<14)
#define E_UNUSED	(1u<<15)

		/*figure out the ent number, and the baselines*/
		if (flags & E_MOREFLAGS)
			flags |= buf->ReadByte()<<8;
		if (flags & E_UNUSED)
		{
			if (protocol == 666)
				fitzflags = buf->ReadByte();
			else
				Com_Printf(PRINT_WARNING, "Entity update contains unused flags\n");
		}
		unsigned int ent = (flags&E_LARGE)? buf->ReadUShort():buf->ReadByte();
		base = (ent<baselines_max)?&baselines[ent]:&default_baseline;

		/*now parse the data*/
		es.modelindex	= (flags&E_MODELIDX	)?buf->ReadByte  (): base->modelindex;
		es.frame		= (flags&E_FRAME	)?buf->ReadByte  (): base->frame; 
		es.colourmap	= (flags&E_COLOURMAP)?buf->ReadByte  (): base->colourmap; 
		es.skin			= (flags&E_SKIN		)?buf->ReadByte  (): base->skin;
		es.effects		= (flags&E_EFFECTS	)?buf->ReadByte  (): base->effects;
		es.origin[0]	= (flags&E_ORIGIN_X	)?buf->ReadSShort(): base->origin[0];
		es.angles[0]	= (flags&E_ANGLE_X	)?buf->ReadByte  (): base->angles[0];
		es.origin[1]	= (flags&E_ORIGIN_Y	)?buf->ReadSShort(): base->origin[1];
		es.angles[1]	= (flags&E_ANGLE_Y	)?buf->ReadByte  (): base->angles[1];
		es.origin[2]	= (flags&E_ORIGIN_Z	)?buf->ReadSShort(): base->origin[2];
		es.angles[2]	= (flags&E_ANGLE_Z	)?buf->ReadByte  (): base->angles[2];

		if (fitzflags & (1u<<0))
			/*alpha = */buf->ReadByte();
		if (fitzflags & (1u<<1))
			es.frame |= buf->ReadByte()<<8;
		if (fitzflags & (1u<<2))
			es.modelindex |= buf->ReadByte()<<8;
		if (fitzflags & (1u<<3))
			/*lerpfinish = */buf->ReadByte();

		if (ent >= entities_max)
		{
			curentities_s *newents = new curentities_s[ent+1];
			memcpy(newents, entities, sizeof(*newents)*entities_max);
			memset(newents+entities_max, 0, sizeof(*newents)*(ent+1-entities_max));
			delete[] entities;
			entities = newents;
			entities_max = ent+1;
		}

		while (entities_visible < ent)
			entities[++entities_visible].state.modelindex = 0;

		if (!(flags&E_NOLERP) || es.frame != entities[ent].state.frame ||
			entities[ent].oldorg[0] != entities[ent].state.origin[0] ||
			entities[ent].oldorg[1] != entities[ent].state.origin[1] ||
			entities[ent].oldorg[2] != entities[ent].state.origin[2])
		{
			entities[ent].oldorg[0] = entities[ent].state.origin[0];
			entities[ent].oldorg[1] = entities[ent].state.origin[1];
			entities[ent].oldorg[2] = entities[ent].state.origin[2];
			entities[ent].oldang[0] = entities[ent].state.angles[0];
			entities[ent].oldang[1] = entities[ent].state.angles[1];
			entities[ent].oldang[2] = entities[ent].state.angles[2];
			entities[ent].oldframe = entities[ent].state.frame;
			entities[ent].oldtime = oldservertime;
		}

		entities[ent].state = es;
	}
	void ParseClientData(netbuf *buf)
	{
#define CD_VIEWHEIGHT	(1u<<0)
#define CD_IDEALPITCH	(1u<<1)
#define CD_ANGLE_X		(1u<<2)
#define CD_ANGLE_Y		(1u<<3)
#define CD_ANGLE_Z		(1u<<4)
#define CD_VEL_X		(1u<<5)
#define CD_VEL_Y		(1u<<6)
#define CD_VEL_Z		(1u<<7)
//#define CD_UNUSED1		(1u<<8)
#define CD_ITEMS		(1u<<9)
//#define CD_UNKNOWN1		(1u<<10)
//#define CD_UNKNOWN2		(1u<<11)
#define CD_WEAPONFRAME	(1u<<12)
#define CD_ARMOURVALUE	(1u<<13)
#define CD_WEAPONMODEL	(1u<<14)
//#define CD_UNUSED2		(1u<<15)
		unsigned int flags;
		flags = buf->ReadUShort();

		stats[STAT_VIEWHEIGHT] = (flags&CD_VIEWHEIGHT)?(signed char)buf->ReadByte():22;
		if (flags & CD_IDEALPITCH)
			buf->ReadByte();

		for (unsigned int i = 0; i < 3; i++)
		{
			if (flags & (CD_ANGLE_X<<i))
				buf->ReadByte();
			if (flags & (CD_VEL_X<<i))
				buf->ReadByte();
		}

		if (flags & CD_ITEMS)
			stats[STAT_ITEMS] = buf->ReadULong();

		stats[STAT_WEAPONFRAME] = (flags & CD_WEAPONFRAME) ? buf->ReadByte() : 0;
		stats[STAT_ARMOURVALUE] = (flags & CD_ARMOURVALUE) ? buf->ReadByte() : 0;
		stats[STAT_WEAPONMODEL] = (flags & CD_WEAPONMODEL) ? buf->ReadByte() : 0; 
		stats[STAT_HEALTH] = buf->ReadSShort();  
		stats[STAT_CURRENTAMMO] = buf->ReadByte();
		stats[STAT_AMMO_SHELLS] = buf->ReadByte();
		stats[STAT_AMMO_NAILS] = buf->ReadByte();
		stats[STAT_AMMO_ROCKETS] = buf->ReadByte();
		stats[STAT_AMMO_CELLS] = buf->ReadByte();
		stats[STAT_WEAPONID] = buf->ReadByte();
	}


	void ParsePlayerColours(netbuf *buf)
	{
		unsigned char pl = buf->ReadByte();
		unsigned char colours = buf->ReadByte();
	}
	void ParsePlayerFrags(netbuf *buf)
	{
		unsigned char pl = buf->ReadByte();
		short frags = buf->ReadSShort();
	}
	void ParsePlayerName(netbuf *buf)
	{
		char name[1024];
		unsigned char pl = buf->ReadByte();
		buf->ReadString(name, sizeof(name));
	}
};
class effectsystem
{
	gamestate *gs;
	unsigned int numeffects;
	struct
	{
		unsigned int id;
		vec3 start;
		vec3 end;
		model_c *mdl;
		float life;
	} effect[16];	//lame. lazy. screw you.
	void AddBeam(int id, vec3 start, vec3 end, model_c *mdl)
	{
		unsigned int i;
		if (!mdl)
			return;

		for (i = (id?0:numeffects); i < numeffects; i++)
		{
			if (effect[i].id == id)
				break;
		}
		if (i == numeffects)
		{
			if (i == countof(effect))
				return;	//overflow
			numeffects++;
			effect[i].id = id;
		}
		effect[i].mdl = mdl;
		effect[i].start = start;
		effect[i].end = end;
		effect[i].life = 0.2f;
	}
public:
	effectsystem(gamestate *thegs) : gs(thegs)
	{
		numeffects = 0;
	}
	void RenderEffects(rstate_t *scene, float frametime)
	{
		for (unsigned int i = 0; i < numeffects; )
		{
			if (effect[i].life < 0)
			{
				numeffects--;
				memcpy(&effect[i], &effect[i+1], sizeof(effect[i])*(numeffects-i));
				continue;
			}
			effect[i].life -= frametime;

			vec3 start = effect[i].start;
//			if (gs->viewentity == effect[i].i)
//				start[i] ;
			vec3 dir = effect[i].end - effect[i].start;
			unsigned int segments = dir.length() / 30;
			dir.normalize();
			dir *= 30;

			vec3 ang = dir.toeularangles();

			rentity_t ent = {};
			ent.animweight[0] = 1;
			ent.matrix.MakeIdentity();	//FIXME
			while(segments)
			{
				ent.matrix[12] = start[0];
				ent.matrix[13] = start[1];
				ent.matrix[14] = start[2];
				if (scene->frustum.Contains(vec3(ent.matrix[12], ent.matrix[13], ent.matrix[14]), 64))
				{
					ang[2] = ((segments&1)?1:-1) * (50*effect[i].life + segments);
					EntAngles(ent.matrix.data, ang.v);
					effect[i].mdl->DrawEntity(&ent, scene);
				}

				start += dir;
				segments--;
			}
			i++;
		}
	}
	void ParseParticles(netbuf *buf)
	{
		vec3 pos, vel;
		pos[0] = buf->ReadSShort() / 8.0;
		pos[1] = buf->ReadSShort() / 8.0;
		pos[2] = buf->ReadSShort() / 8.0;
		vel[0] = buf->ReadByte() - 128;
		vel[1] = buf->ReadByte() - 128;
		vel[2] = buf->ReadByte() - 128;
		unsigned char palcolour = buf->ReadByte();
		unsigned char count = buf->ReadByte();

//		if (count == 255)
//			explosion();
	}

	void ParseTempEntity(netbuf *buf)
	{
		unsigned char te = buf->ReadByte();
		switch(te)
		{
		case 0:	//spike
		case 1:	//superspike
		case 2:	//gunshot
		case 3:	//explosion
		case 4:	//tarexplosion
		case 7:	//wizspike
		case 8:	//knightspike
		case 10://lavasplash
		case 11://teleport
			buf->ReadSShort();
			buf->ReadSShort();
			buf->ReadSShort();
			break;
		case 5://lightning1(shambler)
		case 6://lightning2(player)
		case 9://lightning3(boss)
			{
				unsigned int id;
				vec3 start, end;

				id = buf->ReadUShort();
				start[0] = buf->ReadSShort();
				start[1] = buf->ReadSShort();
				start[2] = buf->ReadSShort();
				end[0] = buf->ReadSShort();
				end[1] = buf->ReadSShort();
				end[2] = buf->ReadSShort();
				start *= 1/8.0;
				end *= 1/8.0;

				if (te == 5)
					AddBeam(id, start, end, R_Model_Get("progs/bolt.mdl"));
				else if (te == 6)
					AddBeam(id, start, end, R_Model_Get("progs/bolt2.mdl"));
				else //if (te == 9)
					AddBeam(id, start, end, R_Model_Get("progs/bolt3.mdl"));
			}
			break;
		default:
			buf->badread = true;
			break;
		}
	}
};

class soundsystem
{
	gamestate *gs;

	//quake uses entities only as a way to clear out playing samples.
public:
	soundsystem(gamestate *thegs) : gs(thegs)
	{
	}
	void ParseSound(netbuf *buf)
	{
		unsigned char flags = buf->ReadByte();
		unsigned short ent;
		unsigned short sampleidx;
		unsigned char chan;
		if (flags & ~(1|2|8|16))
			Com_Printf(PRINT_WARNING, "Unknown flags in ParseSound\n");
		/*volume=*/(flags&1)?buf->ReadByte()/255.0:1.0;
		/*attenuation=*/(flags&2)?buf->ReadByte()/64.0:1.0;

		if (flags & 8)
		{	//non-vanilla
			ent = buf->ReadUShort();
			chan = buf->ReadByte();
		}
		else
		{
			unsigned short ec = buf->ReadUShort();
			chan = ec & 0x7;
			ent = ec >> 3;
		}

		if (flags & 16)	//non-vanilla
			sampleidx = buf->ReadUShort();
		else
			sampleidx = buf->ReadByte();

		/*x=*/buf->ReadSShort();
		/*y=*/buf->ReadSShort();
		/*z=*/buf->ReadSShort();

		/*
		if (sampleidx >= sizeof(gs->soundnames)/sizeof(gs->soundnames))
			sampleidx = 0;
		soundname = gs->soundnames[sampleidx];
		*/
	}
	void ParseStaticSound(netbuf *buf)
	{
		short x=buf->ReadSShort();
		short y=buf->ReadSShort();
		short z=buf->ReadSShort();
		unsigned short sample=buf->ReadByte();
		float volume=buf->ReadByte()/255.0;
		float attenuation=buf->ReadByte()/64.0;
	}
};

class client : public genclient, inputsink
{
	enum constate_e {
		failed,			//given up
		connecting,		//still attempting to connect to the server
		connected,		//we've got a connection but are still waiting for server info
		active			//we got all the data we need. we're now meant to be drawing the game.
	} state;
	netcon *con;
	netpeer serveraddr;
	float resend;
	float inputtime;
	float cltime;	//local time since connection reset.
	float lasttime;	//to calculate cltime properly

	float starttime;	//for timedemos
	unsigned int framecount;	//for timedemos

	gamestate gs;
	effectsystem es;
	soundsystem ss;

	
	unsigned char pending_reliable_buf[65536];
	unsigned int pending_reliable_size;
	unsigned int pending_reliable_offset;
	unsigned int pending_reliable_chunk;
	unsigned int pending_reliable_sequence;

	unsigned char outreliable_buf_data[65536];
	netbuf outreliable;
	float reliableretransmit;
	unsigned int outunreliablesequence;

	unsigned char inreliable_buf_data[65536];
	netbuf inreliable;
	unsigned int inreliablesequence;
	unsigned int inunreliablesequence;

	void ClearState(constate_e newstate)
	{
		gs.ClearState();
		state = newstate;

		move_forward = move_back = move_left = move_right = false;
		move_impulse = 0;
		move_buttons = 0;

		if (newstate == failed)
		{	//also clear connection state too.
			pending_reliable_sequence = 0;
			pending_reliable_size = 0;
			pending_reliable_offset = 0;
			pending_reliable_chunk = 0;
			outreliable.Reset();
			outunreliablesequence = 0;
			
			inreliable.Reset();
			inreliablesequence = 0;
			inunreliablesequence = 0;
			if (con)
				delete con;
			con = NULL;
		}
	}


	void ClientCommand(const char *cmd)
	{
		outreliable.WriteByte(clc_clientcmd);
		outreliable.WriteString(cmd);
	}


	void ParsePacket(netbuf *buf, float time)
	{
		for(;;)
		{
			unsigned char str[2048];

			netprotocol_e cmd = (netprotocol_e)buf->ReadByte();
			switch(cmd)
			{
			case svc_bad:
				if (buf->badread)
					return;	//EOM
				break;
			case svc_nop:
				break;
			case svc_disconnect:
				Com_Printf(PRINT_NOTIFCATION, "svc_disconnect\n");
				state = failed;
				break;
			case svc_updatestat:
				{
					unsigned char statid = buf->ReadByte();
					if (statid >= countof(gs.stats))
						statid = STAT_UNUSED;
					gs.stats[statid] = buf->ReadSLong();
				}
				break;
//	svc_version = 4,
			case svc_setview:
				gs.viewentity = buf->ReadUShort();
				break;
			case svc_sound:
				ss.ParseSound(buf);
				break;
			case svc_time:
				gs.ResetEntities(buf, time);
				break;
			case svc_print:
				Com_Printf(PRINT_INFO, "%s", buf->ReadString(str, sizeof(str)));
				break;
			case svc_stufftext:
				Com_Printf(PRINT_INFO, "stufftext: %s", buf->ReadString(str, sizeof(str)));
				if (!strncmp((char*)str, "reconnect", 9))
				{
					state = connected;
					ClientCommand("new");
				}
				else if (!strcmp((char*)str, "cmd protocols\n"))
				{
					ClientCommand("protocols 666\n");
				}
				else if (!strncmp((char*)str, "cmd ", 4))
				{
					ClientCommand((char*)str + 4);
				}
				break;
			case svc_setangle:
				gs.viewang[0] = buf->ReadByte() * 360 / 256.0;
				gs.viewang[1] = buf->ReadByte() * 360 / 256.0;
				gs.viewang[2] = buf->ReadByte() * 360 / 256.0;
				break;
			case svc_serverinfo:
				cltime = 0;
				if (!gs.ParseServerInfo(buf))
				{
					state = failed;
					return;
				}
				if (!starttime)
				{	//timedemo starts once loading is complete.
					//nq is pretty simple in this regard
					framecount = 0;
					starttime = time;
				}
				break;
			case svc_lightstyle:
				gs.ParseLightStyle(buf);
				break;
			case svc_updatename:
				gs.ParsePlayerName(buf);
				break;
			case svc_updatefrags:
				gs.ParsePlayerFrags(buf);
				break;
			case svc_clientdata:
				gs.ParseClientData(buf);
				break;
//	svc_stopsound = 0x10,
			case svc_updatecolours:
				gs.ParsePlayerColours(buf);
				break;
			case svc_particle:
				es.ParseParticles(buf);
				break;
			case svc_damage:
				/*armour =*/buf->ReadByte();
				/*health =*/buf->ReadByte();
				/*x =*/buf->ReadSShort();
				/*y =*/buf->ReadSShort();
				/*z =*/buf->ReadSShort();
				break;
			case svc_spawnstatic:
				gs.ParseSpawnStatic(buf);
				break;
			case 0x2b:
				gs.ParseSpawnStatic_Fitz(buf);
				break;
//	svc_spawnbinary = 0x15,
			case svc_spawnbaseline:
				gs.ParseSpawnBaseline(buf);
				break;
			case svc_temp_entity:
				es.ParseTempEntity(buf);
				break;
			case svc_setpause:
				gs.paused = !!buf->ReadByte();
				break;
			case svc_signonnum:
				switch(buf->ReadByte())
				{
				case 1:
#ifdef USEVK
					ClientCommand("name spoQ");
#else
					ClientCommand("name foobar");
#endif
					ClientCommand("color 4");
					ClientCommand("prespawn");
					break;
				case 2:
					ClientCommand("spawn");
					break;
				case 3:
					ClientCommand("begin");
					break;
				}
				break;
			case svc_centerprint:
				Com_Printf(PRINT_INFO, "centerprint: %s\n", buf->ReadString(str, sizeof(str)));
				break;
			case svc_killedmonster:
				gs.stats[STAT_FOUNDMONSTERS]++;
				break;
			case svc_foundsecret:
				gs.stats[STAT_FOUNDSECRETS]++;
				break;
			case svc_spawnstaticsound:
				ss.ParseStaticSound(buf);
				break;
			case svc_intermission:
				break;	//nothing... :(
			case svc_finale:
				buf->ReadString(str, sizeof(str));
				break;
			case svc_cdtrack:
				/*track = */buf->ReadByte();
				/*loop = */buf->ReadByte();
				break;
			case svc_sellscreen:
				break;	//nothing... :)
			default:
				if (cmd & svc_entity)
				{
					gs.ParseEntity(buf, cmd&0x7f);
					break;
				}
				Com_Printf(PRINT_ERROR, "Unknown Server Message: %2x\n", cmd);
//				Sys_Error("Unknown Server Message: %2x\n", cmd);
				return;
			}
			if (buf->badread)
			{
				state = failed;
				Com_Printf(PRINT_ERROR, "Illegible Server Message (last parsed %2x)\n", cmd);
			}
		}
	}

	int move_impulse;
	int move_forward;
	int move_back;
	int move_left;
	int move_right;
	int move_buttons;
	virtual void KeyEvent		(int down, int scancode)
	{
		down = !!down;
		if (scancode == 27)
			state = failed;
		else if (scancode == 'w')
			move_forward = down;
		else if (scancode == 's')
			move_back = down;
		else if (scancode == 'a')
			move_left = down;
		else if (scancode == 'd')
			move_right = down;
		else if (scancode == '0')
			move_impulse = 10;
		else if (scancode >= '1' && scancode <= '9')
			move_impulse = scancode - '0';
	}
	virtual void TextInput		(char *text)
	{
	}
	virtual void ButtonEvent	(int down, int button)
	{
		button--;
		if (down)
			move_buttons |= 1<<button;
		else
			move_buttons &= ~(1<<button);
	}
	virtual void MouseDelta		(int x, int y)
	{
		gs.viewang.v[0] += y*0.3;
		if (gs.viewang.v[0] < -90)
			gs.viewang.v[0] = -90;
		if (gs.viewang.v[0] > 90)
			gs.viewang.v[0] = 90;
		gs.viewang.v[1] -= x*0.3;
	}
	virtual void MouseAbsolute	(int x, int y)
	{
	}


public:
	client(const char *server, float time)
		: outreliable(outreliable_buf_data, sizeof(outreliable_buf_data))
		, inreliable(inreliable_buf_data, sizeof(inreliable_buf_data))
		, ss(&gs)
		, es(&gs)
	{
		con = NULL;
		resend = time;
		starttime = 0;
		framecount = 0;
		ClearState(failed);
		if (serveraddr.Resolve(server, 26000))
		{
			con = netcon::Net_Establish(&serveraddr);
			state = connecting;
		}
	};
	bool CheckPackets(float time)
	{
		netpeer peer;
		unsigned char bufdata[65536];
		netbuf buf(bufdata, sizeof(bufdata));

		cltime += time - lasttime;
		lasttime = time;

		if (state == failed)
			return false;

		if (state == connecting)
		{
			if (resend <= time)
			{
				buf.WriteByte(0x80);buf.WriteByte(0x00);	//control type
				buf.WriteByte(0x00);buf.WriteByte(0x0c);	//len
				buf.WriteByte(0x01);						//con request
				buf.WriteString("QUAKE");					//name
				buf.WriteByte(0x03);						//version
				con->Send(&buf, &serveraddr);
				resend = time + 1;
			}
		}
		else
		{
			if (resend <= time)
			{
				if (pending_reliable_offset >= pending_reliable_size)
				{
					pending_reliable_chunk = 0;
					pending_reliable_offset = 0;
					pending_reliable_size = outreliable.ReadData(pending_reliable_buf, sizeof(pending_reliable_buf));
					if (outreliable.readpos != outreliable.writepos)
						pending_reliable_size = 0;
					outreliable.Reset();
				}
				if (pending_reliable_size)
				{
					unsigned short len;
					if (!pending_reliable_chunk)
					{
						pending_reliable_chunk = pending_reliable_size - pending_reliable_offset;
						if (pending_reliable_chunk > 1024)
							pending_reliable_chunk = 1024;
					}

					buf.Reset();
					buf.WriteByte(0x00);
					if (pending_reliable_chunk == pending_reliable_size - pending_reliable_offset)
						buf.WriteByte(0x09);	//eom
					else
						buf.WriteByte(0x01);	//chunk
					len = pending_reliable_chunk + 8;
					buf.WriteByte(len>>8);buf.WriteByte(len);
					buf.WriteByte(pending_reliable_sequence>>24);buf.WriteByte(pending_reliable_sequence>>16);buf.WriteByte(pending_reliable_sequence>>8);buf.WriteByte(pending_reliable_sequence);
					buf.WriteData(pending_reliable_buf+pending_reliable_offset, pending_reliable_chunk);
					con->Send(&buf, &serveraddr);
					resend = time + 1;
				}
			}
		}

		for(;;)
		{
			//FIXME: make sure peer makes sense

			if (serveraddr.demofile)
			{
				if (cltime <= gs.servertime)
					break;
			}

			if (!con->Recv(&buf, &peer))
			{
				if (serveraddr.demofile && framecount)
				{
					Com_Printf(PRINT_NOTIFCATION, "Played %u frames in %g seconds (%g fps)\n", framecount, time - starttime, framecount / (time - starttime));

					framecount = 0;
					starttime = 0;
				}
				break;	//no data! oh noes!
			}

			unsigned short type;
			unsigned short len;	//netchan header is big-endian, everything else is little. oh well.

			if (peer.demofile)
			{	//should interpolate these.
				gs.demo_angles[1] = gs.demo_angles[0];
				gs.demo_time[1] = gs.demo_time[0];

				gs.demo_time[0] = time;
				gs.demo_angles[0][0] = buf.ReadFloat();
				gs.demo_angles[0][1] = buf.ReadFloat();
				gs.demo_angles[0][2] = buf.ReadFloat();

				if (!gs.demo_time[1])
				{
					gs.demo_angles[1] = gs.demo_angles[0];
					gs.demo_time[1] = gs.demo_time[0];
				}
				ParsePacket(&buf, time);
				continue;
			}

			type = buf.ReadByte()<<8;
			type |= buf.ReadByte();
			len = buf.ReadByte()<<8;
			len |= buf.ReadByte();

			if (type & 0x8000)	//control packet
			{
				unsigned char code = buf.ReadByte();
				switch(code)
				{
				case 0x81:
					unsigned int port;
					port = buf.ReadULong();	//yup, long. really. honest.
					if (state == connecting)
					{
						state = connected;
						port &= 0xffff;
						if (port)
							serveraddr.ForcePort(port);
						resend = time;
					}

					break;
				default:
					Com_Printf(PRINT_INFO, "unknown control packet: %2x\n", code);
					break;
				}
				continue;
			}
			else if (state != connected)
				continue;

			unsigned int sequence;
			sequence = buf.ReadByte()<<24;
			sequence |= buf.ReadByte()<<16;
			sequence |= buf.ReadByte()<<8;
			sequence |= buf.ReadByte();

			if (type == 1 || type == 9)
			{
				if (sequence == inreliablesequence)
				{
					inreliable.WriteBuf(&buf);

					if (type == 9)	//eom. parse it now
					{
						ParsePacket(&inreliable, time);
						inreliable.Reset();
					}
					inreliablesequence++;	//now expect the next
				}


				//reliables need an ack (always, even if its a dupe)
				buf.Reset();
				buf.WriteByte(0x00);buf.WriteByte(0x02);	//control type
				buf.WriteByte(0x00);buf.WriteByte(0x08);	//len
				buf.WriteByte(sequence>>24);buf.WriteByte(sequence>>16);buf.WriteByte(sequence>>8);buf.WriteByte(sequence>>0);	//sequence that we're acking
				con->Send(&buf, &serveraddr);
			}
			else if (type == 0x10)
			{
				if (sequence <= inunreliablesequence)
					continue;
				inunreliablesequence = sequence;

				ParsePacket(&buf, time);
			}
			else if (type == 2)
			{
				if (sequence == pending_reliable_sequence)
				{
					pending_reliable_offset += pending_reliable_chunk;
					pending_reliable_sequence++;
					resend = time;
				}
			}
			else
			{
				Com_Printf(PRINT_INFO, "unknown packet type\n");
			}
		}
		return true;
	};

	bool RenderGameView(vid_t *vid, float time)
	{
		rentity_t worldent;
		rstate_t scene;
		float newfrac = 1, oldfrac = 0;
		float interval = gs.localtime - gs.oldlocaltime;
		float frametime;
		static float lasttime;
		float simtime;
		frametime = time - lasttime;
		lasttime = time;

		//paranoia.
		if (frametime < 0)
			frametime = 0;
		if (frametime > 1)
			frametime = 1;

		if (interval > 0.1f)
			interval = 0.1f;
		if (interval < 1/200.0f)
			interval = 1/200.0f;
		if (!interval)
			Com_Printf(PRINT_INFO, "no interval!\n");

		newfrac = (time - gs.localtime) / interval;
		if (newfrac < 0)
			newfrac = 0;
		if (newfrac > 1)
			newfrac = 1;
		oldfrac = 1 - newfrac;

		simtime = gs.oldservertime*oldfrac + gs.servertime*newfrac;

		framecount++;

		if (!gs.entities_visible)	//game does not start rendering until we have at least one entity.
			return true;

		//FIXME: determine desired server time (with drifting) and build a log of packet states (and delta between those)

		memset(&worldent, 0, sizeof(worldent));
		memset(&scene, 0, sizeof(scene));

		scene.mode = vid->mode;


		gamestate::curentities_s *e = &gs.entities[gs.viewentity];

		float aspect = (float)vid->height/vid->width;
		scene.fovx = 90;
		scene.fovy = 360.0/M_PI * atan(aspect * tan(scene.fovx/360.0*M_PI));
		scene.projectionmatrix.MakeProjectionMatrix(scene.fovx, scene.fovy, 4);

		if (state != failed)
		{
			gs.vieworg[0] = (e->state.origin[0]*newfrac + e->oldorg[0]*oldfrac) / 8.0;
			gs.vieworg[1] = (e->state.origin[1]*newfrac + e->oldorg[1]*oldfrac) / 8.0;
			gs.vieworg[2] = (e->state.origin[2]*newfrac + e->oldorg[2]*oldfrac) / 8.0;
			gs.vieworg[2] += gs.stats[STAT_VIEWHEIGHT];
		}
		else
		{
			vec3 fwd,right,up;
			matrix4x4(gs.viewang, vec3(0,0,0)).ToVectors(fwd,right,up);
			if (move_forward)
				gs.vieworg += fwd * (move_forward * 400) * frametime;
			if (move_back)
				gs.vieworg -= fwd * (move_back * 400) * frametime;
			if (move_right)
				gs.vieworg += right * (move_right * 400) * frametime;
			if (move_left)
				gs.vieworg -= right * (move_left * 400) * frametime;
//			if (move_up)
//				gs.vieworg += up * (move_up * 400) * interval;
		}
		scene.vieworigin = gs.vieworg;

		if (serveraddr.demofile && state != failed)
		{
			//demo angles can have a different interval than other stuff (present in reliables too)
			float demofrac = (time - gs.demo_time[0]) / (gs.demo_time[0] - gs.demo_time[1]);
			if (demofrac < 0)
				demofrac = 0;
			if (demofrac > 1)
				demofrac = 1;
			gs.viewang.interpolate_angles(gs.demo_angles[1], demofrac, gs.demo_angles[0], 360);
		}

		scene.viewmatrix.MakeViewMatrix(gs.viewang, scene.vieworigin);
		scene.frustum.FromMatricies(scene.viewmatrix, scene.projectionmatrix);

		scene.viewproj = (scene.projectionmatrix * scene.viewmatrix);
		scene.time = time;

		for (int i = 0; i < sizeof(gs.lightstylestrings)/sizeof(gs.lightstylestrings[0]); i++)
		{
			size_t l = gs.lightstylestrings[i]?strlen(gs.lightstylestrings[i]):0;
			if (l)
			{
				size_t v1 = (size_t)(time*10) % l;
				size_t v2 = (size_t)((time*10)+1) % l;
				float frac = (time*10) - floor(time*10);
				scene.lmscale[i] = ((gs.lightstylestrings[i][v1]*(1-frac) + frac*gs.lightstylestrings[i][v2] - 'a') * 22) / 128.0;
			}
			else
				scene.lmscale[i] = 1;
		}
/*		memcpy(screencbuf.modelviewproj, (scene.projectionmatrix * scene.viewmatrix).data, sizeof(float)*16);
		memcpy(screencbuf.eyepos, scene.vieworigin.v, sizeof(screencbuf.eyepos));
		screencbuf.rtime = time;

//		pglBindBuffer(GL_UNIFORM_BUFFER, cbuf);
		pglBufferData(GL_UNIFORM_BUFFER, sizeof(screencbuf), &screencbuf, GL_STREAM_DRAW);
*/

		worldent.animtime[0] = gs.servertime;
		worldent.animtime[1] = gs.servertime;
		e = gs.entities;
		float msgfrac = (simtime - e->oldtime) / (gs.servertime - gs.oldservertime);
		if (msgfrac > 1)
			msgfrac = 1;
		else if (msgfrac < 0)
			msgfrac = 0;
		for (unsigned int i = 0; i <= gs.entities_visible; i++, e++)
		{
			if (i == gs.viewentity)	//viewentity isn't drawn.
				continue;

			if (e->state.modelindex)
			{
				float efrac;
				if (e->state.modelindex >= countof(gs.modelindexes))
					continue;
				model_c *mdl = gs.modelindexes[e->state.modelindex];
				if (!mdl)
					continue;
				worldent.skinnum = e->state.skin;

				efrac = (simtime - e->oldtime) / 0.1;
				if (efrac > 1.1) efrac = msgfrac;
				else
				{
					if (efrac > 1) efrac = 1;
					if (efrac < 0) efrac = 0;
				}

				worldent.matrix[12] = (e->oldorg[0] + (e->state.origin[0]-e->oldorg[0])*efrac) * (1.0/8.0);
				worldent.matrix[13] = (e->oldorg[1] + (e->state.origin[1]-e->oldorg[1])*efrac) * (1.0/8.0);
				worldent.matrix[14] = (e->oldorg[2] + (e->state.origin[2]-e->oldorg[2])*efrac) * (1.0/8.0);
				if (i && !scene.frustum.Contains(vec3(worldent.matrix[12], worldent.matrix[13], worldent.matrix[14]), mdl->cullradius))
					continue;

				//fixme: lerp
				worldent.anim[0] = e->oldframe;
				worldent.anim[1] = e->state.frame;
				worldent.animweight[1] = efrac;
				worldent.animweight[0] = 1-efrac;

				LerpEntAngles(worldent.matrix.data, e->state.angles, e->oldang, efrac, 256, -1);
				mdl->DrawEntity(&worldent, &scene);
			}
		}

		if (gs.stats[STAT_WEAPONMODEL] > 0 && gs.stats[STAT_WEAPONMODEL] < countof(gs.modelindexes) && gs.modelindexes[gs.stats[STAT_WEAPONMODEL]])
		{
			worldent.skinnum = 0;


			worldent.matrix[12] = scene.vieworigin[0];
			worldent.matrix[13] = scene.vieworigin[1];
			worldent.matrix[14] = scene.vieworigin[2];

			worldent.anim[0] = gs.stats[STAT_WEAPONFRAME];
			worldent.anim[1] = gs.stats[STAT_WEAPONFRAME];
			worldent.animweight[1] = oldfrac;
			worldent.animweight[0] = newfrac;

			LerpEntAngles(worldent.matrix.data, gs.viewang.v, gs.viewang.v, 0, 360, 1);

			//FIXME: depthhack...

			gs.modelindexes[gs.stats[STAT_WEAPONMODEL]]->DrawEntity(&worldent, &scene);
		}

		for (unsigned int i = 0; i <= gs.numstaticents; i++, e++)
		{
			if (gs.staticents[i].modelindex >= countof(gs.modelindexes))
				continue;
			model_c *mdl = gs.modelindexes[gs.staticents[i].modelindex];
			if (!mdl)
				continue;
			worldent.skinnum = gs.staticents[i].skin;


			worldent.matrix[12] = gs.staticents[i].origin[0] / 8.0;
			worldent.matrix[13] = gs.staticents[i].origin[1] / 8.0;
			worldent.matrix[14] = gs.staticents[i].origin[2] / 8.0;
			if (i && !scene.frustum.Contains(vec3(worldent.matrix[12], worldent.matrix[13], worldent.matrix[14]), 64))
				continue;

			worldent.anim[0] = gs.staticents[i].frame;
			worldent.anim[1] = gs.staticents[i].frame;
			worldent.animweight[1] = oldfrac;
			worldent.animweight[0] = newfrac;
			worldent.animtime[0] = gs.servertime;
			worldent.animtime[1] = gs.servertime;

			LerpEntAngles(worldent.matrix.data, gs.staticents[i].angles, gs.staticents[i].angles, 1, 256, -1);

			mdl->DrawEntity(&worldent, &scene);
		}

		es.RenderEffects(&scene, frametime);

		return true;
	}
	void ProcessInput(vid_t *vid, float time)
	{
		if (time > inputtime && gs.entities_max > 0)
		{
			unsigned char buf_data[1024];
			netbuf buf(buf_data, sizeof(buf_data));

			buf.WriteByte(0x00);buf.WriteByte(0x10);
			buf.WriteByte(0x00);buf.WriteByte((gs.protocol == 666)?27:24);
			buf.WriteByte(outunreliablesequence>>24);buf.WriteByte(outunreliablesequence>>16);buf.WriteByte(outunreliablesequence>>8);buf.WriteByte(outunreliablesequence>>0);
			outunreliablesequence++;

			buf.WriteByte(clc_move);
			buf.WriteFloat(gs.servertime);
			if (gs.protocol == 666)
			{
				buf.WriteUShort(gs.viewang[0] * (65535/360.0));
				buf.WriteUShort(gs.viewang[1] * (65535/360.0));
				buf.WriteUShort(gs.viewang[2] * (65535/360.0));
			}
			else
			{
				buf.WriteByte(gs.viewang[0] * (255/360.0));
				buf.WriteByte(gs.viewang[1] * (255/360.0));
				buf.WriteByte(gs.viewang[2] * (255/360.0));
			}
			buf.WriteSShort((move_forward - move_back) * 400);
			buf.WriteSShort((move_right - move_left) * 400);
			buf.WriteSShort(0);
			buf.WriteByte(move_buttons&3);
			buf.WriteByte(move_impulse);
			move_impulse = 0;
			con->Send(&buf, &serveraddr);


			inputtime = time + 1/77.0;
		}
	}

	bool Run(vid_t *vid)
	{
		float time = vid->GetTime();
		vid->input = this;

		char fname[256];
		if (vid->GetDroppedFileName(fname+5, sizeof(fname)-5))
		{
			memcpy(fname, "demo:", 5);
			if (serveraddr.Resolve(fname, 26000))
			{
				con = NULL;
				resend = time;
				starttime = 0;
				framecount = 0;
				ClearState(failed);

				con = netcon::Net_Establish(&serveraddr);
				state = connecting;
			}
		}

		CheckPackets(time);
		ProcessInput(vid, time);
		RenderGameView(vid, time);
		return true;
	}
};

genclient *CL_Init(const char *server, float time)
{
	return new client(server, time);
};
