#include "common.h"
#include "filesystem.h"

#ifndef _WIN32
#include <dirent.h>
#else
#include <windows.h>
#include <direct.h>
#endif

#define MAX_OSPATH 256

file_c::~file_c()
{
	fclose((FILE*)data);
}
void file_c::seek(size_t offset)
{
#if _MSC_VER >= 1400
	_fseeki64((FILE*)data, offset+base, SEEK_SET);
#else
	fseek((FILE*)data, offset+base, SEEK_SET);
#endif
}
void file_c::seektoend(void)
{
#if _MSC_VER >= 1400
	_fseeki64((FILE*)data, base+length, SEEK_SET);
#else
	fseek((FILE*)data, base+length, SEEK_SET);
#endif
}
size_t file_c::tell(void)
{
#if _MSC_VER >= 8000
	return _ftelli64((FILE*)data) - base;
#else
	return ftell((FILE*)data) - base;
#endif
}
size_t file_c::read(void*buffer, size_t size)
{
	return fread(buffer, 1, size, (FILE*)data);
}
size_t file_c::write(const void*buffer, size_t size)
{
	return fwrite(buffer, 1, size, (FILE*)data);
}

static void CreatePathForFile(const char *filename)
{	//filename contains the path of a file, for which we try to ensure the path exists
	char path[1024];
	char *p = path;
	*p = '\0';
	while(*filename && p < path+sizeof(path)-1)
	{
		if (*filename == '/')
		{
			*p = 0;
			_mkdir(path);
		}
		*p++ = *filename++;
	}
}

file_c *file_c::Create(const char *filename, const char *accessmode)
{
	file_c *n;
	FILE *f;

	if (*accessmode == 'w')
		CreatePathForFile(filename);

	f = fopen(filename, accessmode);
	if (!f)
		return NULL;

	n = new file_c;
	n->data = f;
	n->base = 0;
	strcpy(n->name, filename);	//FIXME: buffer bounds

	fseek(f, 0, SEEK_END);
	n->length = ftell(f);
	fseek(f, 0, SEEK_SET);

	return n;
}

class fsstdio : filesys
{
private:
	char mPath[MAX_OSPATH];
public:
	virtual file_c *OpenRFile(const char *name, char *debugname, int debugnamelen)
	{
		char filename[MAX_OSPATH];
		snprintf(filename, sizeof(filename), "%s%s", mPath, name);

		if (debugname)
			snprintf(debugname, debugnamelen, "%s%s", mPath, name);

		return file_c::Create(filename, "rb");
	}

	virtual file_c *OpenWFile(const char *name, char *debugname, int debugnamelen)
	{
		char filename[MAX_OSPATH];
		snprintf(filename, sizeof(filename), "%s%s", mPath, name);

		if (debugname)
			snprintf(debugname, debugnamelen, "%s%s", mPath, name);

		return file_c::Create(filename, "wb");
	}

	static filesys *Create(const char *path)
	{
		fsstdio *fs;
		if (strncmp(path, "file://", 7))
			return NULL;

		fs = new fsstdio();
		strlcpy(fs->mPath, path+7, sizeof(fs->mPath));

		return fs;
	}
	virtual int SizeOfFile(const char *name)
	{
		char osname[MAX_OSPATH];
		int len;
		FILE *f;

		snprintf(osname, sizeof(osname)-1, "%s%s", mPath, name);
		f = fopen (osname, "rb");
		if (!f)
			return -1;
		fseek(f, 0, SEEK_END);
		len = ftell(f);
		fclose(f);

		return len;
	}
	virtual unsigned char *ReadFile(const char *name, unsigned int *retlen)
	{
		char osname[MAX_OSPATH];
		int len;
		FILE *f;
		unsigned char *buffer;

		snprintf(osname, sizeof(osname)-1, "%s%s", mPath, name);
		f = fopen (osname, "rb");
		if (!f)
			return NULL;
		fseek(f, 0, SEEK_END);
		len = ftell(f);
		fseek(f, 0, SEEK_SET);
		buffer = new unsigned char[len+1];	//ensure it's null terminated
		fread(buffer, len, 1, f);
		fclose(f);
		buffer[len] = '\0';
		if (retlen)
			*retlen = len;
		return buffer;
	}
	virtual void EnumerateFiles(const char *filematch, void (*callback) (filesys *fs, const char *filename, void *userdata), void *userdata)
	{
#ifndef _WIN32
		DIR *dir;
		struct dirent* entry;

		char *s;
		char apath[MAX_OSPATH];
		char file[MAX_OSPATH];

		sprintf(apath, "%s%s", mPath, filematch);
//		Com_Printf("apath = %s\n", apath);
		for (s = apath+strlen(apath)-1; s> apath; s--)
		{
			if (*s == '/')
				break;
		}
		if (strlen(mPath) >= strlen(apath))
		{	//indication of a security error.
			Sys_Error("EnumerateFiles: no path\n");
			dir = NULL;
		}
		else
		{
			*s = '\0';
			s++;
//			Com_Printf("apath = %s\n", apath);
			Com_Printf(PRINT_INFO, "Enumerate: %s\n", apath);
			dir = opendir(apath);
			if (!dir)	//urm, it doesn't exist!
			{
				Com_Printf("Couldn't enumerate files in %s\n", apath);
				return;
			}
		}

		while ((entry = readdir(dir)) != NULL)
		{
			if (wildcmp(s, entry->d_name))
			{
				if (*(apath+strlen(mPath)-1))
					snprintf(file, sizeof(file), "%s/%s", apath+strlen(mPath)-1, entry->d_name);
				else
					snprintf(file, sizeof(file), "%s%s", apath+strlen(mPath)-1, entry->d_name);
				callback(file, userdata);
			}
		}
		closedir(dir);
#else	//win32 code
		HANDLE r;
		WIN32_FIND_DATA fd;
		char apath[MAX_OSPATH];
		char file[MAX_OSPATH];
		char *s;

	//	strcpy(apath, match);
		sprintf(apath, "%s%s", mPath, filematch);
		for (s = apath+strlen(apath)-1; s> apath; s--)
		{
			if (*s == '/')
				break;
		}
		if (s == apath)
			*s = '\0';
		else
			s[1] = '\0';

		//this is what we ask windows for.
		sprintf(file, "%s*.*", apath);

		//we need to make apath contain the path in match but not gpath
		strcpy(apath, filematch);
		for (s = apath+strlen(apath)-1; s> apath; s--)
		{
			if (*s == '/')
				break;
		}
		*s = '\0';
		if (s != apath)
			strcat(apath, "/");
		Com_Printf(PRINT_INFO, "Enumerate: %s\n", file);
		r = FindFirstFile(file, &fd);
		if (r==(HANDLE)-1)
			return;
		do
		{
			if (*fd.cFileName == '.');	//don't ever find files with a name starting with '.'
			else if (!(fd.dwFileAttributes & 16))	//is a directory
			{
				if (wildcmp(filematch, fd.cFileName))
				{
					snprintf(file, sizeof(file), "%s%s", apath, fd.cFileName);
					callback(this, file, userdata);
				}
			}
			else
			{
				if (wildcmp(filematch, fd.cFileName))
				{
					snprintf(file, sizeof(file), "%s%s/", apath, fd.cFileName);
					callback(this, file, userdata);
				}
			}
		}
		while(FindNextFile(r, &fd));
		FindClose(r);
#endif
	}
};

fsdrv fsdrvstdio(".", "stdio", fsstdio::Create);

filesys *filesystems;

fsdrv *fsdrivers;
fsdrv::fsdrv(const char *p_extension, const char *p_name, filesys *(*p_openpath) (const char *path))
{
	mNext = fsdrivers;
	fsdrivers = this;
	OpenPath = p_openpath;
	OpenFile = NULL;
	extension = p_extension;
	name = p_name;
}
fsdrv::fsdrv(const char *p_extension, const char *p_name, filesys *(*p_openfile) (file *path, char *debugfname))
{
	mNext = fsdrivers;
	fsdrivers = this;
	OpenPath = NULL;
	OpenFile = p_openfile;
	extension = p_extension;
	name = p_name;
}

void EnumerateChildFileSystems(filesys *fs, const char *filename, void *user)
{
	fsdrv *drv = (fsdrv *)user;
	file *f;
	char debugname[256];
	debugname[0] = 0;

	f = fs->OpenRFile(filename, debugname, sizeof(debugname));

	if (!f)	//urm..
	{
		Com_Printf(PRINT_WARNING, "   Couldn't open %s\n", debugname);
		return;
	}
	Com_Printf(PRINT_INFO, "Opening %s\n", debugname);

	fs = drv->OpenFile(f, debugname);
	if (fs)
	{
		fs->mNext = filesystems;
		filesystems = fs;
	}
	else if (f)
	{
		Com_Printf(PRINT_WARNING, "   Failed\n");
		delete f;
	}
}

void FS_EnumerateFiles(const char *filematch, void (*callback) (filesys *fs, const char *filename, void *userdata), void *userdata)
{
	filesys *fs;
	for (fs = filesystems; fs; fs = fs->mNext)
	{
		fs->EnumerateFiles(filematch, callback, userdata);
	}
}

int FS_SizeOfFile(const char *filename)
{
	filesys *fs;
	unsigned int size;
	for (fs = filesystems; fs; fs = fs->mNext)
	{
		size = fs->SizeOfFile(filename);
		if (size >= 0)
			return size;
	}
	return -1;	//noone has it.
}

unsigned char *FS_ReadFile(const char *filename, unsigned int *size)
{
	filesys *fs;
	unsigned char *buffer;
	for (fs = filesystems; fs; fs = fs->mNext)
	{
		buffer = fs->ReadFile(filename, size);
		if (buffer)
			return buffer;
	}
	return NULL;	//noone has it.
}

file *FS_OpenRFile(const char *filename)
{
	filesys *fs;
	file *buffer;
	for (fs = filesystems; fs; fs = fs->mNext)
	{
		buffer = fs->OpenRFile(filename, NULL, 0);
		if (buffer)
			return buffer;
	}
	return NULL;	//noone has it.
}

file *FS_OpenWFile(const char *filename)
{
	filesys *fs;
	file *buffer;
	for (fs = filesystems; fs; fs = fs->mNext)
	{
		buffer = fs->OpenWFile(filename, NULL, 0);
		if (buffer)
			return buffer;
	}
	return NULL;	//noone has it.
}

void FS_AddGameDir(const char *basepath)
{
	fsdrv *rootdrv, *drv;
	filesys *rootsys;
	for (rootdrv = fsdrivers; rootdrv; rootdrv = rootdrv->mNext)
	{
		if (!strcmp(rootdrv->extension, "."))
			break;
	}
	if (!rootdrv)
		Sys_Error("No root file system driver");

	rootsys = rootdrv->OpenPath(basepath);
	if (!rootsys)	//argh!
	{
		Com_Printf(PRINT_WARNING, "unable to open %s\n", basepath);
		return;
	}

	Com_Printf(PRINT_INFO, "Added path %s\n", basepath);

	rootsys->mNext = filesystems;

	filesystems = rootsys;

	for (drv = fsdrivers; drv; drv = drv->mNext)
	{
		if (drv->OpenFile)
		{
			rootsys->EnumerateFiles(drv->extension, EnumerateChildFileSystems, drv);
		}
	}
}

int wildcmp(const char *wild, const char *string)	//it's handy for things.
{
	const char *cp=NULL, *mp=NULL;

	while ((*string) && (*wild != '*'))
	{
		if ((tolower(*wild) != tolower(*string)) && (*wild != '?'))
			return 0;
		wild++;
		string++;
	}

	while (*string)
	{
		if (*wild == '*')
		{
			if (!*++wild)
				return 1;
			mp = wild;
			cp = string+1;
		}
		else if ((tolower(*wild) == tolower(*string)) || (*wild == '?'))
		{
			wild++;
			string++;
		}
		else
		{
			wild = mp;
			string = cp++;
		}
	}

	while (*wild == '*')
		wild++;
	return !*wild;
}

static void FS_InitFileSystem(const char *basedir, const char *game)
{
	char dest[2048];
	if (!game)
		return;

	if (*basedir && basedir[strlen(basedir)-1] != '/')
		snprintf(dest, sizeof(dest), "file://%s/%s/", basedir, game);
	else
		snprintf(dest, sizeof(dest), "file://%s%s/", basedir, game);
	FS_AddGameDir(dest);	//so it can be placed in your quakedir for it to load q1 maps.
}

void FS_InitFileSystem(const char *basedir, const char *basegame, const char *subgame)
{
	FS_InitFileSystem(basedir, basegame);
	FS_InitFileSystem(basedir, subgame);
}

void FS_ShutdownAllFileSystems()	//but not drivers. :)
{
	filesys *fs;
	while(filesystems)
	{
		fs = filesystems->mNext;
		delete filesystems;
		filesystems = fs;
	}
}
