#include "common.h"
#include "renderer.h"
#include "filesystem.h"

#ifdef _WIN32
#include <windows.h>
#endif
#ifdef USEGL
#include <SDL.h>
#include "glrenderer.h"

GLuint GL_CompileShader(const char *vertexprogram, const char *fragmentprogram)
{
	GLhandleARB vp = pglCreateShader(GL_VERTEX_SHADER);
	GLhandleARB fp = pglCreateShader(GL_FRAGMENT_SHADER);
	pglShaderSource(vp, 1, &vertexprogram, NULL);
	pglShaderSource(fp, 1, &fragmentprogram, NULL);
	pglCompileShader(vp);
	pglCompileShader(fp);
	GLint compiled;
	pglGetShaderiv(vp, GL_COMPILE_STATUS, &compiled);
	if(!compiled)
	{
		GLsizei len;
		char buffer[8192];
		pglGetShaderInfoLog(vp, sizeof(buffer), &len, buffer);
		Sys_Error("Vertex program failed to compile:\n%s\n", buffer);
	}
	pglGetShaderiv(fp, GL_COMPILE_STATUS, &compiled);
	if(!compiled)
	{
		GLsizei len;
		char buffer[8192];
		pglGetShaderInfoLog(fp, sizeof(buffer), &len, buffer);
		Sys_Error("Fragment program failed to compile:\n%s\n", buffer);
	}

	GLuint shader = pglCreateProgram();
	pglAttachShader(shader, vp);
	pglAttachShader(shader, fp);
	pglLinkProgram(shader);

	pglGetProgramiv(shader, GL_LINK_STATUS, &compiled);
	if(!compiled)
	{
		GLsizei len;
		char buffer[8192];
		pglGetProgramInfoLog(shader, sizeof(buffer), &len, buffer);
		Sys_Error("Shader failed to link:\n%s\n", buffer);
	}

	return shader;
}
#endif

static model_c *loadedmodels;
void R_Model_Purge(void)
{
	model_c *mod;
	while(loadedmodels)
	{
		mod = loadedmodels;
		loadedmodels = loadedmodels->next;

		free((void*)mod->name);
		delete mod;
	}

	GL_PurgeModels();
}

model_c *R_Model_Get(const char *name)
{
	model_c *cm;
	modeltype *mt;
	unsigned char *filedata;
	unsigned int filelen;
	unsigned int ident;
	int l, s;

	for (cm = loadedmodels; cm; cm = cm->next)
	{
		if (!strcmp(cm->name, name))
			return cm;
	}
	cm = NULL;

	if (*name == '*')
		filedata = NULL;
	else
		filedata = FS_ReadFile(name, &filelen);
	if (filedata)
		ident = *(unsigned int *)filedata;
	else
		ident = 0;

	for (mt = modeltypes; mt; mt = mt->next)
	{
		if (!mt->identwords && mt->ident == ident)
		{
			cm = mt->LoadFunc(name, filedata, filelen);
			break;
		}
	}
	if (!mt && filedata)
	{
		for (s = 0; filedata[s] > 0 && filedata[s] <= ' '; s++)	//skip leading whitespace
			;
		for (l = s; filedata[l]>' '; l++)	//find the length of the first word.
			;

		for (mt = modeltypes; mt; mt = mt->next)
			if (mt->identwords && !strncmp(mt->identwords, (char*)filedata+s, l))
			{
				cm = mt->LoadFunc(name, filedata, filelen);
				break;
			}
	}
	if (!mt || !cm)
	{
		if (!mt && filedata)
			Com_Printf(PRINT_WARNING, "Unrecognised model type when opening %s\n", name);
		else
			Com_Printf(PRINT_WARNING, "Error while loading %s\n", name);

		for (mt = modeltypes; mt; mt = mt->next)
			if (!mt->ident && !mt->identwords)
			{
				cm = mt->LoadFunc(name, filedata, filelen);
				break;
			}
	}

	if (filedata)
		delete filedata;

	if (cm)
	{
		cm->name = strdup(name);
		cm->next = loadedmodels;
		loadedmodels = cm;
	}

	return cm;
}

double bench_pvs;
double bench_idxbuf;
double bench_draw;
double bench_sdl;
double bench_other;

modeltype *modeltypes;
vid_t *SDL_Vid_Create(void);
vid_t *Vid_Vulkan_Create(void);

vid_t *vid;

#ifdef _MSC_VER
#undef main
#elif defined(USEVK)
#define main gccsucks
#endif
extern "C"
int main(int argc, char* argv[])
{
	double time, last = 0, frametime;
	int frames;
	int oldmode=0;
	double start;
	bool grabmouse = true;
	int ow = 0, oh = 0;

	vec3 startorigin(0,0,0), startangle(0,0,0);

	const char *basedir = "";
	const char *basegame = "id1";
	const char *subgame = "";
	const char *map = NULL;
	const char *serveraddr = NULL;

	for (int i = 1; i < argc; )
	{
		char *arg = argv[i++];
		if (!strcmp(arg, "-basedir"))
			basedir = argv[i++];
		else if (!strcmp(arg, "-basegame"))
			basegame = argv[i++];
		else if (!strcmp(arg, "-game"))
			subgame = argv[i++];
		else if (!strcmp(arg, "-map") || !strcmp(arg, "+map"))
			map = argv[i++];
		else if (!strcmp(arg, "-server") || !strcmp(arg, "+connect"))
			serveraddr = argv[i++];
		else if (!strcmp(arg, "-setpos") || !strcmp(arg, "+setpos"))
		{
			startorigin = vec3(atof(argv[i+0]), atof(argv[i+1]), atof(argv[i+2]));
			i+=3;
		}
		else if (!strcmp(arg, "-setang") || !strcmp(arg, "+setang"))
		{
			startangle = vec3(atof(argv[i+0]), atof(argv[i+1]), atof(argv[i+2]));
			i+=3;
		}
		else
			Com_Printf(PRINT_WARNING, "Unknown argument: %s\n", arg);
	}
#ifdef USEVK
	vid = Vid_Vulkan_Create();
#else
	vid = SDL_Vid_Create();
#endif
	if (!vid->Startup("Crappy Thing"))
		return 0;

	FS_InitFileSystem(basedir, basegame, subgame);

	start = 0;
	frames = 0;
	vid->mousebuttons = 0;


	unsigned int whoa = 0;

	vid->mousex = vid->mousey = 0;

//	FS_ShutdownAllFileSystems();
	FS_AddGameDir("file://");

	last = vid->GetTime();

	class genclient *theclient = NULL;
	if (serveraddr && !theclient)
		theclient = CL_Init(serveraddr, last);
	if (map && !theclient)
		theclient = MapView_Init(map, startorigin, startangle);
	if (!theclient)
		theclient = CL_Init("demo:demo1.dem", last);
	if (!theclient)
		theclient = MapView_Init("start", startorigin, startangle);

	while(vid)
	{
		time = vid->GetTime();
		frametime = time - last;
		last = time;

		vid->PreFrame(grabmouse);	//check for inputs and stuff

		if (vid->mode == 9)
		{
			grabmouse ^= true;
			vid->mode = oldmode;
		}
		else if (vid->mode == 10)
		{
			static int wireframe;
			wireframe ^= true;
#ifndef USEVK
			pglPolygonMode(GL_FRONT_AND_BACK, wireframe?GL_LINE:GL_FILL);
#endif
			vid->mode = oldmode;
		}
		else if (vid->mode == 11)
		{
			if (grabmouse)
				grabmouse = false;
			else
			{
				vid->Shutdown();
				delete vid;
				vid = NULL;
				break;
			}
			vid->mode = oldmode;
		}

		if (vid->width != ow || vid->height != oh)
		{
#ifndef USEVK
			pglViewport(0, 0, vid->width, vid->height);
#endif
			ow = vid->width;
			oh = vid->height;
		}

		if (time - start > 1 || oldmode != vid->mode)
		{
			char buf[256];

			sprintf(buf, 
#ifdef USEVK
				"Vulkan: "
#endif
#ifdef USEGL
	#ifdef USEBINDLESS
				"BindlessGL: "
	#else
				"OpenGL4: "
	#endif
#endif
				"%s: %g fps.", GetBSPModeDesc(vid->mode), frames / (time - start));
			vid->SetTitle(buf);
//			Com_Printf(PRINT_INFO, "%g fps, %g ms/frame, "/*%g pvs, %g frustum(+pvs), */"%g the draw call, %g sdl+swap, %g other\n", frames / (time - start), 1000 * (time - start) / frames, /*1000*bench_pvs, 1000*bench_idxbuf, */1000*bench_draw, 1000*bench_sdl, 1000*bench_other);
			start = time;
			frames = 0;
			oldmode = vid->mode;
		}
		frames++;

		if (theclient)
			theclient->Run(vid);

		GL_FlushModels();

		double time2 = vid->GetTime();
		if (vid->PostFrame() || !theclient)
		{
			R_Model_Purge();
			vid->Shutdown();
			delete vid;
			vid = NULL;
			break;
		}
		double time3 = vid->GetTime();
		bench_sdl = time3-time2;
		bench_other = time3-time - bench_sdl - /*bench_pvs - bench_idxbuf - */bench_draw;
	}

	return 0;
}


void Com_Printf(printtype_e ptype, const char *fmt, ...)
{
	va_list v;
	if (ptype == PRINT_DEBUG)
		return;
	va_start(v, fmt);
	vprintf(fmt, v);
	va_end(v);
}

void Sys_Error(const char *fmt, ...)
{
	char buffer[8192];
	va_list va;
	va_start(va, fmt);
	vsnprintf(buffer, sizeof(buffer), fmt, va);
	va_end(va);

	printf("%s", buffer);

#ifdef _WIN32
	MessageBox(NULL, buffer, "My First VK Hack", 0);
#else
	#ifdef USEBINDLESS
		SDL_ShowSimpleMessageBox(0, "OneDraw Error", buffer, NULL);
	#else
		SDL_ShowSimpleMessageBox(0, "OneMultiDraw Error", buffer, NULL);
	#endif
#endif

	exit(0);
}

msec_c Sys_Milliseconds(void)
{
	return vid->GetTime()*1000;
}


#ifndef _MSC_VER
#ifdef _WIN32
#include <windows.h>
//fuck microsoft. fuck gcc. fuck sdl. fuck the world.
int WINAPI WinMain (HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine, int nCmdShow)
{
	int argc;
	int i, l, c;
	LPWSTR *argvw;
	char **argv;
	char utf8arg[1024];
	argvw = CommandLineToArgvW(GetCommandLineW(), &argc);
	argv = (char**)malloc(argc * sizeof(char*));
	for (i = 0; i < argc; i++)
	{
		for(l = 0, c = 0; argvw[i][l]; l++)
		{
			utf8arg[c++] = argvw[i][l];
//			c += utf8_encode(utf8arg+c, argvw[i][l], sizeof(utf8arg) - c-1);
		}
		utf8arg[c] = 0;
		argv[i] = _strdup(utf8arg);
	}
	return main(argc, argv);
}
#endif
#endif