#include "common.h"
#include "filesystem.h"

//provides multiple handles with seperate offsets within a single larger file handle
class reffile : public file
{
private:
	struct reffile_ref
	{
		file *handle;
		unsigned int refcount;
		size_t offset;	//cache for hopeful speedup
	} *refs;
public:
	size_t offset;
	size_t base;
	size_t length;
	~reffile()
	{
		if (--refs->refcount)
		{
			delete refs->handle;	//finally close the parent file
			delete refs;
		}
	}
	reffile(reffile *prev, unsigned int suboffset, size_t sublength)
	{
		refs = prev->refs;
		refs->refcount++;
		base = suboffset;
		length = sublength;
		offset = 0;
	}
	reffile(file *basefile)
	{
		refs = new reffile_ref;
		refs->handle = basefile;
		refs->refcount = 1;
		offset = basefile->tell();
		basefile->seektoend();
		refs->offset = length = basefile->tell();
		base = 0;
	}

	virtual void seek(size_t newoffset)
	{
		offset = newoffset;
		if (offset > length)
			offset = length;
	};
	virtual void seektoend(void)
	{
		offset = length;
	};
	virtual size_t tell(void)
	{
		return offset;
	};
	virtual size_t read(void*buffer, size_t size)
	{
		if (refs->offset != offset+base)
			refs->handle->seek(offset+base);

		if (offset+size > length)	//don't read beyond the file
			size = length - offset;

		size_t res = refs->handle->read(buffer, size);
		offset += res;
		refs->offset = offset+base;
		return res;
	};
	virtual size_t write(const void*buffer, size_t size)
	{
		return 0;
	};
};

class fspak : filesys
{
private:
	unsigned int m_filecount;
	typedef struct {
		char name[56];
		int offset;
		int length;
	} pakfile;
	pakfile *m_filelist;
	reffile *m_handle;

public:
	char mPath[512];
	static filesys *OpenFile(file *srcfile, char *debugfname)
	{
		unsigned int i;
		unsigned int offset;

		char ident[4];
		srcfile->seek(0);
		if (srcfile->read(ident, 4)==4 && !strncmp(ident, "PACK", 4))
		{
			fspak *pak = new fspak;

			snprintf(pak->mPath, sizeof(pak->mPath), "%s", debugfname);

			srcfile->read(&offset, 4);
			offset = LittleInt(offset);

			srcfile->read(&pak->m_filecount, 4);
			pak->m_filecount = LittleInt(pak->m_filecount);
			pak->m_filecount /= sizeof(pakfile);

			pak->m_handle = new reffile(srcfile);
			pak->m_filelist = new pakfile[pak->m_filecount];
			pak->m_handle->seek(offset);
			if (sizeof(pakfile)*pak->m_filecount == pak->m_handle->read(pak->m_filelist, sizeof(pakfile)*pak->m_filecount))
			{
				for (i = 0; i < pak->m_filecount; i++)
				{
					pak->m_filelist[i].length = LittleInt(pak->m_filelist[i].length);
					pak->m_filelist[i].offset = LittleInt(pak->m_filelist[i].offset);
				}
				return pak;
			}
			delete[] pak->m_filelist;
			delete pak;
		}
		return NULL;
	}
	virtual file *OpenRFile(const char *name, char *debugname, int debugnamelen)	//returns a handle to a file.
	{
		unsigned int i;
		for (i = 0; i < m_filecount; i++)
		{
			if (!strcmp(name, m_filelist[i].name))
			{
				if (debugname)
					snprintf(debugname, debugnamelen, "%s/%s", mPath, name);

				return new reffile(m_handle, m_filelist[i].offset, m_filelist[i].length);
			}
		}
		return NULL;
	}
	virtual file_c *OpenWFile(const char *name, char *debugname, int debugnamelen)	//returns a handle to a file.
	{
		return NULL;
	}
	virtual unsigned char *ReadFile(const char *name, unsigned int *retlen)
	{
		unsigned char *retbuffer;
		unsigned int i;
		for (i = 0; i < m_filecount; i++)
		{
			if (!strcmp(name, m_filelist[i].name))
			{
				if (retlen)
					*retlen = m_filelist[i].length;

				retbuffer = new unsigned char[m_filelist[i].length+1];
				retbuffer[m_filelist[i].length] = '\0';
				m_handle->seek(m_filelist[i].offset);
				m_handle->read(retbuffer, m_filelist[i].length);

				return retbuffer;
			}
		}
		return NULL;
	}
	virtual int SizeOfFile(const char *name)
	{
		unsigned int i;
		for (i = 0; i < m_filecount; i++)
		{
			if (!strcmp(name, m_filelist[i].name))
				return m_filelist[i].length;
		}
		return -1;
	}
	virtual void EnumerateFiles(const char *filematch, void (*callback) (filesys *fs, const char *filename, void *userdata), void *userdata)
	{
		unsigned int i;
		for (i = 0; i < m_filecount; i++)
		{
			if (wildcmp(filematch, m_filelist[i].name))
				callback(this, m_filelist[i].name, userdata);
		}
	}
};

fsdrv fsdrvpak("*.pak", "Quake Pack File", fspak::OpenFile);

