#include "common.h"
#ifdef USEVK
#include "vkrenderer.h"
#include "filesystem.h"

//#define DEBUGLAYERS
//#define DEBUG_LAYER_DRAWSTATE	//this is crashing on me. no idea why.

#ifndef UINT64_MAX
#define UINT64_MAX (((1ull<<63)-1)*2+1)
#endif

#pragma comment(lib, "winmm")

VkInstance		vkinstance;
VkDevice		vkdevice;
VkSurfaceKHR	vksurface;
uint32_t		vkcurrentbuffer;
VkSwapchainKHR	vkswapchain;
uint32_t		vkqueueidx[2];	//render, present.
VkQueue			vkqueuerender;
VkQueue			vkqueuepresent;
struct vkbackbuffer_s
{
	VkImage			image;
	VkImageView		view;
	VkFramebuffer	framebuffer;

	//the system shouldn't return a buffer back to us before its been submitted to the presentation queue.
	//so its safe to just reuse this per-buffer stuff.
	//its possible that acquire can give us a buffer that we don't own yet, but this should still be safe as we won't have scribbled to it by then
	VkFence fence;
	VkSemaphore sem_vsync;
	VkSemaphore sem_presentable;
	VkCommandBuffer cmdbuf;
} *vkbackbuffer, *vkframe;
uint32_t vkbackbuffer_count;
vk_image_t		vkdepth;
VkCommandPool	vkcmdpool;
VkRenderPass	vkrenderpass;
VkPhysicalDevice vkgpu;
VkCommandBuffer vkmaincmd;
VkPhysicalDeviceMemoryProperties vk_memory_properties;
VkDebugReportCallbackEXT vkdebugcallback = VK_NULL_HANDLE;

#define VKFunc(n) PFN_vk##n pvk##n;
VKFunc(CreateDebugReportCallbackEXT)
VKFunc(DestroyDebugReportCallbackEXT)
#ifdef VK_NO_PROTOTYPES
VKFuncs
#else
#pragma comment(lib, "vulkan-1")
#endif
#undef VKFunc

static void set_image_layout(VkCommandBuffer cmd, VkImage image, VkImageAspectFlags aspectMask, VkImageLayout old_image_layout, VkImageLayout new_image_layout)
{
	//images have weird layout representations.
	//we need to use a side-effect of memory barriers in order to convert from one layout to another, so that we can actually use the image.
	VkImageMemoryBarrier imgbarrier = {};
	imgbarrier.sType = VK_STRUCTURE_TYPE_IMAGE_MEMORY_BARRIER;
	imgbarrier.pNext = NULL;
	imgbarrier.srcAccessMask = 0;
	imgbarrier.dstAccessMask = 0;
	imgbarrier.oldLayout = old_image_layout;
	imgbarrier.newLayout = new_image_layout;
	imgbarrier.image = image;
	imgbarrier.subresourceRange.aspectMask = aspectMask;
	imgbarrier.subresourceRange.baseMipLevel = 0;
	imgbarrier.subresourceRange.levelCount = 1;
	imgbarrier.subresourceRange.baseArrayLayer = 0;
	imgbarrier.subresourceRange.layerCount = 1;

	if (new_image_layout == VK_IMAGE_LAYOUT_TRANSFER_DST_OPTIMAL)	/* Make sure anything that was copying from this image has completed */
		imgbarrier.dstAccessMask = VK_ACCESS_TRANSFER_READ_BIT;
	else if (new_image_layout == VK_IMAGE_LAYOUT_COLOR_ATTACHMENT_OPTIMAL)
		imgbarrier.dstAccessMask = VK_ACCESS_COLOR_ATTACHMENT_WRITE_BIT;
	else if (new_image_layout == VK_IMAGE_LAYOUT_DEPTH_STENCIL_ATTACHMENT_OPTIMAL)
		imgbarrier.dstAccessMask = VK_ACCESS_DEPTH_STENCIL_ATTACHMENT_WRITE_BIT;
	else if (new_image_layout == VK_IMAGE_LAYOUT_SHADER_READ_ONLY_OPTIMAL) /* Make sure any Copy or CPU writes to image are flushed */
		imgbarrier.dstAccessMask = VK_ACCESS_SHADER_READ_BIT | VK_ACCESS_INPUT_ATTACHMENT_READ_BIT;

	pvkCmdPipelineBarrier(cmd, VK_PIPELINE_STAGE_TOP_OF_PIPE_BIT, VK_PIPELINE_STAGE_TOP_OF_PIPE_BIT, 0, 0, NULL, 0, NULL, 1, &imgbarrier);
}

//typeBits is some vulkan requirement thing (like textures must be device-local).
//requirements_mask are things that the engine may require (like host-visible).
//note that there is absolutely no guarentee that hardware requirements will match what the host needs.
//thus you may need to use staging.
static uint32_t memory_type_from_properties_try(uint32_t typeBits, VkFlags requirements_mask)
{
	for (uint32_t i = 0; i < 32; i++)
	{
		if ((typeBits & 1) == 1)
		{
			if ((vk_memory_properties.memoryTypes[i].propertyFlags & requirements_mask) == requirements_mask)
				return i;
		}
		typeBits >>= 1;
	}
	return ~0u;
}
static uint32_t memory_type_from_properties_require(uint32_t typeBits, VkFlags requirements_mask)
{
	uint32_t ret = memory_type_from_properties_try(typeBits, requirements_mask);
	if (ret == ~0)
		VkAssert(VK_ERROR_OUT_OF_DEVICE_MEMORY);
	return ret;
}



class vkvid_t : public vid_t
{
	HWND wnd;
	bool closereq;
	bool resized;
	int buttonbits;

	VkFormat depthformat;

	static LRESULT CALLBACK wproc(HWND wnd, UINT msg, WPARAM wparam, LPARAM lparam)
	{
		vkvid_t *v = (vkvid_t*)(LONG_PTR)GetWindowLongPtr(wnd, GWLP_USERDATA);
		switch(msg)
		{
		case WM_CREATE:
			{
				CREATESTRUCT *cs = (CREATESTRUCT*)(LONG_PTR)lparam;
				SetWindowLongPtr(wnd, GWLP_USERDATA, (LONG_PTR)cs->lpCreateParams);
			}
			return false;
		case WM_CLOSE:
			v->closereq = true;
			return false;
		case WM_PAINT:
			{
				PAINTSTRUCT ps;
				HDC hdc=BeginPaint(wnd,(LPPAINTSTRUCT)&ps);

				EndPaint(wnd,(LPPAINTSTRUCT)&ps);
				return TRUE;
			}
			break;
		case WM_SYSKEYDOWN:
		case WM_SYSKEYUP:
		case WM_KEYDOWN:
		case WM_KEYUP:
			if (v->input)
			{
				bool down = msg == WM_KEYDOWN || msg == WM_SYSKEYDOWN;
				if (wparam >= 'A' && wparam <= 'Z')
					v->input->KeyEvent(down, wparam-'A'+'a');
				else if (wparam == VK_ESCAPE)
					v->input->KeyEvent(down, 27);
				else if (wparam >= '0' && wparam <= '9')
					v->input->KeyEvent(down, wparam);
				else if (wparam >= VK_F1 && wparam <= '9')
					v->input->KeyEvent(down, wparam);
				else if (wparam >= VK_F1 && wparam <= VK_F12 && down)
					v->mode = wparam-VK_F1;
				else
					return TRUE;
			}
			return FALSE;
		case WM_LBUTTONUP:
		case WM_LBUTTONDOWN:
		case WM_RBUTTONUP:
		case WM_RBUTTONDOWN:
		case WM_MBUTTONUP:
		case WM_MBUTTONDOWN:
			if (v->input)
			{
				if ((v->buttonbits ^ wparam) & MK_LBUTTON)
					v->input->ButtonEvent(!!(wparam&MK_LBUTTON), 1);
				if ((v->buttonbits ^ wparam) & MK_RBUTTON)
					v->input->ButtonEvent(!!(wparam&MK_RBUTTON), 2);
				if ((v->buttonbits ^ wparam) & MK_MBUTTON)
					v->input->ButtonEvent(!!(wparam&MK_MBUTTON), 3);
			}
			v->buttonbits = wparam;
			return TRUE;
		case WM_SIZE:
			v->resized = true;
			return FALSE;
		default:
			return DefWindowProc(wnd,msg,wparam,lparam);
		}
	}
	virtual bool GetDroppedFileName(char *buf, size_t bufsize)
	{
		return false;
	}
	virtual void SetTitle(const char *newtitle)
	{
		SetWindowTextA(wnd, newtitle);
	}
	static VkBool32 VKAPI_PTR mydebugreportcallback(
				VkDebugReportFlagsEXT                       flags,
				VkDebugReportObjectTypeEXT                  objectType,
				uint64_t                                    object,
				size_t                                      location,
				int32_t                                     messageCode,
				const char*                                 pLayerPrefix,
				const char*                                 pMessage,
				void*                                       pUserData)
	{
		vkvid_t *ctx = (vkvid_t*)pUserData;
		if (flags & VK_DEBUG_REPORT_ERROR_BIT_EXT)
			Com_Printf(PRINT_ERROR, "%s: %s\n", pLayerPrefix, pMessage);
		else if (flags & VK_DEBUG_REPORT_WARNING_BIT_EXT)
			Com_Printf(PRINT_WARNING, "%s: %s\n", pLayerPrefix, pMessage);
		else if (flags & VK_DEBUG_REPORT_DEBUG_BIT_EXT)
			Com_Printf(PRINT_INFO, "%s: %s\n", pLayerPrefix, pMessage);
		else if (flags & VK_DEBUG_REPORT_INFORMATION_BIT_EXT)
			Com_Printf(PRINT_DEBUG, "%s: %s\n", pLayerPrefix, pMessage);
		else if (flags & VK_DEBUG_REPORT_PERFORMANCE_WARNING_BIT_EXT)
			Com_Printf(PRINT_INFO, "%s: %s\n", pLayerPrefix, pMessage);    
		else
			Com_Printf(PRINT_INFO, "%s: %s\n", pLayerPrefix, pMessage);
		return false;
	}
	bool StartupVulkan(void)
	{
		VkResult err;

#ifdef VK_NO_PROTOTYPES
#define VKFunc(n) pvk##n = (PFN_vk##n)pvkGetInstanceProcAddr(VK_NULL_HANDLE, "vk"#n);
		VKInstFuncs
#undef VKFunc
#endif

		{
			uint32_t propcount = 0;
			pvkEnumerateInstanceLayerProperties(&propcount, NULL);
			VkLayerProperties *props = new VkLayerProperties[propcount];
			pvkEnumerateInstanceLayerProperties(&propcount, props);
			for (uint32_t i = 0; i < propcount; i++)
				Com_Printf(PRINT_INFO, "layer %s (%s)\n", props[i].layerName, props[i].description);
			delete[] props;
		}

		{
			uint32_t propcount = 0;
			pvkEnumerateInstanceExtensionProperties(NULL, &propcount, NULL);
			VkExtensionProperties *props = new VkExtensionProperties[propcount];
			pvkEnumerateInstanceExtensionProperties(NULL, &propcount, props);
			for (uint32_t i = 0; i < propcount; i++)
				Com_Printf(PRINT_INFO, "extension %s\n", props[i].extensionName);
			delete[] props;
		}

#define ENGINENAME "MyCrappyEngine"
#define ENGINEVERSION 0
		VkApplicationInfo app={};
		app.sType = VK_STRUCTURE_TYPE_APPLICATION_INFO;
		app.pNext = NULL;
		app.pApplicationName = ENGINENAME;
		app.applicationVersion = ENGINEVERSION;
		app.pEngineName = ENGINENAME;
		app.engineVersion = ENGINEVERSION;
		app.apiVersion = VK_API_VERSION;

		const char *layers[] = {
#ifndef DEBUGLAYERS
			NULL
#else
//	"VK_LAYER_LUNARG_api_dump",
"VK_LAYER_LUNARG_device_limits",
#ifdef DEBUG_LAYER_DRAWSTATE
"VK_LAYER_LUNARG_draw_state",
#endif
"VK_LAYER_LUNARG_image",
"VK_LAYER_LUNARG_mem_tracker",
"VK_LAYER_LUNARG_object_tracker",
"VK_LAYER_LUNARG_param_checker",
"VK_LAYER_LUNARG_screenshot",
"VK_LAYER_LUNARG_swapchain",
"VK_LAYER_LUNARG_threading",
"VK_LAYER_GOOGLE_unique_objects",
//"VK_LAYER_LUNARG_vktrace",
//"VK_LAYER_RENDERDOC_Capture"	//renderdoc gets activated as needed
#endif
	};
		const char *extensions[] = {VK_KHR_WIN32_SURFACE_EXTENSION_NAME,
									VK_KHR_SURFACE_EXTENSION_NAME,
#ifdef _DEBUG
									VK_EXT_DEBUG_REPORT_EXTENSION_NAME
#endif
		};
		VkInstanceCreateInfo inst_info = {};
		inst_info.sType = VK_STRUCTURE_TYPE_INSTANCE_CREATE_INFO;
		inst_info.pApplicationInfo = &app;
		inst_info.enabledLayerCount = layers[0]?countof(layers):0;
		inst_info.ppEnabledLayerNames = layers;
		inst_info.enabledExtensionCount = countof(extensions);
		inst_info.ppEnabledExtensionNames = extensions;

		err = pvkCreateInstance(&inst_info, vkallocationcb, &vkinstance);
		switch(err)
		{
		case VK_ERROR_INCOMPATIBLE_DRIVER:
			Com_Printf(PRINT_ERROR, "VK_ERROR_INCOMPATIBLE_DRIVER: please install an appropriate vulkan driver\n", err);
			return false;
		case VK_ERROR_EXTENSION_NOT_PRESENT:
			Com_Printf(PRINT_ERROR, "VK_ERROR_EXTENSION_NOT_PRESENT: something on a system level is probably misconfigured\n", err);
			return false;
		default:
			Com_Printf(PRINT_ERROR, "Unknown vulkan instance creation error: %x\n", err);
			return false;
		case VK_SUCCESS:
			break;
		}

#ifdef VK_NO_PROTOTYPES
		pvkGetInstanceProcAddr = (PFN_vkGetInstanceProcAddr)pvkGetInstanceProcAddr(vkinstance, "vkGetInstanceProcAddr");
#define VKFunc(n) pvk##n = (PFN_vk##n)pvkGetInstanceProcAddr(vkinstance, "vk"#n);
		VKInst2Funcs
#undef VKFunc
#endif

		{
			pvkCreateDebugReportCallbackEXT = (PFN_vkCreateDebugReportCallbackEXT)pvkGetInstanceProcAddr(vkinstance, "vkCreateDebugReportCallbackEXT");
			pvkDestroyDebugReportCallbackEXT = (PFN_vkDestroyDebugReportCallbackEXT)pvkGetInstanceProcAddr(vkinstance, "vkDestroyDebugReportCallbackEXT");
			if (pvkCreateDebugReportCallbackEXT && pvkDestroyDebugReportCallbackEXT)
			{
				VkDebugReportCallbackCreateInfoEXT dbgCreateInfo = {};
				dbgCreateInfo.sType = VK_STRUCTURE_TYPE_DEBUG_REPORT_CREATE_INFO_EXT;
				dbgCreateInfo.pfnCallback = mydebugreportcallback;
				dbgCreateInfo.pUserData = this;
				dbgCreateInfo.flags =	VK_DEBUG_REPORT_ERROR_BIT_EXT |
										VK_DEBUG_REPORT_WARNING_BIT_EXT	|
										VK_DEBUG_REPORT_INFORMATION_BIT_EXT	|
										VK_DEBUG_REPORT_PERFORMANCE_WARNING_BIT_EXT |
										VK_DEBUG_REPORT_DEBUG_BIT_EXT;
				pvkCreateDebugReportCallbackEXT(vkinstance, &dbgCreateInfo, vkallocationcb, &vkdebugcallback);
			}
		}

		{
			uint32_t gpucount = 0;
			pvkEnumeratePhysicalDevices(vkinstance, &gpucount, NULL);
			if (!gpucount)
			{
				Com_Printf(PRINT_ERROR, "vulkan: no devices known!\n");
				return false;
			}
			VkPhysicalDevice *devs = new VkPhysicalDevice[gpucount];
			pvkEnumeratePhysicalDevices(vkinstance, &gpucount, devs);
			for (uint32_t i = 0; i < gpucount; i++)
			{
				VkPhysicalDeviceProperties props;
				pvkGetPhysicalDeviceProperties(devs[i], &props);
				const char *gputype;
				if (!vkgpu)
					vkgpu = devs[i];
				switch(props.deviceType)
				{
				default:
				case VK_PHYSICAL_DEVICE_TYPE_OTHER:
					gputype = "other";
					break;
				case VK_PHYSICAL_DEVICE_TYPE_INTEGRATED_GPU:
					gputype = "integrated";
					break;
				case VK_PHYSICAL_DEVICE_TYPE_DISCRETE_GPU:
					gputype = "discrete";
					break;
				case VK_PHYSICAL_DEVICE_TYPE_VIRTUAL_GPU:
					gputype = "virtual";
					break;
				case VK_PHYSICAL_DEVICE_TYPE_CPU:
					gputype = "CPU";
					break;
				}
				Com_Printf(PRINT_INFO, "GPU%i %s (%s)\n", i, props.deviceName, gputype);
			}
			delete[] devs;

			if (!vkgpu)
			{
				Com_Printf(PRINT_ERROR, "vulkan: unable to pick a usable device\n");
				return false;
			}
		}

		VkWin32SurfaceCreateInfoKHR createInfo = {};
		createInfo.sType = VK_STRUCTURE_TYPE_WIN32_SURFACE_CREATE_INFO_KHR;
		createInfo.flags = 0;
		createInfo.hinstance = GetModuleHandle(NULL);
		createInfo.hwnd = wnd;

		err = pvkCreateWin32SurfaceKHR(vkinstance, &createInfo, NULL, &vksurface);
		switch(err)
		{
		default:
			Com_Printf(PRINT_ERROR, "Unknown vulkan device creation error: %x\n", err);
			return false;
		case VK_SUCCESS:
			break;
		}



		/*now we know what we're trying to create, go ahead and do the actual creation*/

		{
		    /* Call with NULL data to get count */
			uint32_t queue_count;
			pvkGetPhysicalDeviceQueueFamilyProperties(vkgpu, &queue_count, NULL);
			VkQueueFamilyProperties *queueprops = new VkQueueFamilyProperties[queue_count];
			pvkGetPhysicalDeviceQueueFamilyProperties(vkgpu, &queue_count, queueprops);

			vkqueueidx[0] = ~0u;
			vkqueueidx[1] = ~0u;
			for (uint32_t i = 0; i < queue_count; i++)
			{
				VkBool32 supportsPresent;
				pvkGetPhysicalDeviceSurfaceSupportKHR(vkgpu, i, vksurface, &supportsPresent);

				if ((queueprops[i].queueFlags & VK_QUEUE_GRAPHICS_BIT) && supportsPresent)
				{
					vkqueueidx[0] = i;
					vkqueueidx[1] = i;
					break;
				}
				else if (vkqueueidx[0] == ~0u && (queueprops[i].queueFlags & VK_QUEUE_GRAPHICS_BIT))
					vkqueueidx[0] = i;
				else if (vkqueueidx[1] == ~0u && supportsPresent)
					vkqueueidx[1] = i;
			}

			delete[] queueprops;

			if (vkqueueidx[0] == ~0u || vkqueueidx[1] == ~0u)
			{
				Com_Printf(PRINT_ERROR, "unable to find suitable queues\n", err);
				return false;
			}
		}



		const char *devextensions[] = {	VK_KHR_SWAPCHAIN_EXTENSION_NAME};
		float queue_priorities[1] = {1.0};
		VkDeviceQueueCreateInfo queueinf[2] = {};
		queueinf[0].sType = VK_STRUCTURE_TYPE_DEVICE_QUEUE_CREATE_INFO;
		queueinf[0].pNext = NULL;
		queueinf[0].queueFamilyIndex = vkqueueidx[0];
		queueinf[0].queueCount = countof(queue_priorities);
		queueinf[0].pQueuePriorities = queue_priorities;
		queueinf[1].sType = VK_STRUCTURE_TYPE_DEVICE_QUEUE_CREATE_INFO;
		queueinf[1].pNext = NULL;
		queueinf[1].queueFamilyIndex = vkqueueidx[1];
		queueinf[1].queueCount = countof(queue_priorities);
		queueinf[1].pQueuePriorities = queue_priorities;

		VkDeviceCreateInfo devinf = {};
        devinf.sType = VK_STRUCTURE_TYPE_DEVICE_CREATE_INFO;
		devinf.queueCreateInfoCount = (vkqueueidx[0]==vkqueueidx[0])?1:2;
        devinf.pQueueCreateInfos = queueinf;
		devinf.enabledLayerCount = layers[0]?countof(layers):0;
		devinf.ppEnabledLayerNames = layers;
        devinf.enabledExtensionCount = countof(devextensions);
		devinf.ppEnabledExtensionNames = devextensions;
        devinf.pEnabledFeatures = NULL;

		err = pvkCreateDevice(vkgpu, &devinf, NULL, &vkdevice);
		switch(err)
		{
		case VK_ERROR_INCOMPATIBLE_DRIVER:
			Com_Printf(PRINT_ERROR, "VK_ERROR_INCOMPATIBLE_DRIVER: please install an appropriate vulkan driver\n", err);
			return false;
		case VK_ERROR_EXTENSION_NOT_PRESENT:
			Com_Printf(PRINT_ERROR, "VK_ERROR_EXTENSION_NOT_PRESENT: something on a system level is probably misconfigured\n", err);
			return false;
		default:
			Com_Printf(PRINT_ERROR, "Unknown vulkan device creation error: %x\n", err);
			return false;
		case VK_SUCCESS:
			break;
		}

#ifdef VK_NO_PROTOTYPES
		pvkGetDeviceProcAddr = (PFN_vkGetDeviceProcAddr)pvkGetInstanceProcAddr(vkinstance, "vkGetDeviceProcAddr");
#define VKFunc(n) pvk##n = (PFN_vk##n)pvkGetDeviceProcAddr(vkdevice, "vk"#n);
		VKDevFuncs
#undef VKFunc
#endif

		pvkGetPhysicalDeviceMemoryProperties(vkgpu, &vk_memory_properties);


		VkCommandPoolCreateInfo cpci = {};
		cpci.sType = VK_STRUCTURE_TYPE_COMMAND_POOL_CREATE_INFO;
		cpci.queueFamilyIndex = vkqueueidx[0];
		cpci.flags = VK_COMMAND_POOL_CREATE_RESET_COMMAND_BUFFER_BIT|VK_COMMAND_POOL_CREATE_TRANSIENT_BIT;
		VkAssert(pvkCreateCommandPool(vkdevice, &cpci, vkallocationcb, &vkcmdpool));

		pvkGetDeviceQueue(vkdevice, vkqueueidx[0], 0, &vkqueuerender);
		pvkGetDeviceQueue(vkdevice, vkqueueidx[1], 0, &vkqueuepresent);

		SetupRenderPass();
		return true;
	}
	void DestroySwapChain(void)
	{
		for (uint32_t i = 0; i < vkbackbuffer_count; i++)
		{
			VkAssert(pvkWaitForFences(vkdevice, 1, &vkbackbuffer[i].fence, VK_FALSE, UINT64_MAX));

			pvkDestroyFramebuffer(vkdevice, vkbackbuffer[i].framebuffer, vkallocationcb);
			pvkDestroyImageView(vkdevice, vkbackbuffer[i].view, vkallocationcb);
		}
		DestroyTexture(&vkdepth);
		pvkDestroySwapchainKHR(vkdevice, vkswapchain, vkallocationcb);
	}
	bool StartupVulkanSwapChain(void)
	{
		uint32_t fmtcount;
		VkAssert(pvkGetPhysicalDeviceSurfaceFormatsKHR(vkgpu, vksurface, &fmtcount, NULL));
		VkSurfaceFormatKHR *surffmts = new VkSurfaceFormatKHR[fmtcount];
		VkAssert(pvkGetPhysicalDeviceSurfaceFormatsKHR(vkgpu, vksurface, &fmtcount, surffmts));

		uint32_t presentmodes;
		VkAssert(pvkGetPhysicalDeviceSurfacePresentModesKHR(vkgpu, vksurface, &presentmodes, NULL));
		VkPresentModeKHR *presentmode = new VkPresentModeKHR[presentmodes];
		VkAssert(pvkGetPhysicalDeviceSurfacePresentModesKHR(vkgpu, vksurface, &presentmodes, presentmode));

		VkSurfaceCapabilitiesKHR surfcaps;
		pvkGetPhysicalDeviceSurfaceCapabilitiesKHR(vkgpu, vksurface, &surfcaps);

		VkSwapchainCreateInfoKHR swapinfo = {};
        swapinfo.sType = VK_STRUCTURE_TYPE_SWAPCHAIN_CREATE_INFO_KHR;
        swapinfo.surface = vksurface;
        swapinfo.minImageCount = surfcaps.minImageCount+1;
		if (swapinfo.minImageCount > surfcaps.maxImageCount)
			swapinfo.minImageCount = surfcaps.maxImageCount;
		swapinfo.imageFormat = (!fmtcount || surffmts->format == VK_FORMAT_UNDEFINED)?VK_FORMAT_B8G8R8A8_UNORM:surffmts->format;
        swapinfo.imageColorSpace = (!fmtcount || surffmts->format == VK_FORMAT_UNDEFINED)?VK_COLORSPACE_SRGB_NONLINEAR_KHR:surffmts->colorSpace;
        swapinfo.imageExtent.width = width;
		swapinfo.imageExtent.height = height;
        swapinfo.imageUsage = VK_IMAGE_USAGE_COLOR_ATTACHMENT_BIT;
        swapinfo.preTransform = surfcaps.currentTransform;//VK_SURFACE_TRANSFORM_IDENTITY_BIT_KHR;
        swapinfo.compositeAlpha = VK_COMPOSITE_ALPHA_INHERIT_BIT_KHR;
        swapinfo.imageArrayLayers = 1;
        swapinfo.imageSharingMode = VK_SHARING_MODE_EXCLUSIVE;
        swapinfo.queueFamilyIndexCount = 0;
        swapinfo.pQueueFamilyIndices = NULL;
        swapinfo.presentMode = VK_PRESENT_MODE_FIFO_KHR;	//supposed to be guarenteed support.
        swapinfo.oldSwapchain = vkswapchain;
        swapinfo.clipped = VK_TRUE;	//allow fragment shaders to be skipped on parts that are obscured by another window.

		bool wantvsync = false;
		for (uint32_t i = 0, curpri = 0; i < presentmodes; i++)
		{
			uint32_t priority = 0;
			switch(presentmode[i])
			{
			default://ignore it.
				break;
			case VK_PRESENT_MODE_IMMEDIATE_KHR:
				priority = (wantvsync?0:2) + 1;
				break;
			case VK_PRESENT_MODE_MAILBOX_KHR:
				priority = (wantvsync?0:2) + 2;
				break;
			case VK_PRESENT_MODE_FIFO_KHR:
				priority = (wantvsync?2:0) + 1;
				break;
			case VK_PRESENT_MODE_FIFO_RELAXED_KHR:
				priority = (wantvsync?2:0) + 2;
				break;
			}
			if (priority > curpri)
			{
				curpri = priority;
				swapinfo.presentMode = presentmode[i];
			}
		}

		delete[] presentmode;

		VkSwapchainKHR newvkswapchain = VK_NULL_HANDLE;
		VkAssert(pvkCreateSwapchainKHR(vkdevice, &swapinfo, vkallocationcb, &newvkswapchain));
		if (!newvkswapchain)
			return false;
		if (vkswapchain)
			DestroySwapChain();
		vkswapchain = newvkswapchain;

		VkAssert(pvkGetSwapchainImagesKHR(vkdevice, vkswapchain, &vkbackbuffer_count, NULL));

		VkImage *images = new VkImage[vkbackbuffer_count];
		VkAssert(pvkGetSwapchainImagesKHR(vkdevice, vkswapchain, &vkbackbuffer_count, images));


		VkImageCreateInfo depthinfo = {};
		depthinfo.sType = VK_STRUCTURE_TYPE_IMAGE_CREATE_INFO;
		depthinfo.flags = 0;
		depthinfo.imageType = VK_IMAGE_TYPE_2D;
		depthinfo.format = depthformat;
		depthinfo.extent.width = width;
		depthinfo.extent.height = height;
		depthinfo.extent.depth = 1;
		depthinfo.mipLevels = 1;
		depthinfo.arrayLayers = 1;
		depthinfo.samples = VK_SAMPLE_COUNT_1_BIT;
		depthinfo.tiling = VK_IMAGE_TILING_OPTIMAL;
		depthinfo.usage = VK_IMAGE_USAGE_DEPTH_STENCIL_ATTACHMENT_BIT;
		depthinfo.sharingMode = VK_SHARING_MODE_EXCLUSIVE;
		depthinfo.queueFamilyIndexCount = 0;
		depthinfo.pQueueFamilyIndices = NULL;
		depthinfo.initialLayout = VK_IMAGE_LAYOUT_UNDEFINED;
		VkAssert(pvkCreateImage(vkdevice, &depthinfo, vkallocationcb, &vkdepth.image));

		VkMemoryRequirements mem_reqs;
		pvkGetImageMemoryRequirements(vkdevice, vkdepth.image, &mem_reqs);
		VkMemoryAllocateInfo memAllocInfo = {};
		memAllocInfo.sType = VK_STRUCTURE_TYPE_MEMORY_ALLOCATE_INFO;
		memAllocInfo.allocationSize = mem_reqs.size;
		memAllocInfo.memoryTypeIndex = memory_type_from_properties_require(mem_reqs.memoryTypeBits, 0);
		VkAssert(pvkAllocateMemory(vkdevice, &memAllocInfo, vkallocationcb, &vkdepth.memory));
		VkAssert(pvkBindImageMemory(vkdevice, vkdepth.image, vkdepth.memory, 0));

		VkImageViewCreateInfo depthviewinfo = {};
		depthviewinfo.sType = VK_STRUCTURE_TYPE_IMAGE_VIEW_CREATE_INFO;
		depthviewinfo.format = depthformat;
		depthviewinfo.components.r = VK_COMPONENT_SWIZZLE_IDENTITY;
		depthviewinfo.components.g = VK_COMPONENT_SWIZZLE_IDENTITY;
		depthviewinfo.components.b = VK_COMPONENT_SWIZZLE_IDENTITY;
		depthviewinfo.components.a = VK_COMPONENT_SWIZZLE_IDENTITY;
		depthviewinfo.subresourceRange.aspectMask = VK_IMAGE_ASPECT_DEPTH_BIT;//|VK_IMAGE_ASPECT_STENCIL_BIT;
		depthviewinfo.subresourceRange.baseMipLevel = 0;
		depthviewinfo.subresourceRange.levelCount = 1;
		depthviewinfo.subresourceRange.baseArrayLayer = 0;
		depthviewinfo.subresourceRange.layerCount = 1;
		depthviewinfo.viewType = VK_IMAGE_VIEW_TYPE_2D;
		depthviewinfo.flags = 0;
		depthviewinfo.image = vkdepth.image;
		VkAssert(pvkCreateImageView(vkdevice, &depthviewinfo, vkallocationcb, &vkdepth.imageView));

		VkImageView attachments[2];
		attachments[0] = VK_NULL_HANDLE;
		attachments[1] = vkdepth.imageView;

		VkFramebufferCreateInfo fb_info = {};
		fb_info.sType = VK_STRUCTURE_TYPE_FRAMEBUFFER_CREATE_INFO;
		fb_info.renderPass = vkrenderpass;
		fb_info.attachmentCount = countof(attachments);
		fb_info.pAttachments = attachments;
		fb_info.width = width;
		fb_info.height = height;
		fb_info.layers = 1;


		vkbackbuffer = new vkbackbuffer_s[vkbackbuffer_count];
		memset(vkbackbuffer, 0, sizeof(vkbackbuffer_s)*vkbackbuffer_count);
		for (uint32_t i = 0; i < vkbackbuffer_count; i++)
		{
			VkImageViewCreateInfo ivci = {};
			ivci.sType = VK_STRUCTURE_TYPE_IMAGE_VIEW_CREATE_INFO;
			ivci.format = surffmts->format;
			ivci.components.r = VK_COMPONENT_SWIZZLE_R;
			ivci.components.g = VK_COMPONENT_SWIZZLE_G;
			ivci.components.b = VK_COMPONENT_SWIZZLE_B;
			ivci.components.a = VK_COMPONENT_SWIZZLE_A;
			ivci.subresourceRange.aspectMask = VK_IMAGE_ASPECT_COLOR_BIT;
			ivci.subresourceRange.baseMipLevel = 0;
			ivci.subresourceRange.levelCount = 1;
			ivci.subresourceRange.baseArrayLayer = 0;
			ivci.subresourceRange.layerCount = 1;
			ivci.viewType = VK_IMAGE_VIEW_TYPE_2D;
			ivci.flags = 0;
			ivci.image = images[i];
			vkbackbuffer[i].image = images[i];
			VkAssert(pvkCreateImageView(vkdevice, &ivci, vkallocationcb, &vkbackbuffer[i].view));

			attachments[0] = vkbackbuffer[i].view;
			VkAssert(pvkCreateFramebuffer(vkdevice, &fb_info, vkallocationcb, &vkbackbuffer[i].framebuffer));
		}
		delete[] images;

		
		for (uint32_t i = 0; i < vkbackbuffer_count; i++)
		{
			VkCommandBufferAllocateInfo cbai = {};
			cbai.sType = VK_STRUCTURE_TYPE_COMMAND_BUFFER_ALLOCATE_INFO;
			cbai.commandPool = vkcmdpool;
			cbai.level = VK_COMMAND_BUFFER_LEVEL_PRIMARY;
			cbai.commandBufferCount = 1;
			VkAssert(pvkAllocateCommandBuffers(vkdevice, &cbai, &vkbackbuffer[i].cmdbuf));

			VkSemaphoreCreateInfo seminfo = {};
			seminfo.sType = VK_STRUCTURE_TYPE_SEMAPHORE_CREATE_INFO;
			VkAssert(pvkCreateSemaphore(vkdevice, &seminfo, vkallocationcb, &vkbackbuffer[i].sem_presentable));

			VkFenceCreateInfo fci={};
			fci.sType = VK_STRUCTURE_TYPE_FENCE_CREATE_INFO;
			fci.flags = VK_FENCE_CREATE_SIGNALED_BIT;
			VkAssert(pvkCreateFence(vkdevice,&fci,vkallocationcb,&vkbackbuffer[i].fence));
		}

		return true;
	}
	virtual bool Startup(const char *title)
	{
		WNDCLASS wc;
		closereq = false;
		buttonbits = 0;
		vkframe = NULL;
		mode = 0;

		//VK_FORMAT_D16_UNORM is guarenteed as a depth-stencil attachment.
		//either d24_8x, or d32 must be supported
		//either d24_8s, or d32_8s must be supported
		//stencil formats might not be supported for reading in shaders.

		depthformat = VK_FORMAT_D32_SFLOAT_S8_UINT;//VK_FORMAT_D24_UNORM_S8_UINT;//VK_FORMAT_D16_UNORM;

		wc.style = 0;
		wc.lpfnWndProc = wproc;
		wc.cbClsExtra = 0;
		wc.cbWndExtra = 0;
		wc.hInstance = GetModuleHandle(NULL);
		wc.hIcon = NULL;
		wc.hCursor = LoadCursor (NULL,IDC_ARROW);
		wc.hbrBackground = NULL;
		wc.lpszMenuName = NULL;
		wc.lpszClassName = "MyWClass";
		RegisterClass(&wc);

		RECT wrect = {};
		wrect.right = 640;
		wrect.bottom = 480;
		wrect.left = (GetSystemMetrics(SM_CXSCREEN) - wrect.right)/2;
		wrect.top = (GetSystemMetrics(SM_CYSCREEN) - wrect.bottom)/2;
		wrect.right += wrect.left;
		wrect.bottom += wrect.top;
		AdjustWindowRect(&wrect, WS_OVERLAPPEDWINDOW|WS_SYSMENU, FALSE);

		wnd = CreateWindowEx(0, wc.lpszClassName, title, WS_OVERLAPPEDWINDOW|WS_SYSMENU, wrect.left, wrect.top, wrect.right-wrect.left, wrect.bottom-wrect.top, NULL, NULL, wc.hInstance, this);
		ShowWindow(wnd, SW_NORMAL);

		MSG wmsg;
		while (PeekMessage (&wmsg, NULL, 0, 0, PM_REMOVE))
		{
			TranslateMessage(&wmsg);
			DispatchMessage(&wmsg);
		}

		GetClientRect(wnd, &wrect);
		width = wrect.right - wrect.left;
		height = wrect.bottom - wrect.top;

		if (!wnd)
			return false;

#ifdef VK_NO_PROTOTYPES
		HMODULE vk = LoadLibrary("vulkan-1.dll");
		if (!vk)
			return false;
		pvkGetInstanceProcAddr = (PFN_vkGetInstanceProcAddr)GetProcAddress(vk, "vkGetInstanceProcAddr");
#endif
		if (!pvkGetInstanceProcAddr)
			return false;

		if (!StartupVulkan())
			return false;
		if (!StartupVulkanSwapChain())
			return false;

		return true;
	}
	void SetupRenderPass()
	{
		VkAttachmentReference color_reference = {};
		color_reference.attachment = 0;
		color_reference.layout = VK_IMAGE_LAYOUT_COLOR_ATTACHMENT_OPTIMAL;

		VkAttachmentReference depth_reference = {};
		depth_reference.attachment = 1;
		depth_reference.layout = VK_IMAGE_LAYOUT_DEPTH_STENCIL_ATTACHMENT_OPTIMAL;

		VkAttachmentDescription attachments[2] = {};
		attachments[color_reference.attachment].format = VK_FORMAT_B8G8R8A8_UNORM;
		attachments[color_reference.attachment].samples = VK_SAMPLE_COUNT_1_BIT;
		attachments[color_reference.attachment].loadOp = VK_ATTACHMENT_LOAD_OP_CLEAR;
		attachments[color_reference.attachment].storeOp = VK_ATTACHMENT_STORE_OP_STORE;
		attachments[color_reference.attachment].stencilLoadOp = VK_ATTACHMENT_LOAD_OP_DONT_CARE;
		attachments[color_reference.attachment].stencilStoreOp = VK_ATTACHMENT_STORE_OP_DONT_CARE;
		attachments[color_reference.attachment].initialLayout = VK_IMAGE_LAYOUT_COLOR_ATTACHMENT_OPTIMAL;
		attachments[color_reference.attachment].finalLayout = VK_IMAGE_LAYOUT_COLOR_ATTACHMENT_OPTIMAL;
		attachments[depth_reference.attachment].format = depthformat;
		attachments[depth_reference.attachment].samples = VK_SAMPLE_COUNT_1_BIT;
		attachments[depth_reference.attachment].loadOp = VK_ATTACHMENT_LOAD_OP_CLEAR;
		attachments[depth_reference.attachment].storeOp = VK_ATTACHMENT_STORE_OP_DONT_CARE;
		attachments[depth_reference.attachment].stencilLoadOp = VK_ATTACHMENT_LOAD_OP_DONT_CARE;
		attachments[depth_reference.attachment].stencilStoreOp = VK_ATTACHMENT_STORE_OP_DONT_CARE;
		attachments[depth_reference.attachment].initialLayout = VK_IMAGE_LAYOUT_DEPTH_STENCIL_ATTACHMENT_OPTIMAL;
		attachments[depth_reference.attachment].finalLayout = VK_IMAGE_LAYOUT_DEPTH_STENCIL_ATTACHMENT_OPTIMAL;

		VkSubpassDescription subpass = {};
		subpass.pipelineBindPoint = VK_PIPELINE_BIND_POINT_GRAPHICS;
		subpass.flags = 0;
		subpass.inputAttachmentCount = 0;
		subpass.pInputAttachments = NULL;
		subpass.colorAttachmentCount = 1;
		subpass.pColorAttachments = &color_reference;
		subpass.pResolveAttachments = NULL;
		subpass.pDepthStencilAttachment = &depth_reference;
		subpass.preserveAttachmentCount = 0;
		subpass.pPreserveAttachments = NULL;

		VkRenderPassCreateInfo rp_info = {};
		rp_info.sType = VK_STRUCTURE_TYPE_RENDER_PASS_CREATE_INFO;
		rp_info.attachmentCount = countof(attachments);
		rp_info.pAttachments = attachments;
		rp_info.subpassCount = 1;
		rp_info.pSubpasses = &subpass;
		rp_info.dependencyCount = 0;
		rp_info.pDependencies = NULL;

		VkAssert(pvkCreateRenderPass(vkdevice, &rp_info, vkallocationcb, &vkrenderpass));
	}
	void ClearScreen()
	{
		VkSemaphore vsyncsemaphore;
		VkSemaphoreCreateInfo seminfo = {};
		seminfo.sType = VK_STRUCTURE_TYPE_SEMAPHORE_CREATE_INFO;
		VkAssert(pvkCreateSemaphore(vkdevice, &seminfo, vkallocationcb, &vsyncsemaphore));

		VkResult acquireerror = pvkAcquireNextImageKHR(vkdevice, vkswapchain, UINT64_MAX, vsyncsemaphore, VK_NULL_HANDLE, &vkcurrentbuffer);
		switch(acquireerror)
		{
		case VK_ERROR_OUT_OF_DATE_KHR:
			resized = true;	//gah!
			closereq = true;
			break;
		case VK_SUBOPTIMAL_KHR:
			resized = true;	//recreate it, but live with it for now.
			break;
		case VK_SUCCESS:
			break;	//yay
		default:
			Sys_Error("vkAcquireNextImageKHR == %i", acquireerror);
			break;
		}

		vkframe = &vkbackbuffer[vkcurrentbuffer];
		vkmaincmd = vkframe->cmdbuf;

		if (vkframe->fence)	//make sure that frame is actually finished...
		{
			if (VK_NOT_READY == pvkGetFenceStatus(vkdevice, vkframe->fence))
				VkAssert(pvkWaitForFences(vkdevice, 1, &vkframe->fence, VK_FALSE, UINT64_MAX));
		}

		if (vkframe->sem_vsync)
			pvkDestroySemaphore(vkdevice, vkframe->sem_vsync, vkallocationcb);
		vkframe->sem_vsync = vsyncsemaphore;

		VkAssert(pvkResetFences(vkdevice, 1, &vkframe->fence));

		VkCommandBufferBeginInfo cmdinf = {};
		cmdinf.sType = VK_STRUCTURE_TYPE_COMMAND_BUFFER_BEGIN_INFO;
		cmdinf.flags = VK_COMMAND_BUFFER_USAGE_ONE_TIME_SUBMIT_BIT;
		cmdinf.pInheritanceInfo = NULL;
		VkAssert(pvkBeginCommandBuffer(vkmaincmd, &cmdinf));

		set_image_layout(vkmaincmd, vkbackbuffer[vkcurrentbuffer].image, VK_IMAGE_ASPECT_COLOR_BIT,
                          VK_IMAGE_LAYOUT_PRESENT_SRC_KHR,
                          VK_IMAGE_LAYOUT_COLOR_ATTACHMENT_OPTIMAL);

		VkClearValue	clearvalues[2];
		clearvalues[0].color.float32[0] = 0;
		clearvalues[0].color.float32[1] = 0.2f;
		clearvalues[0].color.float32[2] = 0;
		clearvalues[0].color.float32[3] = 1;
		clearvalues[1].depthStencil.depth = 1.0;
		clearvalues[1].depthStencil.stencil = 0;
		VkRenderPassBeginInfo rpbi = {};
		rpbi.sType = VK_STRUCTURE_TYPE_RENDER_PASS_BEGIN_INFO;
		rpbi.renderPass = vkrenderpass;
		rpbi.framebuffer = vkbackbuffer[vkcurrentbuffer].framebuffer;
		rpbi.renderArea.offset.x = 0;
		rpbi.renderArea.offset.y = 0;
		rpbi.renderArea.extent.width = width;
		rpbi.renderArea.extent.height = height;
		rpbi.clearValueCount = countof(clearvalues);
		rpbi.pClearValues = clearvalues;

		pvkCmdBeginRenderPass(vkmaincmd, &rpbi, VK_SUBPASS_CONTENTS_INLINE);


		VkViewport vp[1] = {};
		vp[0].x = 0;
		vp[0].y = 0;
		vp[0].width = width;
		vp[0].height = height;
		vp[0].minDepth = 0.0;
		vp[0].maxDepth = 1.0;
		VkRect2D scissor[1] = {};
		scissor[0].offset.x = 0;
		scissor[0].offset.y = 0;
		scissor[0].extent.width = width;
		scissor[0].extent.height = height;
		pvkCmdSetViewport(vkmaincmd, 0, countof(vp), vp);
		pvkCmdSetScissor(vkmaincmd, 0, countof(scissor), scissor);

//		Sleep(100);
	}
	virtual void Shutdown()
	{
		pvkDeviceWaitIdle(vkdevice);
		pvkDestroyCommandPool(vkdevice, vkcmdpool, vkallocationcb);
		DestroySwapChain();
		delete[] vkbackbuffer;
		vkbackbuffer = NULL;
		vkbackbuffer_count = 0;
		pvkDestroyRenderPass(vkdevice, vkrenderpass, vkallocationcb);
		pvkDestroyDevice(vkdevice, vkallocationcb);
		pvkDestroySurfaceKHR(vkinstance, vksurface, vkallocationcb);
		if (vkdebugcallback)
			pvkDestroyDebugReportCallbackEXT(vkinstance, vkdebugcallback, vkallocationcb);
		vkdebugcallback = VK_NULL_HANDLE;
		pvkDestroyInstance(vkinstance, vkallocationcb);
		DestroyWindow(wnd);
		wnd = NULL;
	}

	virtual bool PostFrame(void)
	{
		uint32_t timeout = 1000*1000*1000;
		VkResult err;

		pvkCmdEndRenderPass(vkmaincmd);

		//all rendering commands are done, but we need to wait for the rendering itself to actually complete.
		//(we're also meant to transition the framebuffer to something that can be presented cleanly)
		VkImageMemoryBarrier prepresentbarrier = {};
		prepresentbarrier.sType = VK_STRUCTURE_TYPE_IMAGE_MEMORY_BARRIER;
		prepresentbarrier.srcAccessMask = VK_ACCESS_COLOR_ATTACHMENT_WRITE_BIT;
		prepresentbarrier.dstAccessMask = VK_ACCESS_MEMORY_READ_BIT;
		prepresentbarrier.oldLayout = VK_IMAGE_LAYOUT_COLOR_ATTACHMENT_OPTIMAL;
		prepresentbarrier.newLayout = VK_IMAGE_LAYOUT_PRESENT_SRC_KHR;
		prepresentbarrier.srcQueueFamilyIndex = VK_QUEUE_FAMILY_IGNORED;
		prepresentbarrier.dstQueueFamilyIndex = VK_QUEUE_FAMILY_IGNORED;
		prepresentbarrier.image = vkbackbuffer[vkcurrentbuffer].image;
		prepresentbarrier.subresourceRange.aspectMask = VK_IMAGE_ASPECT_COLOR_BIT;
		prepresentbarrier.subresourceRange.baseArrayLayer = 0;
		prepresentbarrier.subresourceRange.baseMipLevel = 0;
		prepresentbarrier.subresourceRange.layerCount = 1;
		prepresentbarrier.subresourceRange.levelCount = 1;
		pvkCmdPipelineBarrier(vkmaincmd, VK_PIPELINE_STAGE_ALL_COMMANDS_BIT, VK_PIPELINE_STAGE_BOTTOM_OF_PIPE_BIT, 0, 0, NULL, 0, NULL, 1, &prepresentbarrier);
		pvkEndCommandBuffer(vkmaincmd);


		VkPipelineStageFlags pipe_stage_flags = VK_PIPELINE_STAGE_BOTTOM_OF_PIPE_BIT;
		VkSubmitInfo subinf[1]={};
		subinf[0].sType = VK_STRUCTURE_TYPE_SUBMIT_INFO;
		subinf[0].waitSemaphoreCount = 1;
		subinf[0].pWaitSemaphores = &vkframe->sem_vsync;	//don't begin rendering until we actually own the backbuffer
		subinf[0].pWaitDstStageMask = &pipe_stage_flags;
		subinf[0].commandBufferCount = 1;
		subinf[0].pCommandBuffers = &vkmaincmd;
		subinf[0].signalSemaphoreCount = 1;
		subinf[0].pSignalSemaphores = &vkframe->sem_presentable;
		VkAssert(pvkQueueSubmit(vkqueuerender,countof(subinf),subinf,vkframe->fence));

		VkPresentInfoKHR present = {};
		present.sType = VK_STRUCTURE_TYPE_PRESENT_INFO_KHR;
		present.waitSemaphoreCount = 1;
		present.pWaitSemaphores = &vkframe->sem_presentable;
		present.swapchainCount = 1;
		present.pSwapchains = &vkswapchain;
		present.pImageIndices = &vkcurrentbuffer;
		err = pvkQueuePresentKHR(vkqueuepresent, &present);

		switch(err)
		{
		default:
			Com_Printf(PRINT_ERROR, "Unknown vulkan vswap error: %x\n", err);
			closereq = true;
			break;
		case VK_SUCCESS:
			break;
		}
		if (closereq)
		{
			closereq = false;

			pvkDeviceWaitIdle(vkdevice);
			return true;
		}
		return false;
	}
	virtual bool PreFrame(bool cursorgrabs)
	{
		MSG wmsg;

		while (PeekMessage (&wmsg, NULL, 0, 0, PM_REMOVE))
		{
			TranslateMessage(&wmsg);
			DispatchMessage(&wmsg);
		}

		if (resized)
		{
			resized = false;
			pvkQueueWaitIdle(vkqueuerender);
			pvkDeviceWaitIdle(vkdevice);

			RECT wrect;
			GetClientRect(wnd, &wrect);
			if (width != wrect.right - wrect.left || height != wrect.bottom - wrect.top)
			{
				width = wrect.right - wrect.left;
				height = wrect.bottom - wrect.top;

				StartupVulkanSwapChain();
			}
		}

		if (cursorgrabs && GetForegroundWindow() == wnd)
		{
			POINT mpos;
			POINT cpos;
			RECT r;
			GetWindowRect(wnd, &r);
			cpos.x = (r.right - r.left)/2 + r.left;
			cpos.y = (r.bottom - r.top)/2 + r.top;
			GetCursorPos(&mpos);
			if (input)
				input->MouseDelta(mpos.x - cpos.x, mpos.y-cpos.y);
			SetCursorPos(cpos.x, cpos.y);
		}
		else
		{
			POINT mpos;
			GetCursorPos(&mpos);
			if (input)
				input->MouseAbsolute(mpos.x, mpos.y);
		}

		ClearScreen();
		return closereq;
	}
	virtual double GetTime()
	{
		static unsigned int firsttime;
		if (!firsttime)
			firsttime = timeGetTime();
		return (timeGetTime()-firsttime)/1000.0;
	}
};

vid_t *Vid_Vulkan_Create(void)
{
	vid_t *vid = new vkvid_t();
	return vid;
}

void DestroyTexture(vk_image_t *image)
{
	if (image->sampler)
		pvkDestroySampler(vkdevice, image->sampler, vkallocationcb);
	pvkDestroyImageView(vkdevice, image->imageView, vkallocationcb);
	pvkDestroyImage(vkdevice, image->image, vkallocationcb);
	pvkFreeMemory(vkdevice, image->memory, vkallocationcb);
}
vk_image_t CreateTexture2DArray(uint32_t width, uint32_t height, uint32_t layers, uint32_t mips)
{
	vk_image_t ret;

	bool staging = layers == 0;
	
	VkImageCreateInfo ici = {};
	ici.sType = VK_STRUCTURE_TYPE_IMAGE_CREATE_INFO;
	ici.flags = 0;
	ici.imageType = VK_IMAGE_TYPE_2D;
	ici.format = VK_FORMAT_R8G8B8A8_UNORM;
	ici.extent.width = width;
	ici.extent.height = height;
	ici.extent.depth = 1;
	ici.mipLevels = mips;
	ici.arrayLayers = staging?1:layers;
	ici.samples = VK_SAMPLE_COUNT_1_BIT;
	ici.tiling = staging?VK_IMAGE_TILING_LINEAR:VK_IMAGE_TILING_OPTIMAL;
	ici.usage = staging?VK_IMAGE_USAGE_TRANSFER_SRC_BIT:(VK_IMAGE_USAGE_SAMPLED_BIT|VK_IMAGE_USAGE_TRANSFER_DST_BIT);
	ici.sharingMode = VK_SHARING_MODE_EXCLUSIVE;
	ici.queueFamilyIndexCount = 0;
	ici.pQueueFamilyIndices = NULL;
	ici.initialLayout = staging?VK_IMAGE_LAYOUT_PREINITIALIZED:VK_IMAGE_LAYOUT_UNDEFINED;

	VkAssert(pvkCreateImage(vkdevice, &ici, vkallocationcb, &ret.image));

	VkMemoryRequirements mem_reqs;
	pvkGetImageMemoryRequirements(vkdevice, ret.image, &mem_reqs);

	VkMemoryAllocateInfo memAllocInfo = {};
	memAllocInfo.sType = VK_STRUCTURE_TYPE_MEMORY_ALLOCATE_INFO;
	memAllocInfo.allocationSize = mem_reqs.size;
	if (staging)
		memAllocInfo.memoryTypeIndex = memory_type_from_properties_require(mem_reqs.memoryTypeBits, VK_MEMORY_PROPERTY_HOST_VISIBLE_BIT);
	else
	{
		memAllocInfo.memoryTypeIndex = memory_type_from_properties_try(mem_reqs.memoryTypeBits, VK_MEMORY_PROPERTY_HOST_VISIBLE_BIT|VK_MEMORY_PROPERTY_DEVICE_LOCAL_BIT);
		if (memAllocInfo.memoryTypeIndex == ~0)
			memAllocInfo.memoryTypeIndex = memory_type_from_properties_try(mem_reqs.memoryTypeBits, VK_MEMORY_PROPERTY_DEVICE_LOCAL_BIT);
		if (memAllocInfo.memoryTypeIndex == ~0)
			memAllocInfo.memoryTypeIndex = memory_type_from_properties_try(mem_reqs.memoryTypeBits, VK_MEMORY_PROPERTY_HOST_VISIBLE_BIT);
		if (memAllocInfo.memoryTypeIndex == ~0)
			memAllocInfo.memoryTypeIndex = memory_type_from_properties_require(mem_reqs.memoryTypeBits, 0);
	}

	VkAssert(pvkAllocateMemory(vkdevice, &memAllocInfo, vkallocationcb, &ret.memory));
	VkAssert(pvkBindImageMemory(vkdevice, ret.image, ret.memory, 0));

	ret.width = width;
	ret.height = height;
	ret.mips = mips;

	ret.sampler = VK_NULL_HANDLE;
	ret.imageView = VK_NULL_HANDLE;

	if (!staging)
	{
		VkImageViewCreateInfo viewInfo = {};
		viewInfo.sType = VK_STRUCTURE_TYPE_IMAGE_VIEW_CREATE_INFO;
		viewInfo.flags = 0;
		viewInfo.image = ret.image;
		viewInfo.viewType = VK_IMAGE_VIEW_TYPE_2D_ARRAY;
		viewInfo.format = VK_FORMAT_R8G8B8A8_UNORM;
		viewInfo.components.r = VK_COMPONENT_SWIZZLE_R;
		viewInfo.components.g = VK_COMPONENT_SWIZZLE_G;
		viewInfo.components.b = VK_COMPONENT_SWIZZLE_B;
		viewInfo.components.a = VK_COMPONENT_SWIZZLE_A;
		viewInfo.subresourceRange.aspectMask = VK_IMAGE_ASPECT_COLOR_BIT;
		viewInfo.subresourceRange.baseMipLevel = 0;
		viewInfo.subresourceRange.levelCount = mips;
		viewInfo.subresourceRange.baseArrayLayer = 0;
		viewInfo.subresourceRange.layerCount = layers;
		VkAssert(pvkCreateImageView(vkdevice, &viewInfo, NULL, &ret.imageView));
	}
	return ret;
}

void CreateTextureSampler(vk_image_t *img, bool linear, bool repeat)
{
	VkSamplerCreateInfo lmsampinfo = {};
	lmsampinfo.sType = VK_STRUCTURE_TYPE_SAMPLER_CREATE_INFO;
	lmsampinfo.magFilter = linear?VK_FILTER_LINEAR:VK_FILTER_NEAREST;
	lmsampinfo.minFilter = linear?VK_FILTER_LINEAR:VK_FILTER_NEAREST;
	lmsampinfo.mipmapMode = VK_SAMPLER_MIPMAP_MODE_LINEAR;
	lmsampinfo.addressModeU = repeat?VK_SAMPLER_ADDRESS_MODE_REPEAT:VK_SAMPLER_ADDRESS_MODE_CLAMP_TO_EDGE;
	lmsampinfo.addressModeV = repeat?VK_SAMPLER_ADDRESS_MODE_REPEAT:VK_SAMPLER_ADDRESS_MODE_CLAMP_TO_EDGE;
	lmsampinfo.addressModeW = repeat?VK_SAMPLER_ADDRESS_MODE_REPEAT:VK_SAMPLER_ADDRESS_MODE_CLAMP_TO_EDGE;
	lmsampinfo.mipLodBias = 0.0;
	lmsampinfo.anisotropyEnable = VK_FALSE;
	lmsampinfo.maxAnisotropy = 8.0;
	lmsampinfo.compareEnable = VK_FALSE;
	lmsampinfo.compareOp = VK_COMPARE_OP_NEVER;
	lmsampinfo.minLod = -1000.0;
	lmsampinfo.maxLod = 1000.0;
	lmsampinfo.borderColor = VK_BORDER_COLOR_FLOAT_OPAQUE_BLACK;
	lmsampinfo.unnormalizedCoordinates = VK_FALSE;
	VkAssert(pvkCreateSampler(vkdevice, &lmsampinfo, NULL, &img->sampler));
}

static void Mipmap(unsigned char *out, uint32_t outpitch, uint32_t outwidth, uint32_t outheight,
				   unsigned char *in, uint32_t inpitch, uint32_t inwidth, uint32_t inheight)
{
	//mips are always half (rounded-down) of the size of the previous - unless its non-square.
	//in which case, one of the input dimensions can already be 1-wide. which is a problem if we're just adding 2*2 blocks.
	//note: in the case of npot, we might end up discarding the spare bottom/right edge pixels.
	if (inwidth == 1)
	{
		for (uint32_t y = 0; y < outheight; y++)
		{
			unsigned char *in1 = in + y*2*inpitch;
			unsigned char *in2 = in + (y*2+(inheight>1))*inpitch;
			unsigned char *o = out + y*outpitch;

			o[0] = (in1[0] + in2[0]+1)>>1;
			o[1] = (in1[1] + in2[1]+1)>>1;
			o[2] = (in1[2] + in2[2]+1)>>1;
			o[3] = (in1[3] + in2[3]+1)>>1;
		}
	}
	else
	{
		for (uint32_t y = 0; y < outheight; y++)
		{
			unsigned char *in1 = in + y*2*inpitch;
			unsigned char *in2 = in + (y*2+(inheight>1))*inpitch;
			unsigned char *o = out + y*outpitch;

			for (uint32_t x = 0; x < outwidth; x++)
			{
				o[0] = (in1[0] + in1[4] + in2[0] + in2[4]+2)>>2;
				o[1] = (in1[1] + in1[5] + in2[1] + in2[5]+2)>>2;
				o[2] = (in1[2] + in1[6] + in2[2] + in2[6]+2)>>2;
				o[3] = (in1[3] + in1[7] + in2[3] + in2[7]+2)>>2;
				in1 += 8;
				in2 += 8;
				o += 4;
			}
		}
	}
}
void FillTextureSync(vk_image_t target, uint32_t width, uint32_t height, uint32_t layer, uint32_t mip, void *imgdata)
{
	VkCommandBuffer vkloadcmd;

	uint32_t mips = 1, basemip;
	struct {
		uint32_t width;
		uint32_t height;
		vk_image_t staging;
	} miplevel[20];

	if (width > target.width || height > target.height)
		return;

	if (mip == ~0u)
	{	//auto-generate mips. on the cpu. like a lunatic.
		for (mips = 0; width >= 1 && height >= 1 && mips < target.mips; mips++)
		{
			//mips keep going until they're both 0.
			if (!width)
				width = 1;
			if (!height)
				height = 1;
			miplevel[mips].width = width;
			miplevel[mips].height = height;
			width >>= 1;
			height >>= 1;
		}
		basemip = 0;
	}
	else
	{
		basemip = mip;
		miplevel[0].width = width;
		miplevel[0].height = height;
		mips = 1;
	}

	VkCommandBufferAllocateInfo cbai = {};
	cbai.sType = VK_STRUCTURE_TYPE_COMMAND_BUFFER_ALLOCATE_INFO;
	cbai.commandPool = vkcmdpool;
	cbai.level = VK_COMMAND_BUFFER_LEVEL_PRIMARY;
	cbai.commandBufferCount = 1;
	VkAssert(pvkAllocateCommandBuffers(vkdevice, &cbai, &vkloadcmd));

	VkCommandBufferInheritanceInfo cmdinh = {};
	cmdinh.sType = VK_STRUCTURE_TYPE_COMMAND_BUFFER_INHERITANCE_INFO;

	VkCommandBufferBeginInfo cmdinf = {};
	cmdinf.sType = VK_STRUCTURE_TYPE_COMMAND_BUFFER_BEGIN_INFO;
	cmdinf.flags = VK_COMMAND_BUFFER_USAGE_ONE_TIME_SUBMIT_BIT;
	cmdinf.pInheritanceInfo = &cmdinh;
	pvkBeginCommandBuffer(vkloadcmd, &cmdinf);

	if (imgdata)
	{
		uint32_t pitch = miplevel[0].width*4;
		for (mip = 0; mip < mips; mip++)
		{
			miplevel[mip].staging = CreateTexture2DArray(miplevel[mip].width, miplevel[mip].height, 0, 1);

			VkImageSubresource subres = {};
			subres.aspectMask = VK_IMAGE_ASPECT_COLOR_BIT;
			subres.mipLevel = 0;
			subres.arrayLayer = 0;
			VkSubresourceLayout layout;
			pvkGetImageSubresourceLayout(vkdevice, miplevel[mip].staging.image, &subres, &layout);
			void *mapdata;
			VkAssert(pvkMapMemory(vkdevice, miplevel[mip].staging.memory, 0, layout.size, 0, &mapdata));
			if (mapdata)
			{
				VkImageSubresource subres = {};
				subres.aspectMask = VK_IMAGE_ASPECT_COLOR_BIT;
				subres.mipLevel = 0;
				subres.arrayLayer = 0;

				if (mip == 0)
				{	//directly copy the base mip
					for (uint32_t y = 0; y < miplevel[mip].height; y++)
						memcpy((char*)mapdata + layout.offset + y*layout.rowPitch, (char*)imgdata + y*miplevel[mip].width*4, miplevel[mip].width*4);
				}
				else
				{
					//other mips require us to filter it a bit.
					Mipmap((unsigned char*)mapdata + layout.offset, layout.rowPitch, miplevel[mip].width, miplevel[mip].height,
						(unsigned char*)imgdata, pitch, width, height);
				}
				imgdata = (char*)mapdata + layout.offset;
				pitch = layout.rowPitch;
				width = miplevel[mip].width;
				height = miplevel[mip].height;
			}
			else
				Sys_Error("Unable to map staging image\n");
		}
		for (mip = 0; mip < mips; mip++)
			pvkUnmapMemory(vkdevice, miplevel[mip].staging.memory);

		VkImageCopy region = {};
		region.srcSubresource.aspectMask = VK_IMAGE_ASPECT_COLOR_BIT;
		region.srcSubresource.mipLevel = 0;
		region.srcSubresource.baseArrayLayer = 0;
		region.srcSubresource.layerCount = 1;
		region.srcOffset.x = 0;
		region.srcOffset.y = 0;
		region.srcOffset.z = 0;
		region.dstSubresource.aspectMask = VK_IMAGE_ASPECT_COLOR_BIT;
//		region.dstSubresource.mipLevel = 0;
		region.dstSubresource.baseArrayLayer = layer;
		region.dstSubresource.layerCount = 1;
		region.dstOffset.x = 0;
		region.dstOffset.y = 0;
		region.dstOffset.z = 0;
//		region.extent.width = width;
//		region.extent.height = height;
		region.extent.depth = 1;

		set_image_layout(vkloadcmd, target.image, VK_IMAGE_ASPECT_COLOR_BIT, VK_IMAGE_LAYOUT_UNDEFINED, VK_IMAGE_LAYOUT_TRANSFER_DST_OPTIMAL);
		for (mip = 0; mip < mips; mip++)
		{
			region.dstSubresource.mipLevel = basemip + mip;
			region.extent.width = miplevel[mip].width;
			region.extent.height = miplevel[mip].height;
			set_image_layout(vkloadcmd, miplevel[mip].staging.image, VK_IMAGE_ASPECT_COLOR_BIT, VK_IMAGE_LAYOUT_PREINITIALIZED, VK_IMAGE_LAYOUT_TRANSFER_SRC_OPTIMAL);
			pvkCmdCopyImage(vkloadcmd, miplevel[mip].staging.image, VK_IMAGE_LAYOUT_TRANSFER_SRC_OPTIMAL, target.image, VK_IMAGE_LAYOUT_TRANSFER_DST_OPTIMAL, 1, &region);
		}
	}
	else
	{
		VkClearColorValue col = {};
		col.float32[0] = 0;
		col.float32[1] = 0;
		col.float32[2] = 0;
		col.float32[3] = 1;
		VkImageSubresourceRange range = {};
		range.aspectMask = VK_IMAGE_ASPECT_COLOR_BIT;
		range.baseArrayLayer = layer;
		range.baseMipLevel = basemip;
		range.layerCount = 1;
		range.levelCount = mips;
		pvkCmdClearColorImage(vkloadcmd, target.image, VK_IMAGE_LAYOUT_GENERAL, &col, 1, &range);
	}
	set_image_layout(vkloadcmd, target.image, VK_IMAGE_ASPECT_COLOR_BIT, VK_IMAGE_LAYOUT_TRANSFER_DST_OPTIMAL, VK_IMAGE_LAYOUT_SHADER_READ_ONLY_OPTIMAL);

	pvkEndCommandBuffer(vkloadcmd);

	VkSubmitInfo subinf[1]={};
	subinf[0].sType = VK_STRUCTURE_TYPE_SUBMIT_INFO;
	subinf[0].waitSemaphoreCount = 0;
	subinf[0].pWaitSemaphores = NULL;
	subinf[0].pWaitDstStageMask = NULL;
	subinf[0].commandBufferCount = 1;
	subinf[0].pCommandBuffers = &vkloadcmd;
	subinf[0].signalSemaphoreCount = 0;
	subinf[0].pSignalSemaphores = NULL;
	VkAssert(pvkQueueSubmit(vkqueuerender, countof(subinf), subinf, VK_NULL_HANDLE));

	pvkQueueWaitIdle(vkqueuerender);

	pvkFreeCommandBuffers(vkdevice, vkcmdpool, 1, &vkloadcmd);

	for (mip = 0; mip < mips; mip++)
	if (miplevel[mip].staging.image != VK_NULL_HANDLE)
	{
		pvkDestroyImage(vkdevice, miplevel[mip].staging.image, vkallocationcb);
		pvkFreeMemory(vkdevice, miplevel[mip].staging.memory, vkallocationcb);
	}
}


vk_buffer_t CreateBuffer(size_t length, uint32_t usage, bool dynamic)
{
	vk_buffer_t ret;

	VkBufferCreateInfo bi = {};
	bi.sType = VK_STRUCTURE_TYPE_BUFFER_CREATE_INFO;
    bi.flags = 0;
    bi.size = length;
	bi.usage = usage|(dynamic?0:VK_BUFFER_USAGE_TRANSFER_DST_BIT);
    bi.sharingMode = VK_SHARING_MODE_EXCLUSIVE;
    bi.queueFamilyIndexCount = 0;
    bi.pQueueFamilyIndices = NULL;
	pvkCreateBuffer(vkdevice, &bi, vkallocationcb, &ret.buffer);
	ret.size = length;

	VkMemoryRequirements mem_reqs;
	pvkGetBufferMemoryRequirements(vkdevice, ret.buffer, &mem_reqs);
	ret.alignment = mem_reqs.alignment;

	VkMemoryAllocateInfo memAllocInfo = {};
	memAllocInfo.sType = VK_STRUCTURE_TYPE_MEMORY_ALLOCATE_INFO;
	memAllocInfo.allocationSize = mem_reqs.size;
	memAllocInfo.memoryTypeIndex = memory_type_from_properties_try(mem_reqs.memoryTypeBits, VK_MEMORY_PROPERTY_HOST_VISIBLE_BIT|VK_MEMORY_PROPERTY_DEVICE_LOCAL_BIT);
	ret.hostvisible = true;
	if (dynamic)
	{
		/*if (memAllocInfo.memoryTypeIndex == ~0)
		{
			memAllocInfo.memoryTypeIndex = memory_type_from_properties_require(mem_reqs.memoryTypeBits, VK_MEMORY_PROPERTY_HOST_VISIBLE_BIT|VK_MEMORY_PROPERTY_HOST_COHERENT_BIT);
			ret.hostvisible = true;
		}*/
		if (memAllocInfo.memoryTypeIndex == ~0)
		{
			memAllocInfo.memoryTypeIndex = memory_type_from_properties_require(mem_reqs.memoryTypeBits, VK_MEMORY_PROPERTY_HOST_VISIBLE_BIT);
			ret.hostvisible = true;
		}
	}
	else
	{
		if (memAllocInfo.memoryTypeIndex == ~0)
		{
			memAllocInfo.memoryTypeIndex = memory_type_from_properties_require(mem_reqs.memoryTypeBits, VK_MEMORY_PROPERTY_DEVICE_LOCAL_BIT);
			ret.hostvisible = false;
		}
	}

	VkAssert(pvkAllocateMemory(vkdevice, &memAllocInfo, vkallocationcb, &ret.memory));
	VkAssert(pvkBindBufferMemory(vkdevice, ret.buffer, ret.memory, 0));

	return ret;
}
void FillBufferSync(vk_buffer_t *target, size_t offset, size_t length, void *data)
{
	VkCommandBuffer vkloadcmd;
	VkBuffer staging_buffer;
	VkDeviceMemory staging_memory;

	if (target->hostvisible)
	{
		//on mobile, we can skip the copy
		void *mapdata;
		VkAssert(pvkMapMemory(vkdevice, target->memory, offset, length, 0, &mapdata));
		if (mapdata)
		{
			memcpy(mapdata, data, length);
			pvkUnmapMemory(vkdevice, target->memory);
		}
		else
			Sys_Error("Unable to map staging image\n");
		return;
	}


	{
		VkCommandBufferAllocateInfo cbai = {};
		cbai.sType = VK_STRUCTURE_TYPE_COMMAND_BUFFER_ALLOCATE_INFO;
		cbai.commandPool = vkcmdpool;
		cbai.level = VK_COMMAND_BUFFER_LEVEL_PRIMARY;
		cbai.commandBufferCount = 1;
		VkAssert(pvkAllocateCommandBuffers(vkdevice, &cbai, &vkloadcmd));

		VkCommandBufferInheritanceInfo cmdinh = {};
		cmdinh.sType = VK_STRUCTURE_TYPE_COMMAND_BUFFER_INHERITANCE_INFO;

		VkCommandBufferBeginInfo cmdinf = {};
		cmdinf.sType = VK_STRUCTURE_TYPE_COMMAND_BUFFER_BEGIN_INFO;
		cmdinf.flags = VK_COMMAND_BUFFER_USAGE_ONE_TIME_SUBMIT_BIT;
		cmdinf.pInheritanceInfo = &cmdinh;
		pvkBeginCommandBuffer(vkloadcmd, &cmdinf);
	}



	VkBufferCreateInfo bi = {};
	bi.sType = VK_STRUCTURE_TYPE_BUFFER_CREATE_INFO;
    bi.flags = 0;
    bi.size = length;
    bi.usage = VK_BUFFER_USAGE_TRANSFER_SRC_BIT;
    bi.sharingMode = VK_SHARING_MODE_EXCLUSIVE;
    bi.queueFamilyIndexCount = 0;
    bi.pQueueFamilyIndices = NULL;
	pvkCreateBuffer(vkdevice, &bi, vkallocationcb, &staging_buffer);


	
	VkMemoryRequirements mem_reqs;
	pvkGetBufferMemoryRequirements(vkdevice, staging_buffer, &mem_reqs);

	VkMemoryAllocateInfo memAllocInfo = {};
	memAllocInfo.sType = VK_STRUCTURE_TYPE_MEMORY_ALLOCATE_INFO;
	memAllocInfo.allocationSize = mem_reqs.size;
	memAllocInfo.memoryTypeIndex = memory_type_from_properties_try(mem_reqs.memoryTypeBits, VK_MEMORY_PROPERTY_HOST_VISIBLE_BIT);

	VkAssert(pvkAllocateMemory(vkdevice, &memAllocInfo, vkallocationcb, &staging_memory));
	VkAssert(pvkBindBufferMemory(vkdevice, staging_buffer, staging_memory, 0));

	void *mapdata;
	VkAssert(pvkMapMemory(vkdevice, staging_memory, 0, length, 0, &mapdata));
	if (mapdata)
	{
		memcpy(mapdata, data, length);
		pvkUnmapMemory(vkdevice, staging_memory);
	}
	else
		Sys_Error("Unable to map staging buffer\n");


	VkBufferCopy bcr = {};
	bcr.srcOffset = 0;
	bcr.dstOffset = offset;
	bcr.size = length;
	pvkCmdCopyBuffer(vkloadcmd, staging_buffer, target->buffer, 1, &bcr);


	{
		pvkEndCommandBuffer(vkloadcmd);

		VkSubmitInfo subinf[1]={};
		subinf[0].sType = VK_STRUCTURE_TYPE_SUBMIT_INFO;
		subinf[0].waitSemaphoreCount = 0;
		subinf[0].pWaitSemaphores = NULL;
		subinf[0].pWaitDstStageMask = NULL;
		subinf[0].commandBufferCount = 1;
		subinf[0].pCommandBuffers = &vkloadcmd;
		subinf[0].signalSemaphoreCount = 0;
		subinf[0].pSignalSemaphores = NULL;
		VkAssert(pvkQueueSubmit(vkqueuerender,countof(subinf),subinf,VK_NULL_HANDLE));

		pvkQueueWaitIdle(vkqueuerender);

		pvkFreeCommandBuffers(vkdevice, vkcmdpool, 1, &vkloadcmd);
	}

	pvkDestroyBuffer(vkdevice, staging_buffer, vkallocationcb);
	pvkFreeMemory(vkdevice, staging_memory, vkallocationcb);
}

void DestroyBuffer(vk_buffer_t *buf)
{
	pvkDestroyBuffer(vkdevice, buf->buffer, vkallocationcb);
	pvkFreeMemory(vkdevice, buf->memory, vkallocationcb);
}


VkShaderModule LoadShaderCode(const char *filename, const unsigned char *blob)
{
	VkShaderModule mod;

	char *data;
	size_t size;
	file *f = filename?FS_OpenRFile(filename):NULL;
	if (f)
	{
		f->seektoend();
		size = f->tell();
		f->seek(0);
		data = new char[size+1];
		data[size] = 0;
		f->read(data, size);
		delete f;
	}
	else if (blob)
	{
		size = *(uint32_t*)blob;
		data = new char[size+1];
		data[size] = 0;
		memcpy(data, blob+4, size);
	}
	else
		return VK_NULL_HANDLE;

	VkShaderModuleCreateInfo info = {};
	info.sType = VK_STRUCTURE_TYPE_SHADER_MODULE_CREATE_INFO;
	info.flags = 0;
	info.codeSize = size;
	info.pCode = (uint32_t*)data;
	VkAssert(pvkCreateShaderModule(vkdevice, &info, vkallocationcb, &mod));
	delete[] data;

	return mod;
}
#endif