/*
Copyright (C) 1996-2001 Id Software, Inc.
Copyright (C) 2002-2009 John Fitzgibbons and others
Copyright (C) 2010-2014 QuakeSpasm developers
Copyright (C) 2016      Spike

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/
// sv_main.c -- server main program

#include "quakedef.h"

server_t	sv;
server_static_t	svs;

static char	localmodels[MAX_MODELS][8];	// inline model names for precache

int				sv_protocol = PROTOCOL_FITZQUAKE; //johnfitz
unsigned int	sv_protocol_pext2; //spike

//============================================================================

void SV_CalcStats(client_t *client, int *statsi, float *statsf)
{
	edict_t *ent = client->edict;
	//FIXME: string stats!
	int items;
	eval_t *val = GetEdictFieldValue(ent, pr_extfields.items2);
	if (val)
		items = (int)ent->v.items | ((int)val->_float << 23);
	else
		items = (int)ent->v.items | ((int)pr_global_struct->serverflags << 28);

	memset(statsi, 0, sizeof(*statsi)*MAX_CL_STATS);
	memset(statsf, 0, sizeof(*statsf)*MAX_CL_STATS);
	statsf[STAT_HEALTH] = ent->v.health;
//	statsf[STAT_FRAGS] = ent->v.frags;	//obsolete
	statsi[STAT_WEAPON] = SV_ModelIndex(PR_GetString(ent->v.weaponmodel));
	statsf[STAT_AMMO] = ent->v.currentammo;
	statsf[STAT_ARMOR] = ent->v.armorvalue;
	statsf[STAT_WEAPONFRAME] = ent->v.weaponframe;
	statsf[STAT_SHELLS] = ent->v.ammo_shells;
	statsf[STAT_NAILS] = ent->v.ammo_nails;
	statsf[STAT_ROCKETS] = ent->v.ammo_rockets;
	statsf[STAT_CELLS] = ent->v.ammo_cells;
	statsf[STAT_ACTIVEWEAPON] = ent->v.weapon;	//sent in a way that does NOT depend upon the current mod...
//	statsf[STAT_TOTALSECRETS] = pr_global_struct->total_secrets;	//don't bother with these, the qc sends extra updates and we don't want to end up with a race condition.
//	statsf[STAT_TOTALMONSTERS] = pr_global_struct->total_monsters;
//	statsf[STAT_SECRETS] = pr_global_struct->found_secrets;
//	statsf[STAT_MONSTERS] = pr_global_struct->killed_monsters;
//	statsi[STAT_TIME] = sv.time*1000;					//in ms, this was a hack to work around vanilla QW clients not having any concept of serverside time.
//	statsi[STAT_MATCHSTARTTIME] = 0*1000;				//in ms, set by the mod to when the current match actually starts (so pregame stuff doesn't interfere with game clocks).
//	stats[STAT_VIEW2] = NUM_FOR_EDICT(PROG_TO_EDICT(ent->v.view2));
//	statsf[STAT_VIEWZOON] = ent->v.viewzoom;

	//FIXME: add support for clientstat/globalstat qc builtins.

	if (client->protocol_pext2 & PEXT2_PREDINFO)
	{	//predinfo also kills clc_clientdata
		statsi[STAT_ITEMS] = items;
		statsf[STAT_VIEWHEIGHT] = ent->v.view_ofs[2];
		statsf[STAT_IDEALPITCH] = ent->v.idealpitch;
		statsf[STAT_PUNCHANGLE_X] = ent->v.punchangle[0];
		statsf[STAT_PUNCHANGLE_Y] = ent->v.punchangle[1];
		statsf[STAT_PUNCHANGLE_Z] = ent->v.punchangle[2];
	}
/*
	if (client->protocol_pext2 & PEXT2_PREDINFO)
	{	//prediction needs some info on the server's rules
		statsf[STAT_MOVEVARS_FRICTION] = sv_friction.value;
		statsf[STAT_MOVEVARS_WATERFRICTION] = sv_waterfriction.value;
		statsf[STAT_MOVEVARS_TICRATE] = host_maxfps.value;
		statsf[STAT_MOVEVARS_TIMESCALE] = sv_gamespeed.value;
		statsf[STAT_MOVEVARS_GRAVITY] = sv_gravity.value;
		statsf[STAT_MOVEVARS_STOPSPEED] = sv_stopspeed.value;
		statsf[STAT_MOVEVARS_MAXSPEED] = client->maxspeed;
		statsf[STAT_MOVEVARS_SPECTATORMAXSPEED] = sv_spectatormaxspeed.value;
		statsf[STAT_MOVEVARS_ACCELERATE] = sv_accelerate.value;
		statsf[STAT_MOVEVARS_AIRACCELERATE] = sv_airaccelerate.value;
		statsf[STAT_MOVEVARS_WATERACCELERATE] = sv_wateraccelerate.value;
		statsf[STAT_MOVEVARS_ENTGRAVITY] = client->entgravity/sv_gravity.value;
		statsf[STAT_MOVEVARS_JUMPVELOCITY] = sv_jumpvelocity.value;	//bah
		statsf[STAT_MOVEVARS_EDGEFRICTION] = sv_edgefriction.value;
		statsf[STAT_MOVEVARS_MAXAIRSPEED] = client->maxspeed;
		statsf[STAT_MOVEVARS_STEPHEIGHT] = 18;
//		statsf[STAT_MOVEVARS_AIRACCEL_QW] = 0;
//		statsf[STAT_MOVEVARS_AIRACCEL_SIDEWAYS_FRICTION] = sv_gravity.value;
	}
*/
}


/*server-side-only flags that re-use encoding bits*/
#define UF_REMOVE		UF_16BIT	/*says we removed the entity in this frame*/
#define UF_MOVETYPE		UF_EFFECTS2	/*this flag isn't present in the header itself*/
#define UF_RESET2		UF_EXTEND1	/*so new ents are reset multiple times to avoid weird baselines*/
//#define UF_UNUSED		UF_EXTEND2	/**/
#define UF_WEAPONFRAME_OLD	UF_EXTEND2
#define UF_VIEWANGLES	UF_EXTEND3	/**/



static unsigned int SVFTE_DeltaPredCalcBits(entity_state_t *from, entity_state_t *to)
{
	unsigned int bits = 0;
//	if (from && from->pmovetype != to->pmovetype)
//		bits |= UFP_MOVETYPE;

//	if (to->movement[0])
//		bits |= UFP_FORWARD;
//	if (to->movement[1])
//		bits |= UFP_SIDE;
//	if (to->movement[2])
//		bits |= UFP_UP;
	if (to->velocity[0])
		bits |= UFP_VELOCITYXY;
	if (to->velocity[1])
		bits |= UFP_VELOCITYXY;
	if (to->velocity[2])
		bits |= UFP_VELOCITYZ;
//	if (to->msec)
//		bits |= UFP_MSEC;

	return bits;
}

static unsigned int SVFTE_DeltaCalcBits(entity_state_t *from, entity_state_t *to)
{
	unsigned int bits = 0;

	if (from->pmovetype != to->pmovetype)
		bits |= UF_PREDINFO|UF_MOVETYPE;
/*	if (from->weaponframe != to->weaponframe)
		bits |= UF_PREDINFO|UF_WEAPONFRAME_OLD;
*/	if (to->pmovetype)
	{
		if (SVFTE_DeltaPredCalcBits(from, to))
			bits |= UF_PREDINFO;

		//moving players get extra data forced upon them which is not deltatracked
		if ((bits & UF_PREDINFO) && (from->velocity[0] || from->velocity[1] || from->velocity[2]))
		{
			//if we've got player movement then write the origin anyway, to cover packetloss
			bits |= UF_ORIGINXY | UF_ORIGINZ;
			//and force angles too, if its not us
//			if (host_client != svs.clients + to->number-1)
//				bits |= UF_ANGLESXZ | UF_ANGLESY;
		}
	}




	if (to->origin[0] != from->origin[0])
		bits |= UF_ORIGINXY;
	if (to->origin[1] != from->origin[1])
		bits |= UF_ORIGINXY;
	if (to->origin[2] != from->origin[2])
		bits |= UF_ORIGINZ;

	if (to->angles[0] != from->angles[0])
		bits |= UF_ANGLESXZ;
	if (to->angles[1] != from->angles[1])
		bits |= UF_ANGLESY;
	if (to->angles[2] != from->angles[2])
		bits |= UF_ANGLESXZ;


	if (to->modelindex != from->modelindex)
		bits |= UF_MODEL;
	if (to->frame != from->frame)
		bits |= UF_FRAME;
	if (to->skin != from->skin)
		bits |= UF_SKIN;
	if (to->colormap != from->colormap)
		bits |= UF_COLORMAP;
	if (to->effects != from->effects)
		bits |= UF_EFFECTS;
	if (to->eflags != from->eflags)
		bits |= UF_FLAGS;
//	if (to->solidsize != from->solidsize)
//		bits |= UF_SOLID;

//	if (to->scale != from->scale)
//		bits |= UF_SCALE;
	if (to->alpha != from->alpha)
		bits |= UF_ALPHA;
//	if (to->fatness != from->fatness)
//		bits |= UF_FATNESS;

//	if (to->hexen2flags != from->hexen2flags || to->abslight != from->abslight)
//		bits |= UF_DRAWFLAGS;

//	if (to->bonecount != from->bonecount || (to->bonecount && memcmp(frombonedata+from->boneoffset, tobonedata+to->boneoffset, to->bonecount*sizeof(short)*7)))
//		bits |= UF_BONEDATA;
//	if (!to->bonecount && (to->basebone != from->basebone || to->baseframe != from->baseframe))
//		bits |= UF_BONEDATA;

//	if (to->colormod[0]!=from->colormod[0]||to->colormod[1]!=from->colormod[1]||to->colormod[2]!=from->colormod[2])
//		bits |= UF_COLORMOD;

//	if (to->glowsize!=from->glowsize||to->glowcolour!=from->glowcolour||to->glowmod[0]!=from->glowmod[0]||to->glowmod[1]!=from->glowmod[1]||to->glowmod[2]!=from->glowmod[2])
//		bits |= UF_GLOW;

//	if (to->tagentity != from->tagentity || to->tagindex != from->tagindex)
//		bits |= UF_TAGINFO;

//	if (to->light[0] != from->light[0] || to->light[1] != from->light[1] || to->light[2] != from->light[2] || to->light[3] != from->light[3] || to->lightstyle != from->lightstyle || to->lightpflags != from->lightpflags)
//		bits |= UF_LIGHT;

	if (to->traileffectnum != from->traileffectnum)
		bits |= UF_TRAILEFFECT;

//	if (to->modelindex2 != from->modelindex2)
//		bits |= UF_MODELINDEX2;

//	if (to->gravitydir[0] != from->gravitydir[0] || to->gravitydir[1] != from->gravitydir[1])
//		bits |= UF_GRAVITYDIR;

	return bits;
}

static void SVFTE_WriteEntityUpdate(unsigned int bits, entity_state_t *state, sizebuf_t *msg, unsigned int pext2)
{
	unsigned int predbits = 0;
	if (bits & UF_MOVETYPE)
	{
		bits &= ~UF_MOVETYPE;
		predbits |= UFP_MOVETYPE;
	}
	if (pext2 & PEXT2_PREDINFO)
	{
		if (bits & UF_VIEWANGLES)
		{
			bits &= ~UF_VIEWANGLES;
			bits |= UF_PREDINFO;
			predbits |= UFP_VIEWANGLE;
		}
	}
	else
	{
		if (bits & UF_VIEWANGLES)
		{
			bits &= ~UF_VIEWANGLES;
			bits |= UF_PREDINFO;
		}
		if (bits & UF_WEAPONFRAME_OLD)
		{
			bits &= ~UF_WEAPONFRAME_OLD;
			predbits |= UFP_WEAPONFRAME_OLD;
		}
	}

//	if (!(pext2 & PEXT2_NEWSIZEENCODING))	//was added at the same time
		bits &= ~UF_BONEDATA;

	/*check if we need more precision for some things*/
	if ((bits & UF_MODEL) && state->modelindex > 255)
		bits |= UF_16BIT;
//	if ((bits & UF_SKIN) && state->skin > 255)
//		bits |= UF_16BIT;
	if ((bits & UF_FRAME) && state->frame > 255)
		bits |= UF_16BIT;

	/*convert effects bits to higher lengths if needed*/
	if (bits & UF_EFFECTS)
	{
		if (state->effects & 0xffff0000) /*both*/
			bits |= UF_EFFECTS | UF_EFFECTS2;
		else if (state->effects & 0x0000ff00) /*2 only*/
			bits = (bits & ~UF_EFFECTS) | UF_EFFECTS2;
	}
	if (bits & 0xff000000)
		bits |= UF_EXTEND3;
	if (bits & 0x00ff0000)
		bits |= UF_EXTEND2;
	if (bits & 0x0000ff00)
		bits |= UF_EXTEND1;

	MSG_WriteByte(msg, (bits>>0) & 0xff);
	if (bits & UF_EXTEND1)
		MSG_WriteByte(msg, (bits>>8) & 0xff);
	if (bits & UF_EXTEND2)
		MSG_WriteByte(msg, (bits>>16) & 0xff);
	if (bits & UF_EXTEND3)
		MSG_WriteByte(msg, (bits>>24) & 0xff);

	if (bits & UF_FRAME)
	{
		if (bits & UF_16BIT)
			MSG_WriteShort(msg, state->frame);
		else
			MSG_WriteByte(msg, state->frame);
	}
	if (bits & UF_ORIGINXY)
	{
		MSG_WriteCoord(msg, state->origin[0], sv.protocolflags);
		MSG_WriteCoord(msg, state->origin[1], sv.protocolflags);
	}
	if (bits & UF_ORIGINZ)
		MSG_WriteCoord(msg, state->origin[2], sv.protocolflags);

	if ((bits & UF_PREDINFO) && !(pext2 & PEXT2_PREDINFO))
	{	/*if we have pred info, (always) use more precise angles*/
		if (bits & UF_ANGLESXZ)
		{
			MSG_WriteAngle16(msg, state->angles[0], sv.protocolflags);
			MSG_WriteAngle16(msg, state->angles[2], sv.protocolflags);
		}
		if (bits & UF_ANGLESY)
			MSG_WriteAngle16(msg, state->angles[1], sv.protocolflags);
	}
	else
	{
		if (bits & UF_ANGLESXZ)
		{
			MSG_WriteAngle(msg, state->angles[0], sv.protocolflags);
			MSG_WriteAngle(msg, state->angles[2], sv.protocolflags);
		}
		if (bits & UF_ANGLESY)
			MSG_WriteAngle(msg, state->angles[1], sv.protocolflags);
	}

	if ((bits & (UF_EFFECTS|UF_EFFECTS2)) == (UF_EFFECTS|UF_EFFECTS2))
		MSG_WriteLong(msg, state->effects);
	else if (bits & UF_EFFECTS2)
		MSG_WriteShort(msg, state->effects);
	else if (bits & UF_EFFECTS)
		MSG_WriteByte(msg, state->effects);

	if (bits & UF_PREDINFO)
	{
		/*movetype is set above somewhere*/
		predbits |= SVFTE_DeltaPredCalcBits(NULL, state);

		MSG_WriteByte(msg, predbits);
/*
		if (predbits & UFP_FORWARD)
			MSG_WriteShort(msg, state->movement[0]);
		if (predbits & UFP_SIDE)
			MSG_WriteShort(msg, state->movement[1]);
		if (predbits & UFP_UP)
			MSG_WriteShort(msg, state->movement[2]);
*/		if (predbits & UFP_MOVETYPE)
			MSG_WriteByte(msg, state->pmovetype);
		if (predbits & UFP_VELOCITYXY)
		{
			MSG_WriteShort(msg, state->velocity[0]);
			MSG_WriteShort(msg, state->velocity[1]);
		}
		if (predbits & UFP_VELOCITYZ)
			MSG_WriteShort(msg, state->velocity[2]);
/*		if (predbits & UFP_MSEC)
			MSG_WriteByte(msg, state->msec);
		if (pext2 & PEXT2_PREDINFO)
		{
			if (predbits & UFP_VIEWANGLE)
			{	//if we have pred info, use more precise angles
				if (bits & UF_ANGLESXZ)
				{
					MSG_WriteShort(msg, state->vangle[0]);
					MSG_WriteShort(msg, state->vangle[2]);
				}
				if (bits & UF_ANGLESY)
					MSG_WriteShort(msg, state->vangle[1]);
			}
		}
		else
		{
			if (predbits & UFP_WEAPONFRAME_OLD)
			{
				if (state->weaponframe > 127)
				{
					MSG_WriteByte(msg, 128 | (state->weaponframe & 127));
					MSG_WriteByte(msg, state->weaponframe>>7);
				}
				else
					MSG_WriteByte(msg, state->weaponframe);
			}
		}
*/
	}

	if (bits & UF_MODEL)
	{
		if (bits & UF_16BIT)
			MSG_WriteShort(msg, state->modelindex);
		else
			MSG_WriteByte(msg, state->modelindex);
	}
	if (bits & UF_SKIN)
	{
		if (bits & UF_16BIT)
			MSG_WriteShort(msg, state->skin);
		else
			MSG_WriteByte(msg, state->skin);
	}
	if (bits & UF_COLORMAP)
		MSG_WriteByte(msg, state->colormap & 0xff);

	if (bits & UF_SOLID)
	{
/*		if (pext2 & PEXT2_NEWSIZEENCODING)
		{
			if (!state->solidsize)
				MSG_WriteByte(msg, 0);
			else if (state->solidsize == ES_SOLID_BSP)
				MSG_WriteByte(msg, 1);
			else if (state->solidsize == ES_SOLID_HULL1)
				MSG_WriteByte(msg, 2);
			else if (state->solidsize == ES_SOLID_HULL2)
				MSG_WriteByte(msg, 3);
			else if (!ES_SOLID_HAS_EXTRA_BITS(state->solidsize))
			{
				MSG_WriteByte(msg, 16);
				MSG_WriteSize16(msg, state->solidsize);
			}
			else
			{
				MSG_WriteByte(msg, 32);
				MSG_WriteLong(msg, state->solidsize);
			}
		}
		else
			MSG_WriteSize16(msg, state->solidsize);
*/	}

	if (bits & UF_FLAGS)
		MSG_WriteByte(msg, state->eflags);

	if (bits & UF_ALPHA)
		MSG_WriteByte(msg, state->alpha);
//	if (bits & UF_SCALE)
//		MSG_WriteByte(msg, state->scale);
/*	if (bits & UF_BONEDATA)
	{
		short *bonedata;
		int i;
		byte bfl = 0;
		if (state->bonecount && boneptr)
			bfl |= 0x80;
		if (state->basebone || state->baseframe)
			bfl |= 0x40;
		MSG_WriteByte(msg, bfl);
		if (bfl & 0x80)
		{
			//this is NOT finalized
			MSG_WriteByte(msg, state->bonecount);
			bonedata = (short*)(boneptr + state->boneoffset);
			for (i = 0; i < state->bonecount*7; i++)
				MSG_WriteShort(msg, bonedata[i]);
		}
		if (bfl & 0x40)
		{
			MSG_WriteByte(msg, state->basebone);
			MSG_WriteShort(msg, state->baseframe);
		}
	}
	if (bits & UF_DRAWFLAGS)
	{
		MSG_WriteByte(msg, state->drawflags);
		if ((state->drawflags & MLS_MASK) == MLS_ABSLIGHT)
			MSG_WriteByte(msg, state->abslight);
	}
	if (bits & UF_TAGINFO)
	{
		MSG_WriteEntity(msg, state->tagentity);
		MSG_WriteByte(msg, state->tagindex);
	}
	if (bits & UF_LIGHT)
	{
		MSG_WriteShort (msg, state->light[0]);
		MSG_WriteShort (msg, state->light[1]);
		MSG_WriteShort (msg, state->light[2]);
		MSG_WriteShort (msg, state->light[3]);
		MSG_WriteByte (msg, state->lightstyle);
		MSG_WriteByte (msg, state->lightpflags);
	}
*/	if (bits & UF_TRAILEFFECT)
		MSG_WriteShort(msg, state->traileffectnum);

/*	if (bits & UF_COLORMOD)
	{
		MSG_WriteByte(msg, state->colormod[0]);
		MSG_WriteByte(msg, state->colormod[1]);
		MSG_WriteByte(msg, state->colormod[2]);
	}
	if (bits & UF_GLOW)
	{
		MSG_WriteByte(msg, state->glowsize);
		MSG_WriteByte(msg, state->glowcolour);
		MSG_WriteByte(msg, state->glowmod[0]);
		MSG_WriteByte(msg, state->glowmod[1]);
		MSG_WriteByte(msg, state->glowmod[2]);
	}
	if (bits & UF_FATNESS)
		MSG_WriteByte(msg, state->fatness);
	if (bits & UF_MODELINDEX2)
	{
		if (bits & UF_16BIT)
			MSG_WriteShort(msg, state->modelindex2);
		else
			MSG_WriteByte(msg, state->modelindex2);
	}

	if (bits & UF_GRAVITYDIR)
	{
		MSG_WriteByte(msg, state->gravitydir[0]);
		MSG_WriteByte(msg, state->gravitydir[1]);
	}
*/
}

static struct entity_num_state_s *snapshot_entstate;
static size_t snapshot_numents;
static size_t snapshot_maxents;

void SVFTE_DestroyFrames(client_t *client)
{
	if (client->previousentities)
		free(client->previousentities);
	client->previousentities = NULL;
	client->numpreviousentities = 0;
	client->maxpreviousentities = 0;


	if (client->pendingentities_bits)
		free(client->pendingentities_bits);
	client->pendingentities_bits = NULL;
	client->numpendingentities = 0;

	while(client->numframes > 0)
	{
		client->numframes--;
		free(client->frames[client->numframes].ents);
	}
	if (client->frames)
		free(client->frames);
	client->frames = NULL;

	client->lastacksequence = 0;
}
static void SVFTE_SetupFrames(client_t *client)
{
	size_t fr;
	//the client will clear out their stats on receipt of the svc_serverinfo packet.
	//we won't send any reliables until they receive it
	//so it should be enough to just clear these here, and they'll get their new stats with the first entity update once they're spawned
	memset(client->oldstats_i, 0, sizeof(client->oldstats_i));
	memset(client->oldstats_f, 0, sizeof(client->oldstats_f));
	client->lastmovemessage = 0;	//it'll clear this too

	if (!client->protocol_pext2)
	{
		SVFTE_DestroyFrames(client);
		return;
	}

	client->numframes = 64;	//must be power-of-two
	client->frames = malloc(sizeof(*client->frames) * client->numframes);
	client->lastacksequence = (int)0x80000000;
	memset(client->frames, 0, sizeof(*client->frames) * client->numframes);
	for (fr = 0; fr < client->numframes; fr++)
		client->frames[fr].sequence = client->lastacksequence;

	client->numpendingentities = sv.num_edicts;
	client->pendingentities_bits = malloc(sizeof(*client->pendingentities_bits) * client->numpendingentities);

	client->pendingentities_bits[0] = UF_REMOVE;
}
static void SVFTE_DroppedFrame(client_t *client, int sequence)
{
	int i;
	struct deltaframe_s *frame = &client->frames[sequence&(client->numframes-1)];
	if (frame->sequence != sequence)
		return;	//this frame was stale... client is running too far behind. we'll probably be spamming resends as a result.
	Con_Printf("Dropped sequence %i\n", sequence);
	frame->sequence = -1;
	//flag their stats for resend
	for (i = 0; i < MAX_CL_STATS/32; i++)
		client->resendstats[i] |= frame->resendstats[i];
	//flag their various entities as needing a resend too.
	for (i = 0; i < frame->numents; i++)
		client->pendingentities_bits[frame->ents[i].num] |= frame->ents[i].bits;

}
void SVFTE_Ack(client_t *client, int sequence)
{	//any gaps in the sequence need to considered dropped
	struct deltaframe_s *frame;
	int dropseq = client->lastacksequence+1;
	if (!client->numframes)
		return;	//client shouldn't be using this.
	if (sequence < client->lastacksequence)
	{
		if (sequence == -1)
			client->pendingentities_bits[0] |= UF_REMOVE;	//client wants a full resend. which might happen from it just starting to record a demo, saving it from writing all the deltas out.
//		else Con_SafePrintf("dupe or stale ack (%s, %i->%i)\n", client->name, client->lastacksequence, sequence);
		return;	//panic
	}
	if ((unsigned)(dropseq-sequence) >= client->numframes)
		dropseq = sequence - client->numframes;
	while(dropseq < sequence)
		SVFTE_DroppedFrame(client, dropseq++);
	client->lastacksequence = sequence;

	frame = &client->frames[sequence&(client->numframes-1)];
	if (frame->sequence == sequence)
	{
		frame->sequence = -1;
		host_client->ping_times[host_client->num_pings%NUM_PING_TIMES] = sv.time - frame->timestamp;
		host_client->num_pings++;
	}
}
static void SVFTE_WriteStats(client_t *client, sizebuf_t *msg)
{
	int statsi[MAX_CL_STATS];
	float statsf[MAX_CL_STATS];
	int i;
	struct deltaframe_s *frame;
	int sequence = NET_QSocketGetSequenceOut(client->netconnection);

	frame = &client->frames[sequence&(client->numframes-1)];

	if (frame->sequence == sequence-(int)client->numframes)	//client is getting behind... this may get really spammy, lets hope it clears up at some point
		SVFTE_DroppedFrame(client, frame->sequence);

	//figure out the current values in a nice easy way (yay for copying to make arrays easier!)
	SV_CalcStats(client, statsi, statsf);

	for (i = 0; i < MAX_CL_STATS; i++)
	{
		//small cleanup
		if (!statsi[i])
			statsi[i] =	statsf[i];
		else
			statsf[i] =	statsi[i];

		//if it changed flag for sending
		if (statsi[i] != client->oldstats_i[i] || statsf[i] != client->oldstats_f[i])
		{
			client->oldstats_i[i] = statsi[i];
			client->oldstats_f[i] = statsf[i];
			client->resendstats[i/32] |= 1u<<(i&31);
		}

		//if its flagged then unflag it, log it, and send it
		if (client->resendstats[i/32] & (1u<<(i&31)))
		{
			client->resendstats[i/32] &= ~(1u<<(i&31));
			frame->resendstats[i/32] |= 1u<<(i&31);

			if ((double)statsi[i] != statsf[i])
			{	//didn't round nicely, so send as a float
				MSG_WriteByte(msg, svcfte_updatestatfloat);
				MSG_WriteByte(msg, i);
				MSG_WriteFloat(msg, statsf[i]);
			}
			else
			{
				if (statsi[i] < 0 || statsi[i] > 255)
				{	//needs to be big
					MSG_WriteByte(msg, svc_updatestat);
					MSG_WriteByte(msg, i);
					MSG_WriteLong(msg, statsi[i]);
				}
				else
				{	//can be fairly small
					MSG_WriteByte(msg, svcdp_updatestatbyte);
					MSG_WriteByte(msg, i);
					MSG_WriteByte(msg, statsi[i]);
				}
			}
		}
	}
}
static void SVFTE_CalcEntityDeltas(client_t *client)
{
	struct entity_num_state_s *olds, *news, *oldstop, *newstop;

	if ((int)client->numpendingentities < sv.num_edicts)
	{
		int newmax = sv.num_edicts+64;
		client->pendingentities_bits = realloc(client->pendingentities_bits, sizeof(*client->pendingentities_bits) * newmax);
		memset(client->pendingentities_bits+client->numpendingentities, 0, sizeof(*client->pendingentities_bits)*(newmax-client->numpendingentities));
		client->numpendingentities = newmax;
	}

	//if we're clearing the list and starting from scratch, just wipe all lingering state
	if (client->pendingentities_bits[0] & UF_REMOVE)
	{
		client->numpreviousentities = 0;
		client->pendingentities_bits[0] = UF_REMOVE;
	}

	news = snapshot_entstate;
	newstop = news + snapshot_numents;
	olds = client->previousentities;
	oldstop = olds+client->numpreviousentities;

	//we have two sets of entity state, pvs culled etc already.
	//figure out which flags changed,
	for (;;)
	{
		if (olds==oldstop && news==newstop)
			break;
		if (news==newstop || (olds!=oldstop && olds->num < news->num))
		{
			//old ent is no longer visible, so flag for removal.
			client->pendingentities_bits[olds->num] = UF_REMOVE;
			olds++;
		}
		else if (olds==oldstop || (news!=newstop && news->num < olds->num))
		{
			//new ent is new this frame, so reset everything.
			client->pendingentities_bits[news->num] = UF_RESET | UF_RESET2;
			//don't need to calc the other bits here, resets are enough
			news++;
		}
		else
		{	//simple entity delta
			//its flagged for removing, that's weird... must be some killer packetloss. turn that back into a reset or something
			if (client->pendingentities_bits[news->num] & UF_REMOVE)
				client->pendingentities_bits[news->num] = (client->pendingentities_bits[news->num] & ~UF_REMOVE) | UF_RESET2;
			client->pendingentities_bits[news->num] |= SVFTE_DeltaCalcBits(&olds->state, &news->state);
			news++;
			olds++;
		}
	}

	//now we know what flags to apply, the client needs a copy of that state for the next frame too.
	//outgoing data can just read off these states too, instead of needing to hit the edicts memory (which may be spread over multiple allocations, yay cache).
	//to avoid a potentially large memcopy, I'm just going to swap these buffers. 
	olds = client->previousentities;
	oldstop = olds + client->maxpreviousentities;

	client->previousentities = snapshot_entstate;
	client->numpreviousentities = snapshot_numents;
	client->maxpreviousentities = snapshot_maxents;

	snapshot_entstate = olds;
	snapshot_numents = 0;
	snapshot_maxents = oldstop-olds;
}
static void SVFTE_WriteEntitiesToClient(client_t *client, sizebuf_t *msg, size_t overflowsize)
{
	struct entity_num_state_s *state, *stateend;
	unsigned int bits, logbits;
	size_t entnum;
	int sequence = NET_QSocketGetSequenceOut(client->netconnection);
	size_t origmaxsize = msg->maxsize;
	size_t rollbacksize;	//I'm too lazy to figure out sizes (especially if someone updates this for bone states or whatever)
	struct deltaframe_s *frame = &client->frames[sequence&(client->numframes-1)];
	frame->sequence = sequence;	//so we know that it wasn't stale later.
	frame->timestamp = sv.time;

	msg->maxsize = overflowsize;

	state = client->previousentities;
	stateend = state + client->numpreviousentities;

	MSG_WriteByte(msg, svcfte_updateentities);

	frame->numents = 0;
	if (client->protocol_pext2 & PEXT2_PREDINFO)
		MSG_WriteShort(msg, (client->lastmovemessage&0xffff));
	MSG_WriteFloat(msg, sv.time);	//should be the time the last physics frame was run.
	for (entnum = client->snapshotresume; entnum < client->numpendingentities; entnum++)
	{
		bits = client->pendingentities_bits[entnum];
		if (!(bits & ~UF_RESET2))
			continue;	//nothing to send (if reset2 is still set, then leave it pending until there's more data

		rollbacksize = msg->cursize;
		client->pendingentities_bits[entnum] = 0;
		logbits = 0;
		if (bits & UF_REMOVE)
		{
			MSG_WriteShort(msg, 0x8000|entnum);
			logbits = UF_REMOVE;
		}
		else
		{
			while (state<stateend && state->num < entnum)
				state++;
			if (state<stateend && state->num == entnum)
			{
				if (bits & UF_RESET2)
				{
					/*if reset2, then this is the second packet sent to the client and should have a forced reset (but which isn't tracked)*/
					logbits = bits & ~UF_RESET2;
					bits = UF_RESET | SVFTE_DeltaCalcBits(&EDICT_NUM(entnum)->baseline, &state->state);
//					Con_Printf("RESET2 %i @ %i\n", j, sequence);
				}
				else if (bits & UF_RESET)
				{
					/*flag the entity for the next packet, so we always get two resets when it appears, to reduce the effects of packetloss on seeing rockets etc*/
					client->pendingentities_bits[entnum] = UF_RESET2;
					bits = UF_RESET | SVFTE_DeltaCalcBits(&EDICT_NUM(entnum)->baseline, &state->state);
					logbits = UF_RESET;
//					Con_Printf("RESET %i @ %i\n", j, sequence);
				}
				else
					logbits = bits;

				MSG_WriteShort(msg, entnum);
//				SV_EmitDeltaEntIndex(msg, j, false, true);
				SVFTE_WriteEntityUpdate(bits, &state->state, msg, client->protocol_pext2);
			}
		}

		if ((size_t)msg->cursize + 2 > origmaxsize)
		{
			msg->cursize = rollbacksize;	//roll back
			client->pendingentities_bits[entnum] |= logbits;	//make sure those bits get re-applied later.
			break;
		}
		if (frame->numents == frame->maxents)
		{
			frame->maxents += 64;
			frame->ents = realloc(frame->ents, sizeof(*frame->ents)*frame->maxents);
		}
		frame->ents[frame->numents].num = entnum;
		frame->ents[frame->numents].bits = logbits;
		frame->numents++;
	}
	msg->maxsize = origmaxsize;
	MSG_WriteShort(msg, 0);	//eom

	//remember how far we got, so we can keep things flushed, instead of only updating the first N entities.
	client->snapshotresume = (entnum<client->numpendingentities?entnum:0);


	if (msg->cursize > 1024 && dev_peakstats.packetsize <= 1024)
		Con_DWarning ("%i byte packet exceeds standard limit of 1024.\n", msg->cursize);
	dev_stats.packetsize = msg->cursize;
	dev_peakstats.packetsize = q_max(msg->cursize, dev_peakstats.packetsize);
}

byte *SV_FatPVS (vec3_t org, qmodel_t *worldmodel);
void SVFTE_BuildSnapshotForClient (edict_t	*clent)
{
	int				e, i;
	byte			*pvs;
	vec3_t			org;
	edict_t			*ent;
	int				maxmodels = MAX_MODELS;
	eval_t			*val;
	unsigned char	eflags;
	int proged = EDICT_TO_PROG(clent);

	struct entity_num_state_s *ents = snapshot_entstate;
	size_t numents = 0;
	size_t maxents = snapshot_maxents;

// find the client's PVS
	VectorAdd (clent->v.origin, clent->v.view_ofs, org);
	pvs = SV_FatPVS (org, sv.worldmodel);
	
// send over all entities (excpet the client) that touch the pvs
	ent = NEXT_EDICT(sv.edicts);
	for (e=1 ; e<sv.num_edicts ; e++, ent = NEXT_EDICT(ent))
	{
		eflags = 0;
		if (ent != clent)	// clent is ALLWAYS sent
		{
			// ignore ents without visible models
			if (!ent->v.modelindex || !PR_GetString(ent->v.model)[0])
				continue;

			//johnfitz -- don't send model>255 entities if protocol is 15
			if ((int)ent->v.modelindex >= maxmodels)
				continue;

			val = GetEdictFieldValue(ent, pr_extfields.viewmodelforclient);
			if (val && val->edict == proged)
				eflags |= EFLAGS_VIEWMODEL;
			else if (val && val->edict)
				continue;
			else
			{
				// ignore if not touching a PV leaf
				for (i=0 ; i < ent->num_leafs ; i++)
					if (pvs[ent->leafnums[i] >> 3] & (1 << (ent->leafnums[i]&7) ))
						break;
				
				// ericw -- added ent->num_leafs < MAX_ENT_LEAFS condition.
				//
				// if ent->num_leafs == MAX_ENT_LEAFS, the ent is visible from too many leafs
				// for us to say whether it's in the PVS, so don't try to vis cull it.
				// this commonly happens with rotators, because they often have huge bboxes
				// spanning the entire map, or really tall lifts, etc.
				if (i == ent->num_leafs && ent->num_leafs < MAX_ENT_LEAFS)
					continue;		// not visible
			}
		}

		//johnfitz -- alpha
		val = GetEdictFieldValue(ent, pr_extfields.alpha);
		if (val)
			ent->alpha = ENTALPHA_ENCODE(val->_float);
		//don't send invisible entities unless they have effects
		if (ent->alpha == ENTALPHA_ZERO && !ent->v.effects)
			continue;
		//johnfitz

//		val = GetEdictFieldValue(ent, pr_extfields.exteriormodelforclient);
//		if (val && val->edict == proged)
//			eflags |= EFLAGS_EXTERIORMODEL;

		//okay, we care about this entity.

		if (numents == maxents)
		{
			maxents += 64;
			ents = realloc(ents, maxents*sizeof(*ents));
		}
		
		ents[numents].num = e;
		VectorCopy(ent->v.origin, ents[numents].state.origin);
		VectorCopy(ent->v.angles, ents[numents].state.angles);
		ents[numents].state.modelindex = ent->v.modelindex;
		ents[numents].state.frame = ent->v.frame;
		ents[numents].state.colormap = ent->v.colormap;
		ents[numents].state.skin = ent->v.skin;
		ents[numents].state.alpha = ent->alpha;
		if ((val = GetEdictFieldValue(ent, pr_extfields.traileffectnum)))
			ents[numents].state.traileffectnum = val->_float;
		else
			ents[numents].state.traileffectnum = 0;
		ents[numents].state.effects = ent->v.effects;
		if (!ent->v.movetype || ent->v.movetype == MOVETYPE_STEP)
			eflags |= EFLAGS_STEP;


		if (ent == clent)	//add velocity, but we only care for the local player (should add prediction for other entities some time too).
		{
			ents[numents].state.pmovetype = ent->v.movetype;
			if ((int)ent->v.flags & FL_ONGROUND)
				eflags |= EFLAGS_ONGROUND;
			ents[numents].state.velocity[0] = ent->v.velocity[0]*8;
			ents[numents].state.velocity[1] = ent->v.velocity[1]*8;
			ents[numents].state.velocity[2] = ent->v.velocity[2]*8;
		}
		else
		{
			ents[numents].state.pmovetype = 0;
			ents[numents].state.velocity[0] = ents[numents].state.velocity[1] = ents[numents].state.velocity[2] = 0;
		}
		ents[numents].state.eflags = eflags;
		numents++;
	}

	snapshot_entstate = ents;
	snapshot_numents = numents;
	snapshot_maxents = maxents;
}

void SV_Pext_f(void);

/*
===============
SV_Protocol_f
===============
*/
void SV_Protocol_f (void)
{
	int i;
	const char *s;

	switch (Cmd_Argc())
	{
	case 1:
		//"FTE+15" or "15", just to be explicit about it
		Con_Printf ("\"sv_protocol\" is \"%s%i\"\n", sv_protocol_pext2?"FTE+":"", sv_protocol);
		break;
	case 2:
		s = Cmd_Argv(1);
		if (!q_strncasecmp(s, "FTE+", 4)||!q_strncasecmp(s, "FTE-", 4))
		{
			i = strtol(s+4, (char**)&s, 0);
			s = "+";
		}
		else if (!q_strncasecmp(s, "Base-", 5)||!q_strncasecmp(s, "Base+", 5))
		{
			i = strtol(s+5, (char**)&s, 0);
			s = "-";
		}
		else if (*s == '-')
		{
			i = strtol(s+1, (char**)&s, 0);
			s = "-";
		}
		else
			i = strtol(s, (char**)&s, 0);

		if (i != PROTOCOL_NETQUAKE && i != PROTOCOL_FITZQUAKE && i != PROTOCOL_RMQ)
			Con_Printf ("sv_protocol must be %i or %i or %i.\nProtocol may be prefixed with FTE+ or Base- to enable/disable FTE extensions.", PROTOCOL_NETQUAKE, PROTOCOL_FITZQUAKE, PROTOCOL_RMQ);
		else
		{
			if (*s == '+')
				sv_protocol_pext2 = PEXT2_SUPPORTED_SERVER;
			else if (*s == '-')
				sv_protocol_pext2 = 0;
			sv_protocol = i;
			if (sv.active)
				Con_Printf ("changes will not take effect until the next level load.\n");
		}
		break;
	default:
		Con_SafePrintf ("usage: sv_protocol <protocol>\n");
		break;
	}
}

/*
===============
SV_Init
===============
*/
void SV_Init (void)
{
	int		i;
	const char	*p;
	extern	cvar_t	sv_maxvelocity;
	extern	cvar_t	sv_gravity;
	extern	cvar_t	sv_nostep;
	extern	cvar_t	sv_freezenonclients;
	extern	cvar_t	sv_friction;
	extern	cvar_t	sv_edgefriction;
	extern	cvar_t	sv_stopspeed;
	extern	cvar_t	sv_maxspeed;
	extern	cvar_t	sv_accelerate;
	extern	cvar_t	sv_idealpitchscale;
	extern	cvar_t	sv_aim;
	extern	cvar_t	sv_altnoclip; //johnfitz
	extern	cvar_t	sv_public;	//spike
	extern	cvar_t	sv_reportheartbeats;	//spike
	extern	cvar_t	com_protocolname;	//spike
	extern	cvar_t	net_masters[];	//spike


	sv.edicts = NULL; // ericw -- sv.edicts switched to use malloc()

	Cvar_RegisterVariable (&sv_maxvelocity);
	Cvar_RegisterVariable (&sv_gravity);
	Cvar_RegisterVariable (&sv_friction);
	Cvar_SetCallback (&sv_gravity, Host_Callback_Notify);
	Cvar_SetCallback (&sv_friction, Host_Callback_Notify);
	Cvar_RegisterVariable (&sv_edgefriction);
	Cvar_RegisterVariable (&sv_stopspeed);
	Cvar_RegisterVariable (&sv_maxspeed);
	Cvar_SetCallback (&sv_maxspeed, Host_Callback_Notify);
	Cvar_RegisterVariable (&sv_accelerate);
	Cvar_RegisterVariable (&sv_idealpitchscale);
	Cvar_RegisterVariable (&sv_aim);
	Cvar_RegisterVariable (&sv_nostep);
	Cvar_RegisterVariable (&sv_freezenonclients);
	Cvar_RegisterVariable (&pr_checkextension);
	Cvar_RegisterVariable (&sv_altnoclip); //johnfitz

	Cvar_RegisterVariable (&sv_public);
	Cvar_RegisterVariable (&sv_reportheartbeats);
	Cvar_RegisterVariable (&com_protocolname);
	for (i = 0; net_masters[i].name; i++)
		Cvar_RegisterVariable (&net_masters[i]);

	Cmd_AddCommand_ClientCommand("pext", SV_Pext_f);
	Cmd_AddCommand ("sv_protocol", &SV_Protocol_f); //johnfitz

	for (i=0 ; i<MAX_MODELS ; i++)
		sprintf (localmodels[i], "*%i", i);

	sv_protocol_pext2 = PEXT2_SUPPORTED_SERVER;
	i = COM_CheckParm ("-protocol");
	if (i && i < com_argc - 1)
		sv_protocol = atoi (com_argv[i + 1]);
	switch (sv_protocol)
	{
	case PROTOCOL_NETQUAKE:
		p = "NetQuake";
		break;
	case PROTOCOL_FITZQUAKE:
		p = "FitzQuake";
		break;
	case PROTOCOL_RMQ:
		p = "RMQ";
		break;
	default:
		Sys_Error ("Bad protocol version request %i. Accepted values: %i, %i, %i.",
				sv_protocol, PROTOCOL_NETQUAKE, PROTOCOL_FITZQUAKE, PROTOCOL_RMQ);
		p = "Unknown";
	}
	Sys_Printf ("Server using protocol %i%s (%s%s)\n", sv_protocol, sv_protocol_pext2?"+":"", sv_protocol_pext2?"FTE-":"", p);
}

/*
=============================================================================

EVENT MESSAGES

=============================================================================
*/

/*
==================
SV_StartParticle

Make sure the event gets sent to all clients
==================
*/
void SV_StartParticle (vec3_t org, vec3_t dir, int color, int count)
{
	int		i, v;

	if (sv.datagram.cursize > MAX_DATAGRAM-16)
		return;
	MSG_WriteByte (&sv.datagram, svc_particle);
	MSG_WriteCoord (&sv.datagram, org[0], sv.protocolflags);
	MSG_WriteCoord (&sv.datagram, org[1], sv.protocolflags);
	MSG_WriteCoord (&sv.datagram, org[2], sv.protocolflags);
	for (i=0 ; i<3 ; i++)
	{
		v = dir[i]*16;
		if (v > 127)
			v = 127;
		else if (v < -128)
			v = -128;
		MSG_WriteChar (&sv.datagram, v);
	}
	MSG_WriteByte (&sv.datagram, count);
	MSG_WriteByte (&sv.datagram, color);
}

/*
==================
SV_StartSound

Each entity can have eight independant sound sources, like voice,
weapon, feet, etc.

Channel 0 is an auto-allocate channel, the others override anything
allready running on that entity/channel pair.

An attenuation of 0 will play full volume everywhere in the level.
Larger attenuations will drop off.  (max 4 attenuation)

==================
*/
void SV_StartSound (edict_t *entity, float *origin, int channel, const char *sample, int volume, float attenuation)
{
	int			sound_num, ent;
	int			i, field_mask;

	if (volume < 0)
		Host_Error ("SV_StartSound: volume = %i", volume);
	else if (volume > 255)
	{
		volume = 255;
		Con_Printf ("SV_StartSound: volume = %i\n", volume);
	}

	if (attenuation < 0 || attenuation > 4)
		Host_Error ("SV_StartSound: attenuation = %f", attenuation);

	if (channel < 0 || channel > 255)
		Host_Error ("SV_StartSound: channel = %i", channel);
	else if (channel > 7)
		Con_Printf ("SV_StartSound: channel = %i\n", channel);

	if (sv.datagram.cursize > MAX_DATAGRAM-16)
		return;

// find precache number for sound
	for (sound_num = 1; sound_num < MAX_SOUNDS && sv.sound_precache[sound_num]; sound_num++)
	{
		if (!strcmp(sample, sv.sound_precache[sound_num]))
			break;
	}

	if (sound_num == MAX_SOUNDS || !sv.sound_precache[sound_num])
	{
		Con_Printf ("SV_StartSound: %s not precacheed\n", sample);
		return;
	}

	ent = NUM_FOR_EDICT(entity);

	field_mask = 0;
	if (volume != DEFAULT_SOUND_PACKET_VOLUME)
		field_mask |= SND_VOLUME;
	if (attenuation != DEFAULT_SOUND_PACKET_ATTENUATION)
		field_mask |= SND_ATTENUATION;

	//johnfitz -- PROTOCOL_FITZQUAKE
	if (ent >= 8192 || channel >= 8)
	{
		if (sv.protocol == PROTOCOL_NETQUAKE)
			return; //don't send any info protocol can't support
		else
			field_mask |= SND_LARGEENTITY;
	}
	if (sound_num >= 256)
	{
		if (sv.protocol == PROTOCOL_NETQUAKE)
			return; //don't send any info protocol can't support
		else
			field_mask |= SND_LARGESOUND;
	}
	//johnfitz

// directed messages go only to the entity the are targeted on
	MSG_WriteByte (&sv.datagram, svc_sound);
	MSG_WriteByte (&sv.datagram, field_mask);
	if (field_mask & SND_VOLUME)
		MSG_WriteByte (&sv.datagram, volume);
	if (field_mask & SND_ATTENUATION)
		MSG_WriteByte (&sv.datagram, attenuation*64);

	//johnfitz -- PROTOCOL_FITZQUAKE
	if (field_mask & SND_LARGEENTITY)
	{
		MSG_WriteShort (&sv.datagram, ent);
		MSG_WriteByte (&sv.datagram, channel);
	}
	else
		MSG_WriteShort (&sv.datagram, (ent<<3) | channel);
	if (field_mask & SND_LARGESOUND)
		MSG_WriteShort (&sv.datagram, sound_num);
	else
		MSG_WriteByte (&sv.datagram, sound_num);
	//johnfitz

	for (i = 0; i < 3; i++)
	{
		if (origin)
			MSG_WriteCoord (&sv.datagram, origin[i], sv.protocolflags);
		else
			MSG_WriteCoord (&sv.datagram, entity->v.origin[i]+0.5*(entity->v.mins[i]+entity->v.maxs[i]), sv.protocolflags);
	}
}

/*
==============================================================================

CLIENT SPAWNING

==============================================================================
*/

/*
================
SV_SendServerinfo

Sends the first message from the server to a connected client.
This will be sent on the initial connection and upon each server load.
================
*/
void SV_SendServerinfo (client_t *client)
{
	const char		**s;
	char			message[2048];
	int				i; //johnfitz
	int				precachelimit;

	if (!sv_protocol_pext2)
	{	//server disabled pext completely, don't bother trying.
		//make sure we try reenabling it again on the next map though. mwahaha.
		client->pextknown = false;
	}
	else if (!client->pextknown)
	{
		client->spawned = false;
		MSG_WriteByte (&client->message, svc_stufftext);
		MSG_WriteString (&client->message, "cmd pext\n");
		client->sendsignon = true;
		return;
	}
	client->protocol_pext2 &= sv_protocol_pext2;

	MSG_WriteByte (&client->message, svc_print);
	sprintf (message, "%c\nFITZQUAKE %1.2f SERVER (%i CRC)\n", 2, FITZQUAKE_VERSION, pr_crc); //johnfitz -- include fitzquake version
	MSG_WriteString (&client->message,message);

	MSG_WriteByte (&client->message, svc_serverinfo);

	if (client->protocol_pext2)
	{	//pext stuff takes the form of modifiers to an underlaying protocol
		MSG_WriteLong (&client->message, PROTOCOL_FTE_PEXT2);
		MSG_WriteLong (&client->message, client->protocol_pext2);	//active extensions that the client needs to look out for
	}

	MSG_WriteLong (&client->message, sv.protocol); //johnfitz -- sv.protocol instead of PROTOCOL_VERSION

	if (sv.protocol == PROTOCOL_RMQ)
	{
		// mh - now send protocol flags so that the client knows the protocol features to expect
		MSG_WriteLong (&client->message, sv.protocolflags);
	}

	MSG_WriteByte (&client->message, svs.maxclients);

	if (!coop.value && deathmatch.value)
		MSG_WriteByte (&client->message, GAME_DEATHMATCH);
	else
		MSG_WriteByte (&client->message, GAME_COOP);

	MSG_WriteString (&client->message, PR_GetString(sv.edicts->v.message));

	//johnfitz -- only send the first 256 model and sound precaches if protocol is 15
	precachelimit = (sv.protocol != PROTOCOL_NETQUAKE || (client->protocol_pext2 & PEXT2_REPLACEMENTDELTAS))?MAX_MODELS:256;
	for (i=0,s = sv.model_precache+1 ; *s && i < precachelimit; s++,i++)
		MSG_WriteString (&client->message, *s);
	MSG_WriteByte (&client->message, 0);

	precachelimit = (sv.protocol != PROTOCOL_NETQUAKE || (client->protocol_pext2 & PEXT2_REPLACEMENTDELTAS))?MAX_SOUNDS:256;
	for (i=0,s = sv.sound_precache+1 ; *s && i < precachelimit; s++,i++)
		MSG_WriteString (&client->message, *s);
	MSG_WriteByte (&client->message, 0);
	//johnfitz

// send music
	MSG_WriteByte (&client->message, svc_cdtrack);
	MSG_WriteByte (&client->message, sv.edicts->v.sounds);
	MSG_WriteByte (&client->message, sv.edicts->v.sounds);

// set view
	MSG_WriteByte (&client->message, svc_setview);
	MSG_WriteShort (&client->message, NUM_FOR_EDICT(client->edict));

	MSG_WriteByte (&client->message, svc_signonnum);
	MSG_WriteByte (&client->message, 1);

	client->sendsignon = true;
	client->spawned = false;		// need prespawn, spawn, etc

	SVFTE_SetupFrames(client);
}

void SV_Pext_f(void)
{
	if (cmd_source == src_command)
	{
		Cmd_ForwardToServer ();
		return;
	}

	if (!host_client->pextknown)
	{
		int i;
		int key;
		int value;
		for (i = 1; i < Cmd_Argc(); i+=2)
		{
			key = strtoul(Cmd_Argv(i), NULL, 0);
			value = strtoul(Cmd_Argv(i+1), NULL, 0);

			if (key == PROTOCOL_FTE_PEXT2)
				host_client->protocol_pext2 = value & PEXT2_SUPPORTED_SERVER;
			//else some other extension that we don't know
		}

		host_client->pextknown = true;
		SV_SendServerinfo(host_client);
	}
}

/*
================
SV_ConnectClient

Initializes a client_t for a new net connection.  This will only be called
once for a player each game, not once for each level change.
================
*/
void SV_ConnectClient (int clientnum)
{
	edict_t			*ent;
	client_t		*client;
	int				edictnum;
	struct qsocket_s *netconnection;
	int				i;
	float			spawn_parms[NUM_SPAWN_PARMS];

	client = svs.clients + clientnum;

	if (client->netconnection)
		Con_DPrintf ("Client %s connected\n", NET_QSocketGetAddressString(client->netconnection));
	else
		Con_DPrintf ("Bot connected\n");

	edictnum = clientnum+1;

	ent = EDICT_NUM(edictnum);

// set up the client_t
	netconnection = client->netconnection;
	net_activeconnections++;

	if (sv.loadgame)
		memcpy (spawn_parms, client->spawn_parms, sizeof(spawn_parms));
	memset (client, 0, sizeof(*client));
	client->netconnection = netconnection;

	strcpy (client->name, "unconnected");
	client->active = true;
	client->spawned = false;
	client->edict = ent;
	client->message.data = client->msgbuf;
	client->message.maxsize = sizeof(client->msgbuf);
	client->message.allowoverflow = true;		// we can catch it

	client->datagram.data = client->datagram_buf;
	client->datagram.maxsize = sizeof(client->datagram_buf);
	client->datagram.allowoverflow = true;		//simply ignored on overflow

	client->pextknown = false;
	client->protocol_pext2 = 0;

	if (sv.loadgame)
		memcpy (client->spawn_parms, spawn_parms, sizeof(spawn_parms));
	else
	{
	// call the progs to get default spawn parms for the new client
		PR_ExecuteProgram (pr_global_struct->SetNewParms);
		for (i=0 ; i<NUM_SPAWN_PARMS ; i++)
			client->spawn_parms[i] = (&pr_global_struct->parm1)[i];
	}

	SV_SendServerinfo (client);
}


/*
===================
SV_CheckForNewClients

===================
*/
void SV_CheckForNewClients (void)
{
	struct qsocket_s	*ret;
	int				i;

//
// check for new connections
//
	while (1)
	{
		ret = NET_CheckNewConnections ();
		if (!ret)
			break;

	//
	// init a new client structure
	//
		for (i=0 ; i<svs.maxclients ; i++)
			if (!svs.clients[i].active)
				break;
		if (i == svs.maxclients)
			Sys_Error ("Host_CheckForNewClients: no free clients");

		svs.clients[i].netconnection = ret;
		SV_ConnectClient (i);
	}
}


/*
===============================================================================

FRAME UPDATES

===============================================================================
*/

/*
==================
SV_ClearDatagram

==================
*/
void SV_ClearDatagram (void)
{
	SZ_Clear (&sv.datagram);
}

/*
=============================================================================

The PVS must include a small area around the client to allow head bobbing
or other small motion on the client side.  Otherwise, a bob might cause an
entity that should be visible to not show up, especially when the bob
crosses a waterline.

=============================================================================
*/

int		fatbytes;
byte	fatpvs[MAX_MAP_LEAFS/8];

void SV_AddToFatPVS (vec3_t org, mnode_t *node, qmodel_t *worldmodel) //johnfitz -- added worldmodel as a parameter
{
	int		i;
	byte	*pvs;
	mplane_t	*plane;
	float	d;

	while (1)
	{
	// if this is a leaf, accumulate the pvs bits
		if (node->contents < 0)
		{
			if (node->contents != CONTENTS_SOLID)
			{
				pvs = Mod_LeafPVS ( (mleaf_t *)node, worldmodel); //johnfitz -- worldmodel as a parameter
				for (i=0 ; i<fatbytes ; i++)
					fatpvs[i] |= pvs[i];
			}
			return;
		}

		plane = node->plane;
		d = DotProduct (org, plane->normal) - plane->dist;
		if (d > 8)
			node = node->children[0];
		else if (d < -8)
			node = node->children[1];
		else
		{	// go down both
			SV_AddToFatPVS (org, node->children[0], worldmodel); //johnfitz -- worldmodel as a parameter
			node = node->children[1];
		}
	}
}

/*
=============
SV_FatPVS

Calculates a PVS that is the inclusive or of all leafs within 8 pixels of the
given point.
=============
*/
byte *SV_FatPVS (vec3_t org, qmodel_t *worldmodel) //johnfitz -- added worldmodel as a parameter
{
	fatbytes = (worldmodel->numleafs+31)>>3;
	Q_memset (fatpvs, 0, fatbytes);
	SV_AddToFatPVS (org, worldmodel->nodes, worldmodel); //johnfitz -- worldmodel as a parameter
	return fatpvs;
}

/*
=============
SV_VisibleToClient -- johnfitz

PVS test encapsulated in a nice function
=============
*/
qboolean SV_VisibleToClient (edict_t *client, edict_t *test, qmodel_t *worldmodel)
{
	byte	*pvs;
	vec3_t	org;
	int		i;

	VectorAdd (client->v.origin, client->v.view_ofs, org);
	pvs = SV_FatPVS (org, worldmodel);

	for (i=0 ; i < test->num_leafs ; i++)
		if (pvs[test->leafnums[i] >> 3] & (1 << (test->leafnums[i]&7) ))
			return true;

	return false;
}

//=============================================================================

/*
=============
SV_WriteEntitiesToClient

=============
*/
void SV_WriteEntitiesToClient (edict_t	*clent, sizebuf_t *msg)
{
	int		e, i;
	int		bits;
	byte	*pvs;
	vec3_t	org;
	float	miss;
	edict_t	*ent;
	eval_t	*val;

// find the client's PVS
	VectorAdd (clent->v.origin, clent->v.view_ofs, org);
	pvs = SV_FatPVS (org, sv.worldmodel);
	
// send over all entities (excpet the client) that touch the pvs
	ent = NEXT_EDICT(sv.edicts);
	for (e=1 ; e<sv.num_edicts ; e++, ent = NEXT_EDICT(ent))
	{

		if (ent != clent)	// clent is ALLWAYS sent
		{
			// ignore ents without visible models
			if (!ent->v.modelindex || !PR_GetString(ent->v.model)[0])
				continue;

			//johnfitz -- don't send model>255 entities if protocol is 15
			if (sv.protocol == PROTOCOL_NETQUAKE && (int)ent->v.modelindex & 0xFF00)
				continue;

			// ignore if not touching a PV leaf
			for (i=0 ; i < ent->num_leafs ; i++)
				if (pvs[ent->leafnums[i] >> 3] & (1 << (ent->leafnums[i]&7) ))
					break;
			
			// ericw -- added ent->num_leafs < MAX_ENT_LEAFS condition.
			//
			// if ent->num_leafs == MAX_ENT_LEAFS, the ent is visible from too many leafs
			// for us to say whether it's in the PVS, so don't try to vis cull it.
			// this commonly happens with rotators, because they often have huge bboxes
			// spanning the entire map, or really tall lifts, etc.
			if (i == ent->num_leafs && ent->num_leafs < MAX_ENT_LEAFS)
				continue;		// not visible
		}

		//johnfitz -- max size for protocol 15 is 18 bytes, not 16 as originally
		//assumed here.  And, for protocol 85 the max size is actually 24 bytes.
		if (msg->cursize + 24 > msg->maxsize)
		{
			//johnfitz -- less spammy overflow message
			if (!dev_overflows.packetsize || dev_overflows.packetsize + CONSOLE_RESPAM_TIME < realtime )
			{
				Con_Printf ("Packet overflow!\n");
				dev_overflows.packetsize = realtime;
			}
			goto stats;
			//johnfitz
		}

// send an update
		bits = 0;

		for (i=0 ; i<3 ; i++)
		{
			miss = ent->v.origin[i] - ent->baseline.origin[i];
			if ( miss < -0.1 || miss > 0.1 )
				bits |= U_ORIGIN1<<i;
		}

		if ( ent->v.angles[0] != ent->baseline.angles[0] )
			bits |= U_ANGLE1;

		if ( ent->v.angles[1] != ent->baseline.angles[1] )
			bits |= U_ANGLE2;

		if ( ent->v.angles[2] != ent->baseline.angles[2] )
			bits |= U_ANGLE3;

		if (ent->v.movetype == MOVETYPE_STEP)
			bits |= U_STEP;	// don't mess up the step animation

		if (ent->baseline.colormap != ent->v.colormap)
			bits |= U_COLORMAP;

		if (ent->baseline.skin != ent->v.skin)
			bits |= U_SKIN;

		if (ent->baseline.frame != ent->v.frame)
			bits |= U_FRAME;

		if (ent->baseline.effects != ent->v.effects)
			bits |= U_EFFECTS;

		if (ent->baseline.modelindex != ent->v.modelindex)
			bits |= U_MODEL;

		//johnfitz -- alpha
		// TODO: find a cleaner place to put this code
		val = GetEdictFieldValue(ent, pr_extfields.alpha);
		if (val)
			ent->alpha = ENTALPHA_ENCODE(val->_float);

		//don't send invisible entities unless they have effects
		if (ent->alpha == ENTALPHA_ZERO && !ent->v.effects)
			continue;
		//johnfitz

		//johnfitz -- PROTOCOL_FITZQUAKE
		if (sv.protocol != PROTOCOL_NETQUAKE)
		{

			if (ent->baseline.alpha != ent->alpha) bits |= U_ALPHA;
			if (bits & U_FRAME && (int)ent->v.frame & 0xFF00) bits |= U_FRAME2;
			if (bits & U_MODEL && (int)ent->v.modelindex & 0xFF00) bits |= U_MODEL2;
			if (ent->sendinterval) bits |= U_LERPFINISH;
			if (bits >= 65536) bits |= U_EXTEND1;
			if (bits >= 16777216) bits |= U_EXTEND2;
		}
		//johnfitz

		if (e >= 256)
			bits |= U_LONGENTITY;

		if (bits >= 256)
			bits |= U_MOREBITS;

	//
	// write the message
	//
		MSG_WriteByte (msg, bits | U_SIGNAL);

		if (bits & U_MOREBITS)
			MSG_WriteByte (msg, bits>>8);

		//johnfitz -- PROTOCOL_FITZQUAKE
		if (bits & U_EXTEND1)
			MSG_WriteByte(msg, bits>>16);
		if (bits & U_EXTEND2)
			MSG_WriteByte(msg, bits>>24);
		//johnfitz

		if (bits & U_LONGENTITY)
			MSG_WriteShort (msg,e);
		else
			MSG_WriteByte (msg,e);

		if (bits & U_MODEL)
			MSG_WriteByte (msg,	ent->v.modelindex);
		if (bits & U_FRAME)
			MSG_WriteByte (msg, ent->v.frame);
		if (bits & U_COLORMAP)
			MSG_WriteByte (msg, ent->v.colormap);
		if (bits & U_SKIN)
			MSG_WriteByte (msg, ent->v.skin);
		if (bits & U_EFFECTS)
			MSG_WriteByte (msg, ent->v.effects);
		if (bits & U_ORIGIN1)
			MSG_WriteCoord (msg, ent->v.origin[0], sv.protocolflags);
		if (bits & U_ANGLE1)
			MSG_WriteAngle(msg, ent->v.angles[0], sv.protocolflags);
		if (bits & U_ORIGIN2)
			MSG_WriteCoord (msg, ent->v.origin[1], sv.protocolflags);
		if (bits & U_ANGLE2)
			MSG_WriteAngle(msg, ent->v.angles[1], sv.protocolflags);
		if (bits & U_ORIGIN3)
			MSG_WriteCoord (msg, ent->v.origin[2], sv.protocolflags);
		if (bits & U_ANGLE3)
			MSG_WriteAngle(msg, ent->v.angles[2], sv.protocolflags);

		//johnfitz -- PROTOCOL_FITZQUAKE
		if (bits & U_ALPHA)
			MSG_WriteByte(msg, ent->alpha);
		if (bits & U_FRAME2)
			MSG_WriteByte(msg, (int)ent->v.frame >> 8);
		if (bits & U_MODEL2)
			MSG_WriteByte(msg, (int)ent->v.modelindex >> 8);
		if (bits & U_LERPFINISH)
			MSG_WriteByte(msg, (byte)(Q_rint((ent->v.nextthink-sv.time)*255)));
		//johnfitz
	}

	//johnfitz -- devstats
stats:
	if (msg->cursize > 1024 && dev_peakstats.packetsize <= 1024)
		Con_DWarning ("%i byte packet exceeds standard limit of 1024.\n", msg->cursize);
	dev_stats.packetsize = msg->cursize;
	dev_peakstats.packetsize = q_max(msg->cursize, dev_peakstats.packetsize);
	//johnfitz
}

/*
=============
SV_CleanupEnts

=============
*/
void SV_CleanupEnts (void)
{
	int		e;
	edict_t	*ent;

	ent = NEXT_EDICT(sv.edicts);
	for (e=1 ; e<sv.num_edicts ; e++, ent = NEXT_EDICT(ent))
	{
		ent->v.effects = (int)ent->v.effects & ~EF_MUZZLEFLASH;
	}
}

/*
==================
SV_WriteDamageToMessage

==================
*/
void SV_WriteDamageToMessage(edict_t *ent, sizebuf_t *msg)
{
	edict_t	*other;
	int		i;

//
// send a damage message
//
	if (ent->v.dmg_take || ent->v.dmg_save)
	{
		other = PROG_TO_EDICT(ent->v.dmg_inflictor);
		MSG_WriteByte (msg, svc_damage);
		MSG_WriteByte (msg, ent->v.dmg_save);
		MSG_WriteByte (msg, ent->v.dmg_take);
		for (i=0 ; i<3 ; i++)
			MSG_WriteCoord (msg, other->v.origin[i] + 0.5*(other->v.mins[i] + other->v.maxs[i]), sv.protocolflags );

		ent->v.dmg_take = 0;
		ent->v.dmg_save = 0;
	}

//
// send the current viewpos offset from the view entity
//
	SV_SetIdealPitch ();		// how much to look up / down ideally

// a fixangle might get lost in a dropped packet.  Oh well.
	if ( ent->v.fixangle )
	{
		MSG_WriteByte (msg, svc_setangle);
		for (i=0 ; i < 3 ; i++)
			MSG_WriteAngle (msg, ent->v.angles[i], sv.protocolflags );
		ent->v.fixangle = 0;
	}
}

/*
==================
SV_WriteClientdataToMessage

==================
*/
void SV_WriteClientdataToMessage (edict_t *ent, sizebuf_t *msg)
{
	int		bits;
	int		i;
	int		items;
	eval_t	*val;

	bits = 0;

	if (ent->v.view_ofs[2] != DEFAULT_VIEWHEIGHT)
		bits |= SU_VIEWHEIGHT;

	if (ent->v.idealpitch)
		bits |= SU_IDEALPITCH;

// stuff the sigil bits into the high bits of items for sbar, or else
// mix in items2
	val = GetEdictFieldValue(ent, pr_extfields.items2);

	if (val)
		items = (int)ent->v.items | ((int)val->_float << 23);
	else
		items = (int)ent->v.items | ((int)pr_global_struct->serverflags << 28);

	bits |= SU_ITEMS;

	if ( (int)ent->v.flags & FL_ONGROUND)
		bits |= SU_ONGROUND;

	if ( ent->v.waterlevel >= 2)
		bits |= SU_INWATER;

	for (i=0 ; i<3 ; i++)
	{
		if (ent->v.punchangle[i])
			bits |= (SU_PUNCH1<<i);
		if (ent->v.velocity[i])
			bits |= (SU_VELOCITY1<<i);
	}

	if (ent->v.weaponframe)
		bits |= SU_WEAPONFRAME;

	if (ent->v.armorvalue)
		bits |= SU_ARMOR;

//	if (ent->v.weapon)
		bits |= SU_WEAPON;

	//johnfitz -- PROTOCOL_FITZQUAKE
	if (sv.protocol != PROTOCOL_NETQUAKE)
	{
		if (bits & SU_WEAPON && SV_ModelIndex(PR_GetString(ent->v.weaponmodel)) & 0xFF00) bits |= SU_WEAPON2;
		if ((int)ent->v.armorvalue & 0xFF00) bits |= SU_ARMOR2;
		if ((int)ent->v.currentammo & 0xFF00) bits |= SU_AMMO2;
		if ((int)ent->v.ammo_shells & 0xFF00) bits |= SU_SHELLS2;
		if ((int)ent->v.ammo_nails & 0xFF00) bits |= SU_NAILS2;
		if ((int)ent->v.ammo_rockets & 0xFF00) bits |= SU_ROCKETS2;
		if ((int)ent->v.ammo_cells & 0xFF00) bits |= SU_CELLS2;
		if (bits & SU_WEAPONFRAME && (int)ent->v.weaponframe & 0xFF00) bits |= SU_WEAPONFRAME2;
		if (bits & SU_WEAPON && ent->alpha != ENTALPHA_DEFAULT) bits |= SU_WEAPONALPHA; //for now, weaponalpha = client entity alpha
		if (bits >= 65536) bits |= SU_EXTEND1;
		if (bits >= 16777216) bits |= SU_EXTEND2;
	}
	//johnfitz

// send the data

	MSG_WriteByte (msg, svc_clientdata);
	MSG_WriteShort (msg, bits);

	//johnfitz -- PROTOCOL_FITZQUAKE
	if (bits & SU_EXTEND1) MSG_WriteByte(msg, bits>>16);
	if (bits & SU_EXTEND2) MSG_WriteByte(msg, bits>>24);
	//johnfitz

	if (bits & SU_VIEWHEIGHT)
		MSG_WriteChar (msg, ent->v.view_ofs[2]);

	if (bits & SU_IDEALPITCH)
		MSG_WriteChar (msg, ent->v.idealpitch);

	for (i=0 ; i<3 ; i++)
	{
		if (bits & (SU_PUNCH1<<i))
			MSG_WriteChar (msg, ent->v.punchangle[i]);
		if (bits & (SU_VELOCITY1<<i))
			MSG_WriteChar (msg, ent->v.velocity[i]/16);
	}

// [always sent]	if (bits & SU_ITEMS)
	MSG_WriteLong (msg, items);

	if (bits & SU_WEAPONFRAME)
		MSG_WriteByte (msg, ent->v.weaponframe);
	if (bits & SU_ARMOR)
		MSG_WriteByte (msg, ent->v.armorvalue);
	if (bits & SU_WEAPON)
		MSG_WriteByte (msg, SV_ModelIndex(PR_GetString(ent->v.weaponmodel)));

	MSG_WriteShort (msg, ent->v.health);
	MSG_WriteByte (msg, ent->v.currentammo);
	MSG_WriteByte (msg, ent->v.ammo_shells);
	MSG_WriteByte (msg, ent->v.ammo_nails);
	MSG_WriteByte (msg, ent->v.ammo_rockets);
	MSG_WriteByte (msg, ent->v.ammo_cells);

	if (standard_quake)
	{
		MSG_WriteByte (msg, ent->v.weapon);
	}
	else
	{
		for(i=0;i<32;i++)
		{
			if ( ((int)ent->v.weapon) & (1<<i) )
			{
				MSG_WriteByte (msg, i);
				break;
			}
		}
	}

	//johnfitz -- PROTOCOL_FITZQUAKE
	if (bits & SU_WEAPON2)
		MSG_WriteByte (msg, SV_ModelIndex(PR_GetString(ent->v.weaponmodel)) >> 8);
	if (bits & SU_ARMOR2)
		MSG_WriteByte (msg, (int)ent->v.armorvalue >> 8);
	if (bits & SU_AMMO2)
		MSG_WriteByte (msg, (int)ent->v.currentammo >> 8);
	if (bits & SU_SHELLS2)
		MSG_WriteByte (msg, (int)ent->v.ammo_shells >> 8);
	if (bits & SU_NAILS2)
		MSG_WriteByte (msg, (int)ent->v.ammo_nails >> 8);
	if (bits & SU_ROCKETS2)
		MSG_WriteByte (msg, (int)ent->v.ammo_rockets >> 8);
	if (bits & SU_CELLS2)
		MSG_WriteByte (msg, (int)ent->v.ammo_cells >> 8);
	if (bits & SU_WEAPONFRAME2)
		MSG_WriteByte (msg, (int)ent->v.weaponframe >> 8);
	if (bits & SU_WEAPONALPHA)
		MSG_WriteByte (msg, ent->alpha); //for now, weaponalpha = client entity alpha
	//johnfitz
}

/*
=======================
SV_SendClientDatagram
=======================
*/
qboolean SV_SendClientDatagram (client_t *client)
{
	byte		buf[MAX_DATAGRAM+1000];
	sizebuf_t	msg;

	if (!client->netconnection)
	{
		//botclient, shouldn't be sent anything.
		SZ_Clear(&client->datagram);
		return true;
	}

	msg.allowoverflow = false;
	msg.data = buf;
	msg.maxsize = MAX_DATAGRAM;//sizeof(buf);
	msg.cursize = 0;

	host_client = client;
	sv_player = client->edict;

	//johnfitz -- if client is nonlocal, use smaller max size so packets aren't fragmented
	if (Q_strcmp(NET_QSocketGetAddressString(client->netconnection), "LOCAL") != 0)
		msg.maxsize = DATAGRAM_MTU;
	//johnfitz

	if (client->protocol_pext2 & PEXT2_REPLACEMENTDELTAS)
	{
		SV_WriteDamageToMessage(client->edict, &msg);
		if (!(client->protocol_pext2 & PEXT2_PREDINFO))
			SV_WriteClientdataToMessage (client->edict, &msg);
		else
			SVFTE_WriteStats(client, &msg);
		if (!client->snapshotresume)
		{
			SVFTE_BuildSnapshotForClient(client->edict);
			SVFTE_CalcEntityDeltas(client);
		}
		SVFTE_WriteEntitiesToClient(client, &msg, sizeof(buf));	//must always write some data, or the stats will break

		//this delta protocol doesn't wipe old state just because there's a new packet.
		//the server isn't required to sync with the client frames either
		//so we can just spam multiple packets to keep our udp data under the MTU
		while (client->snapshotresume)
		{
			NET_SendUnreliableMessage (client->netconnection, &msg);
			SZ_Clear(&msg);
			SVFTE_WriteEntitiesToClient(client, &msg, sizeof(buf));
		}
	}
	else
	{
		MSG_WriteByte (&msg, svc_time);
		MSG_WriteFloat (&msg, sv.time);

// add the client specific data to the datagram
		SV_WriteDamageToMessage (client->edict, &msg);
		SV_WriteClientdataToMessage (client->edict, &msg);

		SV_WriteEntitiesToClient (client->edict, &msg);
	}

// copy the private datagram if there is space
	if (msg.cursize + client->datagram.cursize < msg.maxsize && !client->datagram.overflowed)
		SZ_Write(&msg, client->datagram.data, client->datagram.cursize);
	client->datagram.overflowed = false;
	SZ_Clear(&client->datagram);
// copy the server datagram if there is space
	if (msg.cursize + sv.datagram.cursize < msg.maxsize)
		SZ_Write (&msg, sv.datagram.data, sv.datagram.cursize);

// send the datagram
	if (NET_SendUnreliableMessage (client->netconnection, &msg) == -1)
	{
		SV_DropClient (true);// if the message couldn't send, kick off
		return false;
	}

	return true;
}

/*
=======================
SV_UpdateToReliableMessages
=======================
*/
void SV_UpdateToReliableMessages (void)
{
	int			i, j;
	client_t *client;

// check for changes to be sent over the reliable streams
	for (i=0, host_client = svs.clients ; i<svs.maxclients ; i++, host_client++)
	{
		if (host_client->old_frags != host_client->edict->v.frags)
		{
			for (j=0, client = svs.clients ; j<svs.maxclients ; j++, client++)
			{
				if (!client->active)
					continue;
				MSG_WriteByte (&client->message, svc_updatefrags);
				MSG_WriteByte (&client->message, i);
				MSG_WriteShort (&client->message, host_client->edict->v.frags);
			}

			host_client->old_frags = host_client->edict->v.frags;
		}
	}

	for (j=0, client = svs.clients ; j<svs.maxclients ; j++, client++)
	{
		if (!client->active)
			continue;
		SZ_Write (&client->message, sv.reliable_datagram.data, sv.reliable_datagram.cursize);
	}

	SZ_Clear (&sv.reliable_datagram);
}


/*
=======================
SV_SendNop

Send a nop message without trashing or sending the accumulated client
message buffer
=======================
*/
void SV_SendNop (client_t *client)
{
	sizebuf_t	msg;
	byte		buf[4];

	msg.data = buf;
	msg.maxsize = sizeof(buf);
	msg.cursize = 0;

	MSG_WriteChar (&msg, svc_nop);

	if (NET_SendUnreliableMessage (client->netconnection, &msg) == -1)
		SV_DropClient (true);	// if the message couldn't send, kick off
	client->last_message = realtime;
}

static entity_state_t nullentitystate;
int SV_SendPrespawnBaselines(int idx)
{
	int i;
	edict_t *svent;
	int maxsize;
//	if (sv.protocol == PROTOCOL_NETQUAKE)
//		maxsize = 8000;	//vanilla has low limits
//	else
		maxsize = NET_MAXMESSAGE;	//we can go quite large

	maxsize /= 2;	//but try not to push it too much
	while (1)
	{
		if (idx >= sv.num_edicts)
		{
			SZ_Write (&host_client->message, sv.signon.data, sv.signon.cursize);
			MSG_WriteByte (&host_client->message, svc_signonnum);
			MSG_WriteByte (&host_client->message, 2);
			host_client->sendsignon = true;
			return -1;
		}
		svent = EDICT_NUM(idx);

		if (host_client->message.cursize > maxsize)
			break;

		if (memcmp(&nullentitystate, &svent->baseline, sizeof(nullentitystate)))
		{
			if (host_client->protocol_pext2 & PEXT2_REPLACEMENTDELTAS)
			{	//not really necessary yet, but hey
				MSG_WriteByte(&host_client->message, svcfte_spawnbaseline2);
				MSG_WriteShort(&host_client->message, idx);
				SVFTE_WriteEntityUpdate(SVFTE_DeltaCalcBits(&nullentitystate, &svent->baseline), &svent->baseline, &host_client->message, host_client->protocol_pext2);
			}
			else
			{
				//johnfitz -- PROTOCOL_FITZQUAKE
				int bits = 0;
				if (sv.protocol == PROTOCOL_FITZQUAKE) //still want to send baseline in PROTOCOL_NETQUAKE, so reset these values
				{
					if (svent->baseline.modelindex & 0xFF00)
						bits |= B_LARGEMODEL;
					if (svent->baseline.frame & 0xFF00)
						bits |= B_LARGEFRAME;
					if (svent->baseline.alpha != ENTALPHA_DEFAULT)
						bits |= B_ALPHA;
				}
				if (bits)
					MSG_WriteByte (&host_client->message, svc_spawnbaseline2);
				else
					MSG_WriteByte (&host_client->message, svc_spawnbaseline);
				//johnfitz

				MSG_WriteShort (&host_client->message, idx);

				//johnfitz -- PROTOCOL_FITZQUAKE
				if (bits)
					MSG_WriteByte (&host_client->message, bits);

				if (bits & B_LARGEMODEL)
					MSG_WriteShort (&host_client->message, svent->baseline.modelindex);
				else
					MSG_WriteByte (&host_client->message, svent->baseline.modelindex);

				if (bits & B_LARGEFRAME)
					MSG_WriteShort (&host_client->message, svent->baseline.frame);
				else
					MSG_WriteByte (&host_client->message, svent->baseline.frame);
				//johnfitz

				MSG_WriteByte (&host_client->message, svent->baseline.colormap);
				MSG_WriteByte (&host_client->message, svent->baseline.skin);
				for (i=0 ; i<3 ; i++)
				{
					MSG_WriteCoord(&host_client->message, svent->baseline.origin[i], sv.protocolflags);
					MSG_WriteAngle(&host_client->message, svent->baseline.angles[i], sv.protocolflags);
				}

				//johnfitz -- PROTOCOL_FITZQUAKE
				if (bits & B_ALPHA)
					MSG_WriteByte (&host_client->message, svent->baseline.alpha);
				//johnfitz
			}
		}

		idx++;
	}
	return idx;
}

/*
=======================
SV_SendClientMessages
=======================
*/
void SV_SendClientMessages (void)
{
	int			i;

// update frags, names, etc
	SV_UpdateToReliableMessages ();

// build individual updates
	for (i=0, host_client = svs.clients ; i<svs.maxclients ; i++, host_client++)
	{
		if (!host_client->active)
			continue;

		if (host_client->spawned)
		{
			if (!SV_SendClientDatagram (host_client))
				continue;
		}
		else
		{
		// the player isn't totally in the game yet
		// send small keepalive messages if too much time has passed
		// send a full message when the next signon stage has been requested
		// some other message data (name changes, etc) may accumulate
		// between signon stages
			if (!host_client->sendsignon)
			{
				if (realtime - host_client->last_message > 5)
					SV_SendNop (host_client);
				continue;	// don't send out non-signon messages
			}
			if (host_client->sendsignon >= 2)
			{
				host_client->sendsignon = SV_SendPrespawnBaselines(host_client->sendsignon-2) + 2;
			}
		}

		// check for an overflowed message.  Should only happen
		// on a very fucked up connection that backs up a lot, then
		// changes level
		if (host_client->message.overflowed)
		{
			SV_DropClient (true);
			host_client->message.overflowed = false;
			continue;
		}

		if (host_client->message.cursize || host_client->dropasap)
		{
			if (!NET_CanSendMessage (host_client->netconnection))
			{
//				I_Printf ("can't write\n");
				continue;
			}

			if (host_client->dropasap)
				SV_DropClient (false);	// went to another level
			else
			{
				if (NET_SendMessage (host_client->netconnection
				, &host_client->message) == -1)
					SV_DropClient (true);	// if the message couldn't send, kick off
				SZ_Clear (&host_client->message);
				host_client->last_message = realtime;
				if (host_client->sendsignon == true)
					host_client->sendsignon = false;
			}
		}
	}


// clear muzzle flashes
	SV_CleanupEnts ();
}


/*
==============================================================================

SERVER SPAWNING

==============================================================================
*/

/*
================
SV_ModelIndex

================
*/
int SV_ModelIndex (const char *name)
{
	int		i;

	if (!name || !name[0])
		return 0;

	for (i=0 ; i<MAX_MODELS && sv.model_precache[i] ; i++)
		if (!strcmp(sv.model_precache[i], name))
			return i;
	if (i==MAX_MODELS || !sv.model_precache[i])
		Sys_Error ("SV_ModelIndex: model %s not precached", name);
	return i;
}

/*
================
SV_CreateBaseline
================
*/
void SV_CreateBaseline (void)
{
	edict_t		*svent;
	int			entnum;

	for (entnum = 0; entnum < sv.num_edicts; entnum++)
	{
	// get the current server version
		svent = EDICT_NUM(entnum);
		if (svent->free)
			continue;
		if (entnum > svs.maxclients && !svent->v.modelindex)
			continue;

	//
	// create entity baseline
	//
		VectorCopy (svent->v.origin, svent->baseline.origin);
		VectorCopy (svent->v.angles, svent->baseline.angles);
		svent->baseline.frame = svent->v.frame;
		svent->baseline.skin = svent->v.skin;
		if (entnum > 0 && entnum <= svs.maxclients)
		{
			svent->baseline.colormap = entnum;
			svent->baseline.modelindex = SV_ModelIndex("progs/player.mdl");
			svent->baseline.alpha = ENTALPHA_DEFAULT; //johnfitz -- alpha support
		}
		else
		{
			svent->baseline.colormap = 0;
			svent->baseline.modelindex = SV_ModelIndex(PR_GetString(svent->v.model));
			svent->baseline.alpha = svent->alpha; //johnfitz -- alpha support
		}

		//Spike -- baselines are now generated on a per-client basis.
		//FIXME: should merge the above with other edict->entity_state copies (updates, baselines, spawnstatics)
		//1) this allows per-client extensions.
		//2) this avoids pre-generating a single signon buffer, splitting it over multiple packets.
		//   thereby allowing more than 3k or so entities
	}
}


/*
================
SV_SendReconnect

Tell all the clients that the server is changing levels
================
*/
void SV_SendReconnect (void)
{
	byte	data[128];
	sizebuf_t	msg;

	msg.data = data;
	msg.cursize = 0;
	msg.maxsize = sizeof(data);

	MSG_WriteChar (&msg, svc_stufftext);
	MSG_WriteString (&msg, "reconnect\n");
	NET_SendToAll (&msg, 5.0);

	if (!isDedicated)
		Cmd_ExecuteString ("reconnect\n", src_command);
}


/*
================
SV_SaveSpawnparms

Grabs the current state of each client for saving across the
transition to another level
================
*/
void SV_SaveSpawnparms (void)
{
	int		i, j;

	svs.serverflags = pr_global_struct->serverflags;

	for (i=0, host_client = svs.clients ; i<svs.maxclients ; i++, host_client++)
	{
		if (!host_client->active)
			continue;

	// call the progs to get default spawn parms for the new client
		pr_global_struct->self = EDICT_TO_PROG(host_client->edict);
		PR_ExecuteProgram (pr_global_struct->SetChangeParms);
		for (j=0 ; j<NUM_SPAWN_PARMS ; j++)
			host_client->spawn_parms[j] = (&pr_global_struct->parm1)[j];
	}
}


/*
================
SV_SpawnServer

This is called at the start of each level
================
*/
extern float		scr_centertime_off;
void SV_SpawnServer (const char *server)
{
	static char	dummy[8] = { 0,0,0,0,0,0,0,0 };
	edict_t		*ent;
	int			i;

	// let's not have any servers with no name
	if (hostname.string[0] == 0)
		Cvar_Set ("hostname", "UNNAMED");
	scr_centertime_off = 0;

	Con_DPrintf ("SpawnServer: %s\n",server);
	svs.changelevel_issued = false;		// now safe to issue another

//
// tell all connected clients that we are going to a new level
//
	if (sv.active)
	{
		SV_SendReconnect ();
	}

//
// make cvars consistant
//
	if (coop.value)
		Cvar_Set ("deathmatch", "0");
	current_skill = (int)(skill.value + 0.5);
	if (current_skill < 0)
		current_skill = 0;
	if (current_skill > 3)
		current_skill = 3;

	Cvar_SetValue ("skill", (float)current_skill);

//
// set up the new server
//
	//memset (&sv, 0, sizeof(sv));
	Host_ClearMemory ();

	q_strlcpy (sv.name, server, sizeof(sv.name));

	sv.protocol = sv_protocol; // johnfitz
	
	if (sv.protocol == PROTOCOL_RMQ)
	{
		// set up the protocol flags used by this server
		// (note - these could be cvar-ised so that server admins could choose the protocol features used by their servers)
		sv.protocolflags = PRFL_INT32COORD | PRFL_SHORTANGLE;
	}
	else sv.protocolflags = 0;

// load progs to get entity field count
	PR_LoadProgs ();

// allocate server memory
	/* Host_ClearMemory() called above already cleared the whole sv structure */
	sv.max_edicts = CLAMP (MIN_EDICTS,(int)max_edicts.value,MAX_EDICTS); //johnfitz -- max_edicts cvar
	sv.edicts = (edict_t *) malloc (sv.max_edicts*pr_edict_size); // ericw -- sv.edicts switched to use malloc()

	sv.datagram.maxsize = sizeof(sv.datagram_buf);
	sv.datagram.cursize = 0;
	sv.datagram.data = sv.datagram_buf;

	sv.multicast.maxsize = sizeof(sv.multicast_buf);
	sv.multicast.cursize = 0;
	sv.multicast.data = sv.multicast_buf;

	sv.reliable_datagram.maxsize = sizeof(sv.reliable_datagram_buf);
	sv.reliable_datagram.cursize = 0;
	sv.reliable_datagram.data = sv.reliable_datagram_buf;

	sv.signon.maxsize = sizeof(sv.signon_buf);
	sv.signon.cursize = 0;
	sv.signon.data = sv.signon_buf;

// leave slots at start for clients only
	sv.num_edicts = svs.maxclients+1;
	memset(sv.edicts, 0, sv.num_edicts*pr_edict_size); // ericw -- sv.edicts switched to use malloc()
	for (i=0 ; i<svs.maxclients ; i++)
	{
		ent = EDICT_NUM(i+1);
		svs.clients[i].edict = ent;
	}

	sv.state = ss_loading;
	sv.paused = false;

	sv.time = 1.0;

	q_strlcpy (sv.name, server, sizeof(sv.name));
	q_snprintf (sv.modelname, sizeof(sv.modelname), "maps/%s.bsp", server);
	sv.worldmodel = Mod_ForName (sv.modelname, false);
	if (!sv.worldmodel || sv.worldmodel->type != mod_brush)
	{
		Con_Printf ("Couldn't spawn server %s\n", sv.modelname);
		sv.active = false;
		return;
	}
	sv.models[1] = sv.worldmodel;

//
// clear world interaction links
//
	SV_ClearWorld ();

	sv.sound_precache[0] = dummy;
	sv.model_precache[0] = dummy;
	sv.model_precache[1] = sv.modelname;
	for (i=1 ; i<sv.worldmodel->numsubmodels ; i++)
	{
		sv.model_precache[1+i] = localmodels[i];
		sv.models[i+1] = Mod_ForName (localmodels[i], false);
	}

//
// load the rest of the entities
//
	ent = EDICT_NUM(0);
	memset (&ent->v, 0, progs->entityfields * 4);
	ent->free = false;
	ent->v.model = PR_SetEngineString(sv.worldmodel->name);
	ent->v.modelindex = 1;		// world model
	ent->v.solid = SOLID_BSP;
	ent->v.movetype = MOVETYPE_PUSH;

	if (coop.value)
		pr_global_struct->coop = coop.value;
	else
		pr_global_struct->deathmatch = deathmatch.value;

	pr_global_struct->mapname = PR_SetEngineString(sv.name);

// serverflags are for cross level information (sigils)
	pr_global_struct->serverflags = svs.serverflags;

	ED_LoadFromFile (sv.worldmodel->entities);

	sv.active = true;

// all setup is completed, any further precache statements are errors
	sv.state = ss_active;

// run two frames to allow everything to settle
	host_frametime = 0.1;
	SV_Physics ();
	SV_Physics ();

// create a baseline for more efficient communications
	SV_CreateBaseline ();

	//johnfitz -- warn if signon buffer larger than standard server can handle
	if (sv.signon.cursize > 8000-2) //max size that will fit into 8000-sized client->message buffer with 2 extra bytes on the end
		Con_DWarning ("%i byte signon buffer exceeds standard limit of 7998.\n", sv.signon.cursize);
	//johnfitz

// send serverinfo to all connected clients
	for (i=0,host_client = svs.clients ; i<svs.maxclients ; i++, host_client++)
		if (host_client->active)
			SV_SendServerinfo (host_client);

	Con_DPrintf ("Server spawned.\n");
}

