/*
Copyright (C) 1996-1997 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

// gl_screen.c -- master for refresh, status bar, console, chat, notify, etc

#include "quakedef.h"
#ifdef _WIN32
#include "movie.h"
#endif

/*

background clear
rendering
turtle/net/ram icons
sbar
centerprint / slow centerprint
notify lines
intermission / finale overlay
loading plaque
console
menu

required background clears
required update regions


syncronous draw mode or async
One off screen buffer, with updates either copied or xblited
Need to double buffer?


async draw will require the refresh area to be cleared, because it will be
xblited, but sync draw can just ignore it.

sync
draw

CenterPrint ()
SlowPrint ()
Screen_Update ();
Con_Printf ();

net 
turn off messages option

the refresh is allways rendered, unless the console is full screen


console is:
	notify lines
	half
	full

*/


int		glx, gly, glwidth, glheight;

// only the refresh window will be updated unless these variables are flagged 
int		scr_copytop;
int		scr_copyeverything;

float		scr_con_current;
float		scr_conlines;		// lines of console to display

float		oldscreensize, oldfov;
cvar_t		scr_viewsize			= {"viewsize", "100", true};
cvar_t		scr_fov					= {"fov", "90"};				// 10 - 170
cvar_t		scr_fovspeed			= {"fov_speed", "10"};			//R00k
cvar_t      scr_consize				= {"scr_consize", "0.5"};		// by joe
cvar_t		scr_conspeed			= {"scr_conspeed", "1000"};
cvar_t		scr_centertime			= {"scr_centertime", "2"};
cvar_t		scr_showram				= {"showram", "0"};
cvar_t		scr_showturtle			= {"showturtle", "0"};
cvar_t		scr_showpause			= {"showpause", "1"};
cvar_t		scr_printspeed			= {"scr_printspeed", "8"};
#ifdef GLQUAKE
cvar_t		gl_triplebuffer			= {"gl_triplebuffer", "0", true};
cvar_t		png_compression_level	= {"png_compression_level", "1"};
cvar_t		jpeg_compression_level	= {"jpeg_compression_level", "75"};
#endif
cvar_t		scr_sshot_type			= {"scr_sshot_type", "jpg"};

cvar_t		scr_centerprint_levelname = {"scr_centerprint_levelname","1", true};

qboolean	scr_initialized;		// ready to draw

mpic_t          *scr_ram;
mpic_t          *scr_net; 
mpic_t          *scr_turtle;

int			scr_fullupdate;

int			clearconsole;
int			clearnotify;

int			sb_lines;
	
viddef_t	vid;				// global video state

vrect_t		scr_vrect;

qboolean	scr_disabled_for_loading;
qboolean	scr_drawloading;
float		scr_disabled_time;

qboolean	block_drawing;

void SCR_ScreenShot_f (void);

/*
===============================================================================

CENTER PRINTING

===============================================================================
*/

char	scr_centerstring[1024];
float	scr_centertime_start;	// for slow victory printing
float	scr_centertime_off;
int		scr_center_lines;
int		scr_erase_lines;
int		scr_erase_center;

/*
==============
SCR_CenterPrint

Called for important messages that should stay in the center of the screen
for a few moments
==============
*/
void SCR_CenterPrint (char *str)
{
	if (!str[0]) return; //MH!

	Q_strncpyz (scr_centerstring, str, sizeof(scr_centerstring)); 

	scr_centertime_off = scr_centertime.value;
	scr_centertime_start = cl.time;

// count the number of lines for centering
	scr_center_lines = 1;
	while (*str)
	{
		if (*str == '\n')
			scr_center_lines++;
		str++;
	}
}


void SCR_DrawCenterString (void)
{
	char	*start;
	int	l, j, x, y, remaining;

// the finale prints the characters one at a time
	if (cl.intermission)
		remaining = scr_printspeed.value * (cl.time - scr_centertime_start);
	else
		remaining = 9999;

	scr_erase_center = 0;
	start = scr_centerstring;

	if (scr_center_lines <= 4)
		y = vid.height*0.35;
	else
		y = 48;

	do	
	{
	// scan the width of the line
		for (l=0 ; l<40 ; l++)
			if (start[l] == '\n' || !start[l])
				break;
		x = (vid.width - l*8) / 2;
		for (j=0 ; j<l ; j++, x+=8)
		{
			Draw_Character (x, y, start[j],false);	
			if (!remaining--)
				return;
		}
			
		y += 8;

		while (*start && *start != '\n')
			start++;

		if (!*start)
			break;
		start++;		// skip the \n
	} while (1);
}

extern qboolean sb_showscores;
void SCR_CheckDrawCenterString (void)
{
	scr_copytop = 1;
	if (scr_center_lines > scr_erase_lines)
		scr_erase_lines = scr_center_lines;

	scr_centertime_off -= host_frametime;
	
	if (scr_centertime_off <= 0 && !cl.intermission)
		return;

	if (key_dest != key_game)
		return;

	if ((sb_showscores) && (cl.gametype == GAME_DEATHMATCH))
		return;

	SCR_DrawCenterString ();
}

//=============================================================================

/*
====================
CalcFov
====================
*/
float CalcFov (float fov_x, float width, float height)
{
	fov_x = CLAMP(1, fov_x, 179);
	return ((atan (height / (width / tan(fov_x/360*M_PI))))* 360 / M_PI);
}

/*
=================
SCR_CalcRefdef

Must be called whenever vid changes
Internal use only
=================
*/
static void SCR_CalcRefdef (void)
{
	float		size;
	int		h;
	qboolean	full = false;

	scr_fullupdate = 0;		// force a background redraw
	vid.recalc_refdef = 0;

// force the status bar to redraw
	Sbar_Changed ();

//========================================
	
// bound viewsize
	if (scr_viewsize.value < 30)
		Cvar_Set ("viewsize", "30");
	if (scr_viewsize.value > 120)
		Cvar_Set ("viewsize", "120");

// bound field of view
	if (scr_fov.value < 10)
		Cvar_Set ("fov", "10");
	if (scr_fov.value > 170)
		Cvar_Set ("fov", "170");

// intermission is always full screen	
	size = cl.intermission ? 120 : scr_viewsize.value;

	if (size >= 120)
		sb_lines = 0;		// no status bar at all
	else if (size >= 110)
		sb_lines = 24;		// no inventory
	else
		sb_lines = 24 + 16 + 8;

	if (scr_viewsize.value >= 100.0)
	{
		full = true;
		size = 100.0;
	}
	else
	{
		size = scr_viewsize.value;
	}

	if (cl.intermission)
	{
		full = true;
		size = 100.0;
		sb_lines = 0;
	}
	size /= 100.0;

	h = (!cl_sbar.value && full) ? vid.height : vid.height - sb_lines;

	r_refdef.vrect.width = vid.width * size;

	if (r_refdef.vrect.width < 96)
	{
		size = 96.0 / r_refdef.vrect.width;
		r_refdef.vrect.width = 96;	// min for icons
	}

	r_refdef.vrect.height = vid.height * size;

	if (cl_sbar.value || !full)
	{
		if (r_refdef.vrect.height > vid.height - sb_lines)
			r_refdef.vrect.height = vid.height - sb_lines;
	}
	else if (r_refdef.vrect.height > vid.height)
	{
		r_refdef.vrect.height = vid.height;
	}

	r_refdef.vrect.x = (vid.width - r_refdef.vrect.width) / 2;

	if (full)
		r_refdef.vrect.y = 0;
	else 
		r_refdef.vrect.y = (h - r_refdef.vrect.height) / 2;

	r_refdef.fov_x = scr_fov.value;
	r_refdef.fov_y = CalcFov (r_refdef.fov_x, (float)r_refdef.vrect.width, (float)r_refdef.vrect.height);

	scr_vrect = r_refdef.vrect;
}
/*
=================
SCR_SizeUp_f

Keybinding command
=================
*/
void SCR_SizeUp_f (void)
{
	Cvar_SetValue ("viewsize", scr_viewsize.value + 10);
	vid.recalc_refdef = 1;
}

/*
=================
SCR_SizeDown_f

Keybinding command
=================
*/
void SCR_SizeDown_f (void)
{
	Cvar_SetValue ("viewsize", scr_viewsize.value - 10);
	vid.recalc_refdef = 1;
}

//============================================================================

/*
==================
SCR_Init
==================
*/
void SCR_Init (void)
{

	Cvar_RegisterVariable (&scr_fov);
	Cvar_RegisterVariable (&scr_fovspeed);	// R00k
	Cvar_RegisterVariable (&scr_viewsize);
	Cvar_RegisterVariable (&scr_consize);	// by joe
	Cvar_RegisterVariable (&scr_conspeed);
	Cvar_RegisterVariable (&scr_showram);
	Cvar_RegisterVariable (&scr_showturtle);
	Cvar_RegisterVariable (&scr_showpause);
	Cvar_RegisterVariable (&scr_centertime);
	Cvar_RegisterVariable (&scr_printspeed);
	Cvar_RegisterVariable (&scr_sshot_type);
	Cvar_RegisterVariable (&scr_centerprint_levelname);

#ifdef GLQUAKE
	Cvar_RegisterVariable (&png_compression_level);
	Cvar_RegisterVariable (&gl_triplebuffer);
	Cvar_RegisterVariable (&jpeg_compression_level);
#endif
// register our commands
	Cmd_AddCommand ("screenshot", SCR_ScreenShot_f);
	Cmd_AddCommand ("sizeup", SCR_SizeUp_f);
	Cmd_AddCommand ("sizedown", SCR_SizeDown_f);

	scr_ram = Draw_PicFromWad ("ram");
	scr_net = Draw_PicFromWad ("net");
	scr_turtle = Draw_PicFromWad ("turtle");

#ifdef _WIN32
	Movie_Init ();
#endif

	scr_initialized = true;
}

/*
==============
SCR_DrawRam
==============
*/
void SCR_DrawRam (void)
{
	if (!scr_showram.value)
		return;

	if (!r_cache_thrash)
		return;

	Draw_Pic (scr_vrect.x+32, scr_vrect.y, scr_ram);
}

/*
==============
SCR_DrawTurtle
==============
*/
void SCR_DrawTurtle (void)
{
	static	int	count;
	
	if (!scr_showturtle.value)
		return;

	if (host_frametime < 0.1)
	{
		count = 0;
		return;
	}

	count++;
	if (count < 3)
		return;

	Draw_Pic (scr_vrect.x, scr_vrect.y, scr_turtle);
}

/*
==============
SCR_DrawNet
==============
*/
void SCR_DrawNet (void)
{
	if (realtime - cl.last_received_message < 0.3)
		return;

	if (cls.demoplayback)
		return;

	Draw_Pic (scr_vrect.x+64, scr_vrect.y, scr_net);
}

/*
==============
DrawPause
==============
*/
void SCR_DrawPause (void)
{
	mpic_t	*pic;

	if (!scr_showpause.value)		// turn off for screenshots
		return;

	if (!cl.paused)
		return;

	pic = Draw_CachePic ("gfx/pause.lmp");
	Draw_Pic ((vid.width - pic->width) / 2, (vid.height - 48 - pic->height) / 2, pic);
}

/*
==============
SCR_DrawLoading
==============
*/
void SCR_DrawLoading (void)
{
	mpic_t	*pic;

	if (!scr_drawloading)
		return;
		
	pic = Draw_CachePic ("gfx/loading.lmp");
	Draw_Pic ((vid.width - pic->width) / 2, (vid.height - 48 - pic->height) / 2, pic);		
}

//=============================================================================

/*
==================
SCR_SetUpToDrawConsole
==================
*/
void SCR_SetUpToDrawConsole (void)
{
	Con_CheckResize ();

	if (scr_drawloading)
		return;		// never a console with loading plaque

// decide on the height of the console
	con_forcedup = !cl.worldmodel || cls.signon != SIGNONS;

	if (con_forcedup)
	{
		scr_conlines = vid.height;		// full screen
		scr_con_current = scr_conlines;
	}
	else if (key_dest == key_console)		// by joe
	{
		scr_conlines = vid.height * scr_consize.value;
		if (scr_conlines < 30)
			scr_conlines = 30;
		if (scr_conlines > vid.height - 10)
			scr_conlines = vid.height - 10;
	}
	else
	{
		scr_conlines = 0;			// none visible
	}

	if (scr_conlines < scr_con_current)
	{
		scr_con_current -= scr_conspeed.value * host_frametime * vid.height / 320;
		if (scr_conlines > scr_con_current)
			scr_con_current = scr_conlines;

	}
	else if (scr_conlines > scr_con_current)
	{
		scr_con_current += scr_conspeed.value * host_frametime * vid.height / 320;
		if (scr_conlines < scr_con_current)
			scr_con_current = scr_conlines;
	}

	if (clearconsole++ < vid.numpages)
		Sbar_Changed ();
	else if (clearnotify++ < vid.numpages)
	{
	}
	else
		con_notifylines = 0;
}
	
/*
==================
SCR_DrawConsole
==================
*/
void SCR_DrawConsole (void)
{
	if (scr_con_current)
	{
		scr_copyeverything = 1;
		Con_DrawConsole (scr_con_current, true);
		clearconsole = 0;
	}
	else
	{
		if (key_dest == key_game || key_dest == key_message)
			Con_DrawNotify ();	// only draw notify in game
	}
}


/* 
============================================================================== 
 
				SCREEN SHOTS 
 
============================================================================== 
*/ 

// stuff added from FuhQuake - joe
extern	unsigned short	ramps[3][256];

void ApplyGamma (byte *buffer, int size)
{
	int	i;

	if (!vid_hwgamma_enabled)
		return;

	for (i=0 ; i<size ; i+=3)
	{
		buffer[i+0] = ramps[0][buffer[i+0]] >> 8;
		buffer[i+1] = ramps[1][buffer[i+1]] >> 8;
		buffer[i+2] = ramps[2][buffer[i+2]] >> 8;
	}
}

int SCR_ScreenShot (char *name)
{
	qboolean	ok = false;
	int		buffersize = glwidth * glheight * 3;
	byte		*buffer;
	char		*ext;

	ext = COM_FileExtension (name);

	buffer = Q_malloc (buffersize);
	glReadPixels (glx, gly, glwidth, glheight, GL_RGB, GL_UNSIGNED_BYTE, buffer);
	ApplyGamma (buffer, buffersize);

	if (!Q_strcasecmp(ext, "jpg"))
		ok = Image_WriteJPEG (name, jpeg_compression_level.value, buffer + buffersize - 3 * glwidth, -glwidth, glheight);
	else if (!Q_strcasecmp(ext, "png"))
		ok = Image_WritePNG (name, png_compression_level.value, buffer + buffersize - 3 * glwidth, -glwidth, glheight);
	else
		ok = Image_WriteTGA (name, buffer, glwidth, glheight);

	Q_free (buffer);

	return ok;
}

/* 
================== 
SCR_ScreenShot_f
================== 
*/  
void SCR_ScreenShot_f (void) 
{
	int	i, success;
	char	name[MAX_OSPATH], ext[4], *sshot_dir = "qrack/screenshots";

	if (Cmd_Argc() == 2)
	{
		Q_strncpyz (name, Cmd_Argv(1), sizeof(name));
	}
	else if (Cmd_Argc() == 1)
	{
		// find a file name to save it to 
		if (!Q_strcasecmp(scr_sshot_type.string, "jpg") || !Q_strcasecmp(scr_sshot_type.string, "jpeg"))
			Q_strncpyz (ext, "jpg", 4);
		else if (!Q_strcasecmp(scr_sshot_type.string, "png"))
			Q_strncpyz (ext, "png", 4);
		else
			Q_strncpyz (ext, "tga", 4);

		for (i=0 ; i<999 ; i++) 
		{ 
			Q_snprintfz (name, sizeof(name), "qrack%03i.%s", i, ext);
			if (Sys_FileTime(va("%s/%s/%s", com_basedir, sshot_dir, name)) == -1)
				break;	// file doesn't exist
		} 

		if (i == 1000)
		{
			Con_Printf ("ERROR: Cannot create more than 1000 screenshots\n");
			return;
		}
	}
	else
	{
		Con_Printf ("Usage: %s [filename]", Cmd_Argv(0));
		return;
	}

	success = SCR_ScreenShot (va("%s/%s", sshot_dir, name));
	Con_Printf ("%s %s\n", success ? "Wrote" : "Couldn't write", name);
} 


//=============================================================================


/*
================
SCR_BeginLoadingPlaque
================
*/
void SCR_BeginLoadingPlaque (void)
{
	S_StopAllSounds (true);

	if (cls.state != ca_connected || cls.signon != SIGNONS)
		return;

	if (((cl_autodemo.value == 1) && (!cls.demoplayback)) || (cls.demorecording))
		Cmd_ExecuteString ("stop\n", src_command);

// redraw with no console and the loading plaque
	Con_ClearNotify ();
	scr_centertime_off = 0;
	scr_con_current = 0;

	scr_drawloading = true;
	scr_fullupdate = 0;
	Sbar_Changed ();
	SCR_UpdateScreen ();
	scr_drawloading = false;

	scr_disabled_for_loading = true;
	scr_disabled_time = realtime;
	scr_fullupdate = 0;
}

/*
================
SCR_EndLoadingPlaque
================
*/
void SCR_EndLoadingPlaque (void)
{
	scr_disabled_for_loading = false;
	scr_fullupdate = 0;
	Con_ClearNotify ();
}

//=============================================================================

char		*scr_notifystring;
qboolean	scr_drawdialog;

void SCR_DrawNotifyString (void)
{
	char	*start;
	int	l, j, x, y;

	start = scr_notifystring;

	y = vid.height*0.35;

	do	
	{
	// scan the width of the line
		for (l=0 ; l<40 ; l++)
			if (start[l] == '\n' || !start[l])
				break;
		x = (vid.width - l*8)/2;
		for (j=0 ; j<l ; j++, x+=8)
			Draw_Character (x, y, start[j],false);	
			
		y += 8;

		while (*start && *start != '\n')
			start++;

		if (!*start)
			break;
		start++;		// skip the \n
	} while (1);
}

/*
==================
SCR_ModalMessage

Displays a text string in the center of the screen and waits for a Y or N
keypress.  
==================
*/
int SCR_ModalMessage (char *text)
{
	if (cls.state == ca_dedicated)
		return true;

	scr_notifystring = text;
 
// draw a fresh screen
	scr_fullupdate = 0;
	scr_drawdialog = true;
	SCR_UpdateScreen ();
	scr_drawdialog = false;
	
	S_ClearBuffer ();		// so dma doesn't loop current sound

	do
	{
		key_count = -1;		// wait for a key down and up
		Sys_SendKeyEvents ();
	} while (key_lastpress != 'y' && key_lastpress != 'n' && key_lastpress != K_ESCAPE);

	scr_fullupdate = 0;
	SCR_UpdateScreen ();

	return key_lastpress == 'y';
}


//=============================================================================

void SCR_TileClear (void)
{
	if (r_refdef.vrect.x > 0)
	{
		// left
		Draw_TileClear (0, 0, r_refdef.vrect.x, vid.height - sb_lines);
		// right
		Draw_TileClear (r_refdef.vrect.x + r_refdef.vrect.width, 0, 
			vid.width - r_refdef.vrect.x + r_refdef.vrect.width, 
			vid.height - sb_lines);
	}
	if (r_refdef.vrect.y > 0)
	{
		// top
		Draw_TileClear (r_refdef.vrect.x, 0, 
			r_refdef.vrect.x + r_refdef.vrect.width, 
			r_refdef.vrect.y);
		// bottom
		Draw_TileClear (r_refdef.vrect.x,
			r_refdef.vrect.y + r_refdef.vrect.height, 
			r_refdef.vrect.width, 
			vid.height - sb_lines - 
			(r_refdef.vrect.height + r_refdef.vrect.y));
	}
}
/*
extern float g_worldMins[3];
extern float g_worldMaxs[3];
void SCR_OrthographicWorldPoint(float *point, float *out, float w, float h, float d)
{
	int i;
	float scales[3];
	scales[0] = w;
	scales[1] = h;
	scales[2] = d;
	for (i = 0; i < 3; i++)
	{
		float ex;
		out[i] = point[i];
		if (out[i] > g_worldMaxs[i])
		{
			out[i] = g_worldMaxs[i];
		}
		else if (out[i] < g_worldMins[i])
		{
			out[i] = g_worldMins[i];
		}
		
		ex = g_worldMaxs[i]-g_worldMins[i];
		out[i] = ((out[i]-g_worldMins[i])/ex)*scales[i];
	}
	out[1] = w-out[1];
}

typedef struct radarObject_s
{
	float		pos[3];
	float		ang;
	double		time;
	qboolean	allied;
	qboolean	valid;
} radarObject_t;
radarObject_t g_radarObjects[MAX_SCOREBOARD];

//rww
void SCR_ClearRadar(void)
{
	memset(g_radarObjects, 0, sizeof(g_radarObjects));
}

//rww
void SCR_AddRadarObject(int i, float *pos, float ang, qboolean allied)
{
	radarObject_t *rad = &g_radarObjects[i];
	float orthoWidth = 512;
	float orthoHeight = 512;
	float orthoDepth = 512;

	////VectorCopy(pos, rad->pos);
	//put in automap coordinates now
	SCR_OrthographicWorldPoint(pos, rad->pos, orthoWidth, orthoHeight, orthoDepth);
	rad->ang = ang;
	rad->time = realtime+1.0;
	rad->valid = true;
	rad->allied = allied;
}

//rww
void SCR_MapPosScaledOutput(float scale, float x, float y, float w, float h, float tx, float ty, float *in, float *out)
{
	float orthoWidth = 512;
	float orthoHeight = 512;
	float orthoDepth = 512;
	out[0] = ((in[0]-(tx*orthoWidth)) * (w/(orthoWidth*scale))) + x;
	out[1] = ((in[1]-(ty*orthoHeight)) * (h/(orthoHeight*scale))) + y;
	out[2] = in[2];
}

//rww
void SCR_DrawAngledMapTri(float scale, float x, float y, float w, float h, float tx, float ty, float tw, float th, float ang, float *pos, float *clr)
{
	float transPos[3];
	float triWidth = 32.0f*0.2f;
	float triHeight = 48.0f*0.2f;
	glPushMatrix();

	SCR_MapPosScaledOutput(scale, x, y, w, h, tx, ty, pos, transPos);

	glTranslatef(transPos[0], transPos[1], transPos[2]);
	glRotatef((-ang)+90.0f, 0.0f, 0.0f, 1.0f);

	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
	glDisable(GL_TEXTURE_2D);
	glBegin(GL_TRIANGLES);
		glColor4f(clr[0]*0.25f, clr[1]*0.25f, clr[2]*0.25f, clr[3]);
		glVertex3f(-(triWidth * 0.5f), (triHeight * 0.5f), 0.0f);
		glColor4f(clr[0]*1.0f, clr[1]*1.0f, clr[2]*1.0f, clr[3]);
		glVertex3f(0.0f, -(triHeight * 0.5f), 0.0f);
		glColor4f(clr[0]*0.6f, clr[1]*0.6f, clr[2]*0.6f, clr[3]);
		glVertex3f((triWidth * 0.5f), (triHeight * 0.5f), 0.0f);
	glEnd();
	glColor4f(1.0f, 1.0f, 1.0f, 1.0f);
	glEnable(GL_TEXTURE_2D);
	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);

	glPopMatrix();	
}

//rww
int g_automapState = 1;
void SCR_AutomapToggle_f(void)
{
	g_automapState++;
	if (g_automapState > 3)
	{
		g_automapState = 0;
	}
}

//rww
float SCR_NudgeVal(float prev, float dest, float base)
{
	if (prev == dest)
	{
		return prev;
	}

	if (fabsf(prev) < 0.001f)
	{
		if (prev < 0.0f)
		{
			prev = -0.001f;
		}
		else
		{
			prev = 0.001f;
		}
	}

	if (prev < dest)
	{
		prev += base*host_frametime;
		if (prev > dest)
		{
			prev = dest;
		}
	}
	else
	{
		prev -= base*host_frametime;
		if (prev < dest)
		{
			prev = dest;
		}
	}

	return prev;
}

//rww
extern int window_width;
extern int window_height;
extern int hr_automap;
extern qboolean g_hasAutomap;
extern float Draw_ScaleBasedOnResW(float f);
extern float Draw_ScaleBasedOnResH(float f);
void SCR_DrawAutomap(void)
{
	float fX, fY, fW, fH;
	float tX, tY, tW, tH;
	float orthoWidth = 512;
	float orthoHeight = 512;
	float orthoDepth = 512;
	float mapPos[3];
	float centerOffs;
	static float downScale = 0.15;//25f;
	static float mapWidth = 96.0f;
	static float mapHeight = 96.0f;
	float goalDS = 0.15f;
	float goalMW = 96.0f;
	float goalMH = 96.0f;
	float clr[4];
	int i;

	if (g_automapState)
	{
		if (mapWidth < 8.0f)
		{
			mapWidth = 8.0f;
		}
		if (mapHeight < 8.0f)
		{
			mapHeight = 8.0f;
		}
		if (downScale < 0.01f)
		{
			downScale = 0.01f;
		}
	}
	else
	{
		if (mapWidth < 2.0f)
		{
			mapWidth = 0.0f;
		}
		if (mapHeight < 2.0f)
		{
			mapHeight = 0.0f;
		}
		if (downScale < 0.01f)
		{
			downScale = 0.01f;
		}
	}

	if (g_automapState == 0)
	{
		goalDS = 0.0f;
		goalMW = 0.0f;
		goalMH = 0.0f;
	}
	else if (g_automapState == 1)
	{
		goalDS = 0.15f;
		goalMW = 96.0f;
		goalMH = 96.0f;
	}
	else if (g_automapState == 2)
	{
		goalDS = 0.4f;
		goalMW = 200.0f;
		goalMH = 200.0f;
	}
	else
	{
		goalDS = 1.0f;
		goalMW = 400.0f;
		goalMH = 400.0f;
	}

	downScale = SCR_NudgeVal(downScale, goalDS, 0.4f);
	mapWidth = SCR_NudgeVal(mapWidth, goalMW, 256.0f);
	mapHeight = SCR_NudgeVal(mapHeight, goalMH, 256.0f);

	if (mapWidth < 0.1f || mapHeight < 0.1f)
	{
		return;
	}

	if (!g_hasAutomap)
	{
		return;
	}

	GL_Bind(hr_automap);

	SCR_OrthographicWorldPoint(cl.simorg, mapPos, orthoWidth, orthoHeight, orthoDepth);

	centerOffs = downScale*0.5f;
	tX = (mapPos[0]/orthoWidth)-centerOffs;
	tY = (mapPos[1]/orthoHeight)-centerOffs;
	tW = downScale;
	tH = downScale;
	if (tX < 0.0f)
	{
		tX = 0.0f;
	}
	if (tY < 0.0f)
	{
		tY = 0.0f;
	}
	if (tX+tW > 1.0f)
	{
		tX = 1.0f-tW;
	}
	if (tY+tH > 1.0f)
	{
		tY = 1.0f-tH;
	}

	if (gl_forceWideAspect.value)
	{
		fX = Draw_ScaleBasedOnResW((640.0f*(640.0f/480.0f))-(mapWidth+8.0f));
	}
	else
	{
		fX = Draw_ScaleBasedOnResW(640.0f-(mapWidth+8.0f));
	}
	fY = Draw_ScaleBasedOnResH(8.0f);
	fW = Draw_ScaleBasedOnResW(mapWidth);
	fH = Draw_ScaleBasedOnResH(mapHeight);

	if (gl_forceWideAspect.value)
	{
		float v = (480.0f/640.0f);
		glScissor(fX*((float)window_width/(float)vid.width)*v,
			((float)vid.height-fY-fH)*((float)window_height/(float)vid.height),
			fW*((float)window_width/(float)vid.width)*v,
			fH*((float)window_height/(float)vid.height));
	}
	else
	{
		glScissor(fX*((float)window_width/(float)vid.width),
			((float)vid.height-fY-fH)*((float)window_height/(float)vid.height),
			fW*((float)window_width/(float)vid.width),
			fH*((float)window_height/(float)vid.height));
	}
	glEnable(GL_SCISSOR_TEST);

	glShadeModel(GL_SMOOTH);

	glBegin(GL_QUADS);
		glTexCoord2f (tX, tY);
		glVertex2f (fX, fY);
		glTexCoord2f (tX + tW, tY);
		glVertex2f (fX+fW, fY);
		glTexCoord2f (tX + tW, tY + tH);
		glVertex2f (fX+fW, fY+fH);
		glTexCoord2f (tX, tY + tH);
		glVertex2f (fX, fY+fH);
	glEnd();

	//self
	clr[0] = 0.0f;
	clr[1] = 1.0f;
	clr[2] = 0.0f;
	clr[3] = 1.0f;
	SCR_DrawAngledMapTri(downScale, fX, fY, fW, fH, tX, tY, tW, tH, cl.simangles[YAW], mapPos, clr);

	for (i = 0; i < MAX_CLIENTS; i++)
	{ //other client blips
		radarObject_t *rad = &g_radarObjects[i];
		if (rad->valid && rad->time > realtime)
		{
			glEnable(GL_BLEND);
			glDisable(GL_ALPHA_TEST);
			if (rad->allied)
			{
				clr[0] = 0.0f;
				clr[1] = 1.0f;
				clr[2] = 0.0f;
			}
			else
			{
				clr[0] = 1.0f;
				clr[1] = 0.0f;
				clr[2] = 0.0f;
			}
			clr[3] = rad->time-realtime;
			SCR_DrawAngledMapTri(	downScale,
									fX, fY, fW, fH,
									tX, tY, tW, tH,
									rad->ang, rad->pos, clr);
			glEnable(GL_ALPHA_TEST);
			glDisable(GL_BLEND);
		}
	}

	//the outline
	glDisable(GL_TEXTURE_2D);
	glLineWidth(2.0f);
	glBegin(GL_LINES);
		glColor4f(0.0f, 0.15f, 0.0f, 1.0f);
		glVertex2f (fX, fY);
		glColor4f(0.0f, 0.45f, 0.0f, 1.0f);
		glVertex2f (fX+fW, fY);

		glVertex2f (fX+fW, fY);
		glColor4f(0.0f, 0.15f, 0.0f, 1.0f);
		glVertex2f (fX+fW, fY+fH);

		glVertex2f (fX+fW, fY+fH);
		glColor4f(0.0f, 0.45f, 0.0f, 1.0f);
		glVertex2f (fX, fY+fH);

		glVertex2f (fX, fY+fH);
		glColor4f(0.0f, 0.15f, 0.0f, 1.0f);
		glVertex2f (fX, fY);
	glEnd();
	glEnable(GL_TEXTURE_2D);
	glColor4f(1.0f, 1.0f, 1.0f, 1.0f);
	glLineWidth(1.0f);

	glShadeModel(GL_FLAT);
	glDisable(GL_SCISSOR_TEST);
}
*/

float	oldsbar = 0;
/*
==================
SCR_UpdateScreen

This is called every frame, and can also be called explicitly to flush
text to the screen.

WARNING: be very careful calling this from elsewhere, because the refresh
needs almost the entire 256k of stack space!
==================
*/

void SCR_UpdateScreen (void)
{
	extern void Sbar_DrawPing ();
	extern void Draw_MapShotBackground (char *lvlname, float alpha);
	extern cvar_t vid_refreshrate;
	extern cvar_t cl_levelshots;

	float alpha;

	if (cls.state == ca_dedicated)//r00K
		return;

	if (block_drawing)
		return;

	if (scr_disabled_for_loading)
	{
		if (realtime - scr_disabled_time > 60)
			scr_disabled_for_loading = false;
		else
			return;
	}

	if (!scr_initialized || !con_initialized)
		return;				// not initialized yet

#ifdef _WIN32
	{	// don't suck up any cpu if minimized
		extern	int	Minimized;

		if (Minimized)
			return;
	}
#endif

	vid.numpages = 2 + gl_triplebuffer.value;

	scr_copytop = 0;
	scr_copyeverything = 0;

	if (oldsbar != cl_sbar.value)
	{
		oldsbar = cl_sbar.value;
		vid.recalc_refdef = 1;
	}

	// determine size of refresh window
	if (oldfov != scr_fov.value)
	{      
		if (scr_fov.value > oldfov)
		{
			oldfov += min(scr_fovspeed.value,scr_fov.value);
			if (oldfov > scr_fov.value)
				oldfov = scr_fov.value;
		}
		else
		{
			oldfov -= min(scr_fovspeed.value,scr_fov.value);
			if (oldfov < scr_fov.value)
				oldfov = scr_fov.value;
		}
		vid.recalc_refdef = 1;
	} 

	if (oldscreensize != scr_viewsize.value)
	{
		oldscreensize = scr_viewsize.value;
		vid.recalc_refdef = 1;
	}

	if (vid.recalc_refdef)
		SCR_CalcRefdef ();

	if ((v_contrast.value > 1 && !vid_hwgamma_enabled) || gl_clear.value)
		Sbar_Changed ();

// do 3D refresh drawing, and then update the screen
	GL_BeginRendering (&glx, &gly, &glwidth, &glheight);
	
	SCR_SetUpToDrawConsole ();
	
	V_RenderView ();

	GL_Set2D ();

	vaEnableVertexArray (2);
	vaEnableTexCoordArray (GL_TEXTURE0, VA_TEXTURE0, 2);
	
// added by joe - IMPORTANT: this _must_ be here so that 
//			     palette flashes take effect in windowed mode too.
	R_PolyBlend ();

	// draw any areas not covered by the refresh
	SCR_TileClear ();

	if (scr_drawdialog)
	{
		Sbar_Draw ();
		Draw_FadeScreen ();
		SCR_DrawNotifyString ();
		scr_copyeverything = true;
	}
	else if (scr_drawloading)
	{
		SCR_DrawLoading ();	
	}
	else if (cl.intermission == 1 && key_dest == key_game)
	{
		Sbar_IntermissionOverlay ();
	}
	else if (cl.intermission == 2 && key_dest == key_game)
	{
		Sbar_FinaleOverlay ();
		SCR_CheckDrawCenterString ();
	}
	else if (cl.intermission == 3 && key_dest == key_game)
	{
		SCR_CheckDrawCenterString ();
	}
	else
	{
		Draw_Crosshair ();
		SCR_DrawRam ();
		SCR_DrawNet ();
		SCR_DrawTurtle ();
		SCR_DrawPause ();
		
		if (nehahra)
			SHOWLMP_drawall ();

		SCR_CheckDrawCenterString ();
		SCR_DrawLocalTime ();
		SCR_DrawFPS ();

		if (cls.demorecording)//R00k
		{			
			SCR_DrawREC();
		}

		Sbar_DrawPing ();
		SCR_DrawLocName();
		SCR_DrawSpeed ();
		Sbar_Draw ();
		SCR_DrawConsole ();	
		M_Draw ();
	}

	R_BrightenScreen ();

	V_UpdatePalette ();

	if (cl_levelshots.value)//R00k: v1.92
	{
		if ((cl.lvlshot_time - cl.ctime > 0) && (fabs(cl.time - cl.oldtime)))
		{
			alpha = ((1-(cl.ctime/cl.lvlshot_time))*(cl.lvlshot_time - cl.ctime));//phew!

			if (alpha > 0)
				Draw_MapShotBackground(cl_mapname.string, alpha);
		}
	}

	vaDisableArrays();

#ifdef _WIN32
	Movie_UpdateScreen ();
#endif

	GL_EndRendering ();
}
