#include "light.h"
#include "heapmem.h"
#include "winui.h"


FILE *FindFileInPaks (char *basepath, char *filename);
void BalanceColour (float *c, float target);

qboolean hasbasepal;

void WhiteBias (float *colour, float bias)
{
	int i;

	for (i = 0; i < 3; i++)
	{
		colour[i] += bias;
		colour[i] /= (256.0f + bias);
		colour[i] *= 256.0f;
	}
}


#define SURF_UNDERWATER 0x80

// basic quake palette if we can't find a palette.lmp (this is a fallback, we always attempt to load a palette.lmp and use that instead)
// this is also used for comparison with the loaded palette - if they are the same we use a number of other colours from the palette (aside
// from just fullbrights) for lighting, otherwise we can't rely on those colour indexes to be valid so we won't use them.
// this data was taken directly from a dump of a standard quake1 CD
int basepal[256][3] =
{
	{  0,   0,   0}, { 15,  15,  15}, { 31,  31,  31}, { 47,  47,  47}, { 63,  63,  63}, { 75,  75,  75}, { 91,  91,  91}, {107, 107, 107},
	{123, 123, 123}, {139, 139, 139}, {155, 155, 155}, {171, 171, 171}, {187, 187, 187}, {203, 203, 203}, {219, 219, 219}, {235, 235, 235},
	{ 15,  11,   7}, { 23,  15,  11}, { 31,  23,  11}, { 39,  27,  15}, { 47,  35,  19}, { 55,  43,  23}, { 63,  47,  23}, { 75,  55,  27},
	{ 83,  59,  27}, { 91,  67,  31}, { 99,  75,  31}, {107,  83,  31}, {115,  87,  31}, {123,  95,  35}, {131, 103,  35}, {143, 111,  35},
	{ 11,  11,  15}, { 19,  19,  27}, { 27,  27,  39}, { 39,  39,  51}, { 47,  47,  63}, { 55,  55,  75}, { 63,  63,  87}, { 71,  71, 103},
	{ 79,  79, 115}, { 91,  91, 127}, { 99,  99, 139}, {107, 107, 151}, {115, 115, 163}, {123, 123, 175}, {131, 131, 187}, {139, 139, 203},
	{  0,   0,   0}, {  7,   7,   0}, { 11,  11,   0}, { 19,  19,   0}, { 27,  27,   0}, { 35,  35,   0}, { 43,  43,   7}, { 47,  47,   7},
	{ 55,  55,   7}, { 63,  63,   7}, { 71,  71,   7}, { 75,  75,  11}, { 83,  83,  11}, { 91,  91,  11}, { 99,  99,  11}, {107, 107,  15},
	{  7,   0,   0}, { 15,   0,   0}, { 23,   0,   0}, { 31,   0,   0}, { 39,   0,   0}, { 47,   0,   0}, { 55,   0,   0}, { 63,   0,   0},
	{ 71,   0,   0}, { 79,   0,   0}, { 87,   0,   0}, { 95,   0,   0}, {103,   0,   0}, {111,   0,   0}, {119,   0,   0}, {127,   0,   0},
	{ 19,  19,   0}, { 27,  27,   0}, { 35,  35,   0}, { 47,  43,   0}, { 55,  47,   0}, { 67,  55,   0}, { 75,  59,   7}, { 87,  67,   7},
	{ 95,  71,   7}, {107,  75,  11}, {119,  83,  15}, {131,  87,  19}, {139,  91,  19}, {151,  95,  27}, {163,  99,  31}, {175, 103,  35},
	{ 35,  19,   7}, { 47,  23,  11}, { 59,  31,  15}, { 75,  35,  19}, { 87,  43,  23}, { 99,  47,  31}, {115,  55,  35}, {127,  59,  43},
	{143,  67,  51}, {159,  79,  51}, {175,  99,  47}, {191, 119,  47}, {207, 143,  43}, {223, 171,  39}, {239, 203,  31}, {255, 243,  27},
	{ 11,   7,   0}, { 27,  19,   0}, { 43,  35,  15}, { 55,  43,  19}, { 71,  51,  27}, { 83,  55,  35}, { 99,  63,  43}, {111,  71,  51},
	{127,  83,  63}, {139,  95,  71}, {155, 107,  83}, {167, 123,  95}, {183, 135, 107}, {195, 147, 123}, {211, 163, 139}, {227, 179, 151},
	{171, 139, 163}, {159, 127, 151}, {147, 115, 135}, {139, 103, 123}, {127,  91, 111}, {119,  83,  99}, {107,  75,  87}, { 95,  63,  75},
	{ 87,  55,  67}, { 75,  47,  55}, { 67,  39,  47}, { 55,  31,  35}, { 43,  23,  27}, { 35,  19,  19}, { 23,  11,  11}, { 15,   7,   7},
	{187, 115, 159}, {175, 107, 143}, {163,  95, 131}, {151,  87, 119}, {139,  79, 107}, {127,  75,  95}, {115,  67,  83}, {107,  59,  75},
	{ 95,  51,  63}, { 83,  43,  55}, { 71,  35,  43}, { 59,  31,  35}, { 47,  23,  27}, { 35,  19,  19}, { 23,  11,  11}, { 15,   7,   7},
	{219, 195, 187}, {203, 179, 167}, {191, 163, 155}, {175, 151, 139}, {163, 135, 123}, {151, 123, 111}, {135, 111,  95}, {123,  99,  83},
	{107,  87,  71}, { 95,  75,  59}, { 83,  63,  51}, { 67,  51,  39}, { 55,  43,  31}, { 39,  31,  23}, { 27,  19,  15}, { 15,  11,   7},
	{111, 131, 123}, {103, 123, 111}, { 95, 115, 103}, { 87, 107,  95}, { 79,  99,  87}, { 71,  91,  79}, { 63,  83,  71}, { 55,  75,  63},
	{ 47,  67,  55}, { 43,  59,  47}, { 35,  51,  39}, { 31,  43,  31}, { 23,  35,  23}, { 15,  27,  19}, { 11,  19,  11}, {  7,  11,   7},
	{255, 243,  27}, {239, 223,  23}, {219, 203,  19}, {203, 183,  15}, {187, 167,  15}, {171, 151,  11}, {155, 131,   7}, {139, 115,   7},
	{123,  99,   7}, {107,  83,   0}, { 91,  71,   0}, { 75,  55,   0}, { 59,  43,   0}, { 43,  31,   0}, { 27,  15,   0}, { 11,   7,   0},
	{  0,   0, 255}, { 11,  11, 239}, { 19,  19, 223}, { 27,  27, 207}, { 35,  35, 191}, { 43,  43, 175}, { 47,  47, 159}, { 47,  47, 143},
	{ 47,  47, 127}, { 47,  47, 111}, { 47,  47,  95}, { 43,  43,  79}, { 35,  35,  63}, { 27,  27,  47}, { 19,  19,  31}, { 11,  11,  15},
	{ 43,   0,   0}, { 59,   0,   0}, { 75,   7,   0}, { 95,   7,   0}, {111,  15,   0}, {127,  23,   7}, {147,  31,   7}, {163,  39,  11},
	{183,  51,  15}, {195,  75,  27}, {207,  99,  43}, {219, 127,  59}, {227, 151,  79}, {231, 171,  95}, {239, 191, 119}, {247, 211, 139},
	{167, 123,  59}, {183, 155,  55}, {199, 195,  55}, {231, 227,  87}, {127, 191, 255}, {171, 231, 255}, {215, 255, 255}, {103,   0,   0},
	{139,   0,   0}, {179,   0,   0}, {215,   0,   0}, {255,   0,   0}, {255, 243, 147}, {255, 247, 199}, {255, 255, 255}, {159,  91,  83}
};


int *qpal[256];


// rgb2hsl and hsl2rgb algorithms from http://www.geekymonkey.com/Programming/CSharp/RGB2HSL_HSL2RGB.htm
static void RGB2HSL_Finish (float *base, float h, float s, float l)
{
	while (h > 1.0f) h -= 1.0f;
	while (h < 0.0f) h += 1.0f;

	// the algorithm below assumes a 255 scale for h (not angles as per trad hsl)
	base[0] = h * 255.0f;
	base[1] = s * 255.0f;
	base[2] = l * 255.0f;
}


void RGB2HSL (float *rgb)
{
	float r = rgb[0] / 255.0f;
	float g = rgb[1] / 255.0f;
	float b = rgb[2] / 255.0f;
	float v;
	float m;
	float vm;
	float r2, g2, b2;

	// default to black
	float h = 0;
	float s = 0;
	float l = 0;

	v = r > g ? r : g;
	v = v > b ? v : b;
	m = r > g ? g : r;
	m = m > b ? b : m;

	l = (m + v) / 2.0f;

	if (l <= 0.0)
	{
		RGB2HSL_Finish (rgb, h, s, l);
		return;
	}

	vm = v - m;
	s = vm;

	if (s > 0.0)
	{
		s /= (l <= 0.5f) ? (v + m) : (2.0f - v - m);
	} 
	else 
	{
		RGB2HSL_Finish (rgb, h, s, l);
		return;
	}

	r2 = (v - r) / vm;
	g2 = (v - g) / vm;
	b2 = (v - b) / vm;

	if (r == v)
	{
		h = (g == m ? 5.0f + b2 : 1.0f - g2);
	}
	else if (g == v)
	{
		h = (b == m ? 1.0f + r2 : 3.0f - b2);
	}
	else
	{
		h = (r == m ? 3.0f + g2 : 5.0f - r2);
	}

	h /= 6.0;

	RGB2HSL_Finish (rgb, h, s, l);
}


void HSL2RGB (float *hsl)
{
	float h = hsl[0] / 255.0f;
	float s = hsl[1] / 255.0f;
	float l = hsl[2] / 255.0f;
	double v;
	double r, g, b;

	// default to gray
	r = l;
	g = l;
	b = l;
	v = (l <= 0.5f) ? (l * (1.0f + s)) : (l + s - l * s);

	if (v > 0)
	{
		double m;
		double sv;
		int sextant;
		double fract, vsf, mid1, mid2;

		m = l + l - v;
		sv = (v - m ) / v;
		h *= 6.0f;

		sextant = (int) h;
		fract = h - sextant;
		vsf = v * sv * fract;

		mid1 = m + vsf;
		mid2 = v - vsf;

		switch (sextant) 
		{
		case 0: 
			r = v; 
			g = mid1; 
			b = m; 
			break;

		case 1: 
			r = mid2; 
			g = v; 
			b = m; 
			break;

		case 2: 
			r = m; 
			g = v; 
			b = mid1; 
			break;

		case 3: 
			r = m; 
			g = mid2; 
			b = v; 
			break;

		case 4: 
			r = mid1; 
			g = m; 
			b = v; 
			break;

		case 5: 
			r = v; 
			g = m; 
			b = mid2; 
			break;
		}
	}

	// store back to input
	hsl[0] = r * 255.0f;
	hsl[1] = g * 255.0f;
	hsl[2] = b * 255.0f;
}


void AverageColour (char *texname, int width, int height, byte *data, int threshold, int *colour, float add)
{
}


// simplified versions of the full game memory versions
typedef struct texture_s
{
	int width;
	int height;
	char name[16];
	byte *data;
	qboolean haslight;
	qboolean liquid;
	float colour[3];
	int intensity;

	// texture animations
	int			anim_total;				// total tenths in sequence ( 0 = no)
	int			anim_min, anim_max;		// time for this frame min <=time< max
	struct texture_s *anim_next;		// in the animation sequence
	struct texture_s *alternate_anims;	// bmodels in frmae 1 use these
} texture_t;


typedef struct mplane_s
{
	vec3_t	normal;
	float	dist;
	byte	type;			// for texture axis selection and fast side tests
	byte	signbits;		// signx + signy<<1 + signz<<1
	byte	pad[2];
} mplane_t;


typedef struct
{
	float		vecs[2][4];
	float		mipadjust;
	texture_t	*texture;
	int			flags;
} mtexinfo_t;


#define VERTEXSIZE	5

typedef struct glpoly_s
{
	struct	glpoly_s	*next;
	vec3_t	midpoint;
	int		numverts;
	float	verts[4][VERTEXSIZE];	// variable sized (xyz st)
} glpoly_t;

typedef struct msurface_s
{
	dface_t *face;
	int surfnum;
	mtexinfo_t *texinfo;
	glpoly_t *polys;
	float normal[3];
	int numedges;
	int firstedge;
	int numsubdivisions;
	int flags;
	byte *lightvis;
	int visframe;
} msurface_t;


typedef struct mnode_s
{
// common with leaf
	int			contents;		// 0, to differentiate from leafs
	int			visframe;		// node needs to be traversed if current
	
	float		minmaxs[6];		// for bounding box culling

	struct mnode_s	*parent;

// node specific
	mplane_t	*plane;
	struct mnode_s	*children[2];	

	unsigned short		firstsurface;
	unsigned short		numsurfaces;
} mnode_t;



typedef struct mleaf_s
{
// common with node
	int			contents;		// wil be a negative contents number
	int			visframe;		// node needs to be traversed if current

	float		minmaxs[6];		// for bounding box culling

	struct mnode_s	*parent;

// leaf specific
	int			leafnum;
	byte		*compressed_vis;

	msurface_t	**firstmarksurface;
	int			nummarksurfaces;
	int			key;			// BSP sequence number for leaf's contents
	byte		ambient_sound_level[NUM_AMBIENTS];
} mleaf_t;


typedef struct loadmodel_s
{
	texture_t *textures;
	mplane_t *planes;
	mtexinfo_t *texinfo;
	msurface_t *surfaces;
	msurface_t **marksurfaces;
	mnode_t *nodes;
	mleaf_t *leafs;

	int numcolourpoints;
} loadmodel_t;


loadmodel_t loadmodel;


int AccumulateLight (float *base, int *add)
{
	int gs = (add[0] * 30 + add[1] * 59 + add[2] * 11) / 100;
	float scale;

	// scale brighter colours higher
	if (gs > 224)
		scale = 128;
	else if (gs > 192)
		scale = 64;
	else if (gs > 160)
		scale = 32;
	else if (gs > 128)
		scale = 16;
	else if (gs > 96)
		scale = 8;
	else if (gs > 64)
		scale = 4;
	else if (gs > 32)
		scale = 2;
	else scale = 1;

	base[0] += (float) add[0] * scale;
	base[1] += (float) add[1] * scale;
	base[2] += (float) add[2] * scale;

	return (int) scale;
}


void BoostColour (float *colour, float boost, float whitebias)
{
	int i;

	whitebias *= 2;

	if (boost < 1) return;

	for (i = 0; i < 3; i++)
	{
		colour[i] *= boost;
		colour[i] += whitebias;
		colour[i] *= (256.0f / (256.0f + whitebias));
	}
}


void MakeTextureLightColour (texture_t *tx)
{
	int i;
	int threshold;
	int hit;

	// initial
	tx->colour[0] = tx->colour[1] = tx->colour[2] = 0;
	tx->haslight = false;
	tx->intensity = 0;

	if (!stricmp (tx->name, "clip"))
		return;
	else if (!stricmp (tx->name, "trigger"))
		return;
	else if (!strnicmp (tx->name, "*lava", 5))
		threshold = 0;
	else if (!strnicmp (tx->name, "*slime", 6))
		threshold = 0;
	else if (tx->name[0] == '*')
		return;
	else threshold = 223;

	for (i = 0, hit = 0; i < (tx->width * tx->height); i++)
	{
		// bias slightly towards white
		tx->colour[0] += 1.0f;
		tx->colour[1] += 1.0f;
		tx->colour[2] += 1.0f;

		if (tx->data[i] > threshold)
		{
			tx->colour[0] += qpal[tx->data[i]][0];
			tx->colour[1] += qpal[tx->data[i]][1];
			tx->colour[2] += qpal[tx->data[i]][2];
			hit++;
		}
	}

	// no colours, try for fake light
	if (!hit)
	{
		int highest = 0;
		int lowest = 256;
		int factor;
		int j;

		for (i = 0; i < (tx->width * tx->height); i++)
		{
			// because these are 8 bit palette entries they can never be > 255
			highest = 0;
			lowest = 256;

			// ignore
			if (tx->data[i] > 223) continue;

			// figure highest and lowest colour for this point
			for (j = 0; j < 3; j++)
			{
				if (qpal[tx->data[i]][j] > highest) highest = qpal[tx->data[i]][j];
				if (qpal[tx->data[i]][j] < lowest) lowest = qpal[tx->data[i]][j];
			}

			// accumulate hit points for brighter colours even if we don't add them to the texture colour
			if (highest > 128) hit++;

			if (highest > 223 || (highest - lowest) > 160 || (lowest < 8 && highest > 64))
			{
				tx->colour[0] += qpal[tx->data[i]][0];
				tx->colour[1] += qpal[tx->data[i]][1];
				tx->colour[2] += qpal[tx->data[i]][2];
			}
		}

		if (hit < 8) return;

		for (i = 0; i < 3; i++)
		{
			int final = tx->colour[i];
			final /= 8;
			tx->colour[i] = final * 8;
		}

		BalanceColour (tx->colour, 255);

		// the final scale *should* never be > 255 but we have no absolute guarantee so we ensure that we have a reasonably
		// huge initial lowest value to compare with
		highest = 0;
		lowest = (1 << 30);

		for (i = 0; i < 3; i++)
		{
			if (tx->colour[i] > highest) highest = tx->colour[i];
			if (tx->colour[i] < lowest) lowest = tx->colour[i];
		}

		if (highest <= lowest)
			return;
		else if (highest - lowest < 32)
			factor = 7;
		else if (highest - lowest < 64)
			factor = 6;
		else if (highest - lowest < 96)
			factor = 5;
		else if (highest - lowest < 128)
			factor = 4;
		else if (highest - lowest < 160)
			factor = 3;
		else if (highest - lowest < 192)
			factor = 2;
		else if (highest - lowest < 224)
			factor = 1;
		else factor = 0;

		for (j = 0; j < factor; j++) WhiteBias (tx->colour, 32.0f);

		WhiteBias (tx->colour, (tx->width * tx->height) / (16 * 16));
	}
	else
	{
		BalanceColour (tx->colour, 255);
	}

	if ((int) tx->colour[0] / 16 == (int) tx->colour[1] / 16 && (int) tx->colour[1] / 16 == (int) tx->colour[2] / 16)
	{
		// skip textures where the colours are sufficiently close so as to be as near to white as makes no difference
		tx->intensity = 0;
		return;
	}

	// evaluate intensity
	for (i = 16, tx->intensity = 32; ; i <<= 1, tx->intensity += 32)
		if (hit < i || tx->intensity >= 256) break;

	// special handling for slime - one of the ID1 slime textures has a HUGE red balance, so scale up green to compensate
	if (!strnicmp (tx->name, "*slime", 6))
	{
		qboolean rebal = false;

		// make an attempt to catch these where a higher red balance is intentional, based on the assumption
		// that if so, red will be quite a good bit higher than green...
		while (tx->colour[0] > (tx->colour[1] * 1.125))
		{
			// this is always a possibility...
			if (tx->colour[1] < 1) break;

			// higher = too green, lower = too yellow
			tx->colour[1] *= 1.375f;
			rebal = true;
		}

		if (rebal) BalanceColour (tx->colour, 255);
	}

	{
		//FILE *f = fopen ("lights.txt", "a");
		//fprintf (f, "%-16s\t%3.3f\t%3.3f\t%3.3f\t%i\t%i\n", tx->name, tx->colour[0], tx->colour[1], tx->colour[2], hit, tx->intensity);
		//fclose (f);
	}
}


void Make666Texture666Light666Colour666 (texture_t *tx)
{
	int i;
	int scale;
	int hit;
	int add = 0;
	int threshold = 0;
	qboolean falselight;

	// initial
	tx->colour[0] = tx->colour[1] = tx->colour[2] = 0;
	tx->haslight = false;
	tx->intensity = 0;
	hit = 0;

	// determine how we colour this texture
	// initially assume that we're using false light
	falselight = true;

	if (!strnicmp (tx->name, "*lava", 5))
	{
		falselight = false;
		threshold = 0;
		add = 0;
	}
	else if (!strnicmp (tx->name, "*slime", 6))
	{
		falselight = false;
		threshold = 0;
		add = 0;
	}
	else if (tx->name[0] == '*')
		return;
	else
	{
		falselight = false;
		threshold = 223;
		add = 32;
	}

	// textures with fullbright should never generate false light
	// (let the fullbrights define the colour)
	for (i = 0; i < tx->width * tx->height; i++)
	{
		if (tx->data[i] > threshold)
		{
			add = 32;
			falselight = false;
			break;
		}
	}

	// find false lights
	for (i = 0; ; i++)
	{
		if (tx->name[i] == 0) break;

		/*
		if (!strnicmp (&tx->name[i], "light", 5))
		{
			add = 64;
			falselight = true;
			break;
		}

		if (!strnicmp (&tx->name[i], "lite", 4))
		{
			add = 64;
			falselight = true;
			break;
		}

		if (tx->name[0] == '+' || tx->name[0] == '-')
		{
			add = 96;
			falselight = true;
			break;
		}

		if (!strnicmp (&tx->name[i], "window", 6))
		{
			add = 128;
			falselight = true;
			break;
		}

		if (!strnicmp (&tx->name[i], "wizwin", 6))
		{
			add = 128;
			falselight = true;
			break;
		}
		*/
	}

	for (i = 0; i < tx->width * tx->height; i++)
	{
		// never has colour
		if (tx->data[i] >= 254) continue;

		// same intensity in each channel
		// note - this is a valid condition for accumulating as white is good for lighting with and can blend with colour!!!
		// if (qpal[tx->data[i]][0] == qpal[tx->data[i]][1] && qpal[tx->data[i]][1] == qpal[tx->data[i]][2]) continue;

		if (tx->data[i] > threshold)
		{
			// accumulate
			scale = AccumulateLight (tx->colour, qpal[tx->data[i]]);
			tx->haslight = true;
			tx->intensity += scale;
			hit++;
		}
		else if (falselight)
		{
			int fakecolour[3];
			int j;

			for (j = 0; j < 3; j++)
			{
				// hack to ensure that brighter colours dominate properly
				fakecolour[j] = (((int) qpal[tx->data[i]][j] * (int) qpal[tx->data[i]][j])) / 255.0f;
				fakecolour[j] += 64.0f;
				fakecolour[j] *= 0.8f;
				fakecolour[j] -= 51.2f;
				fakecolour[j] *= 1.25f;

				if (fakecolour[j] < 0) fakecolour[j] = 0;
			}

#define FAKESHIFT 2
			if ((fakecolour[0] >> FAKESHIFT) == (fakecolour[1] >> FAKESHIFT) && 
				(fakecolour[1] >> FAKESHIFT) == (fakecolour[2] >> FAKESHIFT)) continue;

			if (fakecolour[0] || fakecolour[1] || fakecolour[2])
			{
				scale = AccumulateLight (tx->colour, fakecolour);
				tx->haslight = true;
				tx->intensity += scale;
				hit++;
			}
		}
	}

	if (!tx->haslight) return;

	for (i = 0; i < 3; i++)
	{
		tx->colour[i] /= tx->intensity;
		tx->colour[i] += add;
	}

	// hack - boost the green on slime a little to prevent palette crapness
	if (!strnicmp (tx->name, "*slime", 6)) tx->colour[1] *= 1.25;

	BalanceColour (tx->colour, 255);

	// get the final correct intensity
	hit = sqrt (hit);

	// scale up colour as smaller intensities fade too much
	if (hit < 2)
	{
		tx->intensity = 32;
		BoostColour (tx->colour, 5, 32);
	}
	else if (hit < 5)
	{
		tx->intensity = 64;
		BoostColour (tx->colour, 4, 24);
	}
	else if (hit < 10)
	{
		tx->intensity = 128;
		BoostColour (tx->colour, 3, 16);
	}
	else if (hit < 15)
	{
		tx->intensity = 192;
		BoostColour (tx->colour, 2, 8);
	}
	else tx->intensity = 256;
}


void ColourLight (entity_t *light, float *surfpt, dface_t *face, float add)
{
	// retrieve the face number (in case we ever want to index into the surf array
	int facenum = (face - dfaces) / sizeof (dface_t);
	texture_t *tx = loadmodel.texinfo[face->texinfo].texture;

	light->colour[0] += (tx->colour[0] * tx->intensity) / 255;
	light->colour[1] += (tx->colour[1] * tx->intensity) / 255;
	light->colour[2] += (tx->colour[2] * tx->intensity) / 255;
}


void Mod_SequenceAnimations (loadmodel_t *brushmodel, dmiptexlump_t *m)
{
	int		i, j, pixels, num, max, altmax;
	miptex_t	*mt;
	texture_t	*tx, *tx2;
	texture_t	*anims[10];
	texture_t	*altanims[10];

	// sequence the animations
	for (i=0 ; i<m->nummiptex ; i++)
	{
		tx = &brushmodel->textures[i];

		if (!tx || tx->name[0] != '+')
			continue;
		if (tx->anim_next)
			continue;	// allready sequenced

		// find the number of frames in the animation
		memset (anims, 0, sizeof(anims));
		memset (altanims, 0, sizeof(altanims));

		max = tx->name[1];
		altmax = 0;

		if (max >= 'a' && max <= 'z') max -= 'a' - 'A';

		if (max >= '0' && max <= '9')
		{
			max -= '0';
			altmax = 0;
			anims[max] = tx;
			max++;
		}
		else if (max >= 'A' && max <= 'J')
		{
			altmax = max - 'A';
			max = 0;
			altanims[altmax] = tx;
			altmax++;
		}
		else Error ("Bad animating texture %s", tx->name);

		for (j=i+1 ; j<m->nummiptex ; j++)
		{
			tx2 = &brushmodel->textures[j];
			if (!tx2 || tx2->name[0] != '+')
				continue;
			if (strcmp (tx2->name+2, tx->name+2))
				continue;

			num = tx2->name[1];
			if (num >= 'a' && num <= 'z')
				num -= 'a' - 'A';
			if (num >= '0' && num <= '9')
			{
				num -= '0';
				anims[num] = tx2;
				if (num+1 > max)
					max = num + 1;
			}
			else if (num >= 'A' && num <= 'J')
			{
				num = num - 'A';
				altanims[num] = tx2;
				if (num+1 > altmax)
					altmax = num+1;
			}
			else Error ("Bad animating texture %s", tx->name);
		}
		
#define	ANIM_CYCLE	2
		// link them all together
		for (j=0 ; j<max ; j++)
		{
			tx2 = anims[j];
			if (!tx2)
				Error ("Missing frame %i of %s",j, tx->name);
			tx2->anim_total = max * ANIM_CYCLE;
			tx2->anim_min = j * ANIM_CYCLE;
			tx2->anim_max = (j+1) * ANIM_CYCLE;
			tx2->anim_next = anims[ (j+1)%max ];
			if (altmax)
				tx2->alternate_anims = altanims[0];
		}
		for (j=0 ; j<altmax ; j++)
		{
			tx2 = altanims[j];
			if (!tx2)
				Error ("Missing frame %i of %s",j, tx->name);
			tx2->anim_total = altmax * ANIM_CYCLE;
			tx2->anim_min = j * ANIM_CYCLE;
			tx2->anim_max = (j+1) * ANIM_CYCLE;
			tx2->anim_next = altanims[ (j+1)%altmax ];
			if (max)
				tx2->alternate_anims = anims[0];
		}
	}
}


void Mod_LoadTextures (void)
{
	dmiptexlump_t *m;
	miptex_t	*mt;
	int i;
	texture_t *tx;

	{
		//FILE *f = fopen ("lights.txt", "w");
		//fclose (f);
	}

	m = (dmiptexlump_t *) dtexdata;

	loadmodel.textures = QHeap_Alloc (m->nummiptex * sizeof (texture_t));

	for (i = 0; i < m->nummiptex; i++)
	{
		// no texture
		if (m->dataofs[i] == -1) continue;

		// locate the miptex data
		mt = (miptex_t *) ((byte *) m + m->dataofs[i]);

		// copy it in
		strcpy (loadmodel.textures[i].name, mt->name);
		loadmodel.textures[i].width = mt->width;
		loadmodel.textures[i].height = mt->height;
		loadmodel.textures[i].data = (byte *) (mt + 1);

		tx = &loadmodel.textures[i];

		// defaults
		tx->haslight = false;
		tx->liquid = false;
		tx->colour[0] = tx->colour[1] = tx->colour[2] = 0;
		tx->intensity = 0;

		// UpdateProgressNotify ("Loaded %s\r\n", mt->name);

		// don't bother with sky
		if (!strnicmp (tx->name, "sky", 3)) continue;

		MakeTextureLightColour (tx);
	}

	Mod_SequenceAnimations (&loadmodel, m);
}


void Mod_LoadPlanes (void)
{
	int i;
	int j;
	int bits;

	loadmodel.planes = QHeap_Alloc (numplanes * sizeof (mplane_t));

	for (i = 0; i < numplanes; i++)
	{
		bits = 0;

		for (j = 0; j < 3; j++)
		{
			loadmodel.planes[i].normal[j] = dplanes[i].normal[j];

			if (loadmodel.planes[i].normal[j] < 0) bits |= 1 << j;
		}

		loadmodel.planes[i].dist = dplanes[i].dist;
		loadmodel.planes[i].type = dplanes[i].type;
		loadmodel.planes[i].signbits = bits;
	}
}


void Mod_LoadTexinfo (void)
{
	int i;
	int j;

	loadmodel.texinfo = QHeap_Alloc (numtexinfo * sizeof (mtexinfo_t));

	for (i = 0; i < numtexinfo; i++)
	{
		for (j = 0; j < 8; j++)
			loadmodel.texinfo[i].vecs[0][j] = texinfo[i].vecs[0][j];

		loadmodel.texinfo[i].flags = texinfo[i].flags;

		loadmodel.texinfo[i].texture = &loadmodel.textures[texinfo[i].miptex];
	}
}


void Mod_LoadFaces (void)
{
	int i;

	loadmodel.surfaces = QHeap_Alloc (numfaces * sizeof (msurface_t));

	for (i = 0; i < numfaces; i++)
	{
		mplane_t *plane;

		// we only need to address the texinfo here
		loadmodel.surfaces[i].texinfo = loadmodel.texinfo + dfaces[i].texinfo;

		// keep a pointer to the original face info
		loadmodel.surfaces[i].face = &dfaces[i];
		loadmodel.surfaces[i].surfnum = i;
		loadmodel.surfaces[i].firstedge = dfaces[i].firstedge;
		loadmodel.surfaces[i].numedges = dfaces[i].numedges;

		plane = loadmodel.planes + dfaces[i].planenum;
		loadmodel.surfaces[i].normal[0] = plane->normal[0];
		loadmodel.surfaces[i].normal[1] = plane->normal[1];
		loadmodel.surfaces[i].normal[2] = plane->normal[2];

		loadmodel.surfaces[i].lightvis = NULL;

		if (dfaces[i].side)
		{
			// backplane (normal stored for the plane always faces the same (positive) direction)
			loadmodel.surfaces[i].normal[0] *= -1;
			loadmodel.surfaces[i].normal[1] *= -1;
			loadmodel.surfaces[i].normal[2] *= -1;
		}
	}
}


void Mod_LoadMarksurfaces (void)
{
	int i;

	loadmodel.marksurfaces = QHeap_Alloc (nummarksurfaces * sizeof (msurface_t *));

	for (i = 0; i < nummarksurfaces; i++)
	{
		loadmodel.marksurfaces[i] = loadmodel.surfaces + dmarksurfaces[i];
	}
}


void Mod_LoadLeafs (void)
{
	int i;
	int j;

	loadmodel.leafs = QHeap_Alloc ((numleafs + 1) * sizeof (mleaf_t) + numnodes * sizeof (mnode_t));

	for (i = 0; i < numleafs; i++)
	{
		loadmodel.leafs[i].firstmarksurface = loadmodel.marksurfaces + dleafs[i].firstmarksurface;
		loadmodel.leafs[i].nummarksurfaces = dleafs[i].nummarksurfaces;
		loadmodel.leafs[i].contents = dleafs[i].contents;

		if (loadmodel.leafs[i].contents != CONTENTS_EMPTY)
			for (j = 0; j < loadmodel.leafs[i].nummarksurfaces; j++)
				loadmodel.leafs[i].firstmarksurface[j]->flags |= SURF_UNDERWATER;

		if (visdatasize)
			loadmodel.leafs[i].compressed_vis = dvisdata + dleafs[i].visofs;
		else loadmodel.leafs[i].compressed_vis = NULL;
	}
}


void Mod_SetParent (mnode_t *node, mnode_t *parent)
{
	node->parent = parent;

	if (node->contents < 0)
		return;

	Mod_SetParent (node->children[0], node);
	Mod_SetParent (node->children[1], node);
}

void Mod_LoadNodes (void)
{
	int i;
	int j;
	int p;

	// leafs and nodes are in contiguous memory
	loadmodel.nodes = (mnode_t *) &loadmodel.leafs[numleafs];

	for (i = 0; i < numnodes; i++)
	{
		loadmodel.nodes[i].plane = loadmodel.planes + dnodes[i].planenum;
		loadmodel.nodes[i].firstsurface = dnodes[i].firstface;
		loadmodel.nodes[i].numsurfaces = dnodes[i].numfaces;

		for (j = 0; j < 2; j++)
		{
			p = dnodes[i].children[j];

			if (p > 0)
				loadmodel.nodes[i].children[j] = loadmodel.nodes + p;
			else loadmodel.nodes[i].children[j] = (mnode_t *) (loadmodel.leafs + (-1 - p));
		}
	}

	Mod_SetParent (loadmodel.nodes, NULL);	// sets nodes and leafs
}


void Mod_LoadBSP (void)
{
	// we need these items in memory to be able to do stuff...
	Mod_LoadTextures ();
	Mod_LoadPlanes ();
	Mod_LoadTexinfo ();
	Mod_LoadFaces ();
	Mod_LoadMarksurfaces ();
	Mod_LoadLeafs ();
	Mod_LoadNodes ();

	loadmodel.numcolourpoints = 0;
}


mleaf_t *Mod_PointInLeaf (vec3_t p)
{
	mnode_t		*node;
	float		d;
	mplane_t	*plane;

	node = loadmodel.nodes;

	while (1)
	{
		if (node->contents < 0)
			return (mleaf_t *)node;
		plane = node->plane;
		d = DotProduct (p,plane->normal) - plane->dist;
		if (d > 0)
			node = node->children[0];
		else
			node = node->children[1];
	}
	
	return NULL;	// never reached
}


int LeafContentsForPoint (vec3_t p)
{
	// for stuff that doesn't know what an mleaf_t is
	return (Mod_PointInLeaf (p))->contents;
}


int r_visframecount;


float PointDist (vec3_t p1, vec3_t p2);


void RemoveLastPathElement (char *path)
{
	int i;

	for (i = strlen (path) - 1; i > 0; i--)
	{
		if (path[i] == '/' || path[i] == '\\')
		{
			path[i] = 0;
			break;
		}
	}
}


qboolean LocatePalette (void)
{
	// look for it
	FILE *f = OpenQuakeFile ("gfx/palette.lmp");

	if (f)
	{
		// found it
		int i;

		hasbasepal = true;

		for (i = 0; i < 256; i++)
		{
			qpal[i] = (int *) QHeap_Alloc (3 * sizeof (int));

			// these are bytes on disk but we store them as ints
			fread (&qpal[i][0], 1, 1, f);
			fread (&qpal[i][1], 1, 1, f);
			fread (&qpal[i][2], 1, 1, f);

			if (qpal[i][0] != basepal[i][0]) hasbasepal = false;
			if (qpal[i][1] != basepal[i][1]) hasbasepal = false;
			if (qpal[i][2] != basepal[i][2]) hasbasepal = false;
		}

		// done
		fclose (f);
		return true;
	}

	// not found
	return false;
}


void LoadPalette (char *bspname)
{
	int i;

	// true if the basepal is unchanged
	hasbasepal = false;

	// try it
	if (LocatePalette ()) return;

	// not found - set to the base palette
	for (i = 0; i < 256; i++) qpal[i] = basepal[i];
}


void ProcessColour (void)
{
}


void BoundPoly (int numverts, float *verts, vec3_t mins, vec3_t maxs)
{
	int		i, j;
	float	*v;

	mins[0] = mins[1] = mins[2] = 99999999;
	maxs[0] = maxs[1] = maxs[2] = -99999999;

	v = verts;

	for (i = 0; i < numverts; i++)
	{
		for (j = 0; j < 3; j++, v++)
		{
			if (*v < mins[j]) mins[j] = *v;
			if (*v > maxs[j]) maxs[j] = *v;
		}
	}
}

void SubdividePolygon (msurface_t *surf, int numverts, float *verts, int subdivide_size)
{
	int		i, j, k;
	vec3_t	mins, maxs;
	float	m;
	float	*v;
	vec3_t	front[64], back[64];
	int		f, b;
	float	dist[64];
	float	frac;
	glpoly_t	*poly;
	float	s, t;

	if (numverts > 60) Error ("numverts = %i", numverts);

	BoundPoly (numverts, verts, mins, maxs);

	for (i = 0; i < 3; i++)
	{
		m = (mins[i] + maxs[i]) * 0.5;
		m = subdivide_size * floor (m / subdivide_size + 0.5);

		if (maxs[i] - m < 8) continue;
		if (m - mins[i] < 8) continue;

		// cut it
		v = verts + i;
		for (j = 0; j < numverts; j++, v += 3)
			dist[j] = *v - m;

		// wrap cases
		dist[j] = dist[0];
		v-=i;
		VectorCopy (verts, v);

		f = b = 0;
		v = verts;

		for (j = 0; j < numverts; j++, v += 3)
		{
			if (dist[j] >= 0)
			{
				VectorCopy (v, front[f]);
				f++;
			}

			if (dist[j] <= 0)
			{
				VectorCopy (v, back[b]);
				b++;
			}

			if (dist[j] == 0 || dist[j + 1] == 0) continue;

			if ((dist[j] > 0) != (dist[j + 1] > 0))
			{
				// clip point
				frac = dist[j] / (dist[j] - dist[j + 1]);

				for (k = 0; k < 3; k++)
					front[f][k] = back[b][k] = v[k] + frac * (v[3 + k] - v[k]);

				f++;
				b++;
			}
		}

		SubdividePolygon (surf, f, front[0], subdivide_size);
		SubdividePolygon (surf, b, back[0], subdivide_size);
		return;
	}

	if (!numverts) return;

	poly = QHeap_Alloc (sizeof (glpoly_t) + (numverts - 4) * VERTEXSIZE * sizeof (float));
	poly->next = surf->polys;
	surf->polys = poly;
	poly->numverts = numverts;
	poly->midpoint[0] = poly->midpoint[1] = poly->midpoint[2] = 0;

	for (i = 0; i < numverts; i++, verts += 3)
	{
		VectorCopy (verts, poly->verts[i]);

		poly->midpoint[0] += verts[0];
		poly->midpoint[1] += verts[1];
		poly->midpoint[2] += verts[2];

		s = DotProduct (verts, surf->texinfo->vecs[0]);
		t = DotProduct (verts, surf->texinfo->vecs[1]);

		poly->verts[i][3] = s;
		poly->verts[i][4] = t;
	}

	poly->midpoint[0] /= numverts;
	poly->midpoint[1] /= numverts;
	poly->midpoint[2] /= numverts;

	// position up to 32 units away from the surf; ensure it's in empty space
	for (i = 2; i < 32; i++)
	{
		float mp[3];
		mleaf_t *l;

		// move out in the direction of the surface normal
		mp[0] = poly->midpoint[0] + surf->normal[0] * i;
		mp[1] = poly->midpoint[1] + surf->normal[1] * i;
		mp[2] = poly->midpoint[2] + surf->normal[2] * i;

		l = Mod_PointInLeaf (mp);

		if (l->contents == CONTENTS_SOLID || l->contents == CONTENTS_SKY) break;
	}

	poly->midpoint[0] += surf->normal[0] * (i / 2);
	poly->midpoint[1] += surf->normal[1] * (i / 2);
	poly->midpoint[2] += surf->normal[2] * (i / 2);

	surf->numsubdivisions++;
}


void GL_SubdivideSurface (msurface_t *surf, int subdivide_size)
{
	vec3_t		verts[64];
	int			numverts;
	int			i;
	int			lindex;
	float		*vec;

	if (!surf->numedges) return;

	// convert edges back to a normal polygon
	numverts = 0;
	// surf->numsubdivisions++;

	for (i = 0; i < surf->numedges; i++)
	{
		lindex = dsurfedges[surf->firstedge + i];

		if (lindex > 0)
			vec = dvertexes[dedges[lindex].v[0]].point;
		else
			vec = dvertexes[dedges[-lindex].v[1]].point;

		VectorCopy (vec, verts[numverts]);
		numverts++;
	}

	SubdividePolygon (surf, numverts, verts[0], subdivide_size);

	surf->numsubdivisions = sqrt (surf->numsubdivisions);

	if (surf->numsubdivisions < 1) surf->numsubdivisions = 1;

	surf->numsubdivisions = 1;
}


void AddSurfaceLight (msurface_t *surf)
{
}


texture_t *R_TextureAnimation (texture_t *base)
{
	texture_t *tex = base;
	texture_t *best = base;
	int count;

	// no animations
	if (!base->anim_total) return base;

	for (count = 0; count <= base->anim_total; count++)
	{
		if (tex->intensity > best->intensity) best = tex;
		tex = tex->anim_next;
	}

	// return what we got
	return best;
}


void GetLightColoursForPolys (msurface_t *surf, texture_t *tx)
{
	glpoly_t *p;

	for (p = surf->polys; p; p = p->next)
	{
		// create
		colourpoint_t *cp = (colourpoint_t *) QHeap_Alloc (sizeof (colourpoint_t));

		// set props
		cp->angle = 0;
		cp->colour[0] = (float) tx->colour[0] / (float) surf->numsubdivisions;
		cp->colour[1] = (float) tx->colour[1] / (float) surf->numsubdivisions;
		cp->colour[2] = (float) tx->colour[2] / (float) surf->numsubdivisions;
		cp->light = tx->intensity;
		cp->origin[0] = p->midpoint[0];
		cp->origin[1] = p->midpoint[1];
		cp->origin[2] = p->midpoint[2];
		cp->targetent = NULL;

		// link in
		cp->next = colourpoints;
		colourpoints = cp;
	}
}


void CreateColourPointsForFaces (void)
{
	int i;
	msurface_t *surf;
	float slimecolour[3] = {0};
	float lavacolour[3] = {0};
	colourpoint_t *cp;
	int numcp;
	texture_t *tx;

	// load the bsp into memory in (...roughly...) the same format as the game
	Mod_LoadBSP ();

	UpdateProgressNotify ("Lighting %i Surfaces\r\n\r\n", numfaces);
	UpdateProgressNotify ("Building Surface Light Colours...\r\n");

	for (i = 0, surf = loadmodel.surfaces; i < numfaces; i++, surf++)
	{
		int testcolour[3];

		// get animation here as the base animation may not have colour
		tx = R_TextureAnimation (surf->texinfo->texture);

		// skip these
		if (!strnicmp (tx->name, "sky", 3)) continue;
		if (tx->intensity < 1) continue;

		// divide by 4 to exclude all colours reasonably close to white
		testcolour[0] = tx->colour[0] / 4;
		testcolour[1] = tx->colour[1] / 4;
		testcolour[2] = tx->colour[2] / 4;

		// colour is white
		if (testcolour[0] == testcolour[1] && testcolour[1] == testcolour[2]) continue;

		if (!strnicmp (tx->name, "*slime", 6))
		{
			// store slime colour for later reference
			slimecolour[0] = tx->colour[0];
			slimecolour[1] = tx->colour[1];
			slimecolour[2] = tx->colour[2];

			// quake stores 2 copies of liquid surfs - one underwater and one above water
			if (surf->flags & SURF_UNDERWATER) continue;

			GL_SubdivideSurface (surf, 128);
		}
		else if (!strnicmp (tx->name, "*lava", 5))
		{
			// store lava colour for later reference
			lavacolour[0] = tx->colour[0];
			lavacolour[1] = tx->colour[1];
			lavacolour[2] = tx->colour[2];

			// quake stores 2 copies of liquid surfs - one underwater and one above water
			if (surf->flags & SURF_UNDERWATER) continue;

			GL_SubdivideSurface (surf, 128);
		}
		else if (tx->name[0] == '*')
			continue;
		else GL_SubdivideSurface (surf, 64);

		// surfaces are subdivided so that we can have a large surface with a single
		// light texture tiled on it contributing across the full extent of the surface
		GetLightColoursForPolys (surf, tx);
	}

	// test our standard lights and see if any are in lava or slime
	for (i = 0; i < num_entities; i++)
	{
		mleaf_t *leaf = Mod_PointInLeaf (entities[i].origin);

		if (entities[i].light < 1) continue;
		if (entities[i].mdlname[0]) continue;

		if (leaf->contents == CONTENTS_LAVA) MakeColourPoint (&entities[i], lavacolour);
		if (leaf->contents == CONTENTS_SLIME) MakeColourPoint (&entities[i], slimecolour);
	}
}


void R_LeafVisibility (byte *vis)
{
	int i;
	mnode_t *node;

	for (i = 0; i < numleafs; i++)
	{
		if (!vis || (vis[i >> 3] & (1 << (i & 7))))
		{
			node = (mnode_t *) &loadmodel.leafs[i + 1];

			do
			{
				// already added
				if (node->visframe == r_visframecount) break;

				// add it
				node->visframe = r_visframecount;
				node = node->parent;
			} while (node);
		}
	}
}


byte *decompressed;

byte *Mod_DecompressVis (byte *in)
{
	int		c;
	byte	*out;
	int		row;

	row = (numleafs + 7) >> 3;	
	out = decompressed;

	if (!in)
	{
		// no vis info, so make all visible
		while (row)
		{
			*out++ = 0xff;
			row--;
		}

		return decompressed;		
	}

	do
	{
		if (*in)
		{
			*out++ = *in++;
			continue;
		}

		c = in[1];
		in += 2;

		while (c)
		{
			*out++ = 0;
			c--;
		}
	} while (out - decompressed < row);

	return decompressed;
}


void R_RecursiveWorldNode (mnode_t *node, vec3_t modelorg)
{
	int			side;
	mplane_t	*plane;
	msurface_t	*surf;
	double		dot;

	if (node->contents == CONTENTS_SOLID) return;
	if (node->visframe != r_visframecount) return;

	if (node->contents < 0)
	{
		mleaf_t *leaf = (mleaf_t *) node;
		msurface_t **mark = leaf->firstmarksurface;
		int c = leaf->nummarksurfaces;

		if (c)
		{
			do
			{
				(*mark)->visframe = r_visframecount;
				mark++;
			} while (--c);
		}

		return;
	}

	// find which side of the node we are on
	switch (node->plane->type)
	{
	case PLANE_X:
		dot = modelorg[0] - node->plane->dist;
		break;
	case PLANE_Y:
		dot = modelorg[1] - node->plane->dist;
		break;
	case PLANE_Z:
		dot = modelorg[2] - node->plane->dist;
		break;
	default:
		dot = DotProduct (modelorg, node->plane->normal) - node->plane->dist;
		break;
	}

	if (dot >= 0)
		side = 0;
	else side = 1;

	// recurse down the children, front side first
	R_RecursiveWorldNode (node->children[side], modelorg);

	if (node->numsurfaces)
	{
		int c = node->numsurfaces;
		msurface_t *surf = loadmodel.surfaces + node->firstsurface;

		for (; c; c--, surf++)
		{
			// the lightnumber is the r_visframecount + 1 value to retrieve it from there
			int light = r_visframecount - 1;

			if (surf->visframe != r_visframecount) continue;

			// alloc these as required
			if (!surf->lightvis) surf->lightvis = (byte *) QHeap_Alloc ((loadmodel.numcolourpoints + 7) / 8);

			// surf is now visible to the light
			surf->lightvis[light >> 3] |= 1 << (light & 7);
		}
	}

	// recurse down the back side
	R_RecursiveWorldNode (node->children[!side], modelorg);
}


qboolean R_LightVisibility (int surfnum, int lightnum)
{
	msurface_t *surf = &loadmodel.surfaces[surfnum];

	// surf didn't get any visibility info from the world so it must be in a brush model
	// to save on complicating things too much we just always take brushmodel surfs as visible to all lights
	if (!surf->lightvis) return true;

	if (!(surf->lightvis[lightnum >> 3] & (1 << (lightnum & 7))))
		return false;

	return true;
}


float PointDist (vec3_t p1, vec3_t p2);

void FixupColourPoints (void)
{
	int i;
	msurface_t *surf;
	int numcp;
	colourpoint_t *cp;

	// null all of the lightvis points
	for (i = 0; i < numfaces; i++)
		loadmodel.surfaces[i].lightvis = NULL;

	// fix up the colourpoints
	for (cp = colourpoints, numcp = 0; cp; cp = cp->next, numcp++)
	{
		if (cp->colour[0] == cp->colour[1] && cp->colour[1] == cp->colour[2])
		{
			// same intensity so set to white
			cp->hascolour = false;
		}
		else cp->hascolour = true;

		cp->cpnum = numcp;
	}

	loadmodel.numcolourpoints = numcp;

	UpdateProgressNotify ("%i Colour Points Total\r\n\r\n", numcp);
	UpdateProgressNotify ("Building visibility...\r\n");

	// set up light vis points for each surf
	// this goes ape if we just allocate (numleafs + 7) >> 3 so
	// we allocate the full whack here... it's never more than a handful of K anyway
	decompressed = (byte *) QHeap_Alloc (numleafs);

#define MAX_DIST 48
	for (cp = colourpoints; cp; cp = cp->next)
	{
		colourpoint_t *cp2;
		mleaf_t *leaf = Mod_PointInLeaf (cp->origin);
		float factor = 0;
		int numlights = 0;

		// skip these
		if (leaf->contents == CONTENTS_SOLID)
		{
			cp->light = -1;
			continue;
		}

		if (leaf->contents == CONTENTS_SKY)
		{
			cp->light = -1;
			continue;
		}

		/*
		// find all lights close to this one
		for (cp2 = colourpoints; cp2; cp2 = cp2->next)
		{
			float dist;

			// same light
			if (cp->cpnum == cp2->cpnum) continue;

			// distance
			dist = PointDist (cp->origin, cp2->origin);

			// too far
			if (dist > MAX_DIST) continue;

			// blocked
			if (!TestLine (cp->origin, cp2->origin)) continue;

			// accumulate
			numlights++;
			factor += ((float) MAX_DIST - dist) / (float) MAX_DIST;
		}

		if (numlights > 0)
		{
			// scale down to prevent oversaturation
			factor /= (float) (numlights * numlights);
			cp->colour[0] *= factor;
			cp->colour[1] *= factor;
			cp->colour[2] *= factor;
		}
		*/

		// add 1 cos visframes in leafs/nodes/surfs are inited to 0
		r_visframecount = (cp->cpnum + 1);

		if (!visdatasize)
			R_LeafVisibility (NULL);
		else if (leaf == loadmodel.leafs)
			R_LeafVisibility (NULL);
		else if (!leaf->compressed_vis)
			R_LeafVisibility (NULL);
		else R_LeafVisibility (Mod_DecompressVis (leaf->compressed_vis));

		R_RecursiveWorldNode (loadmodel.nodes, cp->origin);
	}
}
