
#include "light.h"

// cache queries
int GetCachedLightOffset (int surfnum);

// lightinfo stuff
void SetupSurfLightInfo (int surfnum, int offset, int size);
byte *SurfGetLight1 (int surfnum);
byte *SurfGetLight3 (int surfnum);
int GetLightOffset (int size);
qboolean CheckFaceVis (float *visorg, int surfnum);

extern float colourrange;

float PointDist (vec3_t p1, vec3_t p2)
{
	int		i;
	vec_t	t;

	for (i = 0, t = 0; i < 3; i++)
		t += (p2[i] - p1[i]) * (p2[i] - p1[i]);

	// don't blow up...
	if (t < 1) t = 1;

	return sqrt(t);
}


/*
============
CastRay

Returns the distance between the points, or -1 if blocked
=============
*/
vec_t CastRay (vec3_t p1, vec3_t p2)
{
	qboolean	trace;

	trace = TestLine (p1, p2);

	if (!trace)
		return -1;		// ray was blocked

	return PointDist (p1, p2);
}

/*
===============================================================================

SAMPLE POINT DETERMINATION

void SetupBlock (dface_t *f) Returns with surfpt[] set

This is a little tricky because the lightmap covers more area than the face.
If done in the straightforward fashion, some of the
sample points will be inside walls or on the other side of walls, causing
false shadows and light bleeds.

To solve this, I only consider a sample point valid if a line can be drawn
between it and the exact midpoint of the face.  If invalid, it is adjusted
towards the center until it is valid.

(this doesn't completely work)

===============================================================================
*/

typedef struct lightinfo_s
{
	int		numlightstyles;
	vec_t	*light;
	vec_t	facedist;
	vec3_t	facenormal;

	int		numsurfpt;
	vec3_t	surfpt[SINGLEMAPSIZE];

	vec3_t	texorg;
	vec3_t	worldtotex[2];	// s = (world - texorg) . worldtotex[0]
	vec3_t	textoworld[2];	// world = texorg + s * textoworld[0]

	vec_t	exactmins[2], exactmaxs[2];
	
	int		texmins[2], texsize[2];
	int		lightstyles[256];
	int		surfnum;
	dface_t	*face;
} lightinfo_t;


/*
================
CalcFaceVectors

Fills in texorg, worldtotex. and textoworld
================
*/
void CalcFaceVectors (lightinfo_t *l)
{
	texinfo_t	*tex;
	int			i, j;
	vec3_t	texnormal;
	float	distscale;
	vec_t	dist, len;

	tex = &texinfo[l->face->texinfo];
	
// convert from float to vec_t
	for (i=0 ; i<2 ; i++)
		for (j=0 ; j<3 ; j++)
			l->worldtotex[i][j] = tex->vecs[i][j];

// calculate a normal to the texture axis.  points can be moved along this
// without changing their S/T
	texnormal[0] = tex->vecs[1][1]*tex->vecs[0][2]
		- tex->vecs[1][2]*tex->vecs[0][1];
	texnormal[1] = tex->vecs[1][2]*tex->vecs[0][0]
		- tex->vecs[1][0]*tex->vecs[0][2];
	texnormal[2] = tex->vecs[1][0]*tex->vecs[0][1]
		- tex->vecs[1][1]*tex->vecs[0][0];
	VectorNormalize (texnormal);

// flip it towards plane normal
	distscale = DotProduct (texnormal, l->facenormal);
	if (!distscale)
		Error ("Texture axis perpendicular to face");
	if (distscale < 0)
	{
		distscale = -distscale;
		VectorSubtract (vec3_origin, texnormal, texnormal);
	}	

// distscale is the ratio of the distance along the texture normal to
// the distance along the plane normal
	distscale = 1/distscale;

	for (i=0 ; i<2 ; i++)
	{
		len = VectorLength (l->worldtotex[i]);
		dist = DotProduct (l->worldtotex[i], l->facenormal);
		dist *= distscale;
		VectorMA (l->worldtotex[i], -dist, texnormal, l->textoworld[i]);
		VectorScale (l->textoworld[i], (1/len)*(1/len), l->textoworld[i]);
	}


// calculate texorg on the texture plane
	for (i=0 ; i<3 ; i++)
		l->texorg[i] = -tex->vecs[0][3]* l->textoworld[0][i] - tex->vecs[1][3] * l->textoworld[1][i];

// project back to the face plane
	dist = DotProduct (l->texorg, l->facenormal) - l->facedist - 1;
	dist *= distscale;
	VectorMA (l->texorg, -dist, texnormal, l->texorg);
	
}

/*
================
CalcFaceExtents

Fills in s->texmins[] and s->texsize[]
also sets exactmins[] and exactmaxs[]
================
*/
void CalcFaceExtents (lightinfo_t *l)
{
	dface_t *s;
	vec_t	mins[2], maxs[2], val;
	int		i,j, e;
	dvertex_t	*v;
	texinfo_t	*tex;
	
	s = l->face;

	mins[0] = mins[1] = 999999;
	maxs[0] = maxs[1] = -99999;

	tex = &texinfo[s->texinfo];
	
	for (i=0 ; i<s->numedges ; i++)
	{
		e = dsurfedges[s->firstedge+i];
		if (e >= 0)
			v = dvertexes + dedges[e].v[0];
		else
			v = dvertexes + dedges[-e].v[1];
		
		for (j=0 ; j<2 ; j++)
		{
			val = v->point[0] * tex->vecs[j][0] + 
				v->point[1] * tex->vecs[j][1] +
				v->point[2] * tex->vecs[j][2] +
				tex->vecs[j][3];
			if (val < mins[j])
				mins[j] = val;
			if (val > maxs[j])
				maxs[j] = val;
		}
	}

	for (i=0 ; i<2 ; i++)
	{	
		l->exactmins[i] = mins[i];
		l->exactmaxs[i] = maxs[i];
		
		mins[i] = floor(mins[i]/16);
		maxs[i] = ceil(maxs[i]/16);

		l->texmins[i] = mins[i];
		l->texsize[i] = maxs[i] - mins[i];
		if (l->texsize[i] > 17)
			Error ("Bad surface extents");
	}
}

/*
=================
CalcPoints

For each texture aligned grid point, back project onto the plane
to get the world xyz value of the sample point
=================
*/
int c_bad;

void CalcPoints (lightinfo_t *l)
{
	int		i;
	int		s, t, j;
	int		w, h, step;
	vec_t	starts, startt, us, ut;
	vec_t	*surf;
	vec_t	mids, midt;
	vec3_t	facemid, move;

//
// fill in surforg
// the points are biased towards the center of the surface
// to help avoid edge cases just inside walls
//
	surf = l->surfpt[0];
	mids = (l->exactmaxs[0] + l->exactmins[0])/2;
	midt = (l->exactmaxs[1] + l->exactmins[1])/2;

	for (j=0 ; j<3 ; j++)
		facemid[j] = l->texorg[j] + l->textoworld[0][j]*mids + l->textoworld[1][j]*midt;

	h = (l->texsize[1] + 1);
	w = (l->texsize[0] + 1);

	starts = l->texmins[0] * 16;
	startt = l->texmins[1] * 16;

	step = 16;

	l->numsurfpt = w * h;

	for (t=0 ; t<h ; t++)
	{
		for (s=0 ; s<w ; s++, surf+=3)
		{
			us = starts + s*step;
			ut = startt + t*step;

		// if a line can be traced from surf to facemid, the point is good
			for (i=0 ; i<6 ; i++)
			{
			// calculate texture point
				for (j=0 ; j<3 ; j++)
					surf[j] = l->texorg[j] + l->textoworld[0][j]*us + l->textoworld[1][j]*ut;

				if (CastRay (facemid, surf) != -1)
					break;	// got it
				if (i & 1)
				{
					if (us > mids)
					{
						us -= 8;
						if (us < mids)
							us = mids;
					}
					else
					{
						us += 8;
						if (us > mids)
							us = mids;
					}
				}
				else
				{
					if (ut > midt)
					{
						ut -= 8;
						if (ut < midt)
							ut = midt;
					}
					else
					{
						ut += 8;
						if (ut > midt)
							ut = midt;
					}
				}
				
				// move surf 8 pixels towards the center
				VectorSubtract (facemid, surf, move);
				VectorNormalize (move);
				VectorMA (surf, 8, move, surf);
			}
			if (i == 2)
				c_bad++;
		}
	}
	
}


/*
===============================================================================

FACE LIGHTING

===============================================================================
*/

int		c_culldistplane, c_proper;

/*
================
SingleLightFace
================
*/
void ColourLight (entity_t *light, float *surfpt, dface_t *face, float add);

void SingleLightFace (entity_t *light, lightinfo_t *l, qboolean test)
{
}


/*
============
FixMinlight
============
*/
void FixMinlight (lightinfo_t *l)
{
}


/*
============
LightFace
============
*/
void LightFace (int surfnum, qboolean test)
{
}


qboolean SingleLightColourFace (colourpoint_t *light, lightinfo_t *l, vec3_t *lightsamp, int *numhits)
{
	vec_t	dist;
	vec3_t	incoming;
	vec_t	angle;
	vec_t	add;
	vec_t	*surf;
	int		size;
	int		c, i;
	vec3_t	rel;
	vec3_t	spotvec;
	vec_t	falloff;
	qboolean hit;
//	float black;

	VectorSubtract (light->origin, bsp_origin, rel);
	dist = scaledist * (DotProduct (rel, l->facenormal) - l->facedist);

	// don't bother with lights behind the surface
	if (dist <= 0) return false;

	// don't bother with light too far away
	if (dist > light->light) return false;

	if (light->targetent)
	{
		VectorSubtract (light->targetent->origin, light->origin, spotvec);
		VectorNormalize (spotvec);

		if (!light->angle)
			falloff = -cos (20 * Q_PI / 180);
		else falloff = -cos (light->angle / 2 * Q_PI / 180);
	}
	else
	{
		// shut up compiler warnings
		falloff = 0;
	}

	surf = l->surfpt[0];

	// not hit until we say it is
	hit = false;

	for (c = 0; c < l->numsurfpt; c++, surf += 3)
	{
		dist = CastRay (light->origin, surf) * scaledist;

		// light doesn't reach
		if (dist < 0) continue;

		VectorSubtract (light->origin, surf, incoming);
		VectorNormalize (incoming);
		angle = DotProduct (incoming, l->facenormal);

		if (light->targetent)
		{
			// spotlight cutoff
			if (DotProduct (spotvec, incoming) > falloff)
				continue;
		}

		angle = (1.0 - scalecos) + scalecos * angle;
		add = light->light - dist;
		add *= angle;
		add = add / ((float) light->light * angle);
		//add = add / (float) light->light;

		if (add < 0.001) continue;
		if (add > 1.0f) add = 1.0f;

		for (i = 0; i < 3; i++)
		{
			float frontlerp = (add * light->colour[i]);
			float backlerp = 255.0f * (1.0f - add);

			lightsamp[c][i] *= (frontlerp + backlerp) / 255.0f;//(add * light->colour[i]);
		}

		WhiteBias (lightsamp[c], 32);

		numhits[c]++;

		// hit now
		hit = true;
	}

	return hit;
}


byte DoLightmap (float lm, float *c, int c1, int c2)
{
	lm *= ((255 - c[c1]) + (255 - c[c2])) / 511.0f;

	if (lm > 255)
		return 255;
	else if (lm < 0)
		return 0;
	else return (byte) lm;
}


void RescaleColour (float *newc, byte original)
{
	float newtotal = newc[0] + newc[1] + newc[2];
	float originaltotal = (float) original * 3;

	// no change
	if ((int) newtotal == (int) originaltotal) return;

	// don't allow this
	if (newtotal < 1 || originaltotal < 1) return;

	newc[0] /= newtotal; newc[0] *= originaltotal;
	newc[1] /= newtotal; newc[1] *= originaltotal;
	newc[2] /= newtotal; newc[2] *= originaltotal;
}


void Rescale666Colour (float *newc, byte original)
{
	// halve it as a luminance of 255 == white always!!!
	RGB2HSL (newc);
	newc[2] = original / 2;
	HSL2RGB (newc);

	// now double the colours to adjust for the halving above
	newc[0] *= 2.0f;
	newc[1] *= 2.0f;
	newc[2] *= 2.0f;
}

void BalanceColour (float *c, float target);

qboolean R_LightVisibility (int surfnum, int lightnum);


float *GetBlurPoint (vec3_t *src, int x, int y, int w, int h)
{
	if (x < 0) x = 0; else if (x >= w) x = w - 1;
	if (y < 0) y = 0; else if (y >= h) y = h - 1;

	return &src[y * w + x][0];
}


void BlurColourMap (vec3_t *src, int w, int h)
{
	int x;
	int y;
	static vec3_t dst[SINGLEMAPSIZE];

	// clamp to lightmap size
	if (w > 18) w = 18;
	if (h > 18) h = 18;

	// blur across rows from src into dst
	for (y = 0; y < h; y++)
	{
		for (x = 0; x < w; x++)
		{
			int i;
			float *p1 = GetBlurPoint (src, x - 1, y, w, h);
			float *p2 = GetBlurPoint (src, x, y, w, h);
			float *p3 = GetBlurPoint (src, x + 1, y, w, h);
			float *pd = GetBlurPoint (dst, x, y, w, h);

			for (i = 0; i < 3; i++) pd[i] = (p1[i] + (p2[i] * 2) + p3[i]) / 4;
		}
	}

	// blur down cols from dst back into src
	for (y = 0; y < h; y++)
	{
		for (x = 0; x < w; x++)
		{
			int i;
			float *p1 = GetBlurPoint (dst, x, y - 1, w, h);
			float *p2 = GetBlurPoint (dst, x, y, w, h);
			float *p3 = GetBlurPoint (dst, x, y + 1, w, h);
			float *pd = GetBlurPoint (src, x, y, w, h);

			for (i = 0; i < 3; i++) pd[i] = (p1[i] + (p2[i] * 2) + p3[i]) / 4;
		}
	}
}


void LightColourFace (int surfnum)
{
	int i, maps;
	dface_t *f = dfaces + surfnum;
	lightinfo_t	l;
	qboolean hit;
	byte *lightmap;
	colourpoint_t *cp;
	static vec3_t colourmap[SINGLEMAPSIZE];
	static int numhits[SINGLEMAPSIZE];

	// face has no lightmaps
	if (f->lightofs == -1) return;

	// non-lit texture
	if (texinfo[f->texinfo].flags & TEX_SPECIAL) return;

	// init the lightmap
	memset (&l, 0, sizeof (l));
	l.surfnum = surfnum;
	l.face = f;

	for (i = 0; i < SINGLEMAPSIZE; i++)
	{
		colourmap[i][0] = colourmap[i][1] = colourmap[i][2] = 255;
		numhits[i] = 0;
	}

	// rotate plane
	VectorCopy (dplanes[f->planenum].normal, l.facenormal);
	l.facedist = dplanes[f->planenum].dist;

	if (f->side)
	{
		// backfacing - switch normal and dist to negative
		VectorSubtract (vec3_origin, l.facenormal, l.facenormal);
		l.facedist = -l.facedist;
	}

	CalcFaceVectors (&l);
	CalcFaceExtents (&l);
	CalcPoints (&l);

	for (cp = colourpoints, hit = false; cp; cp = cp->next)
	{
		qboolean facehit;

		// not valid
		if (cp->light < 1) continue;

		// surf is not visible to the light
		if (!R_LightVisibility (surfnum, cp->cpnum)) continue;

		// process this point and determine hit
		facehit = SingleLightColourFace (cp, &l, colourmap, numhits);
		hit = (hit || facehit);
	}

	// no hit so just keep the original lightmap
	// (this is no longer valid owing to the "fade to white" processing above
	//if (!hit) return;

	// get the base lightmap data pointer (* 3 for 3 component lighting)
	// we copied the original to this (and expanded) to make it valid for modifications
	lightmap = dlightdata3 + (f->lightofs * 3);

	// simulate extra sampling
	BlurColourMap (colourmap, l.texsize[0] + 1, l.texsize[1] + 1);

	for (maps = 0; maps < MAXLIGHTMAPS; maps++)
	{
		int s, t, cm;

		// no more lightmaps
		if (f->styles[maps] == 255) break;

		// do this lightmap
		for (t = 0, cm = 0; t <= l.texsize[1]; t++)
		{
			for (s = 0; s <= l.texsize[0]; s++, cm++)
			{
				int c;
				float lm[3];

				for (c = 0; c < 3; c++)
					lm[c] = ((colourmap[cm][c] * (float) lightmap[c]) / 255.0f);

				// make the new colourmap the same intensity as the old lightmap
				//BalanceColour (lm, lightmap[0]);

				// BalanceColour isn't valid here as we want to do proper RGB/HSL scaling
				RescaleColour (lm, lightmap[0]);

				for (c = 0; c < 3; c++)
				{
					if (lm[c] > 255)
						lightmap[c] = 255;
					else if (lm[c] < 0)
						lightmap[c] = 0;
					else lightmap[c] = (byte) lm[c];
				}

				// next lightmap texel (3 component)
				lightmap += 3;
			}
		}
	}
}

