

#include <windows.h>
#include <commctrl.h>
#include <stdio.h>
#include <direct.h>
#include "light.h"
#include "heapmem.h"

char *va (char *format, ...);


typedef struct
{
	char id[4];
	int dirofs;
	int dirlen;
} dpackheader_t;


typedef struct
{
	char name[56];
	int filepos;
	int filelen;
} dpackfile_t;


typedef struct
{
	char name[MAX_PATH];
	int filepos;
	int filelen;
} packfile_t;

typedef struct pack_s
{
	char filename[MAX_PATH];
	int numfiles;
	packfile_t *files;
} pack_t;


int ValidatePack (char *pakname)
{
	dpackheader_t pheader;
	FILE *f = fopen (pakname, "rb");

	// this shouldn't happen as we've just selected it in a file open dialog
	if (!f) return 0;

	// check the header
	fread (&pheader, sizeof (dpackheader_t), 1, f);
	fclose (f);

	if (pheader.id[0] != 'P' || pheader.id[1] != 'A' || pheader.id[2] != 'C' || pheader.id[3] != 'K') 
		return 0;
	else return 1;
}


void TV_AddPAKItem (HTREEITEM hParent, char *pakname, char *filename);

void ExpandTreeFromPAK (HTREEITEM hParent, char *pakname)
{
	int i;
	dpackheader_t pheader;
	dpackfile_t *pakdir;
	int numpakfiles;
	FILE *f = fopen (pakname, "rb");

	// this shouldn't happen as we've just selected it in a file open dialog
	if (!f) return;

	// check the header
	fread (&pheader, sizeof (dpackheader_t), 1, f);

	if (pheader.id[0] != 'P' || pheader.id[1] != 'A' || pheader.id[2] != 'C' || pheader.id[3] != 'K')
	{
		// not a pack
		fclose (f);
		return;
	}

	// set up the pak directory (using standard malloc as we're going to free it right away)
	pakdir = (dpackfile_t *) malloc (pheader.dirlen);
	numpakfiles = pheader.dirlen / sizeof (dpackfile_t);

	// seek to the directory offset and read it all in
	fseek (f, pheader.dirofs, SEEK_SET);
	fread (pakdir, pheader.dirlen, 1, f);

	// now check all of the directory entries
	for (i = 0; i < numpakfiles; i++)
	{
		// get item extension
		char *ext = &pakdir[i].name[strlen (pakdir[i].name) - 4];

		// check for a BSP (not necessarily always in /maps)
		if (stricmp (ext, ".bsp")) continue;

		// add it
		TV_AddPAKItem (hParent, pakname, pakdir[i].name);
	}

	// not found
	Q_SafeFree (pakdir);
	fclose (f);
}


FILE *FindFileInSpecificPak (char *pakpath, char *pakname, char *filename)
{
	int i;
	dpackheader_t pheader;
	dpackfile_t *pakdir;
	int numpakfiles;
	char fullpakname[MAX_PATH];
	FILE *f;

	// construct the name and open it
	sprintf (fullpakname, "%s\\%s", pakpath, pakname);
	f = fopen (fullpakname, "rb");

	// we don't expect this to happen but let's be sure, eh?
	if (!f) return NULL;

	// check the header
	fread (&pheader, sizeof (dpackheader_t), 1, f);

	if (pheader.id[0] != 'P' || pheader.id[1] != 'A' || pheader.id[2] != 'C' || pheader.id[3] != 'K')
	{
		// not a pack
		fclose (f);
		return NULL;
	}

	// set up the pak directory (using standard malloc as we're going to free it right away)
	pakdir = (dpackfile_t *) malloc (pheader.dirlen);
	numpakfiles = pheader.dirlen / sizeof (dpackfile_t);

	// seek to the directory offset and read it all in
	fseek (f, pheader.dirofs, SEEK_SET);
	fread (pakdir, pheader.dirlen, 1, f);

	// now check all of the directory entries
	for (i = 0; i < numpakfiles; i++)
	{
		// non-case-sensitive search
		if (!stricmp (pakdir[i].name, filename))
		{
			// seek to the position of the file
			fseek (f, pakdir[i].filepos, SEEK_SET);

			// done
			Q_SafeFree (pakdir);
			return f;
		}
	}

	// not found
	Q_SafeFree (pakdir);
	fclose (f);

	// try for it in the file system
	sprintf (fullpakname, "%s\\%s", pakpath, filename);
	return fopen (fullpakname, "rb");
}


void LoadSpecificFileFromSpecificPAK (char *pakname, char *filename, void **buffer)
{
	int i;
	dpackheader_t pheader;
	dpackfile_t *pakdir;
	int numpakfiles;
	FILE *f;

	// open it
	f = fopen (pakname, "rb");

	// we don't expect this to happen but let's be sure, eh?
	if (!f) return;

	// check the header
	fread (&pheader, sizeof (dpackheader_t), 1, f);

	if (pheader.id[0] != 'P' || pheader.id[1] != 'A' || pheader.id[2] != 'C' || pheader.id[3] != 'K')
	{
		// not a pack
		fclose (f);
		return;
	}

	// set up the pak directory (using standard malloc as we're going to free it right away)
	pakdir = (dpackfile_t *) malloc (pheader.dirlen);
	numpakfiles = pheader.dirlen / sizeof (dpackfile_t);

	// seek to the directory offset and read it all in
	fseek (f, pheader.dirofs, SEEK_SET);
	fread (pakdir, pheader.dirlen, 1, f);

	// now check all of the directory entries
	for (i = 0; i < numpakfiles; i++)
	{
		// non-case-sensitive search
		if (!stricmp (pakdir[i].name, filename))
		{
			void *filebuf = QHeap_Alloc (pakdir[i].filelen + 1);
			((char *) filebuf)[pakdir[i].filelen] = 0;

			// seek to the position of the file
			fseek (f, pakdir[i].filepos, SEEK_SET);
			fread (filebuf, pakdir[i].filelen, 1, f);

			// done
			Q_SafeFree (pakdir);
			fclose (f);
			*buffer = filebuf;
			return;
		}
	}

	// not found
	Q_SafeFree (pakdir);
	fclose (f);
}


FILE *FindFileInPaks (char *basepath, char *filename)
{
	WIN32_FIND_DATA ffd;
	HANDLE ffh;
	char searchpath[MAX_PATH];
	FILE *f;

	sprintf (searchpath, "%s\\*.pak", basepath);

	if (!(ffh = FindFirstFile (searchpath, &ffd)))
		return NULL;

	do
	{
		// don't use these types
		if (ffd.dwFileAttributes & FILE_ATTRIBUTE_COMPRESSED) continue;
		if (ffd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) continue;
		if (ffd.dwFileAttributes & FILE_ATTRIBUTE_ENCRYPTED) continue;
		if (ffd.dwFileAttributes & FILE_ATTRIBUTE_HIDDEN) continue;
		if (ffd.dwFileAttributes & FILE_ATTRIBUTE_OFFLINE) continue;
		if (ffd.dwFileAttributes & FILE_ATTRIBUTE_SYSTEM) continue;
		if (ffd.dwFileAttributes & FILE_ATTRIBUTE_TEMPORARY) continue;

		// we got a PAK file so now we attempt to find the file in it
		f = FindFileInSpecificPak (basepath, ffd.cFileName, filename);

		// if we found it we use it
		if (f)
		{
			FindClose (ffh);
			return f;
		}
	} while (FindNextFile (ffh, &ffd));

	// not found
	FindClose (ffh);
	return NULL;
}


/*
=================
CollapsePAKFileName

Takes as input a file name in the format C:\Quake\ID1\PAK1.PAK\maps/dm1.bsp and
collapses it by removing the name of the PAK file to C:\Quake\ID1\maps\dm1.bsp

It is assumed that the program wants to write this file to disk immediately after,
and so any needed directory structures are also created.

It's safe to pass in a filename that is NOT in this format - we will just pass it
back unchanged.
=================
*/
void CollapsePAKFileName (char *pakfilename)
{
	int i;
	dpackheader_t pheader;

	// being by assuming it's a PAK file and try to find the name of it
	for (i = 0; ; i++)
	{
		// out of space!
		if (!pakfilename[i]) break;

		// look for '.PAK' delimiter (this will break if there's a directory entry ending with .pak
		// in the pak file, and presumably there is a crazy mod out there that has it...
		if (!strnicmp (&pakfilename[i], ".PAK\\", 5))
		{
			int valid;
			int j = i;

			// terminate
			pakfilename[i + 4] = 0;

			// validate it
			valid = ValidatePack (pakfilename);

			// put back
			pakfilename[i + 4] = '\\';

			// not a file
			if (!valid) continue;

			// now we know it's really a PAK file we scan backwards to find the name of the file in isolation
			for (; ; i--)
			{
				// too far back
				if (i < 1) return;

				// break on a path seperator
				if (pakfilename[i - 1] == '/') break;
				if (pakfilename[i - 1] == '\\') break;
			}

			for (; ; i++, j++)
			{
				// copy
				pakfilename[i] = pakfilename[j + 5];

				// done
				if (!pakfilename[i]) break;
			}

			break;
		}
	}

	// replace "/" with "\"
	for (i = strlen (pakfilename); i >= 0; i--)
		if (pakfilename[i] == '/')
			pakfilename[i] = '\\';

	// create the directory tree
	for (i = 0; ; i++)
	{
		if (!pakfilename[i]) break;

		if (pakfilename[i] == '\\')
		{
			// null terminate this part of it
			pakfilename[i] = 0;

			// make the directory
			_mkdir (pakfilename);

			// restore
			pakfilename[i] = '\\';
		}
	}
}


typedef struct searchpath_s
{
	char    filename[MAX_PATH];
	pack_t  *pack;          // only one of filename / pack will be used
	struct searchpath_s *next;
} searchpath_t;

searchpath_t    *com_searchpaths;


BOOL TestFileSystemFile (char *filepath)
{
	// try find the file
	FILE *f = fopen (filepath, "rb");

	if (f)
	{
		fclose (f);
		return TRUE;
	}

	return FALSE;
}

BOOL TestFileSystem (char *fspath)
{
	// look for indicators that Quake lives here
	if (TestFileSystemFile (va ("%s\\pak0.pak", fspath))) return TRUE;
	if (TestFileSystemFile (va ("%s\\config.cfg", fspath))) return TRUE;
	if (TestFileSystemFile (va ("%s\\autoexec.cfg", fspath))) return TRUE;
	if (TestFileSystemFile (va ("%s\\progs.dat", fspath))) return TRUE;
	if (TestFileSystemFile (va ("%s\\gfx.wad", fspath))) return TRUE;
	if (TestFileSystemFile (va ("%s\\s0.sav", fspath))) return TRUE;

	// quake doesn't live here...
	return FALSE;
}


pack_t *LoadPAKFile (char *pakname)
{
	int i;
	FILE *f;
	dpackheader_t header;
	pack_t *pak = NULL;

	f = fopen (pakname, "rb");

	// should never happen
	if (!f) return NULL;

	// read it in
	fread (&header, sizeof (dpackheader_t), 1, f);

	if (header.id[0] != 'P' || header.id[1] != 'A' || header.id[2] != 'C' || header.id[3] != 'K')
	{
		// not a pak file
		fclose (f);
		return NULL;
	}

	// set everything up
	pak = (pack_t *) QHeap_Alloc (sizeof (pack_t));
	pak->numfiles = header.dirlen / sizeof (dpackfile_t);
	pak->files = (packfile_t *) QHeap_Alloc (pak->numfiles * sizeof (packfile_t));
	strcpy (pak->filename, pakname);

	// seek to the directory
	fseek (f, header.dirofs, SEEK_SET);

	// read it all in
	for (i = 0; i < pak->numfiles; i++)
	{
		dpackfile_t info;

		// read and copy this item
		fread (&info, sizeof (dpackfile_t), 1, f);
		strcpy (pak->files[i].name, info.name);
		pak->files[i].filelen = info.filelen;
		pak->files[i].filepos = info.filepos;
	}

	return pak;
}


void LoadGameDir (char *fspath)
{
	WIN32_FIND_DATA ffd;
	HANDLE ffh;
	char searchpath[MAX_PATH];
	FILE *f;
	pack_t *pak;
	searchpath_t *search = (searchpath_t *) QHeap_Alloc (sizeof (searchpath_t));

	// add the directory
	strcpy (search->filename, fspath);
	search->pack = NULL;
	search->next = com_searchpaths;
	com_searchpaths = search;

	// add any paks
	sprintf (searchpath, "%s\\*.pak", fspath);

	if (!(ffh = FindFirstFile (searchpath, &ffd)))
		return;

	do
	{
		// don't use these types
		if (ffd.dwFileAttributes & FILE_ATTRIBUTE_COMPRESSED) continue;
		if (ffd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) continue;
		if (ffd.dwFileAttributes & FILE_ATTRIBUTE_ENCRYPTED) continue;
		if (ffd.dwFileAttributes & FILE_ATTRIBUTE_HIDDEN) continue;
		if (ffd.dwFileAttributes & FILE_ATTRIBUTE_OFFLINE) continue;
		if (ffd.dwFileAttributes & FILE_ATTRIBUTE_SYSTEM) continue;
		if (ffd.dwFileAttributes & FILE_ATTRIBUTE_TEMPORARY) continue;

		// we got a PAK file so now we add it
		pak = LoadPAKFile (va ("%s\\%s", fspath, ffd.cFileName));

		if (pak)
		{
			// add it in
			search = (searchpath_t *) QHeap_Alloc (sizeof (searchpath_t));
			search->pack = pak;
			search->next = com_searchpaths;
			com_searchpaths = search;
		}
	} while (FindNextFile (ffh, &ffd));

	// not found
	FindClose (ffh);
}


void BringUpFileSystem (char *fspath)
{
	int i;

	// bring up in the selected dir
	LoadGameDir (fspath);

	for (i = strlen (fspath); i; i--)
	{
		if (fspath[i] == ':') return;
		if (i < 2) return;

		if (fspath[i] == '/' || fspath[i] == '\\')
		{
			// already id1
			if (!stricmp (&fspath[i + 1], "id1")) return;

			fspath[i] = 0;
			strcat (fspath, "\\id1");
			break;
		}
	}

	// bring up id1
	LoadGameDir (fspath);
}


void InitFileSystem (char *bspname)
{
	// path to the filesystem root
	char fspath[MAX_PATH];

	// no paths initially
	com_searchpaths = NULL;

	// copy off the name so that we can safely modify it
	strcpy (fspath, bspname);

	while (1)
	{
		int i;

		for (i = strlen (fspath); i; i--)
		{
			// got to the drive designator without finding anything
			if (fspath[i] == ':') return;

			// file/folder delimiter
			if (fspath[i] == '\\' || fspath[i] == '/')
			{
				// terminate
				fspath[i] = 0;

				// test for quake
				if (TestFileSystem (fspath))
				{
					// got it
					BringUpFileSystem (fspath);

					// done
					return;
				}
			}
		}
	}
}


BOOL FindQuakeFile (char *filename, packfile_t *pakentry)
{
	int i;
	searchpath_t *search;
	pack_t *pak;

	for (search = com_searchpaths; search; search = search->next)
	{
		if (search->pack)
		{
			pak = search->pack;

			for (i = 0; i < pak->numfiles; i++)
			{
				if (!stricmp (pak->files[i].name, filename))
				{
					// copy out the file name and the info
					strcpy (pakentry->name, pak->filename);
					pakentry->filelen = pak->files[i].filelen;
					pakentry->filepos = pak->files[i].filepos;
					return TRUE;
				}
			}
		}
		else
		{
			FILE *f;

			sprintf (pakentry->name, "%s\\%s", search->filename, filename);

			f = fopen (pakentry->name, "rb");

			if (f)
			{
				// set up so that we can use the same code for both types of file
				fseek (f, 0, SEEK_END);
				pakentry->filelen = ftell (f);
				pakentry->filepos = 0;
				fclose (f);
				return TRUE;
			}
		}
	}

	// not found
	return FALSE;
}


FILE *OpenQuakeFile (char *filename)
{
	packfile_t found;
	FILE *f;

	// attempto to find it
	if (!FindQuakeFile (filename, &found)) return NULL;

	f = fopen (found.name, "rb");
	fseek (f, found.filepos, SEEK_SET);
	return f;
}


// int cos light.h doesn't know about BOOL
int LoadQuakeFile (char *filename, void **buffer)
{
	packfile_t found;
	FILE *f;

	// attempto to find it
	if (!FindQuakeFile (filename, &found))
	{
		*buffer = NULL;
		return 0;
	}

	f = fopen (found.name, "rb");
	fseek (f, found.filepos, SEEK_SET);

	// caller is responsible for freeing when done
	*buffer = malloc (found.filelen);
	fread (*buffer, found.filelen, 1, f);
	fclose (f);

	return 1;
}
