#include <io.h>
#include <stdlib.h>

#include "cmdlib.h"
#include "mathlib.h"

#define OP_WAD2TGA 1
#define OP_IMPORTMAP 2
#define OP_WAD2WAL 3
#define OP_WAD2HL 4

// CHANGE THIS ACCORDING TO YOUR OPERATING ENVIRONMENT
_CRTIMP int __cdecl _mkdir(const char *);

void Sys_mkdir (char *path)
{
	_mkdir (path);
}

#define MAX_DIMENSION 512


typedef struct
{ byte magic[4];            // "WAD2", Name of the new WAD format
  int numentries;          // Number of entries
  int diroffset;           // Position of WAD directory in file
} wadhead_t;

typedef struct
{ int offset;                 // Position of the entry in WAD
  int dsize;                  // Size of the entry in WAD file
  int size;                   // Size of the entry in memory
  char type;                   // type of entry, should be 'D'
  char unused[3];              // Never compressed, so why bother
  char name[16];               // 1 to 16 characters, '\0'-padded
} wadentry_t;

typedef struct
{
	char		name[16];
	unsigned	width, height;
	unsigned	offsets[4];
} miptex_t;

typedef struct miptex_s
{
	char name[32];
	unsigned width, height;
	unsigned offsets[4];
	char animname[32];
	int flags;
	int contents;
	int value;
} q2miptex_t;

typedef struct
{
	byte channel[3];
} rgbcolor_t;

typedef struct
{
	rgbcolor_t	colors[256];
} palette_t;

void ExportTGA(char *outpath, int width, int height, byte *pixels)
{
	FILE *tgaexport;
	byte		*buffer;
	int			i, c;

	buffer = malloc(width*height*4 + 18);
	memset (buffer, 0, 18);
	buffer[2] = 2;		// uncompressed type
	buffer[12] = width&255;
	buffer[13] = width>>8;
	buffer[14] = height&255;
	buffer[15] = height>>8;
	buffer[16] = 32;
	// 32-bit is a waste of space, but needed to make GtkRadiant shut up

	// Copy the pixels to the buffer and add the alpha channel
	c = width*height;
	for (i=0 ; i<c ; i++)
	{   // It's 20 19 18 cuz TGA is stupid and stores its pixels backwards
		buffer[i*4 + 20] = pixels[i*3];
		buffer[i*4 + 19] = pixels[i*3+1];
		buffer[i*4 + 18] = pixels[i*3+2];
		buffer[i*4 + 21] = 255;   // It's opaque, now GtkRadiant is happy
	}

	tgaexport = SafeOpenWrite(outpath);

	SafeWrite(tgaexport, buffer, width*height*4 + 18);

	fclose(tgaexport);

	free (buffer);
}

#define DEST_TGA  1
#define DEST_WAL  2
#define DEST_WAD3 3

#define MIPSIZE_SCALAR (85/64)

wadhead_t	wad3head;
FILE *wad3file;
wadentry_t	waddir[1024];   // Allow for some really damn big wads

int wad3size = 0;
int direntries = 0;

void BeginWadFile(char *source, wadhead_t *header)
{
	char dest[1024];

	// Copy the wad header and make a WAD file
	memcpy(&wad3head, header, sizeof(wadhead_t));
	strcpy(dest, source);
	StripExtension(dest);
	strcat(dest, ".hlwad");
	wad3file = SafeOpenWrite(dest);

	wad3head.magic[3] = '3';   // WAD3, not WAD2

	wad3size += sizeof(wadhead_t);
	SafeWrite(wad3file, &wad3head, sizeof(wadhead_t));
}

//#define NO_PAD

void ConvertLump(FILE *wadfile, wadentry_t *wadentry, palette_t *palette)
{
	byte data[MAX_DIMENSION*MAX_DIMENSION*2];
	byte pad[2] = {0, 1};   // Lame

	// WAD2 to WAD3 conversion can be done by just copying lumps and tacking the
	// palette on the end.

	fseek(wadfile, wadentry->offset, SEEK_SET);
	SafeRead(wadfile, &data, wadentry->size);

	SafeWrite(wad3file, &data, wadentry->size);
	SafeWrite(wad3file, pad, 2);
	SafeWrite(wad3file, palette, sizeof(palette_t));

	memcpy(&waddir[direntries], wadentry, sizeof(wadentry_t));
	waddir[direntries].offset = wad3size;   // Fix the offset
	waddir[direntries].size += sizeof(palette_t) + 2;

	wad3size += waddir[direntries].size;
	waddir[direntries].type = 'C';

	// Stupid stupid stupid...
	while(wad3size & 3)
	{
		wad3size++;
		waddir[direntries].size++;
		SafeWrite(wad3file, pad, 1);
	}
	waddir[direntries].dsize = waddir[direntries].size;

	direntries++;
}

void FinishWadFile()
{
	SafeWrite(wad3file, &waddir, sizeof(wadentry_t) * direntries);
	wad3head.diroffset = wad3size;
	fseek(wad3file, 0, SEEK_SET);
	SafeWrite(wad3file, &wad3head, sizeof(wadhead_t));
	fclose(wad3file);
}

void ImportWad(char *source, int desttype)
{
	FILE *wadfile;
	FILE *palettefile;
	FILE *walfile;

	char *workdirtemp;

	palette_t wadpalette;
	palette_t q2palette;

	byte w2wtable[256];

	wadhead_t header;
	miptex_t currenttex;
	static wadentry_t entries[1024];
	static byte pixels[MAX_DIMENSION*MAX_DIMENSION];
	static byte pixels24[MAX_DIMENSION*MAX_DIMENSION*3];
	char storagedir[256];
	char filename[512];
	int i, w, h, j, bytesize;
	int bestdiff, currentdiff, bestcolor;

	int tempmipofs;

	byte *q2mip;
	q2miptex_t q2mipheader;

	q2mip = malloc(90000);

	printf("Loading palette... ");

	palettefile = SafeOpenRead("q1colors.pal");
	SafeRead(palettefile, &wadpalette, sizeof(palette_t));
	fclose(palettefile);

	if(desttype == DEST_WAL)
	{
		printf("\n  Loading Quake II palette...");
		palettefile = SafeOpenRead("q2colors.pal");
		SafeRead(palettefile, &q2palette, sizeof(palette_t));
		fclose(palettefile);

		printf("\n  Building conversion table... ");
		for(i=0;i<255;i++)
		{
			bestdiff = 1000;
			bestcolor = 0;

			// Build the color conversion table
			for(j=0;j<255;j++)
			{
				currentdiff = (int)fabs(wadpalette.colors[i].channel[0] - q2palette.colors[j].channel[0]);
				currentdiff += (int)fabs(wadpalette.colors[i].channel[1] - q2palette.colors[j].channel[1]);
				currentdiff += (int)fabs(wadpalette.colors[i].channel[2] - q2palette.colors[j].channel[2]);
				if(currentdiff < bestdiff)
				{
					bestcolor = j;
					bestdiff = currentdiff;
				}
			}
			w2wtable[i] = bestcolor;
		}
		w2wtable[255] = 255;   // Translucent stays the same
	}

	printf("Done.\n");

	printf("Converting wad textures...\n");
	wadfile = SafeOpenRead(source);

	if(desttype == DEST_TGA)
	{
		Sys_mkdir("baseq3");
		Sys_mkdir("baseq3/textures");
	}
	else if(desttype == DEST_WAL)
	{
		Sys_mkdir("baseq2");
		Sys_mkdir("baseq2/textures");
	}

	StripExtension(source);

	workdirtemp = source;

	w = strlen(source);
	for(i=0;i<w;i++)
	{
		if(source[i] == '/' || source[i] == '\\')
			workdirtemp = source + i + 1;
	}

	if(desttype != DEST_WAD3)
	{
		if(desttype == DEST_TGA)
			sprintf(storagedir, "baseq3/textures/wad2tga_%s", workdirtemp);
		else
			sprintf(storagedir, "baseq2/textures/%s", workdirtemp);

		Sys_mkdir(storagedir);
	}

	// Load header
	SafeRead(wadfile, &header, sizeof(wadhead_t));

	// Swap it

	header.numentries = LittleLong(header.numentries);
	header.diroffset = LittleLong(header.diroffset);

	if(header.magic[0] != 'W' ||
	   header.magic[1] != 'A' ||
	   header.magic[2] != 'D' ||
	   header.magic[3] != '2')
		{
			fclose(wadfile);
			Error("Not a valid WAD2 file.\n");
		}

	if(header.numentries > 1024)
	{
		fclose(wadfile);
		Error("Holy shit, over 1024 textures in a WAD!?  You are truly insane");
	}
	printf("\n  Loading directory... ");
	fseek(wadfile, header.diroffset, SEEK_SET);

	for(i=0;i<header.numentries;i++)
	{
		SafeRead(wadfile, &entries[i], sizeof(wadentry_t));
		// Swap it
		entries[i].offset = LittleLong(entries[i].offset);
		entries[i].dsize = LittleLong(entries[i].dsize);
		entries[i].size = LittleLong(entries[i].size);
	}
	printf("Done.  Loaded %i entries.\n", header.numentries);

	if(desttype == DEST_WAD3)
		BeginWadFile(source, &header);

	printf("\n  Converting miptexes...\n");

	for(i=0;i<header.numentries;i++)
	{
		printf("%i: ", i);
		if(entries[i].type != 'D' && desttype != DEST_WAD3)   // WAD3 conversion copies everything
			continue;   // Not a miptex

		fseek(wadfile, entries[i].offset, SEEK_SET);
		SafeRead(wadfile, &currenttex, sizeof(miptex_t));

		// Stupid Id and their * liquids...
		if(currenttex.name[0] == '*' && desttype != DEST_WAD3)
			currenttex.name[0] = '#';

		printf("    %s\n", currenttex.name);

		if(currenttex.width > MAX_DIMENSION || currenttex.height > MAX_DIMENSION)
			Error("Texture too big, shrink it.\n");

		if(desttype == DEST_TGA)
		{
			sprintf(filename, "%s/%s.tga", storagedir, currenttex.name);

			fseek(wadfile, entries[i].offset + currenttex.offsets[0], SEEK_SET);
			// Seek to the first mip and rip it
			SafeRead(wadfile, &pixels, currenttex.width * currenttex.height);

			for(h=0;h<(int)currenttex.height;h++)
			{
				for(w=0;w<(int)currenttex.width;w++)
				{
					// Expand the pixels to 24-bit and invert the height
					// so the TGA doesn't shit on itself
					pixels24[((currenttex.height-h-1)*currenttex.width+w)*3] = 
						wadpalette.colors[pixels[h*currenttex.width+w]].channel[0];
					pixels24[((currenttex.height-h-1)*currenttex.width+w)*3+1] = 
						wadpalette.colors[pixels[h*currenttex.width+w]].channel[1];
					pixels24[((currenttex.height-h-1)*currenttex.width+w)*3+2] = 
						wadpalette.colors[pixels[h*currenttex.width+w]].channel[2];
				}
			}


			ExportTGA(filename, currenttex.width, currenttex.height, pixels24);
		}
		else if(desttype == DEST_WAL)
		{
			tempmipofs = sizeof(q2miptex_t);

			q2mipheader.animname[0] = '\0';  // Don't animate it
			q2mipheader.contents = 0;
			q2mipheader.flags = 0;
			q2mipheader.width = currenttex.width;
			q2mipheader.height = currenttex.height;
			q2mipheader.name[0] = '\0';   // Name's aren't important
			q2mipheader.value = 0;
			bytesize = q2mipheader.width * q2mipheader.height;

			for(j=0;j<4;j++)
			{
				fseek(wadfile, entries[i].offset + currenttex.offsets[j], SEEK_SET);
				SafeRead(wadfile, q2mip + tempmipofs, bytesize);
				q2mipheader.offsets[j] = tempmipofs;
				tempmipofs += bytesize;
				bytesize /= 4;
			}

			for(j=sizeof(q2miptex_t);j<tempmipofs;j++)
			{
				// Convert each pixel to the Q2 palette using the colormap
				q2mip[j] = w2wtable[q2mip[j]];
			}

			sprintf(filename, "%s/%s.wal", storagedir, currenttex.name);

			memcpy(q2mip, &q2mipheader, sizeof(q2miptex_t));

			walfile = SafeOpenWrite(filename);
			SafeWrite(walfile, q2mip, tempmipofs);
			fclose(walfile);
		}
		else
		{
			ConvertLump(wadfile, &entries[i], &wadpalette);
		}
	}


	if(desttype == DEST_WAD3)
		FinishWadFile();

	fclose(wadfile);
	free(q2mip);
}

int main(int argc, char **argv)
{
	int i;
	int currentoperation = 0;
	char source[1024];

	printf("WadConv Texture Toolbox by Eric 'Riot' Lasota.\n");
	printf("  Syntax: wadconv [-wad2wal] [-wad2tga] [-wad2hl] filename\n");

	for (i=1 ; i<argc ; i++)
	{
		if (!strcmp(argv[i], "-importmap"))
			currentoperation = OP_IMPORTMAP;
		else if(!strcmp(argv[i], "-wad2tga"))
			currentoperation = OP_WAD2TGA;
		else if(!strcmp(argv[i], "-wad2wal"))
			currentoperation = OP_WAD2WAL;
		else if(!strcmp(argv[i], "-wad2hl"))
			currentoperation = OP_WAD2HL;
		else if (argv[i][0] == '-')
			Error ("Unknown option \"%s\"", argv[i]);
		else
			break;
	}
	
	if(i == argc)
		Error ("An inputfile would be nice....");
	strcpy (source, argv[i]);

	if(currentoperation == OP_WAD2TGA)
	{
		DefaultExtension(source, ".wad");
		ImportWad(source, DEST_TGA);
	}
	else if(currentoperation == OP_WAD2WAL)
	{
		DefaultExtension(source, ".wad");
		ImportWad(source, DEST_WAL);
	}
	else if(currentoperation == OP_WAD2HL)
	{
		DefaultExtension(source, ".wad");
		ImportWad(source, DEST_WAD3);
	}


	return 0;
}